/*  ----------------------------------------------------------------<Prolog>-
    Name:       sflhttp.c
    Title:      HTTP and CGI Support functions
    Package:    Standard Function Library (SFL)

    Written:    96/05/31  iMatix SFL project team <sfl@imatix.com>
    Revised:    98/01/19

    Copyright:  Copyright (c) 1991-98 iMatix
    License:    This is free software; you can redistribute it and/or modify
                it under the terms of the SFL License Agreement as provided
                in the file LICENSE.TXT.  This software is distributed in
                the hope that it will be useful, but without any warranty.
 ------------------------------------------------------------------</Prolog>-*/

#include "prelude.h"                    /*  Universal header file            */
#include "sflstr.h"                     /*  String functions                 */
#include "sflsymb.h"                    /*  Symbol functions                 */
#include "sflconv.h"                    /*  Convertion functions             */
#include "sfllist.h"                    /*  Linked-list functions            */
#include "sflmem.h"                     /*  Memory functions                 */
#include "sflfind.h"                    /*  Find text functions              */
#include "sflfile.h"                    /*  Files functions                  */
#include "sflhttp.h"                    /*  Prototypes for functions         */


/*  Constants -------------------------------------------------------------- */

/*  This is the maximum size of a stream of HTTP query data coming from a
 *  file.  Used by cgi_parse_file_vars ().
 */

#define CGI_QUERY_FILE_MAX      65535U
#define MULTI_BUFFER_SIZE       16384


/*- Local functions ---------------------------------------------------------*/

static void   save_multipart_header   (SYMTAB *table, SYMTAB *header,
                                       char *data, char *tmp_name);
static void   multipart_decode_header (char *header, SYMTAB *table);
static DESCR *http_multipart2url      (const SYMTAB *symtab);


/*  ---------------------------------------------------------------------[<]-
    Function: http_escape

    Synopsis: Performs HTTP escaping on a string.  This works as follows:
    all characters except alphanumerics and spaces are converted into the
    3-byte sequence "%xx" where xx is the character's hexadecimal value;
    spaces are replaced by '+'.  Line breaks are stored as "%0D%0A", where
    a 'line break' is any one of: "\n", "\r", "\n\r", or "\r\n".  If the
    result buffer is NULL, calculates the required size, allocates a block
    of memory, and returns that.  Otherwise, returns result, which must be
    large enough for the escaping operation (see http_escape_size()).
    When you all http_escape() with a null target block, you must free the
    returned block using mem_free().  Returns NULL if it could not allocate
    a target block as required.
    ---------------------------------------------------------------------[>]-*/

char *
http_escape (
    const char *string,
    char *result)
{
    static char
        hex_char [] = "0123456789ABCDEF";
    char
        *target;                        /*  Where we store the result        */

    ASSERT (string);
    if (result == NULL)
        if ((result = mem_alloc (http_escape_size (string))) == NULL)
            return (NULL);              /*  Could not allocate a block       */

    target = result;
    while (*string)
      {
        if (isalnum (*string))          /*  Don't escape letters or digits   */
            *target++ = *string;
        else
        if (*string == ' ')             /*  Spaces are replaced by '+'       */
            *target++ = '+';
        else
        if (*string == '\n' || *string == '\r')
          {
            if ((string [1] == '\n' || string [1] == '\r')
            &&  (string [1] != *string))
                string++;
            *target++ = '%';            /*  New line becomes %0A%0D          */
            *target++ = '0';
            *target++ = 'A';
            *target++ = '%';
            *target++ = '0';
            *target++ = 'D';
          }
        else
          {
            *target++ = '%';            /*  Some other escaped character     */
            *target++ = hex_char [(byte) *string >> 4];
            *target++ = hex_char [(byte) *string & 15];
          }
        string++;
      }
    *target = '\0';                     /*  Terminate target string          */
    return (result);
}


/*  ---------------------------------------------------------------------[<]-
    Function: http_escape_size

    Synopsis: Returns the size of a string after HTTP escaping.  See the
    http_escape() function for details of the escaping algorithm.  Includes
    the null terminator in the returned size.
    ---------------------------------------------------------------------[>]-*/

size_t
http_escape_size (
    const char *string)
{
    size_t
        result_size = 1;                /*  Allow for null terminator        */

    ASSERT (string);
    while (*string)
      {
        if (isalnum (*string))          /*  Don't escape letters or digits   */
            result_size++;
        else
        if (*string == ' ')             /*  Spaces are replaced by '+'       */
            result_size++;
        else
        if (*string == '\n' || *string == '\r')
          {
            if ((string [1] == '\n' || string [1] == '\r')
            &&  (string [1] != *string))
                string++;
            result_size += 6;           /*  Line ending becomes %0D%0A       */
          }
        else
            result_size += 3;           /*  Some other escaped character     */

        string++;
      }
    return (result_size);
}


/*  ---------------------------------------------------------------------[<]-
    Function: http_unescape

    Synopsis: Removes HTTP escaping from a string.  See http_escape() for
    details of the escaping algorithm.  If the result string is NULL,
    modifies the source string in place, else fills-in the result string.
    Returns the resulting string.  End-of-line sequences (%0A%0D) are
    stored as a single new-line character, i.e. carriage-returns (%0D) are
    not stored.
    ---------------------------------------------------------------------[>]-*/

char *
http_unescape (
    char *string,
    char *result)
{
    /*  This lookup table gives us a quick way to convert a hex digit        */
    /*  into a binary value.  Note that the index must be [0..127].          */
    static char
        hex_to_bin [128] = {
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,    /*            */
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,    /*            */
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,    /*            */
            0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 0, 0, 0, 0, 0, 0,    /*   0..9     */
            0,10,11,12,13,14,15, 0, 0, 0, 0, 0, 0, 0, 0, 0,    /*   A..F     */
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,    /*            */
            0,10,11,12,13,14,15, 0, 0, 0, 0, 0, 0, 0, 0, 0,    /*   a..f     */
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };  /*            */
    char
        *target;                        /*  Where we store the result        */

    ASSERT (string);
    if (!result)                        /*  If result string is null,        */
        result = string;                /*    modify in place                */
    target = result;

    while (*string)
      {
        if (*string == '%')             /*  Unescape %xx sequence            */
          {
            *target = hex_to_bin [string [1] & 127] * 16
                    + hex_to_bin [string [2] & 127];
            string += 2;                /*  Bump past two hex digits         */
            if (*target != '\r')
                target++;               /*  We do not store CRs              */
          }
        else
        if (*string == '+')             /*  Spaces are escaped as '+'        */
            *target++ = ' ';
        else
            *target++ = *string;        /*  Otherwise just copy              */

        string++;
      }
    *target = '\0';                     /*  Terminate target string          */
    return (result);
}


/*  ---------------------------------------------------------------------[<]-
    Function: http_query2strt

    Synopsis: Parses a HTTP query string, building an array of strings of
    the format "name=value".  The query string is assumed to be in escaped
    format, so http_unescape() is always applied to the query string.
    Within the query string, field=value pairs are delimited by & or ;.
    Returns a pointer to the array.  The array is allocated dynamically.
    The array ends with a NULL string.  To free the table, call strtfree().
    If there was not enough memory to allocate the table, returns NULL.
    ---------------------------------------------------------------------[>]-*/

char **
http_query2strt (
    const char *original_query)
{
    char
        *query,                         /*  Local copy of query string       */
        *query_ptr,                     /*  Pointer into query string        */
        *query_next,                    /*  Pointer to next query chunk      */
        **strings;                      /*  Returned string array            */
    int
        char_nbr,                       /*  Index into query string          */
        string_count,                   /*  Size of string table             */
        string_nbr;                     /*  Index into string table          */

    ASSERT (original_query);

    if (*original_query == '&')         /*  Skip leading & if present        */
        original_query++;

    if ((query = mem_strdup (original_query)) == NULL)
        return (NULL);                  /*  Could not allocate memory        */

    /*  Break query string at & and ; delimiters and count strt size         */
    string_count = 1;                   /*  Last string has no delimiter     */
    for (char_nbr = 0; original_query [char_nbr]; char_nbr++)
        if (query [char_nbr] == '&' || query [char_nbr] == ';')
          {
            query [char_nbr] = '\0';
            string_count++;
          }

    /*  Allocate the array of pointers with one slot for the final NULL      */
    if ((strings = mem_alloc (sizeof (char *) * (string_count + 1))) == NULL)
      {
        mem_free (query);
        return (NULL);                  /*  Could not allocate memory        */
      }

    /*  Query string now consists of a series of substrings, each ending in
     *  a null character.  We have to unescape each substring, which we do
     *  in-place: the unescaped string is never larger than the original
     *  string.
     */
    query_ptr = query;
    for (string_nbr = 0; string_nbr < string_count; string_nbr++)
      {
        /*  Unescape next query string component                             */
        query_next = query_ptr + strlen (query_ptr) + 1;
        http_unescape (query_ptr, NULL);

        /*  Allocate space for "name=value" plus final null char             */
        strings [string_nbr] = mem_strdup (query_ptr);
        query_ptr = query_next;
      }
    strings [string_nbr] = NULL;        /*  Store final null pointer         */
    mem_free (query);                   /*  Release temporary memory         */
    return (strings);
}


/*  ---------------------------------------------------------------------[<]-
    Function: http_query2symb

    Synopsis: Parses a HTTP query string, and populates a symbol table with
    the resulting field values.  The query string is assumed to be escaped,
    so http_unescape() is always applied to the query string.  Within the
    query string, field=value pairs are delimited by & or ;.  Returns a
    SYMTAB pointer to the new table.  If there was not enough memory to
    allocate the table, returns NULL.
    ---------------------------------------------------------------------[>]-*/

SYMTAB *
http_query2symb (
    const char *query)
{
    char
        **strings;                      /*  Formatted string array           */
    SYMTAB
        *symtab;                        /*  Returned symbol table            */

    strings = http_query2strt (query);
    if (strings)
      {
        symtab = strt2symb (strings);
        strtfree (strings);
        return (symtab);
      }
    else
        return (NULL);                  /*  Couldn't create string table     */
}


/*  ---------------------------------------------------------------------[<]-
    Function: http_query2descr

    Synopsis: Parses a HTTP query string, and returns the values as a DESCR
    block, composed of null-delimited strings with an empty string at the
    end.  See strt2descr() and http_query2symb() for more details.  Returns
    the address of the allocated descriptor, or NULL if there was not
    enough memory.
    ---------------------------------------------------------------------[>]-*/

DESCR *
http_query2descr (
    const char *query)
{
    char
        **strings;                      /*  Formatted string array           */
    DESCR
        *descr;                         /*  Returned descriptor              */

    strings = http_query2strt (query);
    if (strings)
      {
        descr = strt2descr (strings);
        strtfree (strings);
        return (descr);
      }
    else
        return (NULL);                  /*  Couldn't create string table     */
}


/*  ---------------------------------------------------------------------[<]-
    Function: http_encode_meta

    Synopsis: Translates special characters into HTML/SGML metacharacters.
    The input buffer is not modified; you supply an output buffer and specify
    the maximum size of this buffer.  The input buffer must end in a null.
    Returns the final size of the translated data excluding the final null
    byte.  If the resulting data is too long, it is brutally truncated.
    ---------------------------------------------------------------------[>]-*/

size_t
http_encode_meta (
    char       *output,
    const char *input,
    size_t      outmax)
{
    /*  This lookup table provides a translation string for each byte
     *  in the character set.  We assume 8-bit characters.  When the
     *  string is NULL, the character is copied without translation.
     */
    static char
        *meta [256];                    /*  Metacharacter translation table  */
    static Bool
        first_time = TRUE;              /*  First time flag                  */
    int
        char_index;                     /*  Index into translation table     */
    size_t
        space_left;                     /*  Space left in destination        */
    const char
        *source;                        /*  Pointer to source string         */
    char
        *dest,                          /*  Pointer to result string         */
        *meta_char;                     /*  Pointer through metachar string  */

    ASSERT (input);
    ASSERT (output);

#   define OUTPUT(c)  if (!space_left) ; else { *dest++ = (c); space_left--; }

    /*  Initialise translation table first time through                      */
    if (first_time)
      {
        first_time = FALSE;
        for (char_index = 0; char_index < 256; char_index++)
            meta [char_index] = NULL;

#if (defined (__UNIX__) || defined (__WINDOWS__))
        /*  UNIX and Windows generally use ISO-8859-1 (Latin-1)              */
        meta [0xC4] = "Auml";
        meta [0xC5] = "Aring";
        meta [0xC6] = "AElig";
        meta [0xD6] = "Ouml";
        meta [0xDC] = "Uuml";
        meta [0xE0] = "agrave";
        meta [0xE1] = "aacute";
        meta [0xE2] = "acirc";
        meta [0xE4] = "auml";
        meta [0xE5] = "aring";
        meta [0xE6] = "aelig";
        meta [0xE7] = "ccedil";
        meta [0xE8] = "egrave";
        meta [0xE9] = "eacute";
        meta [0xEA] = "ecirc";
        meta [0xEB] = "euml";
        meta [0xEC] = "igrave";
        meta [0xED] = "iacute";
        meta [0xEE] = "icirc";
        meta [0xEF] = "iuml";
        meta [0xF2] = "ograve";
        meta [0xF3] = "oacute";
        meta [0xF4] = "ocirc";
        meta [0xF6] = "ouml";
        meta [0xF9] = "ugrave";
        meta [0xFA] = "uacute";
        meta [0xFB] = "ucirc";
        meta [0xFC] = "uuml";
        meta [0xFD] = "yuml";

#elif (defined (__MSDOS__))
        /*  DOS generally uses the PC-1 alphabet                             */
        meta [0x80] = "uuml";
        meta [0x82] = "eacute";
        meta [0x83] = "acirc";
        meta [0x84] = "auml";
        meta [0x85] = "agrave";
        meta [0x86] = "aring";
        meta [0x87] = "ccedil";
        meta [0x88] = "ecirc";
        meta [0x89] = "euml";
        meta [0x8A] = "egrave";
        meta [0x8B] = "iuml";
        meta [0x8C] = "icirc";
        meta [0x8D] = "igrave";
        meta [0x8E] = "Auml";
        meta [0x2F] = "Aring";
        meta [0x91] = "aelig";
        meta [0x92] = "AElig";
        meta [0x93] = "ocirc";
        meta [0x94] = "ouml";
        meta [0x95] = "ograve";
        meta [0x96] = "ucirc";
        meta [0x97] = "ugrave";
        meta [0x98] = "yuml";
        meta [0x99] = "Ouml";
        meta [0x9A] = "Uuml";
        meta [0xA0] = "aacute";
        meta [0xA1] = "iacute";
        meta [0xA2] = "oacute";
        meta [0xA3] = "uacute";
#endif
      }
    if (outmax == 0)                    /*  Special case for zero space      */
        return (0);

    space_left = outmax - 1;            /*  Allow for final null byte        */
    dest = output;
    for (source = input; *source; source++)
      {
        meta_char = meta [(int) *source & 255];
        if (meta_char)
          {
            OUTPUT ('&');
            while (*meta_char)
              {
                OUTPUT (*meta_char);
                meta_char++;
              }
            OUTPUT (';');
          }
        else
            OUTPUT (*source);
      }
    *dest = '\0';
    return ((size_t) (dest - output));
}


/*  ---------------------------------------------------------------------[<]-
    Function: cgi_parse_query_vars

    Synopsis: Parses a CGI query string and loads the resulting variables
    into an existing symbol table, optionally prefixing each name with a
    string.  Returns the number of variables loaded.  The prefix can be
    NULL or empty if not required.
    ---------------------------------------------------------------------[>]-*/

int
cgi_parse_query_vars (
    SYMTAB *symtab,
    const char *query,
    const char *prefix)
{
    static char
        query_var [LINE_MAX];           /*  Query variable name              */
    char
        **query_vars,                   /*  Query as string table            */
        *equals;                        /*  Equal sign in variable           */
    int
        string_nbr,                     /*  Index into string table          */
        variables = 0;                  /*  Number of variables loaded       */

    ASSERT (symtab);
    if ((query_vars = http_query2strt (query)) == NULL)
        return (0);                     /*  Not enough memory                */

    for (string_nbr = 0; query_vars [string_nbr]; string_nbr++)
      {
        equals = strchr (query_vars [string_nbr], '=');
        if (equals)
          {
            *equals = '\0';             /*  Cut into two strings             */
            if (prefix && *prefix)
              {
                ASSERT (strlen (query_vars [string_nbr])
                      + strlen (prefix) < LINE_MAX);
                xstrcpy (query_var, prefix, query_vars [string_nbr], NULL);
              }
            else
                strcpy (query_var, query_vars [string_nbr]);

            sym_assume_symbol (symtab, query_var, equals + 1);
            *equals = '=';              /*  Restore previous state           */
            variables++;                /*  Count this variable              */
          }
      }
    strtfree (query_vars);
    return (variables);
}


/*  ---------------------------------------------------------------------[<]-
    Function: cgi_parse_file_vars

    Synopsis: Parses a CGI query string stored in a file, and loads the
    resulting variables into an existing symbol table, optionally
    prefixing each name with a string.  Returns the number of variables
    loaded.  The prefix can be NULL or empty if not required.  The
    file data is assumed to be escaped (see http_escape()); the data
    should not contain line breaks, spaces, or other unescaped chars.
    The file should already have been opened: a typical use for this
    function is to parse the values supplied in stdin.  The maximum size
    for the file is CGI_QUERY_FILE_MAX characters.
    ---------------------------------------------------------------------[>]-*/

int
cgi_parse_file_vars (
    SYMTAB *symtab,
    FILE   *file,
    const char *prefix,
    size_t size)
{
    char
        *query;                         /*  Data loaded from file            */
    size_t
        read_size;                      /*  Amount of data read from file    */
    int
        variables = 0;                  /*  Number of variables loaded       */

    ASSERT (file);
    ASSERT (symtab);
    ASSERT (size <= CGI_QUERY_FILE_MAX);

    if ((query = mem_alloc (size + 1)) == NULL)
        return (0);

    read_size = fread (query, 1, size, file);
    query [read_size] = '\0';
    variables = cgi_parse_query_vars (symtab, query, prefix);
    mem_free (query);
    return (variables);
}


/*  ---------------------------------------------------------------------[<]-
    Function: http_multipart_decode

    Synopsis: Get form data from a multipart encoded data file. Each field
    have a entry in the symbol table. If the file content a file (INPUT field
    type FILE), the file is stored in a temporary file.
    Name of this file is added in symbol table, %variable_name%_tmp or
    if %variable_name% is numeric, %variable_name - 1%.
    ---------------------------------------------------------------------[>]-*/

DESCR *
http_multipart_decode (const char *mime_file, const char *store_path)
{
    FILE
        *f_source,
        *f_tmp = NULL;
    char
        *tmp_name = NULL,
        *p_head,
        *p_data,
        *p_next,
        *buffer;
    int
        offset,
        read_size;
    static char
        separator [80 + 1];
    static int
        sep_size;
    SYMTAB
        *table,
        *header_tab;
    qbyte
        tmp_index = 1;
    DESCR
        *descr = NULL;

    buffer = mem_alloc (MULTI_BUFFER_SIZE + 1);
    if (buffer == NULL)
        return (NULL);

    table = sym_create_table ();
    if (table == NULL)
      {
        mem_free (buffer);
        return (NULL);
      }

    header_tab = sym_create_table ();
    if (header_tab == NULL)
      {
        mem_free (buffer);
        sym_delete_table (table);
        return (NULL);
      }

    f_source = fopen (mime_file, "rb");
    if (f_source == NULL)
      {
        mem_free (buffer);
        sym_delete_table (table);
        sym_delete_table (header_tab);
        return (NULL);
      }

    memset (separator, 0, sizeof (separator));
    separator [0] = 0x0D;
    separator [1] = 0x0A;
    fgets (&separator [2], 78, f_source);
    strconvch (&separator [2] , '\r', '\0');
    strconvch (&separator [2] , '\n', '\0');
    sep_size  = strlen (separator);

    read_size = fread (buffer, 1, MULTI_BUFFER_SIZE, f_source);
    p_next = buffer;
    while (read_size > 0)
      {
        sym_empty_table (header_tab);
        p_head = p_next;
        p_data = (char *) memfind ((byte *) p_head,
                          MULTI_BUFFER_SIZE - (p_head - buffer),
                          (byte *) "\r\n\r\n", 4, FALSE);
        if (p_data)
          {
            *p_data = '\0';
            p_data += 4;
          }
        if (p_head)
          {
            multipart_decode_header (p_head, header_tab);
            if (sym_lookup_symbol (header_tab, "filename") != NULL)
              {
                if (f_tmp != NULL)
                  {
                    ASSERT (tmp_name != NULL);
                    fclose (f_tmp);
                    f_tmp = NULL;
                    if (get_file_size (tmp_name) == 0)
                        file_delete (tmp_name);
                  }
                tmp_name = get_tmp_file_name (store_path, &tmp_index, "tmp");
                f_tmp = fopen (tmp_name, "wb");
              }
          }
        p_next = (char *) memfind ((byte *) p_data,
                          read_size - (p_data - buffer),
                          (byte *) separator, sep_size, FALSE);
        if (p_next != NULL)
          {
            *p_next = '\0';
            save_multipart_header (table, header_tab, p_data, tmp_name);
            if (f_tmp)
              {
                fwrite (p_data, p_next - p_data, 1, f_tmp);
                fclose (f_tmp);
                f_tmp = NULL;
                if (get_file_size (tmp_name) == 0)
                    file_delete (tmp_name);
              }
            p_next += sep_size;

            /*  Found end of file marker                                     */
            if (*p_next == '-' && *(p_next + 1) == '-')
              {
                if (f_tmp)
                  {
                    fclose (f_tmp);
                    f_tmp = NULL;
                    if (get_file_size (tmp_name) == 0)
                        file_delete (tmp_name);
                  }
                break;
              }
            else
                while (*p_next == '\r' || *p_next == '\n')
                    p_next++;
          }
        else
          {
            if (f_tmp)
                fwrite (p_data, &buffer [read_size - sep_size ] - p_data,
                        1, f_tmp);
            offset = 0;
            while (read_size > 0 && p_next == NULL)
              {
                memmove (buffer, &buffer [read_size - sep_size + offset ],
                                 sep_size);
                read_size = fread (&buffer [sep_size], 1,
                                   MULTI_BUFFER_SIZE - sep_size, f_source);
                p_next = (char *) memfind ((byte *) buffer,
                                  read_size + sep_size,
                                  (byte *) separator, sep_size, FALSE);
                if (p_next != NULL)
                  {
                    *p_next = '\0';
                    save_multipart_header (table, header_tab,
                                           p_data, tmp_name);
                    if (f_tmp)
                      {
                        fwrite (buffer, p_next - buffer, 1, f_tmp);
                        fclose (f_tmp);
                        f_tmp = NULL;
                        if (get_file_size (tmp_name) == 0)
                            file_delete (tmp_name);
                      }
                    p_next += sep_size;

                   /*  Found end of file marker                              */
                   if (*p_next == '-' && *(p_next + 1) == '-')
                     {
                       read_size = 0;
                       break;
                     }
                   else
                       while (*p_next == '\r' || *p_next == '\n')
                           p_next++;
                   read_size += sep_size;
                  }
                else
                  {
                    if (f_tmp)
                        fwrite (buffer, read_size, 1, f_tmp);
                    offset = sep_size;
                  }

              }
          }
      }
    if (f_tmp)
      {
        fclose (f_tmp);
        if (get_file_size (tmp_name) == 0)
            file_delete (tmp_name);
      }
    sym_delete_table (header_tab);
    fclose (f_source);
    mem_free (buffer);

    descr = http_multipart2url (table);
    sym_delete_table (table);

    return (descr);
}


/*  -------------------------------------------------------------------------
    Function: http_multipart2url

    Synopsis: Convert a symbol table to a string in format :
              name=value&name=value&name=value ....
    -------------------------------------------------------------------------*/

static DESCR *
http_multipart2url  (const SYMTAB *symtab)
{
    DESCR
        *descr;                         /*  Formatted descriptor             */
    SYMBOL
        *symbol;                        /*  Pointer to symbol                */
    char
        *p_char,
        *p_val,
        *buffer;
    qbyte
        buffer_length = 0;

    if (!symtab)
        return (NULL);                  /*  Return NULL if argument is null  */

    /*  Calculate length of buffer                                           */
    for (symbol = symtab-> symbols; symbol; symbol = symbol-> next)
        buffer_length += strlen (symbol-> name) + strlen (symbol-> value) + 2;

    buffer = mem_alloc (buffer_length + 1);
    if (buffer == NULL)
        return (NULL);

    p_char = buffer;
    for (symbol = symtab-> symbols; symbol; symbol = symbol-> next)
      {
        if (symbol != symtab-> symbols)
            *p_char++ = '&';
        p_val = symbol-> name;
        while (*p_val)
            *p_char++ = *p_val++;
        *p_char++ = '=';
        p_val = symbol-> value;
        while (*p_val)
            *p_char++ = *p_val++;
      }
    *p_char = '\0';
    descr = mem_alloc (sizeof (DESCR));
    if (descr)
      {
        descr-> size = buffer_length;
        descr-> data = (byte *) buffer;
      }
    else
        mem_free (buffer);

    return (descr);
}


/*  -------------------------------------------------------------------------
    Function: save_multipart_header

    Synopsis: Store field name and value in symbol table.
    -------------------------------------------------------------------------*/

static void
save_multipart_header (SYMTAB *table, SYMTAB *header, char *data,
                       char *tmp_name)
{
    SYMBOL
        *sym_filename,
        *symbol;
    static char
        tmp_val [LINE_MAX + 1];
    char
        *value;
    short
        n_value;

    /* Check if is a file uploaded or form field                             */
    if ((sym_filename = sym_lookup_symbol (header, "filename")) != NULL)
      {
        symbol = sym_lookup_symbol (header, "name");
        if (symbol)
          {
            value = http_escape (sym_filename-> value, NULL);
            if (value)
              {
                sym_assume_symbol (table, symbol-> value, value);
                mem_free (value);
              }
            n_value = atoi (symbol-> value);
            if (n_value > 0)
                sprintf (tmp_val, "%d", n_value - 1);
            else
                xstrcpy (tmp_val, symbol-> value, "_tmp", NULL);

            value = http_escape (tmp_name, NULL);
            if (value)
              {
                sym_assume_symbol (table, tmp_val, value);
                mem_free (value);
              }
          }
      }
    else
      {
        symbol = sym_lookup_symbol (header, "name");
        if (symbol)
          {
            value = http_escape (data, NULL);
            if (value)
              {
                sym_assume_symbol (table, symbol-> value, value);
                mem_free (value);
              }
          }
      }

}


/*  -------------------------------------------------------------------------
    Function: multipart_decode_header

    Synopsis: Decode mime header of a part.
    -------------------------------------------------------------------------*/

static void
multipart_decode_header (char *header, SYMTAB *table)
{
    char
        *p,
        *variable,
        *value = "";

    ASSERT (header);
    ASSERT (table);

    p        = header;
    variable = header;
    while (*p)
      {
        if ((*p == ':' && *(p + 1) == ' ')
        ||   *p == '=')
          {
            *p++ = '\0';
            if (*p == ' ' || *p == '"')
                p++;
            value = p;
          }
        else
        if (*p == ';' || *p == '\r'||  *p == '\n')
          {
            if (*(p - 1) == '"')
                *(p - 1) = '\0';
            else
                *p = '\0';
            http_unescape (value, NULL);
            sym_assume_symbol (table, variable, value);
            p++;
            while (*p == ' ' || *p == '\r' || *p == '\n')
                p++;
            variable = p;
          }
        p++;
      }
    if ( p != header
    && *(p - 1) == '"')
        *(p - 1) = '\0';
    http_unescape (value, NULL);
    sym_assume_symbol (table, variable, value);
}


/*  ---------------------------------------------------------------------[<]-
    Function: is_full_url

    Synopsis: If the specified string starts with a URL scheme, returns
    TRUE, else returns FALSE.  A schema is one or more of [A-Za-z0-9+-.]
    followed by a ':'.
    ---------------------------------------------------------------------[>]-*/

Bool
is_full_url (const char *string)
{
    Bool
        scheme_size = 0;

    ASSERT (string);
    while (*string)
      {
        if (isalpha (*string)
        ||  isdigit (*string)
        ||  *string == '+'
        ||  *string == '-'
        ||  *string == '.')
            scheme_size++;              /*  So far, a valid scheme name      */
        else
        if (*string == ':')
            return (scheme_size > 0);   /*  Okay if ':' was not first char   */
        else
            return (FALSE);             /*  No scheme name found             */
        string++;
      }
    return (FALSE);                     /*  End of string but no scheme      */
}
