/*************************************************************/
/* CMANAGER                                                  */
/*           This is a TCP/IP connection manager.  It will   */
/*           start or quiesce an Internet connection using   */
/*           SLIP capabilities provided in TCP/IP V2 for     */
/*           OS/2.                                           */
/*************************************************************/

/*************************************************************/
/* Queues:                                                   */
/*           CMCommandQueue   (IPC queue for commands)       */
/*                                                           */
/* Commands:                                                 */
/*           START            (activate connection)          */
/*                                                           */
/*           STOP             (deactivate connection after   */
/*                             current activity completed)   */
/*                                                           */
/*           FORCE            (deactivate connection         */
/*                             immediately)                  */
/*                                                           */
/*           EXIT             (terminate the connection      */
/*                             manager)                      */
/*                                                           */
/*************************************************************/

trace OFF

  signal on HALT name CLEANUP
  signal on FAILURE name CLEANUP
  signal on SYNTAX name CLEANUP

  address CMD

  call initialize

  call emit '001I connection manager started'

  do while \stopped()
     command = get_command()

     if (command = '') & c_quiesced() then do
        if inactive() then
          call force
        else
          call syssleep 10
     end /* if */
     
     else do
       parse upper var command verb '15'x id '15'x queue

       verb = strip(verb)
       id = strip(id)
       queue = strip(queue)

       call accept queue

       select
         when (verb == 'START') then call do_start(id)
         when (verb == 'STOP') then call do_quiesce(id)
         when (verb == 'FORCE') then call force(id)
         when (verb == 'EXIT') then result = do_exit(id)
         otherwise call emit '002E unknown command:' command
       end /* select */
     end /* if */

     G._cs = G._ns
  end /* while */

/*************************************************************/
/* Cleanup gracefully after exit or abort                    */
/*************************************************************/

CLEANUP:

  if \datatype(RC, numeric) then
    error = 'not set'
  else
    error = RC

  if \datatype(SIGL, numeric) then
    line = 'not set'
  else
    line = SIGL

  call emit '003I terminating, RC='error', line='line

  call flush_queue G._cq

  if inactive() then
    call force

exit (RC <> 0)

/*************************************************************/
/* Perform environment initialization at startup             */
/*************************************************************/

initialize: procedure expose G.

  G._fax = 1
  G._fax_on = '@d:\software\package\faxworks\fxrcv -on 1> nul 2> nul'
  G._fax_off = '@d:\software\package\faxworks\fxrcv -off 1> nul 2> nul'
  
  G._cq = CMCommandQueue
  G._starts = 0

  call rxqueue delete, G._cq
  call rxqueue create, G._cq

  G._s2r.UP = 0
  G._s2r.QUIESCED = 1
  G._s2r.DOWN = 2

  G._r2s.0 = 'UP'
  G._r2s.1 = 'QUIESCED'
  G._r2s.2 = 'DOWN'

  G._driver = 'SLIP'

  if task_active(G._driver, tasks()) then
    G._cs = G._s2r.UP
  else
    G._cs = G._s2r.DOWN

  G._ns = G._cs

  call emit '017I connection is initially' state_name(G._cs)

  call flush_queue G._cq

  call rxfuncadd syssleep, rexxutil, syssleep

return

/*************************************************************/
/* Get rid of anything sitting in a queue                    */
/*************************************************************/

flush_queue: procedure expose G.

  parse arg queues

  do while (queues <> "")
    parse var queues queue queues

    previous = rxqueue(set, queue)

    do while (queued() > 0)
      pull line

      call emit '004I flushed' '"'line'"' 'from' queue
    end /* while */
 
    previous = rxqueue(set, previous)
  end /* while */

return

/*************************************************************/
/* Get command from command queue (exit immediately if the   */
/* quiesced state is set and nothing is in the command queue)*/
/*************************************************************/

get_command: procedure expose G.

  previous = rxqueue(set, G._cq)

  if \c_quiesced() | (queued() > 0) then do
    command = linein('QUEUE:')
    command = strip(space(command, 1))

    call emit '005I received command' '"'command'"'
  end /* if */  

  else do
    command = ""

    call emit '006W no command in queue'
  end /* if */

  previous = rxqueue(set, previous)

return (command)

/*************************************************************/
/* Perform EXIT operation                                    */
/*************************************************************/

do_exit: procedure expose G.

  parse arg id

  G._stopped = 1

return (0)

/*************************************************************/
/* Perform START operation, including starting SLIP driver   */
/*************************************************************/

do_start: procedure expose G.

  parse arg id

  call emit '007I' id(id) 'requesting START'

  select
    when c_up() then G._ns = up()

    when c_quiesced() then G._ns = up()

    otherwise do
      if task_active(G._driver, tasks()) then
        G._ns = up()
      else do
        call emit '008I starting' G._driver'...' 

        call fax_off

       '@start "IP Connection Driver" /min /n' G._driver
        sRC = RC

       '@SLIPWAIT 240 > nul'
        wRC = RC

        if (rc == 0) & task_active(G._driver, tasks()) then do 
          G._ns = up()

          call emit '009I driver successfully started'
        end /* if */

        else do
          G._ns = down()
          
          call emit '010E driver did not start, SRC='sRC', wRC='wRC
        end /* else */
      end /* else */
    end /* otherwise */
  end /* select */

  if c_up(G._ns) then
    call new_start

return

/*************************************************************/
/* Set quiesced mode if no more pending STARTs               */
/*************************************************************/

do_quiesce: procedure expose G.

  parse arg id

  call emit '016I' id(id) 'requesting STOP'

  call new_stop

  if \starts() then do
    select
      when c_quiesced() then G._ns = quiesced()
      when c_down() then G._ns = down()
      otherwise G._ns = quiesced()
    end
  end /* if */

return

/*************************************************************/
/* Force an immediate shutdown of the SLIP driver            */
/*************************************************************/

force: procedure expose G.

  parse arg id

  call emit '011I' id(id) 'terminating' G._driver

 '@killem' G._driver '> nul'
  result = RC

  if \task_active(G._driver, tasks()) then do
    G._ns = down()

    call emit '012I driver successfully terminated'

    call fax_on
  end /* if */
      
  else do
    G._ns = G._cs

    call emit '013E driver not terminated, RC='result
  end /* else */

return

/*************************************************************/
/* Display and optionally log a message w/ timestamp         */
/*************************************************************/

emit: procedure expose G.

  parse arg message

  say (date(USA) time() message)

return 

/*************************************************************/
/* Return 1 if there are no active sockets                   */
/*************************************************************/

inactive: procedure expose G.

   current = rxqueue(get)

   call flush_queue current

  '@netstat -s | rxqueue' current

   sockets = 0

   do scan = 1 to queued()
     pull information
      
     parse var information socket type . . host state .

     active =,
       (type = 'DGRAM') & (host <> '0.0.0.0') |,
       (type = 'STREAM') & (state <> 'LISTEN')

     if (active == 1) then do
       call emit '014I Waiting on' type 'socket' socket 'to' host

       sockets = sockets + 1
     end /* if */
   end scan

  call emit '115I' sockets 'active sockets'

return (sockets <= 0)

/*************************************************************/
/* Return 1 if the "stopped" flag has been set               */
/*************************************************************/

stopped: procedure expose G.

return (G._stopped == 1)

/*************************************************************/
/* Return 1 if the specified state is UP                     */
/*************************************************************/

c_up: procedure expose G.

  parse arg state

  if (state = '') then
    state = G._cs

return (state = G._s2r.UP)

/*************************************************************/
/* Return 1 if the specified state is DOWN                   */
/*************************************************************/

c_down: procedure expose G.

  parse arg state

  if (state = '') then
    state = G._cs

return (state = G._s2r.DOWN)

/*************************************************************/
/* Return 1 if the specified state is QUIESCED               */
/*************************************************************/

c_quiesced: procedure expose G.

  parse arg state

  if (state = '') then
    state = G._cs

return (state = G._s2r.QUIESCED)

/*************************************************************/
/* Return the numeric state for UP                           */
/*************************************************************/

up: procedure expose G.

return (G._s2r.UP)

/*************************************************************/
/* Return the numeric state for DOWN                         */
/*************************************************************/

down: procedure expose G.

return (G._s2r.DOWN)

/*************************************************************/
/* Return the numeric state for QUIESCED                     */
/*************************************************************/

quiesced: procedure expose G.

return (G._s2r.QUIESCED)

/*************************************************************/
/* Return a generic program id if the id was omitted         */
/*************************************************************/

id: procedure expose G.

  parse arg id

  if (id = '') then
    id = 'program'
 
return (id)

/*************************************************************/
/* Return the name corresponding to a numeric state          */
/*************************************************************/

state_name: procedure expose G.

  parse arg state

return (G._r2s.state)

/*************************************************************/
/* Register START request by incrementing active start count */
/*************************************************************/

new_start: procedure expose G.

  starts = G._starts + 1

  call emit '018I' starts 'active START requests'  

  G._starts = starts

return

/*************************************************************/
/* Register STOP request by decrementing active start count  */
/*************************************************************/

new_stop: procedure expose G.

  starts = G._starts - 1

  If (starts < 0) then
    starts = 0

  call emit '019I' starts 'active START requests'  

  G._starts = starts

return

/*************************************************************/
/* Return 1 if there are any active START requests           */
/*************************************************************/

starts: procedure expose G.

return (G._starts > 0)

/*************************************************************/
/* Turn on fax receive                                       */
/*************************************************************/

fax_on: procedure expose G.

  if (G._fax == 1) then do
    address CMD G._fax_on

    if (RC == 0) then 
      call emit '020I Fax receive enabled'
    else
      call emit '021W Fax receive not enabled, RC='RC
  end /* if */

return

/*************************************************************/
/* Turn off fax receive                                      */
/*************************************************************/

fax_off: procedure expose G.

  if (G._fax == 1) then do
    address CMD G._fax_off

    if (RC == 0) then 
      call emit '022I Fax receive DISABLED'
    else
      call emit '023W Fax receive not disabled, RC='RC
  end /* if */

return

/*************************************************************/
/* Send an acceptance notification to the caller's response  */
/* queue (if any)                                            */
/*************************************************************/

accept: procedure expose G.

  parse arg response

  if (queue <> "") then do
    call emit '024I sending acknowledgement on queue' response

    previous = rxqueue(set, response)

    queue (1)

    call rxqueue set, previous
  end /* if */

return
