/*--------------------------------------------------------------------------*/
/* Connect to ANSRemote/IP service                                          */
/*                                                                          */
/*     This script provides an automatic login capability for ANS'          */
/*     ANSRemote/IP service.                                                */
/*                                                                          */
/*     This script is essentially SLIPUP.CMD as supplied with TCP/IP V2.    */
/*     One important modification was to change the Annex command prompt    */
/*     to the prompt issued by the ANSRemote terminal servers               */
/*                                                                          */
/*     Dependencies:                                                        */
/*                                                                          */
/*     1. TCP/IP for OS/2 V2 (may work on previous versions but not tested) */
/*                                                                          */
/*     2, David Bolen's SLIP driver as supplied with TCP/IP for OS/2 V2     */
/*                                                                          */
/*     3. Assumes SLIP-only connection environment (no NICs).               */
/*                                                                          */
/*--------------------------------------------------------------------------*/

trace OFF

  entry = time(reset)

  parse arg interface , parameters

  signal on HALT name CLEANUP
  signal on SYNTAX name CLEANUP
  signal on FAILURE name CLEANUP

/*--------------------------------------------------------------------------*/
/* Set up modem                                                             */
/*--------------------------------------------------------------------------*/

 '@MODE COM1:38400,n,8,1,BUFFER=ON,RTS=HS,TO=OFF,IDSR=OFF,OCTS=ON,' ||,
   'XON=OFF,ODSR=OFF,DTR=ON'

  cr = '0D'x
  crlf = '0D0A'x

/*--------------------------------------------------------------------------*/
/* Prompt the user to enter username, passwords, dial command               */
/*--------------------------------------------------------------------------*/

   SLIP_user = Value('SLIP.USERNAME',, 'OS2ENVIRONMENT')
   SLIP_dial = Value('SLIP.DIAL_STRING',, 'OS2ENVIRONMENT')
   SLIP_pw = Value('SLIP.PASSWORD',, 'OS2ENVIRONMENT')

   if (SLIP_user == "") | (SLIP_dial == "") | (SLIP_pw == "") then
     signal cleanup

/*--------------------------------------------------------------------------*/
/* Get rid of prior activity, then issue dial command                       */
/*--------------------------------------------------------------------------*/

  call flush_receive

  call send SLIP_dial || cr
  call waitfor 'CONNECT'
  call waitfor crlf

/*--------------------------------------------------------------------------*/
/* Send username and password when prompted                                 */
/*--------------------------------------------------------------------------*/

  call send cr

  call waitfor 'Username:' 
  call flush_receive 'echo'
  call send SLIP_user || cr

  call waitfor 'Password:'
  call flush_receive 'echo'
  call send SLIP_pw || cr

/*--------------------------------------------------------------------------*/
/* When main prompt received, put us in SLIP mode                           */
/*--------------------------------------------------------------------------*/

  call waitfor 'ANSRemote>'
  call flush_receive 'echo'
  call send 'slip' || cr

/*--------------------------------------------------------------------------*/
/* Get terminal server and our IP address                                   */
/*--------------------------------------------------------------------------*/

  call waitfor 'Your address is'

  parse var waitfor_buffer,
    . 'Annex address is' one '.' two '.' three '.' four '.' .

  remote = (one || '.' || two || '.' || three || '.' || four)

  call waitfor crlf

  parse var waitfor_buffer,
    one '.' two '.' three '.' four '.' .

  local = (one || '.' || two || '.' || three || '.' || four)

  call flush_receive 'echo'

/*--------------------------------------------------------------------------*/
/* Tell the user what addresses we found, and configure the interface       */
/*--------------------------------------------------------------------------*/

  call display_parameters local, remote

 '@ifconfig' interface local remote 'netmask 255.255.0.0'
 '@route -f add default' remote '1'

/*--------------------------------------------------------------------------*/
/* Common exit routine                                                      */
/*--------------------------------------------------------------------------*/

CLEANUP:

exit 0


/*--------------------------------------------------------------------------*/
/*                            send ( sendstring)                            */
/*..........................................................................*/
/*                                                                          */
/* Routine to send a character string off to the modem.                     */
/*                                                                          */
/*--------------------------------------------------------------------------*/

send:

   parse arg sendstring
   call slip_com_output interface , sendstring

   return


/*--------------------------------------------------------------------------*/
/*                    waitfor ( waitstring , [timeout] )                    */
/*..........................................................................*/
/*                                                                          */
/* Waits for the supplied string to show up in the COM input.  All input    */
/* from the time this function is called until the string shows up in the   */
/* input is accumulated in the "waitfor_buffer" variable.                   */
/*                                                                          */
/* If timeout is specified, it says how long to wait if data stops showing  */
/* up on the COM port (in seconds).                                                         */
/*                                                                          */
/*--------------------------------------------------------------------------*/

waitfor:

   parse arg waitstring , timeout

   if timeout = '' then
     timeout = 5000    /* L O N G   delay if not specified */
   waitfor_buffer = '' ; done = -1; curpos = 1
   ORI_TIME=TIME('E')

   if (remain_buffer = 'REMAIN_BUFFER') then do
      remain_buffer = ''
   end

   do while (done = -1)
      if (remain_buffer \= '') then do
         line = remain_buffer
         remain_buffer = ''
       end
       else do
         line = slip_com_input(interface,,10)
      end
      waitfor_buffer = waitfor_buffer || line
      index = pos(waitstring,waitfor_buffer)
      if (index > 0) then do
         remain_buffer = substr(waitfor_buffer,index+length(waitstring))
         waitfor_buffer = delstr(waitfor_buffer,index+length(waitstring))
         done = 0
      end
      call charout , substr(waitfor_buffer,curpos)
      curpos = length(waitfor_buffer)+1
      if ((done \= 0) & (TIME('E')>timeout)) then do
        call lineout , ' WAITFOR: timed out '
        done = 1
       end
   end
   timeout=0
   RC=done
 return RC



/*--------------------------------------------------------------------------*/
/*                               readpass ()                                */
/*..........................................................................*/
/*                                                                          */
/* Routine used to read a password from the user without echoing the        */
/* password to the screen.                                                  */
/*                                                                          */
/*--------------------------------------------------------------------------*/

readpass:

  answer = ''
  do until key = cr
    key = slip_getch()
    if key \= cr then do
      answer = answer || key
    end
  end
  say ''
  return answer


/*--------------------------------------------------------------------------*/
/*                             flush_receive ()                             */
/*..........................................................................*/
/*                                                                          */
/* Routine to flush any pending characters to be read from the COM port.    */
/* Reads everything it can until nothing new shows up for 100ms, at which   */
/* point it returns.                                                        */
/*                                                                          */
/* The optional echo argument, if 1, says to echo flushed information.      */
/*                                                                          */
/*--------------------------------------------------------------------------*/

flush_receive:

   parse arg echo

   /* If echoing the flush - take care of waitfor remaining buffer */
   if (echo \= '') & (length(remain_buffer) > 0) then do
      call charout , remain_buffer
      remain_buffer = ''
   end

   /* Eat anything left in the modem or COM buffers */
   /* Stop when nothing new appears for 100ms.      */

   do until line = ''
     line = slip_com_input(interface,,100)
     if echo \= '' then
        call charout , line
   end

   return

/*--------------------------------------------------------------------------*/
/*                          display_parameters()                            */
/*--------------------------------------------------------------------------*/

display_parameters: procedure

  parse arg local, remote

  time = format(time(elapsed), 4, 2)

  msg.0 = 5
  msg.1 = '   SLIP connection successfully completed'
  msg.2 = '   in' time 'seconds...'
  msg.3 = ' '
  msg.4 = '   Local IP address  =' local
  msg.5 = '   Remote IP address =' remote

  do out = 1 to msg.0
    call lineout , msg.out
  end out
 
return 
