/*
 * Copyright (c) 2002 by Dmitry A.Steklenev
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <stdio.h>
#include <stdlib.h>
#include <getopt.h>
#include <direct.h>
#include <dirent.h>
#include <errno.h>
#include <string.h>

static void usage( void )
{
  printf( "Usage: [OPTION]... FILE...\n" );
  printf( "Remove (unlink) the FILE(s).\n\n"
          "  -d  ignored for compatibility\n"
          "  -f  ignored for compatibility\n"
          "  -i  ignored for compatibility\n"
          "  -v  explain what is being done\n"
          "  -r  remove the contents of directories recursively\n" );

  exit( EXIT_SUCCESS );
}

static int delete( char* path, int recurse, int verbose )
{
  struct stat st;
  struct dirent *den;
  DIR *dir;
  char file[_MAX_PATH];
  int rc;

  if( stat( path, &st ) == -1 )
  {
    fprintf( stderr, "SARM: File not found: '%s'\n", path );
    return 1;
  }

  if( st.st_mode & S_IFREG )
  {
    if( verbose )
      fprintf( stdout, "SARM: Delete file '%s'\n", path );
    if( unlink( path ) == -1 )
    {
      fprintf( stderr, "SARM: Unable delete file '%s'\n", path );
      return 1;
    }
  }

  if( st.st_mode & S_IFDIR )
  {
    if( recurse )
    {
      rc = 0;

      if(( dir = opendir( path )) != 0 )
      {
        while(( den = readdir( dir )) != 0 )
          if( strcmp( den->d_name, "."  ) != 0 && strcmp( den->d_name, ".." ) != 0 )
          {
            sprintf( file, "%s\\%s", path, den->d_name );
            rc += delete( file, recurse, verbose );
          }

        closedir( dir );
      }
    }

    if( verbose )
      fprintf( stdout, "SARM: Delete directory '%s'\n", path );
    if( rmdir( path ) == -1 )
    {
      fprintf( stderr, "SARM: Unable delete directory '%s'\n", path );
      return 1;
    }
  }  

  return rc ? 1 : 0;
}

int main( int argc, char **argv )
{
  int recurse = 0;
  int verbose = 0;
  int i, opt;

  while((opt = getopt( argc, argv, "dfivrR")) != -1 )
  {
    switch(opt)
    {
      case 'd':
      case 'f':
      case 'i':
        break;

      case 'v':
        verbose = 1;
        break;

      case 'r':
      case 'R':
        recurse = 1;
        break;

      default:
        usage();
    }
  }

  if( optind == argc )
    usage();

  for( i = optind; i < argc; i++ )
  {
    char *path = strdup( argv[i] ), *p;

    for( p = path; *p; p++ )
      if( *p == '/' ) *p = '\\';

    printf( "SARM: Remove: %s\n", path );
    delete( path, recurse, verbose );
    free  ( path );
  }

  return( EXIT_SUCCESS );
}
