/*--------------------------------------------------------------------------*/
/* This script translates Post Road Mailer Address Books into MR/2 ICE      */
/* addresses and groups.                                                    */
/*                                                                          */
/* So that one can understand what I did to perform the conversion, I'll    */
/* explain a few things as I understand them...                             */
/*                                                                          */
/* The Post Road Mailer (tm) holds both e-mail addresses and groups in      */
/* address books.  Each address book is a separate file with the extension  */
/* ADR.  All of the files are contained in the Post Road Mailer directory.  */
/* Each address book entry can contain quite a bit of information:          */
/* First name, last name, organization, up to 5 e-mail addresses, two       */
/* telephone numbers, a fax number, and a note.                             */
/*                                                                          */
/* The biggest hurdle in creating this converter was the binary format      */
/* of the Post Road Mailer address books.  Needless to say, a hex-editor    */
/* came in very, very useful...                                             */
/*                                                                          */
/*                     *      *      *                                      */
/*                                                                          */
/* MR/2 ICE, on the other hand, holds its e-mail addresses in a single file */
/* with the extension ADR.  You can define other files to be used but that  */
/* doesn't concern this conversion.  In addition, MR/2 ICE holds its group  */
/* definitions in a separate file with the extension GRP.  Both of these    */
/* files are in the MR/2 ICE directory.  And, thankfully, both of these     */
/* files are basically ASCII files.                                         */
/*                                                                          */
/* Each address entry contains a full name, e-mail address, alias and a     */
/* note.                                                                    */
/*                                                                          */
/*                     *      *      *                                      */
/*                                                                          */
/* Now, how I decided to do this conversion.  For e-mail address entries:   */
/*                                                                          */
/* 1).  The first name and last name are combined and used.                 */
/* 2).  Each e-mail address for a given PRM address book entry is given     */
/*      a separate entry in the MR/2 address file.                          */
/* 3).  The alias used for each e-mail address is either the nickname in    */
/*      the PRM address file entry for that e-mail address, or, if a nick-  */
/*      name doesn't exist.  The last name is used.  If a duplicate alias   */
/*      exists, I append a number to new alias.                             */
/* 4).  Finally, the note for the MR/2 ICE address consists of the PRM      */
/*      note, organization, telephone numbers and fax number.  This note    */
/*      is attached to each e-mail address entry.                           */
/*                                                                          */
/* For groups:                                                              */
/*                                                                          */
/* 1).  Every group from each PRM address book is placed in the MR/2 ICE    */
/*      group file.  Groups aliases are checked for duplication.            */
/* 2).  In addition, for each PRM address book, I allow the user to         */
/*      optionally create a MR/2 ICE group consisting of all of the         */
/*      addresses in that address book.  The alias for that group is the    */
/*      file name minus the extension of that address book.                 */
/*                                                                          */
/*      An example would probably be helpful.  If you have two PRM address  */
/*      books, each containing 5 e-mail addresses and 2 groups.  Say the    */
/*      first file is called smeg.adr (with the description "friends") and  */
/*      the second is called head.adr (with the description "enemas").      */
/*                                                                          */
/*      This converter would add all 10 e-mail addresses to the MR/2 ICE    */
/*      address list.  It would add all 4 groups to the MR/2 ICE group      */
/*      file.  In addition, if the user says 'yes' to the add address books */
/*      as groups question, it would add a group with the alias 'smeg'      */
/*      containing the 5 e-mail addresses and it would add a group with the */
/*      alias 'head' with the other 5 e-mail addresses                      */
/*                                                                          */
/*      If you don't like these additional groups, then it's easy enough to */
/*      to any one you don't want.                                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/*  Load the Rexx Functions if they're not already loaded                   */
/*--------------------------------------------------------------------------*/
IF RxFuncQuery('SysLoadFuncs') THEN DO
    /* load the load-function */
    CALL RxFuncAdd 'SysLoadFuncs', 'RexxUtil', 'SysLoadFuncs'

    /* load the Sys* utilities */
    CALL SysLoadFuncs
END

/*--------------------------------------------------------------------------*/
/*  Initialize names, etc.                                                  */

hex01 = '01'x
crlf = '0D'x'0A'x

books_to_groups = 'Y'

description = ''

names.0 = 0
names.0.firstname = ''
names.0.lastname = ''
names.0.nick.1 = ''
names.0.nick.2 = ''
names.0.nick.3 = ''
names.0.nick.4 = ''
names.0.nick.5 = ''
names.0.firstname = ''
names.0.organization = ''
names.0.email.0 = 0
names.0.email.1 = ''
names.0.email.2 = ''
names.0.email.3 = ''
names.0.email.4 = ''
names.0.email.5 = ''
names.0.phone1 = ''
names.0.phone2 = ''
names.0.fax = ''
names.0.notes = ''

groups.0.0 = 0
groups.0.desc = ''
groups.0.1 = ''

mr2namelist.0 = 0
mr2grplist.0 = 0

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
/*  Check prm_location for *.ADR files  (PRM address files)                 */

SAY "Please Enter the path to the Post Road Mailer Directory"
PARSE UPPER PULL prm_location

IF SUBSTR(prm_location,LENGTH(prm_location),1) = '\' THEN
  prm_location = SUBSTR(prm_location,1,LENGTH(prm_location)-1)
tempstring = prm_location||"\*.ADR"
rcode = SysFileTree(tempstring,prmfiles,"FO")
IF prmfiles.0 = 0 THEN DO
  SAY "No Address files found in "prm_location
  SAY "Please restart PRM2MR2 and input the correct PRM location"
  EXIT 1
END
ELSE DO
  SAY "In '"prm_location"' the following address files were found:"
  DO ii = 1 TO prmfiles.0
    SAY "Found "prmfiles.ii
  END
  SAY ""
END
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
/*  Check mr2_location for MR2I.ADR files  (MR2 basic address file)         */

SAY "Please Enter the path to MR/2 ICE Directory"
PARSE UPPER PULL mr2_location

IF SUBSTR(mr2_location,LENGTH(mr2_location),1) = '\' THEN
  mr2_location = SUBSTR(mr2_location,1,LENGTH(mr2_location)-1)

SAY "Checking '"mr2_location"' for mr2i.adr file"
mr2addfile = mr2_location||"\MR2I.ADR"
rcode = STREAM(mr2addfile,'C','QUERY EXISTS')
IF rcode = "" THEN DO
  SAY "No MR/2 Address file found in "mr2_location
  SAY "Will Create a new one"
  rcode = STREAM(mr2addfile,'C','OPEN WRITE')
  SAY ""
END
ELSE DO
  SAY "Updating "rcode
  rcode = STREAM(mr2addfile,'C','OPEN')
  rcode = STREAM(mr2addfile,'C','SEEK < 2')
  last2 = CHARIN(mr2addfile,,2)
  IF last2 \= crlf THEN DO
    rcode = STREAM(mr2addfile,'C','QUERY SIZE')
    IF rcode \= 0 THEN rcode = CHAROUT(mr2addfile,crlf,2)
  END
  rcode = GetMR2NameList(mr2addfile, mr2namelist)
  SAY ""
END
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
/*  Check mr2_location for MR2I.GRP files  (MR2 basic address file)         */

SAY "checking for mr2i.grp file"
mr2grpfile = mr2_location||"\MR2I.GRP"
rcode = STREAM(mr2grpfile,'C','QUERY EXISTS')
IF rcode = "" THEN DO
  SAY "No MR/2 Group file found in "mr2_location
  SAY "Will Create a new one"
  rcode = STREAM(mr2grpfile,'C','OPEN WRITE')
  SAY ""
END
ELSE DO
  SAY "Updating "rcode
  rcode = STREAM(mr2grpfile,'C','OPEN')
  rcode = STREAM(mr2grpfile,'C','SEEK < 2')
  last2 = CHARIN(mr2grpfile,,2)
  IF last2 \= crlf THEN DO
    rcode = STREAM(mr2grpfile,'C','QUERY SIZE')
    IF rcode \= 0 THEN rcode = CHAROUT(mr2grpfile,crlf,2)
  END
  rcode = GetMR2GroupList(mr2grpfile, mr2grplist)
  SAY ""
END
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
/*  Ask about adding address books as groups                                */
DO UNTIL(books_to_groups = 'Y' | books_to_groups = 'N')
  SAY "Do you want to make a group for each PRM address book? (Y/n)"
  PARSE UPPER PULL books_to_groups
  IF (books_to_groups \= 'Y' & books_to_groups \= 'N') THEN
    SAY "Please answer Y or N only"
END
SAY ""
SAY "And off we go..."
SAY ""
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
/*  Start processing input files                                            */

DO ii = 1 TO prmfiles.0
  rcode = STREAM(prmfiles.ii,'C','OPEN READ')
  rcode = Read_PRMAddress_Book(prmfiles.ii,description,names,groups)
  rcode = STREAM(prmfiles.ii,'C','CLOSE')

  IF names.0 \= 0 THEN DO
    SAY "Updating MR/2 Address File with addresses from"
    SAY description
    rcode = Update_Address_Book(mr2addfile,names)
  END
  SAY description " addresses added"
  SAY ""
  SAY "Updating MR/2 Group File"
  rcode = Update_Group_Book(prmfiles.ii,mr2grpfile,description,names,groups)
  SAY description " groups added"
  SAY ""

END

EXIT 0
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
/*  Reads a MR2 address file to get shortnames                              */
GetMR2NameList:
  PARSE ARG mr2addfile, mr2namelist

  mr2namelist.0 = 0
  rcode = STREAM(mr2addfile,'C','CLOSE')
  rcode = STREAM(mr2addfile,'C','OPEN READ')
  DO WHILE(LINES(mr2addfile) \= 0)
    inline = LINEIN(mr2addfile)
    IF SUBSTR(inline,1,1) = hex01 THEN DO
      tempstring = SUBSTR(inline,2,LENGTH(inline)-1)
      comment_length = 0
      PARSE VAR tempstring comment_length
      comment = CHARIN(mr2addfile,,comment_length+2)
    END
    ELSE DO
      pos1 = POS('\',inline)
      pos2 = POS('\',inline,pos1+1)
      pos3 = POS('\',inline,pos2+1)
      IF pos3 \= 0 THEN DO
        mr2namelist.0 = mr2namelist.0 + 1
        index = mr2namelist.0
        mr2namelist.index = STRIP(SUBSTR(inline,pos3+1,LENGTH(inline)-pos3))
      END
    END
  END
  rcode = STREAM(mr2addfile,'C','CLOSE')
  rcode = STREAM(mr2addfile,'C','OPEN WRITE')
  rcode = STREAM(mr2addfile,'C','SEEK < 0')

RETURN 0
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
/*  Reads a MR2 group file to get aliases                                   */
GetMR2GroupList:
  PARSE ARG mr2grpfile, mr2grplist

  mr2grplist.0 = 0
  rcode = STREAM(mr2grpfile,'C','CLOSE')
  rcode = STREAM(mr2grpfile,'C','OPEN READ')
  DO WHILE(LINES(mr2grpfile) \= 0)
    inline = LINEIN(mr2grpfile)
    IF SUBSTR(inline,1,1) = '!' THEN DO
      mr2grplist.0 = mr2grplist.0 + 1
      index = mr2grplist.0
      mr2grplist.index = SUBSTR(inline,2,POS('\',inline)-2)
    END
  END
  rcode = STREAM(mr2grpfile,'C','CLOSE')
  rcode = STREAM(mr2grpfile,'C','OPEN WRITE')
  rcode = STREAM(mr2grpfile,'C','SEEK < 0')
RETURN 0
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
/*  Reads a PRM address book                                                */
Read_PRMAddress_Book:
  PARSE ARG infile, description, names, groups

  shortvalue = Read_Short(infile)
  description = Read_String(infile)
  SAY "Starting with Address Book:  "description
  shortvalue = Read_Short(infile)
  names.0 = Read_Short(infile)
  SAY "Number of Address Entries:  "names.0
  shortvalue = Read_Short(infile)
  shortvalue = Read_Short(infile)
  groups.0 = Read_Short(infile)
  SAY "Number of Groups:  "groups.0
  shortvalue = Read_Short(infile)

  DO jj = 1 to names.0
    names.jj.lastname = Read_String(infile)
    DO kk = 1 TO 5
      names.jj.nick.kk = Read_String(infile)
      IF names.jj.nick.kk = '' THEN names.jj.nick.kk = names.jj.lastname
    END
    names.jj.firstname = Read_String(infile)
    names.jj.organization = Read_String(infile)
    DO kk = 1 TO 5
      names.jj.email.kk = Read_String(infile)
    END
    names.jj.phone1 = Read_String(infile)
    names.jj.phone2 = Read_String(infile)
    names.jj.fax = Read_String(infile)
    names.jj.notes = Read_String(infile)

    names.jj.email.0 = 0
    tempindex = 0
    DO kk = 1 TO 5
      IF names.jj.email.kk \= '' THEN DO
        tempindex = tempindex + 1
        names.jj.email.tempindex = names.jj.email.kk
        names.jj.nick.tempindex = names.jj.nick.kk
      END
    END

    names.jj.email.0 = tempindex

    IF names.jj.phone1 = '' THEN DO
      IF names.jj.phone2 \= '' THEN DO
        names.jj.phone1 = names.jj.phone2
        names.jj.phone2 = ''
      END
    END

    rcode = CHAROUT(stdout,'.')
  END
  SAY ""

  IF groups.0 \= 0 THEN DO

    DO jj = 1 TO groups.0
      groups.jj.desc = Read_String(infile)
      SAY "Group "jj" description:  "groups.jj.desc
      shortvalue = Read_Short(infile)
      groups.jj.0 = Read_Short(infile)
      SAY "Group "jj" number:  "groups.jj.0
      shortvalue = Read_Short(infile)
      DO kk = 1 TO groups.jj.0
        groups.jj.kk = Read_String(infile)
      END
    END
    rcode = CHAROUT(stdout,'.')
  END

  SAY ""
  SAY "Reading "description" address book complete"
  SAY ""
RETURN 0
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
/*  Reads a string from the input PRM address file                          */
Read_String:
  PARSE ARG infile
  stringlength = Read_Short(infile)
  IF stringlength = 0 THEN readstring = ''
  ELSE readstring = CHARIN(infile,,stringlength)
  shortvalue = Read_Short(infile)
RETURN readstring
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
/*  Reads a short from the input PRM address file                           */
Read_Short:
  PARSE ARG infile
  tempstring = CHARIN(infile,,4)
  short_value = C2D(SUBSTR(tempstring,1,1))
  short_value = short_value + C2D(SUBSTR(tempstring,2,1)) * 16
  short_value = short_value + C2D(SUBSTR(tempstring,3,1)) * 256
  short_value = short_value + C2D(SUBSTR(tempstring,4,1)) * 4096
RETURN short_value
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
/*  Updates the MR2 address book                                            */
Update_Address_Book:
  PARSE ARG outfile, names

  DO jj = 1 TO names.0
    IF names.jj.email.0 \= 0 THEN DO
                                  /*----------------------------------------*/
                                  /* set up the full name                   */
                                  /*----------------------------------------*/
      fullname = names.jj.firstname||" "||names.jj.lastname
                                  /*----------------------------------------*/
                                  /* set up phone numbers                   */
                                  /*----------------------------------------*/
      IF names.jj.phone1 \= '' THEN DO
        phone = " Telephone:  "||names.jj.phone1
        IF names.jj.phone2 \= '' THEN phone = phone||" & "||names.jj.phone2
      END
      ELSE phone = ''
                                  /*----------------------------------------*/
                                  /* set up fax numbers                     */
                                  /*----------------------------------------*/
      IF names.jj.fax \= '' THEN DO
        faxnum = " Fax:  "||names.jj.fax
      END
      ELSE faxnum = ''
                                  /*----------------------------------------*/
                                  /* put together the note                  */
                                  /*----------------------------------------*/
      allnotes = names.jj.notes||" "||phone||" "||faxnum
                                  /*----------------------------------------*/
                                  /* do each email address                  */
                                  /*----------------------------------------*/
      DO kk = 1 TO names.jj.email.0
        IF names.jj.email.kk \= '' THEN DO
          firstline = fullname||"\"||names.jj.email.kk||"\N\"
          IF names.jj.nick.kk \= '' THEN DO
            nickname = names.jj.nick.kk
            rcode = CheckNickName(nickname)
            firstline = firstline||nickname
          END
          ELSE DO
            nickname = names.jj.lastname
            rcode = CheckNickName(nickname)
            firstline = firstline||nickname
          END
          names.jj.nick.kk = nickname

          secondline = hex01||LENGTH(allnotes)
                                  /*----------------------------------------*/
                                  /* write entry                            */
                                  /*----------------------------------------*/
          rcode = LINEOUT(outfile,firstline)
          rcode = LINEOUT(outfile,secondline)
          rcode = LINEOUT(outfile,allnotes)
        END
      END
    END
    rcode = CHAROUT(stdout,'+')
  END
  SAY ""

RETURN 0
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
/*  Updates the MR2 groups book                                             */
Update_Group_Book:
  PARSE ARG infile, outfile, description, names, groups

  IF names.0 \= 0 & books_to_groups = 'Y'  THEN DO
    infilename = FILESPEC("NAME",infile)
    dotpos = POS('.',infilename)
    IF dotpos > 9 THEN dotpos = 9
    alias = SUBSTR(infilename,1,dotpos-1)
    rcode = CheckGroupAlias(alias)
    firstline = "!"||alias||"\"||description||"\N"
    secondline = "#"||LENGTH(description)
    thirdline = description
    rcode = LINEOUT(outfile,firstline)
    rcode = LINEOUT(outfile,secondline)
    rcode = LINEOUT(outfile,thirdline)
    DO jj = 1 TO names.0
      IF names.jj.email.0 \= 0 THEN DO
        DO kk = 1 TO names.jj.email.0
          outline = "+"||names.jj.nick.kk
          rcode = LINEOUT(outfile,outline)
        END
      END
    END
    rcode = CHAROUT(stdout,'+')
  END

  IF groups.0 \= 0 THEN DO
    DO jj = 1 TO groups.0

      alias = STRIP(SUBSTR(groups.jj.desc,1,8))
      rcode = CheckGroupAlias(alias)
      firstline = "!"||alias||"\"||groups.jj.desc||"\N"
      secondline = "#"||LENGTH(groups.jj.desc)
      thirdline = groups.jj.desc
      rcode = LINEOUT(outfile,firstline)
      rcode = LINEOUT(outfile,secondline)
      rcode = LINEOUT(outfile,thirdline)

      DO kk = 1 TO groups.jj.0
        temp = groups.jj.kk
        pos1 = POS("<",temp,1) + 1
        pos2 = POS(">",temp,pos1+1)
        email = SUBSTR(temp, pos1, (pos2 - pos1))
        outline = "+"||email
        rcode = LINEOUT(outfile,outline)
      END
      rcode = CHAROUT(stdout,'+')
    END
  END
  SAY ""

RETURN 0
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
CheckNickName:
  PARSE ARG nickname0

  found = 0
  DO ll = 1 TO mr2namelist.0
    IF nickname0 = mr2namelist.ll THEN DO
      ll = mr2namelist.0 + 1
      found = 1
    END
  END
  IF found = 1 THEN DO
    count = 0
    DO WHILE(found = 1)
      found = 0
      count = count + 1
      nickname = nickname0||count
      DO ll = 1 TO mr2namelist.0
        IF nickname = mr2namelist.ll THEN DO
          ll = mr2namelist.0 + 1
          found = 1
        END
      END
    END
    nickname0 = nickname
  END

  mr2namelist.0 = mr2namelist.0 + 1
  index = mr2namelist.0
  mr2namelist.index = nickname0

RETURN 0
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
CheckGroupAlias:
  PARSE ARG groupalias0

  found = 0
  DO ll = 1 TO mr2grplist.0
    IF groupalias0 = mr2grplist.ll THEN DO
      ll = mr2grplist.0 + 1
      found = 1
    END
  END
  IF found = 1 THEN DO
    count = 0
    DO WHILE(found = 1)
      found = 0
      count = count + 1
      groupalias = groupalias0||count
      DO ll = 1 TO mr2grplist.0
        IF groupalias = mr2grplist.ll THEN DO
          ll = mr2grplist.0 + 1
          found = 1
        END
      END
    END
    groupalias0 = groupalias
  END

  mr2grplist.0 = mr2grplist.0 + 1
  index = mr2grplist.0
  mr2grplist.index = groupalias0

RETURN 0
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
