/* routing decisions
 * Copyright (C) 1999  Cendio Systems AB.
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.  See <http://www.fsf.org/copyleft/gpl.txt>.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * RCSID $Id$
 */

#include "os2port.h"
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include <freeswan.h>

#include "constants.h"
#include "defs.h"
#include "log.h"

#include "route.h"


static struct routeinfo *routingtable = NULL;

const char *routing_table_file = ROUTING_TABLE_FILE;


static ssize_t
pluto_getline(char **lineptr, size_t *n, FILE *fp)
{
    ssize_t len = 0;
    int c;

    while ((c = getc(fp)) != EOF)
    {
	if (len+1 >= (ssize_t)*n)	/* +1 to allow for terminating nul */
	{
	    size_t newsize = len * 1.3 + 20;
	    char *tmp = realloc(*lineptr, newsize);
	    if (tmp == NULL)
		return -1;
	    *lineptr = tmp;
	    *n = newsize;
	}
	(*lineptr)[len++] = c;
	if (c == '\n')
	    break;
    }
    if (c == EOF  &&  len == 0)
	return EOF;
    (*lineptr)[len] = '\0';
    return len;
}


static struct routeinfo *
do_read_routing_table(const char *proc_net_route)
{
    FILE		* rf;
    char		* line		= NULL;
    size_t		  linelen	= 0;
    struct routeinfo	* routes	= NULL;
    ssize_t		  nchars;

    rf = fopen(proc_net_route, "r");
    if (rf == NULL)
    {
	log_errno((e, "can't open routing file \"%s\"", proc_net_route));
	return NULL;
    }
    nchars = pluto_getline(&line, &linelen, rf); /* Skip header line */

    while (nchars > 0  &&  (nchars = pluto_getline(&line, &linelen, rf)) > 0)
    {
	int dummy;
	struct routeinfo *route = alloc_thing(struct routeinfo, "route");
	sscanf(line, "%s %x %x %i %i %i %i %x",
	       route->iface,
	       &route->dest.s_addr,
	       &route->gateway.s_addr,
	       &dummy,
	       &dummy,
	       &dummy,
	       &dummy,
	       &route->netmask.s_addr);
	route->mask_nbits = masktobits(route->netmask);
	route->next = routes;
	routes = route;
    }

    free(line);
    fclose(rf);
    return routes;
}


static void
free_routes(struct routeinfo *routes)
{
    while (routes != NULL)
    {
	struct routeinfo *rtmp = routes->next;
	pfree(routes);
	routes = rtmp;
    }
}


bool
reread_routing_table(const char *proc_net_route)
{
    struct routeinfo *routes;

    free_routes(routingtable);
    routes = do_read_routing_table(proc_net_route);
    routingtable = routes;
    return TRUE;
}


struct routeinfo *
find_route(struct in_addr dest)
{
    struct routeinfo *rptr = routingtable;
    struct routeinfo *best = NULL;

    while (rptr != NULL)
    {
	struct in_addr net = subnetof(dest, rptr->netmask);
	if (rptr->dest.s_addr == net.s_addr)
	    if (best == NULL || rptr->mask_nbits > best->mask_nbits)
		best = rptr;
	rptr = rptr->next;
    }
    return best;
}


#ifdef TEST_ROUTE_C


int
main(int argc, char **argv)
{
    struct in_addr a;
    struct routeinfo *ri;

    if (argc != 3) {
	fprintf(stderr, "Usage: %s routefile IP-address\n", argv[0]);
	exit(64);
    }

    atoaddr(argv[2], 0, &a);
    reread_routing_table(argv[1]);
    ri = find_route(a);
    if (!ri)
	printf("No route to host!\n");
    else
    {
	printf("dest:      %s\n", inet_ntoa(ri->dest));
	printf("netmask:   %s\n", inet_ntoa(ri->netmask));
	printf("gateway:   %s\n", inet_ntoa(ri->gateway));
	printf("interface: %s\n", ri->iface);
    }

    return 0;
}

void exit_pluto(int x)
{
    exit(x);
}
#endif	/* TEST */
