/* IPsec DOI and Oakley resolution routines
 * Copyright (C) 1997 Angelos D. Keromytis.
 * Copyright (C) 1998, 1999  D. Hugh Redelmeier.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.  See <http://www.fsf.org/copyleft/gpl.txt>.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * RCSID $Id: ipsec_doi.c,v 1.77 1999/10/22 00:28:14 dhr Exp $
 */
/*
 * changed by dev@fx.dk 24/07/1999 (port to OS/2)
 * changed by dev@fx.dk 09/10/1999
 * changed by dev@fx.dk 29/10/1999 (new snapshot)
 * changed by dev@fx.dk 18/11/1999 (adding aggressive mode)
 */
#include "os2port.h"


#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include <freeswan.h>

#include "constants.h"
#include "defs.h"
#include "state.h"
#include "connections.h"
#include "preshared.h"
#include "packet.h"
#include "demux.h"  /* needs packet.h */
#include "kernel.h"
#include "log.h"
#include "cookie.h"
#include "server.h"
#include "spdb.h"
#include "timer.h"
#include "rnd.h"
/* aggr-patch */
#include "eventlog.h"
//
#include "ipsec_doi.h"  /* needs demux.h and state.h */
#include "whack.h"

#include "sha1.h"
#include "md5.h"
#include "crypto.h" /* requires sha1.h and md5.h */

#include "ipsec_doi-aggr1.inc"

/* pad_up(n, m) is the amount to add to n to make it a multiple of m */
#define pad_up(n, m) (((m) - 1) - (((n) + (m) - 1) % (m)))

/* var := mod(base ** exp, mod), ensuring var is mpz_inited */
#define mpz_init_powm(flag, var, base, exp, mod) { \
    if (!(flag)) \
  mpz_init(&(var)); \
    (flag) = TRUE; \
    mpz_powm(&(var), &(base), &(exp), (mod)); \
    }


/* Convert MP_INT to network form (binary octets, big-endian).
 * We do the malloc; caller must eventually do free.
 */
static chunk_t
mpz_to_n(const MP_INT *mp, size_t bytes)
{
    chunk_t r;
    MP_INT temp1, temp2;
    int i;

    r.len = bytes;
    r.ptr = alloc_bytes(r.len, "host representation of large integer");

    mpz_init(&temp1);
    mpz_init(&temp2);

    mpz_set(&temp1, mp);

    for (i = r.len-1; i >= 0; i--)
    {
      r.ptr[i] = mpz_mdivmod_ui(&temp2, NULL, &temp1, 1 << BITS_PER_BYTE);
      mpz_set(&temp1, &temp2);
    }

    passert(mpz_sgn(&temp1) == 0);  /* we must have done all the bits */
    mpz_clear(&temp1);
    mpz_clear(&temp2);

    return r;
}

/* Convert network form (binary bytes, big-endian) to MP_INT.
 * The *mp must not be previously mpz_inited.
 */
static void
n_to_mpz(MP_INT *mp, const u_char *nbytes, size_t nlen)
{
    size_t i;

    mpz_init_set_ui(mp, 0);

    for (i = 0; i != nlen; i++)
    {
      mpz_mul_ui(mp, mp, 1 << BITS_PER_BYTE);
      mpz_add_ui(mp, mp, nbytes[i]);
    }
}

/* if we haven't already done so, compute a local DH secret (st->st_sec) and
 * the corresponding public value (g).  This is emitted as a KE payload.
 */
static bool
build_and_ship_KE(struct state *st, chunk_t *g
    , const struct oakley_group_desc *group, pb_stream *outs, u_int8_t np)
{
    if (!st->st_sec_in_use)
    {
      u_char tmp[LOCALSECRETSIZE];
      MP_INT mp_g;

      get_rnd_bytes(tmp, LOCALSECRETSIZE);
      st->st_sec_in_use = TRUE;
      n_to_mpz(&st->st_sec, tmp, LOCALSECRETSIZE);

      mpz_init(&mp_g);
      mpz_powm(&mp_g, &groupgenerator, &st->st_sec, group->modulus);
      *g = mpz_to_n(&mp_g, group->bytes);
      mpz_clear(&mp_g);
#ifdef DODGE_DH_MISSING_ZERO_BUG
      if (g->ptr[0] == 0)
      {
        /* generate a new secret to avoid this situation */
        log("regenerating DH private secret to avoid Pluto 1.0 bug"
            " handling public value with leading zero");
        mpz_clear(&st->st_sec);
        st->st_sec_in_use = FALSE;
        freeanychunk(*g);
        return build_and_ship_KE(st, g, group, outs, np);
      }
#endif

      DBG(DBG_CRYPT,
      DBG_dump("Local DH secret:\n", tmp, LOCALSECRETSIZE);
      DBG_dump_chunk("Public DH value sent:\n", *g));
    }
    return out_generic_chunk(np, &isakmp_keyex_desc, outs, *g, "keyex value");
}

/* Compute DH shared secret from our local secret and the peer's public value.
 * We make the leap that the length should be that of the group
 * (see quoted passage at start of ACCEPT_KE).
 */
static void
compute_dh_shared(struct state *st, const chunk_t g
    , const struct oakley_group_desc *group)
{
    MP_INT mp_g, mp_shared;

    passert(st->st_sec_in_use);
    n_to_mpz(&mp_g, g.ptr, g.len);
    mpz_init(&mp_shared);
    mpz_powm(&mp_shared, &mp_g, &st->st_sec, group->modulus);
    mpz_clear(&mp_g);
    st->st_shared = mpz_to_n(&mp_shared, group->bytes);
    mpz_clear(&mp_shared);
#ifdef DODGE_DH_MISSING_ZERO_BUG
    if (st->st_shared.ptr[0] == 0)
      log("shared DH secret has leading zero -- triggers Pluto 1.0 bug");
#endif
    DBG_cond_dump_chunk(DBG_CRYPT, "DH shared secret:\n", st->st_shared);
}

/* ACCEPT_KE
 *
 * Check and accept DH public value (Gi or Gr) from peer's message.
 * According to RFC2409 "The Internet key exchange (IKE)" 5:
 *  The Diffie-Hellman public value passed in a KE payload, in either
 *  a phase 1 or phase 2 exchange, MUST be the length of the negotiated
 *  Diffie-Hellman group enforced, if necessary, by pre-pending the
 *  value with zeros.
 * ??? For now, we accept shorter values to interoperate with old Plutos.
 * This should change some day.
 *
 * This macro is magic -- it cannot be expressed as a function.
 * - it causes the caller to return!
 */
#ifdef DODGE_DH_MISSING_ZERO_BUG
# define ACCEPT_KE(dest, val_name, gr, pbs) { \
    if (pbs_left(&(pbs)) != (gr)->bytes) \
    { \
  log("KE has %u byte DH public value; %u required" \
      , (unsigned) pbs_left(&(pbs)), (unsigned) (gr)->bytes); \
  /* XXX Could send notification back */ \
  if (pbs_left(&(pbs)) > (gr)->bytes) \
      return STF_FAIL + INVALID_KEY_INFORMATION; \
    } \
    clonereplacechunk((dest), (pbs).cur, pbs_left(&(pbs)), val_name); \
    DBG_cond_dump_chunk(DBG_CRYPT, "DH public value received:\n", (dest)); \
}
#else
# define ACCEPT_KE(dest, val_name, gr, pbs) { \
    if (pbs_left(&(pbs)) != (gr)->bytes) \
    { \
  log("KE has %u byte DH public value; %u required" \
      , (unsigned) pbs_left(&(pbs)), (unsigned) (gr)->bytes); \
  /* XXX Could send notification back */ \
  return STF_FAIL + INVALID_KEY_INFORMATION; \
    } \
    clonereplacechunk((dest), (pbs).cur, pbs_left(&(pbs)), val_name); \
    DBG_cond_dump_chunk(DBG_CRYPT, "DH public value received:\n", (dest)); \
}
#endif

/* ACCEPT_PFS_KE
 *
 * Check and accept optional Quick Mode KE payload for PFS.
 * Extends ACCEPT_PFS to check whether KE is allowed or required.
 *
 * This macro is magic -- it cannot be expressed as a function.
 * - it causes the caller to return!
 * - it uses variables local to the caller (md, st)
 */
#define ACCEPT_PFS_KE(dest, val_name, msg_name) { \
    struct payload_digest *const ke_pd = md->chain[ISAKMP_NEXT_KE]; \
    if (ke_pd == NULL) \
    { \
  if (st->st_pfs_group != NULL) \
  { \
      log("missing KE payload in " msg_name " message"); \
      return STF_FAIL + INVALID_KEY_INFORMATION; \
  } \
    } \
    else \
    { \
  if (st->st_pfs_group == NULL) \
  { \
      log(msg_name " message KE payload requires a GROUP_DESCRIPTION attribute in SA"); \
      return STF_FAIL + INVALID_KEY_INFORMATION; \
  } \
  if (ke_pd->next != NULL) \
  { \
      log(msg_name " message contains several KE payloads; we accept at most one"); \
      return STF_FAIL + INVALID_KEY_INFORMATION;  /* ??? */ \
  } \
  ACCEPT_KE(dest, val_name, st->st_pfs_group, ke_pd->pbs); \
    } \
}

static bool
build_and_ship_nonce(chunk_t *n, pb_stream *outs, u_int8_t np, const char *name)
{
    setchunk(*n, alloc_bytes(DEFAULT_NONCE_SIZE, name), DEFAULT_NONCE_SIZE);
    get_rnd_bytes(n->ptr, DEFAULT_NONCE_SIZE);
    return out_generic_chunk(np, &isakmp_nonce_desc, outs, *n, name);
}

/* Figure out which interface / IP address we are using for this exchange.
 * It used to be that we didn't know until we get a message
 * from our peer (in main_inI1_outR1 or main_inR1_outI2)!
 */
/* aggr-patch this func foes to ipsec_doi-aggr2.inc

static void
glean_myidentity(struct state *st)
{
    clonereplacechunk(st->st_myidentity
  , (void *) &st->st_connection->this.host
  , sizeof(st->st_connection->this.host)
  , "st_myidentity in glean_myidentity1()");
    st->st_myidentity_type = ID_IPV4_ADDR;
    DBG(DBG_CONTROL, DBG_log("my identity is %s"
  , inet_ntoa(st->st_connection->this.host)));
}
*/
#include "ipsec_doi-aggr2.inc"

/*
 * Send a notification to the peer. We could make a decision on
 * whether to send the notification, based on the type and the
 * destination, if we care to.
 * XXX It doesn't handle DELETE notifications (which are also
 * XXX informational exchanges).
 */
#if 0 /* not currently used */
//static void
//send_notification(int sock,
//    u_int16_t type,
//    u_char *spi,
//    u_char spilen,
//    u_char protoid,
//    u_char *icookie,
//    u_char *rcookie,
//    msgid_t /*network order*/ msgid,
//    struct sockaddr sa)
//{
//    u_char buffer[sizeof(struct isakmp_hdr) +
//     sizeof(struct isakmp_notification)];
//    struct isakmp_hdr *isa = (struct isakmp_hdr *) buffer;
//    struct isakmp_notification *isan = (struct isakmp_notification *)
//               (buffer + sizeof(struct isakmp_hdr));
//
//    memset(buffer, '\0', sizeof(struct isakmp_hdr) +
//    sizeof(struct isakmp_notification));
//
//    if (icookie != (u_char *) NULL)
//  memcpy(isa->isa_icookie, icookie, COOKIE_SIZE);
//
//    if (rcookie != (u_char *) NULL)
//  memcpy(isa->isa_rcookie, rcookie, COOKIE_SIZE);
//
//    /* Standard header */
//    isa->isa_np = ISAKMP_NEXT_N;
//    isa->isa_version = ISAKMP_MAJOR_VERSION << ISA_MAJ_SHIFT | ISAKMP_MINOR_VERSION;
//    isa->isa_xchg = ISAKMP_XCHG_INFO;
//    isa->isa_msgid = msgid;
//    isa->isa_length = htonl(sizeof(struct isakmp_hdr) +
//          sizeof(struct isakmp_notification) +
//          spilen);
//
//    /* Notification header */
//    isan->isan_type = htons(type);
//    isan->isan_doi = htonl(ISAKMP_DOI_IPSEC);
//    isan->isan_length = htons(sizeof(struct isakmp_notification) + spilen);
//    isan->isan_spisize = spilen;
//    memcpy((u_char *)isan + sizeof(struct isakmp_notification), spi, spilen);
//    isan->isan_protoid = protoid;
//
//    DBG(DBG_CONTROL, DBG_log("sending INFO type %s to %s",
//  enum_show(&notification_names, type),
//  show_sa(&sa)));
//
//    if (sendto(sock, buffer, ntohl(isa->isa_length), 0, &sa,
//         sizeof(sa)) != ntohl(isa->isa_length))
//  log_errno((e, "sendto() failed in send_notification() to %s",
//      show_sa(&sa)));
//    else
//    {
//  DBG(DBG_CONTROL, DBG_log("transmitted %d bytes", ntohl(isa->isa_length)));
//    }
//}
#endif /* not currently used */

/* The whole message must be a multiple of 4 octets.
 * I'm not sure where this is spelled out, but look at
 * rfc2408 3.6 Transform Payload.
 * Note: it talks about 4 BYTE boundaries!
 */
static void
close_message(pb_stream *pbs)
{
    size_t padding =  pad_up(pbs_offset(pbs), 4);

    if (padding != 0)
      (void) out_zero(padding, pbs, "message padding");
    close_output_pbs(pbs);
}

/* Initiate an Oakley Main Mode exchange.
 * --> HDR;SA
 */
static stf_status
main_outI1(
  int whack_sock,
  struct connection *c,
  bool pending_quick,
  lset_t policy,
  unsigned long klives)
{
    u_char space[8192]; /* NOTE: we assume 8192 is big enough to build the packet */
    pb_stream reply;  /* not actually a reply, but you know what I mean */
    pb_stream rbody;

    struct state *st;

    /* set up new state */
    cur_state = st = new_state();

    st->st_connection = c;
    st->st_pending_quick = pending_quick;
    st->st_policy = policy;
    st->st_whack_sock = whack_sock;
    st->st_klives = klives;
    st->st_state = STATE_MAIN_I1;

    get_cookie(ISAKMP_INITIATOR, st->st_icookie, COOKIE_SIZE, c->that.host);

    insert_state(st); /* needs cookies, connection, and msgid (0) */

    /* an event will be scheduled for st before we return */

    log("initiating Main Mode");

    /* set up reply */
    init_pbs(&reply, space, sizeof(space), "reply packet");

    /* HDR out */
    {
      struct isakmp_hdr hdr;

      memset(&hdr, '\0', sizeof(hdr));  /* default to 0 */
      hdr.isa_version = ISAKMP_MAJOR_VERSION << ISA_MAJ_SHIFT | ISAKMP_MINOR_VERSION;
      hdr.isa_np = ISAKMP_NEXT_SA;
      hdr.isa_xchg = ISAKMP_XCHG_IDPROT;
      memcpy(hdr.isa_icookie, st->st_icookie, COOKIE_SIZE);
      /* R-cookie, flags and MessageID are left zero */

      if (!out_struct(&hdr, &isakmp_hdr_desc, &reply, &rbody))
        return STF_INTERNAL_ERROR;
    }

    /* SA out */
    {
      u_char *sa_start = rbody.cur;

      if (!out_sa(&rbody, &oakley_sadb, st, TRUE, ISAKMP_NEXT_NONE))
        return STF_INTERNAL_ERROR;

      /* save initiator SA for later HASH */
      passert(st->st_p1isa.ptr == NULL);  /* no leak!  (MUST be first time) */
      clonetochunk(st->st_p1isa, sa_start, rbody.cur - sa_start
      , "sa in main_outI1");
    }

    close_message(&rbody);
    close_output_pbs(&reply);

    clonetochunk(st->st_tpacket, reply.start, pbs_offset(&reply)
                , "reply packet for main_outI1");

    /* Transmit */

    send_packet(st, "main_outI1");

    /* Set up a retransmission event, half a minute henceforth */
    event_schedule(EVENT_RETRANSMIT, EVENT_RETRANSMIT_DELAY, st);

    whack_log(st->st_whack_sock, RC_NEW_STATE + STATE_MAIN_I1
              , "%s: initiate", enum_name(&state_names, st->st_state));
    cur_state = NULL;
    return STF_NO_REPLY;
}

#include "ipsec_doi-aggr5.inc"
void
ipsecdoi_initiate(
  int whack_sock,
  struct connection *c,
  bool pending_quick,
  lset_t policy,
  unsigned long klives)
{
    /* If there's already an ISAKMP SA established, use that and
     * go directly to Quick Mode.
     * Note: there is no way to initiate with a Road Warrior.
     * XXX If an ISAKMP SA is *being* established, we foolishly
     * XXX try to establish another one, in parallel.  We could
     * XXX issue an event to wait for it to finish and then try
     * XXX to establish it.
     */
    struct state *st = find_phase1_state(c->that.host);

    if (st == NULL)
    {
/* aggr-patch */
      initiator_function *initiator = (LALLIN(c->policy, POLICY_AGGRESSIVE)
           ? aggr_outI1 : main_outI1);
      (void) initiator(whack_sock, c, pending_quick, policy, klives);

//  (void) main_outI1(whack_sock, c, pending_quick, policy, klives);
    }
    else if (pending_quick)
    {
      /* ??? we assume that peer_nexthop_sin isn't important:
       * we already have it from when we negotiated the ISAKMP SA!
       * It isn't clear what to do with the error return.
       */
      (void) quick_outI1(whack_sock, st, c, policy, klives);
    }
    else if (whack_sock != NULL_FD)
    {
      close(whack_sock);
    }
}

/* Replace SA with a fresh one that is similar
 *
 * Shares some logic with ipsecdoi_initiate, but not the same!
 * - we must not reuse the ISAKMP SA if we are trying to replace it!
 * - if trying to replace IPSEC SA, use ipsecdoi_initiate to build
 *   ISAKMP SA if needed.
 * - duplicate whack fd, if live.
 * Does not delete the old state -- someone else will do that.
 */
void
ipsecdoi_replace(struct state *st, unsigned long klives)
{
    int whack_sock = st->st_whack_sock == NULL_FD? NULL_FD : dup(st->st_whack_sock);
    lset_t policy = st->st_policy;

    if (IS_PHASE1(st->st_state))
    {
/* aggr-patch */
      initiator_function *initiator = (LALLIN(policy, POLICY_AGGRESSIVE)
           ? aggr_outI1 : main_outI1);
      initiator(whack_sock, st->st_connection, st->st_pending_quick,
                policy, klives);

//  (void) main_outI1(whack_sock, st->st_connection, st->st_pending_quick
//      , policy, klives);
    }
    else
    {
      /* Add features of actual old state to policy.  This ensures
       * that rekeying doesn't downgrade security.  I admit that
       * this doesn't capture everything.
       */
      if (st->st_pfs_group != NULL)
        policy |= POLICY_PFS;
      if (st->st_ah.present)
      {
        policy |= POLICY_AUTHENTICATE;
        if (st->st_ah.attrs.encapsulation == ENCAPSULATION_MODE_TUNNEL)
          policy |= POLICY_TUNNEL;
      }
      if (st->st_esp.present && st->st_esp.attrs.transid != ESP_NULL)
      {
        policy |= POLICY_ENCRYPT;
        if (st->st_esp.attrs.encapsulation == ENCAPSULATION_MODE_TUNNEL)
          policy |= POLICY_TUNNEL;
      }
      ipsecdoi_initiate(whack_sock, st->st_connection, TRUE, policy, klives);
    }
}

/* SKEYID for preshared keys.
 * See draft-ietf-ipsec-ike-01.txt 4.1
 */
static bool
skeyid_preshared(struct state *st)
{
    const chunk_t pss = get_preshared_secret(st);

    if (pss.ptr == NULL)
    {
      log("preshared secret disappeared!");
      return FALSE;
    }
    else
    {
      struct hmac_ctx ctx;

      hmac_init_chunk(&ctx, st->st_oakley.hasher, pss);
      hmac_update_chunk(&ctx, st->st_ni);
      hmac_update_chunk(&ctx, st->st_nr);
      hmac_final_chunk(st->st_skeyid, "st_skeyid in skeyid_preshared()", &ctx);
      return TRUE;
    }
}

/* Generate the SKEYID_* and new IV
 * See draft-ietf-ipsec-ike-01.txt 4.1
 */
static bool
generate_skeyids_iv(struct state *st)
{
    /* Generate the SKEYID */
    switch (st->st_oakley.auth)
    {
      case OAKLEY_PRESHARED_KEY:
        if (!skeyid_preshared(st))
          return FALSE;
        break;

      case OAKLEY_DSS_SIG:
      case OAKLEY_RSA_SIG:
      /* XXX */

      case OAKLEY_RSA_ENC:
      case OAKLEY_RSA_ENC_REV:
      case OAKLEY_ELGAMAL_ENC:
      case OAKLEY_ELGAMAL_ENC_REV:
      /* XXX */

      default:
        exit_log("generate_skeyids_iv(): unsupported authentication method %s",
                  enum_show(&oakley_auth_names, st->st_oakley.auth));
    }

    /* generate SKEYID_* from SKEYID */
    {
      struct hmac_ctx ctx;

      hmac_init_chunk(&ctx, st->st_oakley.hasher, st->st_skeyid);

      /* SKEYID_D */
      hmac_update_chunk(&ctx, st->st_shared);
      hmac_update(&ctx, st->st_icookie, COOKIE_SIZE);
      hmac_update(&ctx, st->st_rcookie, COOKIE_SIZE);
      hmac_update(&ctx, "\0", 1);
      hmac_final_chunk(st->st_skeyid_d, "st_skeyid_d in generate_skeyids_iv()", &ctx);

      /* SKEYID_A */
      hmac_reinit(&ctx);
      hmac_update_chunk(&ctx, st->st_skeyid_d);
      hmac_update_chunk(&ctx, st->st_shared);
      hmac_update(&ctx, st->st_icookie, COOKIE_SIZE);
      hmac_update(&ctx, st->st_rcookie, COOKIE_SIZE);
      hmac_update(&ctx, "\1", 1);
      hmac_final_chunk(st->st_skeyid_a, "st_skeyid_a in generate_skeyids_iv()", &ctx);

      /* SKEYID_E */
      hmac_reinit(&ctx);
      hmac_update_chunk(&ctx, st->st_skeyid_a);
      hmac_update_chunk(&ctx, st->st_shared);
      hmac_update(&ctx, st->st_icookie, COOKIE_SIZE);
      hmac_update(&ctx, st->st_rcookie, COOKIE_SIZE);
      hmac_update(&ctx, "\2", 1);
      hmac_final_chunk(st->st_skeyid_e, "st_skeyid_e in generate_skeyids_iv()", &ctx);
    }

    /* generate IV */
    {
      union hash_ctx hash_ctx;
      const struct hash_desc *h = st->st_oakley.hasher;

/* aggr-patch */
#if 0 /* TRH_XXX */
      st->st_new_iv_len = h->hash_digest_len;
#else
      st->st_new_iv_len = st->st_oakley.encrypter->blocksize;
#endif
//  st->st_new_iv_len = h->hash_digest_len;
      passert(st->st_new_iv_len <= sizeof(st->st_new_iv));

      h->hash_init(&hash_ctx);
      h->hash_update(&hash_ctx, st->st_gi.ptr, st->st_gi.len);
      h->hash_update(&hash_ctx, st->st_gr.ptr, st->st_gr.len);
      h->hash_final(st->st_new_iv, &hash_ctx);
    }

    /* Oakley Keying Material
     * Derived from Skeyid_e: if it is not big enough, generate more
     * using the PRF.
     * See draft-ietf-ipsec-isakmp-oakley-07.txt Appendix B
     */
    {
      const size_t keysize = st->st_oakley.encrypter->keysize;
      u_char keytemp[MAX_OAKLEY_KEY_LEN + MAX_DIGEST_LEN];
      u_char *k = st->st_skeyid_e.ptr;

      if (keysize > st->st_skeyid_e.len)
      {
        struct hmac_ctx ctx;
        size_t i = 0;

        hmac_init_chunk(&ctx, st->st_oakley.hasher, st->st_skeyid_e);
        hmac_update(&ctx, "\0", 1);
        for (;;)
        {
          hmac_final(&keytemp[i], &ctx);
          i += ctx.hmac_digest_len;
          if (i >= keysize)
            break;
          hmac_reinit(&ctx);
          hmac_update(&ctx, &keytemp[i - ctx.hmac_digest_len], ctx.hmac_digest_len);
        }
        k = keytemp;
      }
      clonereplacechunk(st->st_enc_key, k, keysize, "st_enc_key");
    }

    DBG(DBG_CRYPT,
    DBG_dump_chunk("Skeyid:  ", st->st_skeyid);
    DBG_dump_chunk("Skeyid_d:", st->st_skeyid_d);
    DBG_dump_chunk("Skeyid_a:", st->st_skeyid_a);
    DBG_dump_chunk("Skeyid_e:", st->st_skeyid_e);
    DBG_dump_chunk("enc key:", st->st_enc_key);
    DBG_dump("IV:", st->st_new_iv, st->st_new_iv_len));
    return TRUE;
}

/* Generate HASH_I or HASH_R for ISAKMP Phase I.
 * This will *not* generate other hash payloads (eg. Phase II of Quick Mode,
 * New Group Mode, or ISAKMP Informational Exchanges).
 * If the hashi argument is TRUE, generate HASH_I; if FALSE generate HASH_R.
 * If hashus argument is TRUE, we're generating a hash for our end.
 * See RFC2409 IKE 5.
 */
static void
main_mode_hash(struct state *st, u_char *hash_val, size_t *hash_len, bool hashi, bool hashus)
{
    struct hmac_ctx ctx;

    hmac_init_chunk(&ctx, st->st_oakley.hasher, st->st_skeyid);

    *hash_len = ctx.hmac_digest_len;

    if (hashi)
    {

      DBG_cond_dump(DBG_CRYPT, "g^xi", st->st_gi.ptr , st->st_gi.len);
      DBG_cond_dump(DBG_CRYPT, "g^xr", st->st_gr.ptr , st->st_gr.len);
      DBG_cond_dump(DBG_CRYPT, "CKY-I", st->st_icookie , COOKIE_SIZE);
      DBG_cond_dump(DBG_CRYPT, "CKY-R", st->st_rcookie , COOKIE_SIZE);
      hmac_update_chunk(&ctx, st->st_gi);
      hmac_update_chunk(&ctx, st->st_gr);
      hmac_update(&ctx, st->st_icookie, COOKIE_SIZE);
      hmac_update(&ctx, st->st_rcookie, COOKIE_SIZE);
    }
    else
    {
      hmac_update_chunk(&ctx, st->st_gr);
      hmac_update_chunk(&ctx, st->st_gi);
      hmac_update(&ctx, st->st_rcookie, COOKIE_SIZE);
      hmac_update(&ctx, st->st_icookie, COOKIE_SIZE);
    }

    DBG(DBG_CRYPT, DBG_log("hashing %d bytes of SA",
      st->st_p1isa.len - sizeof(struct isakmp_generic)));

    /* SA_b */
    hmac_update(&ctx, st->st_p1isa.ptr + sizeof(struct isakmp_generic),
    st->st_p1isa.len - sizeof(struct isakmp_generic));

    /* IDio_b (o stands for originator: i or r) */
    {
      /* hash identification payload, without generic payload header */
      struct isakmp_ipsec_id id;  /* body used, in network order! */

      memset(&id, '\0', sizeof(id));

      if (hashus) {
        id.isaiid_idtype = st->st_myidentity_type;
      /* leave protoid and port empty (zero) */
      } else {
        id.isaiid_idtype = st->st_peeridentity_type;
        id.isaiid_protoid = st->st_peeridentity_protocol;
        id.isaiid_port = htons(st->st_peeridentity_port);
      }
      DBG(DBG_CRYPT,
      DBG_log("Hashing %s ID: Type %s, Protocol %d, Port %d",
              hashus? "my" : "his",
      enum_show(&ident_names, id.isaiid_idtype),
      id.isaiid_protoid, htons(id.isaiid_port)));

      /* NOTE: hash does NOT include the generic payload part of
       * Identity Payload
       */
      hmac_update(&ctx,
      (u_char *)&id + sizeof(struct isakmp_generic),
      sizeof(id) - sizeof(struct isakmp_generic));
    }

    if (hashus)
      hmac_update_chunk(&ctx, st->st_myidentity);
    else {
      hmac_update_chunk(&ctx, st->st_peeridentity);
      DBG_cond_dump(DBG_CRYPT, "peer identity", st->st_peeridentity.ptr , st->st_peeridentity.len);
    }
    hmac_final(hash_val, &ctx);
}

/* CHECK_HASH
 *
 * This macro is magic -- it cannot be expressed as a function.
 * - it causes the caller to return!
 * - it declares local variables and expects the "do_hash" argument
 *   expression to reference them (hash_val, hash_len, hash_pbs)
 * - it references a variable local to the caller (md)
 */
#define CHECK_HASH(do_hash, hash_name, msg_name) { \
  pb_stream *const hash_pbs = &md->chain[ISAKMP_NEXT_HASH]->pbs; \
  u_char hash_val[MAX_DIGEST_LEN]; \
  size_t hash_len; \
  do_hash; \
  if (pbs_left(hash_pbs) != hash_len \
  || memcmp(hash_pbs->cur, hash_val, hash_len) != 0) \
  { \
      DBG_cond_dump(DBG_CRYPT, "received " hash_name ":", hash_pbs->cur, pbs_left(hash_pbs)); \
      DBG_cond_dump(DBG_CRYPT, "calculat " hash_name ":", hash_val, pbs_left(hash_pbs)); \
      log("received " hash_name " does not match computed value in " msg_name); \
      /* XXX Could send notification back */ \
      return STF_FAIL + INVALID_HASH_INFORMATION; \
  } \
    }

/* ACCEPT_NONCE
 *
 * This macro is magic.
 * - it can cause the caller to return.
 * - it references variables local to the caller (md, st)
 */
#define ACCEPT_NONCE(dest, name) { \
        pb_stream *nonce_pbs = &md->chain[ISAKMP_NEXT_NONCE]->pbs; \
  size_t len = pbs_left(nonce_pbs); \
  if (len < MINIMUM_NONCE_SIZE || MAXIMUM_NONCE_SIZE < len) \
  { \
      log(name " length not between %d and %d" \
    , MINIMUM_NONCE_SIZE, MAXIMUM_NONCE_SIZE); \
      return STF_FAIL + PAYLOAD_MALFORMED;  /* ??? */ \
  } \
  clonereplacechunk((dest), nonce_pbs->cur, len \
      , "st_" name " in " __FUNCTION__ "()"); \
    }

/* aggr_patch */ /* above */
/*  clonereplacechunk((dest), nonce_pbs->cur, len \ */ \
/*      , "st_ni in main_inI2_outR2()"); \ */


/* START_HASH_PAYLOAD
 *
 * Emit a to-be-filled-in hash payload, noting the field start (r_hashval)
 * and the start of the part of the message to be hashed (r_hash_start).
 * This macro is magic.
 * - it can cause the caller to return
 * - it references variables local to the caller (r_hashval, r_hash_start, st)
 */
#define START_HASH_PAYLOAD(rbody, np) { \
    pb_stream hash_pbs; \
    if (!out_generic(np, &isakmp_hash_desc, &(rbody), &hash_pbs)) \
  return STF_INTERNAL_ERROR; \
    r_hashval = hash_pbs.cur; /* remember where to plant value */ \
    if (!out_zero(st->st_oakley.hasher->hash_digest_len, &hash_pbs, "HASH")) \
  return STF_INTERNAL_ERROR; \
    close_output_pbs(&hash_pbs); \
    r_hash_start = (rbody).cur; /* hash from after HASH payload */ \
}

/* encrypt message, sans fixed part of header
 * IV is fetched from st->st_new_iv and stored into st->st_iv.
 * The theory is that there will be no "backing out", so we commit to IV.
 * We also close the pbs.
 */
static bool
encrypt_message(pb_stream *pbs, struct state *st)
{
    const struct encrypt_desc *e = st->st_oakley.encrypter;
    u_int8_t *enc_start = pbs->start + sizeof(struct isakmp_hdr);
    size_t enc_len = pbs_offset(pbs) - sizeof(struct isakmp_hdr);

    DBG_cond_dump(DBG_CRYPT | DBG_RAW, "encrypting:\n", enc_start, enc_len);
/* aggr-patch */
    DBG_dump_chunk("st_enc_key:\n", st->st_enc_key);
    DBG_dump("st_iv:\n", st->st_iv, st->st_iv_len);
    DBG_dump("st_new_iv:\n", st->st_new_iv, st->st_new_iv_len);
//

    /* pad up to multiple of encryption blocksize */
    {
      size_t padding = pad_up(enc_len, e->blocksize);

      if (padding != 0)
      {
        if (!out_zero(padding, pbs, "encryption padding"))
          return FALSE;
        enc_len += padding;
      }
    }

    DBG(DBG_CRYPT, DBG_log("encrypting using %s", enum_show(&oakley_enc_names, st->st_oakley.encrypt)));

    e->crypt(TRUE, enc_start, enc_len, st);

    update_iv(st);
    DBG_cond_dump(DBG_CRYPT, "next IV:", st->st_iv, st->st_iv_len);
    close_message(pbs);
    return TRUE;
}

/* Compute HASH(1), HASH(2) of Quick Mode.
 * HASH(1) is part of Quick I1 message.
 * HASH(2) is part of Quick R1 message.
 * Used by: quick_outI1, quick_inI1_outR1 (twice), quick_inR1_outI2
 * (see draft-ietf-ipsec-isakmp-oakley-07.txt 5.5)
 */
static void
quick_mode_hash12(
    u_char *dest, size_t *dest_len,
    const u_char *start, const u_char *roof,
    const struct state *st, bool hash2)
{
    struct hmac_ctx ctx;

    hmac_init_chunk(&ctx, st->st_oakley.hasher, st->st_skeyid_a);
    hmac_update(&ctx, (const u_char *) &st->st_msgid, sizeof(st->st_msgid));
    if (hash2)
      hmac_update_chunk(&ctx, st->st_ni); /* include Ni_b in the hash */
    hmac_update(&ctx, start, roof-start);
    hmac_final(dest, &ctx);

    if (dest_len != NULL)
      *dest_len = ctx.hmac_digest_len;

    DBG(DBG_CRYPT,
    DBG_log("HASH(%d) computed:", hash2 + 1);
    DBG_dump("", dest, ctx.hmac_digest_len));
}

/* Compute HASH(3) in Quick Mode (part of Quick I2 message).
 * Used by: quick_inR1_outI2, quick_inI2
 * See RFC2409 "The Internet Key Exchange (IKE)" 5.5.
 * NOTE: this hash (unlike HASH(1) and HASH(2)) ONLY covers the
 * Message ID and Nonces.  This is a mistake.
 */
static void
quick_mode_hash3(u_char *dest, size_t *destlen, struct state *st)
{
    struct hmac_ctx ctx;

    hmac_init_chunk(&ctx, st->st_oakley.hasher, st->st_skeyid_a);
    hmac_update(&ctx, "\0", 1);
    hmac_update(&ctx, (u_char *) &st->st_msgid, sizeof(st->st_msgid));
    hmac_update_chunk(&ctx, st->st_ni);
    hmac_update_chunk(&ctx, st->st_nr);
    hmac_final(dest, &ctx);

    if (destlen != NULL)
      *destlen = ctx.hmac_digest_len;

    DBG_cond_dump(DBG_CRYPT, "HASH(3) computed:", dest, ctx.hmac_digest_len);
}

/* Compute Phase 2 IV.
 * Uses Phase 1 IV from st_iv; puts result in st_new_iv.
 */
/* aggr-patch this func goes to ipsec_doi-aggr3.inc

void
init_phase2_iv(struct state *st, const msgid_t *msgid)
{
    const struct hash_desc *h = st->st_oakley.hasher;
    union hash_ctx ctx;

    st->st_new_iv_len = h->hash_digest_len;
    passert(st->st_new_iv_len <= sizeof(st->st_new_iv));

    h->hash_init(&ctx);
    h->hash_update(&ctx, st->st_iv, st->st_iv_len);
    passert(*msgid != 0);
    h->hash_update(&ctx, (const u_char *)msgid, sizeof(*msgid));
    h->hash_final(st->st_new_iv, &ctx);

    DBG_cond_dump(DBG_CRYPT, "computed Phase 2 IV:"
  , st->st_new_iv, st->st_new_iv_len);
}
*/
#include "ipsec_doi-aggr3.inc"


/* Initiate quick mode.
 * --> HDR*, HASH(1), SA, Nr [, KE ] [, IDci, IDcr ]
 * (see draft-ietf-ipsec-isakmp-oakley-07.txt 5.5)
 */
stf_status
quick_outI1(
  int whack_sock,
  struct state *phase1_st,
  struct connection *c,
  lset_t policy,
  unsigned long klives)
{
    struct state *st = duplicate_state(phase1_st);
    u_char space[8192]; /* NOTE: we assume 8192 is big enough to build the packet */
    pb_stream reply;  /* not really a reply */
    pb_stream rbody;
    u_char
    *r_hashval, /* where in reply to jam hash value */
    *r_hash_start;  /* start of what is to be hashed */
    bool has_client = c->this.has_client ||  c->that.has_client;

    cur_state = st;
    st->st_whack_sock = whack_sock;
    st->st_connection = c;
    st->st_policy = policy;
    st->st_klives = klives;

    st->st_myuserprotoid = st->st_peeruserprotoid = 0;
    st->st_myuserport = st->st_peeruserport = 0;

    st->st_msgid = generate_msgid(c->that.host);
    st->st_state = STATE_QUICK_I1;

/* aggr-patch */
    if (!c->that.has_client)
    {
      c->that.client_net = c->that.host;
      c->that.client_mask = mask32.sin_addr;
    }
//
    insert_state(st); /* needs cookies, connection, and msgid */

    /* an event will be scheduled for st before we return */

    log("initiating Quick Mode %s", bitnamesof(sa_policy_bit_names, policy));

    /* set up reply */
    init_pbs(&reply, space, sizeof(space), "reply packet");

    /* HDR* out */
    {
      struct isakmp_hdr hdr;

      hdr.isa_version = ISAKMP_MAJOR_VERSION << ISA_MAJ_SHIFT | ISAKMP_MINOR_VERSION;
      hdr.isa_np = ISAKMP_NEXT_HASH;
      hdr.isa_xchg = ISAKMP_XCHG_QUICK;
      hdr.isa_msgid = st->st_msgid;
      hdr.isa_flags = ISAKMP_FLAG_ENCRYPTION;
      memcpy(hdr.isa_icookie, st->st_icookie, COOKIE_SIZE);
      memcpy(hdr.isa_rcookie, st->st_rcookie, COOKIE_SIZE);
      if (!out_struct(&hdr, &isakmp_hdr_desc, &reply, &rbody))
        return STF_INTERNAL_ERROR;
    }

    /* HASH(1) -- create and note space to be filled later */
    START_HASH_PAYLOAD(rbody, ISAKMP_NEXT_SA);

    /* SA out */

    /* If PFS specified, use the same group as during Phase 1:
     * since no negotiation is possible, we pick one that is
     * very likely supported.
     */
    st->st_pfs_group = policy & POLICY_PFS? phase1_st->st_oakley.group : NULL;

// dev@fx.dk
    switch (st->st_connection->trans_order) {
      case 0:
      case 1:
        if (!out_sa(&rbody
            , &ipsec_sadb1[policy & (POLICY_ENCRYPT | POLICY_AUTHENTICATE | POLICY_TUNNEL)]
            , st, FALSE, ISAKMP_NEXT_NONCE))
          return STF_INTERNAL_ERROR;
        break;
      case 2:
        if (!out_sa(&rbody
            , &ipsec_sadb2[policy & (POLICY_ENCRYPT | POLICY_AUTHENTICATE | POLICY_TUNNEL)]
            , st, FALSE, ISAKMP_NEXT_NONCE))
          return STF_INTERNAL_ERROR;
        break;
      case 3:
        if (!out_sa(&rbody
            , &ipsec_sadb3[policy & (POLICY_ENCRYPT | POLICY_AUTHENTICATE | POLICY_TUNNEL)]
            , st, FALSE, ISAKMP_NEXT_NONCE))
          return STF_INTERNAL_ERROR;
        break;
      case 4:
        if (!out_sa(&rbody
            , &ipsec_sadb4[policy & (POLICY_ENCRYPT | POLICY_AUTHENTICATE | POLICY_TUNNEL)]
            , st, FALSE, ISAKMP_NEXT_NONCE))
          return STF_INTERNAL_ERROR;
        break;
      default:
          return STF_INTERNAL_ERROR;
    }

    /* Ni out */
    if (!build_and_ship_nonce(&st->st_ni, &rbody
        , policy & POLICY_PFS? ISAKMP_NEXT_KE : has_client? ISAKMP_NEXT_ID : ISAKMP_NEXT_NONE
        , "Ni"))
      return STF_INTERNAL_ERROR;

    /* [ KE ] out (for PFS) */

    if (st->st_pfs_group != NULL)
    {
      if (!build_and_ship_KE(st, &st->st_gi, st->st_pfs_group
          , &rbody, has_client? ISAKMP_NEXT_ID : ISAKMP_NEXT_NONE))
        return STF_INTERNAL_ERROR;
    }

    /* [ IDci, IDcr ] out */
    if (has_client)
    {
      struct isakmp_ipsec_id id;
      pb_stream id_pbs;

      /* IDci (we are initiator) */
      id.isaiid_np = ISAKMP_NEXT_ID;
      id.isaiid_idtype = ID_IPV4_ADDR_SUBNET;
      id.isaiid_protoid = st->st_myuserprotoid;
      id.isaiid_port = st->st_myuserport;

      if (!out_struct(&id, &isakmp_ipsec_identification_desc, &rbody, &id_pbs))
        return STF_INTERNAL_ERROR;

      if (!out_raw(&c->this.client_net
          , sizeof(c->this.client_net)
          , &id_pbs, "initiator's client network"))
        return STF_INTERNAL_ERROR;

      if (!out_raw(&c->this.client_mask.s_addr
          , sizeof(c->this.client_mask.s_addr)
          , &id_pbs, "initiator's client mask"))
        return STF_INTERNAL_ERROR;

      close_output_pbs(&id_pbs);

      /* IDcr (peer is responder) */
      id.isaiid_np = ISAKMP_NEXT_NONE;
      id.isaiid_idtype = ID_IPV4_ADDR_SUBNET;
      id.isaiid_protoid = st->st_peeruserprotoid;
      id.isaiid_port = st->st_peeruserport;
      if (!out_struct(&id, &isakmp_ipsec_identification_desc, &rbody, &id_pbs))
        return STF_INTERNAL_ERROR;

      if (!out_raw(&c->that.client_net
          , sizeof(c->that.client_net)
          , &id_pbs, "peer's client network"))
        return STF_INTERNAL_ERROR;

      if (!out_raw(&c->that.client_mask, sizeof(c->that.client_mask)
          , &id_pbs, "peer's client mask"))
        return STF_INTERNAL_ERROR;

      close_output_pbs(&id_pbs);
    }

    /* finish computing  HASH(1), inserting it in output */
    quick_mode_hash12(r_hashval, NULL, r_hash_start, rbody.cur, st, FALSE);

    /* encrypt message, except for fixed part of header */

    init_phase2_iv(phase1_st, &st->st_msgid);
    st->st_new_iv_len = phase1_st->st_new_iv_len;
    memcpy(st->st_new_iv, phase1_st->st_new_iv, st->st_new_iv_len);

    if (!encrypt_message(&rbody, st))
      return STF_INTERNAL_ERROR;

    /* save packet, now that we know its size */
    clonetochunk(st->st_tpacket, reply.start, pbs_offset(&reply)
                  , "reply packet from quick_outI1");

    /* send the packet */

    send_packet(st, "quick_outI1");

    event_schedule(EVENT_RETRANSMIT, EVENT_RETRANSMIT_DELAY, st);

    whack_log(st->st_whack_sock, RC_NEW_STATE + STATE_QUICK_I1
              , "%s: initiate", enum_name(&state_names, st->st_state));
    cur_state = NULL;
    return STF_NO_REPLY;
}

/* Decode the ID payload of Phase 1 (main_inI3_outR3 and main_inR3) */
/* aggr-patch this func goes to ipsec_doi-aggr4.inc

static bool
decode_peer_id(struct msg_digest *md)
{
    struct state *const st = md->st;
    struct payload_digest *const id_pld = md->chain[ISAKMP_NEXT_ID];
    pb_stream *const id_pbs = &id_pld->pbs;
    struct isakmp_id *const id = &id_pld->payload.id;

    // XXX Check for valid ID types?
    switch (id->isaid_idtype)
    {
    case ID_IPV4_ADDR:
  // I think that RFC2407 (IPSEC DOI) 4.6.2 is confused.
  // It talks about the protocol ID and Port fields of the ID
  // Payload, but they don't exist as such in Phase 1.
  // We use more appropriate names.
  // isaid_doi_specific_a is in place of Protocol ID.
  // isaid_doi_specific_b is in place of Port.
  // Besides, there is no good reason for allowing these to be
  // other than 0 in Phase 1.
  //
  if (!(id->isaid_doi_specific_a == 0 && id->isaid_doi_specific_b == 0)
  && !(id->isaid_doi_specific_a == IPPROTO_UDP && id->isaid_doi_specific_b == IKE_UDP_PORT))
  {
      log("protocol/port in Phase 1 ID payload must be 0/0 or %d/%d"
    " but are %d/%d"
    , IPPROTO_UDP, IKE_UDP_PORT
    , id->isaid_doi_specific_a, id->isaid_doi_specific_b);
      return FALSE;
  }
  if (pbs_left(id_pbs) != sizeof(struct in_addr))
  {
      log("size of ID_IPV4_ADDR identification should be %u"
    " but is %d in ID payload"
    , (unsigned) sizeof(struct in_addr)
    , (unsigned) pbs_left(id_pbs));
      // XXX Could send notification back
      return FALSE;
  }
  st->st_peeridentity_protocol = id->isaid_doi_specific_a;
  st->st_peeridentity_port = id->isaid_doi_specific_b;
  clonereplacechunk(st->st_peeridentity
      , id_pbs->cur, pbs_left(id_pbs)
      , "st_peeridentity in decode_peer_id()");

  DBG(DBG_PARSING,
      DBG_log("Peer's ID type is %s: %s",
    enum_show(&ident_names, id->isaid_idtype),
    inet_ntoa(*(struct in_addr *)id_pbs->cur)));
  break;

    default:
  // XXX Could send notification back
  log("Unacceptable identity type (%s) in ID payload"
      , enum_show(&ident_names, id->isaid_idtype));
  return FALSE;
    }
    st->st_peeridentity_type = id->isaid_idtype;
    return TRUE;
}
*/
#include "ipsec_doi-aggr4.inc"

/* Decode the variable part of an ID packet in (during Quick Mode).
 * This is designed for packets that identify clients, not peers.
 * Currently this will only accept two forms.
 */
static bool
decode_net_id(
    struct isakmp_ipsec_id *id,
    pb_stream *id_pbs,
    struct in_addr *net, struct in_addr *mask,
    const char *which)
{
    switch (id->isaiid_idtype)
    {
      case ID_IPV4_ADDR:
        if (pbs_left(id_pbs) != sizeof(*net))
        {
          log("%s ID payload ID_IPV4_ADDR wrong length in Quick I1"
              , which);
          /* XXX Could send notification back */
          return FALSE;
        }
        memcpy(net, id_pbs->cur, sizeof(*net));
        *mask = mask32.sin_addr;
        DBG(DBG_PARSING | DBG_CONTROL,
        DBG_log("%s is IP address %s", which, inet_ntoa(*net)));
        break;

      case ID_IPV4_ADDR_SUBNET:
        if (pbs_left(id_pbs) != sizeof(*net) + sizeof(*mask))
        {
          log("%s ID payload ID_IPV4_ADDR_SUBNET wrong length in Quick I1"
              , which);
          /* XXX Could send notification back */
          return FALSE;
        }
        memcpy(net, id_pbs->cur, sizeof(*net));
        memcpy(mask, id_pbs->cur + sizeof(*net), sizeof(*mask));
        DBG(DBG_PARSING | DBG_CONTROL,
        {
        char buf[SUBNETTOA_BUF];

        subnettoa(*net, *mask, 0, buf, sizeof(buf));
        DBG_log("%s is IP subnet %s", which, buf);
        });
        break;

      case ID_IPV4_ADDR_RANGE:
        /* ??? NOT YET TESTED */
        if (pbs_left(id_pbs) != sizeof(*net) + sizeof(*mask))
        {
          log("%s ID payload ID_IPV4_ADDR_RANGE wrong length in Quick I1"
              , which);
          /* XXX Could send notification back */
          return FALSE;
        }
        memcpy(net, id_pbs->cur, sizeof(*net));
        memcpy(mask, id_pbs->cur + sizeof(*net), sizeof(*mask));  /* temporary */

        /* check that range is really a subnet -- all we can handle.
         * (a) range is a power of 2
         * (b) start is a multiple of range
         */
        {
          u_int32_t start = ntohl(net->s_addr);
          u_int32_t end = ntohl(mask->s_addr);
          u_int32_t imask = end - start;  /* inverted mask */

          if (start > end)
          {
            log("%s ID payload in Quick I1, ID_IPV4_ADDR_RANGE improper: start is greater than end"
                , which);
            return FALSE;
          }
          /* (a) iff imask is one less than a power of two (i.e. good),
           *     adding one will yield a number with no bits in common.
           * (b) start must not have bits in common with imask.
           * We are too polite to optimize the whole test to
           * ((imask+1) | start) & imask.
           */
          if (((imask+1) & imask) != 0 || (start & imask) != 0)
          {
            log("%s ID payload in Quick I1, ID_IPV4_ADDR_RANGE unacceptable: not a subnet"
                , which);
            return FALSE;
          }
          mask->s_addr = htonl(~imask);
        }
        DBG(DBG_PARSING | DBG_CONTROL,
        {
        char buf[SUBNETTOA_BUF];

        subnettoa(*net, *mask, 0, buf, sizeof(buf));
        DBG_log("%s is IP subnet %s (received as range)"
        , which, buf);
        });
        break;

      default:
        /* XXX support more */
        log("unsupported ID type %s"
            , enum_show(&ident_names, id->isaiid_idtype));
        /* XXX Could send notification back */
        return FALSE;
    }
    return TRUE;
}

/* like decode, but checks that what is received matches what was sent */
static bool
check_net_id(
    struct isakmp_ipsec_id *id,
    pb_stream *id_pbs,
    u_int8_t *protoid, u_int16_t *port,
    struct in_addr *net, struct in_addr *mask,
    const char *which)
{
    struct in_addr net_temp, mask_temp;

    if (!decode_net_id(id, id_pbs, &net_temp, &mask_temp, which))
      return FALSE;

    if (net->s_addr != net_temp.s_addr
    || mask->s_addr != mask_temp.s_addr
    || *protoid != id->isaiid_protoid || *port != id->isaiid_port)
    {
      log("%s ID returned doesn't match my proposal", which);
      return FALSE;
    }
    return TRUE;
}

/*
 * Produce the new key material of Quick Mode.
 * draft-ietf-ipsec-isakmp-oakley-06.txt section 5.5
 * specifies how this is to be done.
 */
static void
compute_proto_keymat(
    struct state *st,
    u_int8_t protoid,
    struct ipsec_proto_info *pi)
{
    size_t needed_len; /* bytes of keying material needed */

    /* Add up the requirements for keying material
     * (It probably doesn't matter if we produce too much!)
     */
    switch (protoid)
    {
      case PROTO_IPSEC_ESP:
        switch (pi->attrs.transid)
        {
          case ESP_NULL:
            needed_len = 0;
            break;
          case ESP_DES:
            needed_len = DES_CBC_BLOCK_SIZE;
            break;
          case ESP_3DES:
            needed_len = DES_CBC_BLOCK_SIZE * 3;
            break;
          default:
            exit_log("transform %s not implemented yet",
            enum_show(&esp_transformid_names, pi->attrs.transid));
        }

        switch (pi->attrs.auth)
        {
          case AUTH_ALGORITHM_NONE:
            break;
          case AUTH_ALGORITHM_HMAC_MD5:
            needed_len += HMAC_MD5_KEY_LEN;
            break;
          case AUTH_ALGORITHM_HMAC_SHA1:
            needed_len += HMAC_SHA1_KEY_LEN;
            break;
          case AUTH_ALGORITHM_DES_MAC:
          default:
            exit_log("AUTH algorithm %s not implemented yet",
            enum_show(&auth_alg_names, pi->attrs.auth));
        }
        break;

      case PROTO_IPSEC_AH:
        switch (pi->attrs.transid)
        {
          case AH_MD5:
            needed_len = HMAC_MD5_KEY_LEN;
            break;
          case AH_SHA:
            needed_len = HMAC_SHA1_KEY_LEN;
          break;
          default:
            exit_log("transform %s not implemented yet",
                    enum_show(&ah_transformid_names, pi->attrs.transid));
        }
        break;

      default:
        exit_log("protocol %s not implemented yet",
                  enum_show(&protocol_names, protoid));
        break;
    }

    pi->keymat_len = needed_len;

    /* Allocate space for the keying material.
     * Although only needed_len bytes are desired, we
     * must round up to a multiple of ctx.hmac_digest_len
     * so that our buffer isn't overrun.
     */
    {
      struct hmac_ctx ctx_me, ctx_peer;
      size_t needed_space;  /* space needed for keying material (rounded up) */
      size_t i;

      hmac_init_chunk(&ctx_me, st->st_oakley.hasher, st->st_skeyid_d);
      ctx_peer = ctx_me;  /* duplicate initial conditions */

      needed_space = needed_len + pad_up(needed_len, ctx_me.hmac_digest_len);
      replace(pi->our_keymat, alloc_bytes(needed_space, "keymat in compute_keymat()"));
      replace(pi->peer_keymat, alloc_bytes(needed_space, "peer_keymat in quick_inI1_outR1()"));

      for (i = 0;; )
      {
        if (st->st_shared.ptr != NULL)
        {
          /* PFS: include the g^xy */
          hmac_update_chunk(&ctx_me, st->st_shared);
          hmac_update_chunk(&ctx_peer, st->st_shared);
        }
        hmac_update(&ctx_me, &protoid, sizeof(protoid));
        hmac_update(&ctx_peer, &protoid, sizeof(protoid));

        hmac_update(&ctx_me, (u_char *)&pi->our_spi, sizeof(pi->our_spi));
        hmac_update(&ctx_peer, (u_char *)&pi->attrs.spi, sizeof(pi->attrs.spi));

        hmac_update_chunk(&ctx_me, st->st_ni);
        hmac_update_chunk(&ctx_peer, st->st_ni);

        hmac_update_chunk(&ctx_me, st->st_nr);
        hmac_update_chunk(&ctx_peer, st->st_nr);

        hmac_final(pi->our_keymat + i, &ctx_me);
        hmac_final(pi->peer_keymat + i, &ctx_peer);

        i += ctx_me.hmac_digest_len;
        if (i >= needed_space)
          break;

        /* more keying material needed: prepare to go around again */

        hmac_reinit(&ctx_me);
        hmac_reinit(&ctx_peer);

        hmac_update(&ctx_me, pi->our_keymat + i - ctx_me.hmac_digest_len,
                    ctx_me.hmac_digest_len);
        hmac_update(&ctx_peer, pi->peer_keymat + i - ctx_peer.hmac_digest_len,
                    ctx_peer.hmac_digest_len);
      }
    }

    DBG(DBG_CRYPT,
    DBG_dump("KEYMAT computed:\n", pi->our_keymat, pi->keymat_len);
    DBG_dump("Peer KEYMAT computed:\n", pi->peer_keymat, pi->keymat_len));
}

static void
compute_keymats(struct state *st)
{
    if (st->st_ah.present)
      compute_proto_keymat(st, PROTO_IPSEC_AH, &st->st_ah);
    if (st->st_esp.present)
      compute_proto_keymat(st, PROTO_IPSEC_ESP, &st->st_esp);
}

/* State Transition Functions.
 * - Called from comm_handle;
 * - state_check[state].processor points to these
 * - these routines are in state-order
 * - these routines must be restartable from any point of error return.
 * - output HDR is handled by comm_handle().
 */

/* Handle a Main Mode Oakley first packet (responder side).
 * HDR;SA --> HDR;SA
 */
stf_status
main_inI1_outR1(struct msg_digest *md)
{
    struct payload_digest *const sa_pd = md->chain[ISAKMP_NEXT_SA];
    struct state *st;
/* aggr-patch */
    struct connection *c = find_host_connection(md->iface->addr, md->sin.sin_addr, NULL, -1);
//    struct connection *c = find_host_connection(md->iface->addr, md->sin.sin_addr);

    pb_stream r_sa_pbs;

/* aggr-patch */
    md->st = st = new_state();
    // put a very short fuse on this state object
    // in case things don't work out.
    //

/* dev@fx.dk */
//    event_schedule(EVENT_SA_EXPIRE, 0, st);

    st->st_policy &= ~POLICY_AGGRESSIVE;
//

    if (c == NULL)
    {
/* aggr-patch */
#if 0 /* TRH was here */
      /* see if a wildcarded connection can be found */
      c = find_host_connection(md->iface->addr, mask0.sin_addr, NULL, -1);

//#ifdef ROAD_WARRIOR_FUDGE
//  // see if a wildcarded connection can be found
//  c = find_host_connection(md->iface->addr, mask0.sin_addr);
//
      if (c != NULL)
      {
        /* create a temporary connection that is a copy of this one */
        c = rw_connection(c, md->sin.sin_addr);
      }
      else
#endif /* ROAD_WARRIOR_FUDGE */
      {
        log("initial Main Mode message from %s"
            " but no connection has been authorized"
            , inet_ntoa(md->sin.sin_addr));
/* aggr-patch */
//        log_event(md, NULL, PE_UNKNOWN_PHASE1_PEER, PR_UNKNOWN_CONNECTION);
//
        /* XXX notification is in order! */
        return STF_IGNORE;
      }
    }

    /* Set up state */
/* aggr-patch */
//    cur_state = md->st; /* (caller will reset cur_state) */
    cur_state = md->st = st = new_state();  // (caller will reset cur_state)
    st->st_connection = c;
    st->st_klives = 1;  /* not our job to try again from start */

    memcpy(st->st_icookie, md->hdr.isa_icookie, COOKIE_SIZE);
    get_cookie(ISAKMP_RESPONDER, st->st_rcookie, COOKIE_SIZE, md->sin.sin_addr);

    insert_state(st); /* needs cookies, connection, and msgid (0) */

    /* put a very short fuse on this state object
     * in case things don't work out.
     */

    event_schedule(EVENT_SA_EXPIRE, 0, st);

    glean_myidentity(st);

    st->st_doi = ISAKMP_DOI_IPSEC;
    st->st_situation = SIT_IDENTITY_ONLY; /* We only support this */
    st->st_state = STATE_MAIN_R1;

#ifdef ROAD_WARRIOR_FUDGE
    if (c->rw_state == rwcs_instance)
    {
      log("responding to Main Mode from Road Warrior %s"
          , inet_ntoa(c->that.host));
    }
    else
#endif /* ROAD_WARRIOR_FUDGE */
    {
      log("responding to Main Mode");
    }

    /* parse_isakmp_sa also spits out a winning SA into our reply,
     * so we have to build our md->reply and emit HDR before calling it.
     */

    /* HDR out.
     * We can't leave this to comm_handle() because we must
     * fill in the cookie.
     */
    {
      struct isakmp_hdr r_hdr = md->hdr;

      memcpy(r_hdr.isa_rcookie, st->st_rcookie, COOKIE_SIZE);
      r_hdr.isa_np = ISAKMP_NEXT_SA;
      if (!out_struct(&r_hdr, &isakmp_hdr_desc, &md->reply, &md->rbody))
        return STF_INTERNAL_ERROR;
    }

    /* start of SA out */
    {
      struct isakmp_sa r_sa = sa_pd->payload.sa;

      r_sa.isasa_np = ISAKMP_NEXT_NONE;
      if (!out_struct(&r_sa, &isakmp_sa_desc, &md->rbody, &r_sa_pbs))
        return STF_INTERNAL_ERROR;
    }

    /* SA body in and out */
    {
      notification_t r = parse_isakmp_sa_body(&sa_pd->pbs, &sa_pd->payload.sa, &r_sa_pbs
      , FALSE, st);

      if (r != NOTHING_WRONG)
        return STF_FAIL + r;
    }

    close_message(&md->rbody);

    /* save initiator SA for HASH */
    clonereplacechunk(st->st_p1isa, sa_pd->pbs.start, pbs_room(&sa_pd->pbs), "sa in main_inI1_outR1()");

    return STF_REPLY;
}

/* Handle HDR;SA from responder (verify proposal),
 * and send back HDR;KE;Ni.
 */
stf_status
main_inR1_outI2(struct msg_digest *md)
{
    struct state *const st = md->st;

    glean_myidentity(st);

    /* verify echoed SA */
    {
      struct payload_digest *const sapd = md->chain[ISAKMP_NEXT_SA];
      notification_t r = parse_isakmp_sa_body(&sapd->pbs
          , &sapd->payload.sa, NULL, TRUE, st);

      if (r != NOTHING_WRONG)
          return STF_FAIL + r;
    }

    /**************** build output packet HDR;KE;Ni ****************/

    /* HDR out.
     * We can't leave this to comm_handle() because the isa_np
     * depends on the type of Auth (eventually).
     */
    {
      struct isakmp_hdr r_hdr = md->hdr;

      r_hdr.isa_np = ISAKMP_NEXT_KE;
      if (!out_struct(&r_hdr, &isakmp_hdr_desc, &md->reply, &md->rbody))
        return STF_INTERNAL_ERROR;
    }

    /* KE out */
    if (!build_and_ship_KE(st, &st->st_gi, st->st_oakley.group
        , &md->rbody, ISAKMP_NEXT_NONCE))
      return STF_INTERNAL_ERROR;

    /* Ni out */
    if (!build_and_ship_nonce(&st->st_ni, &md->rbody, ISAKMP_NEXT_NONE, "Ni"))
      return STF_INTERNAL_ERROR;

    /* finish message */
    close_message(&md->rbody);

    /* Reinsert the state, using the responder cookie we just received */
    unhash_state(st);
    memcpy(st->st_rcookie, md->hdr.isa_rcookie, COOKIE_SIZE);
    insert_state(st); /* needs cookies, connection, and msgid (0) */

    st->st_state = STATE_MAIN_I2;

    return STF_REPLY;
}

/* Handle HDR;KE;Ni from Initiator.  Send a HDR;KE;Nr back.
 */
stf_status
main_inI2_outR2(struct msg_digest *md)
{
    struct state *const st = md->st;

    pb_stream *keyex_pbs = &md->chain[ISAKMP_NEXT_KE]->pbs;

    /* KE in */
    ACCEPT_KE(st->st_gi, "Gi", st->st_oakley.group, *keyex_pbs);

    /* Ni in */
    ACCEPT_NONCE(st->st_ni, "Ni");


    /**************** build output packet HDR;KE;Nr ****************/

    /* HDR out done */

    /* KE out */
    if (!build_and_ship_KE(st, &st->st_gr, st->st_oakley.group
        , &md->rbody, ISAKMP_NEXT_NONCE))
      return STF_INTERNAL_ERROR;

    /* Nr out */
    if (!build_and_ship_nonce(&st->st_nr, &md->rbody, ISAKMP_NEXT_NONE, "Nr"))
      return STF_INTERNAL_ERROR;

    /* finish message */
    close_message(&md->rbody);

    /* next message will be encrypted, but not this one.
     * We could defer this calculation.
     */
    compute_dh_shared(st, st->st_gi, st->st_oakley.group);
#ifdef DODGE_DH_MISSING_ZERO_BUG
    if (st->st_shared.ptr[0] == 0)
      return STF_DROP_DOOMED_EXCHANGE;
#endif
    if (!generate_skeyids_iv(st))
      return STF_FAIL + AUTHENTICATION_FAILED;
    update_iv(st);

    /* Advance state */
    st->st_state = STATE_MAIN_R2;

    return STF_REPLY;
}

/* Handle HDR;KE;Nr from responder.  Send a HDR*;IDii;HASH_I back.
 */
stf_status
main_inR2_outI3(struct msg_digest *md)
{
    struct state *const st = md->st;

    pb_stream *const keyex_pbs = &md->chain[ISAKMP_NEXT_KE]->pbs;

    /* KE in */
    ACCEPT_KE(st->st_gr, "Gr", st->st_oakley.group, *keyex_pbs);

    /* Nr in */
    ACCEPT_NONCE(st->st_nr, "Nr");

    /* done parsing; initialize crypto  */

    compute_dh_shared(st, st->st_gr, st->st_oakley.group);
#ifdef DODGE_DH_MISSING_ZERO_BUG
    if (st->st_shared.ptr[0] == 0)
      return STF_REPLACE_DOOMED_EXCHANGE;
#endif
    if (!generate_skeyids_iv(st))
      return STF_FAIL + AUTHENTICATION_FAILED;

    /**************** build output packet HDR*;IDii;HASH_I ****************/
    /* ??? NOTE: this is almost the same as main_inI3_outR3's code */

    /* HDR* out done */

    /* IDii out */
    {
      struct isakmp_ipsec_id id;
      pb_stream id_pbs;

      id.isaiid_np = ISAKMP_NEXT_HASH;
      id.isaiid_idtype = st->st_myidentity_type;
      id.isaiid_protoid = 0;  /* ??? is this right?  IPPROTO_UDP? */
      id.isaiid_port = 0; /* ??? is this right?  our_port? */
      if (!out_struct(&id, &isakmp_ipsec_identification_desc, &md->rbody, &id_pbs)
          || !out_chunk(st->st_myidentity, &id_pbs, "my identity"))
        return STF_INTERNAL_ERROR;
      close_output_pbs(&id_pbs);
    }

    /* HASH_I out */
    {
      u_char hash_val[MAX_DIGEST_LEN];
      size_t hash_len;

      main_mode_hash(st, hash_val, &hash_len, TRUE, TRUE);

      if (!out_generic_raw(ISAKMP_NEXT_NONE, &isakmp_hash_desc, &md->rbody
          , hash_val, hash_len, "HASH_I"))
      return STF_INTERNAL_ERROR;
    }

    /* encrypt message, except for fixed part of header */

    /* st_new_iv was computed by generate_skeyids_iv */
    if (!encrypt_message(&md->rbody, st))
      return STF_INTERNAL_ERROR;  /* ??? we may be partly committed */

    /* Advance state */
    st->st_state = STATE_MAIN_I3;

    return STF_REPLY;
}

/*
 * Handle HDR*;IDii;HASH_I from initiator. Send a HDR*;IDir;HASH_R back.
 */
stf_status
main_inI3_outR3(struct msg_digest *md)
{
    struct state *const st = md->st;
/* aggr-patch */
    struct connection *c;
//
    /* input code similar to main_inR3 -- should be factored */

    /* IDii in */
/* aggr-patch */
    if (!decode_peer_id(md, st))
//    if (!decode_peer_id(md))
      return STF_FAIL + INVALID_ID_INFORMATION;
/* aggr-patch */
    c = find_host_connection(md->iface->addr, md->sin.sin_addr,
           &st->st_peeridentity, st->st_peeridentity_type);
    if (c == NULL)
    {
      return STF_IGNORE;
    }
    st->st_connection = c;
//
    /* HASH_I in */
    CHECK_HASH(main_mode_hash(st, hash_val, &hash_len, TRUE, FALSE)
                , "HASH_I", "Main I3");

    /**************** build output packet HDR*;IDir;HASH_R ****************/
    /* ??? NOTE: this is almost the same as main_inR2_outI3's code */

    /* IDir out */
    {
      struct isakmp_ipsec_id r_id;
      pb_stream r_id_pbs;

      r_id.isaiid_np = ISAKMP_NEXT_HASH;
      r_id.isaiid_idtype = st->st_myidentity_type;
      r_id.isaiid_protoid = 0;  /* ??? is this right? */
      r_id.isaiid_port = 0; /* ??? is this right? */
      if (!out_struct(&r_id, &isakmp_ipsec_identification_desc, &md->rbody, &r_id_pbs)
          || !out_chunk(st->st_myidentity, &r_id_pbs, "my identity"))
        return STF_INTERNAL_ERROR;
      close_output_pbs(&r_id_pbs);
    }

    /* HASH_R out */
    {
      u_char hash_val[MAX_DIGEST_LEN];
      size_t hash_len;

      main_mode_hash(st, hash_val, &hash_len, FALSE, TRUE);

      if (!out_generic_raw(ISAKMP_NEXT_NONE, &isakmp_hash_desc, &md->rbody
          , hash_val, hash_len, "HASH_R"))
        return STF_INTERNAL_ERROR;
    }

    /* encrypt message, sans fixed part of header */

    if (!encrypt_message(&md->rbody, st))
      return STF_INTERNAL_ERROR;  /* ??? we may be partly committed */

    /* Last block of Phase 1 (R3), kept for Phase 2 IV generation */
    DBG_cond_dump(DBG_CRYPT, "last encrypted block of Phase 1:"
                  , st->st_new_iv, st->st_new_iv_len);

    /* Advance state */
    st->st_state = STATE_MAIN_R3;
    st->st_connection->newest_isakmp_sa = st->st_serialno;

    return STF_REPLY;
}

/* Handle HDR*;IDir;HASH_R from responder.
 */
stf_status
main_inR3(struct msg_digest *md)
{
    struct state *const st = md->st;
    struct connection *c = st->st_connection;

    /* input code similar to main_inI3_outR3 -- should be factored */

    /* IDir in */
/* aggr-patch */
    if (!decode_peer_id(md, st))
//    if (!decode_peer_id(md))
      return STF_FAIL + INVALID_ID_INFORMATION;

/* aggr-patch */
    /* Check that the id received from our peer matches what's in our
       connection description.  */
    if (c->that.id_type > 0
        && (c->that.id_type != st->st_peeridentity_type
        || !chunksequal(c->that.id, st->st_peeridentity)))
    {
      log(("Responders Aggressive Mode packet claiming to be from %s,"
           " but that doesn't match our connection description"),
            show_identity(st->st_peeridentity,
            st->st_peeridentity_type));
      log_event(md, st, PE_UNKNOWN_PHASE1_PEER, PR_UNKNOWN_CONNECTION);
      /* XXX notification is in order! */
      return STF_IGNORE;
    }
//


    /* HASH_R in */
    CHECK_HASH(main_mode_hash(st, hash_val, &hash_len, FALSE, FALSE)
  , "HASH_R", "Main R3");

    /**************** done input ****************/

    /* Advance state */
    st->st_state = STATE_MAIN_I4;
    c->newest_isakmp_sa = st->st_serialno;

    update_iv(st);  /* finalize our Phase 1 IV */

    return STF_UNPEND_QUICK;
}

#include "ipsec_doi-aggr6.inc"

/* Handle first message of Phase 2 -- Quick Mode.
 * HDR*, HASH(1), SA, Ni [, KE ] [, IDci, IDcr ] -->
 * HDR*, HASH(2), SA, Nr [, KE ] [, IDci, IDcr ]
 * (see draft-ietf-ipsec-isakmp-oakley-07.txt 5.5)
 */
stf_status
quick_inI1_outR1(struct msg_digest *md)
{
    /* we build reply packet as we parse the message since
     * the parse_ipsec_sa_body emits the reply SA
     */

    struct state *st = duplicate_state(md->st);
    struct connection *const c = st->st_connection;
    struct payload_digest *const id_pd = md->chain[ISAKMP_NEXT_ID];
    struct in_addr our_net, our_mask, peer_net, peer_mask;

    u_char
    *r_hashval, /* where in reply to jam hash value */
    *r_hash_start;  /* from where to start hashing */

    /* first: fill in missing bits of our new state object */

    st->st_klives = 1;  /* not our job to try again from start */

    st->st_msgid = md->hdr.isa_msgid;
    insert_state(st); /* needs cookies, connection, and msgid */

    st->st_new_iv_len = md->st->st_new_iv_len;
    memcpy(st->st_new_iv, md->st->st_new_iv, st->st_new_iv_len);

    cur_state = st; /* (caller will reset) */
    md->st = st;  /* feed back new state */

    st->st_policy = c->policy;  /* somebody has got to do it */

    /* put a very short fuse on this state object
     * in case things don't work out.
     */
    event_schedule(EVENT_SA_EXPIRE, 0, st);


    /* HASH(1) in */
    CHECK_HASH(quick_mode_hash12(hash_val, &hash_len, hash_pbs->roof, md->message_pbs.roof, st, FALSE)
                , "HASH(1)", "Quick I1");

    /* HDR* out done */

    /* HASH(2) out -- first pass */
    START_HASH_PAYLOAD(md->rbody, ISAKMP_NEXT_SA);

    /* process SA (in and out) */
    {
      struct payload_digest *const sapd = md->chain[ISAKMP_NEXT_SA];
      pb_stream r_sa_pbs;
      struct isakmp_sa sa = sapd->payload.sa;

      /* sa header is unchanged -- except for np */
      sa.isasa_np = ISAKMP_NEXT_NONCE;
      if (!out_struct(&sa, &isakmp_sa_desc, &md->rbody, &r_sa_pbs))
        return STF_INTERNAL_ERROR;

      /* parse and accept body */
      st->st_pfs_group = &unset_group;
      {
        notification_t r = parse_ipsec_sa_body(&sapd->pbs
        , &sapd->payload.sa, &r_sa_pbs, FALSE, st);

        if (r != NOTHING_WRONG)
          return STF_FAIL + r;
      }
    }

    passert(st->st_pfs_group != &unset_group);

    if ((st->st_policy & POLICY_PFS) && st->st_pfs_group == NULL)
    {
      log("we require PFS but Quick I1 SA specifies no GROUP_DESCRIPTION");
      return STF_FAIL + NO_PROPOSAL_CHOSEN; /* ??? */
    }

    /* Ni in */
    ACCEPT_NONCE(st->st_ni, "Ni");

    /* [ KE ] in (for PFS) */
    ACCEPT_PFS_KE(st->st_gi, "Gi", "Quick Mode I1");

    /* [ IDci, IDcr ] in */

    if (id_pd != NULL)
    {
      /* ??? we are assuming IPSEC_DOI */

      /* IDci (initiator is peer) */

      if (!decode_net_id(&id_pd->payload.ipsec_id, &id_pd->pbs
          , &peer_net, &peer_mask, "peer client"))
        return STF_FAIL + INVALID_ID_INFORMATION;

      st->st_peeruserprotoid = id_pd->payload.ipsec_id.isaiid_protoid;
      st->st_peeruserport = id_pd->payload.ipsec_id.isaiid_port;

      /* IDcr (we are responder) */

      if (!decode_net_id(&id_pd->next->payload.ipsec_id, &id_pd->next->pbs
      , &our_net, &our_mask, "our client"))
        return STF_FAIL + INVALID_ID_INFORMATION;

      st->st_myuserprotoid = id_pd->next->payload.ipsec_id.isaiid_protoid;
      st->st_myuserport = id_pd->next->payload.ipsec_id.isaiid_port;
    }
    else
    {
      /* implicit IDci and IDcr: peer and self */
      our_net = c->this.host;
      our_mask = mask32.sin_addr;
      peer_net = c->that.host;
      peer_mask = mask32.sin_addr;
    }

    /* Now that we have identities of client subnets, we must look for
     * a suitable connection (our current one only matches for hosts).
     */
    {
/* aggr-patch */
      struct connection *p = find_client_connection(c,
      st->st_peeridentity_type, st->st_peeridentity,
      our_net, our_mask, peer_net, peer_mask);
//  struct connection *p = find_client_connection(c
//      , our_net, our_mask, peer_net, peer_mask);

/* aggr-patch
#ifdef ROAD_WARRIOR_FUDGE
  if (p == NULL && c->parent != NULL)
  {
      // We didn't find an exact match, so try for one that still
      // needs instantiation.  If the peer's client subnet just
      // contains the peer, search for 0.0.0.0/32 as client.
      //
      p = find_client_connection(c->parent
    , our_net, our_mask
    , peer_net.s_addr == c->that.host.s_addr
      && peer_mask.s_addr == mask32.sin_addr.s_addr
        ? mask0.sin_addr : peer_net
    , peer_mask);
      if (p != NULL)
      {
    // we found one -- instantiate it //
    p = rw_connection(p, c->that.host);
      }
  }
#endif //  ROAD_WARRIOR_FUDGE
*/
      if (p == NULL)
      {
        char
        me[ADDRTOA_BUF], mine[SUBNETTOA_BUF],
        he[ADDRTOA_BUF], his[SUBNETTOA_BUF];

        addrtoa(c->this.host, 0, me, sizeof(me));
        subnettoa(our_net, our_mask, 0, mine, sizeof(mine));
        addrtoa(c->that.host, 0, he, sizeof(he));
        subnettoa(peer_net, peer_mask, 0, his, sizeof(mine));
        log("cannot respond to IPsec SA request"
          " because no connection is known for %s===%s...%s===%s",
            mine, me, he, his);
/* aggr-patch */
        log_event_2(md, st,
        PE_UNKNOWN_PHASE2_CONNECTION, PR_UNKNOWN_CONNECTION,
        our_net, our_mask, peer_net, peer_mask);
//
        return STF_FAIL + INVALID_ID_INFORMATION;
      }
      st->st_connection = p;  /* ??? more plumbing needed? */
      DBG(DBG_CONTROL, DBG_log("using connection \"%s\"", p->name));
    }

    log("responding to Quick Mode");

    /**** finish reply packet: Nr [, KE ] [, IDci, IDcr ] ****/

    /* Nr out */
    if (!build_and_ship_nonce(&st->st_nr, &md->rbody
    , st->st_pfs_group != NULL? ISAKMP_NEXT_KE : id_pd != NULL? ISAKMP_NEXT_ID : ISAKMP_NEXT_NONE
    , "Nr"))
      return STF_INTERNAL_ERROR;

    /* [ KE ] out (for PFS) */

    if (st->st_pfs_group != NULL)
    {
      if (!build_and_ship_KE(st, &st->st_gr, st->st_pfs_group
          , &md->rbody, id_pd != NULL? ISAKMP_NEXT_ID : ISAKMP_NEXT_NONE))
        return STF_INTERNAL_ERROR;

      /* MPZ-Operations might be done after sending the packet... */

      compute_dh_shared(st, st->st_gi, st->st_pfs_group);
#ifdef DODGE_DH_MISSING_ZERO_BUG
      if (st->st_shared.ptr[0] == 0)
        return STF_DROP_DOOMED_EXCHANGE;
#endif
    }

    /* [ IDci, IDcr ] out */
    if  (id_pd != NULL)
    {
      struct isakmp_ipsec_id *p = (void *)md->rbody.cur;  /* UGH! */

      if (!out_raw(id_pd->pbs.start, pbs_room(&id_pd->pbs), &md->rbody, "IDci"))
        return STF_INTERNAL_ERROR;
      p->isaiid_np = ISAKMP_NEXT_ID;

      p = (void *)md->rbody.cur;  /* UGH! */

      if (!out_raw(id_pd->next->pbs.start, pbs_room(&id_pd->next->pbs), &md->rbody, "IDcr"))
        return STF_INTERNAL_ERROR;
      p->isaiid_np = ISAKMP_NEXT_NONE;
    }

    /* Compute reply HASH(2) and insert in output */
    quick_mode_hash12(r_hashval, NULL, r_hash_start, md->rbody.cur, st, TRUE);

    /* encrypt message, except for fixed part of header */

    if (!encrypt_message(&md->rbody, st))
      return STF_INTERNAL_ERROR;  /* ??? we may be partly committed */

    /* Update state of exchange */
    st->st_state = STATE_QUICK_R1;

    return STF_REPLY;
}

/* Handle (the single) message from Responder in Quick Mode.
 * HDR*, HASH(2), SA, Nr [, KE ] [, IDci, IDcr ] -->
 * HDR*, HASH(3)
 * (see draft-ietf-ipsec-isakmp-oakley-07.txt 5.5)
 */
stf_status
quick_inR1_outI2(struct msg_digest *md)
{
    struct state *const st = md->st;

    /* HASH(2) in */
    CHECK_HASH(quick_mode_hash12(hash_val, &hash_len, hash_pbs->roof, md->message_pbs.roof, st, TRUE)
              , "HASH(2)", "Quick R1");

    /* SA in */
    {
      struct payload_digest *const sa_pd = md->chain[ISAKMP_NEXT_SA];
      notification_t r = parse_ipsec_sa_body(&sa_pd->pbs
      , &sa_pd->payload.sa, NULL, TRUE, st);

      if (r != NOTHING_WRONG)
        return STF_FAIL + r;
    }

    /* Nr in */
    ACCEPT_NONCE(st->st_nr, "Nr");

    /* [ KE ] in (for PFS) */
    ACCEPT_PFS_KE(st->st_gr, "Gr", "Quick Mode R1");
    if (st->st_pfs_group != NULL)
    {
      compute_dh_shared(st, st->st_gr, st->st_pfs_group);
#ifdef DODGE_DH_MISSING_ZERO_BUG
      if (st->st_shared.ptr[0] == 0)
        return STF_REPLACE_DOOMED_EXCHANGE;
#endif
    }

    /* [ IDci, IDcr ] in; these must match what we sent */

    {
      struct payload_digest *const id_pd = md->chain[ISAKMP_NEXT_ID];

      if (id_pd != NULL)
      {
        /* ??? we are assuming IPSEC_DOI */

        /* IDci (we are initiator) */

        if (!check_net_id(&id_pd->payload.ipsec_id, &id_pd->pbs
        , &st->st_myuserprotoid, &st->st_myuserport
        , &st->st_connection->this.client_net
        , &st->st_connection->this.client_mask
        , "our client"))
          return STF_FAIL + INVALID_ID_INFORMATION;

        /* IDcr (responder is peer) */

        if (!check_net_id(&id_pd->next->payload.ipsec_id, &id_pd->next->pbs
        , &st->st_peeruserprotoid, &st->st_peeruserport
        , &st->st_connection->that.client_net
        , &st->st_connection->that.client_mask
        , "peer client"))
          return STF_FAIL + INVALID_ID_INFORMATION;
      }
      else
      {
        /* No IDci, IDcr: we must check that the defaults match our proposal.
         * Parallels a sequence of assignments in quick_outI1.
         */
        const struct connection *c = st->st_connection;

        if (c->this.client_net.s_addr != c->this.host.s_addr
            || c->this.client_mask.s_addr != mask32.sin_addr.s_addr
            || c->that.client_net.s_addr != c->that.host.s_addr
            || c->that.client_mask.s_addr != mask32.sin_addr.s_addr)
        {
          log("IDci, IDcr payloads missing in message"
              " but default does not match proposal");
          return STF_FAIL + INVALID_ID_INFORMATION;
        }
      }
    }

    /* ??? We used to copy the accepted proposal into the state, but it was
     * never used.  From sa_pd->pbs.start, length pbs_room(&sa_pd->pbs).
     */

    /**************** build reply packet HDR*, HASH(3) ****************/

    /* HDR* out done */

    /* HASH(3) out -- since this is the only content, no passes needed */
    {
      u_char
      *r_hashval, /* where in reply to jam hash value */
      *r_hash_start;  /* start of what is to be hashed */

      START_HASH_PAYLOAD(md->rbody, ISAKMP_NEXT_NONE);
      quick_mode_hash3(r_hashval, NULL, st);
    }

    /* Derive new keying material */
    compute_keymats(st);

    /* XXX Unless the commit bit is set, we need to send a message
     * XXX to the kernel to establish the new SA.
     * We do this before any state updating so that
     * failure won't look like success.
     */
    if (!install_ipsec_sa(st, TRUE))
      return STF_INTERNAL_ERROR;

    /* encrypt message, except for fixed part of header */

    if (!encrypt_message(&md->rbody, st))
      return STF_INTERNAL_ERROR;  /* ??? we may be partly committed */

    /* Update state of exchange */
    st->st_state = STATE_QUICK_I2;
    st->st_connection->newest_ipsec_sa = st->st_serialno;

    return STF_REPLY;
}

/* Handle last message of Quick Mode.
 * HDR*, HASH(3) -> done
 * (see draft-ietf-ipsec-isakmp-oakley-07.txt 5.5)
 */
stf_status
quick_inI2(struct msg_digest *md)
{
    struct state *const st = md->st;

    /* HASH(3) in */
    CHECK_HASH(quick_mode_hash3(hash_val, &hash_len, st)
      , "HASH(3)", "Quick I2");

    /* Derive new keying material */
    compute_keymats(st);

    /* XXX Unless the commit bit is set, we need to send a message
     * XXX to the kernel to establish the new SA.
     * We do this before any state updating so that
     * failure won't look like success.
     */
    if (!install_ipsec_sa(st, FALSE))
      return STF_INTERNAL_ERROR;

    /* Advance state */
    st->st_state = STATE_QUICK_R2;
    st->st_connection->newest_ipsec_sa = st->st_serialno;

    update_iv(st);  /* not actually used, but tidy */

    return STF_NO_REPLY;
}

