/* machine parsable logging of events
 * Copyright (C) 1999  Cendio Systems AB.
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.  See <http://www.fsf.org/copyleft/gpl.txt>.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * RCSID $Id$
 */

#include "os2port.h"

#include <errno.h>
#include <string.h>
#include <unistd.h>
#include <sys/socket.h>
#include <sys/un.h>

#include <freeswan.h>

#include "constants.h"
#include "defs.h"
#include "server.h"
#include "packet.h"
#include "demux.h"
#include "state.h"
#include "connections.h"
#include "log.h"

#include "eventlog.h"
#include "plutoevents.h"


#define MAX_EVENTLOGGERS	256

static int eventlog_fds[MAX_EVENTLOGGERS];
static int nfds = 0;


static bool
add_eventlog_fd(int fd)
{
    if (nfds >= MAX_EVENTLOGGERS)
	return FALSE;
    eventlog_fds[nfds++] = fd;
    return TRUE;
}


void
close_all_eventloggers(void)
{
    while (nfds > 0)
	close(eventlog_fds[--nfds]);
}


void
eventlog_client_handle(int evlogfd)
{
    int evclient;
    struct sockaddr_un cliaddr;
    int cliaddrlen = sizeof cliaddr;

    evclient = accept(evlogfd, (struct sockaddr*)&cliaddr, &cliaddrlen);
    if (evclient < 0)
    {
	log_errno((e, "accept() failed in eventlog_client_handle()"));
	return;
    }
    if (!add_eventlog_fd(evclient))
    {
	log("too many listeners on eventlogs");
	close(evclient);
	return;
    }
}



void
log_event_2(struct msg_digest *md, struct state *st,
	    enum plutoevents what, enum plutoreasons reason,
	    struct in_addr our_client_net, struct in_addr our_client_mask,
	    struct in_addr peer_client_net, struct in_addr peer_client_mask)
{
    int i;
    struct pluto_event event;
    struct connection *c = st ? st->st_connection : NULL;

/* dev@fx.dk */
    if (st == NULL)
       return;
//	
    event.magic = PE_MAGIC;
    event.version = PE_VERSION;
    event.msglen = sizeof event;
    event.event = what;

    strncpy(event.connection, c ? c->name : "",
	    MIN(sizeof event.connection, sizeof c->name));
    event.connection[sizeof event.connection - 1] = '\0';

    event.me.host = md ? md->iface->addr : c->this.host;
    event.me.id_type = st->st_myidentity_type;
    event.me.id_len = st->st_myidentity.len;
    memcpy(event.me.identity, st->st_myidentity.ptr,
	   MIN(sizeof event.me.identity, st->st_myidentity.len));
    event.me.client_net = our_client_net;
    event.me.client_mask = our_client_mask;

    event.peer.host = md ? md->sin.sin_addr : c->that.host;
    event.peer.id_type = st->st_peeridentity_type;
    event.peer.id_len = st->st_peeridentity.len;
    memcpy(event.peer.identity, st->st_peeridentity.ptr,
	   MIN(sizeof event.peer.identity, st->st_peeridentity.len));
    event.peer.client_net = peer_client_net;
    event.peer.client_mask = peer_client_mask;
    
    event.reason = reason;

    i = 0;
    while (i < nfds)
    {
	int fd = eventlog_fds[i];
	int written = write(fd, &event, event.msglen);
	if (written < event.msglen)
	{
	    close(fd);
	    eventlog_fds[i] = eventlog_fds[--nfds];
	    continue;
	}
	++i;
    }

    return;
}



void
log_event(struct msg_digest *md, struct state *st,
	  enum plutoevents what, enum plutoreasons reason)
      
{
    struct connection *c = st ? st->st_connection : NULL;
    struct in_addr our_client_net, our_client_mask;
    struct in_addr peer_client_net, peer_client_mask;

    if (c)
    {
	our_client_net = c->this.client_net;
	our_client_mask = c->this.client_mask;
	peer_client_net = c->that.client_net;
	peer_client_mask = c->that.client_mask;
    }
    else
    {
	our_client_net = mask0.sin_addr;
	our_client_mask = mask0.sin_addr;
	peer_client_net = mask0.sin_addr;
	peer_client_mask = mask0.sin_addr;
    }

    log_event_2(md, st, what, reason,
		our_client_net, our_client_mask,
		peer_client_net, peer_client_mask);
}
