/* information about connections between hosts and clients
 * Copyright (C) 1998, 1999  D. Hugh Redelmeier.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.  See <http://www.fsf.org/copyleft/gpl.txt>.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * RCSID $Id: connections.c,v 1.43 1999/09/28 21:51:17 dhr Exp $
 */
/*
 * changed by dev@fx.dk 24/07/1999 (port to OS/2)
 * changed by dev@fx.dk 09/10/1999
 * changed by dev@fx.dk 29/10/1999 (new snapshot)
 * changed by dev@fx.dk 19/11/1999 (adding aggressive mode)
 */
#include "os2port.h"

#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <netinet/in.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include <freeswan.h>

#include "constants.h"
#include "defs.h"
#include "connections.h"
#include "packet.h"
#include "demux.h"  /* needs packet.h */
#include "state.h"
#include "ipsec_doi.h"  /* needs demux.h and state.h */
#include "server.h"
#include "kernel.h" /* for no_klips */
#include "log.h"
#include "whack.h"
/* aggr-patch */
#include "route.h"
//
static struct connection *connections = NULL;

/* find a connection by name.
 * move the winner (if any) to the front.
 * If none is found, and wft is not NULL_FD, a diagnostic is logged to whack.
 */
struct connection *con_by_name(const char *nm, int wfd) {
  struct connection *p, *prev;

  prev = NULL;
  for (p = connections; ; p = p->next) {
    if (p == NULL) {
      if (wfd != NULL_FD)
        whack_log(wfd, RC_UNKNOWN_NAME, "no connection with that name");
      break;
    }
    if (strcmp(p->name, nm) == 0) {
      if (prev != NULL) {
        struct connection *nxt = p->next;

        p->next = connections;
        connections = p;
        prev->next = nxt;
      }
      break;
    }
    prev = p;
  }
  return p;
}

static void unorient_connection(struct connection *c) {
#ifdef ROAD_WARRIOR_FUDGE
  if (c->rw_state == rwcs_instance) {
    /* This does everything we need.
     * Note that we will be called by delete_connection,
     * but rw_state will be rwcs_going_away.
     */
    delete_connection(c);
  }
#endif /* ROAD_WARRIOR_FUDGE */
  {
    delete_states_by_connection(c);
    unroute_connection(c);
    c->interface = NULL;
  }
}

/* Delete a connection */

void delete_connection(struct connection *c) {
  struct connection *old_cur_connection = cur_connection;

  cur_connection = c;
#ifdef ROAD_WARRIOR_FUDGE
  /* Must be careful to avoid circularity:
   * we mark c as going away so it won't get deleted recursively.
   */
  passert(c->rw_state != rwcs_going_away);
  if (c->rw_state == rwcs_instance) {
    char that_host[ADDRTOA_BUF];

    addrtoa(c->that.host, 0, that_host, sizeof(that_host));
    log("deleting connection with Road Warrior %s", that_host);
    c->rw_state = rwcs_going_away;
  } else
#endif /* ROAD_WARRIOR_FUDGE */
  {
    log("deleting connection");
  }
/* aggr-patch */
    /* When release_interface calls unorient_connection, it may
       result in a rw_instance being deleted.  If this is the case the
       connection is already unoriented.  We have to check it here to
       avoid a failed assertion.  */
  if (c->interface)
//

    unorient_connection(c); /* won't delete c */

    /* find and delete c from connections list */
  {
    struct connection **p;

    for (p = &connections; *p != c; p = &(*p)->next)
      passert(*p != NULL);    /* we must not come up empty-handed */
    *p = c->next;
  }
  cur_connection = old_cur_connection == c? NULL : old_cur_connection;
/* aggr-patch */
  freeanychunk(c->this.id);
  freeanychunk(c->that.id);
//
  pfree(c);
}

void delete_every_connection(void) {
  while (connections != NULL)
  delete_connection(connections);
}

void release_interface(struct iface *i) {
  struct connection *c, *n;

  passert(i != NULL);
  for (c = connections; c != NULL; c = n) {
    n = c->next;
    if (c->interface == i)
      unorient_connection(c); /* may delete c */
  }
}

static void default_end(struct end *e, struct in_addr dflt_nexthop) {
  if (e->host_nexthop.s_addr == 0)
    e->host_nexthop = dflt_nexthop;
  if (!e->has_client) {
    e->client_net = e->host;
    e->client_mask = mask32.sin_addr;
/* aggr-patch */
    e->has_client = 1;
//
  }
}

/* format the topology of an end, leaving out defaults */
static size_t format_end(char *buf, size_t buf_len, struct end *this, struct end *that, bool is_left) {
  char client[SUBNETTOA_BUF];
  const char *client_sep = "";
  char host[ADDRTOA_BUF];
  char host_port[10];
  char hop[ADDRTOA_BUF];
  const char *hop_sep = "";

  /* [client===] */
  client[0] = '\0';
  if (this->has_client) {
    subnettoa(this->client_net, this->client_mask, 0, client, sizeof(client));
    client_sep = "===";
  }

  /* host */
  addrtoa(this->host, 0, host, sizeof(host));
  host_port[0] = '\0';
  if (this->host_port != IKE_UDP_PORT)
    snprintf(host_port, sizeof(host_port), ":%u", this->host_port);

  /* [---hop] */
  hop[0] = '\0';
  hop_sep = "";
  if (this->host_nexthop.s_addr != that->host.s_addr) {
    addrtoa(this->host_nexthop, 0, hop, sizeof(hop));
    hop_sep = "---";
  }

  if (is_left)
    snprintf(buf, buf_len, "%s%s%s%s%s%s", client, client_sep, host, host_port, hop_sep, hop);
  else
    snprintf(buf, buf_len, "%s%s%s%s%s%s", hop, hop_sep, host, host_port, client_sep, client);
  return strlen(buf);
}

void add_connection(const struct whack_message *wm, int whack_fd) {
  if (con_by_name(wm->name, NULL_FD) != NULL) {
    log("attempt to redefine connection \"%s\"", wm->name);
    whack_log(whack_fd, RC_DUPNAME, "attempt to redefine connection \"%s\"", wm->name);
  } else {
    struct connection *c = alloc_thing(struct connection, "struct connection");

    strcpy(c->name, wm->name);

    c->policy = wm->policy;

    c->sa_ike_life_seconds = wm->sa_ike_life_seconds;
    c->sa_ipsec_life_seconds = wm->sa_ipsec_life_seconds;
    c->sa_rekey_margin = wm->sa_rekey_margin;
    c->sa_rekey_fuzz = wm->sa_rekey_fuzz;
    c->sa_keying_tries = wm->sa_keying_tries;
    c->trans_order = wm->trans_order;

/* aggr-patch */
    c->this = wm->left.end;
    clonetochunk(c->this.id, wm->left.id, wm->left.id_len, "left end id");
    c->that = wm->right.end;
    clonetochunk(c->that.id, wm->right.id, wm->right.id_len, "right end id");
//  c->this = wm->left;
//  c->that = wm->right;

    default_end(&c->this, c->that.host);
    default_end(&c->that, c->this.host);

    /* set internal fields */
    c->next = connections;
    connections = c;
    c->interface = NULL;
    c->routed = FALSE;
    c->newest_isakmp_sa = SOS_NOBODY;
    c->newest_ipsec_sa = SOS_NOBODY;
    c->eroute_owner = SOS_NOBODY;

#ifdef ROAD_WARRIOR_FUDGE
    c->rw_state = rwcs_permanent;
    c->parent = NULL;
#endif

    /* log all about this connection */
    log("added connection description \"%s\"", c->name);
    DBG(DBG_CONTROL,
      char lhs[100];
      char rhs[100];

      (void) format_end(lhs, sizeof(lhs), &c->this, &c->that, TRUE);
      (void) format_end(rhs, sizeof(rhs), &c->that, &c->this, FALSE);

      DBG_log("%s...%s", lhs, rhs);

      DBG_log("ike_life: %lus; ipsec_life: %lus; rekey_margin: %lus;"
    " rekey_fuzz: %lu%%; keyingtries: %lu; policy: %s"
    , (unsigned long) c->sa_ike_life_seconds
    , (unsigned long) c->sa_ipsec_life_seconds
    , (unsigned long) c->sa_rekey_margin
    , (unsigned long) c->sa_rekey_fuzz
    , (unsigned long) c->sa_keying_tries
    , bitnamesof(sa_policy_bit_names, c->policy));
    );
  }
}

#ifdef ROAD_WARRIOR_FUDGE
#include "conn-aggr.inc"
/* aggr-patch this func goes to conn-aggr.inc

struct connection *
rw_connection(struct connection *c, struct in_addr him)
{
    struct connection *d = clone_thing(*c, "temporary connection");

    DBG(DBG_CONTROL, DBG_log("instantiating \"%s\" for %s"
  , c->name, inet_ntoa(him)));
    d->rw_state = rwcs_instance;
    d->parent = c;

    passert(oriented(*d));
    passert(d->that.host.s_addr == 0);
    d->that.host = him;
    default_end(&d->that, d->this.host);

    // We cannot guess what our next_hop should be, but if it was
    / explicitly specified as 0.0.0.0, we set it to be him.
    // (whack will not allow nexthop to be elided in RW case.)
    //
    default_end(&d->this, d->that.host);

    // set internal fields
    d->next = connections;
    connections = d;
    d->routed = FALSE;
    d->newest_isakmp_sa = SOS_NOBODY;
    d->newest_ipsec_sa = SOS_NOBODY;
    d->eroute_owner = SOS_NOBODY;

    return d;
}
*/
#endif /* ROAD_WARRIOR_FUDGE */

bool orient(struct connection *c, bool must) {
  if (!oriented(*c)) {
    struct iface *p;

    /* Note: this loop does not stop when it finds a match:
     * it continues checking to catch any ambiguity.
     */
    for (p = interfaces; p != NULL; p = p->next) {
/* aggr-patch */
      struct in_addr ia;

      ia.s_addr = use_identity_addr
      ? identity_addr.s_addr : p->addr.s_addr;
//
      for (;;) {
        /* check if this interface matches this end */
/* aggr-patch */
        if (c->this.host.s_addr == ia.s_addr
//    if (c->this.host.s_addr == p->addr.s_addr
            && (!no_klips || c->this.host_port == pluto_port)) {
          if (oriented(*c)) {
            if (c->interface == p)
              log("both sides of \"%s\" are our interfaces!", c->name);
            else
              log("two interfaces match \"%s\"", c->name);
            c->interface = NULL;  /* withdraw orientation */
            return FALSE;
          }
          c->interface = p;
        }

        /* done with this interface if it doesn't match that end */
        if (!(c->that.host.s_addr == p->addr.s_addr
            && (!no_klips || c->that.host_port == pluto_port)))
          break;

        /* swap ends and try again */
        {
          struct end t = c->this;

          c->this = c->that;
          c->that = t;
        }
      }
    }
    if (!oriented(*c)) {
      if (must)
        log("cannot find interface for connection \"%s\"", c->name);
      return FALSE;
    }
  }
  return TRUE;
}

void initiate_connection(const char *name, int whackfd) {
  struct connection *c = con_by_name(name, whackfd);

  if (c != NULL) {
    cur_connection = c;
    if (!orient(c, TRUE)) {
      log("could not orient connection");
      whack_log(whackfd, RC_ORIENT, "could not orient connection");
    } else {
      /* We will only request an IPsec SA if policy isn't empty.
       * This is a fudge, but not yet important.
       */
      ipsecdoi_initiate(whackfd == NULL_FD? NULL_FD : dup(whackfd)
                        , c, c->policy != 0, c->policy, c->sa_keying_tries);
    }
    cur_connection = NULL;
  }
}

void terminate_connection(const char *nm, int whackfd) {
#ifdef ROAD_WARRIOR_FUDGE
  /* Loop because more than one may match in ROAD_WARRIOR_FUDGE.
   * But at least one is required (enforced by con_by_name).
   */
  struct connection *c, *n;

  for (c = con_by_name(nm, whackfd); c != NULL; c = n) {
    n = c->next;  /* grab this before c might disappear */
    if (strcmp(c->name, nm) == 0) {
      cur_connection = c;
      log("terminating SAs using connection");
      delete_states_by_connection(c);
      cur_connection = NULL;
    }
  }
#else /* !ROAD_WARRIOR_FUDGE */
  struct connection *c = con_by_name(nm, whackfd);

  if (c != NULL) {
    cur_connection = c;
    log("terminating SAs using connection");
    delete_states_by_connection(c);
    cur_connection = NULL;
  }
#endif /* !ROAD_WARRIOR_FUDGE */
}

/* check nexthop safety
 * Our nexthop must not be within a routed client subnet, and vice versa.
 * Note: we don't think this is true.  We think that KLIPS will
 * not process a packet output by an eroute.
 */
#ifdef NEVER
bool check_nexthop(const struct connection *c) {
  struct connection *d;

  if (inside_client(c->this.host_nexthop, c->that)) {
    log("cannot perform routing for connection \"%s\""
        " because nexthop is within peer's client network", c->name);
    return FALSE;
  }

  for (d = connections; d != NULL; d = d->next) {
    if (d->routed) {
      if (inside_client(c->this.host_nexthop, d->that)) {
        log("cannot do routing for connection \"%s\"
            " because nexthop is contained in"
            " existing routing for connection \"%s\"",
            c->name, d->name);
        return FALSE;
      }
      if (inside_client(d->this.host_nexthop, c->that)) {
        log("cannot do routing for connection \"%s\"
            " because it contains nexthop of"
            " existing routing for connection \"%s\"",
            c->name, d->name);
        return FALSE;
      }
    }
  }
  return TRUE;
}
#endif /* NEVER */

/* Find a connection that owns the route to a connection's peer's client.
 * Preference is given to one that has an eroute too.
 * If eroute is true, our client must match too (as for eroutes).
 */
struct connection *route_owner(struct connection *c, bool eroute) {
  struct connection *d, *best = NULL;

  passert(oriented(*c));

  if (c->routed && c->eroute_owner != SOS_NOBODY)
    return c;

  for (d = connections; d != NULL; d = d->next) {
    if (d->routed) {
      passert(oriented(*d));
      if (same_client(c->that, d->that)
          && (!eroute || same_client(c->this, d->this))) {
        if (d->eroute_owner != SOS_NOBODY)
          return d; /* definite winner */

        best = d; /* winner if no better comes along */
      }
    }
  }
  return best;
}

/* aggr-patch funcs go to conn-aggr2.inc */
/*
struct connection *
find_host_connection(struct in_addr me, struct in_addr him)
{
    struct connection *d;

    for (d = connections; d != NULL; d = d->next)
    {
  if (d->this.host.s_addr == me.s_addr
  || d->that.host.s_addr == me.s_addr)
  {
      if (!oriented(*d))
    (void)orient(d, FALSE);

      // check this.host last -- it is almost certainly us
      if (d->that.host.s_addr == him.s_addr
      && d->this.host.s_addr == me.s_addr
      && (!no_klips || d->this.host_port == pluto_port))
    return d;
  }
    }
    return NULL;
}


#ifdef __OS2__
// dev@fx.dk
#include "connos2.inc"
#endif
*/
#include "conn-aggr2.inc"

void show_connections_status(int wfd) {
  struct connection *c;

  for (c = connections; c != NULL; c = c->next) {
    const char *routed = c->routed? "; routed" : "";
    const char *ifn = oriented(*c)? c->interface->rname : "";

    /* show topology */
    {
      char lhs[100];
      size_t ll = format_end(lhs, sizeof(lhs), &c->this, &c->that, TRUE);
      char rhs[100];
      size_t rl = format_end(rhs, sizeof(rhs), &c->that, &c->this, FALSE);

      if (ll + rl < 70) {
        /* display on one line */
        whack_log(wfd, RC_COMMENT, "\"%s\": %s...%s", c->name, lhs, rhs);
      } else {
        /* split over two lines */
        whack_log(wfd, RC_COMMENT, "\"%s\": %s...", c->name, lhs);
        whack_log(wfd, RC_COMMENT, "\"%s\": ...%s", c->name, rhs);
      }
    }

    whack_log(wfd, RC_COMMENT
              , "\"%s\": ike_life: %lus; ipsec_life: %lus; rekey_margin: %lus;"
              " rekey_fuzz: %lu%%; keyingtries: %lu"
              , c->name
              , (unsigned long) c->sa_ike_life_seconds
              , (unsigned long) c->sa_ipsec_life_seconds
              , (unsigned long) c->sa_rekey_margin
              , (unsigned long) c->sa_rekey_fuzz
              , (unsigned long) c->sa_keying_tries);


    whack_log(wfd, RC_COMMENT
              , "\"%s\": policy: %s; interface: %s%s"
              , c->name
              , bitnamesof(sa_policy_bit_names, c->policy)
              , ifn
              , routed);

    whack_log(wfd, RC_COMMENT
              , "\"%s\": newest ISAKMP SA: #%ld; newest IPsec SA: #%ld; eroute owner: #%lu"
              , c->name
              , c->newest_isakmp_sa
              , c->newest_ipsec_sa
              , c->eroute_owner);
  }
}

