/* REXX */
/* nistime2.cmd -- get current time from NIST and set OS/2 clock */

versionstring = "Version 0.2d   02-nov-2001"

/* Author:
      Pieter Bras
      Cambridge, MA
      pbras@pobox.com

   Copyright (C) 2001 Pieter Bras
   You may redistribute this program provided this copyright notice is
   preserved in full.

Usage:

   nistime2 options listfile

   "listfile" is the name of a file in the current directory containing
   the various NIST time servers, in the order that they should be tried.

   If "listfile" is missing then the program looks for a file named
   nist-srv.lst in the current directory.

   If there is also no file named nist-srv.lst then only the NISTIME default
   NIST time server is tried.

Options (introduced with the '-' character):

   -? or  ?    display help, then exit
   -h or -H    same as -?

   -s0   don't display any messages, and pass -s0 to nistime.exe

   all other options are passed through to nistime.exe
*/

/* uncomment the following line for debugging */
signal on novalue

/* load OS/2 RexxUtil functions (not needed) */
/* if rxFuncQuery( 'SysLoadFuncs') = 1 then do
   call RxFuncAdd 'SysLoadFuncs', 'RexxUtil', 'SysLoadFuncs'
   call SysLoadFuncs
   end
*/

/* define non-printing ascii codes */
HTAB  = '09'X
SPACE = '20'X

msglvl = 1
n_opts = ""
parse arg commandline

/* get name of file containing list of NIST servers */
defaultlist = "nist-srv.lst"        /* default file from NIST */
serverlist = checkopts( strip( commandline))
if serverlist = "" then
   serverlist = defaultlist
else if \FileExists( serverlist) then do
   call message "Cannot find file:" serverlist "-- using:" defaultlist
   serverlist = defaultlist
   end

SrvAddr.0 = 0                       /* number of servers */
rv = BuildServerTable( serverlist)
if SrvAddr.0 = 0 then do            /* if table empty, set up default server */
   call message "Server list empty or not found, using default server..."
   SrvAddr.1 = ''
   SrvName.1 = '(default server)'
   SrvNote.1 = ''
   SrvLoc.1  = ''
   SrvAddr.0 = 1
   end
rv = GetSetTime()
exit rv

/* Try all NIST servers until one of them returns the correct time. */
/* return codes from nistime.exe (ver 0.2c or later):
      0        success: clock not adjusted
      1        success: clock was adjusted
      100-199  unrecoverable error
      200-299  error, possibly recoverable with different choice of server
      300-...  error, possibly recoverable with currently selected server
*/
GetSetTime:
   do i = 1 to SrvAddr.0
      if pos( '3', SrvNote.i) > 0 then
         iterate                       /* test server, do not use */
      call message ''
      call message "Trying:" SrvName.i
      rv = nistime( SrvAddr.i)
      if rv < 200 then                 /* done: success or unrecoverable err */
         leave
      if rv < 300 then                 /* try another server */
         iterate
      call message "Trying again..."   /* try current server one more time */
      rv = nistime( SrvAddr.i)
      if rv < 200 then                 /* done: success or unrecoverable err */
         leave
      end /* do i = ... */
   if rv >= 100 then do
      call message ''
      call message "Unsuccessful: NISTIME error" rv
      end
   return rv

/* Call nistime.exe program to set clock. Returns nistime.exe return code. */
nistime:
   /* address CMD */ "@nistime.exe -S" n_opts arg( 1)
   return RC

/* Build table of NIST time servers. Returns: 0=success, 1=failure */
BuildServerTable:
   parse arg srvlist
   if \FileExists( srvlist) then
      return 1
   if \FileOpen( srvlist) then
      return 1
   i = SrvAddr.0
/* scan until a line starting with '$' is found */
   do while lines( srvlist)
      if left( linein( srvlist), 1) = '$' then
         leave
      end /* do while */
/* parse the lines until another line starting with '$' is found */
   do while lines( srvlist)
      server = translate( linein( srvlist), SPACE, HTAB)
      if left( server, 1) = '$' then
         leave
      i = i + 1
      parse var server SrvName.i SrvAddr.i SrvNote.i SrvLoc.i .
      end /* do while */
   SrvAddr.0 = i
   rv = FileClose( srvlist)
   return 0

/* rv = FileExists( fileName) */ /* TRUE if exists, FALSE otherwise */
FileExists:
   rv = stream( arg( 1), 'C', 'QUERY EXISTS')
   if rv = '' then
      return 0
   else
      return 1

/* rv = FileOpen( fileName, fileMode) */  /* TRUE/FALSE if success/error */
FileOpen:
   rv = stream( arg( 1), 'C', 'OPEN' arg( 2))
   return abbrev( 'READY:', rv)

/* rv = FileClose( fileName) */           /* TRUE/FALSE if success/error */
FileClose:
   rv = stream( arg( 1), 'C', 'CLOSE')
   return abbrev( 'READY:', rv)

message:
   if msglvl = 1 then
      say arg( 1)
   return

checkopts:
   parse arg optlist
   if left( optlist, 1) = '?' then
      signal usage
   do while left( optlist, 1) = '-'
      parse var optlist switch optlist
      if left( switch, 2) = '-?' | left( translate( switch), 2) = '-H' then
         signal usage
      if left( switch, 3) = '-m0' then
         msglvl = 0
      n_opts = n_opts switch              /* pass to nistime.exe */
      end
   return strip( optlist)

usage:
   say 'NISTIME2  ' versionstring
   say ''
   say 'Usage:  nistime2 [options] [serverlist]'
   say ''
   say '   where "serverlist" (optional) is a preferred list of NIST servers;'
   say '   if not found, program will look for a file named "nist-srv.lst".'
   say ''
   say 'Options:'
   say ''
   say '   -? or  ?    display this help message'
   say '   -h or -H    same as -?'
   say "   -s0         don't display messages (also passed to nistime.exe)"
   say ''
   say '   all other options are passed through to nistime.exe'
   exit 0

/* <eof> */
