/*- REXX -ILOG.CMD V1.1 -------------------------------------------*
 |                                                                 |
 | Description:                                                    |
 |                                                                 |
 | This EXEC will summarize the total time you have been connected |
 | to the Internet if you are using IBM (Advantis) as your Internet|
 | provider.  The EXEC will attempt to get the path to the log     |
 | dataset from the 'ETC' environment variable. Also, I'm assuming |
 | that the name of the log dataset has not been changed from      |
 | 'CONNECT.LOG'.  If it has, modify the code to point to the      |
 | correct dataset (the full path is carried in the variable       |
 | 'logfile').  Apparently the date format as output to the log has|
 | been changed with the latest dialer update from MM/DD to        |
 | YYYY/MM/DD. I've coded for this and this EXEC will handle both  |
 | formats.                                                        |
 |                                                                 |
 | If you have any questions or comments, drop me a line at:       |
 | 'ddarrow@ibm.net'.                                              |
 |                                                                 |
 | Date      Maintenance                           Who By          |
 | --------  ------------------------------------  --------------  |
 |                                                                 |
 | 06/15/95  Author.                               Dempsey Darrow  |
 |                                                                 |
 | 06/17/95  Fixed problem where monthly totals    Dempsey Darrow  |
 |           were not displayed when log                           |
 |           did not span multiple months.                         |
 |                                                                 |
 *-----------------------------------------------------------------*/ ;

if   Found_Connect_Log()  /* The log was found */,
then do ;
        call Process_Connect_Log ;

        say yellow ;
     end ;
else do ;
        say 'The search for ''CONNECT.LOG'' using the',
            'environment variable ''ETC'' failed.' ;
        say 'Try hard-coding the path and filename and then run',
            'the program again.' ;
     end ;

say ;
say 'Press ENTER to terminate...' ;

pull x ;

exit ;
/*======================== End of Main EXEC =======================*/ ;
/*-  Section: Found_Connect_Log  ----------------------------------*
 |                                                                 |
 | This section will search for the path to the OS2 Internet       |
 | Connection log dataset by examining the 'ETC' environment       |
 | variable. We will assume that the log dataset name is still the |
 | default: 'CONNECT.LOG'.                                         |
 |                                                                 |
 *-----------------------------------------------------------------*/ ;

Found_Connect_Log:

procedure expose logfile ;

Section_Rc = 1  /* Assume good result */ ;

File_Path = 'VALUE'(etc,,os2environment) ;

do 1 ;

   if   File_Path = ''  /* Variable doesn't exist */,
   then do ;
           Section_Rc = 0 ;

           leave ;
        end ;
   else logfile = File_Path'\connect.log' ;

   if   'STREAM'(logfile,'c','query size') = '',
   then do ;
           Section_Rc = 0 ;

           leave ;
        end ;
   else nop ;

end ;

return Section_Rc ;
/*=================================================================*/ ;
/*-  Section: Process_Connect_Log  --------------------------------*
 |                                                                 |
 | The log has been locate; process it.                            |
 |                                                                 |
 *-----------------------------------------------------------------*/ ;

Process_Connect_Log:

swidth = 78             /* Banner width             */ ;
twidth = 66             /* Data   width             */ ;

"mode co80,25"          /* Set default display size */ ;

black  ='[30m[1A'     /* Define ANSI colors       */ ;
red    ='[31m[1A'     /*                          */ ;
green  ='[32m[1A'     /*                          */ ;
yellow ='[33m[1A'     /*                          */ ;
blue   ='[34m[1A'     /*                          */ ;

call banner             /* Display header           */ ;

Accum_Hours    = 0 ;
Accum_Minutes  = 0 ;
Accum_Seconds  = 0 ;
Total_Hours    = 0 ;
Total_Minutes  = 0 ;
Total_Seconds  = 0 ;
Total_Sessions = 0 ;

count         = 0 ;
Curr_Year     = '' ;
Curr_Month    = '' ;
spacer        = 'COPIES'(' ',12) ;
Session_Count = 0 ;

do while 'LINES'(logfile) ;
   parse value 'LINEIN'(logfile) with logdate logtime logtext ;

   if   'POS'('DISCONNECTED AFTER','TRANSLATE'(logtext)) = 0,
   then nop ;
   else call Process_Session ;

end ;

if   count = 0,
then if   Session_Count = 0  /* Empty connect log */,
     then nop ;
     else do /* All log sessions within one month */ ;
             Total_Sessions = Total_Sessions + Session_Count ;
             count = count + 1 ;
             data.count = logyear Word_Month(Curr_Month),
                          'RIGHT'(Accum_Hours,2,'0')':',
                       || 'RIGHT'(Accum_Minutes,2,'0')':',
                       || 'RIGHT'(Accum_Seconds,2,'0'),
                          'Sessions:' 'RIGHT'(Session_Count,5) ;
          end ;
else do         /* Pick up the final values */ ;
        Total_Sessions = Total_Sessions + Session_Count ;
        count = count + 1 ;
        data.count = '    ' Word_Month(Curr_Month),
                     'RIGHT'(Accum_Hours,2,'0')':',
                  || 'RIGHT'(Accum_Minutes,2,'0')':',
                  || 'RIGHT'(Accum_Seconds,2,'0'),
                     'Sessions:' 'RIGHT'(Session_Count,5) ;
     end ;

/* Close the log file */ ;
call 'STREAM' logfile, 'C', 'CLOSE' ;

say black ;

/* Write out the summary */ ;
do i = 1 to count by 1 ;
   say 'CENTER'(spacer data.i,twidth) ;
end i ;

say 'CENTER'('             ---------------------------------',twidth) ;

lline = spacer 'Total....' || 'RIGHT'(Total_Hours,2,'0')':',
                           || 'RIGHT'(Total_Minutes,2,'0')':',
                           || 'RIGHT'(Total_Seconds,2,'0'),
                              'Sessions:' 'RIGHT'(Total_Sessions,5) ;

say red ;
say 'CENTER'(lline,twidth) ;
say blue ;

return ;
/*=================================================================*/ ;
/*-  Section: banner  ---------------------------------------------*
 |                                                                 |
 | Display the header.                                             |
 |                                                                 |
 *-----------------------------------------------------------------*/ ;

banner:

"echo on" ;
"prompt $p$E[0;"34";"47";"4";"4"m]" ;
"echo off" ;
"cls" ;

frame1 = '+' || 'COPIES'('=',56) || '+' ;
frame2 = '+' || 'COPIES'(' ',56) || '+' ;
text   = '+ ILOG.CMD - V1.1 WARP Internet Connection Times Summary +' ;

say ;
say 'CENTER'(frame1,swidth) ;
say 'CENTER'(frame2,swidth) ;
say 'CENTER'(text,swidth)   ;
say 'CENTER'(frame2,swidth) ;
say 'CENTER'(frame1,swidth) ;
say ;

return ;
/*=================================================================*/ ;
/*-  Section: Process_Session  ------------------------------------*
 |                                                                 |
 | Process a session summary line.                                 |
 |                                                                 |
 *-----------------------------------------------------------------*/ ;

Process_Session:

procedure expose logdate logtime logtext data. count Curr_Year spacer,
                 Curr_Month Accum_Hours Accum_Minutes Accum_Seconds,
                 Total_Hours Total_Minutes Total_Seconds Session_Count,
                 Total_Sessions logyear ;

if   'LENGTH'(logdate) = 5  /* Short date format */,
then do ;
        parse var logdate logmonth '/' logday ;

        logyear = 1995 ;
     end ;
else parse var logdate logyear '/' logmonth '/' logday ;

parse var logtext . . loghours ':' logminutes ':' logseconds . ;

if   Curr_Year = logyear,
then if   Curr_Month = logmonth,
     then do ;
             Session_Count = Session_Count + 1 ;

             call Add_To_Current_Accumulators ;
          end ;
     else do ;
             Total_Sessions = Total_Sessions + Session_Count ;
             count = count + 1 ;

             if   count = 1,
             then data.count = Curr_Year Word_Month(Curr_Month),
                               'RIGHT'(Accum_Hours,2,'0')':',
                            || 'RIGHT'(Accum_Minutes,2,'0')':',
                            || 'RIGHT'(Accum_Seconds,2,'0'),
                               'Sessions:' 'RIGHT'(Session_Count,5) ;
             else data.count = '    ' Word_Month(Curr_Month),
                               'RIGHT'(Accum_Hours,2,'0')':',
                            || 'RIGHT'(Accum_Minutes,2,'0')':',
                            || 'RIGHT'(Accum_Seconds,2,'0'),
                               'Sessions:' 'RIGHT'(Session_Count,5) ;

             Curr_Month = logmonth ;

             Accum_Hours   = 0 ;
             Accum_Minutes = 0 ;
             Accum_Seconds = 0 ;
             Session_Count = 1 ;

             call Add_To_Current_Accumulators ;
          end ;
else if   Curr_Year = '',
     then do ;
             Session_Count = Session_Count + 1 ;

             Curr_Year  = logyear ;
             Curr_Month = logmonth ;

             call Add_To_Current_Accumulators ;
          end ;
     else do ;
             Total_Sessions = Total_Sessions + Session_Count ;
             count = count + 1 ;

             data.count = Curr_Year Word_Month(Curr_Month),
                          'RIGHT'(Accum_Hours,2,'0')':',
                       || 'RIGHT'(Accum_Minutes,2,'0')':',
                       || 'RIGHT'(Accum_Seconds,2,'0')':' ;
                          'Sessions:' 'RIGHT'(Session_Count,5) ;

             Accum_Hours   = 0 ;
             Accum_Minutes = 0 ;
             Accum_Seconds = 0 ;
             Session_Count = 1 ;

             call Add_To_Current_Accumulators ;
          end ;

return ;
/*=================================================================*/ ;
/*-  Section: Add_To_Current_Accumulators  ------------------------*
 |                                                                 |
 | Add session time to current year/month values.                  |
 |                                                                 |
 *-----------------------------------------------------------------*/ ;

Add_To_Current_Accumulators:

procedure expose Accum_Hours Accum_Minutes Accum_Seconds loghours,
                 logminutes logseconds Total_Hours Total_Minutes,
                 Total_Seconds ;

Accum_Seconds = Accum_Seconds + logseconds ;

if   Accum_Seconds > 59,
then do ;
        Accum_Seconds = Accum_Seconds - 60 ;
        Accum_Minutes = Accum_Minutes + 1 ;

        if   Accum_Minutes > 59,
        then do ;
                Accum_Minutes = Accum_Minutes - 60 ;
                Accum_Hours   = Accum_Hours + 1 ;
             end ;
        else nop ;

     end ;
else nop ;

Accum_Minutes = Accum_Minutes + logminutes ;

if   Accum_Minutes > 59,
then do ;
        Accum_Minutes = Accum_Minutes - 60 ;
        Accum_Hours   = Accum_Hours + 1 ;
     end ;
else nop ;

Accum_Hours = Accum_Hours + loghours ;

Total_Seconds = Total_Seconds + logseconds ;

if   Total_Seconds > 59,
then do ;
        Total_Seconds = Total_Seconds - 60 ;
        Total_Minutes = Total_Minutes + 1 ;

        if   Total_Minutes > 59,
        then do ;
                Total_Minutes = Total_Minutes - 60 ;
                Total_Hours   = Total_Hours + 1 ;
             end ;
        else nop ;

     end ;
else nop ;

Total_Minutes = Total_Minutes + logminutes ;

if   Total_Minutes > 59,
then do ;
        Total_Minutes = Total_Minutes - 60 ;
        Total_Hours   = Total_Hours + 1 ;
     end ;
else nop ;

Total_Hours = Total_Hours + loghours ;

return ;
/*=================================================================*/ ;
/*-  Section: Word_Month  -----------------------------------------*
 |                                                                 |
 | Convert numeric months to text.                                 |
 |                                                                 |
 *-----------------------------------------------------------------*/ ;

Word_Month:

procedure ;

arg nmonth ;

select ;

   when nmonth = 1  then tmonth = 'Jan' ;
   when nmonth = 2  then tmonth = 'Feb' ;
   when nmonth = 3  then tmonth = 'Mar' ;
   when nmonth = 4  then tmonth = 'Apr' ;
   when nmonth = 5  then tmonth = 'May' ;
   when nmonth = 6  then tmonth = 'Jun' ;
   when nmonth = 7  then tmonth = 'Jul' ;
   when nmonth = 8  then tmonth = 'Aug' ;
   when nmonth = 9  then tmonth = 'Sep' ;
   when nmonth = 10 then tmonth = 'Oct' ;
   when nmonth = 11 then tmonth = 'Nov' ;

   otherwise             tmonth = 'Dec' ;

end ;

return tmonth ;
/*=================================================================*/ ;

/* End of EXEC */ ;
