/* ********************************************************************* */
/*   By: Jetnick Enterprise                                              */
/*       Don E. Groves, Jr.                                              */
/*   Contact Information:                                                */
/*     E-mail: jetnick@erols.com                                         */
/*        CIS: 71310,3702                                                */
/* Date: 18 Sep 1996                                                     */
/* ********************************************************************* */
/* public Classes and Routines                                           */
/*  .MyTimeZone    Factory for producing TimeStamps                      */
/*  ParseDateStr( inStrDate )                                            */
/*         Convert a Message Date String into a TimeStamp.               */
/* ********************************************************************* */
/*  This file REQUIRES DateTime.CMD to function.                         */
/* ********************************************************************* */
/* This file is primary intented use is as a ::REQUIRES type file.       */
/* ********************************************************************* */
/*  General Documentation for other Programmer type Users                */
/*                                                                       */
/*  Class TimeStamp a Subclass of DateTime                               */
/*                                                                       */
/*   This is a results object from MyTimeZone class/object methods.      */
/*     {New version cannot be produce except from copies of the object   */
/*      or by the MyTimeZone methods.}                                   */
/*                                                                       */
/*       Inherates all methods from DateTime.                            */
/*                                                                       */
/*   Overrides:                                                          */
/*     String        Parameters: (none)                                  */
/*       Returns: A message(DATE:) fully formatted including UTC offset. */
/*                                                                       */
/*     compareBy return UTC useful in generic sorts and compares.        */
/*                                                                       */
/*   Adds the following:                                                 */
/*     UtcOffSet Attribute     {Offset in Seconds from UTC}              */
/*                                                                       */
/*     UTC      Parameters: (none)                                       */
/*       Returns:  NTS value adjusted to UTC                             */
/*                                                                       */
/*     ToUTC    Parameters: (none)                                       */
/*       Returns: a new DateTime object equal to this objects            */
/*               DateTime expressed in  UTC timezone                     */
/*                                                                       */
/*     Tz1      Parameters: (none)                                       */
/*       Returns: String UTC offset. Useful in Message(Date:) fields.    */
/*                                                                       */
/*     MakeLocalTimeStamp Parameters: (none)                             */
/*       Returns: A copy of this object adjusted to the Local time Zone. */
/*                                                                       */
/*     MakeTimeStamp      Parameters: (none)                             */
/*       Returns: a copy of this object.                                 */
/*                                                                       */
/*     MakeUtcTimeStamp   Parameters: (none)                             */
/*       Returns: A copy of this object adjusted to UTC time Zone.       */
/*                                                                       */
/* ********************************************************************* */
/*                                                                       */
/*   Class MyTimeZone  a public Factory for creating TimeStamp objects   */
/*      Has all methods of the TimeStamp with one difference             */
/*     All CLASS members are also Instances methods.                     */
/*                                                                       */
/*   Only methods with a difference parameter sequences are:             */
/*    TZ          Returns current TZ Object.                             */
/*                TZ(string)  sets MyTimeZone~TZ to new value.           */
/*                                                                       */
/*    NEW         Returns a handle to a .MyTimeZone object.              */
/*                                                                       */
/*    ParseDateStr(Message Date String)                                  */
/*       Returns a TimeZone object built from the String                 */
/*       or .NIL on invalid Message Date String.                         */
/*                                                                       */
/*    UtcToLocal( DT )                                                   */
/*      Parameter is a required DateTime object respenting a UTC time    */
/*      Returns: a Local TimeStamp object built from the object.         */
/*                                                                       */
/*    MakeLocalTimeStamp                                                 */
/*      If parameter is supplied then same as UtcToLocal                 */
/*      else return a TimeZone object built from the current Clock.      */
/*                                                                       */
/*    All other methods have the following first parameter format.       */
/*      Paramter(optional) is DateTime for which to build the            */
/*      new object from else the current Clock is used.                  */
/*                                                                       */
/*    TimeStamp         Returns a TimeStamp Object.                      */
/*                                                                       */
/*    MakeTimeStamp     Returns a TimeStamp Object.                      */
/*                                                                       */
/* ********************************************************************* */
/*                                                                       */
/*  class TZ       This is the TimeZone information object returned by   */
/*                 MyTimeZone method TZ.                                 */
/*                                                                       */
/*       Following methods are documented as available:                  */
/*                                                                       */
/*    String       returns objects TZ string respentation                */
/*                                                                       */
/*    MakeString   {same as above}                                       */
/*                                                                       */
/*    Report       Returns multi-record textual string detailing         */
/*                 interpetation of TZ input string.                     */
/*                 {each record marked by LF ('0A'x) characters}         */
/*                                                                       */
/* ********************************************************************* */
/*  All other classes defined in this file arenot visable outside of it. */
/* ********************************************************************* */
/*                                                                       */
/* ********************************************************************* */
/* ********************************************************************* */
parse source . invhow myName
RtCode = 0
SELECT
 WHEN invhow == 'FUNCTION'
 THEN RtCode = .MyTimeZone
 WHEN invhow == 'COMMAND'
 THEN DO
    IF Arg(1)~length \= 0
    THEN DO
       elsp = .Elasp_time~NEW
       .OUTPUT~LINEOUT('Start report time:'~' '(elsp))
       .OUTPUT~LINEOUT('')
    END
    .OUTPUT~LINEOUT('Current MessDate is:'~' '(.MyTimeZone))
    .OUTPUT~LineOut(.MyTimeZone~TZ~Report)
    IF Arg(1)~length \= 0
    THEN DO
       .OUTPUT~LINEOUT('')
       .OUTPUT~LINEOUT('Total report time:'~' '(elsp))
    END
    RtCode = 1
 END
 WHEN invhow == 'SUBROUTINE'
  THEN nop  /* happens when '::requires MyTimeZone' or 'call MyTimeZone' is used */
 OTHERWISE
   nop /* I've no idea, but then maybe everything will work out OK anyway. */
END
Return RtCode

::requires DateTime

 /* This is a Factory for creating 'TimeStamp' objects. */
 /* All public 'Method's take an optional DateTime object. */
 /* Except */
 /*   NEW          {no parameters} */
 /*   ParseDateStr A String repesenting the Date, Time and TZ to use. */
 /*   UtcToLocal   A DateTime in UTC to make a local TimeStamp from. */
::class MyTimeZone subclass Clock public

::method TZ       CLASS     /* Returns current TZ Object */
  use arg zt
  IF ARG() = 0              /* If no arguments to get-in and get-out fast */
  THEN forward to (.TimeZoneS)
return .TimeZoneS~TZ(zt~MakeString)   /* LOCK MyTimeZone while update taking place. */
::method TZ                /* Returns copy of current TZ Object */
  forward to (Self~Class)

::method ArgHelper private class
return Arg(2,'A')   /* Strip off one Argument and return the remaining. */

::method UNKNOWN CLASS      /* How some of the magic is handled */
  use arg cMsg, ARGS
    /* Give a reasonable Error message for Assignment attempts. */
  IF cMsg~RIGHT(1) == '='
  THEN RAISE syntax 97.1 array(Self~OBJECTNAME,cMsg)
    /* Create a 'TimeStamp' with the possible argument. */
  forward continue message 'TimeStamp' ARGUMENTS (ARGS)
  hrult = RESULT   /* save the resulting object */
  forward continue message 'ArgHelper' ARGUMENTS (ARGS)
    /* Now let the first resulting object handle the message */
    /* and any remaining arguments. */
  forward to (hrult) message (cMsg) ARGUMENTS ( RESULT )

::method MakeDateTime CLASS  /* Create a DateTime object from an optional */
  if arg(1,'O')            /* DateTime object. If no arg then use current */
  THEN forward class (super) /* DateTime. {Sounds silly, but the combo happens.} */
forward to (Arg(1)) ARGUMENTS ( .ARRAY~NEW )
          /* Also allows this object to handle copies of itself, while */
          /* maintaining the same syntax for all methods, */

::method TimeStamp CLASS
  use arg dt      /* Create a TimeStamp from an optional DateTime object */
  if arg(1,'O')
  then dt = .Clock    /* if no arg use current time */
forward to (.TimeZoneS) Array (dt~MakeDateTime, .False )
::method TimeStamp           /* Make instances look the same */
  forward to (Self~class)
::method MakeTimeStamp CLASS
  forward message 'TimeStamp'
::method MakeTimeStamp       /* Make instances look the same */
  forward to (Self~class)

::method UtcToLocal Class  /* converts a UtC DateTime object to Local Time */
  use arg DT
  if arg(1,'O')
  THEN RAISE syntax 93.901 array(1)
forward to (.TimeZoneS) message 'TimeStamp' Array (dt~MakeDateTime, .True )
::method UtcToLocal
  forward to (Self~class)

::method MakeLocalTimeStamp CLASS
  if arg(1,'O')
  THEN forward message 'TimeStamp'
  forward message 'UtcToLocal'
::method MakeLocalTimeStamp  /* Make instances look the same */
  forward to (Self~class)

/* ****************************************** */
/* Parses a Message Date String Data          */
/*  Returns                                   */
/*   a TimeStamp object built from the string */
/*   or .NIL on error                         */
/* ****************************************** */
::method ParseDateStr class
  use arg inStrDate
  IF ARG(1,'E')
  THEN DO
     inStrDate = inStrDate~MakeString
     tz = ''
     parse value inStrDate with dowt day month year tm tz TRASH
     if dowt~datatype('N')
     THEN DO
        tz = ''
        parse value inStrDate with day month year tm tz TRASH
        if tm~length = 8
        THEN dowt = 'm,'   /* flag that it is ok to proceed */
        ELSE DO
           if tm~length = 5
           THEN DO
              parse value tm with hh ':' mm
              if hh~datatype('W')
              THEN if hh < 24
              then if mm~datatype('W')
              then if mm < 60
              THEN DO
                 tm = tm~''(':00')
                 dowt = 'm,'  /* flag that it is ok to proceed */
              end
           END
        END
     END
     if dowt~pos(',') \= 0
     then do
        IF year < 100
        THEN year= 1900 + year
        IF year < 1970
        THEN year= 100 + year
        /* Trap possible invalid Date and/or Time information. */
        SIGNAL ON SYNTAX name pDateTrap
        strDate = DATE('N',day month year,'N')
        IF DATE('B',strDate,'N') >= Self~NTS_Base
        THEN DO
           strTime = TIME('N',tm,'N')
           forward to (.TimeZoneS) Array (strDate, strTime, tz)
        END
     end
  end
pDateTrap:
return .nil   /* invalid Date Time Stamp String */
::method ParseDateStr        /* Make instances look the same */
  forward to (Self~class)

  /* This is the results object from MyTimeZone class/object methods */
  /* Yes it's a DateTime type object. */
::class TimeStamp Subclass DateTime
::method UtcOffSet Attribute     /* Seconds from UTC */
::method init
  use arg DT, UtcOffSet
  forward continue class (Super) Array ( dt )
  Self~UtcOffSet = UtcOffSet
return Self
::method UTC       /* Returns NTS normalized to UTC. */
return Self~Nts - Self~UtcOffSet
::method ToUTC     /* Returns a new DateTime object adjusted to UTC. */
  forward to (.DateTime) message 'NEW' Array ( Self~UTC )
::method compareBy     /* override to use Normalized value */
  forward message 'UTC'
::method Tz1       /* String UTC offset */
  tz1 = 'UTC'
  wrk = Self~UtcOffSet
  IF wrk~ABS \= 0
  THEN DO
     IF wrk < 0
     then tz1 = '-'
     else tz1 = '+'
     wrk = wrk~Abs
     tz1 = tz1~''('00'~''(wrk % 3600)~Right(2))
     wrk = wrk // 3600
     tz1 = tz1~''('00'~''(wrk % 60)~Right(2))
  END
forward to (tz1) message 'MakeString'
::method String          /* A message DATE: fully formatted */
return (Self~String:Super)~' '(Self~tz1)
::method MessDate        /* A message DATE: fully formatted */
  forward message 'String'
::method MakeLocalTimeStamp
  forward to (.MyTimeZone) message 'UtcToLocal' array (Self~ToUtc)
::method MakeTimeStamp
  forward to (.TimeStamp) message 'NEW' array (Self~Nts,Self~UtcOffSet)
::method MakeUtcTimeStamp
  forward to (.TimeStamp) message 'NEW' array (Self~Utc,0)

  /* Holds the Start/END information of DayLight Savings */
  /* used by .TZ class to pass infomation internally. */
::class TzDli_Data
::method init
  expose Month Week Day rTime
  use arg Month, Week, Day, rTime
return Self
::method Month
  expose Month Week Day rTime
return Month
::method Week
  expose Month Week Day rTime
return Week
::method Day
  expose Month Week Day rTime
return Day
::method rTime
  expose Month Week Day rTime
return rTime

::class TzDli subclass TzDli_Data
::method String
return (Self~Month)~''(',')~''(Self~Week)~''(',')~''(Self~Day)~''(',')~''(Self~rTime)
::method MakeString
  forward message 'String'
::method UNKNOWN  /* let the string respentation handle anything else */
  forward to (Self~String) message (arg(1)) ARGUMENTS ( arg(2) )
::method MakeTzDli
return .TzDli~New(Self~Month, Self~Week, Self~Day, Self~rTime)
::method Find
  use arg DT
  wrk = (DT~Date('S')~Left(4))~''('00'~''(Self~Month)~Right(2))~''('01')
  wrk = .DateTime~NEW(DATE('N',wrk,'S'),TIME('N',Self~rTime,'S'))
  SecPerDay  = wrk~SecPerDay  /* a handy constant */
  SELECT
   WHEN Self~WEEK > 0   /* from Start of Month */
   THEN DO
      IF Self~WEEK > 1
      THEN  wrk = wrk + ((Self~WEEK - 1) * (SecPerDay * 7))
      IF wrk~Dow \= Self~DAY
      THEN wrk = wrk + ((7 - wrk~Dow) * SecPerDay)
   END
   WHEN Self~WEEK < 0  /* from End of Month */
   THEN DO
      wrk = wrk + (27 * SecPerDay) /* get near the end of the month */
      mth = wrk~Date('M')
      Do until wrk~Date('M') \== mth
        wrk = wrk + SecPerDay
      END
      wrk = wrk - SecPerDay
      /* wrk now equals last day of month */
      IF Self~WEEK~ABS > 1
      THEN wrk = wrk - ((Self~WEEK~ABS - 1) * (SecPerDay * 7))
      IF wrk~Dow \= Self~DAY
      THEN wrk = wrk - ((wrk~Dow - Self~Day) * SecPerDay)
   END
   OTHERWISE DO /* Fixed Day of Month */
      wrk = wrk + (( Self~Day - 1 ) * SecPerDay)
   END
  END
return wrk
::METHOD TzDli CLASS
  use arg mh,wk,day,zt
  DlS = .nil
  IF mh > 0 & mh < 13 & zt >= 0 & zt < (24 * 3600) & wk~ABS >= 0 & wk~ABS < 5
  THEN IF ( wk = 0 & day > 0 ) | (wk \= 0 & wk~ABS < 5 & day >= 0 & day < 7)
  THEN DlS = .TzDli~NEW( mh, wk, day, zt )
return DlS

 /* The Decoded TZ information get stored in one of theses */
::class TZ_Data
::method init
  expose DlS STD difDlS UtcOffSet DlSStart DlSEnd
  use arg DlS, STD, UtcOffSet, difDlS, DlSStart, DlSEnd
return Self
::method DlS        /* This Local TimeZone String daylight Savings */
  expose DlS STD difDlS UtcOffSet DlSStart DlSEnd
forward to (DlS) message 'MakeString'
::method STD        /* This Local TimeZone String Standard Time */
  expose DlS STD difDlS UtcOffSet DlSStart DlSEnd
forward to (StD) message 'MakeString'
::method difDlS     /* Difference between daylight Savings and Standard. */
  expose DlS STD difDlS UtcOffSet DlSStart DlSEnd
forward to (difDlS) message 'MakeString'
::method UtcOffSet  /* Seconds from UTC standard */
  expose DlS STD difDlS UtcOffSet DlSStart DlSEnd
forward to (UtcOffSet) message 'MakeString'
::method DlSStart   /* Start datedata of Daylight Savings information */
  expose DlS STD difDlS UtcOffSet DlSStart DlSEnd
return DlSStart
::method DlSEnd     /* End datedata of Daylight Savings information */
  expose DlS STD difDlS UtcOffSet DlSStart DlSEnd
return DlSEnd

::class TZ subclass TZ_Data
::method TimeStamp               /* returns a TimeStamp built from DT */
  use arg DT, who
  UtcOffSet = Self~UtcOffSet
  difDlS    = Self~difDlS
  IF difDlS \= 0
  THEN DO     /* daylight Savings is observed. */
     wrk    = Self~DlSStart~Find(DT)~Nts
     EndWrk = Self~DlSEnd~Find(DT)~Nts
     IF who
     THEN DO  /* Is it a UTC refence object? */
        wrk    = wrk    - UtcOffSet
        Endwrk = Endwrk - UtcOffSet
     END
     Nts    = DT~Nts
     IF ((wrk < Nts) & (endWrk > Nts))    /* DayLight Savings Time */
     THEN UtcOffSet = UtcOffSet + difDlS
  END
  IF who     /* Is it a UTC refence object? */
  THEN DT = DT + UtcOffSet    /* Then adjust to local time zone. */
forward to (.TimeStamp) message 'New' Array(DT, UtcOffSet)
::method MakeTZ
return .TZ~NEW(Self~DlS, Self~STD, Self~UtcOffSet, Self~difDlS, Self~DlSStart~MakeTzDli, Self~DlSEnd~MakeTzDli)
::method String          /* return current TZ string respentation */
  secoff = Self~UtcOffSet
  IF secOff \= 0
  THEN DO
     IF (secoff~Abs // 3600) \= 0
     then hrs = TIME('N',secoff~ABS,'S')~Left(5)
     else hrs = secoff~Abs % 3600
     hrs = hrs~strip('L','0')
     IF secoff > 0
     then hrs = '-'~''(hrs)
     ELSE hrs = '+'~''(hrs)
  END
  ELSE hrs = '0'
return Self~STD~''(hrs)~''(Self~DlS)~''(',')~''(Self~DlSStart)~''(',')~''(Self~DlSEnd)~''(',')~''(Self~difDlS)
::method MakeString
  forward message 'String'
::method TZ               /* Returns current TZ string respentation */
  forward message 'String'
::method UNKNOWN  /* let the string respentation handle anything else */
  forward to (Self~String) message (Arg(1)) ARGUMENTS (ARG(2))
::method Report
  forward to (.Formatter~New) Array (Self)
/* ******************************************************************** */
/* TZ = ,array~of(EST+5EDT,4,1,0,3600,10,-1,0,7200,3600)                */
/*                                                                      */
/* TZ[1]  = 'EST+5EDT'  {EST +5 EDT}                                    */
/* TZ[2]  = 4      month                                                */
/* TZ[3]  = 1      starting week or 0                                   */
/* TZ[4]  = 0      dayofweek(0-6) or start day if 3 was zero            */
/* TZ[5]  = 3600   starting time in seconds local                       */
/* TZ[6]  = 10     starting month                                       */
/* TZ[7]  = -1     starting week of winter time (see 3 for details)     */
/* TZ[8]  = 0      dayofweek(0-6) or start day if 7 was zero            */
/* TZ[9]  = 7200   winter starting time in seconds local                */
/* TZ[10] = 3600   Offset differences                                   */
/*                                                                      */
/* TZ[1]                                                                */
/*  Std  3 Characters (must begin with a letter {space is legal after}) */
/* ---                                                                  */
/* of   sign {optional}     West is Plus East is minus                  */
/* utc   x numbers and or ':'   {H}H {: MM {: SS}}                      */
/* ---                                                                  */
/*  dls  3 Characters (must begin with a letter {space is legal after}) */
/*                                                                      */
/* default:                                                             */
/*    SET TZ=EST+5EDT,4,1,0,3600,10,-1,0,7200,3600                      */
/* ******************************************************************** */
::METHOD TZ  CLASS
  Use arg EST5EDT
  IF ARG(1,'O')
  THEN EST5EDT = ''
  ELSE EST5EDT = EST5EDT~MakeString
  argsGood = .False
  UtcOffSet = 'ERR'
  DLSSTD = ''
  parse value EST5EDT with tz1','DLSSTD
  if tz1~length > 0
  THEN DO
     StD = TZ1~LEFT(3)
     tz1 = TZ1~SubStr(4)
     lnum = ''
     Do while '0123456789+-:'~pos(tz1~LEFT(1)) \= 0
       lnum = lnum~''(tz1~LEFT(1))
       tz1 = tz1~Substr(2)
     END
     IF lnum~Length \= 0
     THEN DO
        DlS = tz1~LEFT(3)
        tsign = 1
        IF tz1~left(1) = '-'
        THEN DO
           tsign = -1
           lnum = lnum~SubStr(2)
        END
        IF lnum~POS(':') \= 0
        THEN DO
           parse value lnum with hr':'mm':'.
           IF hr~''(mm)~DataType('N')
           THEN UtcOffSet = 3600 * hr + 60 * mm
        END
        ELSE IF Lnum~DataType('N')
        THEN UtcOffSet = 3600 * lnum
        IF UtcOffSet~DataType('N')
        THEN IF UtcOffSet > ( 12 * 3600 )
        THEN UtcOffSet = 'ERR'
        ELSE DO
           UtcOffSet = UtcOffSet * tSign
           UtcOffSet = -1 * UtcOffSet
           IF DLSSTD~LENGTH > 0
           THEN DO
              /* decode the long string */
              tz = .ARRAY~NEW
              DO i = 1 to 9 WHILE DLSSTD~LENGTH > 0
                parse value DLSSTD with ln','DLSSTD
                IF ln~DataType('N')
                THEN tz[i] = ln
                else LEAVE
              END
              IF TZ~Items = 9 & DLSSTD~LENGTH = 0
              THEN DO
                 DlSStart =  .TzDli~TzDli( TZ[1], TZ[2], TZ[3], TZ[4] )
                 IF DlSStart \= .nil
                 THEN DO
                    DlSEnd = .TzDli~TzDli( TZ[5], TZ[6], TZ[7], TZ[8] )
                    IF DlSEnd \= .nil & TZ[9] >= 0 & TZ[9] < (24 * 3600)
                    THEN DO
                       difDlS   = TZ[9]
                       argsGood = .TRUE
                    END
                 END
              END
           END
        END
     END
  END
  IF \ UtcOffSet~DataType('N')
  THEN DO
     DlS = 'EDT'
     STD = 'EST'
     UtcOffSet = (-5 * 3600)
  END
  IF \ argsGood
  THEN DO
     difDlS   = 3600
     DlSStart = .TzDli~NEW( 4, 1,0,3600)
     DlSEnd   = .TzDli~NEW(10,-1,0,7200)
  END
return .TZ~NEW(DlS, STD, UtcOffSet, difDlS, DlSStart, DlSEnd)

::class Formatter_Buf           /* Helper object to format the report */
::method init                   /* for a TZ object method Report. */
  expose buf
  buf = ''
return Self
::method CharOut private
  expose buf
  use arg cln
  buf = buf~''(cln~MakeString)
return Self
::method EofSignal private    /* return the buffer as a string */
  expose buf
forward to (buf) message 'MakeString' Arguments ( .Array~New )
::class Formatter subclass Formatter_Buf
::method Fmtnum private
  use arg num
  pre = '   '
  forward message 'CharOut' Array ((pre)~''(num)~right(3))
::method FmtMonth  private
  use arg month
  Self~Fmtnum(month)~CharOut(' ')
forward message 'CharOut' Array (DATE('M','96/'~''('00'~''(Month)~Right(2))~''('/01'),'O'))
::method FmtDay  private
  use arg Day
  Self~Fmtnum(day)~CharOut(' ')
forward message 'CharOut' Array (DATE('W',.DateTime~NTS_Base + 3 + Day,'B'))
::method RepTime private
  use arg secoff
  hrs = ' '
  IF Arg(2,'O')
  THEN DO
     Select
      when secoff > 0 then hrs = '+'
      when secoff < 0 then hrs = '-'
      OTHERWISE NOP
    END
  END
  Self~CharOut(hrs)~CharOut(TIME('N',secoff~ABS,'S'))
forward message 'CharOut' Array ((' (')~''(secoff)~''(')'))
::method Tittle private
  use arg what
forward message 'CharOut' Array (('0A'x)~''('  ')~''(what))
::method RptMain private
  use arg what
forward message 'Tittle' Array ('  '~''(what)~''(' ')~Left(31,'.')~''(' = '))
::method SubHeader private
  use arg what
forward message 'RptMain' Array ('  '~''(what))
::method Subline private
  use arg what
forward message 'SubHeader' Array ('  '~''(what))
::method ReptHelper private
  use arg dls, who
  Self~SubHeader((who)~' '('information'))~CharOut(dls)
  Self~Subline('Month')~FmtMonth(dls~Month)
  IF dls~week \= 0
  THEN DO
     Self~SubLine('Week of month')~FmtNum(dls~Week)
     Self~CharOut(' from the ')
     IF dls~week > 0
     THEN Self~CharOut('beginning')
     ELSE Self~CharOut('end')
     Self~CharOut(' of month')
     Self~SubLine('Day of week')~FmtDay(dls~Day)
  END
  ELSE Self~SubLine('Fixed day of month')~FmtNum(dls~Day)
  Self~SubLine('Time of day')~RepTime(dls~rTime,1)
  Self~SubLine((who)~''('s this year'))
forward message 'CharOut' Array (dls~Find(.clock))
::method Report        /* format the Report and return. */
  use arg TZ           /* a TZ object */
  Self~Tittle('TZ information report')
  Self~RptMain('TZ repesentation')~CharOut(TZ)
  Self~RptMain('Standard string')~CharOut(TZ~STD)
  Self~RptMain('Offset from UTC (seconds)')~RepTime(TZ~UtcOffSet)
  IF TZ~difDlS \= 0
  THEN DO
     Self~RptMain('Daylight savings string')~CharOut(TZ~DlS)
     Self~RptMain('Daylight savings is')~CharOut('Observed')
     Self~reptHelper(TZ~DLSStart, 'Start')
     Self~reptHelper(TZ~DLSEnd  , 'End'  )
     Self~SubHeader('Difference is')~RepTime(TZ~difDlS)
  END
  ELSE Self~RptMain('Daylight savings is')~CharOut('NOT observed')
  Self~Tittle('End of report '~Left(76,'.'))
forward message 'EofSignal'

 /* Static local TimeZone Table holder. */
 /* Lets the constant information be built only once per session. */
 /* Used by: .MyTimeZone class */
 /*  To confussing to make these Expose(s) part of the class proper. */
::class TimeZoneS
::method init class
  expose Gmtime. TzCur TzDef
  rpErr = .nil
  Gmtime.    = rpErr
  Gmtime.ERR = rpErr
  Gmtime.GMT = 0
  Gmtime.UTC = 0
  Gmtime.UT  = 0
  Gmtime.ADT = (-03 * 3600 )
  Gmtime.AST = (-04 * 3600 )
  Gmtime.EDT = (-04 * 3600 )
  Gmtime.EST = (-05 * 3600 )
  Gmtime.CDT = (-05 * 3600 )
  Gmtime.CST = (-06 * 3600 )
  Gmtime.MDT = (-06 * 3600 )
  Gmtime.MST = (-07 * 3600 )
  Gmtime.PDT = (-07 * 3600 )
  Gmtime.PST = (-08 * 3600 )
  TzCur      = .TZ~TZ(VALUE('TZ',,'OS2ENVIRONMENT'))
  TzDef      = .nil
return Self

::Method TZ  class    /* always returns a copy */
  expose Gmtime. TzCur TzDef
  IF ARG(1,'E')
  THEN TzCur = .TZ~TZ(ARG(1))
forward to (TzCur) message 'MakeTZ' Arguments ( .Array~new)

::Method TimeStamp class     /* for internal MyTimeZone.CMD useage only */
  expose Gmtime. TzCur TzDef
forward to (TzCur)

::Method ParseDateStr class     /* for internal MyTimeZone.CMD useage only */
  expose Gmtime. TzCur TzDef
  use arg strDate, strTime, zt
  UtcOffSet = Self~TimeZ(zt)
  if UtcOffSet \= .nil   /* was TZ valid */
  THEN DO
     DT = .DateTime~NEW(strDate,strTime)
     IF \ UtcOffSet~dataType('N')
     THEN DO             /* If no timeZone inf use default */
        IF TzDef = .nil      /* Delay initalization of TzDef until needed */
        THEN TzDef = .TZ~TZ
        forward to (TzDef) message 'TimeStamp' Array ( DT, .False )
     END
     forward to (.TimeStamp) message 'New' Array (DT, UtcOffSet)
  END
return .nil

::Method timeZ private  class /* returns offset field converted to seconds */
  expose Gmtime. TzCur TzDef
  use arg zt
  /* check timezone stamp */
  zt=zt~strip  /* strip any leading or trailing white space */
  Select
   WHEN zt~length = 0 THEN zt = '*'  /* Flag that ZT was blank */
   WHEN zt~datatype('W') = 1
   then DO
      if zt \= 0
      then DO
         ztt  = zt~substr(2,2)
         zttm = zt~substr(4,2)
         ztts = 1
         SELECT
          WHEN zt~substr(1,1) == '-' then ztts = -1
          WHEN zt~substr(1,1) == '+' then NOP
          OTHERWISE do
             ztt = zt~substr(1,2)
             zttm = zt~substr(3,2)
          END
         END
         if zttm~strip~length = 0
         then zttm = 0
         zt = ztts * (( ztt * 3600 ) + ( zttm * 60 ))
         if zt~abs > (12 * 3600) | zt~DataType('W') = 0
         then zt = GmTime.ERR
      end
   END
   OTHERWISE zt = GmTime.[zt~Translate]
  END
return zt

/* ****************************************************************** */
/* Parses a Message Date String Data                                  */
/*  Returns                                                           */
/*   a TimeStamp object built from the string                         */
/*   or .NIL on error                                                 */
/* {Hey, I first though I wanted a function type interface for this.} */
/* ****************************************************************** */
::ROUTINE ParseDateStr public
  use arg inStrDate
return .MyTimeZone~ParseDateStr(inStrDate)

/* ************************************ */
/* EOF MyTimeStame.CMD                  */
/* ************************************ */

