IMPLEMENTATION MODULE FtpdSession;

        (********************************************************)
        (*                                                      *)
        (*       Part of the ftp server - handles a session     *)
        (*                                                      *)
        (*  Programmer:         P. Moylan                       *)
        (*  Started:            21 August 1997                  *)
        (*  Last edited:        6 June 2000                     *)
        (*  Status:             OK                              *)
        (*                                                      *)
        (********************************************************)


FROM SYSTEM IMPORT ADDRESS;

IMPORT ChanConsts, RndFile, IOChan, Strings;

FROM OS2 IMPORT
    (* const*)  FERR_DISABLEHARDERR,
    (* proc *)  DosError;

FROM Storage IMPORT
    (* proc *)  ALLOCATE, DEALLOCATE;

FROM FtpCommands IMPORT
    (* type *)  Session,
    (* proc *)  SetTransactionLogLevel, OpenSession, CloseSession,
                HandleCommand, KillDataChannel;

FROM LoggedOnUsers IMPORT
    (* proc *)  ClientAddressAcceptable;

FROM FtpTransfers IMPORT
    (* proc *)  NotifyMaxUsers, SendMessageFile;

FROM Sockets IMPORT
    (* type *)  Socket, SockAddr,
    (* proc *)  send, recv, soclose, so_cancel;

FROM TransLog IMPORT
    (* type *)  TransactionLogID,
    (* proc *)  CreateLogID, LogTransaction, DiscardLogID;

FROM Conversions IMPORT
    (* proc *)  CardinalToString;

FROM InetUtilities IMPORT
    (* proc *)  ConvertCard, AddEOL, IPToString;

FROM SplitScreen IMPORT
    (* proc *)  WriteStringAt;

FROM Queues IMPORT
    (* type *)  Queue,
    (* proc *)  CreateQueue, DestroyQueue, AddToQueue, TakeFromQueue;

FROM Semaphores IMPORT
    (* type *)  Semaphore,
    (* proc *)  CreateSemaphore, DestroySemaphore, Signal;

FROM LowLevel IMPORT
    (* proc *)  EVAL;

FROM Timer IMPORT
    (* proc *)  Sleep, TimedWait;

FROM TaskControl IMPORT
    (* type *)  Lock,
    (* proc *)  CreateLock, DestroyLock, Obtain, Release, CreateTask,
                CreateTask1, TaskExit;

(************************************************************************)

TYPE
    (* Data used in creating a new instance of the session handler task. *)

    NewSessionPointer = POINTER TO
                           RECORD
                               socket: Socket;
                               IPAddress: CARDINAL;
                           END (*RECORD*);

    (* Data needed by the timeout checker task. *)

    KeepAlivePointer = POINTER TO
                           RECORD
                               SocketOpen, dying: BOOLEAN;
                               sem: Semaphore;
                               socket: Socket;
                               session: Session;
                               TimedOut: BOOLEAN;
                           END (*RECORD*);

(************************************************************************)

VAR
    (* A queue of KeepAlive records, which are passed to the    *)
    (* TimeoutChecker tasks as they are created.                *)

    KeepAliveQueue: Queue;

    (* Maximum allowed number of simultaneous users. *)

    MaxUsers: CARDINAL;

    (* Timeout delay, in milliseconds. *)

    MaxTime: CARDINAL;

    (* Count of active users, and a lock to protect it. *)

    UserCount: CARDINAL;
    UserCountLock: Lock;

    (* Flag to control whether we write the number of users to the screen. *)

    LogToScreen: BOOLEAN;

    (* Version number. *)

    version: ARRAY [0..15] OF CHAR;

(************************************************************************)

PROCEDURE SetVersion (v: ARRAY OF CHAR);

    (* Stores the version number. *)

    BEGIN
        Strings.Assign (v, version);
    END SetVersion;

(************************************************************************)

PROCEDURE SetTransactionLogging (level: CARDINAL);

    (* Option to control transaction logging: 0 for none, 1 for disk,   *)
    (* 2 for screen, 3 for both.                                        *)

    BEGIN
        IF level > 3 THEN
            level := 3;
        END (*IF*);
        LogToScreen := level > 1;
        SetTransactionLogLevel (level);
    END SetTransactionLogging;

(************************************************************************)

PROCEDURE SetMaxUsers (limit: CARDINAL);

    (* Sets an upper limit on the number of simultaneous users. *)

    BEGIN
        MaxUsers := limit;
        NotifyMaxUsers (limit);
    END SetMaxUsers;

(************************************************************************)

PROCEDURE SetTimeout (seconds: CARDINAL);

    (* Specifies how long a session can be idle before it is forcibly   *)
    (* closed.                                                          *)

    BEGIN
        IF seconds > MAX(CARDINAL) DIV 1000 THEN
            MaxTime := MAX(CARDINAL);
        ELSE
            MaxTime := 1000*seconds;
        END (*IF*);
    END SetTimeout;

(********************************************************************************)

PROCEDURE UpdateCount (increment: INTEGER): CARDINAL;

    (* Updates the count of the number of users, and returns the new    *)
    (* count.  Special case: if this would take us beyond the MaxUsers  *)
    (* limit, then the count is not updated and the returned value      *)
    (* is zero.                                                         *)

    VAR value, pos: CARDINAL;  Buffer: ARRAY [0..15] OF CHAR;

    BEGIN
        Obtain (UserCountLock);
        IF increment > 0 THEN INC (UserCount, increment);
        ELSIF increment < 0 THEN DEC (UserCount, -increment)
        END (*IF*);
        value := UserCount;
        IF value > MaxUsers THEN
            DEC (UserCount, increment);  value := 0;
        ELSIF LogToScreen THEN
            pos := 0;
            ConvertCard (value, Buffer, pos);
            Buffer[pos] := ' ';  INC(pos);
            Buffer[pos] := CHR(0);
            WriteStringAt (0, 69, Buffer);
        END (*IF*);
        Release (UserCountLock);
        RETURN value;
    END UpdateCount;

(************************************************************************)

PROCEDURE NumberOfUsers(): CARDINAL;

    (* Returns the number of users who are currently logged on. *)

    BEGIN
        RETURN UpdateCount(0);
    END NumberOfUsers;

(************************************************************************)

PROCEDURE TimeoutChecker;

    (* A new instance of this task is created for each client session.  *)
    (* It kills the corresponding SessionHandler task if more than      *)
    (* MaxTime milliseconds have passed since the last Signal() on the  *)
    (* sessions KeepAlive semaphore.                                    *)

    (* This is a workaround.  I would have preferred to set the         *)
    (* timeout in the socket options, but I haven't yet figured out     *)
    (* how to do it.  An older version of IBM's sockets documentation   *)
    (* gave details on the send and receive timeouts, but this seems    *)
    (* to have disappeared from later versions of the documentation.    *)

    VAR p: KeepAlivePointer;

    BEGIN
        p := TakeFromQueue (KeepAliveQueue);
        REPEAT
            TimedWait (p^.sem, MaxTime, p^.TimedOut);
        UNTIL p^.TimedOut OR p^.dying;
        IF p^.SocketOpen THEN
            KillDataChannel (p^.session);
            so_cancel (p^.socket);
        END (*IF*);

        (* Wait for the socket to be closed. *)

        WHILE p^.SocketOpen DO
            Sleep (500);
        END (*WHILE*);
        DestroySemaphore (p^.sem);
        DISPOSE (p);

    END TimeoutChecker;

(************************************************************************)

PROCEDURE SessionHandler (arg: ADDRESS);

    (* The task that handles a client session, i.e. this is where all the real  *)
    (* work is done.  There might be several instances of this task running,    *)
    (* one for each session that is still open.                                 *)

    CONST CR = CHR(13);  LF = CHR(10);

    VAR S: Socket;
        CmdBuffer: ARRAY [0..511] OF CHAR;

        (* Temporary buffer for BuildCommand. *)

        TmpBuffer: ARRAY [0..127] OF CHAR;
        tmppos, tmplength: CARDINAL;  ch: CHAR;

    (********************************************************************)

    PROCEDURE BuildCommand(): BOOLEAN;

        VAR length: CARDINAL;
            IACreceived, IgnoreNext: BOOLEAN;

        BEGIN
            length := 0;
            IACreceived := FALSE;  IgnoreNext := FALSE;
            LOOP
                IF tmppos >= tmplength THEN
                    tmplength := recv (S, TmpBuffer, SIZE(TmpBuffer), 0);
                    IF (tmplength = MAX(CARDINAL)) OR (tmplength = 0) THEN
                        RETURN FALSE;
                    END (*IF*);
                    tmppos := 0;
                END (*IF*);
                ch := TmpBuffer[tmppos];  INC(tmppos);

                (* This next section skips over Telnet control codes (which we  *)
                (* don't really want to know about).  A Telnet control code is  *)
                (* two or three bytes long, where the first byte is CHR(255).   *)

                IF IgnoreNext THEN
                    IgnoreNext := FALSE;
                ELSIF IACreceived THEN
                    IACreceived := FALSE;
                    IF ORD(ch) = 255 THEN
                        IF length < SIZE(CmdBuffer) THEN
                            CmdBuffer[length] := ch;  INC(length);
                        END (*IF*);
                    ELSIF ORD(ch) > 250 THEN
                        IgnoreNext := TRUE;
                    END (*IF*);
                ELSIF ORD(ch) = 255 THEN
                    IACreceived := TRUE;

                (* Command should end with CR LF, but for simplicity we'll      *)
                (* ignore the CR.                                               *)

                ELSIF ch = CR THEN  (* Do nothing *)
                ELSIF ch = LF THEN
                    IF length < SIZE(CmdBuffer) THEN
                        CmdBuffer[length] := CHR(0);
                    END (*IF*);
                    RETURN TRUE;
                ELSIF length < SIZE(CmdBuffer) THEN
                    CmdBuffer[length] := ch;  INC(length);
                END (*IF*);

            END (*LOOP*);

        END BuildCommand;

    (********************************************************************)

    VAR NSP: NewSessionPointer;
        Sess: Session;
        UserNumber, j: CARDINAL;  KA: KeepAlivePointer;
        KeepAliveSemaphore: Semaphore;
        Quit: BOOLEAN;
        IPBuffer: ARRAY [0..16] OF CHAR;
        LogID: TransactionLogID;
        LogMessage: ARRAY [0..127] OF CHAR;

    BEGIN                   (* Body of SessionHandler *)

        DosError (FERR_DISABLEHARDERR);
        Sess := NIL;
        NSP := arg;
        S := NSP^.socket;

        (* Create the log file ID for this session. *)

        CardinalToString (S, IPBuffer, 6);  IPBuffer[6] := CHR(0);
        LogID := CreateLogID (IPBuffer);

        (* Initial transaction log message. *)

        IPToString (NSP^.IPAddress, TRUE, IPBuffer);
        Strings.Assign ("New client ", LogMessage);
        Strings.Append (IPBuffer, LogMessage);
        LogTransaction (LogID, LogMessage);

        (* Check for acceptable client address. *)

        IF NOT ClientAddressAcceptable (NSP^.IPAddress) THEN
            Strings.Assign ("Client address rejected ", LogMessage);
            Strings.Append (IPBuffer, LogMessage);
            LogTransaction (LogID, LogMessage);
            DiscardLogID (LogID);
            CmdBuffer := "421 Connection refused";
            j := AddEOL (CmdBuffer);
            EVAL (send (S, CmdBuffer, j, 0));
            EVAL (soclose (S));
            DISPOSE (NSP);
            TaskExit;
        END (*IF*);

        UserNumber := UpdateCount (+1);

        (* Check for too many users. *)

        IF UserNumber = 0 THEN
            LogTransaction (LogID, "Too many users");
            DiscardLogID (LogID);
            CmdBuffer := "421 User limit exceeded, try again later";
            j := AddEOL (CmdBuffer);
            EVAL (send (S, CmdBuffer, j, 0));
            EVAL (soclose (S));
            DISPOSE (NSP);
            TaskExit;
        END (*IF*);

        (* Create the session information structure. *)

        CreateSemaphore (KeepAliveSemaphore, 0);
        Sess := OpenSession (S, UserNumber, NSP^.IPAddress, LogID, KeepAliveSemaphore);
        DISPOSE (NSP);

        (* Create an instance of the TimeoutChecker task. *)

        NEW (KA);
        WITH KA^ DO
            SocketOpen := TRUE;  socket := S;
            dying := FALSE;
            sem := KeepAliveSemaphore;
            session := Sess;
            TimedOut := FALSE;
        END (*WITH*);
        AddToQueue (KeepAliveQueue, KA);
        CreateTask (TimeoutChecker, 3, "ftpd timeout");

        (* Send the "welcome" message. *)

        SendMessageFile (S, "Welcome0.MSG", "220", UserNumber, NIL, FALSE);
        CmdBuffer := "220 FtpServer ";
        Strings.Append (version, CmdBuffer);
        Strings.Append (" ready", CmdBuffer);
        j := AddEOL (CmdBuffer);
        Quit := send (S, CmdBuffer, j, 0) = MAX(CARDINAL);

        (* Here's the main command processing loop.  We leave it when the client  *)
        (* issues a QUIT command, or when socket communications are lost, or      *)
        (* when we get a timeout on the KeepAlive semaphore.                      *)

        tmppos := 0;  tmplength := 0;
        LOOP
            IF Quit THEN EXIT(*LOOP*) END(*IF*);
            IF BuildCommand() THEN
                Signal (KeepAliveSemaphore);
                HandleCommand (Sess, CmdBuffer, Quit);
            ELSE
                EXIT (*LOOP*);
            END (*IF*);
        END (*LOOP*);

        (* Work out whether the session was terminated by a QUIT, or a timeout, *)
        (* or a communications failure.                                         *)

        IF KA^.TimedOut THEN
            LogTransaction (LogID, "Timed out");
        ELSIF NOT Quit THEN
            LogTransaction (LogID, "Session aborted");
        END (*IF*);

        CloseSession (Sess);
        KA^.dying := TRUE;  Signal (KA^.sem);
        soclose (S);
        KA^.SocketOpen := FALSE;
        EVAL (UpdateCount(-1));
        DiscardLogID (LogID);
        TaskExit;

    (*
    EXCEPT
        LogTransaction (LogID, "SessionHandler detected exception.");
        CloseSession (Sess);
        UserNumber := UpdateCount (-1);
        soclose(S);
        TaskExit;
    *)

    END SessionHandler;

(********************************************************************************)

PROCEDURE NewSession (S: Socket;  addr: SockAddr);

    (* Starts and runs a client session.  The session runs in a separate        *)
    (* thread; this procedure returns after starting the session, it does not   *)
    (* wait until the session is over.                                          *)

    VAR NSP: NewSessionPointer;

    BEGIN
        NEW (NSP);
        WITH NSP^ DO
            socket := S;  IPAddress := addr.in_addr.addr;
        END (*WITH*);
        CreateTask1 (SessionHandler, 3, "ftpd session", NSP);
    END NewSession;

(********************************************************************************)
(*                            MODULE INITIALISATION                             *)
(********************************************************************************)

BEGIN
    CreateLock (UserCountLock);
    Obtain (UserCountLock);  UserCount := 0;  Release (UserCountLock);
    CreateQueue (KeepAliveQueue);
    LogToScreen := FALSE;
FINALLY
    DestroyLock (UserCountLock);
    DestroyQueue (KeepAliveQueue);
END FtpdSession.

