MODULE LogAnalysis;

        (*****************************************************)
        (*        Analysis of FtpServer user log file        *)
        (*                                                   *)
        (*     Programmer:    P. Moylan                      *)
        (*     Started:       25 August 1999                 *)
        (*     Last edited:   27 August 1999                 *)
        (*     Status:        Working                        *)
        (*                                                   *)
        (*****************************************************)

IMPORT Strings, STextIO, TextIO, IOChan, IOConsts, ChanConsts, SeqFile, ProgramArgs;

FROM Storage IMPORT
    (* proc *)  ALLOCATE, DEALLOCATE;

(********************************************************************************)

CONST
    Nul = CHR(0);  CtrlZ = CHR(26);

TYPE
    LogLine = ARRAY [0..511] OF CHAR;
    FilenameString = ARRAY [0..255] OF CHAR;
    TimeString = ARRAY [0..18] OF CHAR;
    LineType = (eof, blank, starttime, endtime, upload, partupload, download,
                partdownload, delete, other);

    PhraseType = ARRAY LineType OF ARRAY [0..15] OF CHAR;
    UserDataPtr = POINTER TO UserDataRecord;
    FileListPtr = POINTER TO FilenameRecord;

    UserDataRecord = RECORD
                         next: UserDataPtr;
                         StartTime, EndTime: TimeString;
                         LoginName: LogLine;
                         uploadlist, downloadlist,
                           parttranslist, deletedlist: FileListPtr;
                     END (*RECORD*);

    FilenameRecord = RECORD
                         next: FileListPtr;
                         name: FilenameString;
                         count: CARDINAL;
                     END (*RECORD*);


(********************************************************************************)

CONST
    keyphrase = PhraseType {"", "", "Logged in", "Finished", "Uploaded",
                            "Tried to put", "Downloaded", "Tried to get",
                            "Deleted", ""};

    NullTime = "0000-00-00 00:00:00";
    MaxTime = "9999-12-31 23:59:59";

(********************************************************************************)

VAR
    LogfileName: FilenameString;
    SessionList: UserDataPtr;
    StartTime, EndTime: TimeString;

(********************************************************************************)
(*                                 MISCELLANEOUS                                *)
(********************************************************************************)

PROCEDURE SWriteCard (val: CARDINAL);

    (* Screen output of a cardinal number. *)

    BEGIN
        IF val > 9 THEN
            SWriteCard (val DIV 10);  val := val MOD 10;
        END (*IF*);
        STextIO.WriteChar (CHR(ORD('0')+val));
    END SWriteCard;

(********************************************************************************)

PROCEDURE SWriteRJCard (val, places: CARDINAL);

    (* Screen output of a cardinal number, right justified so as to take up     *)
    (* 'places' character positions.                                            *)

    BEGIN
        IF places = 0 THEN
            (* Can't do anything sensible. *)
        ELSIF val < 10 THEN
            WHILE places > 1 DO
                STextIO.WriteChar (' ');  DEC(places);
            END (*WHILE*);
            STextIO.WriteChar (CHR(ORD('0')+val MOD 10));
        ELSIF places = 1 THEN
            STextIO.WriteChar ('*');
        ELSE
            SWriteRJCard (val DIV 10, places-1);
            STextIO.WriteChar (CHR(ORD('0')+val MOD 10));
        END (*IF*);
    END SWriteRJCard;

(********************************************************************************)
(*                          OPERATIONS ON A FILE LIST                           *)
(********************************************************************************)

PROCEDURE MergeIntoList (VAR (*INOUT*) master: FileListPtr;
                                       VAR (*IN*) name: FilenameString);

    (* Adds "name" into the master list, either by inserting a new list entry   *)
    (* or incrementing the count field of an existing entry.                    *)

    VAR previous, current, p: FileListPtr;  comparison: Strings.CompareResults;

    BEGIN
        (* Find a suitable insertion point. *)

        previous := NIL;  current := master;  comparison := Strings.greater;
        LOOP
            IF current = NIL THEN EXIT(*LOOP*) END(*IF*);
            comparison := Strings.Compare (name, current^.name);
            IF comparison <> Strings.greater THEN EXIT(*LOOP*) END(*IF*);
            previous := current;  current := current^.next;
        END (*LOOP*);

        (* Now do the insertion. *)

        IF comparison = Strings.equal THEN
            INC (current^.count);
        ELSE
            NEW (p);
            p^.count := 1;
            p^.name := name;

            (* Insert between previous and current. *)

            IF previous = NIL THEN
                master := p;
            ELSE
                previous^.next := p;
            END (*IF*);
            p^.next := current;

        END (*IF*);

    END MergeIntoList;

(********************************************************************************)

PROCEDURE UpdateFileList (VAR (*INOUT*) master: FileListPtr;  p: FileListPtr);

    (* Merges the p^ file list into the master list.  *)

    BEGIN
        WHILE p <> NIL DO
            MergeIntoList (master, p^.name);
            p := p^.next;
        END (*WHILE*);
    END UpdateFileList;

(********************************************************************************)

PROCEDURE MakeFileList (p: UserDataPtr;
                        VAR (*OUT*) result1, result2, result3, result4: FileListPtr);

    (* Creates four lists of files, with counts, from the user data list. *)

    BEGIN
        result1 := NIL;  result2 := NIL;
        result3 := NIL;  result4 := NIL;
        WHILE p <> NIL DO
            UpdateFileList (result1, p^.uploadlist);
            UpdateFileList (result2, p^.downloadlist);
            UpdateFileList (result3, p^.parttranslist);
            UpdateFileList (result4, p^.deletedlist);
            p := p^.next;
        END (*WHILE*);
    END MakeFileList;

(********************************************************************************)

PROCEDURE WriteFileList (p: FileListPtr);

    (* Puts a summary to standard output. *)

    VAR total: CARDINAL;

    BEGIN
        IF p = NIL THEN
            STextIO.WriteString ("  (None)");
            STextIO.WriteLn;
            RETURN;
        END (*IF*);
        total := 0;
        REPEAT
            INC (total, p^.count);
            SWriteRJCard (p^.count, 8);  STextIO.WriteString ("  ");
            STextIO.WriteString (p^.name);  STextIO.WriteLn;
            p := p^.next;
        UNTIL p = NIL;
        STextIO.WriteString ("A total of ");  SWriteCard (total);
        STextIO.WriteString (" file transfers.");  STextIO.WriteLn;
    END WriteFileList;

(********************************************************************************)

PROCEDURE DiscardFileList (VAR (*INOUT*) p: FileListPtr);

    (* Disposes of the linked list. *)

    VAR q: FileListPtr;

    BEGIN
        WHILE p <> NIL DO
            q := p^.next;
            DISPOSE (p);
            p := q;
        END (*WHILE*);
    END DiscardFileList;

(********************************************************************************)
(*                   OPERATIONS ON THE USER DATA RECORDS                        *)
(********************************************************************************)

PROCEDURE EmptyUserRecord (VAR (*IN*) R: UserDataRecord): BOOLEAN;

    (* Returns TRUE iff R contains no transfer data. *)

    BEGIN
        RETURN (R.uploadlist = NIL) AND (R.downloadlist = NIL)
                   AND (R.parttranslist = NIL) AND (R.deletedlist = NIL);
    END EmptyUserRecord;

(********************************************************************************)

PROCEDURE DiscardUserData (VAR (*INOUT*) p: UserDataPtr);

    (* Disposes of the linked list. *)

    VAR q: UserDataPtr;

    BEGIN
        WHILE p <> NIL DO
            q := p^.next;
            DiscardFileList (p^.uploadlist);
            DiscardFileList (p^.downloadlist);
            DiscardFileList (p^.parttranslist);
            DiscardFileList (p^.deletedlist);
            DISPOSE (p);
            p := q;
        END (*WHILE*);
    END DiscardUserData;

(********************************************************************************)
(*                        OPERATIONS ON THE LOGFILE LINES                       *)
(********************************************************************************)

PROCEDURE Classify (VAR (*INOUT*) buffer: ARRAY OF CHAR): LineType;

    (* Takes one log line, works out what sort of record it is.  As a           *)
    (* side-effect, we also strip the header keyword from the line.             *)

    VAR pos: CARDINAL;  found: BOOLEAN;  result: LineType;
        key: ARRAY [0..15] OF CHAR;

    BEGIN
        result := other;
        IF buffer[0] = CtrlZ THEN result := eof
        ELSIF buffer[0] = Nul THEN result := blank
        ELSE
            Strings.FindNext (':', buffer, 0, found, pos);
            IF found THEN
                Strings.Extract (buffer, 0, pos, key);
                result := MIN(LineType);
                WHILE NOT Strings.Equal (key, keyphrase[result])
                                 AND (result <> other) DO
                    INC (result);
                END (*WHILE*);
                IF result <> other THEN
                    REPEAT
                        INC (pos);
                    UNTIL buffer[pos] <> ' ';
                    Strings.Delete (buffer, 0, pos);
                END (*IF*);
            END (*IF*);
        END (*IF*);
        RETURN result;
    END Classify;

(********************************************************************************)
(*                       READING THE DATA FROM THE LOG FILE                     *)
(********************************************************************************)

PROCEDURE ReadOneLine (cid: IOChan.ChanId;  VAR (*OUT*) buffer: ARRAY OF CHAR);

    (* Input of a single line of data. *)

    BEGIN
        TextIO.ReadString (cid, buffer);
        IF IOChan.ReadResult(cid) = IOConsts.endOfInput THEN
            buffer[0] := CtrlZ;  buffer[1] := Nul;
        ELSE
            TextIO.SkipLine (cid);
        END (*IF*);
    END ReadOneLine;

(********************************************************************************)

PROCEDURE ExtractFileName (VAR (*IN*) buffer: LogLine;
                           VAR (*OUT*) result: FilenameString);

    VAR pos: CARDINAL;  found: BOOLEAN;

    BEGIN
        Strings.Assign (buffer, result);
        Strings.FindNext (' ', result, 0, found, pos);
        IF found THEN
            result[pos] := Nul;
        END (*IF*);
    END ExtractFileName;

(********************************************************************************)

PROCEDURE ReadOneGroup (cid: IOChan.ChanId): UserDataPtr;

    (* Handles the data from one client session. *)

    VAR buffer: LogLine;  result: UserDataPtr;  flp: FileListPtr;

    BEGIN
        NEW (result);
        WITH result^ DO
            next := NIL;
            StartTime := NullTime;  EndTime := NullTime;
            LoginName[0] := Nul;
            downloadlist := NIL;
        END (*WITH*);
        REPEAT
            ReadOneLine (cid, buffer);
        UNTIL buffer[0] <> Nul;
        LOOP
            CASE Classify(buffer) OF
              | eof:
                    EXIT (*LOOP*);
              | blank:
                    EXIT (*LOOP*);
              | starttime:
                    Strings.Assign (buffer, result^.StartTime);
              | endtime:
                    Strings.Assign (buffer, result^.EndTime);
              | upload:
                    NEW (flp);
                    flp^.next := result^.uploadlist;
                    ExtractFileName (buffer, flp^.name);
                    result^.uploadlist := flp;
              | partupload:
                    NEW (flp);
                    flp^.next := result^.parttranslist;
                    ExtractFileName (buffer, flp^.name);
                    result^.parttranslist := flp;
              | download:
                    NEW (flp);
                    flp^.next := result^.downloadlist;
                    ExtractFileName (buffer, flp^.name);
                    result^.downloadlist := flp;
              | partdownload:
                    NEW (flp);
                    flp^.next := result^.parttranslist;
                    ExtractFileName (buffer, flp^.name);
                    result^.parttranslist := flp;
              | delete:
                    NEW (flp);
                    flp^.next := result^.deletedlist;
                    ExtractFileName (buffer, flp^.name);
                    result^.deletedlist := flp;
              | other:
                    Strings.Assign (buffer, result^.LoginName);
            ELSE
                STextIO.WriteString ("Fault in header type");
                STextIO.WriteLn;
                EXIT (*LOOP*);
            END (*CASE*);
            ReadOneLine (cid, buffer);
        END (*LOOP*);
        IF EmptyUserRecord (result^) THEN
            DISPOSE (result);
        END (*IF*);
        RETURN result;
    END ReadOneGroup;

(********************************************************************************)

PROCEDURE ReadTheData(): UserDataPtr;

    (* Returns a linked list of all client data. *)

    VAR cid: IOChan.ChanId;  res: ChanConsts.OpenResults;
        head, client, tail: UserDataPtr;

    BEGIN
        head := NIL;  tail := NIL;
        StartTime := MaxTime;  EndTime := NullTime;
        SeqFile.OpenRead (cid, LogfileName, SeqFile.read+SeqFile.text, res);
        IF res = ChanConsts.opened THEN
            REPEAT
                client := ReadOneGroup (cid);
                IF client <> NIL THEN
                    IF Strings.Compare (client^.StartTime, StartTime) = Strings.less THEN
                        StartTime := client^.StartTime;
                    END (*IF*);
                    IF Strings.Compare (client^.EndTime, EndTime) = Strings.greater THEN
                        EndTime := client^.EndTime;
                    END (*IF*);
                    client^.next := NIL;
                    IF tail = NIL THEN head := client
                    ELSE tail^.next := client
                    END (*IF*);
                    tail := client;
                END (*IF*);
            UNTIL client = NIL;
            SeqFile.Close (cid);
        END (*IF*);
        RETURN head;
    END ReadTheData;

(********************************************************************************)
(*                      WRITING A REPORT TO STANDARD OUTPUT                     *)
(********************************************************************************)

PROCEDURE WriteSummary (clientlist: UserDataPtr);

    VAR uplist, downlist, faillist, deletelist: FileListPtr;

    BEGIN
        STextIO.WriteString ("FTP summary for period ");
        STextIO.WriteString (StartTime);
        STextIO.WriteString (" to ");
        STextIO.WriteString (EndTime);
        STextIO.WriteLn;
        STextIO.WriteString ("-----------------------------------------------------------------");
        STextIO.WriteLn;
        MakeFileList (clientlist, uplist, downlist, faillist, deletelist);
        DiscardUserData (clientlist);

        STextIO.WriteLn;  STextIO.WriteString ("UPLOADS");  STextIO.WriteLn;
        WriteFileList (uplist);
        DiscardFileList (uplist);

        STextIO.WriteLn;  STextIO.WriteString ("DOWNLOADS");  STextIO.WriteLn;
        WriteFileList (downlist);
        DiscardFileList (downlist);

        STextIO.WriteLn;  STextIO.WriteString ("FAILED TRANSFERS");  STextIO.WriteLn;
        WriteFileList (faillist);
        DiscardFileList (faillist);

        STextIO.WriteLn;  STextIO.WriteString ("DELETIONS");  STextIO.WriteLn;
        WriteFileList (deletelist);
        DiscardFileList (deletelist);

    END WriteSummary;

(********************************************************************************)
(*                              INITIALISATION                                  *)
(********************************************************************************)

PROCEDURE GetLogfileName;

    (* Picks up program argument from the command line. *)

    CONST DefaultLogfileName = "FTPUSERS.LOG";

    VAR j: CARDINAL;
        args: IOChan.ChanId;

    BEGIN
        args := ProgramArgs.ArgChan();
        IF ProgramArgs.IsArgPresent() THEN

            TextIO.ReadString (args, LogfileName);

            (* Remove leading and trailing spaces from the result. *)

            j := 0;
            WHILE LogfileName[j] = ' ' DO
                INC(j);
            END (*WHILE*);
            IF j > 0 THEN
                Strings.Delete (LogfileName, 0, j);
            END (*IF*);
            j := LENGTH(LogfileName);
            LOOP
                IF j = 0 THEN EXIT(*LOOP*) END(*IF*);
                DEC (j);
                IF LogfileName[j] <> ' ' THEN EXIT(*LOOP*) END(*IF*);
                LogfileName[j] := Nul;
            END (*LOOP*);

        END (*IF*);

        (* If no file name supplied, use the default name. *)

        IF LogfileName[0] = Nul THEN
            LogfileName := DefaultLogfileName;
        END (*IF*);

    END GetLogfileName;

(********************************************************************************)

BEGIN
    GetLogfileName;
    SessionList := ReadTheData();
    WriteSummary(SessionList);
END LogAnalysis.

