IMPLEMENTATION MODULE FtpTransfers;

        (********************************************************)
        (*                                                      *)
        (*          FTP server: file operations                 *)
        (*                                                      *)
        (*  Programmer:         P. Moylan                       *)
        (*  Started:            24 August 1997                  *)
        (*  Last edited:        14 September 1999               *)
        (*  Status:             OK                              *)
        (*                                                      *)
        (********************************************************)

FROM SYSTEM IMPORT CARD16, LOC, ADDRESS, ADR, CAST;

IMPORT FileSys, IOChan, ChanConsts, IOConsts, RndFile, Strings, SysClock;

FROM TimeConv IMPORT
    (* proc *)  millisecs;

FROM LowLevel IMPORT
    (* proc *)  EVAL;

FROM Storage IMPORT
    (* proc *)  ALLOCATE, DEALLOCATE;

FROM Sockets IMPORT
    (* const*)  NotASocket,
    (* type *)  Socket, AddressFamily, SocketType, SockAddr,
    (* proc *)  socket, connect, send, recv, soclose, setsockopt,
                bind, listen, getsockname, accept,
                getpeername, so_cancel;

FROM Internet IMPORT
    (* const*)  Zero8, INADDR_ANY;

FROM InetUtilities IMPORT
    (* proc *)  Swap2, Swap4, WriteError, Synch, OpenLogFile,
                CloseLogFile, IPToString, ConvertDecimal, WaitForDataSocket,
                AppendString, ConvertCard, ConvertCardZ, AddEOL;

FROM NameLookup IMPORT
    (* proc *)  StartNameLookup, CancelNameLookup, GetName;

FROM FDUsers IMPORT
    (* type *)  User, UserCategory, FName, ListingOption, ListingOptions,
    (* proc *)  ReadUserData, DestroyUserData, PasswordAcceptable,
                MakeFName, DiscardFName, SetWorkingDirectory,
                ListDirectory, FileExists,
                HaveSeenMessage, OpenForReading, OpenForWriting, OpenForAppend,
                RemoveFile, MayListFiles, CanReadFile, CanWriteFile, CanDeleteFile,
                RenameTo, CanDeleteDirectory, RemoveDirectory, GetDateTime,
                CreateDirectory, GetFileSize, NameOfCurrentDirectory,
                PermissionString, MakeFullName,
                SpaceAvailable, GetUserNumber;

FROM FDFiles IMPORT
    (* proc *)  FWriteChar, FWriteString, FWriteLn, FWriteCard, FWriteLJCard,
                FWriteZCard, SetFileSize;

FROM Keyboard IMPORT
    (* proc *)  NotDetached;

FROM Semaphores IMPORT
    (* type *)  Semaphore,
    (* proc *)  Signal;

FROM OS2 IMPORT
    (* proc *)  DosError, DosSleep,
    (* const*)  FERR_DISABLEHARDERR;

(********************************************************************************)

CONST MillisecondsPerDay = FLOAT (24 * 60 * 60 * 1000);

TYPE
    FileNameString = ARRAY [0..255] OF CHAR;

    (* The LogEntry type is used for user logging. *)

    Operation = (upload, partupload, download, partdownload, delete);

    LogEntryPointer = POINTER TO LogEntry;

    LogEntry = RECORD
                   next: LogEntryPointer;
                   what: Operation;
                   bytes: CARDINAL;
                   duration: REAL;
                   filename: FileNameString;
               END (*RECORD*);

    (* The fields in the ClientFileInfo record have the following meanings.     *)
    (*   user            The data structure that keeps track of the user        *)
    (*                   permissions.                                           *)
    (*   UserNum         User number.  Has no significance except as the        *)
    (*                    identification number to be used in the welcome       *)
    (*                    message.                                              *)
    (*   KickMe          A semaphore.  The session will time out unless we      *)
    (*                    do a Signal on it every so often.                     *)
    (*   iname           File name to be used for the next transfer operation.  *)
    (*   flags           Options for a directory listing                        *)
    (*   CommandSocket   Socket used for the command channel.                   *)
    (*   ServerName      A SockAddr structure for the default server-side       *)
    (*                     data port.                                           *)
    (*   DataPort        The remote client address to use for data transfers.   *)
    (*   SpeedLimit      Upper limit on data transfer rate, in bytes per second.*)
    (*                     Must be nonzero.                                     *)
    (*   ASCIITransfer   TRUE iff current transfer type is ASCII                *)
    (*   StreamMode      TRUE iff current transfer mode is stream               *)
    (*   PassiveOperation  TRUE iff we've entered Passive Mode                  *)
    (*   IsManager       TRUE if this user is a manager                         *)
    (*   AnonUser        TRUE iff user password is e-mail address               *)
    (*   FileStructure   File structure code, ignored at present.  In fact I    *)
    (*                    don't think this will ever have any use, so maybe     *)
    (*                    this field can be eliminated.                         *)
    (*   RestartPoint    Place to resume an interrupted transfer.               *)
    (*   AmountToAllocate  Prespecified file size.                              *)
    (*   Log             Saved details for user logging.                        *)
    (*                                                                          *)
    (* The Log record includes the following fields:                            *)
    (*   StartTime       Time when the user logged in                           *)
    (*   InCount         Count of the amount of data this user has uploaded     *)
    (*                    during this session.                                  *)
    (*   OutCount        Like InCount, but it counts outgoing data.             *)
    (*   UserName        User identification                                    *)
    (*   entrycount      Number of entries in the transaction record list       *)
    (*   first, last     Head and tail of a linked list of transaction records  *)

    ClientFileInfo = POINTER TO ClientInfoRecord;
    ClientInfoRecord = RECORD
                           user: User;
                           UserNum: CARDINAL;
                           KickMe: Semaphore;
                           iname: FName;
                           flags: ListingOptions;
                           CommandSocket: Socket;
                           ServerName: SockAddr;
                           DataPort: RECORD
                                         socket: Socket;
                                         host: CARDINAL;
                                         port: CARD16;
                                     END (*RECORD*);
                           SpeedLimit: CARDINAL;
                           ASCIITransfer, StreamMode: BOOLEAN;
                           PassiveOperation: BOOLEAN;
                           IsManager, AnonUser: BOOLEAN;
                           FileStructure: CHAR;
                           RestartPoint, AmountToAllocate: CARDINAL;
                           Log: RECORD
                                    ClientIP: CARDINAL;
                                    StartTime: SysClock.DateTime;
                                    UserName: ARRAY [0..31] OF CHAR;
                                    entrycount: CARDINAL;
                                    first, last: LogEntryPointer;
                                END (*RECORD*);
                       END (*RECORD*);

(********************************************************************************)

CONST NUL = CHR(0);  CR = CHR(13);  LF = CHR(10);  CtrlZ = CHR(26);

VAR MaxUsers: CARDINAL;
    LogLevel: CARDINAL;
    FreeSpaceThreshold: CARDINAL;
    ScreenEnabled: BOOLEAN;

(********************************************************************************)
(*                             USER LOGGING                                     *)
(********************************************************************************)

PROCEDURE SetLogLevel (level: CARDINAL);

    (* Sets the amount of logging we want. *)

    BEGIN
        LogLevel := level;
    END SetLogLevel;

(********************************************************************************)

PROCEDURE FWriteTime (cid: IOChan.ChanId;  Time: SysClock.DateTime);

    BEGIN
        FWriteZCard (cid, Time.year, 4);  FWriteChar (cid, '-');
        FWriteZCard (cid, Time.month, 2);  FWriteChar (cid, '-');
        FWriteZCard (cid, Time.day, 2);  FWriteChar (cid, ' ');
        FWriteZCard (cid, Time.hour, 2);  FWriteChar (cid, ':');
        FWriteZCard (cid, Time.minute, 2);  FWriteChar (cid, ':');
        FWriteZCard (cid, Time.second, 2);
    END FWriteTime;

(********************************************************************************)

PROCEDURE WriteLog (SS: ClientFileInfo);

    (* Writes the details for this session to the user log file. *)

    VAR cid: IOChan.ChanId;  Now: SysClock.DateTime;
        current: LogEntryPointer;  buffer: ARRAY [0..16] OF CHAR;
        NodeName: FileNameString;  rate: CARDINAL;

    BEGIN
        cid := OpenLogFile();
        WITH SS^.Log DO

            (* Login details. *)

            FWriteLn (cid);
            FWriteString (cid, UserName);  FWriteString (cid, "  ");
            IPToString (ClientIP, buffer);
            FWriteString (cid, buffer);

            GetName (ClientIP,  NodeName);
            IF NodeName[0] <> CHR(0) THEN
                FWriteString (cid, " ");
                FWriteString (cid, NodeName);
            END (*IF*);
            FWriteLn (cid);

            FWriteString (cid, "Logged in:    ");
            FWriteTime (cid, StartTime);  FWriteLn (cid);

            (* Now the details of the actual files transferred. *)

            current := first;
            WHILE current <> NIL DO
                WITH current^ DO
                    CASE what OF
                        | upload:       FWriteString (cid, "Uploaded:     ");
                        | partupload:   FWriteString (cid, "Tried to put: ");
                        | download:     FWriteString (cid, "Downloaded:   ");
                        | partdownload: FWriteString (cid, "Tried to get: ");
                        | delete:       FWriteString (cid, "Deleted:      ");
                    END (*CASE*);
                    FWriteString (cid, filename);
                    IF what <> delete THEN
                        FWriteString (cid, " (");  FWriteLJCard (cid, bytes);
                        FWriteString (cid, " bytes, ");

                        (* Duration is in milliseconds; report it as seconds to *)
                        (* two decimal places.                                  *)

                        WHILE duration < 0.0 DO
                            duration := duration + MillisecondsPerDay;
                        END (*WHILE*);
                        duration := duration/1000.0;
                        IF duration <= 0.005 THEN
                            rate := 0;
                        ELSE
                            rate := TRUNC (FLOAT(bytes)/duration + 0.5);
                        END (*IF*);
                        FWriteLJCard (cid, TRUNC(duration));
                        FWriteChar (cid, '.');
                        duration := duration - FLOAT(TRUNC(duration));
                        FWriteZCard (cid, TRUNC(100.0*duration + 0.5), 2);
                        FWriteString (cid, " seconds");
                        IF rate <> 0 THEN
                            FWriteString (cid, ", ");  FWriteLJCard (cid, rate);
                            FWriteString (cid, " bytes/s");
                        END (*IF*);
                        FWriteString (cid, ")");
                    END (*IF*);
                    FWriteLn (cid);
                END (*WITH*);
                first := current;  current := first^.next;
                DEC (entrycount);
                DISPOSE (first);
            END (*WHILE*);

            (* Logout details. *)

            SysClock.GetClock (Now);
            FWriteString (cid, "Finished:     ");
            FWriteTime (cid, Now);  FWriteLn (cid);

        END (*WITH*);
        CloseLogFile;
    END WriteLog;

(********************************************************************************)

PROCEDURE AddToLog (SS: ClientFileInfo;  kind: Operation;
                                                count: CARDINAL;  time: REAL);

    (* Adds a record to the user's transfer log. *)

    VAR p: LogEntryPointer;

    BEGIN
        IF LogLevel > 0 THEN
            NEW (p);
            WITH p^ DO
                next := NIL;  what := kind;  bytes := count;
                duration := time;
                MakeFullName (SS^.iname, filename);
            END (*WITH*);
            WITH SS^.Log DO
                IF first = NIL THEN first := p
                ELSE last^.next := p;
                END (*IF*);
                INC (entrycount);
                last := p;
            END (*WITH*);
            IF SS^.Log.entrycount > 99 THEN
                WriteLog (SS);
            END (*IF*);
        END (*IF*);
    END AddToLog;

(********************************************************************************)
(*                    LOGGING IN AND RELATED OPERATIONS                         *)
(********************************************************************************)

PROCEDURE NotifyMaxUsers (limit: CARDINAL);

    (* Limit on number of simultaneous users. *)

    BEGIN
        MaxUsers := limit;
    END NotifyMaxUsers;

(********************************************************************************)

PROCEDURE SetFreeSpaceThreshold (kilobytes: CARDINAL);

    (* The amount of space that must be unused on a drive before a client       *)
    (* can write to that drive.                                                 *)

    BEGIN
        FreeSpaceThreshold := kilobytes;
    END SetFreeSpaceThreshold;

(********************************************************************************)

PROCEDURE CreateSession (S: Socket;  UserNumber: CARDINAL;
                               KeepAlive: Semaphore): ClientFileInfo;

    (* Creates a new session state record.  Some of the user information is   *)
    (* still missing, but will be filled in when FindUser is called.          *)

    VAR result: ClientFileInfo;  peer: SockAddr;  size: CARDINAL;

    BEGIN
        NEW (result);
        WITH result^ DO
            user := NIL;
            KickMe := KeepAlive;
            SysClock.GetClock (Log.StartTime);
            flags := ListingOptions{};
            iname := NIL;
            CommandSocket := S;
            size := SIZE(SockAddr);
            getsockname (S, ServerName, size);
            WITH ServerName.in_addr DO
                port := Swap2 (Swap2(port) - 1);
            END (*WITH*);
            WITH DataPort DO
                host := 0;  port := 0;
                socket := NotASocket;
            END (*WITH*);
            UserNum := UserNumber;
            SpeedLimit := MAX(CARDINAL);
            IsManager := FALSE;  AnonUser := FALSE;
            ASCIITransfer := TRUE;  StreamMode := TRUE;
            PassiveOperation := FALSE;  FileStructure := 'F';
            RestartPoint := 0;
            AmountToAllocate := 0;
            WITH Log DO
                size := SIZE(peer);
                IF getpeername (S, peer, size) THEN
                    IF ScreenEnabled THEN
                        WriteError;
                    END (*IF*);
                ELSE
                    ClientIP := peer.in_addr.addr;
                END (*IF*);
                StartNameLookup (ClientIP);
                Strings.Assign ("?", UserName);
                entrycount := 0;
                first := NIL;  last := NIL;
            END (*WITH*);
        END (*WITH*);
        RETURN result;
    END CreateSession;

(********************************************************************************)

PROCEDURE CloseSession (SS: ClientFileInfo);

    (* Destroys SS, also updates the user log. *)

    BEGIN
        IF SS <> NIL THEN
            CloseDataPort (SS);
            IF (LogLevel > 2) OR ((LogLevel > 0) AND (SS^.Log.first <> NIL)) THEN
                WriteLog (SS);
            ELSE
                CancelNameLookup (SS^.Log.ClientIP);
            END (*IF*);
            DiscardFName (SS^.iname);
            DestroyUserData (SS^.user);
            DISPOSE (SS);
        END(*IF*);
    END CloseSession;

(********************************************************************************)

PROCEDURE GetUserName (SS: ClientFileInfo;  VAR (*OUT*) Name: ARRAY OF CHAR);

    (* Returns the user name (the one we use for logging).  *)

    BEGIN
        Strings.Assign (SS^.Log.UserName, Name);
    END GetUserName;

(********************************************************************************)

PROCEDURE FindUser (Name: ARRAY OF CHAR;
                              VAR (*INOUT*) SessionInfo: ClientFileInfo;
                              VAR (*OUT*) category: UserCategory);

    (* Input: Name contains the argument of the  USER command.       *)
    (*        S is this session's command socket.                    *)
    (* Output: Access permission data for that user.                 *)

    BEGIN
        WITH SessionInfo^ DO
            DiscardFName (iname);
            user := ReadUserData (Name, SpeedLimit, category);
            IF category <> NoSuchUser THEN

                (* Note: don't set the port, because a default port is  *)
                (* usually set before this procedure is called.         *)

                ASCIITransfer := TRUE;  StreamMode := TRUE;
                IsManager := category = Manager;
                AnonUser := category = GuestUser;
                PassiveOperation := FALSE;  FileStructure := 'F';
                RestartPoint := 0;
                AmountToAllocate := 0;
                Strings.Assign (Name, Log.UserName);
            END (*IF*);
        END (*WITH*);

    END FindUser;

(********************************************************************************)

PROCEDURE PasswordOK (SS: ClientFileInfo;  VAR (*IN*) pass: ARRAY OF CHAR): BOOLEAN;

    (* Tests for a password match. *)

    BEGIN
        IF SS^.AnonUser THEN
            Strings.Assign (pass, SS^.Log.UserName);
        END (*IF*);
        RETURN PasswordAcceptable (SS^.user, pass);
    END PasswordOK;

(********************************************************************************)

PROCEDURE CloseUser (SS: ClientFileInfo);

    (* Returns the user to a "not logged in" state. *)

    BEGIN
        IF SS <> NIL THEN
            DiscardFName (SS^.iname);
            DestroyUserData (SS^.user);
        END(*IF*);
    END CloseUser;

(********************************************************************************)

PROCEDURE KillDataChannel (SS: ClientFileInfo);

    (* Aborts the data transfer, if any, now in progress for this session. *)

    BEGIN
        IF SS <> NIL THEN
            WITH SS^.DataPort DO
                IF socket <> NotASocket THEN
                    so_cancel (SS^.DataPort.socket);
                END (*IF*);
            END (*WITH*);
        END (*IF*);
    END KillDataChannel;

(********************************************************************************)

PROCEDURE SendFile1 (S: Socket;  filename, prefix: ARRAY OF CHAR;
                                       UserNumber: CARDINAL;  SS: ClientFileInfo;
                                       VAR (*INOUT*) NewLine: BOOLEAN);   FORWARD;

(********************************************************************************)

PROCEDURE SendAMessage (S: Socket;  cid: IOChan.ChanId;  prefix: ARRAY OF CHAR;
                                       UserNumber: CARDINAL;  SS: ClientFileInfo;
                                       VAR (*INOUT*) NewLine: BOOLEAN);

    (* Sends the contents of a text file (which the caller must have already    *)
    (* opened) to socket S.  Each line has "prefix-" prepended to it, and there *)
    (* is also some macro expansion of % codes in the file.                     *)

    (* On entry, NewLine=TRUE means that we should start with the prefix, i.e.  *)
    (* we're not already in the middle of a line.  On return, NewLine=TRUE      *)
    (* means that the last thing we sent was the end-of-line code.              *)

    VAR buffer: ARRAY [0..255] OF CHAR;  pos: CARDINAL;

    (****************************************************************************)

    PROCEDURE AppendNumber (N: CARDINAL);

        (* Appends the string value of N to the buffer, with MAX(CARDINAL)      *)
        (* translated to 'unlimited'.                                           *)

        BEGIN
            IF N = MAX(CARDINAL) THEN
                AppendString ("unlimited", buffer, pos);
            ELSE
                ConvertCard (N, buffer, pos);
            END (*IF*);
        END AppendNumber;

    (****************************************************************************)

    PROCEDURE ReadChar (VAR (*OUT*) ch: CHAR): BOOLEAN;

        VAR status: IOConsts.ReadResults;  amount: CARDINAL;

        BEGIN
            IOChan.RawRead (cid, ADR(ch), SIZE(ch), amount);
            status := IOChan.ReadResult(cid);
            RETURN (status = IOConsts.allRight) AND (amount > 0);
        END ReadChar;

    (****************************************************************************)

    PROCEDURE ReadDelimitedString (VAR (*OUT*) filename: ARRAY OF CHAR): BOOLEAN;

        (* Reads a string surrounded by delimiters, i.e. the first character    *)
        (* is discarded and we continue until we get the same character (which  *)
        (* is also discarded) or end-of-line.                                   *)

        TYPE CharSet = SET OF CHAR;

        VAR success: BOOLEAN;  ch: CHAR;  j: CARDINAL;  stoppers: CharSet;

        BEGIN
            stoppers := CharSet {CR, LF, CtrlZ};
            success := ReadChar(ch);
            IF success THEN
                INCL (stoppers, ch);  j := 0;
                LOOP
                    success := ReadChar(ch);
                    IF NOT success OR (ch IN stoppers) THEN
                        EXIT (*LOOP*);
                    END (*IF*);
                    IF j <= HIGH(filename) THEN
                        filename[j] := ch;  INC(j);
                    END (*IF*);
                END (*LOOP*);
                IF j <= HIGH(filename) THEN
                    filename[j] := CHR(0);
                END (*IF*);
            END (*IF*);
            RETURN success;
        END ReadDelimitedString;

    (****************************************************************************)

    VAR ch: CHAR;  continue, OptionFlag: BOOLEAN;
        localtime: SysClock.DateTime;
        GroupUserNumber, GroupLimit: CARDINAL;
        filename: FileNameString;

    BEGIN
        IF SS = NIL THEN
            GroupUserNumber := UserNumber;  GroupLimit := MaxUsers;
        ELSE
            GetUserNumber (SS^.user, GroupUserNumber, GroupLimit);
        END (*IF*);

        OptionFlag := FALSE;  ch := CHR(0);  pos := 0;
        REPEAT
            continue := ReadChar (ch);
            IF continue THEN
                IF NewLine AND (ch <> CtrlZ) THEN
                    AppendString (prefix, buffer, pos);
                    buffer[pos] := '-';  INC(pos);
                    NewLine := FALSE;
                END (*IF*);
                IF ch = LF THEN
                    buffer[pos] := CHR(0);  pos := AddEOL (buffer);
                    continue := send (S, buffer, pos, 0) <> MAX(CARDINAL);
                    pos := 0;  NewLine := TRUE;
                ELSIF ch = CtrlZ THEN
                    continue := FALSE;
                ELSIF ch <> CR THEN
                    IF OptionFlag THEN
                        CASE ch OF
                          | 'i': continue := ReadDelimitedString(filename);
                                 IF continue THEN
                                     SendFile1 (S, filename, prefix,
                                                UserNumber, SS, NewLine);
                                 END (*IF*);
                          | 'm': AppendNumber (GroupLimit);
                          | 'M': AppendNumber (MaxUsers);
                          | 't','T':
                                 SysClock.GetClock (localtime);
                                 ConvertCardZ (localtime.hour, buffer, 2, pos);
                                 buffer[pos] := ':';  INC(pos);
                                 ConvertCardZ (localtime.minute, buffer, 2, pos);
                          | 'u': ConvertCard (GroupUserNumber, buffer, pos);
                          | 'U': ConvertCard (UserNumber, buffer, pos);
                          | '%': buffer[pos] := '%';  INC(pos);
                          | ELSE
                                 buffer[pos] := '%';  INC(pos);
                                 buffer[pos] := ch;  INC(pos);
                        END (*CASE*);
                        OptionFlag := FALSE;
                    ELSIF ch = '%' THEN
                        OptionFlag := TRUE;
                    ELSE
                        buffer[pos] := ch;  INC(pos);
                    END (*IF*);
                END (*IF*);
            END (*IF*);
        UNTIL NOT continue;

        (* Send out anything still left in the output buffer. *)

        IF pos > 0 THEN
            EVAL (send (S, buffer, pos, 0));
        END (*IF*);

    END SendAMessage;

(********************************************************************************)

PROCEDURE SendFile1 (S: Socket;  filename, prefix: ARRAY OF CHAR;
                                       UserNumber: CARDINAL;  SS: ClientFileInfo;
                                       VAR (*INOUT*) NewLine: BOOLEAN);

    (* Sends the contents of a text file to socket S.  Each line has "prefix-"  *)
    (* prepended to it, and there is also some macro expansion of % macro codes *)
    (* in the file.  If the file can't be opened, nothing is sent.              *)

    (* On entry, NewLine=TRUE means that we should start with the prefix, i.e.  *)
    (* we're not already in the middle of a line.  On return, NewLine=TRUE      *)
    (* means that the last thing we sent was the end-of-line code.              *)

    VAR cid: IOChan.ChanId;  status: ChanConsts.OpenResults;

    BEGIN
        RndFile.OpenOld (cid, filename,
                         ChanConsts.read+ChanConsts.old+ChanConsts.raw, status);
        IF status = ChanConsts.opened THEN
            SendAMessage (S, cid, prefix, UserNumber, SS, NewLine);
            RndFile.Close (cid);
        END (*IF*);
    END SendFile1;

(********************************************************************************)

PROCEDURE SendMessageFile (S: Socket;  filename, prefix: ARRAY OF CHAR;
                                       UserNumber: CARDINAL;  SS: ClientFileInfo);

    (* Sends the contents of a text file to socket S.  Each line has "prefix-"  *)
    (* prepended to it, and there is also some macro expansion of % macro codes *)
    (* in the file.  If the file can't be opened, nothing is sent.              *)

    VAR NewLine: BOOLEAN;  CRLF: ARRAY [0..1] OF CHAR;

    BEGIN
        NewLine := TRUE;
        SendFile1 (S, filename, prefix, UserNumber, SS, NewLine);
        IF NOT NewLine THEN
            CRLF[0] := CR;  CRLF[1] := LF;
            EVAL (send (S, CRLF, 2, 0));
        END (*IF*);
    END SendMessageFile;

(********************************************************************************)

PROCEDURE SendDirectoryMessage (SS: ClientFileInfo;  prefix: ARRAY OF CHAR);

    (* Sends the user a copy of Dir.MSG, if it exists in the user's     *)
    (* current directory and the user hasn't already seen it.           *)

    VAR msgname: FName;  filename: FileNameString;

    BEGIN
        IF NOT HaveSeenMessage (SS^.user) THEN
            msgname := MakeFName (SS^.user, "DIR.MSG");
            MakeFullName (msgname, filename);
            DiscardFName (msgname);
            SendMessageFile (SS^.CommandSocket, filename, prefix, SS^.UserNum, SS);
        END (*IF*);
    END SendDirectoryMessage;

(********************************************************************************)
(*                           FILE TRANSFER SETUP                                *)
(********************************************************************************)

PROCEDURE SetTransferType (SS: ClientFileInfo;  option: CHAR);

    (* Sets transfer type to 'A'=ASCII or 'I'=Image.  *)

    BEGIN
        SS^.ASCIITransfer := (option = 'A');
    END SetTransferType;

(********************************************************************************)

PROCEDURE SetTransferMode (SS: ClientFileInfo;  option: CHAR);

    (* Sets transfer mode to 'S'=stream or (others not implemented). *)

    BEGIN
        SS^.StreamMode := (option = 'S');
    END SetTransferMode;

(********************************************************************************)

PROCEDURE SetFileStructure (SS: ClientFileInfo;  option: CHAR);

    (* Sets file structure to 'F'=file or 'R'=record.  In this implementation   *)
    (* these are treated identically.  Page structure is not implemented.       *)

    BEGIN
        SS^.FileStructure := option;
    END SetFileStructure;

(********************************************************************************)

PROCEDURE SetPort (SS: ClientFileInfo;  IPaddr: CARDINAL;  port: CARDINAL);

    (* Sets the client address for future data transfers. *)

    BEGIN
        WITH SS^ DO
            DataPort.host := IPaddr;  DataPort.port := port;
        END (*WITH*);
    END SetPort;

(********************************************************************************)

PROCEDURE SetRestartPoint (SS: ClientFileInfo;  marker: ARRAY OF CHAR);

    (* Sets a restart marker for a subsequent RETR or STOR operation.  *)

    VAR j: CARDINAL;

    BEGIN
        j := 0;
        SS^.RestartPoint := ConvertDecimal (marker, j);
    END SetRestartPoint;

(********************************************************************************)

PROCEDURE AllocateFileSize (SS: ClientFileInfo;  amount: ARRAY OF CHAR);

    (* Sets the file size for a subsequent STOR or APPE. *)

    VAR j: CARDINAL;

    BEGIN
        j := 0;
        SS^.AmountToAllocate := ConvertDecimal (amount, j);
    END AllocateFileSize;

(********************************************************************************)

PROCEDURE EnterPassiveMode (SS: ClientFileInfo;  VAR (*OUT*) myaddr: SockAddr): BOOLEAN;

    (* Creates a data socket, binds it to a local port, and then listens.       *)
    (* On return myaddr holds the port details.                                 *)

    VAR s: Socket;  size: CARDINAL;

    BEGIN
        s := socket (AF_INET, SOCK_STREAM, AF_UNSPEC);
        SS^.DataPort.socket := s;
        myaddr := SS^.ServerName;
        myaddr.in_addr.port := 0;
        size := SIZE(myaddr);
        WITH SS^ DO
            PassiveOperation := NOT(bind (s, myaddr, size) OR listen (s, 1)
                                         OR getsockname (s, myaddr, size));
            IF PassiveOperation THEN
                RETURN TRUE;
            ELSE
                WITH DataPort DO
                    soclose (socket);
                    socket := NotASocket;
                END (*WITH*);
                RETURN FALSE;
            END (*IF*);
        END (*WITH*);
    END EnterPassiveMode;

(********************************************************************************)

PROCEDURE OpenDataPort (SS: ClientFileInfo): BOOLEAN;

    CONST SO_LINGER = 0080H;          (* linger on close if data present *)

    VAR s, ns: Socket;  peer: SockAddr;  option, size: CARDINAL;
        (*LingerData: ARRAY [0..1] OF CARDINAL;*)

    BEGIN
        IF SS^.PassiveOperation THEN

            (* Phase 2: we have our port, now accept a connection. *)

            s := SS^.DataPort.socket;
            IF WaitForDataSocket (FALSE, s, SS^.CommandSocket) THEN
                size := SIZE(peer);
                ns := accept (s, peer, size);
            ELSE
                ns := NotASocket;
            END (*IF*);
            SS^.PassiveOperation := FALSE;
            IF ns = NotASocket THEN
                IF ScreenEnabled THEN
                    WriteError;
                END (*IF*);
                RETURN FALSE;
            ELSE
                SS^.DataPort.socket := ns;
                soclose (s);
                s := ns;
            END (*IF*);

        ELSE

            (* Normal, non-passive operation. *)

            CloseDataPort (SS);
            s := socket (AF_INET, SOCK_STREAM, AF_UNSPEC);
            SS^.DataPort.socket := s;

            (* Bind to the data port at our end, allowing reuse of the port     *)
            (* we're binding to.  No error check is needed; if the bind fails,  *)
            (* we can carry on without it.                                      *)

            option := MAX(CARDINAL);
            setsockopt (s, 0FFFFH, 4, option, SIZE(CARDINAL));
            bind (s, SS^.ServerName, SIZE(SockAddr));

            (* Socket open, connect to the client. *)

            WITH peer DO
                family := AF_INET;
                WITH in_addr DO
                    port := SS^.DataPort.port;
                    addr := SS^.DataPort.host;
                    zero := Zero8;
                END (*WITH*);
            END (*WITH*);

            IF connect (s, peer, SIZE(peer)) THEN

                (* Connection failed. *)

                IF ScreenEnabled THEN
                    WriteError;
                END (*IF*);
                RETURN FALSE;

            END (*IF*);

        END (*IF PassiveOperation*);

        (* "Linger on close" is disabled for now, because there is at least     *)
        (* one version of the TCP/IP stack that can't handle it.                *)

        (*
        LingerData[0] := 1;  LingerData[1] := MAX(CARDINAL);
        IF setsockopt (s, 0FFFFH, SO_LINGER, LingerData, SIZE(LingerData)) THEN
            WriteError;
        END (*IF*);
        *)

        RETURN TRUE;

    END OpenDataPort;

(********************************************************************************)

PROCEDURE CloseDataPort (SS: ClientFileInfo);

    (* Closes the data port used by this client, if it was still open. *)

    BEGIN
        WITH SS^.DataPort DO
            IF socket <> NotASocket THEN
                soclose (socket);
                socket := NotASocket;
            END (*IF*);
        END (*WITH*);
    END CloseDataPort;

(********************************************************************************)
(*                          FILE TRANSFER OPERATIONS                            *)
(********************************************************************************)

PROCEDURE SendAscii (CommandSocket, S: Socket;  VAR (*IN*) source: ARRAY OF LOC;
                                                   amount: CARDINAL): CARDINAL;

    (* Sends "amount" bytes of data from source to S.  The value returned is    *)
    (* the actual number of characters sent; this could be different from       *)
    (* amount because of things like conversion of line terminators.  If the    *)
    (* transfer fails, we return a result of MAX(CARDINAL).                     *)

    VAR j, k, count, N: CARDINAL;  ch: CHAR;  AtEOL, AtEOF: BOOLEAN;
        CRLF: ARRAY[0..1] OF CHAR;

    BEGIN
        CRLF[0] := CR;  CRLF[1] := LF;
        k := 0;  count := 0;  AtEOL := FALSE;  AtEOF := FALSE;
        REPEAT
            (* Skip past any carriage returns. *)

            WHILE (k < amount) AND (CAST(CHAR,source[k]) = CR) DO
                INC (k);
            END (*WHILE*);

            j := k;

            (* Scan to next CR or LF or end-of-file. *)

            LOOP
                IF k >= amount THEN
                    AtEOF := TRUE;  EXIT(*LOOP*);
                END(*IF*);
                ch := CAST (CHAR, source[k]);
                IF ch = LF THEN
                    AtEOL := TRUE;  EXIT;
                ELSIF ch = CtrlZ THEN
                    AtEOF := TRUE;  EXIT;
                ELSIF ch = CR THEN
                    EXIT;
                END (*IF*);
                INC (k);
            END (*LOOP*);

            (* Send the data from position j to position k-1. *)

            IF k > j THEN
                IF WaitForDataSocket (TRUE, S, CommandSocket) THEN
                    N := send (S, source[j], k-j, 0);
                ELSE
                    N := MAX(CARDINAL);
                END (*IF*);
                IF N = MAX(CARDINAL) THEN
                    count := MAX(CARDINAL);
                    AtEOF := TRUE;  AtEOL := FALSE;
                ELSE
                    INC (count, N);
                END (*IF*);
            END (*IF*);

            (* Send the end-of-line code, if needed. *)

            IF AtEOL THEN
                AtEOF := send (S, CRLF, 2, 0) = MAX(CARDINAL);
                IF AtEOF THEN
                    count := MAX(CARDINAL);
                ELSE
                    AtEOL := FALSE;
                    INC (k);  INC (count, 2);
                END (*IF*);
            END (*IF*);

        UNTIL AtEOF;

        RETURN count;

    END SendAscii;

(********************************************************************************)

PROCEDURE PutFile (CommandSocket, S: Socket;  id: IOChan.ChanId;
                   RateLimit: REAL;  Ascii: BOOLEAN;  KeepAlive: Semaphore;
                   VAR (*OUT*) BytesSent: CARDINAL): BOOLEAN;

    (* Sends a file on a previously opened socket S. *)

    CONST BufferSize = 2048;

    VAR BuffPtr: POINTER TO ARRAY [0..BufferSize-1] OF LOC;
        amount: CARDINAL;  success: BOOLEAN;
        starttime, duration: REAL;

    BEGIN
        NEW (BuffPtr);
        BytesSent := 0;  success := TRUE;
        starttime := FLOAT(millisecs());
        LOOP
            IOChan.RawRead (id, BuffPtr, BufferSize, amount);
            IF IOChan.ReadResult(id) <> IOConsts.allRight THEN
                EXIT(*LOOP*)
            END (*IF*);
            Signal (KeepAlive);
            IF Ascii THEN
                amount := SendAscii (CommandSocket, S, BuffPtr^, amount);
            ELSIF WaitForDataSocket (TRUE, S, CommandSocket) THEN
                amount := send (S, BuffPtr^, amount, 0);
            ELSE
                amount := MAX(CARDINAL);
            END (*IF*);
            IF amount = MAX(CARDINAL) THEN
                success := FALSE;
            ELSE
                INC (BytesSent, amount);
            END (*IF*);
            IF NOT success THEN
                EXIT(*LOOP*)
            END (*IF*);

            (* If the transfer is too fast, insert a time delay. *)
            (* RateLimit is in bytes per millisecond.            *)

            duration := FLOAT(millisecs()) - starttime;
            WHILE duration < 0.0 DO
                duration := duration + MillisecondsPerDay;
            END (*WHILE*);
            IF FLOAT(BytesSent) > RateLimit * duration THEN
                DosSleep (TRUNC(FLOAT(BytesSent)/RateLimit - duration));
            END (*IF*);

        END (*LOOP*);
        DISPOSE (BuffPtr);
        RETURN success;
    END PutFile;

(********************************************************************************)

PROCEDURE GetFile (CommandSocket, S: Socket;  id: IOChan.ChanId;
                   RateLimit: REAL;
                   KeepAlive: Semaphore;  VAR (*OUT*) totalbytes: CARDINAL): BOOLEAN;

    (* Retrieves a file on a previously opened socket.  Returns TRUE iff the    *)
    (* transfer was successful.                                                 *)

    CONST BufferSize = 2048;

    VAR BuffPtr: POINTER TO ARRAY [0..BufferSize-1] OF LOC;
        amount: CARDINAL;  success: BOOLEAN;
        starttime, duration: REAL;

    BEGIN
        NEW (BuffPtr);
        totalbytes := 0;
        success := TRUE;
        starttime := FLOAT(millisecs());
        LOOP
            IF WaitForDataSocket (FALSE, S, CommandSocket) THEN
                amount := recv (S, BuffPtr^, BufferSize, 0);
            ELSE
                amount := MAX(CARDINAL);
            END (*IF*);
            IF amount = 0 THEN EXIT(*LOOP*)
            ELSIF amount = MAX(CARDINAL) THEN
                success := FALSE;
                EXIT(*LOOP*);
            END(*IF*);
            Signal (KeepAlive);
            INC (totalbytes, amount);
            IOChan.RawWrite (id, BuffPtr, amount);

            (* If the transfer is too fast, insert a time delay. *)
            (* RateLimit is in bytes per millisecond.            *)

            duration := FLOAT(millisecs()) - starttime;
            WHILE duration < 0.0 DO
                duration := duration + MillisecondsPerDay;
            END (*WHILE*);
            IF FLOAT(totalbytes) > RateLimit * duration THEN
                DosSleep (TRUNC(FLOAT(totalbytes)/RateLimit - duration));
            END (*IF*);

        END (*LOOP*);
        DISPOSE (BuffPtr);
        RETURN success;
    END GetFile;

(********************************************************************************)

PROCEDURE SendDirectory (SS: ClientFileInfo;  ShowAllDetails: BOOLEAN): BOOLEAN;

    (* Sends a directory listing as specified by the last SetFileName call.     *)
    (* Returns TRUE iff the transfer was successful.                            *)

    BEGIN
        IF OpenDataPort (SS) THEN
            WITH SS^ DO
                IF ShowAllDetails THEN
                    flags := flags + ListingOptions{ListDotDot, ShowDetails};
                END (*IF*);
                IF IsManager THEN INCL(flags, SystemAndHidden) END (*IF*);
                ListDirectory (DataPort.socket, iname, flags);
                Synch (DataPort.socket);
            END (*WITH*);
            RETURN TRUE;
        ELSE
            RETURN FALSE;
        END (*IF*);
    END SendDirectory;

(********************************************************************************)

PROCEDURE SendFile (SS: ClientFileInfo): BOOLEAN;

    (* Sends the file whose name was specified by the last SetFileName call to  *)
    (* the client.  Returns TRUE for a successful send.                         *)

    VAR cid: IOChan.ChanId;  count: CARDINAL;  starttime: REAL;
        success: BOOLEAN;

    BEGIN
        success := FALSE;
        IF OpenDataPort (SS) THEN
            IF SS^.RestartPoint >= GetFileSize (SS^.iname) THEN
                SS^.RestartPoint := 0;
                success := TRUE;
                CloseDataPort (SS);
                AddToLog (SS, download, 0, 0.0);
            ELSIF OpenForReading (cid, SS^.iname) THEN
                RndFile.SetPos (cid,
                    RndFile.NewPos (cid, SS^.RestartPoint, 1, RndFile.StartPos(cid)));
                SS^.RestartPoint := 0;
                starttime := FLOAT(millisecs());
                success := PutFile (SS^.CommandSocket, SS^.DataPort.socket,
                                    cid, 0.001*FLOAT(SS^.SpeedLimit),
                                    SS^.ASCIITransfer, SS^.KickMe, count);
                RndFile.Close (cid);
                CloseDataPort (SS);
                IF success THEN
                    AddToLog (SS, download, count, FLOAT(millisecs()) - starttime);
                ELSIF LogLevel > 1 THEN
                    AddToLog (SS, partdownload, count, FLOAT(millisecs()) - starttime);
                END (*IF*);
            ELSE
                CloseDataPort (SS);
            END (*IF*);
        END (*IF*);
        DiscardFName (SS^.iname);
        RETURN success;
    END SendFile;

(********************************************************************************)

PROCEDURE FetchFile (SS: ClientFileInfo): BOOLEAN;

    (* Reads the file whose name was specified by the last SetFileName call     *)
    (* from the client.  Returns TRUE for a successful transfer.                *)

    VAR cid: IOChan.ChanId;  count: CARDINAL;  starttime: REAL;
        success: BOOLEAN;

    BEGIN
        IF OpenDataPort (SS) THEN
            IF OpenForWriting (cid, SS^.iname) THEN
                IF SS^.AmountToAllocate <> 0 THEN
                    SetFileSize (cid, SS^.AmountToAllocate);
                END (*IF*);
                SS^.AmountToAllocate := 0;
                RndFile.SetPos (cid,
                    RndFile.NewPos (cid, SS^.RestartPoint, 1, RndFile.StartPos(cid)));
                SS^.RestartPoint := 0;
                starttime := FLOAT(millisecs());
                success := GetFile (SS^.CommandSocket, SS^.DataPort.socket, cid,
                                    0.001*FLOAT(SS^.SpeedLimit), SS^.KickMe, count);
                RndFile.Close (cid);
                CloseDataPort (SS);
                IF success THEN
                    AddToLog (SS, upload, count, FLOAT(millisecs()) - starttime);
                ELSIF LogLevel > 1 THEN
                    AddToLog (SS, partupload, count, FLOAT(millisecs()) - starttime);
                END (*IF*);
                DiscardFName (SS^.iname);
                RETURN success;
            ELSE
                CloseDataPort (SS);
                DiscardFName (SS^.iname);
                RETURN FALSE;
            END (*IF*);
        ELSE
            DiscardFName (SS^.iname);
            RETURN FALSE;
        END (*IF*);
    END FetchFile;

(********************************************************************************)

PROCEDURE AppendFile (SS: ClientFileInfo): BOOLEAN;

    (* Like FetchFile, except that if the file already exists then the new data *)
    (* are appended to the end of the file.                                     *)

    VAR cid: IOChan.ChanId;  count: CARDINAL;  starttime: REAL;
        success: BOOLEAN;

    BEGIN
        success := FALSE;
        IF OpenDataPort (SS) THEN
            IF OpenForAppend (cid, SS^.iname) THEN
                IF SS^.AmountToAllocate <> 0 THEN
                    SetFileSize (cid, SS^.AmountToAllocate);
                END (*IF*);
                SS^.AmountToAllocate := 0;
                starttime := FLOAT(millisecs());
                success := GetFile (SS^.CommandSocket, SS^.DataPort.socket, cid,
                                    0.001*FLOAT(SS^.SpeedLimit), SS^.KickMe, count);
                RndFile.Close (cid);
                CloseDataPort (SS);
                IF success THEN
                    AddToLog (SS, upload, count, FLOAT(millisecs()) - starttime);
                ELSIF LogLevel > 1 THEN
                    AddToLog (SS, partupload, count, FLOAT(millisecs()) - starttime);
                END (*IF*);
            ELSE
                CloseDataPort (SS);
            END (*IF*);
        END (*IF*);
        DiscardFName (SS^.iname);
        RETURN success;
    END AppendFile;

(********************************************************************************)

PROCEDURE DeleteFile (SS: ClientFileInfo): BOOLEAN;

    (* Deletes the file whose name was specified by the last SetFileName call   *)
    (* from the client.  Returns TRUE for a successful deletion.                *)

    VAR result: BOOLEAN;

    BEGIN
        result := RemoveFile (SS^.iname);
        IF result THEN
            AddToLog (SS, delete, 0, 0.0);
        END (*IF*);
        DiscardFName (SS^.iname);
        RETURN result;
    END DeleteFile;

(********************************************************************************)

PROCEDURE RenameFile (SS: ClientFileInfo;  NewName: ARRAY OF CHAR): BOOLEAN;

    (* Renames to NewName the file whose name was specified by the last         *)
    (* SetFileName call from the client.  Returns TRUE for success.             *)

    VAR dstName: FName;  result: BOOLEAN;

    BEGIN
        dstName := MakeFName (SS^.user, NewName);
        result := CanWriteFile (dstName, SS^.IsManager)
                         AND RenameTo (SS^.iname, dstName);
        DiscardFName (dstName);
        RETURN result;
    END RenameFile;

(********************************************************************************)
(*                           DIRECTORY OPERATIONS                               *)
(********************************************************************************)

PROCEDURE MakeDirectory (SS: ClientFileInfo): BOOLEAN;

    (* Creates a directory as specified by the last SetFileName call.  Returns  *)
    (* TRUE for a successful operation.                                         *)

    BEGIN
        RETURN CreateDirectory (SS^.iname);
    END MakeDirectory;

(********************************************************************************)

PROCEDURE DeleteDirectory (SS: ClientFileInfo): BOOLEAN;

    (* Deletes the file whose name was specified by the last SetFileName call   *)
    (* from the client.  Returns TRUE for a successful deletion.                *)

    BEGIN
        RETURN RemoveDirectory (SS^.iname);
    END DeleteDirectory;

(********************************************************************************)

PROCEDURE CurrentDirectory (SS: ClientFileInfo;  VAR (*OUT*) DirString: ARRAY OF CHAR);

    (* Gives back the name of the current directory for this user.  *)

    BEGIN
        NameOfCurrentDirectory (SS^.user, DirString);
    END CurrentDirectory;

(********************************************************************************)

PROCEDURE SetDirectory (SS: ClientFileInfo;  MessageEnabled: BOOLEAN;
                                              DirString: ARRAY OF CHAR): BOOLEAN;

    (* Changes user to the specified directory.  The pathname can be absolute   *)
    (* (starting with '/') or relative to the current directory.                *)
    (* Returns FALSE if the requested directory does not exist, or if the user  *)
    (* does not have the right to see it.                                       *)

    VAR success: BOOLEAN;  iname: FName;

    BEGIN
        iname := MakeFName (SS^.user, DirString);
        success := SetWorkingDirectory (SS^.user, iname);
        DiscardFName (iname);

        (* Give the user a copy of Dir.MSG, if it exists and is wanted. *)

        IF success AND MessageEnabled THEN
            SendDirectoryMessage (SS, "250");
        END (*IF*);

        RETURN success;

    END SetDirectory;

(********************************************************************************)

PROCEDURE GetCurrentPermissions (SS: ClientFileInfo;  VAR (*OUT*) result: ARRAY OF CHAR);

    (* Returns a string indicating read/write/delete permissions for the        *)
    (* user's current directory.                                                *)

    BEGIN
        PermissionString (SS^.user, result);
    END GetCurrentPermissions;

(********************************************************************************)

PROCEDURE SetFileName (SS: ClientFileInfo;  NameString: ARRAY OF CHAR;
                                                         InterpretFlags: BOOLEAN);

    (* Specifies the name of the file that will next be involved in a data      *)
    (* transfer for this user.  If InterpretFlags is TRUE, also detects some    *)
    (* Unix-like arguments and sets SS^.flags.                                  *)

    (* The Unix-like arguments have the meanings:                               *)
    (*     F   a trailing '/' will be added to directory names.                 *)
    (*     R   recurse on subdirectories.                                       *)
    (*     a   in standard Unix 'ls', this means to include names starting with *)
    (*         a '.'.  In OS/2 the leading '.' has no special significance,     *)
    (*         so we interpret this command as meaning that the '..' should be  *)
    (*         included in the directory listing.                               *)
    (*     d   when NameString is a directory name, we list its name rather     *)
    (*         than the contents of the directory.                              *)
    (*     l   detailed listing: mode, number of links, owner, size in bytes,   *)
    (*         time of last modification.  For the number of links we have a    *)
    (*         dummy entry, and the 'owner' field is filled with the RASH code. *)
    (* These arguments are meaningful only for the LIST and NLST commands.      *)
    (* Otherwise they have no effect.                                           *)

    (* Earlier I had supported the following as well, but I've decided to       *)
    (* drop it as being irrelevant to our purposes.                             *)
    (*     A   not documented in Unix man, not sure why I added this; but       *)
    (*         apparently it was supposed to hide filenames starting with '.'.  *)

    VAR j: CARDINAL;

    BEGIN
        WITH SS^ DO

            flags := ListingOptions {MayExpand};
            IF InterpretFlags THEN
                WHILE NameString[0] = '-' DO
                    j := 1;
                    WHILE (NameString[j] <> CHR(0)) AND (NameString[j] <> ' ') DO
                        CASE NameString[j] OF
                          |  'F':  INCL (flags, AddSlash);
                          |  'R':  INCL (flags, Recurse);
                          |  'a':  INCL (flags, ListDotDot);
                          |  'd':  EXCL (flags, MayExpand);
                          |  'l':  INCL (flags, ShowDetails);

                            (* I'm not sure whether I need to handle the following: *)
                            (*    -D   generate output suited to Emacs' dired mode  *)

                        ELSE
                            (* Ignore options that are irrelevant to us. *)
                        END (*CASE*);
                        INC (j);
                    END (*WHILE*);
                    WHILE NameString[j] = ' ' DO
                        INC (j);
                    END (*WHILE*);
                    Strings.Delete (NameString, 0, j);
                END (*WHILE*);
            END (*IF*);

            DiscardFName (iname);
            iname := MakeFName (user, NameString);

        END (*WITH*);

   END SetFileName;

(********************************************************************************)

PROCEDURE CreateUniqueFileName (SS: ClientFileInfo;
                                 VAR (*OUT*) NameString: ARRAY OF CHAR): BOOLEAN;

    (* Like SetFileName, except that we create the name internally, making      *)
    (* sure that it's unique for the current directory.  The result is FALSE    *)
    (* if we're not able to create the unique name.                             *)

    CONST numstart = 3;

    (****************************************************************************)

    PROCEDURE increment (endpos: CARDINAL): BOOLEAN;

        (* Increments the ASCII decimal number in SS^.FileName[numstart..endpos].*)
        (* Returns FALSE if the number overflows.                                *)

        BEGIN
            IF endpos < numstart THEN
                RETURN FALSE;
            ELSIF NameString[endpos] <> '9' THEN
                INC (NameString[endpos]);
                RETURN TRUE;
            ELSE
                NameString[endpos] := '0';
                RETURN increment (endpos-1);
            END (*IF*);
        END increment;

    (****************************************************************************)

    CONST lastdigit = 7;

    VAR nameOK: BOOLEAN;

    BEGIN
        Strings.Assign ("FTP00000", NameString);
        nameOK := TRUE;
        SS^.iname := MakeFName (SS^.user, NameString);
        WHILE nameOK AND FileExists (SS^.iname, TRUE) DO
            nameOK := increment (lastdigit);
            DiscardFName (SS^.iname);
            SS^.iname := MakeFName (SS^.user, NameString);
        END (*WHILE*);
        RETURN nameOK;
    END CreateUniqueFileName;

(********************************************************************************)

PROCEDURE GetFileDate (SS: ClientFileInfo;  VAR (*OUT*) result: ARRAY OF CHAR);

    (* Returns the date/time of the file's directory entry, in a string of the  *)
    (* form "yyyymmddhhmmss" (exactly 14 characters).  If the file is not       *)
    (* accessible, the result is the null string.                               *)

    BEGIN
        GetDateTime (SS^.iname, result);
    END GetFileDate;

(********************************************************************************)

PROCEDURE GetSize (SS: ClientFileInfo): CARDINAL;

    (* Returns the size in bytes of the file specified in SetFileName.  If the  *)
    (* file is not accessible, the result is returned as MAX(CARDINAL).         *)

    BEGIN
        RETURN GetFileSize (SS^.iname);
    END GetSize;

(********************************************************************************)

PROCEDURE ListingIsPossible (SS: ClientFileInfo): BOOLEAN;

    (* Returns TRUE iff this user is allowed to see a listing of the file(s)    *)
    (* specified in the last SetFileName call.                                  *)

    BEGIN
        RETURN MayListFiles (SS^.iname);
    END ListingIsPossible;

(********************************************************************************)

PROCEDURE FileAvailable (SS: ClientFileInfo): BOOLEAN;

    (* Returns TRUE iff the file exists and the user is allowed to read it.  *)

    BEGIN
        WITH SS^ DO
            RETURN CanReadFile (iname, IsManager);
        END (*WITH*);
    END FileAvailable;

(********************************************************************************)

PROCEDURE CanWrite (SS: ClientFileInfo): BOOLEAN;

    (* Returns TRUE iff user can write the file whose name was last set.  *)

    BEGIN
        WITH SS^ DO
            RETURN CanWriteFile (iname, IsManager)
                    AND (SpaceAvailable(iname) > FreeSpaceThreshold);
        END (*WITH*);
    END CanWrite;

(********************************************************************************)

PROCEDURE CanDelete (SS: ClientFileInfo): BOOLEAN;

    (* Returns TRUE iff user can delete the file whose name was last set.  *)

    BEGIN
        WITH SS^ DO
            RETURN CanDeleteFile (iname, IsManager);
        END (*WITH*);
    END CanDelete;

(********************************************************************************)

PROCEDURE CanRemoveDirectory (SS: ClientFileInfo): BOOLEAN;

    (* Returns TRUE iff user can delete the directory whose name was last set.  *)

    BEGIN
        RETURN CanDeleteDirectory (SS^.iname);
    END CanRemoveDirectory;

(********************************************************************************)
(*                              INITIALISATION                                  *)
(********************************************************************************)

BEGIN
    LogLevel := 1;  MaxUsers := MAX(CARDINAL);
    ScreenEnabled := NotDetached();
END FtpTransfers.

