IMPLEMENTATION MODULE Users;

        (********************************************************)
        (*                                                      *)
        (*         Editor for the user access rights            *)
        (*                                                      *)
        (*  Programmer:         P. Moylan                       *)
        (*  Started:            2 December 1997                 *)
        (*  Last edited:        14 July 1998                    *)
        (*  Status:             OK                              *)
        (*                                                      *)
        (********************************************************)

(************************************************************************)
(*                                                                      *)
(*     Syntax for the permissions file:                                 *)
(*                                                                      *)
(*     <result>  ->  <userclass> <password> { <volumeinfo> }*           *)
(*     <volumeinfo> ->  <homedir> { <dirrule> }                         *)
(*     <userclass> -> G | U | N | M                                     *)
(*     <password>  -> <namestring>                                      *)
(*     <homedir>    ->  <namestring>  |  <namestring> = <namestring>    *)
(*     <diritem>   -> <namestring> <dirrule>                            *)
(*     <dirrule>   -> { <permission> }* { ( <dirlist> ) }               *)
(*     <dirlist>   ->  <diritem> { , <diritem> }*                       *)
(*     <dirlist>   -> { <diritem> }+                                    *)
(*     <permission> ->  V+ | V- | R+ | R- | W- | W+ | D- | D+           *)
(*     <namestring> ->  <string1> | " <string2> " | ' <string3> '       *)
(*     <string1>   -> any string not including space char               *)
(*     <string2>   -> any string not including double quote             *)
(*     <string3>   -> any string not including single quote             *)
(*                                                                      *)
(*  Notation: {} means optional, {}* means zero or more, {}+ means      *)
(*            one or more.                                              *)
(*                                                                      *)
(************************************************************************)

FROM SYSTEM IMPORT ADDRESS, ADR;

IMPORT OS2, Strings;

FROM PermissionEditor IMPORT
    (* type *)  CharSet, DirEntryPtr, ExitCode, CharArrayPointer,
    (* proc *)  CreateDirEntry,
                Scan, NameString, DirRule,
                AppendQuotedString, SizeOfPermissionData, StorePermissionData,
                KillList, UpdateDirectoryTree,
                InsertThreads, ShowTree, EditPermissionListBox, CleanUpTree;

FROM SetupINI IMPORT
    (* proc *)  OurINIHandle;

FROM ListBoxes IMPORT
    (* type *)  ListBox,
    (* proc *)  CreateListBox, LBAppend, CursorMovements,
                LBCurrentItemNumber, DestroyListBox, HighlightOn, HighlightOff,
                ClearListBox, LBCurrent, ReplaceCurrent, LBInsertAfter,
                LBDeleteCurrent;

FROM InetUtilities IMPORT
    (* proc *)  INIGet, ToLower;

FROM ScreenEditor IMPORT
    (* type *)  Structure,
    (* proc *)  CreateField, MenuField, CardinalField, StringField, Combine,
                ScreenEdit, DeleteStructure;

FROM FieldEditor IMPORT
    (* type *)  FieldType,
    (* proc *)  DefineFieldType, DiscardFieldType;

FROM Menus IMPORT
    (* type *)  Menu, ItemText,
    (* proc *)  CreateMenu, MenuColours;

FROM Windows IMPORT
    (* type *)  Window, Colour, FrameType, DividerType,
    (* proc *)  OpenWindowHidden, CloseWindow, WriteChar, WriteString, WriteLn,
                SetCursor, GetKey, EditString, PressAnyKey, GetScreenSize;

FROM MultiScreen IMPORT
    (* type *)  VirtualScreen,
    (* proc *)  MapToVirtualScreen;

FROM LowLevel IMPORT
    (* proc *)  EVAL;

FROM Storage IMPORT
    (* proc *)  ALLOCATE, DEALLOCATE;

(********************************************************************************)

CONST Nul = CHR(0);

TYPE
    FileNameSubscript = [0..255];
    FileNameString = ARRAY FileNameSubscript OF CHAR;
    PassString = ARRAY [0..31] OF CHAR;
    PassStringPtr = POINTER TO PassString;

    (* The fields in a VolumePermission record have the following meanings.     *)
    (*      Drive        The name of the device or partition                    *)
    (*      next         Pointer to the next VolumePermission for this user.    *)
    (*      Home         Home directory, an absolute path.  This always         *)
    (*                                   ends with a '/'.                       *)
    (*      BaseEntry    Pointer to tree of directory permissions               *)

    Volume = POINTER TO VolumePermission;

    VolumePermission = RECORD
                           Drive: FileNameString;
                           next: Volume;
                           Home: FileNameString;
                           BaseEntry: DirEntryPtr;
                       END (*RECORD*);

    UserCategory = (NoSuchUser, NoPasswordNeeded, GuestUser, NormalUser, Manager);

TYPE CategoryMapType = ARRAY UserCategory OF CHAR;
CONST CategoryMap = CategoryMapType {'?', 'N', 'G', 'U', 'M'};

    (* The fields in a UserPermission record have the following meanings.       *)
    (*      category     What sort of user this is                              *)
    (*      Password     The user's password                                    *)
    (*      UserLimit    Maximum number of instances of this user               *)
    (*      FirstVol     Points to a linked list of Volume entries.             *)

TYPE
    User = POINTER TO UserPermission;

    UserPermission = RECORD
                         category: UserCategory;
                         Password: PassString;
                         UserLimit: CARDINAL;
                         FirstVol: Volume;
                     END (*RECORD*);

VAR
    (* Type for field editor. *)

    ShortString: FieldType;

    (* Number of display rows on screen.  *)

    ScreenRows: CARDINAL;

(********************************************************************************)
(*                    CHECKING WHETHER A USER EXISTS                            *)
(********************************************************************************)

PROCEDURE UserInINIFile (VAR (*OUT*) hini: OS2.HINI;
                                username: ARRAY OF CHAR): BOOLEAN;

    (* If username is in INI file, initialises hini and returns TRUE. *)

    VAR size: CARDINAL;  result: BOOLEAN;

    BEGIN
        hini := OurINIHandle();
        IF hini = OS2.NULLHANDLE THEN
            result := FALSE;
        ELSE
            result := OS2.PrfQueryProfileSize (hini, username, NIL, size)
                                   AND (size <> 0);
        END (*IF*);
        RETURN result;
    END UserInINIFile;

(********************************************************************************)

PROCEDURE FindUser (UserName: ARRAY OF CHAR): BOOLEAN;

    (* Returns TRUE iff the user data exists.        *)

    VAR hini: OS2.HINI;

    BEGIN
        RETURN UserInINIFile (hini, UserName);
    END FindUser;

(********************************************************************************)
(*                    CREATION AND DELETION OF VOLUMES                          *)
(********************************************************************************)

PROCEDURE MakeVolume (U: User;  previous: Volume): Volume;

    VAR V: Volume;

    BEGIN
        NEW (V);
        WITH V^ DO
            Drive[0] := CHR(0);
            Home := "";
            BaseEntry := NIL;
        END (*WITH*);
        IF previous = NIL THEN
            V^.next := NIL;  U^.FirstVol := V;
        ELSE
            V^.next := previous^.next;  previous^.next := V;
        END (*IF*);
        RETURN V;
    END MakeVolume;

(********************************************************************************)

PROCEDURE DeleteVolume (U: User;  VAR (*INOUT*) V: Volume);

    VAR previous, current: Volume;

    BEGIN
        previous := NIL;  current := U^.FirstVol;
        WHILE (current <> NIL) AND (current <> V) DO
            previous := current;  current := current^.next;
        END (*WHILE*);
        IF current <> NIL THEN
            KillList (V^.BaseEntry);
            IF previous = NIL THEN
                U^.FirstVol := V^.next;
            ELSE
                previous^.next := V^.next;
            END (*IF*);
            DISPOSE (V);
        END (*IF*);
    END DeleteVolume;

(********************************************************************************)
(*                                  PARSER                                      *)
(********************************************************************************)

PROCEDURE ReadUserData (name: ARRAY OF CHAR): User;

    (* Fetches the password, etc., for the user whose username is specified     *)
    (* as the argument.  Returns with category = NoSuchUser if the user's data  *)
    (* could not be found.                                                      *)

    VAR NextChar: CHAR;
        hini: OS2.HINI;
        bufptr: CharArrayPointer;
        BufferPosition, BufferSize: CARDINAL;

    (****************************************************************************)

    PROCEDURE VolumeInfo (U: User;  previous: Volume): Volume;

        (* Creates a new volume record, and tack it onto the tail of the list.  *)

        (*     <volumeinfo> ->  <homedir>  { <dirrule> }                        *)
        (*     <homedir>    ->  <namestring>  |  <namestring> = <namestring>    *)

        VAR V: Volume;

        BEGIN
            V := MakeVolume (U, previous);

            (* Read the home directory. *)

            NameString (bufptr, BufferPosition, BufferSize, V^.Home, NextChar, CharSet{'='});
            IF NextChar = '=' THEN
                V^.Drive := V^.Home;
                NextChar := Scan (bufptr, BufferPosition, BufferSize);
                NameString (bufptr, BufferPosition, BufferSize, V^.Home, NextChar, CharSet{});
            ELSIF V^.Home[1] = ':' THEN
                V^.Drive[0] := CAP(V^.Home[0]);
                V^.Drive[1] := Nul;
            ELSE
                V^.Drive[0] := Nul;
            END (*IF*);

            (* Create a descriptor record for the user's root directory.  *)

            V^.BaseEntry := CreateDirEntry ("/");

            (* If there's anything left, it must be the optional directory rule.    *)

            IF NextChar <> Nul THEN
                 DirRule (bufptr, BufferPosition, BufferSize, V^.BaseEntry, NextChar);
            END (*IF*);
            RETURN V;

        END VolumeInfo;

    (************************************************************************)
    (*                     BODY OF READUSERDATA                             *)
    (*                                                                      *)
    (*     <result>  ->  <userclass> <password> { <volumeinfo> }*           *)
    (*     <userclass> -> G | U | N | M                                     *)
    (*     <password>  -> <namestring>                                      *)
    (*                                                                      *)
    (* Note: for simplicity, a password must always be specified, even if   *)
    (* it is not going to be used.                                          *)
    (*                                                                      *)
    (************************************************************************)

    VAR result: User;
        previousV: Volume;

    BEGIN       (* Body of ReadUserData *)

        NEW (result);  result^.FirstVol := NIL;

        IF UserInINIFile (hini, name) THEN

            IF NOT INIGet (hini, name, "Category", result^.category) THEN
                result^.category := NormalUser;
            END (*IF*);
            IF NOT INIGet (hini, name, "Password", result^.Password) THEN
                result^.Password := "@";
            END (*IF*);
            IF NOT INIGet (hini, name, "UserLimit", result^.UserLimit) THEN
                result^.UserLimit := MAX(CARDINAL);
            END (*IF*);
            IF OS2.PrfQueryProfileSize (hini, name, "Volume", BufferSize)
                                                    AND (BufferSize <> 0) THEN
                ALLOCATE (bufptr, BufferSize);
                OS2.PrfQueryProfileData (hini, name, "Volume", bufptr, BufferSize);
                BufferPosition := 0;
            ELSE
                bufptr := NIL;
            END (*IF*);

            IF bufptr <> NIL THEN
                NextChar := Scan (bufptr, BufferPosition, BufferSize);
                previousV := NIL;
                WHILE NextChar <> Nul DO
                    previousV := VolumeInfo (result, previousV);
                    IF NextChar = ';' THEN
                        NextChar := Scan (bufptr, BufferPosition, BufferSize);
                    END (*IF*);
                END (*WHILE*);
                DEALLOCATE (bufptr, BufferSize);
            END (*IF*);

        END (*IF*);

        RETURN result;

    END ReadUserData;

(************************************************************************)

PROCEDURE StoreVolumeInfo (V: Volume;  VAR (*INOUT*) result: ARRAY OF CHAR;
                                       VAR (*INOUT*) index: CARDINAL);

    (* Puts the volume V information into the result array, starting    *)
    (* at result[index].  On return index has been updated to the next  *)
    (* unused array slot.                                               *)

    BEGIN
        AppendQuotedString (V^.Drive, result, index);
        result[index] := '=';  INC(index);
        AppendQuotedString (V^.Home, result, index);
        StorePermissionData (V^.BaseEntry, result, index);
    END StoreVolumeInfo;

(************************************************************************)

PROCEDURE SizeOfThisVolumeData (V: Volume): CARDINAL;

    (* Works out how much space will be needed in the INI file to       *)
    (* record the information for this volume.                          *)

    BEGIN
        RETURN LENGTH(V^.Drive) + LENGTH(V^.Home) + 5
                                + SizeOfPermissionData (V^.BaseEntry);
    END SizeOfThisVolumeData;

(************************************************************************)

PROCEDURE SizeOfAllVolumeData (V: Volume): CARDINAL;

    (* Works out how much space will be needed in the INI file to       *)
    (* record the information for this volume and all following volumes.*)

    VAR result: CARDINAL;

    BEGIN
        result := 0;
        IF V <> NIL THEN
            REPEAT
                INC (result);                    (* allow for semicolon *)
                INC (result, SizeOfThisVolumeData(V));
                V := V^.next;
            UNTIL V = NIL;
            DEC (result);                     (* one semicolon too many *)
        END (*IF*);
        RETURN result;
    END SizeOfAllVolumeData;

(********************************************************************************)

PROCEDURE EncodeVolumeInformation (V: Volume;  VAR (*OUT*) bufptr: CharArrayPointer;
                                               VAR (*OUT*) BufferSize: CARDINAL);

    (* Takes a linked list of volumes, and writes the information to bufptr^ in *)
    (* a form suitable for writing to the INI file.  It is the caller's         *)
    (* responsibility to deallocate the buffer space after use.                 *)

    VAR index: CARDINAL;

    BEGIN

        BufferSize := SizeOfAllVolumeData (V);
        ALLOCATE (bufptr, BufferSize);

        (* Store all volume information in the bufptr^ array. *)

        index := 0;
        IF V <> NIL THEN
            StoreVolumeInfo (V, bufptr^, index);  V := V^.next;
        END (*IF*);
        WHILE V <> NIL DO
            bufptr^[index] := ';';  INC(index);
            StoreVolumeInfo (V, bufptr^, index);  V := V^.next;
        END (*WHILE*);

    END EncodeVolumeInformation;

(********************************************************************************)
(*                      WRITING USER DATA TO THE INI FILE                       *)
(********************************************************************************)

PROCEDURE StoreUserData (U: User;  username: ARRAY OF CHAR);

    (* Writes permission data to the INI file. *)

    VAR hini: OS2.HINI;
        size: CARDINAL;
        bufptr: CharArrayPointer;

    BEGIN
        hini := OurINIHandle();
        IF hini = OS2.NULLHANDLE THEN
            RETURN;
        END (*IF*);
        OS2.PrfWriteProfileData (hini, username, "Category",
                                ADR(U^.category), SIZE(UserCategory));
        OS2.PrfWriteProfileData (hini, username, "Password",
                                ADR(U^.Password), SIZE(PassString));
        OS2.PrfWriteProfileData (hini, username, "UserLimit",
                                ADR(U^.UserLimit), SIZE(CARDINAL));

        (* Store all volume information in the bufptr^ array. *)

        EncodeVolumeInformation (U^.FirstVol, bufptr, size);

        (* Copy the result from bufptr^ to the INI file. *)

        IF size > 0 THEN
            OS2.PrfWriteProfileData (hini, username, "Volume", bufptr, size);
            DEALLOCATE (bufptr, size);
        END (*IF*);

    END StoreUserData;

(********************************************************************************)
(*                      WRITING A USER'S PERMISSION DATA                        *)
(********************************************************************************)

PROCEDURE WritePermissionData (U: User;  name: ARRAY OF CHAR);

    (* Writes permission data to both the INI file and the user's PRM file. *)

    BEGIN
        StoreUserData (U, name);
    END WritePermissionData;

(********************************************************************************)
(*                        DISCARDING A USER RECORD                              *)
(********************************************************************************)

PROCEDURE DestroyUserData (VAR (*INOUT*) U: User);

    (* Discards the data structure.  *)

    VAR V, V1: Volume;

    BEGIN
        IF U <> NIL THEN
            V := U^.FirstVol;

            (* Kill each volume. *)

            WHILE V <> NIL DO
                KillList (V^.BaseEntry);  V1 := V^.next;
                DISPOSE (V);  V := V1;
            END (*WHILE*);

            DISPOSE (U);
        END (*IF*);
    END DestroyUserData;

(************************************************************************)

PROCEDURE RemoveUser (name: ARRAY OF CHAR);

    (* Deletes this user's INI file entry, or permission file, *)
    (* whichever is found first.                               *)

    VAR hini: OS2.HINI;

    BEGIN
        hini := OurINIHandle();
        IF hini <> OS2.NULLHANDLE THEN
           OS2.PrfWriteProfileData (hini, name, NIL, NIL, 0);
        END (*IF*);
    END RemoveUser;

(************************************************************************)
(*                        ShortString EDITOR                            *)
(************************************************************************)

PROCEDURE WriteShortString (w: Window;  p: ADDRESS;  width: CARDINAL);

    VAR address: PassStringPtr;

    BEGIN
        address := p;
        WriteString (w, address^);
    END WriteShortString;

(************************************************************************)

PROCEDURE EditShortString (w: Window;  VAR (*INOUT*) p: ADDRESS;
                                         stringsize, width: CARDINAL);

    VAR address: PassStringPtr;

    BEGIN
        address := p;
        EditString (w, address^, stringsize, width);
    END EditShortString;

(********************************************************************************)
(*                       EDITING THE DIRECTORY PERMISSIONS                      *)
(********************************************************************************)

PROCEDURE UpdateDirectoryDisplay (V: Volume;  LB: ListBox);

    (* Ensures that the directory tree in V contains precisely the directories  *)
    (* that actually exist.  Then removes the old contents of LB, and creates   *)
    (* an updated list from the current state of V.                             *)

    BEGIN
        UpdateDirectoryTree (V^.Home, V^.BaseEntry);
        InsertThreads (V^.BaseEntry);
        ClearListBox (LB);
        ShowTree (LB, V^.BaseEntry, 1);
    END UpdateDirectoryDisplay;

(********************************************************************************)

PROCEDURE EditVolumeInfo (V: Volume;  Screen: VirtualScreen);

    (* Allows the user to edit the volume name, the home directory, and the     *)
    (* permissions for this volume.                                             *)

    CONST BoxTop = 7;  BoxLeft = 1;  BoxWidth = 70;

    VAR ddw, LBWindow, bottombar, PleaseWait: Window;
        R: Structure;  LB: ListBox;
        abort, EditPermissions: BOOLEAN;
        ch: CHAR;
        BoxHeight, j: CARDINAL;
        OldHome: FileNameString;

    BEGIN
        BoxHeight := ScreenRows - BoxTop - 4;
        OldHome := "";

        (* Set up the listbox of directory permissions.  *)

        OpenWindowHidden (LBWindow, white, blue, BoxTop, BoxTop+BoxHeight+1,
                      BoxLeft, BoxLeft+BoxWidth+2, noframe, nodivider);
        MapToVirtualScreen (LBWindow, Screen);
        LB := CreateListBox (LBWindow, 1, 1, BoxHeight, BoxWidth);

        (* Create the drive/home window. *)

        OpenWindowHidden (ddw, white, black, 1, BoxTop-1, 0, 79, noframe, nodivider);
        MapToVirtualScreen (ddw, Screen);
        SetCursor (ddw, BoxTop-2, 1);  WriteString (ddw, "Directories");

        (* Create the bottom bar. *)

        OpenWindowHidden (bottombar, yellow, red, ScreenRows-1, ScreenRows-1, 0, 79, noframe, nodivider);
        MapToVirtualScreen (bottombar, Screen);
        WriteString (bottombar, " Esc exit  ");
        WriteChar (bottombar, CHR(24));  WriteChar (bottombar, CHR(25));
        WriteString (bottombar, " select");

        (* Create the drive/directory editing structure. *)

        SetCursor (ddw, 1, 2);  WriteString (ddw, "Virtual drive name");
        SetCursor (ddw, 2, 2);  WriteString (ddw, "Home directory");

        R := CreateField (ADR(V^.Drive), SIZE(FileNameString), ShortString, 1, 24, 32);
        Combine (R, StringField (V^.Home, 2, 24, 32));

        (* Here's the main editing loop. *)

        SetCursor (ddw, 1, 1);
        LOOP
            ScreenEdit (ddw, R, abort);
            IF abort THEN EXIT(*LOOP*) END(*IF*);

            (* Make sure that V^.Home finishes with a '/'. *)

            j := LENGTH (V^.Home);
            IF j = 0 THEN V^.Home[0] := '/'
            ELSIF V^.Home[j-1] = '\' THEN V^.Home[j-1] := '/'
            ELSIF V^.Home[j-1] <> '/' THEN
                IF j <= MAX(FileNameSubscript) THEN
                    V^.Home[j] := '/';  INC(j);
                END (*IF*);
                IF j <= MAX(FileNameSubscript) THEN
                    V^.Home[j] := CHR(0);
                END (*IF*);
            END (*IF*);

            (* Check the character that took us off the edge.  If it's *)
            (* "cursor down", proceed to editing permissions.  If it's *)
            (* "cursor up", leave this procedure.                      *)

            EditPermissions := FALSE;
            ch := GetKey (ddw);
            IF CAP(ch) = 'X' THEN
                EXIT(*LOOP*)
            ELSIF ch = CHR(0) THEN
                ch := GetKey (ddw);
                IF CAP(ch) = 'H' THEN EXIT(*LOOP*) END(*IF*);
                EditPermissions := ch = 'P';
            END (*IF*);

            IF EditPermissions THEN
                IF NOT Strings.Equal (V^.Home, OldHome) THEN
                    OpenWindowHidden (PleaseWait, yellow, red, BoxTop, BoxTop,
                          25, 62, noframe, nodivider);
                    MapToVirtualScreen (PleaseWait, Screen);
                    WriteString (PleaseWait, " Rescanning directories, please wait");
                    UpdateDirectoryDisplay (V, LB);
                    CloseWindow (PleaseWait);
                    OldHome := V^.Home;
                END (*IF*);
                CASE EditPermissionListBox (V^.BaseEntry, LBWindow, LB, BoxHeight) OF
                  |  TrueExit:   EXIT (*LOOP*);
                  |  CursorOffTop:   (* Go back to editing top part *);
                                 SetCursor (ddw, 2, 1);
                END (*CASE*);
            END (*IF*);

        END (*LOOP*);

        DestroyListBox (LB);
        CloseWindow (LBWindow);
        DeleteStructure (R);
        CleanUpTree (V^.BaseEntry);
        CloseWindow (ddw);
        CloseWindow (bottombar);

    END EditVolumeInfo;

(********************************************************************************)
(*                     EDITING THE LIST OF HOME DIRECTORIES                     *)
(********************************************************************************)

PROCEDURE SelectedVolume (LB: ListBox;  U: User): Volume;

    (* Returns the Volume that's currently selected in the ListBox. *)

    VAR j: CARDINAL;  V: Volume;

    BEGIN
        j := LBCurrentItemNumber(LB);
        V := U^.FirstVol;
        WHILE (j > 1) AND (V <> NIL) DO
            DEC(j);  V := V^.next;
        END (*WHILE*);
        RETURN V;
    END SelectedVolume;

(********************************************************************************)

PROCEDURE DriveAndHome (V: Volume;  VAR (*OUT*) result: FileNameString);

    (* Formats the drive name and home for this volume into a one-line string. *)

    CONST DriveFieldSize = 12;
    VAR j: CARDINAL;

    BEGIN
        Strings.Assign (V^.Drive, result);
        j := LENGTH (result);
        IF j > DriveFieldSize THEN
            j := DriveFieldSize;
        END (*IF*);
        WHILE j < DriveFieldSize DO
            result[j] := ' ';  INC(j);
        END (*WHILE*);
        result[j] := Nul;
        Strings.Append (V^.Home, result);
    END DriveAndHome;

(********************************************************************************)

PROCEDURE EditHomeDirListBox (U: User;  w: Window;  Screen: VirtualScreen;
                                                      LB: ListBox): ExitCode;

    (* This procedure edits a listbox of home directories for user U. *)

    CONST Esc = CHR(27);

    VAR bottombar: Window;  ch: CHAR;
        result: ExitCode;  V: Volume;
        BoxEntry: FileNameString;

    BEGIN
        HighlightOn (LB);
        OpenWindowHidden (bottombar, yellow, red, ScreenRows-1, ScreenRows-1, 0, 79, noframe, nodivider);
        MapToVirtualScreen (bottombar, Screen);
        WriteString (bottombar, " A add  E edit  Del delete  Esc exit  ");
        WriteChar (bottombar, CHR(24));  WriteChar (bottombar, CHR(25));
        WriteString (bottombar, " select");
        LOOP
            IF CursorMovements(LB) THEN
                result := CursorOffTop;
                EXIT (*LOOP*);
            END (*IF*);
            ch := GetKey(w);
            IF ch = CHR(0) THEN
                ch := GetKey(w);
                IF ch = 'S' THEN                     (* Del = delete *)
                    V := SelectedVolume (LB, U);
                    IF V <> NIL THEN
                        DeleteVolume (U, V);
                        LBDeleteCurrent (LB);
                    END (*IF*);
                END (*IF*);
            ELSIF CAP(ch) = 'A' THEN                    (* A = add *)
                V := MakeVolume (U, SelectedVolume (LB, U));
                EditVolumeInfo (V, Screen);
                DriveAndHome (V, BoxEntry);
                LBInsertAfter (LB, BoxEntry);
            ELSIF (CAP(ch) = 'E') OR (ch = CHR(13)) THEN   (* E = edit *)
                V := SelectedVolume (LB, U);
                IF V <> NIL THEN
                    EditVolumeInfo (V, Screen);
                    DriveAndHome (V, BoxEntry);
                    ReplaceCurrent (LB, BoxEntry);
                END (*IF*);
            ELSIF ch = Esc THEN                    (* Esc = exit *)
                result := TrueExit;
                EXIT (*LOOP*);
            END (*IF*);
        END (*LOOP*);
        HighlightOff (LB);
        CloseWindow (bottombar);
        RETURN result;
    END EditHomeDirListBox;

(********************************************************************************)
(*                          MASTER EDITING PROCEDURE                            *)
(********************************************************************************)

PROCEDURE SkipKey (w: Window);

    (* Discards the next keyboard character. *)

    VAR ch: CHAR;

    BEGIN
        ch := GetKey (w);
        IF ch = CHR(0) THEN
            EVAL (GetKey (w));
        END (*IF*);
    END SkipKey;

(********************************************************************************)

PROCEDURE EditUser (Screen: VirtualScreen;  VAR (*INOUT*) name: ARRAY OF CHAR);

    (* Top-level editor for the user permission data. *)

    CONST BoxTop = 11;  BoxLeft = 1;  BoxWidth = 70;

    VAR TopBar, Background, UserData, HomeWindow, bottombar, ErrorWindow: Window;
        UserName: PassString;
        R: Structure;
        abort, EditHomeDir: BOOLEAN;  ch: CHAR;
        U: User;
        Vol: Volume;
        UserList: ListBox;
        HomeLabel: FileNameString;
        M: Menu;
        CatLabel: ARRAY [0..4] OF ItemText;
        BoxHeight, CategoryNumber: CARDINAL;

    BEGIN
        BoxHeight := ScreenRows - BoxTop - 4;
        Strings.Assign (name, UserName);

        OpenWindowHidden (TopBar, yellow, red, 0, 0, 0, 79, noframe, nodivider);
        MapToVirtualScreen (TopBar, Screen);
        WriteString (TopBar, "Username: ");
        WriteString (TopBar, UserName);

        OpenWindowHidden (Background, white, black, 1, ScreenRows-2, 0, 79, noframe, nodivider);
        MapToVirtualScreen (Background, Screen);
        SetCursor (Background, BoxTop-2, 1);  WriteString (Background, "Home directories");

        OpenWindowHidden (bottombar, yellow, red, ScreenRows-1, ScreenRows-1, 0, 79, noframe, nodivider);
        MapToVirtualScreen (bottombar, Screen);
        WriteString (bottombar, " Esc exit  ");
        WriteChar (bottombar, CHR(24));  WriteChar (bottombar, CHR(25));
        WriteString (bottombar, " select");

        U := ReadUserData (name);
        IF U = NIL THEN
            NEW (U);
            Strings.Assign ("", U^.Password);
            U^.FirstVol := NIL;
        END (*IF*);

        OpenWindowHidden (UserData, white, blue, 2, 7, 1, 70, noframe, nodivider);
        MapToVirtualScreen (UserData, Screen);

        (* Create the menu of user categories. *)

        CatLabel[1] := "NoPassword";
        CatLabel[2] := "Guest";
        CatLabel[3] := "User";
        CatLabel[4] := "Manager";
        CreateMenu (M, 4, CatLabel, 4);
        MenuColours (M, white, blue, blue, cyan, yellow);
        CategoryNumber := ORD (U^.category);

        (* Create the username/category/password editing structure. *)

        SetCursor (UserData, 1, 2);  WriteString (UserData, "User name");
        SetCursor (UserData, 2, 2);  WriteString (UserData, "Category");
        SetCursor (UserData, 3, 2);  WriteString (UserData, "Password");
        SetCursor (UserData, 4, 2);  WriteString (UserData, "User limit");

        R := CreateField (ADR(UserName), SIZE(UserName), ShortString, 1, 20, 32);
        Combine (R, MenuField (CategoryNumber, 2, 20, 1, 48, M));
        Combine (R, CreateField (ADR(U^.Password), SIZE(PassString), ShortString, 3, 20, 32));
        Combine (R, CardinalField (U^.UserLimit, 4, 20, 12));

        (* Set up the list of home directories.  *)

        OpenWindowHidden (HomeWindow, white, blue, BoxTop, BoxTop+BoxHeight+1,
                      BoxLeft, BoxLeft+BoxWidth+2, noframe, nodivider);
        MapToVirtualScreen (HomeWindow, Screen);
        UserList := CreateListBox (HomeWindow, 1, 1, BoxHeight, BoxWidth);
        Vol := U^.FirstVol;

        WHILE Vol <> NIL DO
            DriveAndHome (Vol, HomeLabel);
            LBAppend (UserList, HomeLabel);
            Vol := Vol^.next;
        END (*WHILE*);

        (* Here's the main editing loop. *)

        SetCursor (UserData, 1, 1);
        LOOP
            LOOP
                ScreenEdit (UserData, R, abort);
                ToLower (UserName);
                U^.category := VAL (UserCategory, CategoryNumber);
                IF abort THEN EXIT(*LOOP*) END(*IF*);

                (* Check the character that took us off the edge.  If it's *)
                (* "cursor down", proceed to editing directories.  If it's *)
                (* "cursor up", leave this procedure.                      *)

                EditHomeDir := FALSE;
                ch := GetKey (UserData);
                IF ch = CHR(0) THEN
                    ch := GetKey (UserData);
                    IF ch = 'H' THEN
                        EXIT (*LOOP*);
                    ELSE
                        EditHomeDir := ch = 'P';
                    END (*IF*);
                END (*IF*);

                IF EditHomeDir THEN
                    CASE EditHomeDirListBox (U, UserData, Screen, UserList) OF
                      |  TrueExit:   EXIT (*LOOP*);
                      |  CursorOffTop:   (* Go back to editing top part *);
                                     SetCursor (UserData, 4, 1);
                    END (*CASE*);
                END (*IF*);

            END (*LOOP*);

            (* At this point, "name" holds the original username, and "UserName"    *)
            (* holds the new name.  We need to check for the following errors:      *)
            (*    1.  UserName is empty.                                            *)
            (*    2.  UserName duplicates an existing name.                         *)

            IF UserName[0] = CHR(0) THEN
                OpenWindowHidden (ErrorWindow, white, black, BoxTop+2, BoxTop+4, 20, 59, noframe, nodivider);
                MapToVirtualScreen (ErrorWindow, Screen);
                WriteLn (ErrorWindow);
                WriteString (ErrorWindow, "  You have not assigned a user name");
                SkipKey (ErrorWindow);
                CloseWindow (ErrorWindow);
            ELSE
                Strings.Assign (name, HomeLabel);
                ToLower (HomeLabel);
                IF NOT Strings.Equal (HomeLabel, UserName) AND FindUser (UserName) THEN
                    OpenWindowHidden (ErrorWindow, white, black, BoxTop+2, BoxTop+5, 16, 63, noframe, nodivider);
                    MapToVirtualScreen (ErrorWindow, Screen);
                    WriteLn (ErrorWindow);
                    WriteString (ErrorWindow, " You've duplicated the name of an existing user");
                    WriteLn (ErrorWindow);
                    WriteString (ErrorWindow, "       Please assign a different name");
                    SkipKey (ErrorWindow);
                    CloseWindow (ErrorWindow);
                ELSE
                    EXIT (*LOOP*);
                END (*IF*);
            END (*IF*);

        END (*LOOP*);

        DestroyListBox (UserList);
        DeleteStructure (R);
        CloseWindow (UserData);
        CloseWindow (TopBar);
        CloseWindow (HomeWindow);

        (* Destroy the old user permission file, and create a new one. *)

        IF name[0] <> Nul THEN
            RemoveUser (name);
        END (*IF*);
        Strings.Assign (UserName, name);
        WritePermissionData (U, name);
        DestroyUserData (U);

        CloseWindow (bottombar);
        CloseWindow (Background);

    END EditUser;

(********************************************************************************)
(*                              INITIALISATION                                  *)
(********************************************************************************)

VAR dummy: CARDINAL;

BEGIN
    GetScreenSize (ScreenRows, dummy);
    ShortString := DefineFieldType (WriteShortString, EditShortString);
FINALLY
    DiscardFieldType (ShortString);
END Users.

