IMPLEMENTATION MODULE Security;

        (********************************************************)
        (*                                                      *)
        (*           Part of FtpServer Setup program            *)
        (*                                                      *)
        (*  Programmer:         P. Moylan                       *)
        (*  Started:            1 February 1998                 *)
        (*  Last edited:        9 July 1998                     *)
        (*  Status:             Working                         *)
        (*                                                      *)
        (********************************************************)

FROM SYSTEM IMPORT
    (* type *)  LOC, CARD8, ADDRESS,
    (* proc *)  CAST, ADR;

IMPORT Strings;

FROM SetupINI IMPORT
    (* proc *)  OurINIHandle;

FROM InetUtilities IMPORT
    (* proc *)  INIGet, INIPut, ConvertCard, Swap4;

FROM MultiScreen IMPORT
    (* type *)  ScreenGroup, VirtualScreen,
    (* proc *)  CreateScreenGroup, CreateVirtualScreen, MapToVirtualScreen,
                RemoveVirtualScreen, RemoveScreenGroup;

FROM ListBoxes IMPORT
    (* type *)  ListBox,
    (* proc *)  CreateListBox, HighlightOn, HighlightOff, CursorBackward,
                CursorForward, LBAppend,
                LBCurrent, LBDeleteCurrent, LBInsertAfter, ReplaceCurrent;

FROM ScreenEditor IMPORT
    (* type *)  Structure,
    (* proc *)  CardinalField, CreateField, Combine, ScreenEdit, DeleteStructure;

FROM FieldEditor IMPORT
    (* type *)  FieldType,
    (* proc *)  DefineFieldType, DiscardFieldType;

FROM Windows IMPORT
    (* type *)  Window, Colour, FrameType, DividerType,
    (* proc *)  OpenWindowHidden, CloseWindow, SetCursor,
                WriteString, WriteLn, GetKey, GetScreenSize, EditString;

FROM Keyboard IMPORT
    (* proc *)  PutBack, StuffKeyboardBuffer, StuffKeyboardBuffer2;

FROM TaskControl IMPORT
    (* proc *)  CreateTask;

FROM LowLevel IMPORT
    (* proc *)  EVAL, IAND;

FROM OS2 IMPORT
    (* type *)  HINI,
    (* proc *)  PrfQueryProfileSize, PrfQueryProfileData, PrfWriteProfileData;

FROM Storage IMPORT
    (* proc *)  ALLOCATE, DEALLOCATE;

(************************************************************************)

TYPE
    AddrPtr = POINTER TO AddrRecord;
    AddrRecord = RECORD
                     previous, next: AddrPtr;
                     address, mask: CARDINAL;
                 END (*RECORD*);

VAR
    (* Number of display rows on screen.  *)

    ScreenRows: CARDINAL;

    (* The screen group for our screen page.  *)

    OurGroup: ScreenGroup;

    (* The screen page used by this module. *)

    OurPage: VirtualScreen;

    (* Header and footer windows. *)

    TopBar, BottomBar: Window;

    (* Limit on number of users from the same IP address. *)

    SameIPLimit: CARDINAL;

    (* Handle for our INI file. *)

    hini: HINI;

    (* List heads for the allow/exclude lists. *)

    AddrListHead: ARRAY [0..1] OF AddrPtr;

    (* Field type for dotted address editing. *)

    DottedField: FieldType;

    (* IP address list listbox height. *)

    BoxHeight: CARDINAL;

(************************************************************************)
(*          GETTING/PUTTING IP ADDRESS CONTROLS FROM INI FILE           *)
(************************************************************************)

PROCEDURE LoadIPAddressList (IPAllow: BOOLEAN): AddrPtr;

    (* Constructs an IP address list from the INI file data.  It's      *)
    (* either the "allow" or the "deny" list, depending on the          *)
    (* Boolean parameter.                                               *)

    VAR option: ARRAY [0..7] OF CHAR;  j, size: CARDINAL;
        bufptr: POINTER TO ARRAY [0..1023] OF CARDINAL;
        head, current: AddrPtr;

    BEGIN
        IF IPAllow THEN
            Strings.Assign ("IPAllow", option);
        ELSE
            Strings.Assign ("IPDeny", option);
        END (*IF*);
        head := NIL;
        IF PrfQueryProfileSize (hini, "$SYS", option, size) THEN
            IF size <> 0 THEN
                ALLOCATE (bufptr, size);
                PrfQueryProfileData (hini, "$SYS", option, bufptr, size);
                j := 0;  current := NIL;
                WHILE size > 0 DO
                    IF current = NIL THEN
                        NEW (head);
                        head^.previous := NIL;
                        current := head;
                    ELSE
                        NEW (current^.next);
                        current^.next^.previous := current;
                        current := current^.next;
                    END (*IF*);
                    current^.next := NIL;
                    current^.address := bufptr^[j];  INC(j);
                    current^.mask := bufptr^[j];  INC(j);
                    DEC (size, 2*SIZE(CARDINAL));
                END (*WHILE*);
                DEALLOCATE (bufptr, size);
            END (*IF*);
        END (*IF*);
        RETURN head;
    END LoadIPAddressList;

(************************************************************************)

PROCEDURE StoreIPAddressList (IPAllow: BOOLEAN;  head: AddrPtr);

    (* Stores an IP address list to the INI file.  It's either the      *)
    (* "allow" or the "deny" list, depending on the Boolean parameter.  *)

    VAR option: ARRAY [0..7] OF CHAR;  j, size: CARDINAL;
        bufptr: POINTER TO ARRAY [0..1023] OF CARDINAL;
        current: AddrPtr;

    BEGIN
        IF IPAllow THEN
            Strings.Assign ("IPAllow", option);
        ELSE
            Strings.Assign ("IPDeny", option);
        END (*IF*);
        size := 0;  current := head;
        WHILE current <> NIL DO
            INC (size, 2*SIZE(CARDINAL));  current := current^.next;
        END (*WHILE*);
        IF size = 0 THEN bufptr := NIL
        ELSE ALLOCATE (bufptr, size);
        END (*IF*);
        current := head;  j := 0;
        WHILE current <> NIL DO
            bufptr^[j] := current^.address;  INC(j);
            bufptr^[j] := current^.mask;  INC(j);
            current := current^.next;
        END (*WHILE*);
        PrfWriteProfileData (hini, "$SYS", option, bufptr, size);
        IF bufptr <> NIL THEN
            DEALLOCATE (bufptr, size);
        END (*IF*);
    END StoreIPAddressList;

(************************************************************************)
(*          CONVERTING AN ADDRESS RECORD TO HUMAN-READABLE FORM         *)
(************************************************************************)

PROCEDURE CardToDotted (value: ARRAY OF LOC;  VAR (*OUT*) text: ARRAY OF CHAR;
                                              VAR (*INOUT*) place: CARDINAL);

    (* Converts the four-byte value to a string of the form a.b.c.d,    *)
    (* starting at text[place].  On return, place is the first location *)
    (* in the text array that hasn't been used.                         *)

    VAR j: [0..2];

    BEGIN
        FOR j := 0 TO 2 DO
            ConvertCard (CAST(CARD8,value[j]), text, place);
            text[place] := '.';  INC(place);
        END (*FOR*);
        ConvertCard (CAST(CARD8,value[3]), text, place);
        text[place] := CHR(0);
    END CardToDotted;

(************************************************************************)

PROCEDURE AddrRecordToText (p: AddrPtr;  VAR (*OUT*) text: ARRAY OF CHAR);

    (* Converts p^ to a text string. *)

    CONST MaskPos = 17;

    VAR place: CARDINAL;

    BEGIN
        place := 0;
        CardToDotted (p^.address, text, place);
        WHILE place < MaskPos DO
            text[place] := ' ';  INC (place);
        END (*WHILE*);
        CardToDotted (p^.mask, text, place);
    END AddrRecordToText;

(************************************************************************)

PROCEDURE DottedToCard (text: ARRAY OF CHAR): CARDINAL;

    (* Converts a dotted-quad text string to a cardinal value. *)

    VAR pos, result: CARDINAL;  j: [0..3];

    (********************************************************************)

    PROCEDURE ConvertOneComponent(): CARDINAL;

        TYPE CharSet = SET OF CHAR;
        CONST Digits = CharSet {'0'..'9'};

        VAR answer: CARDINAL;

        BEGIN
            answer := 0;
            WHILE text[pos] IN Digits DO
                answer := 10*answer + (ORD(text[pos]) - ORD('0'));
                INC (pos);
            END (*WHILE*);
            RETURN answer;
        END ConvertOneComponent;

    (********************************************************************)

    BEGIN
        pos := 0;  result := 0;
        FOR j := 0 TO 3 DO
            result := 256*result + ConvertOneComponent();
            IF text[pos] = '.' THEN INC(pos) END(*IF*);
        END (*FOR*);
        RETURN Swap4(result);
    END DottedToCard;

(************************************************************************)

PROCEDURE WriteDottedField (w: Window;  addr: ADDRESS;  width: CARDINAL);

    VAR text: ARRAY [0..40] OF CHAR;  position: CARDINAL;
        loc: POINTER TO CARDINAL;

    BEGIN
        loc := addr;  position := 0;
        CardToDotted (loc^, text, position);
        WriteString (w, text);
    END WriteDottedField;

(************************************************************************)
(*                     EDITING AN ADDRESS RECORD                        *)
(************************************************************************)

PROCEDURE EditDottedField (w: Window;  VAR (*INOUT*) addr: ADDRESS;
                                             dummy, width: CARDINAL);

    VAR text: ARRAY [0..31] OF CHAR;  position: CARDINAL;
        place: POINTER TO CARDINAL;

    BEGIN
        place := addr;  position := 0;
        CardToDotted (place^, text, position);
        EditString (w, text, 32, width);
        place^ := DottedToCard (text);
    END EditDottedField;

(************************************************************************)

PROCEDURE EditAddrRecord (p: AddrPtr);

    (* Allows screen editing of the address and mask values. *)

    VAR w: Window;  R: Structure;  abort: BOOLEAN;

    BEGIN
        OpenWindowHidden (w, white, blue, 2, 5, 10, 70, noframe, nodivider);
        MapToVirtualScreen (w, OurPage);
        SetCursor (w, 2, 2);  WriteString (w, "Mask");
        SetCursor (w, 1, 2);  WriteString (w, "Address");
        R := CreateField (ADR(p^.address), 0, DottedField, 1, 20, 16);
        Combine (R, CreateField (ADR(p^.mask), 0, DottedField, 2, 20, 16));
        ScreenEdit (w, R, abort);
        IF NOT abort THEN
            p^.address := IAND(p^.address, p^.mask);
        END (*IF*);
        CloseWindow (w);
    END EditAddrRecord;

(************************************************************************)
(*                  EDITING THE INCLUDE/EXCLUDE LISTS                   *)
(************************************************************************)

PROCEDURE IPListEditor (listnumber: CARDINAL;  w: Window;  LB: ListBox;
                                VAR (*INOUT*) current: AddrPtr);

    (********************************************************************)

    PROCEDURE UpARow(): BOOLEAN;
        BEGIN
            IF (current = NIL) OR (current^.previous = NIL) THEN RETURN FALSE;
            ELSE
                current := current^.previous;
                RETURN CursorBackward (LB);
            END (*IF*);
        END UpARow;

    (********************************************************************)

    PROCEDURE DownARow(): BOOLEAN;
        BEGIN
            IF (current = NIL) OR (current^.next = NIL) THEN RETURN FALSE
            ELSE
                current := current^.next;
                EVAL (CursorForward(LB));
                RETURN TRUE;
            END (*IF*);
        END DownARow;

    (********************************************************************)

    VAR ch: CHAR;  text: ARRAY [0..79] OF CHAR;
        j: CARDINAL;  q: AddrPtr;

    BEGIN
        HighlightOn (LB);
        LOOP
            ch := GetKey(w);
            IF CAP(ch) = 'A' THEN                       (* A = add *)
                NEW (q);
                WITH q^ DO
                    previous := current;  next := NIL;
                    address := 0;  mask := MAX(CARDINAL);
                END (*WITH*);
                EditAddrRecord (q);
                IF current = NIL THEN
                    AddrListHead[listnumber] := q;
                ELSE
                    q^.next := current^.next;
                    current^.next := q;
                    IF q^.next <> NIL THEN
                        q^.next^.previous := q;
                    END (*IF*);
                END (*IF*);
                AddrRecordToText (q, text);
                LBInsertAfter (LB, text);
                current := q;
            ELSIF CAP(ch) = 'E' THEN                    (* E = edit *)
                IF current <> NIL THEN
                    EditAddrRecord (current);
                    AddrRecordToText (current, text);
                    ReplaceCurrent (LB, text);
                END (*IF*);
            ELSIF CAP(ch) = 'X' THEN                    (* X = exit *)
                PutBack (ch);
                EXIT (*LOOP*);
            ELSIF ch = CHR(0) THEN
                ch := GetKey(w);
                IF (ch = "K") OR (ch = "M") THEN        (* cursor left/right *)
                    PutBack (ch);  PutBack (CHR(0));
                    EXIT (*LOOP*);
                ELSIF ch = "H" THEN                     (* cursor up *)
                    IF NOT UpARow() THEN
                        PutBack ("H");  PutBack (CHR(0));
                        EXIT (*LOOP*);
                    END (*IF*);
                ELSIF ch = "P" THEN                     (* cursor down *)
                    EVAL (DownARow());
                ELSIF ch = "G" THEN                     (* home *)
                    WHILE UpARow() DO
                    END (*WHILE*);
                ELSIF ch = "O" THEN                     (* end *)
                    WHILE DownARow() DO
                    END (*WHILE*);
                ELSIF ch = "I" THEN                     (* page up *)
                    j := BoxHeight;
                    WHILE (j > 0) AND UpARow() DO
                        DEC (j);
                    END (*WHILE*);
                ELSIF ch = "Q" THEN                     (* page down *)
                    j := BoxHeight;
                    WHILE (j > 0) AND DownARow() DO
                        DEC (j);
                    END (*WHILE*);
                ELSIF ch = 'S' THEN                     (* Del = delete *)
                    IF current <> NIL THEN
                        q := current;
                        IF current^.previous = NIL THEN
                            AddrListHead[listnumber] := current^.next;
                        ELSE
                            current^.previous^.next := current^.next;
                        END (*IF*);
                        IF current^.next = NIL THEN
                            current := current^.previous;
                        ELSE
                            current^.next^.previous := current^.previous;
                            current := current^.next;
                        END (*IF*);
                        DISPOSE (q);
                        LBDeleteCurrent (LB);
                    END (*IF*);
                END (*IF*);
            END (*IF*);
        END (*LOOP*);
        HighlightOff (LB);
    END IPListEditor;

(************************************************************************)
(*                          MAIN EDITING TASK                           *)
(************************************************************************)

PROCEDURE SecurityEditor;

    (* Runs as a separate task.  This is for allowing the user to       *)
    (* edit the "same IP limit" and the IP permit/deny lists.           *)

    CONST BoxTop = 6;  BoxLeft = 2;  BoxWidth = 33;

    VAR w, bottombar: Window;  R: Structure;
        ch: CHAR;  abort, EditIPLists: BOOLEAN;
        listnumber: [0..1];
        IPControl: ARRAY [0..1] OF RECORD
                                       win: Window;
                                       LB: ListBox;
                                       current: AddrPtr;
                                   END (*RECORD*);
        text: ARRAY [0..39] OF CHAR;

    BEGIN
        IF NOT INIGet (hini, "$SYS", "SameIPLimit", SameIPLimit) THEN
            SameIPLimit := MAX(CARDINAL);
        END (*IF*);;

        OpenWindowHidden (w, black, white, 2, 4, 10, 70, noframe, nodivider);
        MapToVirtualScreen (w, OurPage);
        SetCursor (w, 1, 2);  WriteString (w, "Same IP limit");
        R := CardinalField (SameIPLimit, 1, 32, 12);

        BoxHeight := ScreenRows - BoxTop - 7;
        FOR listnumber := 0 TO 1 DO
            WITH IPControl[listnumber] DO
                OpenWindowHidden (win, black, white, BoxTop, BoxTop+BoxHeight+4,
                     BoxLeft+40*listnumber, BoxLeft+40*listnumber+BoxWidth+2, noframe, nodivider);
                MapToVirtualScreen (win, OurPage);
                SetCursor (win, 1, 6);
                IF listnumber = 0 THEN
                    WriteString (win, "PERMIT");
                ELSE
                    WriteString (win, "  DENY");
                END (*IF*);
                WriteString (win, " ACCESS FROM");
                SetCursor (win, 2, 3);
                WriteString (win, "Address             Mask");
                LB := CreateListBox (win, 4, 1, BoxHeight, BoxWidth);
                current := LoadIPAddressList (listnumber=0);
                AddrListHead[listnumber] := current;
                WHILE current <> NIL DO
                    AddrRecordToText (current, text);
                    LBAppend (LB, text);
                    current := current^.next;
                END (*WHILE*);
                current := AddrListHead[listnumber];
            END (*WITH*);
        END (*FOR*);

        listnumber := 0;
        LOOP
            ScreenEdit (w, R, abort);

            (* Check the character that took us off the edge.  If it's *)
            (* "cursor down", proceed to editing the IP lists.         *)

            EditIPLists := FALSE;
            ch := GetKey (w);
            IF ch = CHR(0) THEN
                ch := GetKey (w);
                EditIPLists := ch = 'P';
            END (*IF*);

            IF EditIPLists THEN
                OpenWindowHidden (bottombar, yellow, red, ScreenRows-1, ScreenRows-1, 0, 39, noframe, nodivider);
                MapToVirtualScreen (bottombar, OurPage);
                WriteString (bottombar, " A add  E edit  Del delete  X exit");
                LOOP
                    WITH IPControl[listnumber] DO
                        IPListEditor (listnumber, win, LB, current);
                    END (*WITH*);

                    (* Check the character that took us off the edge.  *)
                    (* Cursor up means go back to editing the R        *)
                    (* structure.  Cursor left or right can mean       *)
                    (* going to the other list.  Other characters can  *)
                    (* be ignored.                                     *)

                    ch := GetKey (w);
                    IF CAP(ch) = 'X' THEN                       (* X = exit *)
                        StuffKeyboardBuffer2 (CHR(63));
                        StuffKeyboardBuffer ('X');
                    ELSIF ch = CHR(0) THEN
                        ch := GetKey (w);
                        IF ch = 'K' THEN listnumber := 0
                        ELSIF ch = 'M' THEN listnumber := 1
                        ELSIF ch = 'H' THEN EXIT (*LOOP*);
                        END (*IF*);
                    END (*IF*);

                END (*LOOP*);

                CloseWindow (bottombar);

            END (*IF*);

        END (*LOOP*);

    END SecurityEditor;

(************************************************************************)
(*                         INITIALISATION                               *)
(************************************************************************)

PROCEDURE CreateHeadings;

    BEGIN
        OpenWindowHidden (TopBar, yellow, red, 0, 0, 0, 79, noframe, nodivider);
        MapToVirtualScreen (TopBar, OurPage);
        WriteString (TopBar, "    FTPSERVER SECURITY SETUP");
        OpenWindowHidden (BottomBar, yellow, red, ScreenRows-1, ScreenRows-1, 0, 79, noframe, nodivider);
        MapToVirtualScreen (BottomBar, OurPage);
        SetCursor (BottomBar, 0, 67);
        WriteString (BottomBar, "F5 next page");
    END CreateHeadings;

(************************************************************************)

VAR dummy: CARDINAL;

BEGIN
    DottedField := DefineFieldType (WriteDottedField, EditDottedField);
    GetScreenSize (ScreenRows, dummy);
    hini := OurINIHandle();
    OurGroup := CreateScreenGroup (1);
    OurPage := CreateVirtualScreen (OurGroup);
    CreateHeadings;
    CreateTask (SecurityEditor, 2, "Security editor");
FINALLY
    INIPut (hini, "$SYS", "SameIPLimit", SameIPLimit);
    CloseWindow (BottomBar);
    StoreIPAddressList (TRUE, AddrListHead[0]);
    StoreIPAddressList (FALSE, AddrListHead[1]);
    CloseWindow (TopBar);
    RemoveVirtualScreen (OurPage);
    RemoveScreenGroup (OurGroup);
    DiscardFieldType (DottedField);
END Security.

