IMPLEMENTATION MODULE LoggedOnUsers;

        (********************************************************)
        (*                                                      *)
        (*     Keeps track of the currently logged-on users     *)
        (*                                                      *)
        (*  Programmer:         P. Moylan                       *)
        (*  Started:            24 October 1997                 *)
        (*  Last edited:        02 July 1998                    *)
        (*  Status:             OK so far                       *)
        (*                                                      *)
        (********************************************************)


IMPORT SysClock, Strings, OS2;

FROM FtpTransfers IMPORT
    (* type *)  ClientFileInfo,
    (* proc *)  CreateSession, CloseSession, FindUser, GetUserName,
                SetPort, CloseUser, KillDataChannel;

FROM FDUsers IMPORT
    (* type *)  UserCategory;

FROM Semaphores IMPORT
    (* type *)  Semaphore;

FROM TaskControl IMPORT
    (* type *)  Lock,
    (* proc *)  CreateLock, Obtain, Release;

FROM Storage IMPORT
    (* proc *)  ALLOCATE, DEALLOCATE;

FROM InetUtilities IMPORT
    (* proc *)  IPToString, SendString, SendEOL, ConvertCardZ,
                WriteError, OpenINIFile, INIGet,
                SendCard;

FROM Sockets IMPORT
    (* type *)  Socket, SockAddr,
    (* proc *)  getpeername, so_cancel;

FROM Conversions IMPORT
    (* proc *)  CardinalToString, StringToCardinal;

FROM LowLevel IMPORT
    (* proc *)  EVAL, IAND;

FROM GlassTTY IMPORT
    (* proc *)  WriteString;

FROM Keyboard IMPORT
    (* proc *)  NotDetached;

(********************************************************************************)

TYPE
    (* A place to store information about a particular session.                 *)
    (*   SessionID        a unique session identifier                           *)
    (*   FileInfo         the user's file permissions, current options, etc.    *)
    (*   StartTime        the time when this session started, for logging       *)
    (*   ClientIPAddress  the client's IP address, for logging                  *)
    (*   IsGuest          TRUE iff this is a guest user                         *)

    ClientDataPointer = POINTER TO
                          RECORD
                              SessionID: ARRAY [0..5] OF CHAR;
                              FileInfo: ClientFileInfo;
                              StartTime: SysClock.DateTime;
                              ClientIPAddress: CARDINAL;
                              IsGuest: BOOLEAN;
                          END (*RECORD*);

    (* A record used to form a linked list of all ClientDataPointer     *)
    (* objects which are currently active.                              *)

    SessionPointer = POINTER TO
                         RECORD
                             this: ClientDataPointer;
                             next: SessionPointer;
                         END (*RECORD*);

    (* A record used in the checking of which IP addresses to allow and *)
    (* which to exclude.                                                *)

    AddressListPointer = POINTER TO
                             RECORD
                                 IPAddress, mask: CARDINAL;
                                 next: AddressListPointer;
                             END (*RECORD*);

    (* A linked list of such records. *)

    IPAddressList = RECORD
                        head, tail: AddressListPointer;
                    END (*RECORD*);

(********************************************************************************)

VAR
    (* List of active sessions, and a lock to protect it. *)

    SessionList: RECORD
                     head, tail: SessionPointer;
                 END (*RECORD*);
    SessionListLock: Lock;

    (* Count of number of guest users, a lock to protect it, and  *)
    (* a limit on the number of guest users.                      *)

    GuestCount: CARDINAL;
    GuestCountLock: Lock;
    GuestLimit: CARDINAL;

    (* List of permitted client addresses, list of excluded client addresses,   *)
    (* and a shared lock to protect these two.                                  *)

    AllowList, DenyList: IPAddressList;
    AddressCheckLock: Lock;

    (* Limit on number of simultaneous connections from the same IP address. *)

    SameIPLimit: CARDINAL;

(********************************************************************************)

PROCEDURE SetGuestLimit (limit: CARDINAL);

    (* Sets the limit on the number of simultaneous guest users. *)

    BEGIN
        GuestLimit := limit;
    END SetGuestLimit;

(********************************************************************************)

PROCEDURE SetSameIPLimit (limit: CARDINAL);

    (* Sets the limit on simultaneous connections from the same IP address. *)

    BEGIN
        SameIPLimit := limit;
    END SetSameIPLimit;

(********************************************************************************)

PROCEDURE StartSession (CommandSocket: Socket;  UserNumber, IPaddr: CARDINAL;
                        KeepAlive: Semaphore;
                        VAR (*OUT*) UserHandle: ClientFileInfo): ClientDataPointer;

    (* Creates a new session state record.  During lengthy operations           *)
    (* we have to do a Signal(KeepAlive) every so often in order to stop the    *)
    (* session from timing out.  The returned UserHandle can be used as a user  *)
    (* identifier in calls to module FtpTransfers; it remains valid until       *)
    (* EndSession is called.                                                    *)

    VAR SP: SessionPointer;  p: ClientDataPointer;
        peer: SockAddr;  size: CARDINAL;

    BEGIN
        (* Create the FtpTransfer data structure. *)

        UserHandle := CreateSession (CommandSocket, UserNumber, KeepAlive);

        (* Create this module's data structure. *)

        NEW (p);
        WITH p^ DO
            CardinalToString (CommandSocket, SessionID, 6);
            FileInfo := UserHandle;
            SysClock.GetClock (StartTime);
            ClientIPAddress := IPaddr;
            IsGuest := FALSE;
        END (*WITH*);

        (* Add it to our list of active sessions. *)

        NEW (SP);
        WITH SP^ DO
            this := p;  next := NIL;
        END (*WITH*);
        Obtain (SessionListLock);
        IF SessionList.head = NIL THEN
            SessionList.head := SP;
        ELSE
            SessionList.tail^.next := SP;
        END (*IF*);
        SessionList.tail := SP;
        Release (SessionListLock);

        (* Set the initial default data port. *)

        size := SIZE(peer);
        IF getpeername (CommandSocket, peer, size) THEN
            IF NotDetached() THEN
                WriteError;
            END (*IF*);
        END (*IF*);
        SetPort (p^.FileInfo, peer.in_addr.addr, peer.in_addr.port);

        RETURN p;

    END StartSession;

(********************************************************************************)

PROCEDURE IdentifySession (CommandSocket: Socket): ClientDataPointer;

    (* Returns the session corresponding to this socket number. *)

    VAR ID: ARRAY [0..5] OF CHAR;  current: SessionPointer;

    BEGIN
        CardinalToString (CommandSocket, ID, 6);
        Obtain (SessionListLock);
        current := SessionList.head;
        WHILE (current <> NIL) AND NOT Strings.Equal (current^.this^.SessionID, ID) DO
            current := current^.next;
        END (*WHILE*);
        Release (SessionListLock);
        IF current = NIL THEN RETURN NIL
        ELSE RETURN current^.this;
        END (*IF*);
    END IdentifySession;

(********************************************************************************)

PROCEDURE ClientAddressAcceptable (IPAddress: CARDINAL): BOOLEAN;

    (* Returns TRUE if we are willing to accept connections from this address.  *)
    (* At this stage we have not yet decided whether to accept the new client,  *)
    (* so a client record for the new session does not yet exist.               *)

    VAR current: AddressListPointer;  acceptable: BOOLEAN;
        SP: SessionPointer;  count: CARDINAL;

    BEGIN
        Obtain (AddressCheckLock);

        (* Is this address on the AllowList? *)

        acceptable := FALSE;
        current := AllowList.head;
        LOOP
            IF (current = NIL) OR acceptable THEN EXIT(*LOOP*) END (*IF*);
            IF IAND (IPAddress, current^.mask) = current^.IPAddress THEN
                acceptable := TRUE;
            END (*IF*);
            current := current^.next;
        END (*LOOP*);

        (* If the address was on the AllowList, skip the DenyList check. *)

        IF NOT acceptable THEN
            acceptable := TRUE;
            current := DenyList.head;
            LOOP
                IF (current = NIL) OR NOT acceptable THEN EXIT(*LOOP*) END (*IF*);
                IF IAND (IPAddress, current^.mask) = current^.IPAddress THEN
                    acceptable := FALSE;
                END (*IF*);
                current := current^.next;
            END (*LOOP*);
        END (*IF*);
        Release (AddressCheckLock);

        (* If we've passed all checks so far, we still have to see whether      *)
        (* there are too many existing users with the same address.             *)

        IF acceptable THEN
            Obtain (SessionListLock);
            SP := SessionList.head;  count := 0;
            WHILE SP <> NIL DO
                IF SP^.this^.ClientIPAddress = IPAddress THEN
                    INC (count);
                END (*IF*);
                SP := SP^.next;
            END (*WHILE*);
            Release (SessionListLock);
            acceptable := count < SameIPLimit;
        END (*IF*);

        RETURN acceptable;

    END ClientAddressAcceptable;

(********************************************************************************)

PROCEDURE EndSession (VAR (*INOUT*) p: ClientDataPointer);

    (* Discards the session information. *)

    VAR previous, current: SessionPointer;

    BEGIN
        IF p <> NIL THEN
            RemoveUser (p);
            CloseSession (p^.FileInfo);
            Obtain (SessionListLock);
            previous := NIL;  current := SessionList.head;
            LOOP
                IF current = NIL THEN EXIT(*LOOP*) END(*IF*);
                IF current^.this = p THEN
                    IF previous = NIL THEN
                        SessionList.head := current^.next;
                    ELSE
                        previous^.next := current^.next;
                    END (*IF*);
                    IF current^.next = NIL THEN
                        SessionList.tail := previous;
                    END (*IF*);
                    DISPOSE (current);
                    EXIT (*LOOP*);
                END (*IF*);
                previous := current;  current := current^.next;
            END (*LOOP*);
            Release (SessionListLock);
            DISPOSE (p);
        END (*IF*);
    END EndSession;

(********************************************************************************)

PROCEDURE NewUser (client: ClientDataPointer;  Username: ARRAY OF CHAR): UserCategory;

    (* We assume that a session has already been established, but that there    *)
    (* is not currently a logged-in user.  This procedure produces an updated   *)
    (* session state that reflects the Username.                                *)

    VAR category: UserCategory;

    BEGIN
        FindUser (Username, client^.FileInfo, category);
        IF category = GuestUser THEN
            Obtain (GuestCountLock);
            IF GuestCount < GuestLimit THEN
                INC (GuestCount);
                client^.IsGuest := TRUE;
            ELSE
                category := OverflowUser;
            END (*IF*);
            Release (GuestCountLock);
        END (*IF*);
        RETURN category;
    END NewUser;

(********************************************************************************)

PROCEDURE AbortDataOperations (client: ClientDataPointer);

    (* Aborts the data transfer, if any, now in progress for this session. *)

    BEGIN
        IF client <> NIL THEN
            KillDataChannel (client^.FileInfo);
        END (*IF*);
    END AbortDataOperations;

(********************************************************************************)

PROCEDURE KillUser (ID: ARRAY OF CHAR);

    (* Attempts to kill the session whose ID is given.  This involves aborting the      *)
    (* data operations, and then cancelling operations on the session's command         *)
    (* socket.  The cancel will cause a higher-level module to detect that the          *)
    (* session is no longer active, and then it is up to that higher-level module       *)
    (* to clean up neatly.  Because we count on higher-level software to do the         *)
    (* operations that will lead to an EndSession call, the present procedure does      *)
    (* not actually remove the session from the list of active sessions.                *)

    VAR client: ClientDataPointer;  CommandSocket: Socket;

    BEGIN
        CommandSocket := StringToCardinal (ID);
        client := IdentifySession (CommandSocket);
        IF client <> NIL THEN
            KillDataChannel (client^.FileInfo);
            so_cancel (CommandSocket);
        END (*IF*);
    END KillUser;

(****************************************************************************************)

PROCEDURE RemoveUser (client: ClientDataPointer);

    (* Logs out the user, but does not terminate the session.  *)

    BEGIN
        IF client^.IsGuest THEN
            Obtain (GuestCountLock);
            DEC (GuestCount);
            client^.IsGuest := FALSE;
            Release (GuestCountLock);
        END (*IF*);
        CloseUser (client^.FileInfo);
    END RemoveUser;

(****************************************************************************************)

PROCEDURE DateTimeToString (Time: SysClock.DateTime;  VAR (*OUT*) buffer: ARRAY OF CHAR);

    VAR j: CARDINAL;

    BEGIN
        j := 0;
        ConvertCardZ (Time.year, buffer, 4, j);
        buffer[j] := '-';  INC(j);
        ConvertCardZ (Time.month, buffer, 2, j);
        buffer[j] := '-';  INC(j);
        ConvertCardZ (Time.day, buffer, 2, j);
        buffer[j] := ' ';  INC(j);
        ConvertCardZ (Time.hour, buffer, 2, j);
        buffer[j] := ':';  INC(j);
        ConvertCardZ (Time.minute, buffer, 2, j);
        buffer[j] := ':';  INC(j);
        ConvertCardZ (Time.second, buffer, 2, j);
        buffer[j] := CHR(0);
    END DateTimeToString;

(********************************************************************************)

PROCEDURE ListAllUsers (S: Socket);

    (* Sends a user listing to socket S. *)

    VAR p: SessionPointer;  CDP: ClientDataPointer;
        j: CARDINAL;
        success: BOOLEAN;
        buffer: ARRAY [0..31] OF CHAR;

    BEGIN
        Obtain (SessionListLock);
        p := SessionList.head;
        WHILE p <> NIL DO
            CDP := p^.this;
            success := SendString (S, " ") AND SendString (S, CDP^.SessionID);
            IPToString (CDP^.ClientIPAddress, buffer);
            j := LENGTH(buffer);
            WHILE j < 18 DO
                buffer[j] := ' ';  INC(j);
            END (*WHILE*);
            buffer[j] := CHR(0);
            success := success AND SendString (S, "  ") AND SendString (S, buffer);
            DateTimeToString (CDP^.StartTime, buffer);
            success := success AND SendString (S, buffer);
            GetUserName (CDP^.FileInfo, buffer);
            EVAL (success AND SendString (S, "  ")
                             AND SendString (S, buffer) AND SendEOL (S));
            p := p^.next;
        END (*WHILE*);
        Release (SessionListLock);
    END ListAllUsers;

(********************************************************************************)

PROCEDURE SessionIDof (p: ClientDataPointer;  VAR (*OUT*) result: ARRAY OF CHAR);

    (* Returns the session ID in result. *)

    BEGIN
        Strings.Assign (p^.SessionID, result);
    END SessionIDof;

(************************************************************************)
(*                          INITIALISATION                              *)
(************************************************************************)

PROCEDURE LoadIPAddressList (hini: OS2.HINI;  VAR (*OUT*) List: IPAddressList;
                                                  option: ARRAY OF CHAR);

    (* Constructs an IP address list from the INI file data. *)

    VAR j, size, remaining: CARDINAL;
        bufptr: POINTER TO ARRAY [0..1023] OF CARDINAL;
        current: AddressListPointer;

    BEGIN
        List.head := NIL;  List.tail := NIL;
        IF (hini <> OS2.NULLHANDLE)
                 AND (OS2.PrfQueryProfileSize (hini, "$SYS", option, size)) THEN
            IF size <> 0 THEN
                ALLOCATE (bufptr, size);
                OS2.PrfQueryProfileData (hini, "$SYS", option, bufptr, size);
                j := 0;  current := NIL;  remaining := size DIV (2*SIZE(CARDINAL));
                WHILE remaining > 0 DO
                    IF current = NIL THEN
                        NEW (current);
                        List.head := current;
                    ELSE
                        NEW (current^.next);
                        current := current^.next;
                    END (*IF*);
                    List.tail := current;
                    current^.next := NIL;
                    current^.IPAddress := bufptr^[j];  INC(j);
                    current^.mask := bufptr^[j];  INC(j);
                    DEC (remaining);
                END (*WHILE*);
                DEALLOCATE (bufptr, size);
            END (*IF*);
        END (*IF*);
    END LoadIPAddressList;

(************************************************************************)

PROCEDURE LoadINIData;

    (* Reads some parameters from ftpd.ini.  *)

    VAR hini: OS2.HINI;

    BEGIN
        SameIPLimit := MAX(CARDINAL);
        hini := OpenINIFile ("ftpd.ini");
        LoadIPAddressList (hini, AllowList, "IPAllow");
        LoadIPAddressList (hini, DenyList, "IPDeny");
        IF hini <> OS2.NULLHANDLE THEN
            IF NOT INIGet (hini, "$SYS", "SameIPLimit", SameIPLimit) THEN
                SameIPLimit := MAX(CARDINAL);
            END (*IF*);
            OS2.PrfCloseProfile (hini);
        END (*IF*);
    END LoadINIData;

(************************************************************************)

BEGIN
    WITH SessionList DO
        head := NIL;  tail := NIL;
    END (*WITH*);
    LoadINIData;
    CreateLock (SessionListLock);
    GuestCount := 0;  GuestLimit := 0;
    CreateLock (GuestCountLock);
    CreateLock (AddressCheckLock);
END LoggedOnUsers.

