MODULE Ftpd;

        (********************************************************)
        (*                                                      *)
        (*               Simple FTP server                      *)
        (*                                                      *)
        (*  Programmer:         P. Moylan                       *)
        (*  Started:            19 August 1997                  *)
        (*  Last edited:        11 August 1998                  *)
        (*  Status:             Working                         *)
        (*                                                      *)
        (********************************************************)

FROM SYSTEM IMPORT LOC, CARD8, CARD16, CAST;

IMPORT TextIO, OS2;

FROM FtpdSession IMPORT
    (* proc *)  SetVersion, SetTransactionLogging, SetMaxUsers, SetTimeout,
                NewSession, NumberOfUsers;

FROM LoggedOnUsers IMPORT
    (* proc *)  SetGuestLimit, SetSameIPLimit;

FROM FtpTransfers IMPORT
    (* proc *)  SetLogLevel, SetFreeSpaceThreshold;

FROM Internet IMPORT
    (* const*)  Zero8, INADDR_ANY,
    (* proc *)  inet_addr;

FROM Sockets IMPORT
    (* const*)  NotASocket,
    (* type *)  Socket, AddressFamily, SocketType, SockAddr,
    (* proc *)  socket, bind, sock_errno, psock_errno, soclose,
                accept, connect, listen, gethostid,
                sock_init, so_cancel, setsockopt;

FROM InetUtilities IMPORT
    (* proc *)  Swap2, Swap4, DumpRecord, WriteError, IPToString,
                ConvertDecimal, OpenINIFile, INIGet, WaitForSocket,
                AddToTransactionLog;

FROM GlassTTY IMPORT
    (* proc *)  WriteString, WriteLn, WriteChar, WriteInt, WriteCard,
                SetCursor;

FROM Keyboard IMPORT
    (* proc *)  NotDetached, InKey, StuffKeyboardBuffer;

FROM IOChan IMPORT
    (* type *)  ChanId;

FROM ProgramArgs IMPORT
    (* proc *)  ArgChan, IsArgPresent;

FROM LowLevel IMPORT
    (* proc *)  EVAL;

FROM Timer IMPORT
    (* proc *)  Sleep;

FROM TaskControl IMPORT
    (* proc *)  CreateTask;

FROM CtrlC IMPORT
    (* type *)  BreakHandler,
    (* proc *)  SetBreakHandler;

(********************************************************************************)

CONST version = "0.70";
    DefaultPort = 21;
    DefaultMaxUsers = 10;
    DefaultTimeout = 900;               (* seconds   *)
    DefaultLogLevel = 1;
    DefaultFreeSpaceThreshold = 10;     (* megabytes *)
    DefaultGuestLimit = DefaultMaxUsers-1;

VAR MainSocket: Socket;
    CalledFromInetd: BOOLEAN;
    RapidShutdown: BOOLEAN;
    ScreenEnabled: BOOLEAN;
    ServerPort: CARDINAL;

(********************************************************************************)

PROCEDURE ShutdownChecker;

    (* A separate task that looks for a Q command from the keyboard.  *)

    VAR ch: CHAR;  StillRunning: BOOLEAN;

    BEGIN
        StillRunning := TRUE;
        LOOP
            ch := InKey();
            IF CAP(ch) = 'Q' THEN
                RapidShutdown := TRUE;  ch := 'G';
            END (*IF*);
            IF CAP(ch) = 'G' THEN
                IF StillRunning THEN
                    so_cancel (MainSocket);
                    StillRunning := FALSE;
                END (*IF*);
                IF RapidShutdown THEN
                    EXIT (*LOOP*);
                END (*IF*);
            END(*IF*);
        END (*LOOP*);

    END ShutdownChecker;

(********************************************************************************)

PROCEDURE ["C"] ControlCHandler(): BOOLEAN;

    (* Intercepts a Ctrl/C from the keyboard. *)

    BEGIN
        RapidShutdown := TRUE;
        StuffKeyboardBuffer ('Q');
        RETURN TRUE;
    END ControlCHandler;

(********************************************************************************)

PROCEDURE GetParameters;

    (* Picks up program arguments from the command line. *)

    TYPE CharNumber = [0..79];

    VAR j: CARDINAL;
        args: ChanId;
        Options: ARRAY CharNumber OF CHAR;

    (****************************************************************************)

    PROCEDURE SkipBlanks;

        BEGIN
            LOOP
                IF Options[j] <> ' ' THEN EXIT(*LOOP*) END(*IF*);
                IF j = MAX(CharNumber) THEN
                    Options[j] := CHR(0);  EXIT (*LOOP*);
                ELSE
                    INC (j);
                END (*IF*);
            END (*LOOP*);
        END SkipBlanks;

    (****************************************************************************)

    PROCEDURE GetNumber(): CARDINAL;

        BEGIN
            SkipBlanks;
            RETURN ConvertDecimal (Options, j);
        END GetNumber;

    (****************************************************************************)

    VAR level: CARDINAL;

    BEGIN
        args := ArgChan();
        IF IsArgPresent() THEN
            TextIO.ReadString (args, Options);
            j := 0;  SkipBlanks;
            LOOP
                CASE CAP(Options[j]) OF
                    CHR(0):   EXIT (*LOOP*);
                  | 'D':      INC (j);  level := GetNumber();
                                        IF NOT ScreenEnabled THEN
                                            level := level MOD 2;
                                        END (*IF*);
                                        SetTransactionLogging (level);
                  | 'F':      INC (j);  SetFreeSpaceThreshold (1024*GetNumber());
                  | 'G':      INC (j);  SetGuestLimit (GetNumber());
                  | 'L':      INC (j);  SetLogLevel (GetNumber());
                  | 'M':      INC (j);  SetMaxUsers (GetNumber());
                  | 'P':      INC (j);  ServerPort := GetNumber();
                  | 'T':      INC (j);  SetTimeout (GetNumber());
                  | '0'..'9': MainSocket := GetNumber();
                              CalledFromInetd := TRUE;
                ELSE
                    IF ScreenEnabled THEN
                        WriteString ("Unknown option ");
                        WriteChar (Options[j]);  WriteLn;
                    END (*IF*);
                    INC(j);
                END (*CASE*);
                SkipBlanks;
            END (*LOOP*);
        END (*IF*);
    END GetParameters;

(************************************************************************)

PROCEDURE LoadINIData;

    (* Loads setup parameters from "ftpd.ini". *)

    VAR hini: OS2.HINI;

    PROCEDURE GetItem (name: ARRAY OF CHAR;
                            VAR (*OUT*) variable: ARRAY OF LOC): BOOLEAN;

        BEGIN
            RETURN INIGet (hini, "$SYS", name, variable);
        END GetItem;

    (********************************************************************)

    VAR MaxUsers, FreeSpaceThreshold, UserLogging,
           TimeoutLimit, GuestLimit, SameIPLimit, TransLevel: CARDINAL;

    BEGIN
        ServerPort := DefaultPort;
        MaxUsers := DefaultMaxUsers;
        TimeoutLimit := DefaultTimeout;
        UserLogging := DefaultLogLevel;
        TransLevel := 0;
        FreeSpaceThreshold := DefaultFreeSpaceThreshold;
        GuestLimit := MaxUsers - 1;
        SameIPLimit := MAX(CARDINAL);

        hini := OpenINIFile ("ftpd.ini");
        IF hini <> OS2.NULLHANDLE THEN
            EVAL(GetItem ("ServerPort", ServerPort));
            EVAL(GetItem ("MaxUsers", MaxUsers));  GuestLimit := MaxUsers - 1;
            EVAL(GetItem ("TimeOut", TimeoutLimit));
            EVAL(GetItem ("TransLevel", TransLevel));
            EVAL(GetItem ("LogLevel", UserLogging));
            EVAL(GetItem ("SpaceThreshold", FreeSpaceThreshold));
            EVAL(GetItem ("GuestLimit", GuestLimit));
            EVAL(GetItem ("SameIPLimit", SameIPLimit));
            OS2.PrfCloseProfile (hini);
        END (*IF*);

        SetMaxUsers (MaxUsers);
        SetTimeout (TimeoutLimit);
        IF NOT ScreenEnabled THEN
            TransLevel := TransLevel MOD 2;
        END (*IF*);
        SetTransactionLogging (TransLevel);
        SetLogLevel (UserLogging);
        SetFreeSpaceThreshold (1024*FreeSpaceThreshold);
        SetGuestLimit (GuestLimit);
        SetSameIPLimit (SameIPLimit);

    END LoadINIData;

(********************************************************************************)

PROCEDURE WriteHostID (ID: ARRAY OF LOC);

    VAR result: ARRAY [0..16] OF CHAR;

    BEGIN
        IPToString (ID, result);
        WriteString (result);
    END WriteHostID;

(********************************************************************************)

PROCEDURE RunTheServer;

    (*  OPERATING AS A SERVER                                                       *)
    (*     1. (Compulsory) Call "bind" to bind the socket with a local address.     *)
    (*        You can usually afford to specify INADDR_ANY as the machine           *)
    (*        address, but you'd normally bind to a specific port number.           *)
    (*     2. Call "listen" to indicate your willingness to accept connections.     *)
    (*     3. Call "accept", getting a new socket (say ns) from the client.         *)
    (*     4. Use procedures "send" and "recv" to transfer data, using socket ns.   *)
    (*        (Meanwhile, your original socket remains available to accept          *)
    (*        more connections, so you can continue with more "accept" operations   *)
    (*        in parallel with these data operations.  If so, you should of course  *)
    (*        be prepared to run multiple threads.)                                 *)
    (*     5. Use "soclose(ns)" to terminate the session with that particular       *)
    (*        client.                                                               *)
    (*     6. Use "soclose" on your original socket to clean up at the end.         *)

    VAR ns: Socket;  myaddr, client: SockAddr;
        temp: CARDINAL;

    BEGIN
        RapidShutdown := FALSE;
        IF sock_init() <> 0 THEN
            IF ScreenEnabled THEN
                WriteString ("No network.");
            END (*IF*);
            RETURN;
        END (*IF*);
        CalledFromInetd := FALSE;
        GetParameters;
        SetVersion (version);

        IF CalledFromInetd THEN

            IF ScreenEnabled THEN
                WriteString ("FtpServer started from inetd, socket ");
                WriteInt (MainSocket);  WriteLn;
            END (*IF*);
            AddToTransactionLog ("Server started.");
            NewSession (MainSocket, client);

        ELSE

            MainSocket := socket (AF_INET, SOCK_STREAM, AF_UNSPEC);

            (* Allow reuse of the port we're binding to. *)

            temp := 1;
            setsockopt (MainSocket, 0FFFFH, 4, temp, SIZE(CARDINAL));

            IF ScreenEnabled THEN
                WriteString ("FtpServer v");  WriteString (version);  WriteLn;
                WriteString ("Copyright (C) 1997-98 Peter Moylan.");
                WriteLn;  WriteHostID (Swap4(gethostid()));  WriteString (" port ");
                WriteCard (ServerPort);
                WriteString (", socket ");
                WriteInt(MainSocket);  WriteString ("     ");  WriteLn;
                EVAL (SetBreakHandler(ControlCHandler));
            END (*IF*);
            AddToTransactionLog ("Server started.");
            CreateTask (ShutdownChecker, 1, "ftp kbd");

            (* Now have the socket, bind to our machine. *)

            WITH myaddr DO
                family := AF_INET;
                WITH in_addr DO
                    port := Swap2 (ServerPort);
                    (* Bind to all interfaces. *)
                    addr := INADDR_ANY;
                    zero := Zero8;
                END (*WITH*);
            END (*WITH*);

            IF bind (MainSocket, myaddr, SIZE(myaddr)) THEN
                IF ScreenEnabled THEN
                    WriteError;
                    WriteString ("Cannot bind to server port, exiting.");
                    WriteLn;
                END (*IF*);

            ELSE

                (* Go into listening mode. *)

                IF listen (MainSocket, 1) THEN
                    IF ScreenEnabled THEN
                        WriteError;
                    END (*IF*);
                END (*IF*);

                WHILE WaitForSocket (MainSocket, MAX(CARDINAL)) > 0 DO
                    temp := SIZE(client);
                    ns := accept (MainSocket, client, temp);
                    IF ns <> NotASocket THEN

                        (* Allow the reception of out-of-band data. *)

                        temp := 1;
                        IF setsockopt (ns, 0FFFFH, 0100H, temp, SIZE(CARDINAL)) THEN
                        END (*IF*);

                        (* Start the client session. *)

                        NewSession (ns, client);
                    END (*IF*);
                END (*WHILE*);

            END (*IF*);

            IF soclose(MainSocket) THEN
                psock_errno ("");
            END (*IF*);

        END (*IF*);

        (* End of operation, shut down the server. *)

        IF NOT RapidShutdown THEN
            IF CalledFromInetd THEN
                Sleep (3000);
            ELSIF NumberOfUsers() > 0 THEN
                AddToTransactionLog ("Waiting for existing users to finish");
            END (*IF*);
            WHILE (NumberOfUsers() > 0) AND NOT RapidShutdown DO
                Sleep (1000);
            END (*WHILE*);
        END (*IF*);

        AddToTransactionLog ("FtpServer closing down");

    END RunTheServer;

(********************************************************************************)
(*                                 MAIN PROGRAM                                 *)
(********************************************************************************)

BEGIN
    ScreenEnabled := NotDetached();
    LoadINIData;
    RunTheServer;
END Ftpd.

