IMPLEMENTATION MODULE FDUsers;

        (********************************************************)
        (*                                                      *)
        (*  FTP server: module to deal with user access rights  *)
        (*                                                      *)
        (*  Programmer:         P. Moylan                       *)
        (*  Started:            30 August 1997                  *)
        (*  Last edited:        9 August 1998                   *)
        (*  Status:             Working                         *)
        (*                                                      *)
        (********************************************************)

(************************************************************************)
(*                                                                      *)
(*     Syntax for the permissions file:                                 *)
(*                                                                      *)
(*     <result>  ->  <userclass> <password> { <volumeinfo> }*           *)
(*     <volumeinfo> ->  <homedir> { <dirrule> }  { ; }                  *)
(*     <userclass> -> G | U | N | M                                     *)
(*     <password>  -> <namestring>                                      *)
(*     <homedir>    ->  <namestring>  |  <namestring> = <namestring>    *)
(*     <diritem>   -> <namestring> <dirrule>                            *)
(*     <dirrule>   -> { <permission> }* { ( <dirlist> ) }               *)
(*     <dirlist>   ->  <diritem> { , <diritem> }*                       *)
(*     <dirlist>   -> { <diritem> }+                                    *)
(*     <permission> ->  V+ | V- | R+ | R- | W- | W+ | D- | D+           *)
(*     <namestring> ->  <string1> | " <string2> " | ' <string3> '       *)
(*     <string1>   -> any string not including space char               *)
(*     <string2>   -> any string not including double quote             *)
(*     <string3>   -> any string not including single quote             *)
(*                                                                      *)
(*  Notation: {} means optional, {}* means zero or more, {}+ means      *)
(*            one or more.                                              *)
(*                                                                      *)
(************************************************************************)

FROM SYSTEM IMPORT ADR, CARD16;

IMPORT IOChan, ChanConsts, RndFile, FileSys, Strings, SysClock, OS2;

FROM LowLevel IMPORT
    (* proc *)  EVAL;

FROM TaskControl IMPORT
    (* type *)  Lock,
    (* proc *)  CreateLock, Obtain, Release;

FROM FDFiles IMPORT
    (* type *)  FileAttr, FileAttribute, DirectoryEntry,
    (* proc *)  FirstDirEntry, NextDirEntry, DirSearchDone, FreeSpace;

FROM Volume IMPORT
    (* proc *)  CreateDir, RmvDir;

FROM Storage IMPORT
    (* proc *)  ALLOCATE, DEALLOCATE;

FROM InetUtilities IMPORT
    (* proc *)  SendChar, SendString, SendEOL, SendCard, SendZCard,
                OpenINIFile, INIGet;

FROM Sockets IMPORT
    (* type *)  Socket,
    (* proc *)  send;

(********************************************************************************)

CONST Nul = CHR(0);

TYPE
    CharSet = SET OF CHAR;
    NameString = ARRAY [0..31] OF CHAR;
    FileNameString = ARRAY [0..255] OF CHAR;

    (* The following declarations relate to the data structure that we keep,    *)
    (* for a logged-in user, to show which directories and files are accessible.*)
    (* For each directory, we have a linked list of subdirectories.             *)

    Permission = (Visible, AllowRead, AllowWrite, AllowDelete);
    PermissionSet = SET OF Permission;

    DirEntryPtr = POINTER TO DirEntry;

    DirEntry = RECORD
                   flags: PermissionSet;
                   parent, FirstChild, next: DirEntryPtr;
                   BeenHereBefore: BOOLEAN;
                   name: FileNameString;
               END (*RECORD*);

    (* The fields in a VolumePermission record have the following meanings.     *)
    (*      VolumeName   The name of the device or partition                    *)
    (*      next         Pointer to the next VolumePermission for this user.    *)
    (*      Home         Home directory, an absolute path.  This always         *)
    (*                                   ends with a '/'.                       *)
    (*      BaseEntry    Pointer to tree of directory permissions               *)
    (*      SuperDir     Pointer to the "superdirectory" volume for this user,  *)
    (*                     or NIL if this user doesn't have a superdirectory.   *)
    (*                                                                          *)
    (* The superdirectory is distinguished by the property V^.SuperDir = V      *)

    Volume = POINTER TO VolumePermission;

    VolumePermission = RECORD
                           VolumeName: FileNameString;
                           next: Volume;
                           Home: FileNameString;
                           BaseEntry: DirEntryPtr;
                           SuperDir: Volume;
                       END (*RECORD*);

    (* An FName structure is our internal representation of a file name.        *)
    (*     dir          Directory string, a complete path name.  Always ends    *)
    (*                    with a '/', unless it's an empty string.              *)
    (*     fname        file name within the directory.  This is an empty       *)
    (*                    string if the file in question is a directory.        *)
    (*     vol          volume on which this file lives                         *)
    (*     EntryPtr     points to the DirEntry record for the "dir" field, or   *)
    (*                    NIL if we've detected a malformed file name.          *)

    FName = POINTER TO
                RECORD
                    dir, fname: FileNameString;
                    vol: Volume;
                    EntryPtr: DirEntryPtr;
                END (*RECORD*);

    (* The fields in a UserPermission record have the following meanings.       *)
    (*      Name         The user's login name                                  *)
    (*      Password     The user's password                                    *)
    (*      MultiVol     TRUE iff user can access more than one volume.         *)
    (*      FirstVol     Points to a linked list of Volume entries.             *)
    (*      UserNumber   A serial number to use in welcome messages.            *)
    (*      UserLimit    Maximum number of instances of this user.  Storing     *)
    (*                   a copy of this for each instance might seem to be      *)
    (*                   redundant; but this allows us to cover the possibility *)
    (*                   that the limit has changed between instances.          *)
    (*      CurrentVol   The user's current volume.                             *)
    (*      CurrentDir   Current directory, an absolute path.  This always      *)
    (*                      ends with a '/', except in the special case where   *)
    (*                      it's a null string.                                 *)
    (*      PosInTree    Pointer to current position within tree                *)

    User = POINTER TO UserPermission;

    UserPermission = RECORD
                         Name, Password: NameString;
                         MultiVol: BOOLEAN;
                         FirstVol, CurrentVol: Volume;
                         PosInTree: DirEntryPtr;
                         CurrentDir: FileNameString;
                         UserNumber, UserLimit: CARDINAL;
                     END (*RECORD*);

    (* A structure for maintaining a linked list of user counts. *)

    CountList = POINTER TO RECORD
                               next: CountList;
                               count: CARDINAL;
                               name: NameString;
                           END (*RECORD*);

(********************************************************************************)

VAR
    (* A linked list showing, for each username, how many users are currently   *)
    (* logged in under that name.                                               *)

    UserCount: CountList;

    (* Critical section protection for this list. *)

    UserCountLock: Lock;

(********************************************************************************)
(*                         CHECK FOR TOO MANY USERS                             *)
(********************************************************************************)

PROCEDURE AllocateUserNumber (username: NameString;  limit: CARDINAL): CARDINAL;

    (* Works out which instance we are of this username.  If the result goes    *)
    (* over the limit, the returned result is 0.                                *)

    VAR current, newelement: CountList;
        result: CARDINAL;

    BEGIN
        Obtain (UserCountLock);

        (* Find the list entry for this username. *)

        current := UserCount;
        WHILE (current <> NIL) AND NOT Strings.Equal(current^.name, username) DO
            current := current^.next;
        END (*WHILE*);

        (* If this username is not already in the list, add it. *)

        IF current = NIL THEN
            NEW (newelement);
            WITH newelement^ DO
                next := UserCount;
                count := 0;
                name := username;
            END (*WITH*);
            UserCount := newelement;
            current := newelement;
        END (*IF*);

        (* Increment the count, unless it's already at the limit. *)

        IF current^.count < limit THEN
            INC (current^.count);
            result := current^.count;
        ELSE
            result := 0;
        END (*IF*);

        Release (UserCountLock);
        RETURN result;

    END AllocateUserNumber;

(********************************************************************************)
(*                                PARSER                                        *)
(********************************************************************************)

PROCEDURE ReadUserData (name: ARRAY OF CHAR;
                          VAR (*OUT*) category: UserCategory): User;

    (* Fetches the password, etc., for the user whose username is specified     *)
    (* as the argument.  Returns with category = NoSuchUser if the user's data  *)
    (* could not be found.  The result is NIL in this case, and also in the     *)
    (* case of an overflow user.                                                *)

    CONST Space = " ";

    VAR NextChar: CHAR;
        hini: OS2.HINI;
        bufptr: POINTER TO ARRAY [0..MAX(CARDINAL) DIV 4] OF CHAR;
        BufferPosition, BufferSize: CARDINAL;

    (****************************************************************************)

    PROCEDURE Scan;

        (* Puts the next input character into variable NextChar. *)

        BEGIN
            IF BufferPosition >= BufferSize THEN
                NextChar := Nul;
            ELSE
                NextChar := bufptr^[BufferPosition];
                INC (BufferPosition);
            END (*IF*);
        END Scan;

    (****************************************************************************)

    PROCEDURE LoadString (VAR (*OUT*) result: ARRAY OF CHAR;  Stoppers: CharSet);

        (* Reads a string up to but not including a character in Stoppers.      *)

        VAR j: CARDINAL;

        BEGIN
            j := 0;
            WHILE (j <= HIGH(result)) AND NOT (NextChar IN Stoppers) DO
                result[j] := NextChar;  INC(j);
                Scan;
            END (*WHILE*);
            IF j <= HIGH(result) THEN result[j] := Nul END(*IF*);
        END LoadString;

    (****************************************************************************)

    PROCEDURE GetNameString (VAR (*OUT*) result: ARRAY OF CHAR;  Stoppers: CharSet);

        (*     <namestring> ->  <string1> | " <string2> " | ' <string3> '       *)
        (*     <string1>   -> any string not including space char               *)
        (*     <string2>   -> any string not including double quote             *)
        (*     <string3>   -> any string not including single quote             *)
        (* The strings in <string1> also may not contain characters in Stoppers.*)

        CONST SingleQuote = "'";  DoubleQuote = '"';

        VAR Delimiter: CHAR;

        BEGIN
            INCL (Stoppers, Nul);
            IF (NextChar = SingleQuote) OR (NextChar = DoubleQuote) THEN
                Delimiter := NextChar;
                Stoppers := CharSet {Nul, Delimiter};
                Scan;
                LoadString (result, Stoppers);
                IF NextChar = Delimiter THEN
                    Scan;
                END (*IF*);
            ELSE
                INCL (Stoppers, Space);
                LoadString (result, Stoppers);
            END (*IF*);
        END GetNameString;

    (****************************************************************************)

    PROCEDURE DirItem (mother: DirEntryPtr): DirEntryPtr;  FORWARD;

    (****************************************************************************)

    PROCEDURE DirList (mother: DirEntryPtr): DirEntryPtr;

        (*     <dirlist>   ->  <diritem> { , <diritem> }*       *)
        (* Result returned: a linked list of directory nodes.   *)

        CONST Comma = ",";

        VAR result, lastnode: DirEntryPtr;

        BEGIN
            result := DirItem (mother);  lastnode := result;
            WHILE NextChar = Comma DO
                Scan;
                lastnode^.next := DirItem (mother);
                lastnode := lastnode^.next;
            END (*WHILE*);
            RETURN result;
        END DirList;

    (****************************************************************************)

    PROCEDURE DirRule (pnode: DirEntryPtr);

        (* Fills in the permissions and subdirectory info for pnode.  *)
        (*     <dirrule>   -> { <permission> }* { ( <dirlist> ) }     *)
        (*     <permission> ->  V+ | V- | R+ | R- | W- | W+ | D- | D+ *)

        VAR option: Permission;

        BEGIN
            (* Default flags are inherited from parent. *)

            IF pnode^.parent = NIL THEN
                pnode^.flags := PermissionSet {Visible, AllowRead}
            ELSE
                pnode^.flags := pnode^.parent^.flags
            END (*IF*);

            (* Look for optional permission codes. *)

            WHILE CAP(NextChar) IN CharSet {'V', 'R', 'W', 'D'} DO
                CASE CAP(NextChar) OF
                  | 'V':  option := Visible;
                  | 'R':  option := AllowRead;
                  | 'W':  option := AllowWrite;
                  | 'D':  option := AllowDelete;
                END (*CASE*);
                Scan;
                IF NextChar = '-' THEN
                    Scan;
                    EXCL (pnode^.flags, option);
                ELSIF NextChar = '+' THEN
                    Scan;
                    INCL (pnode^.flags, option);
                END (*IF*);

            END (*WHILE*);

            (* Look for optional list of subdirectories. *)

            IF NextChar = '(' THEN
                Scan;
                IF NextChar <> ')' THEN
                    pnode^.FirstChild := DirList(pnode);
                END (*IF*);
                IF NextChar = ')' THEN
                    Scan;
                END (*IF*);
            END (*IF*);

        END DirRule;

    (****************************************************************************)

    PROCEDURE DirItem (mother: DirEntryPtr): DirEntryPtr;

        (*     <diritem>   -> <namestring> <dirrule>                            *)

        VAR result: DirEntryPtr;

        BEGIN
            NEW (result);
            WITH result^ DO
                flags := PermissionSet {Visible, AllowRead};
                parent := mother;  FirstChild := NIL;  next := NIL;
                BeenHereBefore := FALSE;
                GetNameString (name, CharSet{});
            END (*WITH*);
            DirRule (result);
            RETURN result;
        END DirItem;

    (****************************************************************************)

    PROCEDURE VolumeInfo (U: User;  previous: Volume): Volume;

        (* Creates a new volume record, and tack it onto the tail of the list.  *)

        (*     <volumeinfo> ->  <homedir>  { <dirrule> }  { ; }                 *)
        (*     <homedir>    ->  <namestring>  |  <namestring> = <namestring>    *)

        VAR V: Volume;

        BEGIN
            NEW (V);
            WITH V^ DO
                VolumeName[0] := CHR(0);
                next := NIL;
                Home := "";
                BaseEntry := NIL;
                SuperDir := NIL;
            END (*WITH*);
            IF previous = NIL THEN
                U^.FirstVol := V;
            ELSE
                previous^.next := V;
            END (*IF*);

            (* Read the home directory. *)

            GetNameString (V^.Home, CharSet{'='});
            IF NextChar = '=' THEN
                V^.VolumeName := V^.Home;  Scan;
                GetNameString (V^.Home, CharSet{});
            ELSIF V^.Home[1] = ':' THEN
                V^.VolumeName[0] := CAP(V^.Home[0]);
                V^.VolumeName[1] := Nul;
            END (*IF*);

            (* Create a descriptor record for the user's root directory.  *)

            NEW (V^.BaseEntry);
            WITH V^.BaseEntry^ DO
                flags := PermissionSet {Visible, AllowRead};
                parent := NIL;  FirstChild := NIL;  next := NIL;
                name := "/";
                BeenHereBefore := FALSE;
            END (*WITH*);

            (* If there's anything left, it must be the optional directory rule.    *)

            IF NextChar <> Nul THEN
                DirRule (V^.BaseEntry);
            END (*IF*);

            (* A trailing semicolon is optional in this version. *)

            IF NextChar = ';' THEN Scan END(*IF*);

            RETURN V;

        END VolumeInfo;

    (****************************************************************************)

    PROCEDURE AddSuperDirectory (U: User);

        (* Adds an artifical volume that sits above all volumes, and contains   *)
        (* one dummy directory.                                                 *)

        VAR V, V1: Volume;

        BEGIN
            NEW (V);
            WITH V^ DO
                VolumeName := "/";  next := U^.FirstVol;
                Home := "";
                NEW (BaseEntry);
                WITH BaseEntry^ DO
                    flags := PermissionSet {Visible, AllowRead};
                    parent := NIL;  FirstChild := NIL;  next := NIL;
                    BeenHereBefore := TRUE;
                    name := "";
                END (*WITH*);
            END (*WITH*);
            U^.FirstVol := V;
            V1 := V;
            WHILE V1 <> NIL DO
                V1^.SuperDir := V;  V1 := V1^.next;
            END (*WHILE*);
        END AddSuperDirectory;

    (************************************************************************)

    PROCEDURE UserInINIFile (username: ARRAY OF CHAR): BOOLEAN;

        (* If username is in INI file, initialises hini and returns TRUE. *)

        VAR size: CARDINAL;  result: BOOLEAN;

        BEGIN
            hini := OpenINIFile ("ftpd.ini");
            IF hini = OS2.NULLHANDLE THEN
                result := FALSE;
            ELSE
                result := OS2.PrfQueryProfileSize (hini, username, NIL, size)
                                       AND (size <> 0);
            END (*IF*);
            RETURN result;
        END UserInINIFile;

    (************************************************************************)
    (*                     BODY OF READUSERDATA                             *)
    (*                                                                      *)
    (*     <result>  ->  <userclass> <password> { <volumeinfo> }*           *)
    (*     <userclass> -> G | U | N | M                                     *)
    (*     <password>  -> <namestring>                                      *)
    (*                                                                      *)
    (* Note: for simplicity, a password must always be specified, even if   *)
    (* it is not going to be used.                                          *)
    (*                                                                      *)
    (************************************************************************)

    VAR result: User;
        previousV, DefaultVolume: Volume;

    BEGIN       (* Body of ReadUserData *)


        NEW (result);  result^.FirstVol := NIL;
        Strings.Assign (name, result^.Name);

        IF UserInINIFile (result^.Name)
                AND INIGet (hini, result^.Name, "Category", category) THEN

            IF category = GuestUser THEN
                result^.Password := '@';
            ELSE
                EVAL (INIGet (hini, result^.Name, "Password", result^.Password));
            END (*IF*);

            IF NOT INIGet (hini, result^.Name, "UserLimit", result^.UserLimit) THEN
                result^.UserLimit := MAX(CARDINAL);
            END (*IF*);

            (* Check for too many users. *)

            result^.UserNumber := AllocateUserNumber (result^.Name, result^.UserLimit);
            IF result^.UserNumber = 0 THEN
                DISPOSE (result);
                category := OverflowUser;
            END (*IF*);

        ELSE

            DISPOSE (result);
            category := NoSuchUser;

        END (*IF*);

        (* If the volume information exists, allocate a buffer for it. *)

        IF (result <> NIL)
                AND OS2.PrfQueryProfileSize (hini, result^.Name, "Volume", BufferSize)
                AND (BufferSize <> 0) THEN
            ALLOCATE (bufptr, BufferSize);
            OS2.PrfQueryProfileData (hini, result^.Name, "Volume", bufptr, BufferSize);
        ELSE
            bufptr := NIL;
            BufferSize := 0;
        END (*IF*);
        BufferPosition := 0;
        Scan;

        (* Load the volume information. *)

        previousV := NIL;
        WHILE NextChar <> Nul DO
            previousV := VolumeInfo (result, previousV);
        END (*WHILE*);

        (* Close the INI file. *)

        OS2.PrfCloseProfile (hini);
        IF bufptr <> NIL THEN
            DEALLOCATE (bufptr, BufferSize);
        END (*IF*);

        IF result <> NIL THEN
            DefaultVolume := result^.FirstVol;
            result^.MultiVol := (DefaultVolume <> NIL) AND (DefaultVolume^.next <> NIL);

            (* Special addition for a super-directory. *)

            IF result^.MultiVol THEN
                AddSuperDirectory (result);
            END (*IF*);

            (* Set the user's initial directory. *)

            WITH result^ DO
                CurrentVol := FirstVol;
                IF FirstVol = NIL THEN
                    CurrentDir := "";
                    PosInTree := NIL;
                ELSE
                    CurrentDir := FirstVol^.Home;
                    PosInTree := FirstVol^.BaseEntry;
                END (*IF*);
            END (*WITH*);

        END (*IF*);

        RETURN result;

    END ReadUserData;

(********************************************************************************)

PROCEDURE PasswordAcceptable (U: User;  VAR (*IN*) pass: ARRAY OF CHAR): BOOLEAN;

    (* Tests for a password match. *)

    VAR dummy: CARDINAL;  found: BOOLEAN;

    BEGIN
        IF U = NIL THEN
            found := FALSE;
        ELSIF (U^.Password[0] = '@') AND (U^.Password[1] = Nul) THEN
            Strings.FindNext ('@', pass, 0, found, dummy);
        ELSE
            found := Strings.Equal (pass, U^.Password);
        END (*IF*);
        RETURN found;
    END PasswordAcceptable;

(********************************************************************************)

PROCEDURE GetUserNumber (U: User;  VAR (*OUT*) UserNumber, Limit: CARDINAL);

    (* Returns the user number and user limit, within the category defined by   *)
    (* this user's username.                                                    *)

    BEGIN
        IF U = NIL THEN
            UserNumber := 0;  Limit := 0;
        ELSE
            UserNumber := U^.UserNumber;
            Limit := U^.UserLimit;
        END (*IF*);
    END GetUserNumber;

(********************************************************************************)
(*                        DISCARDING A USER RECORD                              *)
(********************************************************************************)

PROCEDURE KillList (VAR (*INOUT*) p: DirEntryPtr);

    (* Discards a directory tree.  *)

    VAR q: DirEntryPtr;

    BEGIN
        WHILE p <> NIL DO
            KillList (p^.FirstChild);
            q := p^.next;
            DISPOSE (p);
            p := q;
        END (*IF*);
    END KillList;

(********************************************************************************)

PROCEDURE DestroyUserData (VAR (*INOUT*) U: User);

    (* Discards the data structure.  *)

    VAR V, V1: Volume;
        current: CountList;

    BEGIN
        IF U <> NIL THEN

            (* Decrement the number of instances of this username. *)

            Obtain (UserCountLock);
            current := UserCount;
            WHILE (current <> NIL) AND NOT Strings.Equal(current^.name, U^.Name) DO
                current := current^.next;
            END (*WHILE*);
            IF current <> NIL THEN
                DEC (current^.count);
            END (*IF*);
            Release (UserCountLock);

            (* Kill each volume. *)

            V := U^.FirstVol;
            WHILE V <> NIL DO
                KillList (V^.BaseEntry);  V1 := V^.next;
                DISPOSE (V);  V := V1;
            END (*WHILE*);

            DISPOSE (U);
        END (*IF*);
    END DestroyUserData;

(********************************************************************************)
(*                           STRING COMPARISON                                  *)
(********************************************************************************)

PROCEDURE NameMatch (first, second: ARRAY OF CHAR): BOOLEAN;

    (* Equality test, modulo upper/lower case differences.  That is, the result *)
    (* is TRUE even if the letters in first and second don't have the same      *)
    (* case, as long as the strings are otherwise equal.                        *)

    VAR k: CARDINAL;

    BEGIN
        k := 0;
        LOOP
            IF k > HIGH(second) THEN RETURN (k > HIGH(first));
            ELSIF k > HIGH(first) THEN RETURN FALSE;
            ELSIF first[k] = Nul THEN RETURN (second[k] = Nul);
            ELSIF CAP(first[k]) <> CAP(second[k]) THEN RETURN FALSE;
            ELSE INC(k);
            END (*IF*);
        END (*LOOP*);
    END NameMatch;

(********************************************************************************)

PROCEDURE WildMatch (first, second: ARRAY OF CHAR): BOOLEAN;

    (* Same as NameMatch, except that second is allowed to contain the          *)
    (* wildcard characters '*' and '?'.                                         *)

    VAR k1, k2: CARDINAL;

    BEGIN
        k1 := 0;  k2 := 0;
        LOOP
            IF k2 > HIGH(second) THEN RETURN (k1 > HIGH(first));
            ELSIF second[k2] = '*' THEN
                INC(k2);
                WHILE (k1 <= HIGH(first)) AND (first[k1] <> Nul)
                                AND (CAP(first[k1]) <> CAP(second[k2])) DO
                    INC (k1);
                END (*WHILE*);
            ELSIF k1 > HIGH(first) THEN RETURN FALSE;
            ELSIF first[k1] = Nul THEN RETURN (second[k2] = Nul);
            ELSIF second[k2] = '?' THEN INC(k1);  INC (k2);
            ELSIF CAP(first[k1]) <> CAP(second[k2]) THEN RETURN FALSE;
            ELSE INC(k1);  INC(k2);
            END (*IF*);
        END (*LOOP*);
    END WildMatch;

(********************************************************************************)
(*                             VOLUME OPERATIONS                                *)
(********************************************************************************)

PROCEDURE SpaceAvailable (name: FName): CARDINAL;

    (* Returns the free space, in kilobytes, on the drive that would be used    *)
    (* to store this file if we accepted it.                                    *)

    BEGIN
        IF (name = NIL) OR (name^.dir[0] = Nul) THEN
            RETURN 0;
        ELSIF name^.dir[1] = ':' THEN
            RETURN FreeSpace (name^.dir[0]);
        ELSE
            (* Temporary workaround for the fact that I don't know how to       *)
            (* calculate free space on a network drive.                         *)
            RETURN MAX(CARDINAL);
        END (*IF*);
    END SpaceAvailable;

(********************************************************************************)

PROCEDURE SearchVolume (V: Volume;  drive: ARRAY OF CHAR): Volume;

    (* Maps a virtual drive name to a Volume.  The result could be NIL    *)
    (* for an illegal or inaccessible drive.  Starts the search from V.   *)

    BEGIN
        WHILE (V <> NIL) AND NOT NameMatch (V^.VolumeName, drive) DO
            V := V^.next;
        END (*WHILE*);
        RETURN V;
    END SearchVolume;

(********************************************************************************)
(*                          FILE NAME PROCESSING                                *)
(********************************************************************************)

PROCEDURE MakeFullName (Name: FName;  VAR (*OUT*) FullName: ARRAY OF CHAR);

    (* Converts a relative file name to a full file name.  *)

    VAR k: CARDINAL;

    BEGIN
        Strings.Assign (Name^.dir, FullName);
        IF Name^.fname[0] = Nul THEN

            (* Strip trailing '/'. *)

            k := LENGTH (FullName);
            IF k > 0 THEN
                DEC (k);
                IF FullName[k] = '/' THEN
                    FullName[k] := Nul;
                END (*IF*);
            END (*IF*);
        ELSE
            Strings.Append (Name^.fname, FullName);
        END (*IF*);

    END MakeFullName;

(********************************************************************************)

PROCEDURE SplitHead (VAR (*INOUT*) src: ARRAY OF CHAR;  VAR (*OUT*) head: ARRAY OF CHAR);

    (* Finds the first '/' character in src.  Everything up to the separator is *)
    (* assigned to head, the separator itself is discarded, the remainder is    *)
    (* assigned to src.  If there is no '/', src becomes the null string.       *)

    (* (Any '\' found is treated as a '/'.)                                     *)

    VAR k: CARDINAL;  ch: CHAR;

    BEGIN
        k := 0;  ch := Nul;
        LOOP
            IF k > HIGH(src) THEN
                EXIT (*LOOP*);
            END (*IF*);
            ch := src[k];
            IF (ch = Nul) OR (ch = '/') OR (ch = '\')
                       OR (k > HIGH(head)) THEN
                EXIT (*LOOP*);
            ELSE
                head[k] := ch;  INC(k);
            END (*IF*);
        END (*LOOP*);
        IF k <= HIGH(head) THEN
            head[k] := Nul;
        END (*IF*);
        IF ch <> Nul THEN
            INC (k);
        END (*IF*);
        IF k > HIGH(src) THEN
            Strings.Assign ("", src);
        ELSE
            Strings.Delete (src, 0, k);
        END (*IF*);
    END SplitHead;

(********************************************************************************)

PROCEDURE SplitTail (VAR (*INOUT*) src: ARRAY OF CHAR;  VAR (*OUT*) head: ARRAY OF CHAR);

    (* Finds the last '/' character in src.  Everything up to the separator is  *)
    (* assigned to head, the separator is discarded, and src is set to whatever *)
    (* is left after the separator.  If there is no '/', head becomes the null  *)
    (* string.  Special case: if the last '/' is actually the first character   *)
    (* of src, then we return with head = "/".                                  *)

    (* (Any '\' found is treated as a '/'.)                                     *)

    VAR k: CARDINAL;  ch: CHAR;

    BEGIN
        k := Strings.Length (src);  ch := Nul;
        LOOP
            IF k = 0 THEN EXIT(*LOOP*) END(*IF*);
            DEC (k);
            ch := src[k];
            IF ch = '\' THEN ch := '/' END(*IF*);
            IF ch = '/' THEN EXIT (*LOOP*) END(*IF*);
        END (*LOOP*);
        IF ch = '/' THEN
            Strings.Assign (src, head);
            Strings.Delete (src, 0, k+1);
            IF k = 0 THEN INC(k) END(*IF*);
        END (*IF*);
        head[k] := Nul;
    END SplitTail;

(********************************************************************************)

PROCEDURE StripTail (VAR (*INOUT*) name: ARRAY OF CHAR);

    (* Deletes the last subdirectory specification from name, for example       *)
    (* we change A/B/C/D/ to A/B/C/.                                            *)

    VAR k: CARDINAL;

    BEGIN
        k := Strings.Length (name);
        IF k > 0 THEN
            REPEAT
                DEC (k);
            UNTIL (k = 0) OR (name[k-1] = '/') OR (name[k-1] = '\');
        END (*IF*);
        name[k] := Nul;
    END StripTail;

(********************************************************************************)

PROCEDURE TailMatch (first, second: ARRAY OF CHAR): BOOLEAN;

    (* Tests whether the tail of first matches second. *)

    VAR head: FileNameString;

    BEGIN
        SplitTail (first, head);
        RETURN NameMatch (first, second);
    END TailMatch;

(********************************************************************************)
(*                              DIRECTORIES                                     *)
(********************************************************************************)

PROCEDURE FileIsADirectory (parent: FName;  name: ARRAY OF CHAR): BOOLEAN;

    (* Returns TRUE iff the file exists and is a directory.   *)

    VAR V: Volume;  FullName: FileNameString;  D: DirectoryEntry;  result: BOOLEAN;

    BEGIN
        V := parent^.vol;
        IF V = NIL THEN result := FALSE
        ELSIF V^.SuperDir = V THEN
            result := (name[0] = Nul) OR ((name[0] = '.') AND (name[1] = Nul));
        ELSE
            FullName := parent^.dir;
            Strings.Append (name, FullName);

            (* Watch out for wildcard matches! *)

            result := FirstDirEntry (FullName, TRUE, D)
                         AND (directory IN D.attr)
                         AND TailMatch (name, D.name);
            DirSearchDone (D);
        END (*IF*);

        RETURN result;

    END FileIsADirectory;

(********************************************************************************)

PROCEDURE FindDirEntry (parent: FName;  name: ARRAY OF CHAR;
                                                  NoCheck: BOOLEAN): DirEntryPtr;

    (* Looks for name in the directory described by the directory part of       *)
    (* parent.  (The fname component of parent is not used.)  If subdirectory   *)
    (* name is not already in the directory tree, but name exists and is a      *)
    (* directory, an entry is added to the tree.  If the NoCheck flag is set,   *)
    (* we trust the caller's assertion that name is indeed a directory,         *)
    (* so we don't do the check again.                                          *)
    (* Returns NIL if the file doesn't exist.                                   *)

    (* Special case: we don't want ".." records wasting memory, so we deal      *)
    (* with them in such a way that they don't get added to the tree.           *)

    VAR V: Volume;  father, previous, current: DirEntryPtr;

    BEGIN
        V := parent^.vol;
        father := parent^.EntryPtr;
        IF Strings.Equal (name, "..") THEN
            current := father^.parent;
            IF (current = NIL) AND (V^.SuperDir <> NIL) AND (V^.SuperDir <> V) THEN
                current := V^.SuperDir^.BaseEntry;
            END (*IF*);
            RETURN current;
        END (*IF*);

        previous := NIL;  current := father^.FirstChild;
        LOOP
            IF current = NIL THEN EXIT(*LOOP*) END(*IF*);
            IF NameMatch (name, current^.name) THEN EXIT(*LOOP*) END(*IF*);
            previous := current;  current := current^.next;
        END (*LOOP*);

        IF current = NIL THEN
            IF NoCheck OR FileIsADirectory (parent, name) THEN
                NEW (current);
                Strings.Assign (name, current^.name);
                current^.parent := father;
                current^.FirstChild := NIL;  current^.next := NIL;
                current^.flags := father^.flags;
                current^.BeenHereBefore := FALSE;
                IF previous = NIL THEN
                    father^.FirstChild := current;
                ELSE
                    previous^.next := current;
                END (*IF*);
            END (*IF*);
        END (*IF*);

        RETURN current;

    END FindDirEntry;

(********************************************************************************)
(*                         THE INTERNAL DATA TYPE FName                         *)
(********************************************************************************)

PROCEDURE MakeFName (U: User;  filename: ARRAY OF CHAR): FName;

    (* Creates an FName descriptor from the filename.  The result is NIL for    *)
    (* a name that implies a nonexistent directory.                             *)

    VAR result: FName;

    (****************************************************************************)

    PROCEDURE ChangeToVolume (newV: Volume);

        (* Sets result to the root directory of newV. *)

        BEGIN
            WITH result^ DO
                vol := newV;
                IF vol = NIL THEN
                    dir := "";  EntryPtr := NIL;
                ELSE
                    dir := vol^.Home;  EntryPtr := vol^.BaseEntry;
                END (*IF*);
            END (*WITH*);
        END ChangeToVolume;

    (****************************************************************************)

    VAR head: FileNameString;
        NewEntryPtr: DirEntryPtr;  FinalSlash, finished: BOOLEAN;
        k: CARDINAL;

    BEGIN
        (* For later reference, we want to know whether the name ends with '/'. *)

        k := LENGTH (filename);
        FinalSlash := (k > 0) AND ((filename[k-1] = '/') OR (filename[k-1] = '\'));

        NEW (result);
        WITH result^ DO
            Strings.Assign (filename, fname);

            (* Start at the user's root directory or the user's current     *)
            (* directory, depending on whether the name starts with a '/'.  *)

            IF fname[0] = '/' THEN
                ChangeToVolume (U^.FirstVol);
                Strings.Delete (fname, 0, 1);
            ELSE
                vol := U^.CurrentVol;
                dir := U^.CurrentDir;
                EntryPtr := U^.PosInTree;
            END (*IF*);

            IF vol = NIL THEN
                EntryPtr := NIL;
            END (*IF*);

            (* Now strip all leading directory information from fname, and  *)
            (* move it to dir, updating EntryPtr as we go.                  *)

            finished := (EntryPtr = NIL) OR (fname[0] = CHR(0));
            WHILE NOT finished DO

                SplitHead (fname, head);
                IF Strings.Equal (head, "..") THEN

                    (* Change to parent directory. *)

                    EntryPtr := EntryPtr^.parent;
                    IF EntryPtr = NIL THEN
                        IF vol <> U^.FirstVol THEN
                            ChangeToVolume (U^.FirstVol);
                        END (*IF*);
                    ELSE
                        StripTail (dir);
                    END (*IF*);

                ELSIF (head[0] = '.') AND (head[1] = Nul) THEN

                    (* Change to current directory, i.e. do nothing. *)

                ELSIF vol^.SuperDir = vol THEN

                    (* Change from root to a real volume. *)

                    ChangeToVolume (SearchVolume(vol, head));

                ELSE
                    (* Try to move to a subdirectory of the current directory. *)

                    NewEntryPtr := FindDirEntry (result, head, FALSE);
                    IF NewEntryPtr = NIL THEN

                        (* 'head' is not the name of a subdirectory. *)

                        finished := TRUE;
                        IF (fname[0] = CHR(0)) AND NOT FinalSlash THEN

                            (* Still OK, we must have arrived at a file name or *)
                            (* mask instead of a subdirectory name.             *)

                            fname := head;

                        ELSE

                            (* Illegal file name, clear EntryPtr to show error. *)

                            EntryPtr := NIL;

                        END (*IF*);

                    ELSE

                        (* Move to subdirectory. *)

                        EntryPtr := NewEntryPtr;
                        Strings.Append (head, dir);
                        Strings.Append ('/', dir);

                    END (*IF*);

                END (*IF*);

                finished := finished OR (EntryPtr = NIL) OR (fname[0] = CHR(0));

            END (*WHILE*);

        END (*WITH*);

        RETURN result;

    END MakeFName;

(********************************************************************************)

PROCEDURE DiscardFName (VAR (*INOUT*) name: FName);

    (* Disposes of the storage used by name. *)

    BEGIN
        IF name <> NIL THEN
            DISPOSE (name);
        END (*IF*);
    END DiscardFName;

(********************************************************************************)
(*                                FILE OPERATIONS                               *)
(********************************************************************************)

PROCEDURE OpenForReading (VAR (*OUT*) cid: IOChan.ChanId;  name: FName): BOOLEAN;

    (* Opens the file, returns TRUE iff successful. *)

    VAR result: ChanConsts.OpenResults;
        FullName: FileNameString;

    BEGIN
        MakeFullName (name, FullName);
        RndFile.OpenOld (cid, FullName, ChanConsts.read+ChanConsts.old+ChanConsts.raw, result);
        RETURN result = ChanConsts.opened;
    END OpenForReading;

(********************************************************************************)

PROCEDURE OpenForWriting (VAR (*OUT*) cid: IOChan.ChanId;  name: FName): BOOLEAN;

    (* Opens the file, returns TRUE iff successful. *)

    VAR result: ChanConsts.OpenResults;  success: BOOLEAN;
        FullName: FileNameString;

    BEGIN
        MakeFullName (name, FullName);
        IF FileSys.Exists (FullName) THEN
            FileSys.Remove (FullName, success);
            IF NOT success THEN
                RETURN FALSE;
            END (*IF*);
        END (*IF*);
        RndFile.OpenClean (cid, FullName, ChanConsts.write+ChanConsts.raw, result);
        RETURN result = ChanConsts.opened;
    END OpenForWriting;

(********************************************************************************)

PROCEDURE OpenForAppend (VAR (*OUT*) cid: IOChan.ChanId;  name: FName): BOOLEAN;

    (* Opens the file, returns TRUE iff successful. *)

    VAR result: ChanConsts.OpenResults;  FullName: FileNameString;

    BEGIN
        MakeFullName (name, FullName);
        IF FileSys.Exists (FullName) THEN
            RndFile.OpenOld (cid, FullName, ChanConsts.write+ChanConsts.raw, result);
            RndFile.SetPos (cid, RndFile.EndPos(cid));
        ELSE
            RndFile.OpenClean (cid, FullName, ChanConsts.write+ChanConsts.raw, result);
        END (*IF*);
        RETURN result = ChanConsts.opened;
    END OpenForAppend;

(********************************************************************************)

PROCEDURE RemoveFile (name: FName): BOOLEAN;

    (* Deletes the file, returns TRUE iff successful. *)

    VAR result: BOOLEAN;
        FullName: FileNameString;

    BEGIN
        MakeFullName (name, FullName);
        FileSys.Remove (FullName, result);
        RETURN result;
    END RemoveFile;

(********************************************************************************)

PROCEDURE RenameTo (OldName: FName;  NewName: ARRAY OF CHAR): BOOLEAN;

    (* Renames OldName to NewName.  Returns TRUE for success.  *)

    VAR result: BOOLEAN;
        FullName1, FullName2: FileNameString;

    BEGIN
        MakeFullName (OldName, FullName1);
        Strings.Assign (NewName, OldName^.fname);
        MakeFullName (OldName, FullName2);
        FileSys.Rename (FullName1, FullName2, result);
        RETURN result;
    END RenameTo;

(********************************************************************************)
(*                            DIRECTORY OPERATIONS                              *)
(********************************************************************************)

PROCEDURE Encode (flags: PermissionSet;  VAR (*OUT*) result: ARRAY OF CHAR);

    (* Translates flags into a three-character string.  The Visible attribute   *)
    (* is not encoded, since we're never going to use this encoding for an      *)
    (* invisible directory.                                                     *)

    BEGIN
        Strings.Assign ("---", result);
        IF AllowRead IN flags THEN
            result[0] := 'r';
        END (*IF*);
        IF AllowWrite IN flags THEN
            result[1] := 'w';
        END (*IF*);
        IF AllowDelete IN flags THEN
            result[2] := 'x';
        END (*IF*);
    END Encode;

(********************************************************************************)

PROCEDURE PermissionString (U: User;  VAR (*OUT*) result: ARRAY OF CHAR);

    (* Returns a string indicating read/write/delete permissions for the        *)
    (* user's current directory.                                                *)

    VAR p: DirEntryPtr;

    BEGIN
        Strings.Assign ("---", result);
        p := U^.PosInTree;
        IF p <> NIL THEN
            Encode (p^.flags, result);
        END (*IF*);
    END PermissionString;

(********************************************************************************)

PROCEDURE FileExists (name: FName;  ShowHidden: BOOLEAN): BOOLEAN;

    (* Returns TRUE iff the file exists.  System and hidden files will appear   *)
    (* to be nonexistent unless ShowHidden is TRUE.                             *)

    VAR FullName: FileNameString;  D: DirectoryEntry;  result: BOOLEAN;
        V: Volume;

    BEGIN
        IF name = NIL THEN
            RETURN FALSE;
        END (*IF*);
        V := name^.vol;
        IF V = NIL THEN
            RETURN FALSE;
        ELSIF V^.SuperDir = V THEN
            RETURN name^.fname[0] = Nul;
        ELSE
            MakeFullName (name, FullName);
            result := FirstDirEntry (FullName, FALSE, D)
                         AND TailMatch (FullName, D.name)
                         AND (ShowHidden
                             OR NOT ((hidden IN D.attr) OR (system IN D.attr)));
            DirSearchDone (D);
            RETURN result;
        END (*IF*);
    END FileExists;

(********************************************************************************)

PROCEDURE HaveSeenMessage (U: User): BOOLEAN;

    (* Returns a flag that says whether this is the first call of this          *)
    (* procedure for this user and this user's current directory.               *)

    VAR result: BOOLEAN;

    BEGIN
        IF (U=NIL) OR (U^.PosInTree = NIL) THEN
            result := TRUE;
        ELSE
            result := U^.PosInTree^.BeenHereBefore;
            U^.PosInTree^.BeenHereBefore := TRUE;
        END (*IF*);
        RETURN result;
    END HaveSeenMessage;

(********************************************************************************)

PROCEDURE NameOfCurrentDirectory (U: User;  VAR (*OUT*) DirString: ARRAY OF CHAR);

    (* Gives back the name of the current directory for this user.  *)

    VAR V: Volume;  RelDir: FileNameString;  k: CARDINAL;

    BEGIN
        V := U^.CurrentVol;
        IF V = NIL THEN
            Strings.Assign ("?", DirString);
        ELSIF V^.SuperDir = V THEN
            Strings.Assign ("/", DirString);
        ELSE
            RelDir := U^.CurrentDir;
            k := LENGTH (V^.Home);
            IF k > 0 THEN
                Strings.Delete (RelDir, 0, k);
            END (*IF*);
            Strings.Assign ("/", DirString);
            IF U^.MultiVol THEN
                Strings.Append (V^.VolumeName, DirString);
                IF RelDir[0] <> Nul THEN
                    Strings.Append ("/", DirString);
                END (*IF*);
            END (*IF*);
            Strings.Append (RelDir, DirString);
        END (*IF*);
    END NameOfCurrentDirectory;

(********************************************************************************)

PROCEDURE SetWorkingDirectory (U: User;  newdir: FName): BOOLEAN;

    (* Changes user to the specified directory.  Returns FALSE if the requested    *)
    (* directory does not exist, or if the user does not have the right to see it. *)

    VAR success: BOOLEAN;

    BEGIN
        success := (newdir <> NIL) AND (newdir^.fname[0] = CHR(0))
                           AND (newdir^.EntryPtr <> NIL)
                           AND (Visible IN newdir^.EntryPtr^.flags);
        IF success THEN
            WITH U^ DO
                CurrentVol := newdir^.vol;
                PosInTree := newdir^.EntryPtr;
                CurrentDir := newdir^.dir;
            END (*WITH*);
        END (*IF*);
        RETURN success;
    END SetWorkingDirectory;

(********************************************************************************)

PROCEDURE SendDirDetails (S: Socket;  D: DirectoryEntry;
                                            permissions: ARRAY OF CHAR): BOOLEAN;

    (* Decoding of date, time, attributes, etc of a directory entry. *)

    (****************************************************************************)

    PROCEDURE SendDateTime (datecode, timecode: CARD16): BOOLEAN;

        (* This version sends American-style dates, for compatibility with      *)
        (* ftp clients that are fussy about the format.                         *)

        VAR day, month, year: CARDINAL;

        (************************************************************************)

        PROCEDURE AgeInMonths(): INTEGER;

            VAR Now: SysClock.DateTime;
            BEGIN
                SysClock.GetClock (Now);
                RETURN 12*(VAL(INTEGER,Now.year) - VAL(INTEGER,year))
                       + VAL(INTEGER,Now.month) - VAL(INTEGER,month);
            END AgeInMonths;

        (************************************************************************)

        TYPE MonthNameType = ARRAY [0..15] OF ARRAY [0..2] OF CHAR;

        CONST MonthName = MonthNameType {'M00', 'Jan', 'Feb', 'Mar', 'Apr', 'May',
                                         'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov',
                                         'Dec', 'M13', 'M14', 'M15'};

        VAR minute, hour: CARDINAL;  age: INTEGER;  success: BOOLEAN;

        BEGIN
            day := datecode MOD 32;  datecode := datecode DIV 32;
            month := datecode MOD 16;  year := 1980 + datecode DIV 16;
            success := SendString (S, MonthName[month]) AND SendChar (S, ' ')
                          AND SendZCard (S, day, 2) AND SendChar (S, ' ');
               IF success THEN
                   age := AgeInMonths();
                   IF (age >= 0) AND (age <= 6) THEN
                       timecode := timecode DIV 32;
                       minute := timecode MOD 64;  hour := timecode DIV 64;
                       success := SendZCard (S, hour, 2) AND SendChar (S, ':')
                                AND SendZCard (S, minute, 2);
                   ELSE
                       success := SendChar (S, ' ') AND SendZCard (S, year, 4);
                   END (*IF*);
               END (*IF*);
            RETURN success;
        END SendDateTime;

    (****************************************************************************)

    PROCEDURE SendAttributes (attr: FileAttr): BOOLEAN;

        (************************************************************************)

        PROCEDURE PutAttr (totest: FileAttribute;  code: CHAR): BOOLEAN;

            BEGIN
                IF NOT (totest IN attr) THEN
                    code := '-';
                END (*IF*);
                RETURN SendChar (S, code);
            END PutAttr;

        (************************************************************************)

        BEGIN
            RETURN PutAttr (directory, 'd') AND SendString (S, "------")
               AND SendString (S, permissions) AND SendString (S, "   0 0     ")
               AND PutAttr (readonly, 'R')
               AND PutAttr (archive, 'A')
               AND PutAttr (system, 'S')
               AND PutAttr (hidden, 'H');
        END SendAttributes;

    (****************************************************************************)

    BEGIN
        RETURN SendAttributes (D.attr)
                  AND SendCard (S, D.size, 15)
                  AND SendChar (S, ' ')
                  AND SendDateTime (D.datePkd, D.timePkd)
                  AND SendChar (S, ' ');
    END SendDirDetails;

(********************************************************************************)

PROCEDURE ListSuperDirectory (S: Socket;  V: Volume;  options: ListingOptions);

    (* Sends a directory listing for the superdirectory.  *)

    VAR success: BOOLEAN;  pstring: ARRAY [0..2] OF CHAR;

    BEGIN
        V := V^.next;
        success := (V <> NIL);
        WHILE success DO

            (* Note re date: we can't report Jan 01 1980 because that  *)
            (* triggers a bug in Netscape.                             *)

            IF ShowDetails IN options THEN
                Encode (V^.BaseEntry^.flags, pstring);
                success := SendChar (S, "d") AND SendString (S, "------")
                             AND SendString (S, pstring)
                             AND SendString (S, "   0 0     ----")
                             AND SendString (S, "              0 Jan 02  1980 ");
            END (*IF*);
            success := success AND SendString (S, V^.VolumeName);
            IF AddSlash IN options THEN
                success := success AND SendChar (S, '/');
            END (*IF*);
            success := success AND SendEOL(S);

            V := V^.next;
            success := success AND (V <> NIL);

        END (*WHILE*);

    END ListSuperDirectory;

(********************************************************************************)

PROCEDURE ListRealDirectory (S: Socket;  arg: FName;  options: ListingOptions);

    (* Sends a directory listing, as specified by arg.  *)

    VAR D: DirectoryEntry;  p: DirEntryPtr;
        Name: FileNameString;
        success, SendIt: BOOLEAN;
        pstring: ARRAY [0..2] OF CHAR;

    BEGIN
        (* An empty filename mask means "entire directory", unless the -d       *)
        (* option suppresses this interpretation.                               *)

        IF (arg^.fname[0] = Nul) AND (MayExpand IN options) THEN
            arg^.fname[0] := '*';  arg^.fname[1] := Nul;
        END (*IF*);

        (* Treat the ".." entry separately, to circumvent problems related to   *)
        (* the status of ".." in the root directory of a volume.                *)

        success := TRUE;
        IF (ListDotDot IN options) AND WildMatch ("..", arg^.fname) THEN

            p := arg^.EntryPtr^.parent;
            IF (p = NIL) AND (arg^.vol^.SuperDir <> NIL) THEN
                p := arg^.vol^.SuperDir^.BaseEntry;
            END (*IF*);

            IF (p <> NIL) AND (Visible IN p^.flags) THEN
                IF ShowDetails IN options THEN
                    Encode (p^.flags, pstring);
                    success := SendString (S, "d------")
                           AND SendString (S, pstring)
                           AND SendString (S, "   0 0     ----")
                           (*AND SendString (S, "              0              ");*)
                           AND SendString (S, "              0 Jan 02  1980 ");
                END (*IF*);
                success := success AND SendString (S, "..");
                IF AddSlash IN options THEN
                    success := success AND SendChar (S, '/');
                END (*IF*);
                success := success AND SendEOL(S);
            END (*IF*);

        END (*IF*);

        (* Now the main part of the listing. *)

        MakeFullName (arg, Name);
        success := success AND FirstDirEntry (Name, FALSE, D);
        WHILE success DO

            SendIt := TRUE;  p := arg^.EntryPtr;

            (* Skip over the "." and ".." entries. *)

            IF D.name[0] = '.' THEN
                SendIt := (D.name[1] <> Nul) AND
                                 ((D.name[1] <> '.') OR (D.name[2] <> Nul));
            END (*IF*);

            (* Work out whether we want to send this entry. *)

            IF SendIt THEN
                SendIt := (SystemAndHidden IN options)
                          OR NOT ((hidden IN D.attr) OR (system IN D.attr));
            END (*IF*);

            (* If it's a directory, check whether it should be hidden. *)

            IF SendIt AND (directory IN D.attr) THEN
                p := FindDirEntry (arg, D.name, TRUE);
                SendIt := (p <> NIL) AND (Visible IN p^.flags);
            END (*IF*);

            IF SendIt THEN
                IF ShowDetails IN options THEN
                    Encode (p^.flags, pstring);
                    success := SendDirDetails (S, D, pstring);
                END (*IF*);
                success := success AND SendString (S, D.name);
                IF AddSlash IN options THEN
                    IF directory IN D.attr THEN
                        success := success AND SendChar (S, '/');
                    END (*IF*);
                END (*IF*);
                success := success AND SendEOL(S);
            END (*IF*);

            success := success AND NextDirEntry (D);

        END (*WHILE*);
        DirSearchDone (D);

    END ListRealDirectory;

(********************************************************************************)

PROCEDURE ListDirectory (S: Socket;  arg: FName;  options: ListingOptions);

    (* Sends a directory listing, as specified by arg and options.  *)

      VAR V: Volume;

      BEGIN
        V := arg^.vol;
        IF V <> NIL THEN
            IF V^.SuperDir = V THEN
                IF arg^.fname[0] = Nul THEN
                    ListSuperDirectory (S, V, options);
                END (*IF*);
            ELSE
                ListRealDirectory (S, arg, options);
            END (*IF*);
        END (*IF*);
    END ListDirectory;

(********************************************************************************)

PROCEDURE CreateDirectory (Name: FName): BOOLEAN;

    (* Creates a new directory. *)

    VAR FullName: FileNameString;  result: CARDINAL;

    BEGIN
        IF (Name^.EntryPtr <> NIL) AND (AllowWrite IN Name^.EntryPtr^.flags) THEN
            MakeFullName (Name, FullName);
            CreateDir (FullName, result);
        ELSE
            result := 1;
        END (*IF*);
        RETURN result = 0;
    END CreateDirectory;

(********************************************************************************)

PROCEDURE RemoveDirectory (Name: FName): BOOLEAN;

    (* Deletes a directory. *)

    VAR FullName: FileNameString;  result: CARDINAL;

    BEGIN
        MakeFullName (Name, FullName);
        RmvDir (FullName, result);
        RETURN result = 0;
    END RemoveDirectory;

(********************************************************************************)

PROCEDURE CanReadDir (iname: FName): BOOLEAN;

    (* Returns TRUE iff the user has permission to read the directory   *)
    (* implied by iname.                                                *)

    VAR p: DirEntryPtr;

    BEGIN
        IF iname = NIL THEN
            RETURN FALSE;
        ELSE
            p := iname^.EntryPtr;
            RETURN (p <> NIL) AND (Visible IN p^.flags);
        END (*IF*);
    END CanReadDir;

(********************************************************************************)

PROCEDURE CanReadFile (Name: FName;  Manager: BOOLEAN): BOOLEAN;

    (* Returns TRUE iff the file exists and is readable by this user.   *)

    BEGIN
        RETURN (Name <> NIL) AND (Name^.fname[0] <> Nul)
               AND FileExists (Name, Manager)
               AND (Name^.EntryPtr <> NIL)
               AND (AllowRead IN Name^.EntryPtr^.flags);
    END CanReadFile;

(********************************************************************************)

PROCEDURE GetFileSize (Name: FName): CARDINAL;

    (* Returns the size in bytes of file "Name".  If the file is not    *)
    (* accessible, the result is returned as MAX(CARDINAL).             *)

    VAR D: DirectoryEntry;  FullName: FileNameString;  result: CARDINAL;

    BEGIN
        IF CanReadDir (Name) THEN
            MakeFullName (Name, FullName);
            IF FirstDirEntry (FullName, FALSE, D) THEN
                result := D.size;
            ELSE
                result := MAX(CARDINAL);
            END (*IF*);
            DirSearchDone (D);
        ELSE
            result := MAX(CARDINAL);
        END (*IF*);
        RETURN result;
    END GetFileSize;

(********************************************************************************)

PROCEDURE GetDateTime (iname: FName;  VAR (*OUT*) result: ARRAY OF CHAR);

    (* Returns the date/time of the file's directory entry, in a string of the  *)
    (* form "yyyymmddhhmmss" (exactly 14 characters).  If the file is not       *)
    (* accessible, the result is the null string.                               *)

    VAR place: CARDINAL;

    (****************************************************************************)

    PROCEDURE Encode1 (value: CARDINAL);

        BEGIN
            result[place] := CHR(ORD('0') + value);  INC (place);
        END Encode1;

    (****************************************************************************)

    PROCEDURE Encode (value: CARDINAL);

        BEGIN
            Encode1 (value DIV 10);  Encode1 (value MOD 10);
        END Encode;

    (****************************************************************************)

    VAR D: DirectoryEntry;  FullName: FileNameString;  V: Volume;
        year, month, day, hour, minute, second: CARDINAL;

    BEGIN
        V := iname^.vol;
        place := 0;
        IF V = NIL THEN
            (* Do nothing *)
        ELSIF V^.SuperDir = V THEN
            Strings.Assign ("19800101000000", result);
            place := 14;
        ELSIF CanReadDir (iname) THEN
            MakeFullName (iname, FullName);
            IF FirstDirEntry (FullName, FALSE, D) THEN
                year := D.datePkd;
                day := year MOD 32;  year := year DIV 32;
                month := year MOD 16;  year := 1980 + year DIV 16;
                hour := D.timePkd;
                second := 2*(hour MOD 32);  hour := hour DIV 32;
                minute := hour MOD 64;  hour := hour DIV 64;
                Encode (year DIV 100);  Encode (year MOD 100);
                Encode (month);  Encode (day);
                Encode (hour);  Encode (minute);  Encode (second);
            END (*IF*);
        END (*IF*);
        IF place <= HIGH(result) THEN
            result[place] := Nul;
        END (*IF*);
    END GetDateTime;

(********************************************************************************)

PROCEDURE CanWriteFile (Name: FName;  Manager: BOOLEAN): BOOLEAN;

    (* Returns TRUE iff we can create a file of this name.  (If the file        *)
    (* already exists, we must have delete permission for it.)                  *)

    VAR perm: PermissionSet;  result: BOOLEAN;

    BEGIN
        IF (Name = NIL) OR (Name^.EntryPtr = NIL) THEN
            perm := PermissionSet{};
        ELSE
            perm := Name^.EntryPtr^.flags;
        END (*IF*);
        result := AllowWrite IN perm;
        IF result AND FileExists (Name, Manager) THEN
            result := AllowDelete IN perm;
        END (*IF*);
        RETURN result;
    END CanWriteFile;

(********************************************************************************)

PROCEDURE CanDeleteFile (Name: FName;  Manager: BOOLEAN): BOOLEAN;

    (* Returns TRUE iff the file exists and is readable by this user.   *)

    BEGIN
        RETURN FileExists (Name, Manager) AND (Name^.EntryPtr <> NIL)
               AND (AllowDelete IN Name^.EntryPtr^.flags);
    END CanDeleteFile;

(********************************************************************************)

PROCEDURE CanDeleteDirectory (Name: FName): BOOLEAN;

    (* Returns TRUE iff user can delete directory DirName.  *)

    VAR pos: DirEntryPtr;  V: Volume;

    BEGIN
        IF (Name <> NIL) AND (Name^.fname[0] = Nul) THEN
            pos := Name^.EntryPtr;
        ELSE
            pos := NIL;
        END (*IF*);

        (* The permission we're looking for is actually in the directory        *)
        (* record of the parent of this directory.                              *)

        IF pos <> NIL THEN
            pos := pos^.parent;
            IF pos = NIL THEN
                V := Name^.vol;
                IF (V^.SuperDir <> NIL) AND (V^.SuperDir <> V) THEN
                    pos := V^.SuperDir^.BaseEntry;
                END (*IF*);
            END (*IF*);
        END (*IF*);

        RETURN (pos <> NIL) AND (AllowDelete IN pos^.flags);

        (*
        RETURN (Name <> NIL) AND (Name^.fname[0] = Nul)
               AND (Name^.EntryPtr <> NIL)
               AND (AllowDelete IN Name^.EntryPtr^.flags);
        *)
    END CanDeleteDirectory;

(********************************************************************************)
(*                              INITIALISATION                                  *)
(********************************************************************************)

BEGIN
    UserCount := NIL;
    CreateLock (UserCountLock);
END FDUsers.

