IMPLEMENTATION MODULE PermissionEditor;

        (********************************************************)
        (*                                                      *)
        (*         Editor for the user access rights            *)
        (*                                                      *)
        (*  Programmer:         P. Moylan                       *)
        (*  Started:            2 December 1997                 *)
        (*  Last edited:        27 July 1998                    *)
        (*  Status:             Working                         *)
        (*                                                      *)
        (********************************************************)

(************************************************************************)
(*                                                                      *)
(*     Syntax for the permissions file:                                 *)
(*                                                                      *)
(*     <result>  ->  <userclass> <password> { <volumeinfo> }*           *)
(*     <volumeinfo> ->  <homedir> { <dirrule> }                         *)
(*     <userclass> -> G | U | N | M                                     *)
(*     <password>  -> <namestring>                                      *)
(*     <homedir>    ->  <namestring>  |  <namestring> = <namestring>    *)
(*     <diritem>   -> <namestring> <dirrule>                            *)
(*     <dirrule>   -> { <permission> }* { ( <dirlist> ) }               *)
(*     <dirlist>   ->  <diritem> { , <diritem> }*                       *)
(*     <dirlist>   -> { <diritem> }+                                    *)
(*     <permission> ->  V+ | V- | R+ | R- | W- | W+ | D- | D+           *)
(*     <namestring> ->  <string1> | " <string2> " | ' <string3> '       *)
(*     <string1>   -> any string not including space char               *)
(*     <string2>   -> any string not including double quote             *)
(*     <string3>   -> any string not including single quote             *)
(*                                                                      *)
(*  Notation: {} means optional, {}* means zero or more, {}+ means      *)
(*            one or more.                                              *)
(*                                                                      *)
(************************************************************************)

IMPORT Strings;

FROM SYSTEM IMPORT ADR, ADDRESS;

FROM ListBoxes IMPORT
    (* type *)  ListBox,
    (* proc *)  HighlightOn, HighlightOff, CursorBackward, CursorForward,
                LBAppend, ReplaceCurrent, LBDeleteCurrent,
                LBInsertAfter, DisableScreenOutput, Repaint;

FROM MultiScreen IMPORT
    (* type *)  VirtualScreen,
    (* proc *)  VirtualScreenOf, MapToVirtualScreen;

FROM Windows IMPORT
    (* type *)  Window, Colour, FrameType, DividerType,
    (* proc *)  OpenWindowHidden, CloseWindow, WriteChar, WriteString, WriteLn,
                PressAnyKey, EditString, GetKey, GetScreenSize;

FROM FDFiles IMPORT
    (* type *)  FileAttr, FileAttribute, DirectoryEntry,
                FirstDirEntry, NextDirEntry, DirSearchDone;

FROM LowLevel IMPORT
    (* proc *)  EVAL;

FROM Storage IMPORT
    (* proc *)  ALLOCATE, DEALLOCATE;

(********************************************************************************)

CONST Nul = CHR(0);
    Space = " ";

TYPE
    FileNameSubscript = [0..255];
    FileNameString = ARRAY FileNameSubscript OF CHAR;

    (* The following declarations relate to the data structure that we keep,    *)
    (* for a logged-in user, to show which directories and files are accessible.*)
    (* For each directory, we have a linked list of subdirectories.  The left   *)
    (* and right pointers are filled in when we need to thread the structure.   *)
    (* The altright pointer is a saved copy of right for use when this node has *)
    (* a collapsed subtree.  It turns out that you can't do a similar thing for *)
    (* the left threads; they have to be readjusted each time we expand or      *)
    (* collapse a subtree.                                                      *)

    Permission = (Visible, AllowRead, AllowWrite, AllowDelete);
    PermissionSet = SET OF Permission;

    DirEntryPtr = POINTER TO DirEntry;

    DirEntry = RECORD
                   collapsed: BOOLEAN;
                   indent: CARDINAL;
                   flags: PermissionSet;
                   parent, FirstChild, next: DirEntryPtr;
                   name: FileNameString;
                   left, right, altright: DirEntryPtr;
               END (*RECORD*);

VAR ScreenRows: CARDINAL;

(********************************************************************************)
(*                       PERMISSION ENCODING/DECODING                           *)
(********************************************************************************)

PROCEDURE CharToPermission (ch: CHAR): Permission;

    (* Translates a code (V,R,W,D) to a permission. *)

    BEGIN
        ch := CAP(ch);
        IF ch = 'R' THEN RETURN AllowRead
        ELSIF ch = 'W' THEN RETURN AllowWrite
        ELSIF ch = 'D' THEN RETURN AllowDelete
        ELSE RETURN Visible
        END (*IF*);
    END CharToPermission;

(********************************************************************************)
(*                          CREATING A NEW NODE                                 *)
(********************************************************************************)

PROCEDURE CreateDirEntry (dirname: ARRAY OF CHAR): DirEntryPtr;

    (* Creates a new entry. *)

    VAR result: DirEntryPtr;

    BEGIN
        NEW (result);
        WITH result^ DO
            collapsed := FALSE;
            flags := PermissionSet {Visible, AllowRead};
            parent := NIL;  FirstChild := NIL;  next := NIL;
            Strings.Assign (dirname, name);
        END (*WITH*);
        RETURN result;
    END CreateDirEntry;

(********************************************************************************)
(*                                PARSER                                        *)
(********************************************************************************)

PROCEDURE Scan (bufptr: CharArrayPointer;  VAR (*INOUT*) position: CARDINAL;
                                            BufferSize: CARDINAL): CHAR;

    (* Returns the character at bufptr^[position], and updates position. *)

    VAR NextChar: CHAR;

    BEGIN
        IF position >= BufferSize THEN
            NextChar := Nul;
        ELSE
            NextChar := bufptr^[position];  INC(position);
        END (*IF*);
        RETURN NextChar;
    END Scan;

(****************************************************************************)

PROCEDURE LoadString (bufptr: CharArrayPointer;  VAR (*INOUT*) position: CARDINAL;
                            BufferSize: CARDINAL;  VAR (*OUT*) result: ARRAY OF CHAR;
                           VAR (*INOUT*) NextChar: CHAR;  Stoppers: CharSet);

    (* Reads a string up to but not including a character in Stoppers.      *)

    VAR j: CARDINAL;

    BEGIN
        j := 0;
        WHILE (j <= HIGH(result)) AND NOT (NextChar IN Stoppers) DO
            result[j] := NextChar;  INC(j);
            NextChar := Scan (bufptr, position, BufferSize);
        END (*WHILE*);
        IF j <= HIGH(result) THEN result[j] := Nul END(*IF*);
    END LoadString;

(****************************************************************************)

PROCEDURE NameString (bufptr: CharArrayPointer;  VAR (*INOUT*) position: CARDINAL;
                            BufferSize: CARDINAL;  VAR (*OUT*) result: ARRAY OF CHAR;
                            VAR (*INOUT*) NextChar: CHAR;  Stoppers: CharSet);

    (*     <namestring> ->  <string1> | " <string2> " | ' <string3> '       *)
    (*     <string1>   -> any string not including space char               *)
    (*     <string2>   -> any string not including double quote             *)
    (*     <string3>   -> any string not including single quote             *)
    (* The strings in <string1> also may not contain characters in Stoppers.*)

    CONST SingleQuote = "'";  DoubleQuote = '"';

    VAR Delimiter: CHAR;

    BEGIN
        INCL (Stoppers, Nul);
        IF (NextChar = SingleQuote) OR (NextChar = DoubleQuote) THEN
            Delimiter := NextChar;
            Stoppers := CharSet {Nul, Delimiter};
            NextChar := Scan (bufptr, position, BufferSize);
            LoadString (bufptr, position, BufferSize, result, NextChar, Stoppers);
            IF NextChar = Delimiter THEN
                NextChar := Scan (bufptr, position, BufferSize);
            END (*IF*);
        ELSE
            INCL (Stoppers, Space);
            LoadString (bufptr, position, BufferSize, result, NextChar, Stoppers);
        END (*IF*);
    END NameString;

(****************************************************************************)

PROCEDURE DirItem (bufptr: CharArrayPointer;  VAR (*INOUT*) position: CARDINAL;
                            BufferSize: CARDINAL;  mother: DirEntryPtr;
                         VAR (*INOUT*) NextChar: CHAR): DirEntryPtr;  FORWARD;

(****************************************************************************)

PROCEDURE DirList (bufptr: CharArrayPointer;  VAR (*INOUT*) position: CARDINAL;
                            BufferSize: CARDINAL;  mother: DirEntryPtr;
                                   VAR (*INOUT*) NextChar: CHAR): DirEntryPtr;

    (*     <dirlist>   ->  <diritem> { , <diritem> }*       *)
    (* Result returned: a linked list of directory nodes.   *)

    CONST Comma = ",";

    VAR result, lastnode: DirEntryPtr;

    BEGIN
        result := DirItem (bufptr, position, BufferSize, mother, NextChar);
        lastnode := result;
        WHILE NextChar = Comma DO
            NextChar := Scan (bufptr, position, BufferSize);
            lastnode^.next := DirItem (bufptr, position, BufferSize, mother, NextChar);
            lastnode := lastnode^.next;
        END (*WHILE*);
        RETURN result;
    END DirList;

(****************************************************************************)

PROCEDURE DirRule (bufptr: CharArrayPointer;  VAR (*INOUT*) position: CARDINAL;
                            BufferSize: CARDINAL;  pnode: DirEntryPtr;
                            VAR (*INOUT*) NextChar: CHAR);

    (* Fills in the permissions and subdirectory info for pnode.  *)
    (*     <dirrule>   -> { <permission> }* { ( <dirlist> ) }     *)
    (*     <permission> ->  V+ | V- | R+ | R- | W- | W+ | D- | D+ *)

    VAR option: Permission;

    BEGIN
        (* Default flags are inherited from parent. *)

        IF pnode^.parent = NIL THEN
            pnode^.flags := PermissionSet {Visible, AllowRead}
        ELSE pnode^.flags := pnode^.parent^.flags
        END (*IF*);

        (* Look for optional permission codes. *)

        WHILE CAP(NextChar) IN CharSet {'V', 'R', 'W', 'D'} DO
            CASE CAP(NextChar) OF
              | 'V':  option := Visible;
              | 'R':  option := AllowRead;
              | 'W':  option := AllowWrite;
              | 'D':  option := AllowDelete;
            END (*CASE*);
            NextChar := Scan (bufptr, position, BufferSize);
            IF NextChar = '-' THEN
                NextChar := Scan (bufptr, position, BufferSize);
                EXCL (pnode^.flags, option);
            ELSIF NextChar = '+' THEN
                NextChar := Scan (bufptr, position, BufferSize);
                INCL (pnode^.flags, option);
            END (*IF*);
        END (*WHILE*);

        (* Look for optional list of subdirectories. *)

        IF NextChar = '(' THEN
            NextChar := Scan (bufptr, position, BufferSize);
            pnode^.FirstChild := DirList (bufptr, position, BufferSize, pnode, NextChar);
            IF NextChar = ')' THEN
                NextChar := Scan (bufptr, position, BufferSize);
            END (*IF*);
        END (*IF*);

    END DirRule;

(****************************************************************************)

PROCEDURE DirItem (bufptr: CharArrayPointer;  VAR (*INOUT*) position: CARDINAL;
                            BufferSize: CARDINAL;  mother: DirEntryPtr;
                                VAR (*INOUT*) NextChar: CHAR): DirEntryPtr;

    (*     <diritem>   -> <namestring> <dirrule>                            *)

    VAR result: DirEntryPtr;

    BEGIN
        NEW (result);
        WITH result^ DO
            collapsed := FALSE;
            flags := PermissionSet {Visible, AllowRead};
            parent := mother;  FirstChild := NIL;  next := NIL;
            NameString (bufptr, position, BufferSize, name, NextChar, CharSet{});
        END (*WITH*);
        DirRule (bufptr, position, BufferSize, result, NextChar);
        RETURN result;
    END DirItem;

(************************************************************************)
(*         ENCODING THE VOLUME INFORMATION AS AN ASCII STRING           *)
(************************************************************************)

PROCEDURE AppendString (string: ARRAY OF CHAR;
                                       VAR (*INOUT*) result: ARRAY OF CHAR;
                                       VAR (*INOUT*) index: CARDINAL);

    (* Puts string into the result array, starting at result[index].    *)
    (* On return index has been updated to the next unused array slot.  *)

    VAR j: CARDINAL;

    BEGIN
        IF LENGTH(string) > 0 THEN
            FOR j := 0 TO LENGTH(string)-1 DO
                result[index] := string[j];  INC(index);
            END (*FOR*);
        END (*IF*);
    END AppendString;

(************************************************************************)

PROCEDURE AppendQuotedString (string: ARRAY OF CHAR;
                                       VAR (*INOUT*) result: ARRAY OF CHAR;
                                       VAR (*INOUT*) index: CARDINAL);

    (* Puts quotation marks around string, and puts it into the result  *)
    (* array, starting at result[index].  On return index has been      *)
    (* updated to the next unused array slot.                           *)

    BEGIN
        result[index] := '"';  INC(index);
        AppendString (string, result, index);
        result[index] := '"';  INC(index);
    END AppendQuotedString;

(************************************************************************)

PROCEDURE StorePermAndSubdir (D: DirEntryPtr;  DefaultFlags: PermissionSet;
                              VAR (*INOUT*) result: ARRAY OF CHAR;
                              VAR (*INOUT*) index: CARDINAL);

    (* Puts the flags for D, then information for all subdirectories    *)
    (* of D, into the result array, starting at result[index].          *)
    (* On return index has been updated to the next unused array slot.  *)

    TYPE CodeArray = ARRAY Permission OF CHAR;

    CONST PermissionCode = CodeArray {'V','R','W','D'};

    VAR child: DirEntryPtr;  perm: Permission;  difference: PermissionSet;

    BEGIN
        difference := DefaultFlags - D^.flags;
        FOR perm := MIN(Permission) TO MAX(Permission) DO
            IF perm IN difference THEN
                result[index] := PermissionCode[perm];  INC (index);
                result[index] := '-';  INC (index);
            END (*IF*);
        END (*FOR*);
        difference := D^.flags - DefaultFlags;
        FOR perm := MIN(Permission) TO MAX(Permission) DO
            IF perm IN difference THEN
                result[index] := PermissionCode[perm];  INC (index);
                result[index] := '+';  INC (index);
            END (*IF*);
        END (*FOR*);

        child := D^.FirstChild;
        IF child <> NIL THEN
            result[index] := '(';  INC(index);
            REPEAT
                AppendQuotedString (child^.name, result, index);
                StorePermAndSubdir (child, D^.flags, result, index);
                child := child^.next;
                IF child <> NIL THEN
                    result[index] := ',';  INC(index);
                END (*IF*);
            UNTIL child = NIL;
            result[index] := ')';  INC(index);
        END (*IF*);

    END StorePermAndSubdir;

(************************************************************************)

PROCEDURE SizeOfPermAndSubdir (D: DirEntryPtr;
                                   DefaultFlags: PermissionSet): CARDINAL;

    (* Works out how much space to allocate for the flags for D, plus   *)
    (* information for all subdirectories of D.                         *)

    VAR size: CARDINAL;  child: DirEntryPtr;
        perm: Permission;  difference: PermissionSet;

    BEGIN
        size := 0;
        difference := DefaultFlags / D^.flags;
        FOR perm := MIN(Permission) TO MAX(Permission) DO
            IF perm IN difference THEN
                INC (size, 2);
            END (*IF*);
        END (*FOR*);

        child := D^.FirstChild;
        IF child <> NIL THEN
            INC (size);
            REPEAT
                INC (size, 3 + LENGTH(child^.name)
                            + SizeOfPermAndSubdir (child, D^.flags));
                child := child^.next;
            UNTIL child = NIL;
        END (*IF*);

        RETURN size;

    END SizeOfPermAndSubdir;

(************************************************************************)

PROCEDURE SizeOfPermissionData (D: DirEntryPtr): CARDINAL;

    (* Returns the number of characters needed to store the INI file    *)
    (* data for this directory.                                         *)

    BEGIN
        IF D = NIL THEN
            RETURN 0;
        ELSE
            RETURN SizeOfPermAndSubdir (D, PermissionSet {Visible, AllowRead});
        END (*IF*);
    END SizeOfPermissionData;

(************************************************************************)

PROCEDURE StorePermissionData (D: DirEntryPtr;
                                  VAR (*INOUT*) result: ARRAY OF CHAR;
                                  VAR (*INOUT*) index: CARDINAL);

    (* Puts the flags for D, then information for all subdirectories    *)
    (* of D, into the result array, starting at result[index].          *)
    (* On return index has been updated to the next unused array slot.  *)

    BEGIN
        IF D <> NIL THEN
            StorePermAndSubdir (D, PermissionSet {Visible, AllowRead},
                                                 result, index);
        END (*IF*);
    END StorePermissionData;

(********************************************************************************)
(*                        DISCARDING A USER RECORD                              *)
(********************************************************************************)

PROCEDURE KillList (VAR (*INOUT*) p: DirEntryPtr);

    (* Discards a directory tree.  *)

    VAR q: DirEntryPtr;

    BEGIN
        WHILE p <> NIL DO
            KillList (p^.FirstChild);
            q := p^.next;
            DISPOSE (p);
            p := q;
        END (*IF*);
    END KillList;

(********************************************************************************)
(*                           STRING COMPARISON                                  *)
(********************************************************************************)

PROCEDURE NameMatch (first, second: ARRAY OF CHAR): BOOLEAN;

    (* Equality test, modulo upper/lower case differences.  That is, the result *)
    (* is TRUE even if the letters in first and second don't have the same      *)
    (* case, as long as the strings are otherwise equal.                        *)

    VAR k: CARDINAL;

    BEGIN
        k := 0;
        LOOP
            IF k > HIGH(first) THEN RETURN (k > HIGH(second));
            ELSIF first[k] = Nul THEN RETURN (second[k] = Nul);
            ELSIF CAP(first[k]) <> CAP(second[k]) THEN RETURN FALSE;
            ELSE INC(k);
            END (*IF*);
        END (*LOOP*);
    END NameMatch;

(************************************************************************)
(*                   "Not yet implemented" message                      *)
(************************************************************************)

(*
PROCEDURE NotImplemented;

    VAR w: Window;

    BEGIN
        OpenWindow (w, yellow, red, 12, 15, 15, 64, noframe, nodivider);
        WriteLn (w);
        WriteString (w, " Sorry, that feature is not yet implemented");
        PressAnyKey (w);
        CloseWindow (w);
    END NotImplemented;
*)

(********************************************************************************)
(*                       UPDATING DIRECTORY INFORMATION                         *)
(********************************************************************************)

PROCEDURE PruneTree (VAR (*INOUT*) p: DirEntryPtr;  defaults: PermissionSet);

    (* Removes the tree nodes that aren't necessary because their permissions   *)
    (* are directly inherited from their parents.                               *)

    VAR previous, current, next: DirEntryPtr;

    BEGIN
        previous := NIL;  current := p;
        WHILE current <> NIL DO
            next := current^.next;
            PruneTree (current^.FirstChild, current^.flags);
            IF (current^.FirstChild = NIL) AND (current^.flags = defaults) THEN
                IF previous = NIL THEN p := next;
                ELSE previous^.next := next;
                END (*IF*);
                DISPOSE (current);
            ELSE
                previous := current;
            END (*IF*);
            current := next;
        END (*WHILE*);
    END PruneTree;

(********************************************************************************)

PROCEDURE CleanUpTree (VAR (*INOUT*) root: DirEntryPtr);

    (* Removes the tree nodes that aren't necessary because their permissions   *)
    (* are directly inherited from their parents.                               *)

    BEGIN
        PruneTree (root, PermissionSet {Visible, AllowRead} );
    END CleanUpTree;

(********************************************************************************)

PROCEDURE CopyPermissions (VAR (*INOUT*) src: DirEntryPtr;  dest: DirEntryPtr;
                                             defaultflags: PermissionSet);

    (* Takes the permission information from src and inserts it into the dest   *)
    (* tree.  In the process the src tree is destroyed.                         *)

    VAR this, current, previous: DirEntryPtr;

    BEGIN
        this := dest;
        WHILE this <> NIL DO

            (* Search for a src entry that matches this entry. *)

            current := src;  previous := NIL;
            WHILE (current <> NIL) AND NOT NameMatch (current^.name, this^.name) DO
                previous := current;  current := current^.next;
            END (*WHILE*);
            IF current <> NIL THEN

                (* Match found.  Copy the information, then destroy the *)
                (* current tree.                                        *)

                this^.flags := current^.flags;
                IF previous = NIL THEN
                    src := current^.next;
                ELSE
                    previous^.next := current^.next;
                END (*IF*);
                current^.next := NIL;
                CopyPermissions (current^.FirstChild, this^.FirstChild, this^.flags);
                KillList (current);
            ELSE

                (* No match with any src directory, so use the  *)
                (* default flag values.                         *)

                this^.flags := defaultflags;
                CopyPermissions (current, this^.FirstChild, defaultflags);
            END (*IF*);
            this := this^.next;

        END (*WHILE*);
        KillList (src);

    END CopyPermissions;

(********************************************************************************)

PROCEDURE CopyDownPermissions (thisnode: DirEntryPtr);

    (* All the children of thisnode inherit its permissions. *)

    VAR current: DirEntryPtr;

    BEGIN
        current := thisnode^.FirstChild;
        WHILE current <> NIL DO
            current^.flags := thisnode^.flags;
            CopyDownPermissions (current);
            current := current^.next;
        END (*WHILE*);
    END CopyDownPermissions;

(********************************************************************************)

PROCEDURE BuildSubdirectoryStructure (father: DirEntryPtr;
                                        fathername: ARRAY OF CHAR): DirEntryPtr;

    (* From the given directory father, builds a DirEntry structure         *)
    (* holding all the subdirectories of father.  (This also includes all   *)
    (* the lower-level subdirectories.)  The existing nodes below father,   *)
    (* if any, are left untouched.                                          *)

    VAR D: DirectoryEntry;  q, previous, current: DirEntryPtr;
        SearchString, MyFullName: FileNameString;
        MoreToGo: BOOLEAN;

    BEGIN

        Strings.Assign (fathername, SearchString);
        Strings.Append ('*', SearchString);
        q := NIL;  previous := NIL;
        MoreToGo := FirstDirEntry (SearchString, TRUE, D);
        WHILE MoreToGo DO
            IF (D.name[0] = '.') AND (D.name[1]=Nul)
                                      OR ((D.name[1]='.') AND (D.name[2] = Nul)) THEN

                (* Ignore . and .. entries *)

            ELSIF directory IN D.attr THEN
                NEW (current);
                WITH current^ DO
                    parent := father;
                    FirstChild := NIL;
                    next := NIL;
                    Strings.Assign (D.name, name);
                END (*WITH*);
                IF previous = NIL THEN
                    q := current;
                ELSE
                    previous^.next := current;
                END (*IF*);
                previous := current;
            END (*IF*);
            MoreToGo := NextDirEntry (D);

        END (*WHILE*);
        DirSearchDone (D);

        (* We've found all the immediate subdirectories of parentname. *)
        (* Now traverse the list to fill in the lower-level directory  *)
        (* records.                                                    *)

        current := q;
        WHILE current <> NIL DO
            Strings.Assign (fathername, MyFullName);
            Strings.Append (current^.name, MyFullName);
            Strings.Append ('/', MyFullName);
            current^.FirstChild := BuildSubdirectoryStructure (current, MyFullName);
            current := current^.next;
        END (*WHILE*);

        RETURN q;

    END BuildSubdirectoryStructure;

(********************************************************************************)

PROCEDURE UpdateDirectoryList (parent: DirEntryPtr;  parentname: ARRAY OF CHAR);

    (* On entry parentname contains the full name of a directory, and parent    *)
    (* is the root of the DirEntry tree; there might be some nodes missing from *)
    (* this tree.  This procedure rebuilds the list so that all subdirectories  *)
    (* of parentname (and their subdirectories as well) are included.           *)

    VAR q: DirEntryPtr;

    BEGIN
        q := BuildSubdirectoryStructure (parent, parentname);
        CopyPermissions (parent^.FirstChild, q, parent^.flags);
        parent^.FirstChild := q;
    END UpdateDirectoryList;

(********************************************************************************)

PROCEDURE UpdateDirectoryTree (rootname: ARRAY OF CHAR;  VAR (*INOUT*) p: DirEntryPtr);

    (* Rebuilds a directory tree. *)

    VAR D: DirectoryEntry;  RootIsADirectory: BOOLEAN;  j: CARDINAL;

    BEGIN
        (*j := LENGTH (rootname);*)

        (* The LENGTH call seems to be causing a crash, so let's do this *)
        (* calculation a more primitive way.                             *)

        j := 0;
        LOOP
            IF j > HIGH(rootname) THEN EXIT(*LOOP*)
            ELSIF rootname[j] = Nul THEN EXIT(*LOOP*);
            ELSE
                INC (j);
            END (*IF*);
        END (*LOOP*);

        IF j = 0 THEN
            RootIsADirectory := FALSE;
        ELSE
            DEC (j);
            IF (rootname[j] = '\') OR (rootname[j] = '/') THEN
                rootname[j] := Nul;
            ELSE
                INC(j);
            END (*IF*);

            IF (j = 0) OR (rootname[j-1] = ':') THEN
                RootIsADirectory := TRUE;
            ELSE
                RootIsADirectory := FirstDirEntry (rootname, TRUE, D)
                                                           AND (directory IN D.attr);
                DirSearchDone (D);
            END (*IF*);
            rootname[j] := '/';  rootname[j+1] := Nul;
        END (*IF*);
        IF RootIsADirectory THEN
            IF p = NIL THEN
                NEW (p);
                WITH p^ DO
                    flags := PermissionSet {Visible, AllowRead};
                    parent := NIL;  FirstChild := NIL;  next := NIL;
                    name := "/";
                END (*WITH*);
            END (*IF*);
            UpdateDirectoryList (p, rootname);
        ELSE
            p := NIL;
        END (*IF*);
    END UpdateDirectoryTree;

(********************************************************************************)

PROCEDURE InsertThreadsRecursive (p, leftptr, rightptr: DirEntryPtr): DirEntryPtr;

    (* On entry p is pointing to a directory tree.  This procedure adds left    *)
    (* and right threads to turn it into a threaded tree.  The function result  *)
    (* is a pointer to the node at the end of the thread.                       *)

    VAR nextright, tail: DirEntryPtr;

    BEGIN
        p^.left := leftptr;
        IF p^.next = NIL THEN
            nextright := rightptr;
        ELSE
            nextright := p^.next;
        END (*IF*);
        IF p^.FirstChild = NIL THEN
            p^.right := nextright;
            tail := p;
        ELSE
            p^.right := p^.FirstChild;
            tail := InsertThreadsRecursive (p^.FirstChild, p, nextright);
        END (*IF*);
        IF p^.next <> NIL THEN
            tail := InsertThreadsRecursive (p^.next, tail, rightptr);
        END (*IF*);
        RETURN tail;
    END InsertThreadsRecursive;

(********************************************************************************)

PROCEDURE InsertAltThreads (p, rightptr: DirEntryPtr);

    (* Like InsertThreadsRecursive, but fills in the altright threads.  The     *)
    (* altright thread is the thread to follow if we want to skip over the      *)
    (* subtrees of the current node.  There is no similar "altleft" concept;    *)
    (* the left threads have to be recomputed dynamically.                      *)

    BEGIN
        p^.collapsed := FALSE;
        IF p^.next = NIL THEN
            p^.altright := rightptr;
        ELSE
            p^.altright := p^.next;
            InsertAltThreads (p^.next, rightptr);
        END (*IF*);
        IF p^.FirstChild <> NIL THEN
            InsertAltThreads (p^.FirstChild, p^.altright);
        END (*IF*);
    END InsertAltThreads;

(********************************************************************************)

PROCEDURE InsertThreads (p: DirEntryPtr);

    (* On entry p is pointing to a directory tree.  This procedure adds left    *)
    (* and right threads to turn it into a threaded tree.                       *)

    BEGIN
        IF p <> NIL THEN
            InsertAltThreads (p, NIL);
            EVAL (InsertThreadsRecursive (p, NIL, NIL));
        END (*IF*);
    END InsertThreads;

(********************************************************************************)
(*                       EDITING THE DIRECTORY PERMISSIONS                      *)
(********************************************************************************)

PROCEDURE PermissionsToString (flags: PermissionSet;
                                 VAR (*OUT*) result: ARRAY OF CHAR);

    (* Turns the flags into a four-character string.  We assume that result is  *)
    (* at least four characters long, and we don't add a string terminator.     *)

    BEGIN
        IF Visible IN flags THEN result[0] := 'V' ELSE result[0] := ' ' END(*IF*);
        IF AllowRead IN flags THEN result[1] := 'R' ELSE result[1] := ' ' END(*IF*);
        IF AllowWrite IN flags THEN result[2] := 'W' ELSE result[2] := ' ' END(*IF*);
        IF AllowDelete IN flags THEN result[3] := 'D' ELSE result[3] := ' ' END(*IF*);
    END PermissionsToString;

(********************************************************************************)

PROCEDURE DirNameToText (p: DirEntryPtr;  VAR (*OUT*) Entry: ARRAY OF CHAR);

    (* Creates a text string containing the permissions, collapsed marker, and  *)
    (* directory name for this node.                                            *)

    VAR j: CARDINAL;

    BEGIN
        PermissionsToString (p^.flags, Entry);
        FOR j := 4 TO p^.indent+4 DO
            IF j <= MAX(FileNameSubscript) THEN
                Entry[j] := ' ';
            END (*IF*);
        END (*FOR*);
        IF p^.collapsed THEN
            Entry[5] := '+';
        END (*IF*);
        IF p^.indent+5 <= HIGH(Entry) THEN
            Entry[p^.indent+5] := Nul;
        END (*IF*);
        Strings.Append (p^.name, Entry);
    END DirNameToText;

(********************************************************************************)

PROCEDURE EditPermissionListBox (root: DirEntryPtr;  w: Window;  LB: ListBox;
                                                   BoxHeight: CARDINAL): ExitCode;

    VAR current: DirEntryPtr;
        RowBuffer: FileNameString;

    (****************************************************************************)

    PROCEDURE RedisplaySubtree;

        (* Rebuilds the listbox contents, for the subtree headed by current.    *)
        (* On exit current is the first entry beyond that subtree in the        *)
        (* listbox - unless we reach the end of the list, in which case         *)
        (* current is the last entry.                                           *)

        VAR child: DirEntryPtr;

        BEGIN
            child := current^.FirstChild;
            IF current^.collapsed OR (child = NIL) THEN
                IF current^.right <> NIL THEN
                    current := current^.right;
                    EVAL (CursorForward(LB));
                END (*IF*);
            ELSE
                EVAL (CursorForward(LB));
                WHILE child <> NIL DO
                    current := child;
                    DirNameToText (child, RowBuffer);
                    ReplaceCurrent (LB, RowBuffer);
                    RedisplaySubtree;
                    child := child^.next;
                END (*WHILE*);
            END (*IF*);
        END RedisplaySubtree;

    (****************************************************************************)

    PROCEDURE RemoveSubtree;

        (* Takes the subtrees of current out of the listbox display, and        *)
        (* readjusts left and right thread pointers.                            *)

        (************************************************************************)

        PROCEDURE WipeChildren (node: DirEntryPtr);

            (* Takes the subtrees of node out of the listbox display. *)

            VAR child: DirEntryPtr;

            BEGIN
                child := node^.FirstChild;
                WHILE child <> NIL DO
                    LBDeleteCurrent (LB);
                    IF NOT child^.collapsed THEN
                        WipeChildren (child);
                    END (*IF*);
                    child := child^.next;
                END (*WHILE*);
            END WipeChildren;

        (************************************************************************)

        VAR temp: DirEntryPtr;

        BEGIN
            temp := current^.right;  current^.right := current^.altright;
            current^.altright := temp;
            IF current^.right <> NIL THEN
                current^.right^.left := current;
            END (*IF*);
            EVAL (CursorForward (LB));
            WipeChildren (current);
            IF current^.right <> NIL THEN
                current := current^.right;
            END (*IF*);
        END RemoveSubtree;

    (****************************************************************************)

    PROCEDURE ReinsertSubtree;

        (* Puts the subtrees of current back into the listbox display, and      *)
        (* readjusts left and right thread pointers.                            *)

        (************************************************************************)

        PROCEDURE ReinstateChildren (node: DirEntryPtr): DirEntryPtr;

            (* Puts the subtrees of node back into the listbox display. *)
            (* Returns a pointer to the last non-collapsed node.        *)

            VAR child, tail: DirEntryPtr;

            BEGIN
                child := node^.FirstChild;  tail := child;
                WHILE child <> NIL DO
                    DirNameToText (child, RowBuffer);
                    LBInsertAfter (LB, RowBuffer);
                    IF child^.collapsed OR (child^.FirstChild = NIL) THEN
                        tail := child;
                    ELSE
                        tail := ReinstateChildren (child);
                    END (*IF*);
                    child := child^.next;
                END (*WHILE*);
                RETURN tail;
            END ReinstateChildren;

        (************************************************************************)

        VAR treeatright, tail: DirEntryPtr;

        BEGIN
            treeatright := current^.right;  current^.right := current^.altright;
            current^.altright := treeatright;
            tail := ReinstateChildren (current);
            IF treeatright <> NIL THEN
                treeatright^.left := tail;
            END (*IF*);
            current := tail;
        END ReinsertSubtree;

    (****************************************************************************)

    PROCEDURE StepBackward(): BOOLEAN;

        (* Moves one step back in the list, or returns FALSE if this would      *)
        (* take us off the top.                                                 *)

        BEGIN
            IF CursorBackward (LB) THEN
                current := current^.left;
                RETURN TRUE;
            ELSE
                RETURN FALSE;
            END (*IF*);
        END StepBackward;

    (****************************************************************************)

    PROCEDURE StepForward;

        (* Moves one step forward in the list, if possible. *)

        BEGIN
            EVAL (CursorForward (LB));
            IF current^.right <> NIL THEN
                current := current^.right;
            END (*IF*);
        END StepForward;

    (****************************************************************************)

    VAR bottombar: Window;  ch: CHAR;
        result: ExitCode;
        perm: Permission;
        Screen: VirtualScreen;
        j: CARDINAL;

    BEGIN                                   (* Body of EditPermissionListBox *)
        Screen := VirtualScreenOf (w);
        IF root = NIL THEN
            OpenWindowHidden (bottombar, yellow, red, 14, 15,
                  23, 53, noframe, nodivider);
            MapToVirtualScreen (bottombar, Screen);
            WriteString (bottombar, "Home directory does not exist.");
            PressAnyKey (bottombar);
            CloseWindow (bottombar);
            RETURN CursorOffTop;
        END (*IF*);

        OpenWindowHidden (bottombar, yellow, red, ScreenRows-1, ScreenRows-1, 0, 79, noframe, nodivider);
        MapToVirtualScreen (bottombar, Screen);
        WriteString (bottombar, " VRWD (toggle)  I inherit  P propagate  + expand  - collapse  X exit  ");

        HighlightOn (LB);
        current := root;
        LOOP
            ch := GetKey(w);
            IF ch = CHR(0) THEN
                ch := GetKey(w);
                IF ch = "H" THEN                        (* cursor up *)
                    IF NOT StepBackward() THEN
                        result := CursorOffTop;
                        EXIT (*LOOP*);
                    END (*IF*);
                ELSIF ch = "P" THEN                     (* cursor down *)
                    StepForward;
                ELSIF ch = "G" THEN                     (* home *)
                    DisableScreenOutput (LB);
                    WHILE StepBackward() DO
                    END (*WHILE*);
                    Repaint (LB);
                ELSIF ch = "O" THEN                     (* end *)
                    DisableScreenOutput (LB);
                    WHILE current^.right <> NIL DO
                        StepForward;
                    END (*WHILE*);
                    Repaint (LB);
                ELSIF ch = "I" THEN                     (* page up *)
                    DisableScreenOutput (LB);
                    FOR j := 1 TO BoxHeight-1 DO
                        EVAL (StepBackward());
                    END (*FOR*);
                    Repaint (LB);
                ELSIF ch = "Q" THEN                     (* page down *)
                    DisableScreenOutput (LB);
                    FOR j := 1 TO BoxHeight-1 DO
                        StepForward;
                    END (*FOR*);
                    Repaint (LB);
                END (*IF*);
            ELSIF CAP(ch) IN CharSet {'V','R','W','D'} THEN
                perm := CharToPermission (ch);
                IF perm IN current^.flags THEN EXCL(current^.flags, perm)
                ELSE INCL(current^.flags, perm);
                END (*IF*);
                DirNameToText (current, RowBuffer);
                ReplaceCurrent (LB, RowBuffer);
            ELSIF CAP(ch) = 'I' THEN                    (* I = inherit *)
                IF current^.parent = NIL THEN
                    current^.flags := PermissionSet {Visible, AllowRead}
                ELSE
                    current^.flags := current^.parent^.flags;
                END (*IF*);
                DirNameToText (current, RowBuffer);
                ReplaceCurrent (LB, RowBuffer);
            ELSIF CAP(ch) = 'P' THEN                    (* P = propagate *)
                CopyDownPermissions (current);
                RedisplaySubtree;
            ELSIF CAP(ch) = '+' THEN                    (* + = expand *)
                IF current^.collapsed THEN
                    current^.collapsed := FALSE;
                    DirNameToText (current, RowBuffer);
                    DisableScreenOutput (LB);
                    ReplaceCurrent (LB, RowBuffer);
                    ReinsertSubtree;
                    Repaint (LB);
                END (*IF*);
                StepForward;
            ELSIF CAP(ch) = '-' THEN                    (* - = collapse *)
                IF (current^.FirstChild <> NIL)
                           AND NOT current^.collapsed THEN
                    current^.collapsed := TRUE;
                    DirNameToText (current, RowBuffer);
                    DisableScreenOutput (LB);
                    ReplaceCurrent (LB, RowBuffer);
                    RemoveSubtree;
                    Repaint (LB);
                ELSE
                    StepForward;
                END (*IF*);
            ELSIF CAP(ch) = 'X' THEN                    (* X = exit *)
                result := TrueExit;
                EXIT (*LOOP*);
            END (*IF*);

        END (*LOOP*);
        HighlightOff (LB);
        CloseWindow (bottombar);
        RETURN result;
    END EditPermissionListBox;

(********************************************************************************)

PROCEDURE ShowTree (LB: ListBox;  p: DirEntryPtr;  indent: CARDINAL);

    VAR Entry: FileNameString;

    BEGIN
        IF p <> NIL THEN
            p^.indent := indent;
            DirNameToText (p, Entry);
            LBAppend (LB, Entry);
            p := p^.FirstChild;
            WHILE p <> NIL DO
                ShowTree (LB, p, indent+3);
                p := p^.next;
            END (*WHILE*);
        END (*IF*);
    END ShowTree;

(********************************************************************************)
(*                              INITIALISATION                                  *)
(********************************************************************************)

VAR dummy: CARDINAL;

BEGIN
    GetScreenSize (ScreenRows, dummy);
END PermissionEditor.

