/*************************************************
*     Exim - an Internet mail transport agent    *
*************************************************/

/* Copyright (c) University of Cambridge 1995 - 1997 */
/* See the file NOTICE for conditions of use and distribution. */

/* Functions concerned with verifying things. */


#include "exim.h"



/*************************************************
*           Local static variables               *
*************************************************/


static char ident_buffer[128];
static BOOL sender_ok;
static BOOL sender_is_local;

/* Values for sender_ok */

#define SENDER_OK_OK          0    /* really and truly OK */
#define SENDER_OK_EXCEPTION   1    /* OK by virtue of the exception list */
#define SENDER_OK_WARNING     2    /* OK by virtue of being in warning mode */
#define SENDER_OK_DEFER       3    /* check didn't complete */
#define SENDER_OK_NOT         4    /* Not OK */



/*************************************************
*            Verify an email address             *
*************************************************/

/* The local flag indicates whether an unqualified address is acceptable or
not. The local_domain flag is set TRUE if the address turns out to be in a
local domain, FALSE if it is in a remote domain. If the file is NULL, don't
print anything. If newaddress is not null, return the address, possibly
modified by the routing process. The yield is OK, FAIL, or DEFER. If debugging,
output fuller information, provided there is an output file. If log_details is
TRUE, write details of failures to the reject log. If address_test is true,
don't skip directors and routers that have no_verify set - we are doing a test
rather than a verify.

Arguments:
  s               address to verify
  f               if not NULL, write the result to this file
  local_domain    set TRUE if address turns out to be in a local domain
  newaddress      if not NULL, return the top-level address, possibly modified
  options         various option bits:
                    vopt_is_recipient => this is a recipient address, otherwise
                      it's a sender address - this affects qualification and
                      rewriting
                    vopt_local => qualify an unqualified address; else error
                    vopt_log_details => write details of failures to the
                      reject log
                    vopt_address_test => testing an address (-bt) rather than
                      verifying
                    vopt_expn => called from SMTP EXPN command

Returns:          OK
                  FAIL
                  DEFER
*/

int
verify_address(char *s, FILE *f, BOOL *local_domain, char **newaddress,
  int options)
{
int start, end, domain;
BOOL allok = TRUE;
BOOL full_info = (f == NULL)? FALSE : (debug_level > 0);
BOOL is_recipient = (options & vopt_is_recipient) != 0;
BOOL log_details  = (options & vopt_log_details) != 0;
BOOL address_test = (options & vopt_address_test) != 0;
BOOL expn         = (options & vopt_expn) != 0;
int yield = OK;
int verify_type = expn? v_expn :
          address_test? v_none :
          is_recipient? v_recipient : v_sender;
address_item *addr_orig;
address_item *addr_new;
address_item *addr_remote = NULL;
address_item *addr_local = NULL;
address_item *addr_succeed = NULL;
char *ko_prefix, *cr;
char *errmess;

char *receiver =
  parse_extract_address(s, &errmess, &start, &end, &domain, FALSE);

if (expn)
  {
  ko_prefix = "553 ";
  cr = "\r";
  }
else ko_prefix = cr = "";

if (receiver == NULL)
  {
  if (f != NULL)
    fprintf(f, "%s%s - bad address: %s%s\n", ko_prefix, s, errmess, cr);
  if (log_details)
    log_write(0, LOG_REJECT, "%s - bad address: %s", s, errmess);
  return FAIL;
  }

/* Add qualify domain if permitted. */

if (parse_find_at(receiver) == NULL)
  {
  if ((options & vopt_local) == 0)
    {
    if (f != NULL)
      fprintf(f, "%sA domain is required for \"%s\"%s\n", ko_prefix, receiver,
        cr);
    if (log_details)
      log_write(0, LOG_REJECT, "a domain is required for \"%s\"", receiver);
    return FAIL;
    }
  receiver = rewrite_address_qualify(receiver, is_recipient);
  }

/* Rewrite and report on it. It is guaranteed that the address of the new
string will be different to the old, even though the old one is freed in the
process. */

if (rewrite_rules != NULL)
  {
  char *old = receiver;
  receiver = rewrite_address(receiver, is_recipient, FALSE);
  if (receiver != old)
    {
    if (f != NULL && !expn) fprintf(f, "Address rewritten as: %s\n", receiver);
    if (log_details)
      log_write(0, LOG_REJECT, "%s rewritten as: %s", old, receiver);
    }
  }

/* Set up an initial address structure. */

addr_new = addr_orig = deliver_make_addr(receiver);

/* We need a loop, since a directed address might generate a number of new
addresses. We must also cope with generated pipes and files at the top
level. (See also the code/comment in deliver.c.) However, it is usually
the case that the forwardfile director has its verify flag turned off.

The loop is used after directing, however, only when the verify_actions flag is
set, and this can only be set locally. Remote enquiries just get information
about the top level address, not anything that it generated.

In the case of a router discovering that an apparently remote address is in
fact local, the loop is always re-run. */

while (addr_new != NULL)
  {
  int rc;
  address_item *addr = addr_new;
  addr_new = addr->next;
  addr->next = NULL;

  /* Handle generated pipe, file or reply addresses. We don't get these
  when handling EXPN, as it does only one level of expansion. */

  if (addr->pfr)
    {
    allok = FALSE;
    if (addr->orig[0] == '|')
      {
      if (f != NULL)
        fprintf(f, "%s -> %s %s\n", addr->parent->orig,
          addr->orig, addr->allow_pipe? "" : "*** forbidden ***");
      continue;
      }
    else if (addr->orig[0] == '/')
      {
      if (f != NULL)
        fprintf(f, "%s -> %s %s\n", addr->parent->orig,
          addr->orig, addr->allow_file? "" : "*** forbidden ***");
      continue;
      }
    else if (addr->orig[0] == '>')
      {
      if (f != NULL)
        fprintf(f, "%s -> mail %s %s\n", addr->parent->orig,
          addr->orig+1, addr->allow_reply? "" : "*** forbidden ***");
      continue;
      }
    }

  /* All addresses should either have been made fully qualified above,
  or been qualified when generated by a director, so panic if we find
  an unqualified one. */

  if (parse_find_at(addr->orig) == NULL)
    log_write(0, LOG_PANIC_DIE, "Unqualified address found: %s", addr->orig);

  /* Determine locality - this sets "local_part", "domain", and "local"
  fields. Pass back the locality if a variable is supplied. */

  deliver_setlocal(addr);

  if (local_domain != NULL) *local_domain = addr->local;

  /* If we are handing an EXPN command and this is the original address,
  and it is not local, we don't proceed any further. (Actually, if it's
  expn, this will always be the original address, since we don't process
  generated addresses any more.) */

  if (expn && addr == addr_orig && !addr->local)
    {
    fprintf(f, "550 Not a local domain\r\n");
    return FALSE;
    }

  /* DEBUG and/or log_details: show what's been done to this address */

  DEBUG(7)
    {
    debug_printf("address %s\n", addr->orig);
    debug_printf("  local_part=%s domain=%s local=%d\n",
      addr->local_part, addr->domain, addr->local);
    }

  if (log_details)
    log_write(0, LOG_REJECT, "Verifying %s: local part = %s domain = %s "
      "local domain = %s",
      addr->orig, addr->local_part, addr->domain,
      (addr->local)? "true":"false");

  /* Handle a local address with the directors, or a remote address with the
  routers, and output or return the result except when full_info is set, in
  which case continue for other (generated) addresses. When full_info is set,
  f will not be NULL. Don't output anything for success in that case at this
  stage. Note that a director may set up local or remote delivery. */

  rc = (addr->local)?
    direct_address(addr, &addr_local, &addr_remote, &addr_new, &addr_succeed,
      verify_type) :
    route_address(addr, &addr_local, &addr_remote, &addr_new, verify_type);

  /* If a remote address turned out to be local after all, set it up for
    reprocessing and restart the loop. */

  if (rc == ISLOCAL)
    {
    if (log_details)
      log_write(0, LOG_REJECT, "routing %s caused it to become local",
        addr->orig);
    if (addr->local_part[0] == ',' || addr->local_part[0] == ':')
      addr->orig = string_sprintf("@%s%s", addr->domain, addr->local_part);
    else
      addr->orig = string_sprintf("%s@%s", addr->local_part, addr->domain);
    addr->next = addr_new;
    addr_new = addr;
    continue;
    }

  /* Handle hard failures */

  if (rc == FAIL)
    {
    allok = FALSE;
    if (f != NULL) fprintf(f, "%s%s %s:%s%s%s%s%s\n", ko_prefix,
      addr->orig,
      address_test? "is undeliverable" : "failed to verify",
      (addr->basic_errno <= 0)? "" : strerror(addr->basic_errno),
      (addr->basic_errno <= 0)? "" : ": ",
      expn? " " : "\n  ",
      (addr->message != NULL)? addr->message :
        (addr->basic_errno <= 0)? "unknown error" : "", cr);

    if (log_details) log_write(0, LOG_REJECT,
      "%s is undeliverable: %s%s%s", addr->orig,
      (addr->basic_errno <= 0)? "" : strerror(addr->basic_errno),
      (addr->basic_errno <= 0)? "" : ": ",
      (addr->message != NULL)? addr->message :
        (addr->basic_errno <= 0)? "unknown error" : "");

    if (!full_info) return FAIL; else yield = FAIL;
    }

  /* If the yield is ERROR or PANIC, there has been some cock-up in the
  directors or routers. This doesn't really mean the address is undeliverable
  or unverifyable, so we treat it the same as DEFER. */

  else if (rc == DEFER || rc == ERROR || rc == PANIC)
    {
    allok = FALSE;
    if (f != NULL)
      fprintf(f, "%s%s cannot be resolved at this time:%s%s%s%s%s\n",
      ko_prefix,
      addr->orig,
      expn? " " : "\n  ",
      (addr->basic_errno <= 0)? "" : strerror(addr->basic_errno),
      (addr->basic_errno <= 0)? "" : ": ",
      (addr->message != NULL)? addr->message :
        (addr->basic_errno <= 0)? "unknown error" : "", cr);

    if (log_details)
      log_write(0, LOG_REJECT, "%s cannot be resolved at this time: %s%s%s",
      addr->orig,
      (addr->basic_errno <= 0)? "" : strerror(addr->basic_errno),
      (addr->basic_errno <= 0)? "" : ": ",
      (addr->message != NULL)? addr->message :
        (addr->basic_errno <= 0)? "unknown error" : "");

    if (!full_info) return DEFER;
      else if (yield == OK) yield = DEFER;
    }

  /* If we are handling EXPN, we do not want to continue to route beyond
  the top level. Output what has been generated. */

  else if (expn)
    {
    char *ok_prefix = "250-";
    if (addr_new == NULL)
      {
      if (addr_local == NULL)
        fprintf(f, "250 mail to %s is discarded\r\n", s);
      else
        fprintf(f, "250 %s\r\n", addr_local->orig);
      }
    else while (addr_new != NULL)
      {
      address_item *addr = addr_new;
      addr_new = addr->next;
      if (addr_new == NULL) ok_prefix = "250 ";
      fprintf(f, "%s%s\r\n", ok_prefix, addr->orig);
      }
    return OK;
    }

  /* Handle successful routing or directing when short info wanted */

  else if (!full_info)
    {
    if (f != NULL) fprintf(f, "%s %s\n",
      addr->orig,
      address_test? "is deliverable" : "verified");
    if (newaddress != NULL)
      {
      char *prefix = (addr->prefix == NULL)? "" : addr->prefix;
      char *suffix = (addr->suffix == NULL)? "" : addr->suffix;
      *newaddress = (addr->local_part[0] == ',' || addr->local_part[0] == ':')?
         string_sprintf("@%s%s", addr->domain, addr->local_part) :
         string_sprintf("%s%s%s@%s", prefix, addr->local_part,
           suffix, addr->domain);
      }
    return OK;
    }

  }

/* Display the full results of the successful directing and routing,
including any generated addresses. Control gets here only when full_info is
set, which requires f not to be NULL, and this occurs only when a top-level
verify is called with the debugging switch on, or as a result of an SMTP EXPN
command (permitted only for specified hosts/nets). For verification, if the
debugging value is greater than 1, show everything; for 1 (which is also -v)
show what "the normal user" might be interested in. For address testing,
always show everything.

If there are no local and no remote addresses, and there were no pipes, files,
or autoreplies, and there were no errors or deferments, the message is to be
discarded, usually because of the use of :blackhole: in an alias file. */

if (allok && addr_local == NULL && addr_remote == NULL)
  fprintf(f, "mail to %s is discarded\n", s);

full_info = debug_level > 1 || address_test;

while (addr_local != NULL)
  {
  address_item *addr = addr_local;
  address_item *p = addr->parent;
  addr_local = addr->next;

  fprintf(f, "%s", addr->orig);

  while (p != NULL)
    {
    fprintf(f, "\n    <-- %s", p->orig);
    p = p->parent;
    }
  fprintf(f, "\n  local delivery to %s in domain %s\n",
    addr->local_part, addr->domain);
  if (full_info)
    {
    if (addr->director != NULL) fprintf(f, "  director = %s, ",
      addr->director->name);
    if (addr->router != NULL) fprintf(f, "  router = %s, ", addr->router->name);
    fprintf(f, "transport = %s\n", addr->transport->name);
    }
  }

while (addr_remote != NULL)
  {
  address_item *addr = addr_remote;
  address_item *p = addr->parent;
  addr_remote = addr->next;

  fprintf(f, "%s", addr->orig);

  while (p != NULL)
    {
    fprintf(f, "\n    <-- %s", p->orig);
    p = p->parent;
    }

  /* Show director or router, and transport */

  if (addr->director != NULL)
    {
    fprintf(f, "\n  remote delivery to %s in domain %s\n",
      addr->local_part, addr->domain);
    if (full_info) fprintf(f, "  director = %s, ", addr->director->name);
    }

  else if (addr->router != NULL)
    {
    if (addr->local_part[0] == ',' || addr->local_part[0] == ':')
      fprintf(f, "\n  remote delivery to @%s%s", addr->domain,
      addr->local_part);
    else
      fprintf(f, "\n  remote delivery to %s@%s", addr->local_part,
      addr->domain);

    if (strcmp(addr->domain, addr->route_domain) != 0)
      fprintf(f, " via domain %s", addr->route_domain);
    fprintf(f, "\n");

    if (full_info) fprintf(f, "  router = %s, ", addr->router->name);
    }

  else if (full_info) fprintf(f, "  director/router unset, ");

  if (full_info)
    fprintf(f, "transport = %s\n", (addr->transport == NULL)? "unset" :
      addr->transport->name);

  /* Show any hosts that are set up; fiddle a bit to get a nice format */

  if (addr->host_list != NULL)
    {
    host_item *h;
    int maxlen = 0;
    int maxaddlen = 0;
    for (h = addr->host_list; h != NULL; h = h->next)
      {
      int len = (int)strlen(h->name);
      if (len > maxlen) maxlen = len;
      if (h->address != NULL)
        {
        len = (int)strlen(h->address);
        if (len > maxaddlen) maxaddlen = len;
        }
      }
    for (h = addr->host_list; h != NULL; h = h->next)
      {
      int len = (int)strlen(h->name);
      fprintf(f, "  host %s ", h->name);
      while (len++ < maxlen) fprintf(f, " ");
      if (h->address != NULL)
        {
        fprintf(f, "[%s] ", h->address);
        len = (int)strlen(h->address);
        while (len++ < maxaddlen) fprintf(f," ");
        }
      if (h->mx >= 0) fprintf(f, "MX=%d", h->mx);
      fprintf(f, "\n");
      }
    }
  }

return yield;  /* Will be DEFER or FAIL if any one has */
}





/*************************************************
*          Setup host list matching data         *
*************************************************/

/* This function is called for a number of parameters containing lists
of strings of the form [ident@]host, to scan the strings it contains and build
a chain of control blocks of the broken-down or compiled data for subsequent
use. If the host is precisely "@" it refers to the primary hostname. The daemon
calls this at its start if it is listening for SMTP calls. It is also called
from verify_sender_preliminary for other kinds of SMTP input.

Arguments:
  list      colon-separated list of items
  anchor    where to chain the created blocks

Returns:    nothing
*/

void
verify_setup_hostlist(char *list, host_item **anchor)
{
char *s, *t;
char buffer[1024];

DEBUG(2) debug_printf("verify_setup_hostlist called\n");
if (*anchor != NULL) return;                  /* Just in case */

for (s = string_nextinlist(&list, ':', buffer, sizeof(buffer));
     s != NULL;
     s = string_nextinlist(&list, ':', buffer, sizeof(buffer)))
  {
  host_item *h = store_malloc(sizeof(host_item));
  h->next = NULL;
  *anchor = h;
  anchor = &(h->next);

  /* Make a permanent copy of the string. */

  s = string_copy(s);

  /* If there's an '@' in the string that is not the last character, it starts
  with an ident string; carve that off. */

  if ((t = strchr(s, '@')) != NULL && t[1] != 0)
    {
    *t = 0;
    h->ident_string = s;
    s = t+1;
    }
  else h->ident_string = NULL;

  /* If the host name string is precisely "@", then substitute the primary
  host name. */

  h->name = (strcmp(s, "@") == 0)? string_copy(primary_hostname) : s;
  h->compiled_name = NULL;
  h->address = NULL;

  /* If the name is of the form of an IP address, we can set the address
  field immediately. */

  if (string_is_ip_address(h->name)) h->address = h->name;

  /* Else if the name is not wildcarded (starting with * or ^) or a file lookup
  (contains ';') then we can look up the IP address(es) once and for all.
  Ignore failure - this will just cause slower reverse lookups to happen later.
  Move the anchor pointer on to the end for multi-homed hosts. */

  else if (s[0] != '*' && s[0] != '^' && strchr(s, ';') == NULL)
    {
    char *fully_qualified_name;
    (void) host_find_byname(h, &fully_qualified_name);
    while (*anchor != NULL) anchor = &((*anchor)->next);  /* Multihomed */
    }
  }
}




/*************************************************
*         Check host+ident is in list            *
*************************************************/

/* This function is called from a number of places to test whether the current
calling host (plus ident) is in a list of hosts + idents.

Arguments:
  list                 host list for checking
  anchor               anchor for broken down data (may already be done if
                         via daemon)
  find_failed_return   the value to return if the host name has to be found
                         by gethostbyaddr(), and that fails

Returns:     TRUE if current host+ident is in the list,
             FALSE if definitely not in the list,
             find_failed_return if gethostbyaddr() failed.
*/

BOOL
verify_check_host(char *list, host_item **anchor, BOOL find_failed_return)
{
host_item *h;
BOOL ffr = find_failed_return;
char *sender_host_ipv4;

/* We can't do anything if the calling host's address is not available or if
the list is empty. */

if (sender_host_address == NULL || list == NULL) return FALSE;

/* Initialize the munged data if not already done so (for calls via the
daemon it should already be set up). */

if (*anchor == NULL) verify_setup_hostlist(list, anchor);

/* If the sender host address starts off ::ffff: it is an IPv6 address in
IPv4-compatible mode. Find the IPv4 part for checking against IPv4 addresses.
*/

sender_host_ipv4 = (strncmp(sender_host_address, "::ffff:", 7) == 0)?
  sender_host_address + 7 : sender_host_address;

/* Scan each listed item, checking the ident string if present, and then
checking the address. For single names, the address will be in the host item.
If the address isn't set in the host item, we have to match on the textual name
of the host, which should normally be wildcarded, though this case can also
arise if the forward lookup failed. Textual name matching requires a reverse
lookup of the sender address, if that hasn't already been done.

If a reverse lookup fails, we normally return find_failed_return. This is set
true for calls to test reject lists and false for calls to test accept lists.
However, there is a requirement to be able to relax this. If any "host name" is
of the form "+allow_unknown", then if a subsequent reverse lookup fails, we
return the negation of find_failed_return. */

for (h = *anchor; h != NULL; h = h->next)
  {
  char *ident;

  /* If the host name is "+allow_unknown", negate the value returned for a
  subsequent failed reverse lookup. */

  if (strcmp(h->name, "+allow_unknown") == 0)
    {
    ffr = ~find_failed_return;
    continue;
    }

  /* Handle ident checking. If the test string starts with \ just ignore that
  character; otherwise if it starts with ! the test is negated. */

  if ((ident = h->ident_string) != NULL)
    {
    BOOL match = TRUE;
    if (sender_ident == NULL) continue;
    if (*ident == '\\') ident++;
      else if (*ident == '!') { ident++; match = FALSE; }
    if ((strcmp(ident, sender_ident) == 0) != match) continue;
    }

  /* If we have the IP address already, test that. If it is an IPv4 address
  ensure that IPv6 compatibility addresses are correctly compared. */

  if (h->address != NULL)
    {
    char *s = (strchr(h->address, ':') == NULL)?
      sender_host_ipv4 : sender_host_address;
    if (strcmp(h->address, s) == 0) return TRUE; else continue;
    }

  /* Otherwise match on the name. Optimise for the special case of "*".
  If we can't find the host name, return the value given in find_failed_
  return. */

  if (strcmp(h->name, "*") == 0) return TRUE;

  if (sender_host_name == NULL)
    {
    sender_host_name = host_find_byaddr(sender_host_address);
    if (sender_host_name == NULL)
      {
      DEBUG(2) debug_printf("failed to get host name for %s in "
        "verify_check_host: returning %d\n", sender_host_address, ffr);
      return ffr;
      }
    host_build_sender_fullhost();
    }

  if (match_check_string(sender_host_name, h->name, &(h->compiled_name), -1,
    TRUE, NULL)) return TRUE;
  }

return FALSE;
}




/*************************************************
*          Setup net list matching data          *
*************************************************/

/* This function is called for a number of parameters containing lists
of strings of the form <ip-address>/<mask>, to scan the strings it contains and
build a chain of control blocks of the broken-down or compiled data for
subsequent use. For IPv4 addresses, the mask is an integer, or in the form of
another v4 address. For IPv6 addresses the mask is always an integer. The
daemon calls this at its start if it is listening for SMTP calls. The string
has already been syntax checked at read-in time.

Arguments:
  list      colon-separated list of items
  anchor    where to chain the created blocks

Returns:    nothing
*/

void
verify_setup_netlist(char *list, ip_net_item **anchor)
{
char *s;
char buffer[256];

DEBUG(2) debug_printf("verify_setup_netlist called\n");
if (*anchor != NULL) return;                  /* Just in case */

for (s = string_nextinlist(&list, ':', buffer, sizeof(buffer));
     s != NULL;
     s = string_nextinlist(&list, ':', buffer, sizeof(buffer)))
  {
  int i;
  char *error;
  ip_net_item *n = store_malloc(sizeof(ip_net_item));

  n->next = NULL;
  *anchor = n;
  anchor = &(n->next);

  /* If the string starts with a slash, it is taken as a file name, and
  not much more can be done here. */

  if (*s == '/')
    {
    n->filename = string_copy(s);
    continue;
    }

  /* Else set the filename pointer to null to indicate the presence of
  binary data, use the common function to decode an address and a mask,
  and apply the mask to the saved address. */

  n->filename = NULL;
  n->size = host_amton(s, n->address, n->mask, &error);
  for (i = 0; i < n->size; i++) n->address[i] &= n->mask[i];
  }
}



/*************************************************
*         Check address is in net                *
*************************************************/

/* This function is called from a number of places to test whether the IP
address of the current calling host is in a list of networks.

Arguments:
  list       net list for checking
  anchor     anchor for broken down data (may already be done if via daemon)

Returns:     TRUE if IP address of current host is in the list
*/

BOOL
verify_check_net(char *list, ip_net_item **anchor)
{
/* We can't do anything if the calling host's address is not available, or if
the list is empty. */

if (sender_host_address == NULL || list == NULL) return FALSE;

/* Initialize the munged data if not already done so (for calls via the
daemon it should already be set up). */

if (*anchor == NULL) verify_setup_netlist(list, anchor);

/* Use the generic function that handles both IPv4 and IPv6 addresses */

return match_net_isinlist(sender_host_address, *anchor);
}




/*************************************************
*             Get valid header address           *
*************************************************/

/* Scan the originator headers of the message, looking for an address that
verifies successfully. This function is called from the second verification
function below. RFC 822 says:

    o   The "Sender" field mailbox should be sent  notices  of
        any  problems in transport or delivery of the original
        messages.  If there is no  "Sender"  field,  then  the
        "From" field mailbox should be used.

    o   If the "Reply-To" field exists, then the reply  should
        go to the addresses indicated in that field and not to
        the address(es) indicated in the "From" field.

So we check a Sender field if there is one, else a Reply_to field,
else a From field.

Arguments:
  newaddr    address of pointer to point at the verified address, or
               NULL, if value of the address not wanted
  header_id  address of pointer to point at name of header used,
               or NULL if not wanted

Returns:     result of the verification attempt: OK, FAIL, or DEFER;
             FAIL is given if no appropriate headers are found
*/

static int
verify_get_header_address(char **newaddr, char **header_id)
{
static char *header_names[] = { "Sender", "Reply-to", "From" };
header_line *hh[3];
header_line *h;
int yield = FAIL;
int i;

hh[0] = hh[1] = hh[2] = NULL;

for (h = header_list; h != NULL; h = h->next)
  {
  if (h->type == htype_from) hh[2] = h;
  else if (h->type == htype_replyto) hh[1] = h;
  else if (h->type == htype_sender) hh[0] = h;
  }

for (i = 0; i < 3; i++)
  {
  if (hh[i] != NULL)
    {
    BOOL is_local;
    int terminator, new_ok;
    char *ss;
    char *s = strchr(hh[i]->text, ':') + 1;

    while (isspace(*s)) s++;
    ss = parse_find_address_end(s, FALSE);
    terminator = *ss;
    *ss = 0;
    new_ok =
      verify_address(s, NULL, &is_local, newaddr,
        sender_verify_log_details? vopt_log_details : 0);
    *ss = terminator;

    if (new_ok == OK)
      {
      if (header_id != NULL) *header_id = header_names[i];
      return OK;
      }

    if (new_ok == DEFER) yield = DEFER;
    }
  }

return yield;
}




/*************************************************
*            Verify the sender of a message      *
*************************************************/

/* The next two functions operate in tandem, and the second must not be
called without first calling the first one. The reason for this approach is
that some SMTP mailers treat any error returned after the data has been
transmitted as temporary (contrary to RFC821) and keep retrying, even after
they have been sent a 5xx error at the previous attempt. To get round this,
exim keeps a database of failed messages and their hosts, and if the same bad
address is received from the same host soon afterwards, it is rejected at the
preliminary stage (meaning after MAIL FROM for SMTP) in the hope that the far
end might now give up.

The reason for not rejecting at this stage in all cases is that remote
postmasters, when told their systems have been sending out bad messages, always
ask "what were the headers?" and so one needs to have read them in order to log
them. This also helps track down mail forgers. It also makes it possible to
replace bad envelope sender addresses with good ones from inside the message if
that option is configured.

September 1996: Some mailers keep on trying even after getting a 5xx error for
MAIL FROM. If the same bad address is received from the same host for a third
time in a short time, MAIL FROM is accepted, but refuse_all_rcpts is set, and
all subsequent RCPT TO commands get rejected with a 550.

The RFCs imply that the final local-part@domain of a route address should be
intelligible to all parties. It is unfortunately the case that some mailers
abuse this and supply final domain addresses that are meaningful only to
them.

As we are interested only in whether we can route back to this address, we
don't worry about this case, but a possible upgrade would be to make checking
the final address a configurable option. That would then catch

  @valid.domain:junk@junk.domain

which at present gets through. Later: If the new collapse_source_routes option
is set, source routes are collapsed during parsing, so won't appear here, which
is in effect the upgrade mentioned. */




/*************************************************
*        First check on sender address           *
*************************************************/

/* This function is called as soon as a sender address has been received
from an SMTP connection. Unless the host is in the list of those from which no
verification is required, it verifies the address. If it is bad and
sender_verify_reject is FALSE, it gives an OK response with a warning message.

When sender_verify_reject is TRUE and a bad address is received, it checks to
see if the same address has recently been rejected. If not, it sets a flag for
verify_sender() to interrogate. If it has been rejected once recently, an error
return is given for MAIL FROM. If it has been rejected more than once recently,
then MAIL FROM is accepted, but a flag is set to cause all subsequent RCPT TO
commands to be rejected.

Arguments:
  errcode    set this to an SMTP error code on failure
  errmess    set this to point to an error message on failure

Returns:     TRUE if address verified, or did not fail recently, or host
               is in the exception list, i.e. if is OK to proceed
*/

BOOL
verify_sender_preliminary(int *errcode, char **errmess)
{
char *newaddr;
open_db dbblock;
open_db *dbm_file;
dbdata_reject *reject;
int rejectlen;
char buffer[SENDER_ADDRESS_MAXLENGTH + 256];

/* If the sender address is empty, it's an error message with, in effect,
no sender, and we can't check anything. */

if (sender_address[0] == 0)
  {
  sender_ok = SENDER_OK_OK;
  return TRUE;
  }

/* See if this is one of the trusted hosts/identd combinations or nets
for which we accept all addresses. If so, do no further checking. */

if (verify_check_host(sender_verify_except_hosts,
      &sender_verify_except_hostlist, FALSE) ||
    verify_check_net(sender_verify_except_nets,
      &sender_verify_except_netlist))
  {
  DEBUG(2) debug_printf("matched in sender_verify_except list\n");
  sender_ok = SENDER_OK_EXCEPTION;
  return TRUE;
  }

/* Run a verification on the address. */

switch(verify_address(sender_address, NULL, &sender_is_local, &newaddr,
    sender_verify_log_details? vopt_log_details : 0))
  {
  /* After a successful return, the address may have been changed (typically
  a domain will be canonicized or expanded by a router). A new copy is always
  given. */

  case OK:
  DEBUG(2) debug_printf("%s verified ok as %s\n", sender_address, newaddr);
  store_free(sender_address);
  sender_address = newaddr;
  sender_ok = SENDER_OK_OK;
  return TRUE;

  /* Defer is usually a DNS time out. If try_verify switch is set, which means
  accept if soft error, accept it with an appropriate message. Otherwise, pass
  it on, leaving sender_ok set to DEFER, leaving possible (temporary) rejection
  to the second verification function. */

  case DEFER:
  sender_ok = SENDER_OK_DEFER;
  if (sender_try_verify)
    {
    DEBUG (2) debug_printf("%s verification deferred: accepted unverified\n");
    *errcode = 250;
    *errmess = "warning: temporarily unable to resolve sender address: "
      "accepted unverified";
    if (sender_verify_log_details)
      log_write(0, LOG_REJECT, "%s verification deferred but sender_try_verify "
        "is true", sender_address);
    sender_ok = SENDER_OK_WARNING;
    }
  return TRUE;
  }

/* Otherwise it's a hard failure. If rejection is not required, accept with
a warning. */

if (!sender_verify_reject)
  {
  *errcode = 250;
  *errmess = sender_is_local?
    "warning: unknown local-part in sender address" :
    "warning: cannot route to sender address";

  if (sender_verify_log_details)
    log_write(0, LOG_REJECT, "%s failed verification but sender_verify_reject "
      "is false", sender_address);

  sender_ok = SENDER_OK_WARNING;
  return TRUE;
  }

/* It's really unacceptable */

sender_ok = SENDER_OK_NOT;

/* See if this address from this host has recently been rejected. If it has
not, pass now, leaving rejection to the second function, which will create
a DBM entry if necessary (it might not be necessary if the sender is fixed
up from the headers). */

dbm_file = dbfn_open("reject", O_RDWR, &dbblock);
if (dbm_file == NULL) return TRUE;

sprintf(buffer, "%s:%.200s", sender_address,
  (sender_host_name != NULL)? sender_host_name :
  (sender_host_address != NULL)? sender_host_address : "");
reject = dbfn_read_with_length(dbm_file, buffer, &rejectlen);

if (reject == NULL || time(NULL) - reject->time_stamp > 24*60*60)
  {
  dbfn_close(dbm_file);
  return TRUE;
  }

/* There's been a recent rejection. If there has only been one, reject now
(i.e. reject the MAIL FROM) and up the count. For compatibility with older
versions of Exim that didn't have the count, inspect the length of the
record. This change made in September 1996; after a few months this could
be removed. */

if (rejectlen < sizeof(dbdata_reject) || !reject->rejected_mail_from)
  {
  dbdata_reject newreject;
  *errcode = 550;
  *errmess = sender_is_local?
    "unknown local-part in sender" : "cannot route to sender";
  newreject.rejected_mail_from = TRUE;
  dbfn_write(dbm_file, buffer, &newreject, sizeof(dbdata_reject));
  dbfn_close(dbm_file);
  DEBUG(2) debug_printf("%s verification failed after MAIL FROM\n",
    sender_address);
  return FALSE;
  }

/* There has been a previous recent rejection after MAIL FROM; the mailer
at the far end is horribly broken. Allow through this MAIL FROM with warning
text, but set refuse_all_rcpts to cause all RCPT TO commands to be failed
with 550 - which seems to be the only thing some mailers understand. */

dbfn_close(dbm_file);
refuse_all_rcpts = TRUE;
*errcode = 250;
*errmess = "reject all recipients: 3 times bad sender";
DEBUG(2) debug_printf("%s verification failed - will reject all recipients",
  sender_address);
return TRUE;
}




/*************************************************
*        Second check on sender address          *
*************************************************/


/* This function is called when a message has been completely read, but the
headers haven't yet been written to the spool file, if the sender_verify
option is set. The sender check actually took place in the preliminary
function; its result is left in sender_ok. If it is bad, it may (depending on
the configuration) be permitted to replace it with a value taken from one of
the headers (From, Sender) if that address is viable.

Arguments:
  errcode    set this to an SMTP error code on failure
  errmess    set this to point to an error message on failure

Returns:     TRUE if address verified or fixed up, FALSE otherwise
*/

BOOL
verify_sender(int *errcode, char **errmess)
{
open_db dbblock;
open_db *dbm_file;
dbdata_reject reject;
char buffer[256];

/* The sender verified correctly on the preliminary check. However, sometimes
we are required to inspect the headers as well. */

switch(sender_ok)
  {
  /* Wasn't really OK, but we are running in warning mode. If sender_verify_
  fixup is set, log what it would have done if we weren't running in warning
  mode. */

  case SENDER_OK_WARNING:
  if (sender_verify_fixup)
    {
    char *newaddr, *header_id;
    if (verify_get_header_address(&newaddr, &header_id) == OK)
      log_write(0, LOG_MAIN, "return-path %s is rewritable as %s using %s",
        sender_address, newaddr, header_id);
    }

  /* Fall through */

  /* Sender was really, really OK. Verify a sender in an appropriate header
  if configured to do so. */

  case SENDER_OK_OK:
  if (headers_sender_verify ||
      (sender_address[0] == 0 && headers_sender_verify_errmsg))
    {
    switch (verify_get_header_address(NULL, NULL))
      {
      case DEFER:
      if (headers_checks_fail)
        {
        DEBUG(2) debug_printf("verification of sender from message "
          "headers deferred\n");
        *errcode = 451;
        *errmess = "can't currently verify any sender in message headers "
          "(please try again later): return path is";
        return FALSE;
        }

      log_write(3, LOG_REJECT, "warning: from%s%s <%s>: can't check for "
        "valid sender in headers",
        (sender_fullhost == NULL)? "" : " ",
        (sender_fullhost == NULL)? "" : host_and_ident(),
        sender_address);
      break;

      case FAIL:
      if (headers_checks_fail)
        {
        DEBUG(2) debug_printf("no valid sender in message headers: "
          "return path is <%s>\n", sender_address);
        *errcode = 550;
        *errmess = "no valid sender in message headers: return path is";
        return FALSE;
        }

      log_write(3, LOG_REJECT, "warning: from%s%s <%s>: no valid sender "
        "in message headers",
        (sender_fullhost == NULL)? "" : " ",
        (sender_fullhost == NULL)? "" : host_and_ident(),
        sender_address);
      break;
      }
    }

  /* Fall through */

  /* Sender was accepted by exception. Don't verify a sender from the headers
  in this case. */

  case SENDER_OK_EXCEPTION:
  return TRUE;
  }

/* Original sender not OK, or the verification attempt deferred. If configured,
attempt a fix-up by scanning the messages's originator headers for a valid
address. */

if (sender_verify_fixup)
  {
  char *newaddr, *header_id;
  if (verify_get_header_address(&newaddr, &header_id) == OK)
    {
    log_write(0, LOG_MAIN, "return-path %s rewritten as %s using %s",
      sender_address, newaddr, header_id);
    header_add(htype_other,
      "X-BadReturnPath: %s rewritten as %s\n  using \"%s\" header\n",
      sender_address, newaddr, header_id);
    sender_address = newaddr;
    return TRUE;
    }
  }

/* If the original verification attempt deferred, give a temporary error
return. */

if (sender_ok == SENDER_OK_DEFER)
  {
  DEBUG(2) debug_printf("%s verification deferred\n", sender_address);
  *errcode = 451;
  *errmess = "temporarily unable to verify sender address (try again later)";
  return FALSE;
  }

/* Otherwise it's a hard failure. Update the database to record this rejection.
O_RDWR (rather than O_WRONLY) is needed by Berkeley native DB. If opening
fails, don't worry. */

dbm_file = dbfn_open("reject", O_RDWR|O_CREAT, &dbblock);
if (dbm_file != NULL)
  {
  sprintf(buffer, "%s:%.200s", sender_address,
    (sender_host_name != NULL)? sender_host_name :
    (sender_host_address != NULL)? sender_host_address : "");
  reject.rejected_mail_from = FALSE;
  dbfn_write(dbm_file, buffer, &reject, sizeof(dbdata_reject));
  dbfn_close(dbm_file);
  }

/* Now give a hard error */

*errcode = 550;
*errmess = sender_is_local?
  "unknown local-part in sender" : "cannot route to sender";

DEBUG(2) debug_printf("%s verification failed after data\n", sender_address);
return FALSE;
}





/*************************************************
*            Get RFC 1413 identification         *
*************************************************/

/* Attempt to get an id from the sending machine via the RFC 1413 protocol. If
the timeout is set to zero, then the query is not done. There may also be lists
of hosts and nets which are exempt. We copy as many bytes from the result as
will fit into ident_buffer and then free the store that ident_id has got
(several K, I think). To guard against malefactors sending non-printing
characters which could, for example, disrupt a message's headers, make sure the
string consists of printing characters only.

Argument: the socket of the connection for which the ident value is required
Returns:  nothing

Side effect: any received ident value is put in sender_ident (NULL otherwise)
*/

void
verify_get_ident(int socket)
{
sender_ident = NULL;
if (rfc1413_query_timeout > 0 &&
    (rfc1413_except_hosts == NULL ||
      !verify_check_host(rfc1413_except_hosts, &rfc1413_except_hostlist, FALSE))
    &&
    (rfc1413_except_nets == NULL ||
      !verify_check_net(rfc1413_except_nets, &rfc1413_except_netlist)))
  {
  char *ident_ptr = ident_id(socket, rfc1413_query_timeout);
  if (ident_ptr != NULL)
    {
    strncpy(ident_buffer, ident_ptr, sizeof(ident_buffer));
    ident_buffer[sizeof(ident_buffer) - 1] = 0;
    sender_ident = string_printing(ident_buffer, FALSE);
    /* NB: free() not store_free() as it wasn't got by store_malloc() */
    free(ident_ptr);
    }
  }
}

/* End of verify.c */
