/*************************************************
*     Exim - an Internet mail transport agent    *
*************************************************/

/* Copyright (c) University of Cambridge 1995 - 1997 */
/* See the file NOTICE for conditions of use and distribution. */


#include "../exim.h"
#include "pipe.h"



/* Options specific to the pipe transport. They must be in alphabetic
order (note that "_" comes before the lower case letters). Those starting
with "*" are not settable by the user but are used by the option-reading
software for alternative value types. Some options are stored in the transport
instance block so as to be publicly visible; these are flagged with opt_public.
The "directory" option is obsolete, being replaced by "home_directory". */

optionlist pipe_transport_options[] = {
  { "batch",             opt_local_batch | opt_public,
      (void *)(offsetof(transport_instance, local_batch)) },
  { "batch_max",         opt_int | opt_public,
      (void *)(offsetof(transport_instance, batch_max)) },
  { "bsmtp",             opt_local_batch | opt_public,
      (void *)(offsetof(transport_instance, local_smtp)) },
  { "bsmtp_helo",        opt_bool | opt_public,
      (void *)(offsetof(transport_instance, bsmtp_helo)) },
  { "command",           opt_stringptr,
      (void *)(offsetof(pipe_transport_options_block, cmd)) },
  { "current_directory", opt_stringptr | opt_public,
      (void *)(offsetof(transport_instance, current_dir)) },
  { "delivery_date_add", opt_bool,
      (void *)(offsetof(pipe_transport_options_block, delivery_date_add)) },
  { "directory",         opt_stringptr | opt_public | opt_hidden,
      (void *)(offsetof(transport_instance, home_dir)) },
  { "envelope_to_add", opt_bool,
      (void *)(offsetof(pipe_transport_options_block, envelope_to_add)) },
  { "from_hack",         opt_bool,
      (void *)(offsetof(pipe_transport_options_block, from_hack)) },
  { "home_directory",    opt_stringptr | opt_public,
      (void *)(offsetof(transport_instance, home_dir)) },
  { "ignore_status",     opt_bool,
      (void *)(offsetof(pipe_transport_options_block, ignore_status)) },
  { "log_fail_output",   opt_bool | opt_public,
      (void *)(offsetof(transport_instance, log_fail_output)) },
  { "log_output",        opt_bool | opt_public,
      (void *)(offsetof(transport_instance, log_output)) },
  { "max_output",        opt_mkint,
      (void *)(offsetof(pipe_transport_options_block, max_output)) },
  { "path",              opt_stringptr,
      (void *)(offsetof(pipe_transport_options_block, path)) },
  { "prefix",            opt_stringptr,
      (void *)(offsetof(pipe_transport_options_block, prefix)) },
  { "restrict_to_path",  opt_bool,
      (void *)(offsetof(pipe_transport_options_block, restrict_to_path)) },
  { "retry_use_local_part", opt_bool | opt_public,
      (void *)offsetof(transport_instance, retry_use_local_part) },
  { "return_fail_output",opt_bool | opt_public,
      (void *)(offsetof(transport_instance, return_fail_output)) },
  { "return_output",     opt_bool | opt_public,
      (void *)(offsetof(transport_instance, return_output)) },
  { "return_path_add",   opt_bool,
      (void *)(offsetof(pipe_transport_options_block, return_path_add)) },
  { "suffix",            opt_stringptr,
      (void *)(offsetof(pipe_transport_options_block, suffix)) },
  { "timeout",           opt_time,
      (void *)(offsetof(pipe_transport_options_block, timeout)) },
  { "use_shell",         opt_bool,
      (void *)(offsetof(pipe_transport_options_block, use_shell)) },
};

/* Size of the options list. An extern variable has to be used so that its
address can appear in the tables drtables.c. */

int pipe_transport_options_count =
  sizeof(pipe_transport_options)/sizeof(optionlist);

/* Default private options block for the pipe transport. */

pipe_transport_options_block pipe_transport_option_defaults = {
  NULL,           /* cmd */
  "/usr/bin",     /* path */
  "From ${if def:return_path{$return_path}{MAILER-DAEMON}} ${tod_bsdinbox}\n",
                  /* prefix */
  "\n",           /* suffix */
  20480,          /* max_output */
  60*60,          /* timeout */
  FALSE,          /* from_hack */
  FALSE,          /* ignore_status */
  FALSE,          /* return_path_add */
  FALSE,          /* delivery_date_add */
  FALSE,          /* envelope_to_add */
  FALSE,          /* restrict_to_path */
  FALSE,          /* use_shell */
};



/*************************************************
*          Initialization entry point            *
*************************************************/

/* Called for each instance, after its options have been read, to
enable consistency checks to be done, or anything else that needs
to be set up. */

void
pipe_transport_init(transport_instance *tblock)
{
pipe_transport_options_block *ob =
  (pipe_transport_options_block *)(tblock->options_block);

/* Retry_use_local_part defaults TRUE if unset */

if (tblock->retry_use_local_part == 2) tblock->retry_use_local_part = TRUE;

/* Only one of return_output/return_fail_output or log_output/log_fail_output
should be set. */

if (tblock->return_output && tblock->return_fail_output)
  log_write(0, LOG_PANIC_DIE|LOG_CONFIG,
    "both return_output and return_fail_output set for %s transport",
    tblock->name);

if (tblock->log_output && tblock->log_fail_output)
  log_write(0, LOG_PANIC_DIE|LOG_CONFIG,
    "both log_output and log_fail_output set for %s transport",
    tblock->name);

/* If batch SMTP is set, ensure the generic local batch option matches */

if (tblock->local_smtp != local_batch_off)
  tblock->local_batch = tblock->local_smtp;

/* The restrict_to_path and use_shell options are incompatible */

if (ob->restrict_to_path && ob->use_shell)
  log_write(0, LOG_PANIC_DIE|LOG_CONFIG,
    "both restrict_to_path and use_shell set for %s transport",
    tblock->name);
}



/*************************************************
*          Set up direct (non-shell) command     *
*************************************************/

/* This function is called when a command line is to be parsed by the transport
and executed directly, without the use of /bin/sh.

Arguments:
  argvptr            pointer to anchor for argv vector
  cmd                points to the command string
  expand_arguments   true if expansion is to occur
  addr               chain of addresses
  tname              the transport name
  ob                 the transport options block

Returns:             TRUE if all went well; otherwise an error will be
                     set in the first address and FALSE returned
*/

static BOOL
set_up_direct_command(char ***argvptr, char *cmd, BOOL expand_arguments,
  address_item *addr, char *tname, pipe_transport_options_block *ob)
{
char **argv;
char buffer[64];

/* Set up "transport <name>" to be put in any error messages, and then
call the common function for creating an argument list and expanding
the items if necessary. If it fails, this function fails (error information
is in the addresses). */

sprintf(buffer, "%.50s transport", tname);
if (!transport_set_up_command(argvptr, cmd, expand_arguments,
      (addr->local_part[0]== '|')? FAIL : PANIC,
      addr, buffer))
  return FALSE;

/* Point to the set-up arguments. */

argv = *argvptr;

/* If restrict_to_path is set, check that the name of the command does not
contain any slashes. */

if (ob->restrict_to_path)
  {
  if (strchr(argv[0], '/') != NULL)
    {
    addr->transport_return = FAIL;
    addr->message = string_sprintf("\"/\" found in \"%s\" (command for %s "
      "transport) - failed for security reasons", cmd, tname);
    return FALSE;
    }
  }

/* If the command is not an absolute path, search the PATH directories
for it. */

/*
if (argv[0][0] != '/')
  {
  char *p;
  char *listptr = ob->path;
  char buffer[1024];

  for (p = string_nextinlist(&listptr, ':', buffer, sizeof(buffer));
       p != NULL;
       p = string_nextinlist(&listptr, ':', buffer, sizeof(buffer)))
    {
    struct stat statbuf;
    sprintf(big_buffer, "%.256s/%.256s", p, argv[0]);
    if (stat(big_buffer, &statbuf) == 0)
      {
      argv[0] = string_copy(big_buffer);
      break;
      }
    }
  if (p == NULL)
    {
    addr->transport_return = FAIL;
    addr->message = string_sprintf("\"%s\" command not found for %s transport",
      argv[0], tname);
    return FALSE;
    }
  }
*/
  
return TRUE;
}


/*************************************************
*               Set up shell command             *
*************************************************/

/* This function is called when a command line is to be passed to /bin/sh
without parsing inside the transport.

Arguments:
  argvptr            pointer to anchor for argv vector
  cmd                points to the command string
  expand_arguments   true if expansion is to occur
  addr               chain of addresses
  tname              the transport name
  ob                 the transport options block

Returns:             TRUE if all went well; otherwise an error will be
                     set in the first address and FALSE returned
*/

static BOOL
set_up_shell_command(char ***argvptr, char *cmd, BOOL expand_arguments,
  address_item *addr, char *tname, pipe_transport_options_block *ob)
{
char **argv;

*argvptr = argv = (char **)store_malloc((4)*sizeof(char *));

argv[0] = "cmd.exe";
argv[1] = "/c";

/* We have to take special action to handle the special "variable" called
$pipe_addresses, which is not recognized by the normal expansion function. */

DEBUG(7) debug_printf("shell pipe command before expansion:\n  %s\n", cmd);

if (expand_arguments)
  {
  char *s = cmd;
  char *p = strstr(cmd, "pipe_addresses");

  if (p != NULL && (
         (p > cmd && p[-1] == '$') ||
         (p > cmd + 1 && p[-2] == '$' && p[-1] == '{' && p[14] == '}')))
    {
    address_item *ad;
    char *q = p + 14;
    int size = (int)strlen(cmd) + 64;
    int offset;

    if (p[-1] == '{') { q++; p--; }

    s = store_malloc(size);
    offset = p - cmd - 1;
    strncpy(s, cmd, offset);

    for (ad = addr; ad != NULL; ad = ad->next)
      {
      if (ad != addr) string_cat(s, &size, &offset, " ", 1);
      string_cat(s, &size, &offset, ad->orig, (int)strlen(ad->orig));
      }

    string_cat(s, &size, &offset, q, (int)strlen(q));
    s[offset] = 0;
    }

  argv[2] = expand_string(s);
  if (argv[2] == NULL)
    {
    addr->transport_return = (addr->local_part[0] == '|')? FAIL : PANIC;
    addr->message = string_sprintf("Expansion of command \"%s\" "
      "in %s transport failed: %s",
      cmd, tname, expand_string_message);
    return FALSE;
    }
  DEBUG(7) debug_printf("shell pipe command after expansion:\n  %s\n", argv[2]);
  }
else argv[2] = cmd;

argv[3] = (char *)0;
return TRUE;
}




/*************************************************
*              Main entry point                  *
*************************************************/

/* See local README for interface details. */

void
pipe_transport_entry(
  transport_instance *tblock,      /* data for this instantiation */
  address_item *addr)              /* address(es) we are working on */
{
pid_t pid, outpid;
int fd, rc;
int pipefd[2];
int envcount = 0;
pipe_transport_options_block *ob =
  (pipe_transport_options_block *)(tblock->options_block);
BOOL smtp_dots = FALSE;
BOOL return_path_add = ob->return_path_add;
BOOL delivery_date_add = ob->delivery_date_add;
BOOL envelope_to_add = ob->envelope_to_add;
BOOL expand_arguments;
char **argv;
char *envp[20];
char *env_local_part;
char *cmd;

DEBUG(2) debug_printf("%s transport entered\n", tblock->name);

/* Set up for the good case */

addr->transport_return = OK;
addr->basic_errno = 0;

/* Pipes are not accepted as general addresses, but they can be generated from
.forward files or alias files. In those cases, the command to be obeyed is
pointed to by addr->local_part; it starts with the pipe symbol. In other cases,
the command is supplied as one of the pipe transport's options. */

if (addr->local_part[0] == '|')
  {
  cmd = addr->local_part + 1;
  while (isspace(*cmd)) cmd++;
  expand_arguments = addr->expand_pipe;
  }
else
  {
  cmd = ob->cmd;
  expand_arguments = TRUE;
  }

/* The default way of processing the command is to split it up into arguments
here, and run it directly. This offers some security advantages. However, there
are installations that want by default to run commands under /bin/sh always, so
there is an option to do that. */

if (ob->use_shell)
  {
  if (!set_up_shell_command(&argv, cmd, expand_arguments, addr, tblock->name,
    ob)) return;
  }
else if (!set_up_direct_command(&argv, cmd, expand_arguments, addr,
  tblock->name, ob)) return;

/* Set up the environment for the command. */

env_local_part = (deliver_localpart == NULL)? "" : deliver_localpart;

envp[envcount++] = string_sprintf("LOCAL_PART=%s", env_local_part);
envp[envcount++] = string_sprintf("LOGNAME=%s", env_local_part);
envp[envcount++] = string_sprintf("USER=%s", env_local_part);

envp[envcount++] = string_sprintf("DOMAIN=%s", (deliver_domain == NULL)?
  "" : deliver_domain);

envp[envcount++] = string_sprintf("HOME=%s", (deliver_home == NULL)?
  "" : deliver_home);

envp[envcount++] = string_sprintf("MESSAGE_ID=%s", message_id_external);
envp[envcount++] = string_sprintf("PATH=%s", ob->path);
envp[envcount++] = string_sprintf("QUALIFY_DOMAIN=%s", qualify_domain_sender);
envp[envcount++] = string_sprintf("SENDER=%s", sender_address);
envp[envcount++] = "SHELL=cmd.exe";

if (addr->host_list != NULL)
  envp[envcount++] = string_sprintf("HOST=%s", addr->host_list->name);

envp[envcount] = NULL;


/* If the -N option is set, can't do any more. */

if (dont_deliver)
  {
  debug_printf("*** delivery by %s transport bypassed by -N option",
    tblock->name);
  return;
  }


/* Handling the output from the pipe is tricky. If a file for catching this
output is provided, we could just hand that fd over to the pipe, but this isn't
very safe because the pipe might loop and carry on writing for ever (which is
exactly what happened in early versions of Exim). Therefore we must hand over a
pipe fd, read our end of the pipe and count the number of bytes that come
through, chopping the sub-process if it exceeds some limit.

However, this means we want to run a sub-process with both its input and output
attached to pipes. We can't handle that easily from a single parent process
using straightforward code such as the transport_write_message() function
because the subprocess might not be reading its input because it is trying to
write to a full output pipe. The complication of redesigning the world to
handle this is too great - simpler just to run another process to do the
reading of the output pipe. */


/* Make the pipe for handling the output - do this always, even if a
return_file is not provided. */

if (pipe(pipefd) < 0)
  {
  addr->transport_return = DEFER;
  addr->message = string_sprintf(
    "Failed to create pipe for handling output in %s transport",
      tblock->name);
  return;
  }

/* HACK -- HACK -- HACK -- HACK; envp (2nd of child_open forced to NULL to
   avoid losing main environment. Will be properly fixed later. */

if ((pid = child_open(argv, NULL, &fd,
     pipefd[pipe_write])) < 0)
  {
  addr->transport_return = DEFER;
  addr->message = string_sprintf(
    "Failed to create child process for %s transport", tblock->name);
  return;
  }

/* Close off the end of the output pipe we are not using. */

close(pipefd[pipe_write]);

/* Now fork a process to handle the output that comes down the pipe. */

if ((outpid = fork()) < 0)
  {
  addr->basic_errno = errno;
  addr->transport_return = DEFER;
  addr->message = string_sprintf(
    "Failed to create process for handling output in %s transport",
      tblock->name);
  close(pipefd[pipe_read]);
  return;
  }

/* This is the code for the output-handling subprocess. Read from the pipe
in chunks, and write to the return file if one is provided. Keep track of
the number of bytes handled. If the limit is exceeded, try to kill the
subprocess, and in any case close the pipe and exit, which should cause the
subprocess to fail. */

if (outpid == 0)
  {
  int count = 0;
  close(fd);
  set_process_info("reading output from %s", cmd);
  while ((rc = read(pipefd[pipe_read], big_buffer, big_buffer_size)) > 0)
    {
    if (addr->return_file >= 0)
      write(addr->return_file, big_buffer, rc);
    count += rc;
    if (count > ob->max_output)
      {
      char *message = "\n\n*** Too much output - remainder discarded ***\n";
      DEBUG(2) debug_printf("Too much output from pipe - killed\n");
      if (addr->return_file >= 0)
        write(addr->return_file, message, (int)strlen(message));
      kill(pid, SIGKILL);
      break;
      }
    }
  close(pipefd[pipe_read]);
  _exit(0);
  }

close(pipefd[pipe_read]);  /* Not used in this process */


/* Carrying on now with the main parent process. Attempt to write the message
to it down the pipe. It is a fallacy to think that you can detect write errors
when the sub-process fails to read the pipe. The parent process may complete
writing and close the pipe before the sub-process completes. Sleeping for a bit
here lets the sub-process get going, but it may still not complete. So we
ignore all writing errors. */

DEBUG(2) debug_printf("Writing message to pipe\n");

/* If the local_smtp option is not unset, we need to write SMTP prefix
information. The various different values for batching are handled outside; if
there is more than one address available here, all must be included. Force
SMTP dot-handling. */

if (tblock->local_smtp != local_smtp_off)
  {
  address_item *a;
  char *sender = (addr->errors_address != NULL)?
    addr->errors_address : sender_address;
  smtp_dots = TRUE;
  return_path_add = delivery_date_add = envelope_to_add = FALSE;

  if (tblock->bsmtp_helo)
    {
    write(fd, "HELO ", 5);
    write(fd, primary_hostname, (int)strlen(primary_hostname));
    write(fd, "\n", 1);
    }

  write(fd, "MAIL FROM: <", 12);
  write(fd, sender, (int)strlen(sender));
  write(fd, ">\n", 2);

  for (a = addr; a != NULL; a = a->next)
    {
    write(fd, "RCPT TO: <", 10);
    if (a->local_part[0] == ',' || a->local_part[0] == ':')
      {
      write(fd, "@", 1);
      write(fd, a->domain, (int)strlen(a->domain));
      write(fd, a->local_part, (int)strlen(a->local_part));
      }
    else
      {
      write(fd, a->local_part, (int)strlen(a->local_part));
      write(fd, "@", 1);
      write(fd, a->domain, (int)strlen(a->domain));
      }
    write(fd, ">\n", 2);
    }

  write(fd, "DATA\n", 5);
  }

/* Now any other configured prefix. */

if (ob->prefix != NULL)
  {
  char *prefix = expand_string(ob->prefix);
  if (prefix == NULL)
    {
    addr->transport_return = PANIC;
    addr->message = string_sprintf("Expansion of \"%s\" (prefix for %s "
      "transport) failed", ob->prefix, tblock->name);
    return;
    }
  write(fd, prefix, (int)strlen(prefix));
  }

if (!transport_write_message(addr, fd,
    (return_path_add? topt_add_return_path : 0) |
    (delivery_date_add? topt_add_delivery_date : 0) |
    (envelope_to_add? topt_add_envelope_to : 0) |
    (ob->from_hack? topt_from_hack : 0 ) |
    (smtp_dots? topt_smtp_dots : 0),        /* no CRLF */
    addr->errors_address, 0, tblock->add_headers, tblock->remove_headers))
  {
  if (errno != EPIPE && errno != ERRNO_WRITEINCOMPLETE)
    {
    addr->transport_return = PANIC;
    addr->basic_errno = errno;
    if (errno == ERRNO_ADDHEADER_FAIL)
      addr->message = string_sprintf("Failed to expand added headers");
    else if (errno == ERRNO_FILTER_FAIL)
      addr->message = string_sprintf("Filter process failure");
    else
      addr->message = string_sprintf("Error %d", errno);
    return;
    }
  else DEBUG(9)
    {
    debug_printf("transport error %s ignored\n",
      (errno == EPIPE)? "EPIPE" : "WRITEINCOMPLETE");
    }
  }

/* Now any configured suffix */

if (ob->suffix != NULL)
  {
  char *suffix = expand_string(ob->suffix);
  if (suffix == NULL)
    {
    addr->transport_return = PANIC;
    addr->message = string_sprintf("Expansion of \"%s\" (suffix for %s "
      "transport) failed", ob->suffix, tblock->name);
    return;
    }
  write(fd, suffix, (int)strlen(suffix));
  }

/* If local_smtp, write the terminating dot. */

if (tblock->local_smtp != local_smtp_off) write(fd, ".\n", 2);

/* OK, the writing is now all done. Close the pipe. */

(void) close(fd);

/* Wait for the child process to complete and take action if the returned
status is nonzero or if something went wrong. */

if ((rc = child_close(pid, ob->timeout)) != 0)
  {
  /* The process did not complete in time; kill it and fail the delivery. */

  if (rc == -256)
    {
    kill(pid, SIGKILL);
    addr->transport_return = FAIL;
    addr->message = string_sprintf("pipe delivery process timed out",
      tblock->name);
    }

  /* Wait() failed. */

  else if (rc == -257)
    {
    addr->transport_return = PANIC;
    addr->message = string_sprintf("Wait() failed for child process of %s "
      "transport: %s", tblock->name, strerror(errno));
    }

  /* Either the process completed, but yielded a non-zero (necessarily
  positive) status, or the process was terminated by a signal (rc will contain
  the negation of the signal number). Treat killing by signal as failure unless
  status is being ignored. */

  else if (rc < 0)
    {
    if (!ob->ignore_status)
      {
      addr->transport_return = FAIL;
      addr->message = string_sprintf("Child process of %s transport (running "
        "command \"%s\") was terminated by signal %d", tblock->name, cmd, -rc);
      }
    }

  /* For positive values (process terminated with non-zero status), we need a
  status code to request deferral. A number of systems contain the following
  line in sysexits.h:

      #define EX_TEMPFAIL 75 temp failure; user is invited to retry

  Based on this, use exit code EX_TEMPFAIL to mean "defer" when not ignoring
  the returned status. Otherwise fail. */

  else
    {
    if (!ob->ignore_status)
      {
      int size, ptr, i;
      addr->transport_return = (rc == EX_TEMPFAIL)? DEFER : FAIL;

      /* Ensure the message contains the expanded command and arguments. This
      doesn't have to be brilliantly efficient - it is an error situation. */

      addr->message = string_sprintf("Child process of %s transport returned "
        "%d from command:", tblock->name, rc);
      ptr = (int)strlen(addr->message);
      size = ptr + 1;
      for (i = 0; i < sizeof(argv)/sizeof(int *) && argv[i] != NULL; i++)
        {
        BOOL quote = FALSE;
        addr->message = string_cat(addr->message, &size, &ptr, " ", 1);
        if (strpbrk(argv[i], " \t") != NULL)
          {
          quote = TRUE;
          addr->message = string_cat(addr->message, &size, &ptr, "\"", 1);
          }
        addr->message = string_cat(addr->message, &size, &ptr, argv[i],
          (int)strlen(argv[i]));
        if (quote)
          addr->message = string_cat(addr->message, &size, &ptr, "\"", 1);
        }
      addr->message[ptr] = 0;  /* Ensure concatenated string terminated */
      }
    }
  }

/* Ensure all subprocesses (in particular, the output handling process)
are complete before we pass this point. */

while (wait(&rc) >= 0);

DEBUG(2) debug_printf("%s transport yielded %d\n", tblock->name,
  addr->transport_return);
}

/* End of transport/pipe.c */
