/*************************************************
*     Exim - an Internet mail transport agent    *
*************************************************/

/* Copyright (c) University of Cambridge 1995 - 1997 */
/* See the file NOTICE for conditions of use and distribution. */


#include "../exim.h"
#include "appendfile.h"



/* Options specific to the appendfile transport. They must be in alphabetic
order (note that "_" comes before the lower case letters). Some of them are
stored in the publicly visible instance block - these are flagged with the
opt_public flag. */

optionlist appendfile_transport_options[] = {
  { "batch",             opt_local_batch | opt_public,
      (void *)(offsetof(transport_instance, local_batch)) },
  { "batch_max",         opt_int | opt_public,
      (void *)(offsetof(transport_instance, batch_max)) },
  { "bsmtp",             opt_local_batch | opt_public,
      (void *)(offsetof(transport_instance, local_smtp)) },
  { "bsmtp_helo",        opt_bool | opt_public,
      (void *)(offsetof(transport_instance, bsmtp_helo)) },
  { "create_directory",  opt_bool,
      (void *)(offsetof(appendfile_transport_options_block, create_directory)) },
  { "create_file",       opt_stringptr,
      (void *)(offsetof(appendfile_transport_options_block, create_file_string)) },
  { "current_directory", opt_stringptr | opt_public,
      (void *)(offsetof(transport_instance, current_dir)) },
  { "delivery_date_add", opt_bool,
      (void *)(offsetof(appendfile_transport_options_block, delivery_date_add)) },
  { "directory",         opt_stringptr,
      (void *)(offsetof(appendfile_transport_options_block, dirname)) },
  { "envelope_to_add",   opt_bool,
      (void *)(offsetof(appendfile_transport_options_block, envelope_to_add)) },
  { "file",              opt_stringptr,
      (void *)(offsetof(appendfile_transport_options_block, filename)) },
  { "file_must_exist",   opt_bool,
      (void *)(offsetof(appendfile_transport_options_block, file_must_exist)) },
  { "from_hack",         opt_bool,
      (void *)(offsetof(appendfile_transport_options_block, from_hack)) },
  { "lock_interval",     opt_time,
      (void *)(offsetof(appendfile_transport_options_block, lock_interval)) },
  { "lock_retries",      opt_int,
      (void *)(offsetof(appendfile_transport_options_block, lock_retries)) },
  { "lockfile_timeout",  opt_time,
      (void *)(offsetof(appendfile_transport_options_block, lockfile_timeout)) },
  { "maildir_format",    opt_bool,
      (void *)(offsetof(appendfile_transport_options_block, maildir_format )) } ,
  { "maildir_retries",   opt_int,
      (void *)(offsetof(appendfile_transport_options_block, maildir_retries)) },
  { "notify_comsat",     opt_bool,
      (void *)(offsetof(appendfile_transport_options_block, notify_comsat)) },
  { "prefix",            opt_stringptr,
      (void *)(offsetof(appendfile_transport_options_block, prefix)) },
  { "quota",             opt_stringptr,
      (void *)(offsetof(appendfile_transport_options_block, quota)) },
  { "require_lockfile",  opt_bool,
      (void *)(offsetof(appendfile_transport_options_block, require_lockfile)) },
  { "retry_use_local_part", opt_bool | opt_public,
      (void *)offsetof(transport_instance, retry_use_local_part) },
  { "return_path_add",   opt_bool,
      (void *)(offsetof(appendfile_transport_options_block, return_path_add)) },
  { "suffix",            opt_stringptr,
      (void *)(offsetof(appendfile_transport_options_block, suffix)) },
  { "use_fcntl_lock",    opt_bool,
      (void *)(offsetof(appendfile_transport_options_block, use_fcntl)) },
  { "use_lockfile",      opt_bool,
      (void *)(offsetof(appendfile_transport_options_block, use_lockfile)) },
};

/* Size of the options list. An extern variable has to be used so that its
address can appear in the tables drtables.c. */

int appendfile_transport_options_count =
  sizeof(appendfile_transport_options)/sizeof(optionlist);

/* Default private options block for the appendfile transport. */

appendfile_transport_options_block appendfile_transport_option_defaults = {
  NULL,           /* file name */
  NULL,           /* dir name */
  "From ${if def:return_path{$return_path}{MAILER-DAEMON}} ${tod_bsdinbox}\n",
                  /* prefix */
  "\n",           /* suffix */
  "anywhere",     /* create_file_string (string value for create_file) */
  NULL,           /* quota */
  0,              /* quota_value */
  30*60,          /* lockfile_timeout */
  10,             /* lock_retries */
   3,             /* lock_interval */
  10,             /* maildir_retries */
  create_anywhere,/* create_file */
  TRUE,           /* create_directory */
  FALSE,          /* notify_comsat */
  TRUE,           /* require_lockfile */
  TRUE,           /* use_lockfile */
  TRUE,           /* use_fcntl */
  TRUE,           /* from_hack */
  TRUE,           /* return_path_add */
  TRUE,           /* delivery_date_add */
  TRUE,           /* envelope_to_add */
  FALSE,          /* file_must_exist */
  FALSE           /* maildir format */
};



/*************************************************
*              Setup entry point                 *
*************************************************/

/* Called for each delivery in the privileged state, just before the uid/gid
are changed and the main entry point is called. We use this function to
expand any quota setting, so that it can access files that may not be readable
by the user.

Argument:   tblock   points to the transport instance
Returns:             NULL if OK, pointer to error string if not
*/

char *
appendfile_transport_setup(transport_instance *tblock)
{
appendfile_transport_options_block *ob =
  (appendfile_transport_options_block *)(tblock->options_block);
double d;
char *s, *rest;

if (ob->quota == NULL)
  {
  ob->quota_value = 0;
  return NULL;
  }

s = expand_string(ob->quota);
if (s == NULL)
  return string_sprintf("Expansion of \"%s\" (quota setting "
    "for %s transport) failed: %s", ob->quota, tblock->name,
    expand_string_message);

d = strtod(s, &rest);
if (tolower(*rest) == 'k') { d *= 1024.0; rest++; }
if (tolower(*rest) == 'm') { d *= 1024.0*1024.0; rest++; }
while (isspace(*rest)) rest++;

if (*rest != 0)
  return string_sprintf("Malformed quota setting \"%s\" for "
    "%s transport", s, tblock->name);

ob->quota_value = (int)d;
return NULL;
}



/*************************************************
*          Initialization entry point            *
*************************************************/

/* Called for each instance, after its options have been read, to
enable consistency checks to be done, or anything else that needs
to be set up. */

void
appendfile_transport_init(transport_instance *tblock)
{
appendfile_transport_options_block *ob =
  (appendfile_transport_options_block *)(tblock->options_block);

/* Set up the setup entry point, to be called in the privileged state */

tblock->setup = appendfile_transport_setup;

/* Retry_use_local_part defaults TRUE if unset */

if (tblock->retry_use_local_part == 2) tblock->retry_use_local_part = TRUE;

/* Lock_retries must be greater than zero */

if (ob->lock_retries == 0) ob->lock_retries = 1;

/* Only one of a file name or directory name must be given. */

if (ob->filename != NULL && ob->dirname != NULL)
  log_write(0, LOG_PANIC_DIE|LOG_CONFIG2, "%s transport:\n  "
  "only one of \"file\" or \"directory\" can be specified", tblock->name);

/* If a file name was specified, it must be an absolute path. Can check here
only if there are no expansions. Also, at least one locking mechanism must
be specified. */

if (ob->filename != NULL)
  {
  if (ob->filename[0] != '/' && ob->filename[0] != '$')
    log_write(0, LOG_PANIC_DIE|LOG_CONFIG2, "%s transport:\n  "
    "the file option must specify an absolute path", tblock->name);
  if (!ob->use_fcntl && (!ob->use_lockfile || !ob->require_lockfile))
    log_write(0, LOG_PANIC_DIE|LOG_CONFIG2, "%s transport:\n  "
    "either lockfile or fcntl locking (or both) must be specified",
    tblock->name);
  }

/* If a directory name was specified, it must be an absolute path. Can check
here only if there are no expansions. */

if (ob->dirname != NULL && ob->dirname[0] != '/' && ob->dirname[0] != '$')
  log_write(0, LOG_PANIC_DIE|LOG_CONFIG2, "%s transport:\n  "
  "the directory option must specify an absolute path", tblock->name);

/* If "create_file" is set, check that a valid option is given, and set the
integer variable. */

if (ob->create_file_string != NULL)
  {
  int value = 0;
  if (strcmp(ob->create_file_string, "anywhere") == 0) value = create_anywhere;
  else if (strcmp(ob->create_file_string, "belowhome") == 0) value =
    create_belowhome;
  else if (strcmp(ob->create_file_string, "inhome") == 0)
    value = create_inhome;
  else
    log_write(0, LOG_PANIC_DIE|LOG_CONFIG,
      "invalid value given for \"file_create\" for the %s transport: %s",
        tblock->name, ob->create_file_string);
  ob->create_file = value;
  }

/* If batch SMTP is set, ensure the generic local batch option matches. */

if (tblock->local_smtp != local_batch_off)
  tblock->local_batch = tblock->local_smtp;
}



/*************************************************
*                  Notify comsat                 *
*************************************************/

/* The comsat daemon is the thing that provides asynchronous notification of
the arrival of local messages, if requested by the user by "biff y". It is a
BSD thing that uses a TCP/IP protocol for communication. A message consisting
of the text "user@offset" must be sent, where offset is the place in the
mailbox where new mail starts. There is no scope for telling it which file to
look at, which makes it a less than useful if mail is being delivered into a
non-standard place such as the user's home directory.

Arguments:
  user       user name
  offset     offset in mailbox

Returns:     nothing
*/

static void
notify_comsat(char *user, int offset)
{
int fd;
struct sockaddr_in sa;
struct hostent *hp;
struct servent *sp;
char buffer[256];

DEBUG(2) debug_printf("notify_comsat called\n");

if ((hp = gethostbyname("localhost")) == NULL)
  {
  DEBUG(2) debug_printf("\"localhost\" unknown\n");
  return;
  }

if (hp->h_addrtype != AF_INET)
  {
  DEBUG(2) debug_printf("local host not TCP/IP\n");
  return;
  }

if ((sp = getservbyname("biff", "udp")) == NULL)
  {
  DEBUG(2) debug_printf("biff/udp is an unknown service");
  return;
  }

sa.sin_port = sp->s_port;
sa.sin_family = hp->h_addrtype;
memcpy(&sa.sin_addr, hp->h_addr, hp->h_length);

if ((fd = socket(AF_INET, SOCK_DGRAM, 0)) == -1)
  {
  DEBUG(2) debug_printf("failed to create comsat socket: %s\n",
    strerror(errno));
  return;
  }

sprintf(buffer, "%.200s@%d\n", user, offset);
if (sendto(fd, buffer, (int)strlen(buffer) + 1, 0, (struct sockaddr *)(&sa),
    sizeof(sa)) < 0)
  {
  DEBUG(2) debug_printf("send to comsat failed: %s\n", strerror(errno));
  }

close(fd);
}



/*************************************************
*       Write block and check for errors         *
*************************************************/

/* On failing to write the given number of bytes, but having written
some, try to write the rest in order to force an over-quota error to
appear. Otherwise no error is set.

Arguments:
  fd        file descriptor to write to
  buffer    pointer to block
  count     number of bytes

Returns:    TRUE if write successful
*/

static BOOL
write_block(int fd, char *buffer, int count)
{
int rc;
if ((rc = write(fd, buffer, count)) != count &&
    (rc < 0 || write(fd, buffer+rc, count-rc) != count-rc)) return FALSE;
return TRUE;
}



/*************************************************
*             Write formatted string             *
*************************************************/

/*
Arguments:
  fd          file descriptor
  format      string format
  ...         arguments for format

Returns:      the yield of write_block()
*/

static BOOL
write_string(int fd, char *format, ...)
{
va_list ap;
va_start(ap, format);
if (!string_vformat(big_buffer, BIG_BUFFER_SIZE, format, ap))
  log_write(0, LOG_PANIC_DIE, "overlong formatted string in appendfile");
va_end(ap);
return write_block(fd, big_buffer, strlen(big_buffer));
}



/*************************************************
*              Main entry point                  *
*************************************************/

/* See local README for general interface details.

Appendfile delivery is tricky and has led to various security problems in other
mailers. The logic used here is therefore laid out in some detail. When this
function is called, we are running in a subprocess which has had its gid and
uid set to the appropriate values. Therefore, we cannot write directly to the
exim logs. Any errors must be handled by setting appropriate return codes.
Note that the default setting for addr->transport_return is DEFER, so it need
not be set unless some other value is required.

(1)  If the addr->pfr flag is set and the local_part field starts with '/' then
     this is a delivery to a file after .forward or alias expansion. Otherwise,
     there must be a configured file name or directory name.

The following items apply in the case when a file name (as opposed to a
directory name) is given:

(2f) If the file name to be expanded contains a reference to $local_part,
     check that the user name doesn't contain a / because that would cause
     a reference to another directory.

(3f) Expand the file name.

(4f) If the file name is /dev/null, return success (optimization).

(5f) If a lock file is required, create it (see separate comments below about
     the algorithm for doing this). It is important to do this before opening
     the mailbox if NFS is in use.

(6f) Stat the file.

(7f) If the file already exists:

     If it's not a regular file, complain to local administrator and defer
     delivery with a local error code that causes subsequent delivery to
     be prevented until manually enabled.

     Open with O_WRONLY + O_APPEND, thus failing if the file has vanished.

     If open fails because the file does not exist, go to (6); on any other
     failure except EWOULDBLOCK, complain & defer. For EWOULDBLOCK (NFS
     failure), just defer.

(8f) If file does not exist initially:

     Open with O_WRONLY + O_EXCL + O_CREAT.

     If open fails because the file already exists, go to (6). To avoid looping
     for ever in a situation where the file is continuously being created and
     deleted, all of this happens inside a loop that operates lock_retries
     times and includes the fcntl locking. If the loop completes without the
     file getting opened, complain and defer with a code that freezes delivery
     attempts.

     If open fails for any other reason, defer for subsequent delivery except
     when this is a file delivery resulting from an alias or forward expansion
     and the error is EPERM or ENOENT or EACCES, in which case FAIL as this is
     most likely a user rather than a configuration error.

(9f) We now have the file checked and open for writing. Lock it using fcntl -
     if fcntl() fails, close the file and goto (6), up to lock_retries times,
     after sleeping for a while. If it still fails, give up and defer delivery.

(10f)Save the access time (for subsequent restoration) and the size of the
     file (for comsat and for re-setting if delivery fails in the middle -
     e.g. for quota exceeded).

The following items apply in the case when a directory name is given:

(2d) Create a new file in the directory using a temporary name, by opening for
     writing and with O_CREAT. If maildir format is being used, the file
     is created in a temporary subdirectory with a prescribed name.

The following items apply in all cases:

(11) We now have the file open for writing, and locked if it was given as a
     file name. Write the message and flush the file, unless there is a setting
     of the local quota option, in which case we can check for its excession
     without doing any writing.

     For any other error (most commonly disc full), do the same.

The following applies after appending to a file:

(12f)Restore the atime; notify_comsat if required; close the file (which
     unlocks it if it was locked).

The following applies after writing a unique file in a directory:

(12d)For non-maildir-format: Generate a unique name for the file from the time
     and the inode number, and rename the file after writing it. For maildir
     format, rename the file into the new directory.

This transport yields FAIL only when a file name is generated by an alias or
forwarding operation and attempting to open it gives EPERM, ENOENT, or EACCES.
All other failures return DEFER (in addr->transport_return). */


void
appendfile_transport_entry(
  transport_instance *tblock,      /* data for this instantiation */
  address_item *addr)              /* address we are working on */
{
appendfile_transport_options_block *ob =
  (appendfile_transport_options_block *)(tblock->options_block);
struct stat statbuf;
char *path;
char *filename;
char *hitchname;
char *lockname = NULL;     /* keep picky compiler happy */
char *newname = NULL;      /* ditto */
time_t now = time(NULL);
struct utimbuf times;
BOOL smtp_dots = FALSE;
BOOL return_path_add = ob->return_path_add;
BOOL delivery_date_add = ob->delivery_date_add;
BOOL envelope_to_add = ob->envelope_to_add;
BOOL isdirectory = FALSE;
int saved_size = 0;
int hd = -1;
int fd = -1;
int yield = FAIL;
int i, len;

DEBUG(9) debug_printf("appendfile transport entered\n");

/* See if this is the address_file or address_directory transport, used to
deliver to files specified via .forward or an alias file. If it is, assume
a directory-style delivery if the last character is '/'. (For address_file
this shouldn't ever be the case.) */

if (addr->pfr && addr->local_part[0] == '/')
  {
  path = addr->local_part;
  if (path[(int)strlen(path)-1] == '/') isdirectory = TRUE;
  }

/* Handle a non-address file delivery. One of the file or directory options is
mandatory; it will have already been checked to be an absolute path. */

else
  {
  char *fdname = ob->filename;
  if (fdname == NULL)
    {
    fdname = ob->dirname;
    isdirectory = TRUE;
    }

  if (fdname == NULL)
    {
    addr->transport_return = PANIC;
    addr->message = string_sprintf("Mandatory file or directory option "
      "missing from %s transport", tblock->name);
    return;
    }

  path = expand_string(fdname);

  if (path == NULL)
    {
    addr->transport_return = PANIC;
    addr->message = string_sprintf("Expansion of \"%s\" (file or directory "
      "name for %s transport) failed: %s", fdname, tblock->name,
      expand_string_message);
    return;
    }

  /* If the path contained a reference to $local_part, check that it
  doesn't attempt to change directory. Note that deliver_localpart may be
  unset during some deliveries (e.g. batched SMTP). */

  if (deliver_localpart != NULL &&
      strchr(deliver_localpart, '/') != NULL &&
     (strstr(fdname, "${local_part}") != NULL ||
      strstr(fdname, "$local_part") != NULL))
    {
    addr->message = "appendfile: user name contains /";
    addr->basic_errno = ERRNO_USERSLASH;
    addr->special_action = SPECIAL_FREEZE;
    return;
    }

  /* Check that the expanded name is absolute. This check is needed in case
  it started with an expansion item. If not, the check is done at
  initialization time. */

  if (path[0] != '/')
    {
    addr->message = string_sprintf("appendfile: file or directory name "
      "\"%s\" is not absolute", path);
    addr->basic_errno = ERRNO_NOTABSOLUTE;
    addr->special_action = SPECIAL_FREEZE;
    return;
    }
  }
  

DEBUG(9) debug_printf("appendfile: notify_comsat=%d quota=%d\n"
  "  %s=%s\n  prefix=%s\n  suffix=%s\n",
  ob->notify_comsat, ob->quota_value,
  isdirectory? "directory" : "file",
  path,
  (ob->prefix == NULL)? "null" : string_printing(ob->prefix, FALSE),
  (ob->suffix == NULL)? "null" : string_printing(ob->suffix, FALSE));


/* If the -N option is set, can't do any more. */

if (dont_deliver)
  {
  debug_printf("*** delivery by %s transport bypassed by -N option\n",
    tblock->name);
  addr->transport_return = OK;
  return;
  }

/* Handle the case of a file name. If the file name is /dev/null, we can save
ourselves some effort and just give a success return right away. */

if (!isdirectory)
  {
  BOOL file_opened = FALSE;

  if (strcmp(path, "/dev/null") == 0)
    {
    addr->transport_return = OK;
    return;
    }

  /* The name of the file to be opened */

  filename = path;

  /* The locking of mailbox files is worse than the naming of cats, which is
  known to be "a difficult matter" (T.S. Eliot) and just as cats must have
  three different names, so several different styles of locking are used.

  Research in other programs that lock mailboxes shows that there is no
  universally standard method. Having mailboxes NFS-mounted on the system that
  is delivering mail is not the best thing, but people do run like this,
  and so the code must do its best to cope.

  Two different locking approaches are taken. Unless no_use_lockfile is set,
  we attempt to build a lock file in a way that will work over NFS, and we
  also use fcntl. Failures to lock cause retries after a sleep, but only for
  a certain number of tries. Lockfiles should always be used with NFS, because
  of the following:

  The logic for creating the lock file is:

  . The name of the lock file is <mailbox-name>.lock

  . First, create a "hitching post" name by adding the primary host name,
    current time and pid to the lock file name. This should be unique.

  . Create the hitching post file using WRONLY + CREAT + EXCL.

  . If that fails EACCES, we assume it means that the user is unable to create
    files in the mail spool directory. Some installations might operate in this
    manner, so there is a configuration option to allow this state not to be an
    error - we proceed to lock using fcntl only, after the file is open.

  . Otherwise, an error causes a deferment of the address.

  . Rename the hitching post to the lock file name.

  . If the rename succeeds, we have successfully created the lock file.

  . If the rename does not succeed, proceed as follows:

    o Close and unlink it the hitching post file.

  . This method allows for the lock file to be created by some other process
    right up to the moment of the attempt to hard link it, and is also robust
    against NFS server crash-reboots, which would probably result in timeouts
    in the middle of rename().

  . System crashes may cause lock files to get left lying around, and some means
    of flushing them is required. The approach of writing a pid (used by smail
    and by elm) into the file isn't useful when NFS may be in use. Pine uses a
    timeout, which seems a better approach. Since any program that writes to a
    mailbox using a lock file should complete its task very quickly, Pine
    removes lock files that are older than 5 minutes. We allow the value to be
    configurable on the director. */


  /* Build a lock file if configured to do so - the existence of a lock
  file is subsequently checked by looking for a non-negative value of the
  file descriptor hd - even though the file is no longer open. */

  if (ob->use_lockfile)
    {
    len = (int)strlen(filename);
    lockname = store_malloc(len + 8);
    sprintf(lockname, "%s.lock", filename);
    hitchname = store_malloc(len + 32 + (int)strlen(primary_hostname));
    sprintf(hitchname, "%s.%s.%08x.%08x", lockname, primary_hostname,
      (unsigned int)now, (unsigned int)getpid());

    DEBUG(9) debug_printf("lock name: %s\nhitch name: %s\n", lockname,
      hitchname);

    /* Lock file creation retry loop */

    for (i = 0; i < ob->lock_retries; sleep(ob->lock_interval), i++)
      {
      int rc;
      hd = os_open(hitchname, O_WRONLY | O_CREAT | O_EXCL, S_IREAD | S_IWRITE);

      if (hd < 0)
        {
        if (errno != EACCES || ob->require_lockfile)
          {
          addr->basic_errno = errno;
          addr->message =
            string_sprintf("creating lock file hitching post %s", hitchname);
          return;
          }
        else break;
        }

      /* Attempt to hitch the hitching post to the lock file. If rename()
      succeeds (the common case, we hope) all is well. */

      close (hd) ;
      if ((rc = os_rename(hitchname, lockname)) != 0) os_unlink(hitchname);
      if (rc != 0)
        {
        if (ob->lockfile_timeout > 0 && stat(lockname, &statbuf) == 0 &&
            now - statbuf.st_ctime > ob->lockfile_timeout)
          {
          DEBUG(2) debug_printf("unlinking timed-out lock file\n");
          os_unlink(lockname);
          }
        DEBUG(9) debug_printf("rename of hitching post failed - retrying\n");
        continue;
        }

      DEBUG(9) debug_printf("lock file created\n");
      break;
      }

    /* Check for too many tries at creating the lock file */

    if (i >= ob->lock_retries)
      {
      addr->basic_errno = ERRNO_LOCKFAILED;
      addr->message = string_sprintf("failed to lock mailbox %s (lock file)",
        filename);
      return;
      }
    }


  /* We now have to get the file open and lock it with fcntl(). Stat the file,
  and act on existence or non-existence. This is in a loop to handle the case
  of a file's being created or deleted as we watch, and also to handle retries
  when the locking fails. Rather than holding the file open while waiting for
  the fcntl() lock, we close and do the whole thing again. This should be
  safer, especially for NFS files, which might get altered from other hosts,
  making their cached sizes incorrect. */

  for (i = 0; i < ob->lock_retries; i++)
    {
/* flock_t lock_data; */
    file_opened = FALSE;

    if(stat(filename, &statbuf) != 0)
      {
      /* Let's hope that failure to stat (other than non-existence) is a
      rare event. */

      if (errno != ENOENT)
        {
        addr->basic_errno = errno;
        addr->message = string_sprintf("attempting to stat mailbox %s",
          filename);
        goto RETURN;
        }

      /* File does not exist. If it is required to pre-exist this state is an
      error. */

      if (ob->file_must_exist)
        {
        addr->basic_errno = errno;
        addr->message = string_sprintf("mailbox %s does not exist, "
          "but file_must_exist is set", filename);
        addr->special_action = SPECIAL_FREEZE;
        goto RETURN;
        }

      /* If file creation is permitted in certain directories only, check that
      this is actually the case. Current checks are for in or below the
      home directory. */

      if (ob->create_file != create_anywhere)
        {
        BOOL OK = FALSE;
        if (deliver_home != NULL)
          {
          int len = (int)strlen(deliver_home);
	  char *file = filename;
          while (file[0] == '/' && file[1] == '/') file++;
          if (strncmp(file, deliver_home, len) == 0 && file[len] == '/' &&
            (ob->create_file == create_belowhome ||
              strchr(file+len+2, '/') == NULL)) OK = TRUE;
          }

        if (!OK)
          {
          addr->basic_errno = errno;
          addr->message = string_sprintf("mailbox %s does not exist, "
            "but creation outside the home directory is not permitted",
            filename);
          addr->special_action = SPECIAL_FREEZE;
          goto RETURN;
          }
        }

      /* Attempt to create and open the file. If open fails because of
      pre-existence, go round the loop again. For any other error, defer the
      address, except for an alias or forward generated file name with EPERM,
      ENOENT, or EACCES, as those are most likely to be user errors rather
      than Exim config errors. When a symbolic link is permitted and points
      to a non-existent file, we get here with use_lstat = FALSE. In this case
      we mustn't use O_EXCL, since it doesn't work. */

      fd = os_open(filename, O_WRONLY | O_APPEND | O_CREAT | O_EXCL, S_IREAD | S_IWRITE);
      if (fd < 0)
        {
        if (errno == EEXIST) continue;
        addr->basic_errno = errno;
        addr->message = string_sprintf("while creating mailbox %s",
          filename);
        if (addr->pfr && addr->local_part[0] == '/' &&
            (errno == EPERM || errno == ENOENT || errno == EACCES))
          addr->transport_return = FAIL;
        goto RETURN;
        }

      }


    /* The file already exists. */

    else
      {
      /* We are happy with the existing file. Open it, and then do further
      tests to ensure that it is the same file that we were just looking at.
      If the file does not now exist, restart this loop, going back to using
      lstat again. For an NFS error, just defer; other opening errors are
      more serious. */

      fd = os_open(filename, O_WRONLY | O_APPEND, S_IREAD | S_IWRITE);
      if (fd < 0)
        {
        if (errno == ENOENT)
          {
          continue;
          }
        addr->basic_errno = errno;
        if (errno != EWOULDBLOCK)
          {
          addr->message = string_sprintf("while opening mailbox %s", filename);
          addr->special_action = SPECIAL_FREEZE;
          }
        goto RETURN;
        }

      /* This fstat really shouldn't fail, as we have an open file! There's a
      dilemma here. We use fstat in order to be sure we are peering at the file
      we have got open. */

      if (fstat(fd, &statbuf) < 0)
        {
        addr->basic_errno = errno;
        addr->message = string_sprintf("attempting to stat open mailbox %s",
          filename);
        addr->special_action = SPECIAL_FREEZE;
        goto RETURN;
        }

      /* The file is OK. Carry on to do the locking. */
      }

    /* We now have an open file, and must lock it using fcntl if configured to
    do so. If a lock file is also required, it was created above and hd was
    left >= 0. At least one form of locking is required by the initialization
    function. If fcntl() fails, close the file and go round the loop all over
    again, after waiting for a bit. */

    file_opened = TRUE;
    if (!ob->use_fcntl) break;

/*
    lock_data.l_type = F_WRLCK;
    lock_data.l_whence = lock_data.l_start = lock_data.l_len = 0;
    if (fcntl(fd, F_SETLK, &lock_data) >= 0) break;
*/
    break ; /* fake successful fcntl */
    DEBUG(9) debug_printf("fcntl failed - retrying\n");
    close(fd);
    fd = -1;
    sleep(ob->lock_interval);
    }

  /* Test for exceeding the maximum number of tries. Either the file remains
  locked, or, if we haven't got it open, something is terribly wrong... */

  if (i >= ob->lock_retries)
    {
    if (!file_opened)
      {
      addr->basic_errno = ERRNO_EXISTRACE;
      addr->message = string_sprintf("mailbox %s: existence unclear", filename);
      addr->special_action = SPECIAL_FREEZE;
      }
    else
      {
      addr->basic_errno = ERRNO_LOCKFAILED;
      addr->message = string_sprintf("failed to lock mailbox %s (fcntl)",
        filename);
      }
    goto RETURN;
    }

  DEBUG(9) debug_printf("mailbox %s is locked\n", filename);

  /* Save access time (for subsequent restoration), modification time (for
  restoration if updating fails), size of file (for comsat and for re-setting if
  delivery fails in the middle - e.g. for quota exceeded). */

  if (fstat(fd, &statbuf) < 0)
    {
    addr->basic_errno = errno;
    addr->message = string_sprintf("while fstatting opened mailbox %s",
      filename);
    goto RETURN;
    }

  times.actime = statbuf.st_atime;
  times.modtime = statbuf.st_mtime;
  saved_size = statbuf.st_size;
  }

/* Handle the case of creating a unique file in a given directory. A temporary
name is used to create the file. Later, when it is written, the name is changed
to a unique one. There is no need to lock the file. An attempt is made to
create the directory if it does not exist. */

else if (!ob->maildir_format)
  {
  DEBUG(9) debug_printf("delivering to unique file (non-maildir)\n");
  filename = string_sprintf("%s/temp.%d.%s", path, (int)getpid(),
    primary_hostname);
  fd = os_open(filename, O_WRONLY|O_CREAT, S_IREAD | S_IWRITE);
  if (fd < 0 &&                                    /* failed to open, and */
      (errno != ENOENT ||                          /* either not exists */
       !ob->create_directory ||                    /* or not allowed to make */
       mkdir(path, 0) < 0 ||                       /* or failed to create dir */
       (fd = os_open(filename, O_WRONLY|O_CREAT, S_IREAD | S_IWRITE)) < 0)) /* or then failed to open */
    {
    addr->basic_errno = errno;
    addr->message = string_sprintf("while creating file %s", filename);
    return;
    }

  /* Build the new unique name. It starts with 'q' for smail compatibility.
  The fiddling around is because string_base62 returns its result in a fixed
  buffer always (exactly 6 chars plus zero). */

  if (stat(filename, &statbuf) < 0)
    {
    addr->basic_errno = errno;
    addr->message = string_sprintf("after stat() for %s", filename);
    goto RETURN;
    }

  strcpy(big_buffer, string_base62(time(NULL)));
  newname = string_sprintf("%s/q%s-%s", path, big_buffer,
    string_base62(statbuf.st_ino));
  }

/* Handle the case of a unique file in maildir format. The file is written to
the tmp subdirectory, with a prescribed form of name. */

else
  {
  char *subdirs[] = { "/tmp", "/new", "/cur" };

  DEBUG(9) debug_printf("delivering in maildir format in %s\n", path);

  /* First ensure that the path we have is a directory; if it does not exist,
  create it. Then make sure the tmp, new & cur subdirs of the maildir are
  there. If not we abort the delivery (even though the cur subdir is not
  actually needed for delivery). Handle all 4 directory tests/creates in a
  loop so that code can be shared. */

  for (i = 0; i < 4; i++)
    {
    int j;
    char *dir, *mdir;

    if (i == 0)
      {
      mdir = "";
      dir = path;
      }
    else
      {
      mdir = subdirs[i-1];
      dir = mdir + 1;
      }

    /* Check an existing path is a directory. This is inside a loop because
    there is a potential race condition when creating the directory - some
    other process may get there first. Give up after trying several times,
    though. */

    for (j = 0; j < 10; j++)
      {
      if (stat(dir, &statbuf) == 0)
        {
        if (S_ISDIR(statbuf.st_mode)) break;   /* out of the race loop */
        addr->message = string_sprintf("%s%s is not a directory", path,
          mdir);
        addr->basic_errno = ERRNO_NOTDIRECTORY;
        addr->special_action = SPECIAL_FREEZE;
        return;
        }

      /* Try to make if non-existent and configured to do so */

      if (errno == ENOENT && ob->create_directory)
        {
        if (mkdir(dir, 0) != 0)
          {
          if (errno == EEXIST) continue;     /* repeat the race loop */
          addr->message = string_sprintf ("cannot mkdir %s%s", path, mdir);
	  addr->basic_errno = errno;
	  addr->special_action = SPECIAL_FREEZE;
	  return;
          }
        DEBUG(9) debug_printf("created directory %s%s\n", path, mdir);
        break;   /* out of the race loop */
        }

      /* stat() error other than ENOENT, or ENOENT and not creatable */

      addr->message = string_sprintf("stat() error for %s%s", path, mdir);
      addr->basic_errno = errno;
      addr->special_action = SPECIAL_FREEZE;
      return;
      }

    /* If we went round the loop 10 times, the directory was flickering in and
    out of existence like someone in a malfunctioning Star Trek transporter. */

    if (j >= 10)
      {
      addr->message = string_sprintf("existence of %s%s unclear\n");
      addr->basic_errno = errno;
      addr->special_action = SPECIAL_FREEZE;
      return;
      }

    /* First time through the directories loop, cd to the main directory */

    if (i == 0 && chdir(path) != 0)
      {
      addr->message = string_sprintf ("cannot chdir to %s", path);
      addr->basic_errno = errno;
      addr->special_action = SPECIAL_FREEZE;
      return;
      }
    }

  /* All relevant directories now exist. Attempt to open the temporary
  file a limited number of times. I think this rather scary-looking for
  statement is actually OK. If open succeeds, the loop is broken; if not,
  there is a test on the value of i. */

  for (i = 1;; i++)
    {
    char *basename = string_sprintf("%lu.%lu.%s", time(NULL), getpid(),
      primary_hostname);

    filename = string_sprintf("tmp/%s", basename);
    newname = string_sprintf("new/%s", basename);

    if (stat(filename, &statbuf) == 0)
      errno = EEXIST;
    else if (errno == ENOENT)
      {
      fd = os_open(filename, O_WRONLY | O_CREAT | O_EXCL, S_IREAD | S_IWRITE);
      if (fd >= 0) break;
      DEBUG (9) debug_printf ("open failed for %s: %s\n",
        filename, strerror(errno));
      }

    if (i >= ob->maildir_retries)
      {
      addr->message = string_sprintf ("failed to open %s (%d tr%s)",
        filename, i, (i == 1)? "y" : "ies");
      addr->basic_errno = errno;
      addr->transport_return = DEFER;
      return;
      }

    /* Open or stat failed but we haven't tried too many times yet. */

    store_free(basename);
    store_free(filename);
    store_free(newname);
    sleep(2);
    }

  }


/* At last we can write the message to the file, preceded by any configured
prefix line, and followed by any configured suffix line. If there are any
writing errors, we must defer. */

DEBUG(9) debug_printf("writing to file %s\n", filename);

yield = OK;
errno = 0;

/* If there is a local quota setting, check that we are not going to exceed it
with this message. This of course applies only file appending, not when writing
in a directory. */

if (ob->quota_value > 0)
  {
  if (saved_size + message_size > ob->quota_value)
    {
    yield = DEFER;
    errno = ERRNO_EXIMQUOTA;
    }
  }

/* If the local_smtp option is not unset, we need to write SMTP prefix
information. The various different values for batching are handled outside; if
there is more than one address available here, all must be included. If any
address is a file, use it's parent in the RCPT TO. Force SMTP dot-handling. */

if (tblock->local_smtp != local_smtp_off && yield == OK)
  {
  smtp_dots = TRUE;
  return_path_add = delivery_date_add = envelope_to_add = FALSE;
  if (tblock->bsmtp_helo)
    {
    write(fd, "HELO ", 5);
    write(fd, primary_hostname, (int)strlen(primary_hostname));
    write(fd, "\n", 1);
    }
  if (!write_string(fd, "MAIL FROM: <%s>\n",
    (addr->errors_address != NULL)? addr->errors_address : sender_address))
      yield = DEFER;
  else
    {
    address_item *a;
    for (a = addr; a != NULL; a = a->next)
      {
      address_item *b = (a->pfr)? a->parent: a;
      if ((b->local_part[0] == ',' || b->local_part[0] == ':')?
        !write_string(fd,
          "RCPT TO: <@%s%s>\n", b->domain, b->local_part) :
        !write_string(fd,
          "RCPT TO: <%s@%s>\n", b->local_part, b->domain))
        { yield = DEFER; break; }
      }
    if (yield == OK && !write_string(fd, "DATA\n")) yield = DEFER;
    }
  }

/* Now any other configured prefix. */

if (yield == OK && ob->prefix != NULL)
  {
  char *prefix = expand_string(ob->prefix);
  if (prefix == NULL)
    {
    addr->transport_return = PANIC;
    addr->message = string_sprintf("Expansion of \"%s\" (prefix for %s "
      "transport) failed", ob->prefix, tblock->name);
    goto RETURN;
    }
  if (!write_string(fd, "%s", prefix)) yield = DEFER;
  }

/* The options to write_message request a return-path header if configured, the
unix "from" hack if configured, no CRs added before LFs, and no SMTP dot
handling except when local_smtp is set. Pass the errors_address from the
address if present. This is used to create return-path if requested (forced
off for local_smtp). */

if (yield == OK &&
  !transport_write_message(addr, fd,
    topt_writing_file |
      (return_path_add? topt_add_return_path : 0) |
      (delivery_date_add? topt_add_delivery_date : 0) |
      (envelope_to_add? topt_add_envelope_to : 0 ) |
      (ob->from_hack? topt_from_hack : 0 ) |
      (smtp_dots? topt_smtp_dots : 0),
    addr->errors_address, 0, tblock->add_headers, tblock->remove_headers))
  yield = DEFER;

/* Now a configured suffix. */

if (yield == OK && ob->suffix != NULL)
  {
  char *suffix = expand_string(ob->suffix);
  if (suffix == NULL)
    {
    addr->transport_return = PANIC;
    addr->message = string_sprintf("Expansion of \"%s\" (suffix for %s "
      "transport) failed", ob->suffix, tblock->name);
    goto RETURN;
    }
  if (!write_string(fd, "%s", suffix)) yield = DEFER;
  }

/* If local_smtp, write the terminating dot. */

if (yield == OK && tblock->local_smtp != local_smtp_off &&
  !write_block(fd, ".\n", 2)) yield = DEFER;

/* Force out the remaining data to check for any errors */

if (yield == OK && fsync(fd) < 0) yield = DEFER;

/* Handle error while writing the file. Control should come here directly after
the error, with the reason in errno. */

if (yield != OK)
  {
  /* Save the error number. If positive, it will ultimately cause a strerror()
  call to generate some text. */

  addr->basic_errno = errno;

  /* Handle system quota excession. Set more_errno to the time since the file
  was last read, and add an explanatory phrase for the error message, since
  some systems don't have special quota-excession errors. */

  if (errno == errno_quota)
    {
    addr->more_errno = time(NULL) - times.actime;
    #ifndef EDQUOT
    addr->message = string_sprintf("probably user's quota exceeded while "
      "writing to file %s", filename);
    #endif
    DEBUG(9) debug_printf("System quota exceeded for %s: time since file "
      "read = %s\n", filename, readconf_printtime(addr->more_errno));
    }

  /* Handle Exim's own quota-imposition */

  else if (errno == ERRNO_EXIMQUOTA)
    {
    addr->more_errno = time(NULL) - times.actime;
    addr->message = string_sprintf("MTA-imposed quota exceeded while "
      "writing to file %s", filename);
    DEBUG(9) debug_printf("Exim quota exceeded for %s: time since file "
      "read = %s\n", filename, readconf_printtime(addr->more_errno));
    }

  /* Handle a process failure while writing via a filter; the return
  from child_close() is in more_errno. */

  else if (errno == ERRNO_FILTER_FAIL)
    addr->message = string_sprintf("filter process failure %d while writing "
      "to %s", addr->more_errno, filename);

  /* Handle failure to add a header - usually expansion failure. */

  else if (errno == ERRNO_ADDHEADER_FAIL)
    {
    yield = PANIC;
    addr->message = string_sprintf("failed to expand added headers while "
      "writing to %s", filename);
    }

  /* For other errors, a general-purpose explanation */

  else addr->message = string_sprintf("error while writing to %s",
    filename);

  /* For a file, reset the file size to what it was before we started, leaving
  the last modification time unchanged, so it will get reset also. All systems
  investigated so far have ftruncate(), whereas not all have the F_FREESP
  fcntl() call (BSDI & FreeBSD do not). */

  if (!isdirectory) ftruncate(fd, saved_size);
  }

/* Handle successful writing - we want the modification time to be now for
appended files. Remove the default backstop error number. For a directory, now
is the time to rename the file with a unique name. As soon as such a name
appears it may get used by another process, so we close the file first and
check that all is well. */

else
  {
  times.modtime = now;
  addr->basic_errno = 0;

  /* Handle the case of writing to a new file in a directory. This applies
  both to maildir and non-maildir formats. */

  if (isdirectory)
    {
    if (fstat(fd, &statbuf) < 0)
      {
      addr->basic_errno = errno;
      addr->message = string_sprintf("while fstatting opened message file %s",
        filename);
      yield = DEFER;
      }

    else if (close(fd) < 0)
      {
      addr->basic_errno = errno;
      addr->message = string_sprintf("close() error for %s", filename);
      yield = DEFER;
      }

    else
      {
      fd = -1;
      if (os_rename(filename, newname) < 0)
        {
        addr->basic_errno = errno;
        addr->message = string_sprintf("while renaming %s as %s",
          filename, newname);
        yield = DEFER;
        }
      else
        {
        DEBUG(9) debug_printf("renamed %s as %s\n", filename, newname);
        filename = NULL;   /* Prevents attempt to unlink at end */
        }
      }
    }
  }


/* For a file, restore the last access time (atime), and set the modification
time as required - changed if write succeeded, unchanged if not. */

if (!isdirectory) utime(filename, &times);

/* Notify comsat if configured to do so. It only makes sense if the configured
file is the one that the comsat daemon knows about. */

if (ob->notify_comsat && yield == OK && deliver_localpart != NULL)
  notify_comsat(deliver_localpart, saved_size);

/* Pass back the final return code in the address structure */

DEBUG(2) debug_printf("appendfile yields %d with errno=%d more_errno=%d\n",
  yield, addr->basic_errno, addr->more_errno);

addr->transport_return = yield;

/* Close the file, which will release the fcntl lock. For a directory write it
is closed above, except in cases of error which goto RETURN, when we also need
to remove the original file. If everything has gone right but close fails,
defer the message. Then unlink the lock file, if present. This point in the
code is jumped to from a number of places when errors are detected, in order to
get the file closed and the lock file tidied away. */

RETURN:

if (fd >= 0 && close(fd) < 0 && yield == OK)
  {
  addr->basic_errno = errno;
  addr->message = string_sprintf("while closing %s", filename);
  addr->transport_return = DEFER;
  }

if (hd >= 0) os_unlink(lockname);
if (isdirectory && filename != NULL) os_unlink(filename);
}

/* End of transport/appendfile.c */
