/*************************************************
*     Exim - an Internet mail transport agent    *
*************************************************/

/* Copyright (c) University of Cambridge 1995 - 1997 */
/* See the file NOTICE for conditions of use and distribution. */

/* Functions for handling an incoming SMTP call. */


#include "exim.h"


#define cmd_buffer_size 512      /* Ref. RFC 821 */


/* Structure for SMTP command list */

typedef struct {
  char *name;
  int len;
  int cmd;
} smtp_cmd_list;

/* Codes for identifying commands */

enum { HELO_CMD, EHLO_CMD, MAIL_CMD, RCPT_CMD, DATA_CMD, VRFY_CMD,
  EXPN_CMD, QUIT_CMD, RSET_CMD, NOOP_CMD, DEBUG_CMD, HELP_CMD,
  ETRN_CMD, EOF_CMD, OTHER_CMD };



/*************************************************
*                Local static variables          *
*************************************************/

static BOOL host_allow_relay_anywhere;
static BOOL host_allow_relay_anywhere_set;
static BOOL nonrelay_host_failed_accept;
static BOOL sender_allow_relay_anywhere;
static BOOL sender_allow_relay_anywhere_set;

static BOOL host_refuse_all_rcpts;
static BOOL sender_refuse_all_rcpts;

static char *smtp_data;

static char cmd_buffer[cmd_buffer_size + 1];

static smtp_cmd_list cmd_list[] = {
  { "helo",       sizeof("helo")-1,       HELO_CMD },
  { "ehlo",       sizeof("ehlo")-1,       EHLO_CMD },
  { "mail from:", sizeof("mail from:")-1, MAIL_CMD },
  { "rcpt to:",   sizeof("rcpt to:")-1,   RCPT_CMD },
  { "data",       sizeof("data")-1,       DATA_CMD },
  { "vrfy",       sizeof("vrfy")-1,       VRFY_CMD },
  { "expn",       sizeof("expn")-1,       EXPN_CMD },
  { "quit",       sizeof("quit")-1,       QUIT_CMD },
  { "rset",       sizeof("rset")-1,       RSET_CMD },
  { "noop",       sizeof("noop")-1,       NOOP_CMD },
  { "debug",      sizeof("debug")-1,      DEBUG_CMD },
  { "help",       sizeof("help")-1,       HELP_CMD },
  { "etrn",       sizeof("etrn")-1,       ETRN_CMD} };

static smtp_cmd_list *cmd_list_end =
  cmd_list + sizeof(cmd_list)/sizeof(smtp_cmd_list);



/*************************************************
*          SMTP command read timeout             *
*************************************************/

/* Signal handler for timing out incoming SMTP commands

Argument: signal number (SIGALRM)
Returns:  nothing
*/

static void
command_timeout_handler(int sig)
{
if (!smtp_batched_input)
  {
  DEBUG(3) debug_printf("421 %s: SMTP command timeout - closing connection\n",
    primary_hostname);
  fprintf(smtp_out, "421 %s: SMTP command timeout - closing connection\r\n",
    primary_hostname);
  fflush(smtp_out);
  }
log_write(4, LOG_MAIN, "SMTP command timeout%s%s",
  (sender_fullhost != NULL)? " while connected to " : "",
  (sender_fullhost != NULL)? sender_fullhost : "");
exit(EXIT_FAILURE);
}



/*************************************************
*               SIGTERM received                 *
*************************************************/

/* Signal handler for handling SIGTERM

Argument: signal number (SIGTERM)
Returns:  nothing
*/

static void
command_sigterm_handler(int sig)
{
if (!smtp_batched_input)
  {
  DEBUG(3) debug_printf("421 %s: Service not available - closing connection\n",
    primary_hostname);
  fprintf(smtp_out, "421 %s: Service not available - closing connection\r\n",
    primary_hostname);
  }
log_write(0, LOG_MAIN, "SMTP connection closed after SIGTERM%s%s",
  sender_fullhost? " while connected to " : "",
  sender_fullhost? sender_fullhost : "");
exit(EXIT_FAILURE);
}



/*************************************************
*           Read one command line                *
*************************************************/

/* Strictly, SMTP commands coming over the net are supposed to end with CRLF.
There are sites that don't do this, and in any case internal SMTP probably
should check only for LF. Consequently, we check here for LF only. The line
ends up with [CR]LF removed from its end. If we get an overlong line, treat as
an unknown command. Carry on in the face of interrupts. (There appears to be no
standard way of disabling them.) The command is read into the static
cmd_buffer.

Arguments:  none
Returns:    a code identifying the command (enumerated above)
*/

static int
smtp_read_command(void)
{
int c;
int ptr = 0;
smtp_cmd_list *p;

alarm(smtp_receive_timeout);

while ((c = getc(smtp_in)) != '\n' && c != EOF)
  {
  if (ptr >= cmd_buffer_size) { alarm(0); return OTHER_CMD; }
  cmd_buffer[ptr++] = c;
  }
alarm(0);

/* If hit end of file, return pseudo EOF command. Whether we have a
part-line already read doesn't matter, since this is an error state. */

if (c == EOF) return EOF_CMD;

/* Remove any CR and white space at the end of the line, and terminate the
string. */

while (ptr > 0 && isspace(cmd_buffer[ptr-1])) ptr--;
cmd_buffer[ptr] = 0;

DEBUG(3) debug_printf("SMTP<< %s\n", cmd_buffer);

/* Scan command list and return identity, having set the data pointer
to the start of the actual data characters. */

for (p = cmd_list; p < cmd_list_end; p++)
  {
  if (strncmpic(cmd_buffer, p->name, p->len) == 0)
    {
    smtp_data = cmd_buffer + p->len;
    while (isspace(*smtp_data)) smtp_data++;
    return p->cmd;
    }
  }

return OTHER_CMD;
}



/*************************************************
*          Forced closedown of call              *
*************************************************/

/* This function is called from log.c when Exim is dying because of a serious
disaster. If an incoming non-batched SMTP channel is open, it sends the reply
string, and gives an error to all subsequent commands except QUIT. The
existence of an SMTP call is detected by the non-NULLness of smtp_in.

Argument:   SMTP reply string to send, excluding the code
Returns:    nothing
*/

void
smtp_closedown(char *message)
{
if (smtp_in == NULL || smtp_batched_input) return;
fprintf(smtp_out, "421 %s\r\n", message);
fflush(smtp_out);
for (;;)
  {
  switch(smtp_read_command())
    {
    case EOF_CMD:
    case QUIT_CMD:
    return;

    default:
    fprintf(smtp_out, "503 Command out of sequence\r\n");
    fflush(smtp_out);
    break;
    }
  }
}




/*************************************************
*  Initialize for incoming batched SMTP message  *
*************************************************/

/* This function is called from smtp_setup_msg() in the case when
smtp_batched_input is true. This happens when -bS is used to pass a whole batch
of messages in one file with SMTP commands between them. All errors must be
reported by sending a message, and only MAIL FROM, RCPT TO, and DATA are
relevant. After an error on a sender, or an invalid recipient, the remainder
of the message is skipped.

Argument: none
Returns:  > 0 message successfully started (reached DATA)
          = 0 QUIT read or end of file reached
          < 0 should not occur
*/

static int
smtp_setup_batch_msg()
{
BOOL skipping = FALSE;
BOOL toomany = FALSE;
int done = 0;
int rcount = 0;

if (feof(smtp_in)) return 0;       /* Treat EOF as QUIT */

received_protocol = "bsmtp";
check_relay = FALSE;           /* No relay checking, whatever config says */
raw_sender = NULL;             /* No sender yet */

/* Deal with SMTP commands. The reading routine sets up a timeout
for each one. If the timeout happens, or we get SIGTERM, exim just
gives up and dies. */

signal(SIGALRM, command_timeout_handler);
signal(SIGTERM, command_sigterm_handler);

/* This loop is exited by setting done to a POSITIVE value. The values
are 2 larger than the required yield of the function. */

while (done <= 0)
  {
  char *errmess;
  char *receiver = NULL;
  char *orig_receiver = NULL;
  int errcode, start, end, sender_domain, receiver_domain;

  switch(smtp_read_command())
    {
    /* The HELO/EHLO commands are simply ignored, except that they do
    a reset of the state. */

    case HELO_CMD:
    case EHLO_CMD:
    case RSET_CMD:
    accept_free_recipients();
    sender_address = NULL;
    skipping = toomany = FALSE;
    break;


    /* The MAIL FROM command requires an address as an operand. All we
    do here is to parse it for syntactic correctness. The form "<>" is
    a special case which converts into an empty string. The start/end
    pointers in the original are not used further for this address, as
    it is the canonical extracted address which is all that is kept. */

    case MAIL_CMD:
    skipping = toomany = FALSE;
    if (sender_address != NULL)
      {
      moan_smtp_batch(cmd_buffer, "503 Sender already given");
      skipping = TRUE;
      break;
      }

    if (smtp_data[0] == 0)
      {
      moan_smtp_batch(cmd_buffer, "501 MAIL FROM must have an address operand");
      skipping = TRUE;
      break;
      }

    /* The TRUE flag allows "<>" as a sender address */

    raw_sender =
      parse_extract_address(smtp_data, &errmess, &start, &end, &sender_domain,
        TRUE);
    if (raw_sender == NULL)
      {
      moan_smtp_batch(cmd_buffer, "501 %s", errmess);
      skipping = TRUE;
      break;
      }
    sender_address = string_copy(raw_sender);

    /* Qualify unqualified sender addresses. There doesn't seem much point
    in causing trouble here. */

    if (sender_domain == 0 && sender_address[0] != 0 && sender_address[0] != '@')
      {
      sender_address = rewrite_address_qualify(sender_address, FALSE);
      DEBUG(9) debug_printf("unqualified address %s accepted\n",
        raw_sender);
      }

    /* If configured to check sender addresses, do the preliminary check
    now, unless the sender is local (in which case whatever is given here
    is ignored anyway). The check will fail if the message is to be refused at
    this stage. Another check function is called after the message has been
    received, to do more checking when the headers are available. */

    errmess = NULL;
    refuse_all_rcpts = FALSE;

    if (sender_verify || sender_try_verify)
      {
      if (!sender_local && !verify_sender_preliminary(&errcode, &errmess))
        {
        moan_smtp_batch(cmd_buffer, "%d rejected MAIL FROM: %s <%s>\n",
          errcode, errmess, raw_sender);
        log_write(1, LOG_MAIN|LOG_REJECT, "rejected MAIL FROM in SMTP batch: "
          "%s <%s>", errmess, raw_sender);
        store_free(sender_address);
        sender_address = NULL;
        skipping = TRUE;
        break;
        }
      }

    /* RFC 821 says MAIL FROM resets state at start of message */
    accept_free_recipients();
    rcount = 0;
    break;


    /* The RCPT TO command requires an address as an operand. All we do
    here is to parse it for syntactic correctness. There may be any number
    of RCPT TO commands, specifying multiple senders. We build them all into
    a data structure that is in argc/argv format. The start/end values
    given by parse_extract_address are not used, as we keep only the
    extracted address. */

    case RCPT_CMD:
    if (skipping) break;     /* Ignore if bad sender */

    if (sender_address == NULL)
      {
      moan_smtp_batch(cmd_buffer, "503 No sender yet given");
      skipping = TRUE;
      break;
      }

    if (smtp_data[0] == 0)
      {
      moan_smtp_batch(cmd_buffer, "501 RCPT TO must have an address operand");
      skipping = TRUE;
      break;
      }

    if (refuse_all_rcpts)
      {
      moan_smtp_batch(cmd_buffer, "550 cannot route to sender address <%s>",
        sender_address);
      skipping = TRUE;
      break;
      }

    /* Check maximum number allowed */

    if (recipients_max > 0 && rcount + 1 > recipients_max)
      {
      moan_smtp_batch(cmd_buffer, "%s too many recipients",
        recipients_max_reject? "550": "421");
      toomany = skipping = TRUE;
      break;
      }

    /* Don't allow "<>" as a recipient address */

    orig_receiver =
      parse_extract_address(smtp_data, &errmess, &start, &end,
        &receiver_domain, FALSE);
    if (orig_receiver == NULL)
      {
      moan_smtp_batch(cmd_buffer, "501 %s", errmess);
      skipping = TRUE;
      break;
      }
    receiver = string_copy(orig_receiver);

    /* If the receiver address is unqualified, qualify it. There doesn't seem
    much point in causing trouble here. */

    if (receiver_domain == 0 && receiver[0] != '@')
      {
      DEBUG(9) debug_printf("unqualified address %s accepted\n",
        receiver);
      receiver = rewrite_address_qualify(receiver, TRUE);
      }

    /* Add to the list of receivers, and set value to NULL to prevent
    freeing. */

    accept_add_recipient(receiver);
    rcount++;
    receiver = NULL;
    break;


    /* The DATA command is legal only if it follows successful MAIL FROM
    and RCPT TO commands. This function is complete when a valid DATA
    command is encountered. If skipping is set, it means there was a
    bad MAIL FROM, or RCPT TO, and some recipients were probably ignored.

    If the count of recipients is zero, they were all bad (or ignored). If
    toomany is set, there were too many recipients, and if
    recipients_max_reject is true, the entire message must be rejected. In all
    cases, we must skip over the remainder of this message and then try again
    for the next message. */

    case DATA_CMD:
    if (skipping || sender_address == NULL || recipients_count <= 0 ||
        (toomany && recipients_max_reject))
      {
      if (sender_address == NULL)
        moan_smtp_batch(cmd_buffer, "503 MAIL FROM command must precede DATA");
      else if (toomany)
        moan_smtp_batch(cmd_buffer, "554 Too many recipients");
      else if (!skipping)
        moan_smtp_batch(cmd_buffer,
          "503 Valid RCPT TO <recipient> must precede DATA");
      accept_read_message_data_smtp(smtp_in, NULL, NULL);
      accept_free_recipients();
      sender_address = NULL;
      skipping = FALSE;
      break;
      }
    done = 3;
    break;


    /* The VRFY, EXPN, HELP, ETRN, and DEBUG commands are ignored. */

    case VRFY_CMD:
    case EXPN_CMD:
    case HELP_CMD:
    case DEBUG_CMD:
    case NOOP_CMD:
    case ETRN_CMD:
    break;


    case EOF_CMD:
    case QUIT_CMD:
    accept_free_recipients();
    done = 2;
    break;


    default:
    moan_smtp_batch(cmd_buffer, "500 Command unrecognized");
    skipping = TRUE;
    break;
    }

  /* Free temporary store. */

  if (orig_receiver != NULL) store_free(orig_receiver);
  if (receiver != NULL) store_free(receiver);
  }

/* Reset the signal handlers used in this function, and if no
message is in progress, ensure the store is cleaned up. */

signal(SIGALRM, SIG_DFL);
signal(SIGTERM, SIG_DFL);

if (done < 3)
  {
  if (raw_sender != NULL)
    {
    store_free(raw_sender);
    raw_sender = NULL;
    }
  accept_free_recipients();
  }

return done - 2;  /* Convert yield values */
}




/*************************************************
*          Build host+ident message              *
*************************************************/

/* Used when logging rejections below and in other modules

Arguments: none
Returns:   pointer to host name and ident value, if present
*/

char *
host_and_ident(void)
{
return (sender_ident == NULL ||
  !string_format(big_buffer, BIG_BUFFER_SIZE, "%s (%s)", sender_fullhost,
    sender_ident))?
      sender_fullhost : big_buffer;
}





/*************************************************
*        Output tailored rejection message       *
*************************************************/

/* This function is called when an error caused by an administrative
prohibition is about to be sent down an SMTP channel. It gives the
administrator a change to send additional information. If prohibition_message
is set, it is expanded and written as a series of continued response lines,
assuming the the standard response will be output afterwards.

Arguments:
  f          the file to write to
  errorcode  the SMTP error code
  reason     a value to put in $prohibition_reason for the expansion

Returns:     nothing
*/

void
smtp_send_prohibition_message(FILE *f, int errorcode, char *reason)
{
char *msg, *s;

if (prohibition_message == NULL) return;

prohibition_reason = reason;
msg = expand_string(prohibition_message);
prohibition_reason = NULL;

if (msg == NULL)
  {
  log_write(0, LOG_MAIN|LOG_PANIC,
    "Expansion of \"%s\" (prohibition_message) failed: %s",
    prohibition_message, expand_string_message);
  return;
  }

s = msg;
while (isspace((uschar)*s)) s++;
if (*s == 0) return;            /* Empty message ignored */

while (*s != 0)
  {
  char *ss = s + 1;
  while (*ss != 0 && *ss != '\n' && *ss != '|') ss++;
  fprintf(f, "%d-%.*s\r\n", errorcode, ss-s, s);
  DEBUG(3) debug_printf("%d-%.*s\r\n", errorcode, ss-s, s);
  if (*ss == 0) break;
  s = ss + 1;
  while (isspace((uschar)*s)) s++;
  }

store_free(msg);
}




/*************************************************
*       Initialize for SMTP incoming message     *
*************************************************/

/* This function conducts the initial dialogue at the start of an incoming SMTP
message, and builds a list of recipients. However, if the incoming message
is part of a batch (-bS option) a separate function is called since it would
be messy having tests splattered about all over this function. This function
therefore handles the case where interaction is occurring. The input and output
files are set up in smtp_in and smtp_out.

The global recipients_list is set to point to a vector of string pointers,
whose number is given by recipients_count. The global variable sender_address
is set to the sender's address. The yield is +1 if a message has been
successfully started, 0 if a QUIT command was encountered or the connection was
refused from the particular host, or -1 if the connection was lost.

If there are host accept/reject configuration settings, this is where
they are checked.

Argument:
  first   TRUE for the first message read on this connection; this causes
            various connection-oriented checks to be done that can be skipped
            for subsequent messages

Returns:  > 0 message successfully started (reached DATA)
          = 0 QUIT read or end of file reached or call refused
          < 0 lost connection
*/

int
smtp_setup_msg(BOOL first)
{
int done = 0;
int rcount = 0;
BOOL toomany = FALSE;
BOOL helo_seen = FALSE;
char rbl_msg_buffer[128];

DEBUG(2) debug_printf("smtp_setup_msg entered\n");

/* Initialize */

accept_free_recipients();
sender_address = NULL;
raw_sender = NULL;

/* Batched SMTP is handled in a different function. */

if (smtp_batched_input) return smtp_setup_batch_msg();


/* The following code is obeyed for the first message in the SMTP connection
only. */

if (first)
  {
  char *p, *s;

  received_protocol = "smtp";        /* Reset later if EHLO is received */
  host_allow_relay_anywhere_set = FALSE;
  host_refuse_all_rcpts = FALSE;
  rbl_msg_buffer[0] = 0;

  /* When a message is input locally via the -bs option, sender_host_unknown
  is set. In this case no checking of acceptable hosts is done. Otherwise
  obey the code that deals with IP source routing, if configured to do so. Then
  check for forbidden hosts and reject the call if it is one of them. When -bs
  is used from inetd, this flag is not set, causing the sending host to be
  checked. Batched input, using the -bS option, calls a different function and
  does not come through here.

  (1) If sender_{host,net}_accept is set, reject a connection not from those
  hosts/nets. Otherwise accept a connection from any host that is not in
  sender_{host,net}_reject. However, if the connection is from a host listed in
  sender_{host,net}_reject_recipients, set the flag to reject all recipients,
  that is, to fail all messages (as opposed to rejecting connections). There
  can be ident values associated with any host.

  (2) If smtp_accept_max and smtp_accept_reserve are set, keep some connections
  in reserve for certain hosts and/or networks.

  (3) If check_relay is set, further checks are done on individual recipient
  addresses, at RCPT TO time, in an order which is supposed to minimize the
  work done, which is why nothing is done here - the host doesn't need to
  be checked unless the domain is a restricted one. */

  if (sender_host_unknown) check_relay = FALSE; else
    {
    BOOL reserved_host = FALSE;
    BOOL hostcheck;

    /* If the current host matches host_lookup_nets, set the name by doing a
    reverse lookup. On failure, sender_host_name will be NULL. This may or
    may not be serious - optional checks later. */

    if (verify_check_net(host_lookup_nets, &host_lookup_netlist))
      {
      sender_host_name = host_find_byaddr(sender_host_address);
      host_build_sender_fullhost();
      }

    /* Reject connections from any host not in an accept list. */

    if (sender_host_accept != NULL || sender_net_accept != NULL)
      {
      if (!verify_check_host(sender_host_accept, &sender_host_accept_hosts,
              FALSE) &&
          !verify_check_net(sender_net_accept, &sender_net_accept_nets))
        {
        log_write(1, LOG_MAIN|LOG_REJECT, "connection from %s refused "
          "(accept)%s", host_and_ident(), host_findbyaddr_msg);
        smtp_send_prohibition_message(smtp_out, 554, "host_accept");
        fprintf(smtp_out, "554 SMTP service not available%s\r\n",
          host_findbyaddr_msg);
        DEBUG(3) debug_printf("554 SMTP service not available%s\n",
          host_findbyaddr_msg);
        return 0;
        }
      }

    /* Reject connections from any host in a reject list. */

    if (sender_host_reject != NULL || sender_net_reject != NULL)
      {
      if ((hostcheck = verify_check_host(sender_host_reject,
            &sender_host_reject_hosts, TRUE)) == TRUE ||
          verify_check_net(sender_net_reject, &sender_net_reject_nets))
        {
        log_write(1, LOG_MAIN|LOG_REJECT, "connection from %s refused "
          "(reject %s)%s", host_and_ident(), hostcheck? "host" : "net",
          host_findbyaddr_msg);
        smtp_send_prohibition_message(smtp_out, 554, "host_reject");
        fprintf(smtp_out, "554 SMTP service not available%s\r\n",
          host_findbyaddr_msg);
        DEBUG(3) debug_printf("554 SMTP service not available%s\n",
          host_findbyaddr_msg);
        return 0;
        }
      }

    /* Reject recipients from any host in a reject_recipients list. */

    if (sender_host_reject_recipients != NULL ||
        sender_net_reject_recipients != NULL)
      {
      if ((hostcheck = verify_check_host(sender_host_reject_recipients,
            &sender_host_reject_recipients_hosts, TRUE) == TRUE) ||
          verify_check_net(sender_net_reject_recipients,
            &sender_net_reject_recipients_nets))
        {
        log_write(1, LOG_MAIN|LOG_REJECT, "recipients from %s refused (%s)",
          host_and_ident(), hostcheck? "host" : "net");
        host_refuse_all_rcpts = TRUE;
        }
      }

    /* If any Realtime Blocking List domains are configured, look up this host
    in them, and if it is found in any one of them, reject recipients if
    so configured. Otherwise just log and create a warning header if required.
    Do not do this if the host is in rbl_except_nets. */

    if (rbl_domains != NULL &&
          !verify_check_net(rbl_except_nets, &rbl_except_netlist))
      {
      char domain[256];
      char *listptr = rbl_domains;

      while (string_nextinlist(&listptr, ':', domain, sizeof(domain)) != NULL)
        {
        DEBUG(9) debug_printf("checking RBL domain %s\n", domain);

        switch(host_check_rbl(sender_host_address, domain, rbl_msg_buffer,
               sizeof(rbl_msg_buffer)))
          {
          case DNS_SUCCEED:
          if (rbl_reject_recipients)
            {
            log_write(1, LOG_MAIN|LOG_REJECT, "recipients from %s refused "
              "(RBL %s)", host_and_ident(), domain);
            host_refuse_all_rcpts = TRUE;
            }
          else
            {
            log_write(1, LOG_MAIN|LOG_REJECT, "%s in RBL list, but "
              "rbl_reject_recipients is false", host_and_ident());
            if (rbl_warn_header)
              {
              char *name = "X-RBL-Warning: ";
              int slen = (int)strlen(name) + 1 + (int)strlen(rbl_msg_buffer);
              rbl_header = store_malloc(sizeof(header_line) + slen);
              sprintf(rbl_header->text, "%s%s\n", name, rbl_msg_buffer);
              rbl_header->type = htype_other;
              rbl_header->slen = slen;
              }
            }
          listptr = NULL;        /* To break the loop */
          break;

          case DNS_NOMATCH:
          break;                 /* Continue looking */

          case DNS_AGAIN:        /* Continue at the moment; should there be */
          case DNS_FAIL:         /* an option for a soft reject? */
          break;
          }
        }
      }

    /* Check for reserved slots. Note that the count value doesn't include
    this process, as it gets upped in the parent process. */

    if (smtp_accept_max > 0 &&
        smtp_accept_count + 1 > smtp_accept_max - smtp_accept_reserve)
      {
      if (!verify_check_host(smtp_reserve_hosts, &smtp_reserve_hostlist, FALSE)
          &&
          !verify_check_net(smtp_reserve_nets, &smtp_reserve_netlist))
        {
        log_write(1, LOG_MAIN, "Connection from %s temporarily refused: not in "
          "reserve list: connected=%d max=%d reserve=%d", host_and_ident(),
          smtp_accept_count, smtp_accept_max, smtp_accept_reserve);
        DEBUG(3) debug_printf("421 %s: Too many concurrent SMTP connections; "
          "please try again later\n", primary_hostname);
        fprintf(smtp_out, "421 %s: Too many concurrent SMTP connections; "
          "please try again later\r\n", primary_hostname);
        return 0;
        }
      reserved_host = TRUE;
      }

    /* If a load level above which only messages from reserved hosts are
    accepted is set, check the load. For incoming calls via the daemon, the
    check is done in the superior process if there are no reserved hosts, to
    save a fork. In all cases, the load average will already be available
    in a global variable at this point. */

    if (smtp_load_reserve >= 0 &&
         load_average > smtp_load_reserve &&
         !reserved_host &&
         !verify_check_host(smtp_reserve_hosts, &smtp_reserve_hostlist, FALSE)
          &&
         !verify_check_net(smtp_reserve_nets, &smtp_reserve_netlist))
      {
      log_write(1, LOG_MAIN, "Connection from %s temporarily refused: not in "
        "reserve list and load average = %.2f", host_and_ident(),
        (double)load_average/1000.0);
      DEBUG(3) debug_printf("421 %s: Too much load; "
        "please try again later\n", primary_hostname);
      fprintf(smtp_out, "421 %s: Too much load; "
        "please try again later\r\n", primary_hostname);
      return 0;
      }

    /* Determine whether unqualified senders or recipients are permitted
    for this host. Unfortunately, we have to do this every time, in order to
    set the flags so that they can be inspected when considering qualifying
    addresses in the headers. For a site that permits no qualification, this
    won't take long, however. */

    allow_unqualified_sender = verify_check_host(sender_unqualified_hosts,
      &sender_unqualified_hostlist, FALSE) ||
    verify_check_net(sender_unqualified_nets, &sender_unqualified_netlist);

    allow_unqualified_recipient = verify_check_host(receiver_unqualified_hosts,
      &receiver_unqualified_hostlist, FALSE) ||
    verify_check_net(receiver_unqualified_nets, &receiver_unqualified_netlist);
    }

  /* Output the initial message for an SMTP connection. It may contain
  newlines, which then cause a multi-line response to be given. */

  s = expand_string(smtp_banner);
  p = s;

  if (s == NULL)
    log_write(0, LOG_PANIC_DIE, "Expansion of \"%s\" (smtp_banner) failed: %s",
      smtp_banner, expand_string_message);

  while (*p != 0)
    {
    int c;
    DEBUG(3) debug_printf("220%c", (strchr(p, '\n') == NULL)? ' ' : '-');
    fprintf(smtp_out, "220%c", (strchr(p, '\n') == NULL)? ' ' : '-');
    while ((c = *p) != 0)
      {
      p++; /* NB can't include in while because used in previous while */
      if (c == '\n') break;
      DEBUG(3) debug_printf("%c", c);
      fputc(c, smtp_out);
      }
    DEBUG(3) debug_printf("\n");
    putc('\r', smtp_out);
    putc('\n', smtp_out);
    }

  store_free(s);
  fflush(smtp_out);
  }

/* Now deal with SMTP commands. The reading routine sets up a timeout
for each one. If the timeout happens, or we get SIGTERM, exim just
gives up and dies. */

signal(SIGALRM, command_timeout_handler);
signal(SIGTERM, command_sigterm_handler);

/* This loop is exited by setting done to a POSITIVE value. The values
are 2 larger than the required yield of the function. */

while (done <= 0)
  {
  char *s;
  char *errmess;
  char *receiver = NULL;
  char *orig_receiver = NULL;
  char *hello = NULL;
  void (*oldsignal)(int);
  BOOL ok;
  int pid;
  int multiline = ' ';
  int errcode, start, end, sender_domain, receiver_domain;

  switch(smtp_read_command())
    {
    /* The HELO/EHLO commands are permitted to appear in the middle of
    a session as well as at the beginning. They have the effect of a
    reset in addition to their other functions. Their absence at the
    start cannot be taken to be an error. */

    case HELO_CMD:
    hello = "HELO";
    received_protocol = "smtp";      /* could be resetting in principle */
    multiline = ' ';
    /* fall through with hello != NULL */

    case EHLO_CMD:
    if (hello == NULL)
      {
      hello = "EHLO";
      received_protocol = "esmtp";
      multiline = '-';
      }

    /* The data for HELO/EHLO is supposed to be the domain name of the sending
    host, or an ip literal, optionally followed by other information. Check
    the text up to the first white space; then check that any subsequent text
    consists of printing characters. */

    s = smtp_data;
    helo_seen = FALSE;
    ok = FALSE;

    if (*s == '[')
      {
      while (*s != 0 && !isspace((uschar)*s)) s++;
      if (s[-1] == ']')
        {
        s[-1] = 0;
        ok = string_is_ip_address(smtp_data+1);
        s[-1] = ']';
        }
      }
    else if (*s != 0)
      {
      ok = TRUE;
      while (*s != 0 && !isspace((uschar)*s))
        {
        if (!isalnum((uschar)*s) && *s != '.' && *s != '-')
          {
          ok = FALSE;
          break;
          }
        s++;
        }
      }

    /* Check remaining characters, if any */

    if (ok) while (*s != 0)
      {
      if (*s == '\t') *s = ' ';
      if (!isprint((uschar)*s++)) { ok = FALSE; break; }
      }

    /* Reject the HELO if its argument was invalid or non-existent */

    if (!ok)
      {
      DEBUG(3)
        debug_printf("501 syntactically invalid %s argument(s)\n",hello);
      fprintf(smtp_out, "501 syntactically invalid %s argument(s)\r\n", hello);
      if (*smtp_data == 0) strcpy(smtp_data, "(no argument given)");
      s = string_printing(smtp_data, FALSE);
      log_write(0, LOG_REJECT, "rejected %s: syntactically invalid "
        "argument(s): %s", hello, s);
      if (s != smtp_data) store_free(s);
      break;
      }

    /* If sender_host_unknown is true, we have got here via the -bs interface,
    not called from inetd. In this case, HELO/EHLO should *not* set the helo
    name. Otherwise, we are running an IP connection and the host address will
    be set. If the helo name is the primary name of this host and we haven't
    done a reverse lookup, force one now. If helo_verify is set, ensure that
    the HELO name matches the actual host. */

    if (!sender_host_unknown)
      {
      char *p = smtp_data;
      if (sender_helo_name != NULL) store_free(sender_helo_name);
      sender_helo_name = store_malloc((int)strlen(smtp_data) + 1);
      strcpy(sender_helo_name, smtp_data);

      while (*p != 0 && !isspace(*p)) { *p = tolower(*p); p++; }
      *p = 0;

      /* Force a reverse lookup if HELO quoted our name or the name of one of
      our primary domains. */

      if (sender_host_name == NULL &&
          (strcmpic(smtp_data, primary_hostname) == 0 ||
            match_isinlist(smtp_data, local_domains, &re_local_domains, TRUE)))
        sender_host_name = host_find_byaddr(sender_host_address);

      host_build_sender_fullhost();

      /* Verify if required. This doesn't give much security, but it does make
      some people happy to be able to do it. Note that HELO is legitimately
      allowed to quote an address literal. */

      if (helo_verify)
        {
        BOOL OK = TRUE;

        /* Check an address literal. Allow for IPv6 ::ffff: literals. */

        if (sender_helo_name[0] == '[')
          {
          OK = strncmp(sender_helo_name+1, sender_host_address,
            (int)strlen(sender_host_address)) == 0;
          }

        /* Do a reverse lookup if one hasn't already been done. If that fails,
        or the name doesn't match, try checking with a forward lookup. */

        else
          {
          if (sender_host_name == NULL)
            sender_host_name = host_find_byaddr(sender_host_address);

          if (sender_host_name == NULL ||
               strcmpic(sender_host_name, sender_helo_name) != 0)
            {
            host_item h;
            h.name = sender_helo_name;
            h.address = NULL;
            h.next = NULL;
            if (host_find_byname(&h, NULL) == HOST_FIND_FAILED)
              OK = FALSE;
            else
              {
              host_item *hh = &h;
              while (hh != NULL)
                {
                host_item *thishh;
                if (strcmp(hh->address, sender_host_address) == 0)
                  OK = TRUE;
                store_free(hh->address);
                thishh = hh;
                hh = hh->next;
                if (thishh != &h) store_free(thishh);
                }
              }
            }
          }

        if (!OK)
          {
          DEBUG(3) debug_printf("550 %s argument does not match calling host\n",
            hello);
          fprintf(smtp_out, "550 %s argument does not match calling host\n",
            hello);
          log_write(0, LOG_MAIN|LOG_REJECT, "rejected %s from %s", hello,
            sender_fullhost);
          break;
          }
        }

      }

    /* Generate an OK reply, including the ident if present, and also
    the IP address if present. Reflecting back the ident is intended
    as a deterrent to mail forgers. */

    fprintf(smtp_out, "250%c%s: Hello %s%s%s", multiline, primary_hostname,
      (sender_ident == NULL)?  "" : sender_ident,
      (sender_ident == NULL)?  "" : " at ",
      (sender_host_name == NULL)? smtp_data : sender_host_name);

    DEBUG(3) debug_printf("250%c%s: Hello %s%s%s", multiline, primary_hostname,
      (sender_ident == NULL)?  "" : sender_ident,
      (sender_ident == NULL)?  "" : " at ",
      (sender_host_name == NULL)? smtp_data : sender_host_name);

    if (sender_host_address != NULL)
      {
      fprintf(smtp_out, " [%s]", sender_host_address);
      DEBUG(3) debug_printf(" [%s]", sender_host_address);
      }

    fprintf(smtp_out, "\r\n");
    DEBUG(3) debug_printf("\n");

    /* If we received EHLO, we have started a multiline response. Finish it
    off with the functions supported. */

    if (multiline == '-')
      {
      fprintf(smtp_out, "250-Supported functions are:\r\n");
      DEBUG(3) debug_printf("250-Supported functions are:\n");

      /* I'm not entirely happy with this, as an MTA is supposed to check
      that it has enough room to accept a message of maximum size before
      it sends this. However, there seems little point in not sending it. */

      if (message_size_limit > 0)
        {
        fprintf(smtp_out, "250-SIZE %d\r\n", message_size_limit);
        DEBUG(3) debug_printf("250-SIZE %d\n", message_size_limit);
        }

      /* Exim does not do protocol conversion or data conversion. It is 8-bit
      clean; if it has an 8-bit character in its hand, it just sends it. It
      cannot therefore specify 8BITMIME and remain consistent with the RFCs.
      However, some users want this option simply in order to stop MUAs
      mangling messages that contain top-bit-set characters. It is therefore
      provided as an option. */

      if (accept_8bitmime)
        {
        fprintf(smtp_out, "250-8BITMIME\r\n");
        DEBUG(3) debug_printf("250-8BITMIME\n");
        }

      /* Advertise ETRN if any hosts are permitted to issue it; a check is
      made when any host actually does. */

      if (smtp_etrn_hosts != NULL || smtp_etrn_nets != NULL)
        {
        fprintf(smtp_out, "250-ETRN\r\n");
        DEBUG(3) debug_printf("250-ETRN\n");
        }

      /* Advertise EXPN if any hosts are permitted to issue it; a check is
      made when any host actually does. */

      if (smtp_expn_hosts != NULL || smtp_expn_nets != NULL)
        {
        fprintf(smtp_out, "250-EXPN\r\n");
        DEBUG(3) debug_printf("250-EXPN\n");
        }

      /* Finish off the multiline reply with one that is always available. */

      fprintf(smtp_out, "250 HELP\r\n");
      DEBUG(3) debug_printf("250 HELP\n");
      }

    fflush(smtp_out);

    /* Ensure we are in the reset state, and note that HELO/EHLO has
    been seen. */

    accept_free_recipients();
    sender_address = NULL;
    helo_seen = TRUE;
    break;


    /* The MAIL FROM command requires an address as an operand. All we
    do here is to parse it for syntactic correctness. The form "<>" is
    a special case which converts into an empty string. The start/end
    pointers in the original are not used further for this address, as
    it is the canonical extracted address which is all that is kept. */

    case MAIL_CMD:
    if (helo_verify && !helo_seen)
      {
      DEBUG(3) debug_printf("550 HELO or EHLO is required for this host\n");
      fprintf(smtp_out, "550 HELO or EHLO is required for this host\r\n");
      log_write(0, LOG_MAIN|LOG_REJECT, "rejected MAIL FROM from %s: no "
        "HELO/EHLO given", sender_fullhost);
      break;
      }

    if (sender_address != NULL)
      {
      DEBUG(3) debug_printf("503 Sender already given\n");
      fprintf(smtp_out, "503 Sender already given\r\n");
      break;
      }

    if (smtp_data[0] == 0)
      {
      DEBUG(3) debug_printf("501 MAIL FROM must have an address operand\n");
      fprintf(smtp_out, "501 MAIL FROM must have an address operand\r\n");
      break;
      }

    /* Check that there is enough disc space, if configured to do so. There
    doesn't seem to be an ideal place to put this check, but putting it here
    does permit VRFY and EXPN etc. to be used when space is short. */

    if (!accept_check_fs())
      {
      DEBUG(3) debug_printf("452 space shortage, please try later\n");
      fprintf(smtp_out, "452 space shortage, please try later\r\n");
      break;
      }

    /* Reset the flag that refuses all recipients for bad senders when
    the reject style is configured that way. */

    sender_refuse_all_rcpts = FALSE;

    /* Loop for handling ESMTP additions to the MAIL FROM command. At
    present this is done by rather ad-hoc coding, since only two are
    supported. */

    if (strcmp(received_protocol, "esmtp") == 0) for (;;)
      {
      /* If this session was initiated with EHLO and message_size_limit is
      non-zero, Exim will have indicated that it supports the SIZE option.
      The remote is then permitted to add "SIZE=n" on the end of the MAIL
      FROM command. Just parse this out by ad hoc code for the moment. We
      know the command line starts "MAIL FROM", so the backwards searches
      will always terminate without the need for explicit stops. */

      if (message_size_limit > 0)
        {
        char *p = smtp_data + (int)strlen(smtp_data);
        while (isspace(p[-1])) *(--p) = 0;
        while (isdigit(p[-1])) p--;
        if (*p != 0 && strncmpic(p-5, "SIZE=", 5) == 0)
          {
          int size = atoi(p);
          p[-5] = 0;
          if (size > message_size_limit)
            {
            DEBUG(3) debug_printf("552 Message size exceeds maximum permitted\n");
            fprintf(smtp_out, "552 Message size exceeds maximum permitted\r\n");
            break;
            }
          }
        }

      /* If this session was initiated with EHLO and accept_8bitmime is set,
      Exim will have indicated that it supports the BODY=8BITMIME option. In
      fact, it does not support this according to the RFCs, in that it does not
      take any special action for forwarding messages containing 8-bit
      characters. That is why accept_8bitmime is not the default setting, but
      some sites want the action that is provided. Parse out the keyword by
      ad hoc code pro tem. We know the line starts "MAIL FROM", so the backwards
      searches will always terminate without explicit stops. */

      if (accept_8bitmime)
        {
        char *p = smtp_data + (int)strlen(smtp_data);
        while (isspace(p[-1])) *(--p) = 0;
        while (isalnum(p[-1])) p--;
        if (*p != 0 && strncmpic(p-5, "BODY=", 5) == 0) p[-5] = 0;
          else break;  /* ESMTP loop */
        }
      else break;      /* ESMTP loop */
      }

    /* Now extract the address. The TRUE flag allows "<>" as a sender
    address. */

    raw_sender =
      parse_extract_address(smtp_data, &errmess, &start, &end, &sender_domain,
        TRUE);
    if (raw_sender == NULL)
      {
      DEBUG(3) debug_printf("501 %s: %s\n", smtp_data, errmess);
      fprintf(smtp_out, "501 %s: %s\r\n", smtp_data, errmess);
      break;
      }
    sender_address = string_copy(raw_sender);

    /* If sender_address is unqualified, reject it, unless this is a
    locally generated message, in which case it will be ignored anyway.
    However, if the sending host or net is listed as permitted to send
    unqualified addresses - typically local machines behaving as MUAs -
    then just qualify the address. The flag is set above at the start
    of the SMTP connection. */

    if (!sender_local && sender_domain == 0 && sender_address[0] != 0 &&
        sender_address[0] != '@')
      {
      if (allow_unqualified_sender)
        {
        sender_domain = (int)strlen(sender_address) + 1;
        sender_address = rewrite_address_qualify(sender_address, FALSE);
        DEBUG(9) debug_printf("unqualified address %s accepted\n",
          raw_sender);
        }
      else
        {
        DEBUG(3) debug_printf("501 %s: sender address must contain a domain\n",
          smtp_data);
        fprintf(smtp_out, "501 %s: sender address must contain a domain\r\n",
          smtp_data);
        log_write(1, LOG_MAIN|LOG_REJECT, "unqualified sender rejected: "
          "<%s>%s%s%s",
          raw_sender,
          (sender_fullhost != NULL)? " H=" : "",
          (sender_fullhost != NULL)? host_and_ident() : "",
          host_findbyaddr_msg);
        store_free(sender_address);
        sender_address = NULL;
        break;
        }
      }

    /* If configured to reject any senders explicitly (spam filtering),
    check now, but don't bother if the host has already been rejected. The
    sender check is independent of sender verification, and does not
    happen for local senders. We must always allow through the null
    sender, though. */

    if (!host_refuse_all_rcpts && !sender_local && sender_address[0] != 0 &&
         (sender_reject != NULL || sender_reject_recipients != NULL ||
          sender_accept != NULL || sender_accept_recipients != NULL))
      {
      BOOL rejectcheck = TRUE;
      if ((sender_accept != NULL &&
            (rejectcheck =
               match_address_list(sender_address, sender_domain,
                 sender_accept, &re_sender_accept, -1, ':')) == FALSE
          ) ||
          (match_address_list(sender_address, sender_domain,
            sender_reject, &re_sender_reject, -1, ':') &&
          !match_address_list(sender_address, sender_domain,
            sender_reject_except, &re_sender_reject_except, -1, ':')))
        {
        smtp_send_prohibition_message(smtp_out, 550,
          rejectcheck? "sender_reject" : "sender_accept");
        fprintf(smtp_out, "550 rejected: administrative prohibition%s\r\n",
          host_findbyaddr_msg);
        DEBUG(3) debug_printf("550 rejected: administrative prohibition%s\n",
          host_findbyaddr_msg);
        log_write(1, LOG_MAIN|LOG_REJECT, "sender rejected: <%s>%s%s%s",
          raw_sender,
          (sender_fullhost != NULL)? " H=" : "",
          (sender_fullhost != NULL)? host_and_ident() : "",
          host_findbyaddr_msg);
        store_free(sender_address);
        sender_address = NULL;
        break;     /* Ends case statement - MAIL FROM finished */
        }

      if ((sender_accept_recipients != NULL &&
           !match_address_list(sender_address, sender_domain,
           sender_accept_recipients, &re_sender_accept_recipients, -1, ':')) ||
          (match_address_list(sender_address, sender_domain,
            sender_reject_recipients, &re_sender_reject_recipients, -1, ':') &&
           !match_address_list(sender_address, sender_domain,
              sender_reject_except, &re_sender_reject_except, -1, ':')
          ))
        {
        sender_refuse_all_rcpts = TRUE;
        log_write(1, LOG_MAIN|LOG_REJECT, "recipients refused from <%s>%s%s",
          raw_sender,
          (sender_fullhost != NULL)? " H=" : "",
          (sender_fullhost != NULL)? host_and_ident() : "");
        }
      }

    /* If configured to check sender addresses, do the preliminary check
    now, unless the sender is local (in which case whatever is given here
    is ignored anyway). The check will fail if the message is to be refused at
    this stage. Another check function is called after the message has been
    received, to do more checking when the headers are available. However,
    don't bother with the check if all recipients are in any case going to
    be refused. */

    errmess = NULL;
    refuse_all_rcpts = FALSE;

    if (sender_verify || sender_try_verify)
      {
      char *old_sender_address = sender_address;

      if (!sender_local &&
          !host_refuse_all_rcpts &&
          !sender_refuse_all_rcpts &&
          !verify_sender_preliminary(&errcode, &errmess))
        {
        smtp_send_prohibition_message(smtp_out, errcode, "sender_verify");
        fprintf(smtp_out, "%d rejected: %s <%s>\r\n", errcode, errmess,
          raw_sender);
        DEBUG(3) debug_printf("%d rejected: %s %s\n", errcode,
          errmess, raw_sender);
        log_write(1, LOG_MAIN|LOG_REJECT, "rejected MAIL FROM: %s <%s>%s%s",
          errmess,
          raw_sender,
          (sender_fullhost != NULL)? " H=" : "",
          (sender_fullhost != NULL)? host_and_ident() : "");
        store_free(sender_address);
        sender_address = NULL;
        break;
        }

      /* Verification may cause rewriting of the address. We need to reset
      sender_domain as it might be used later on when checking recipients
      for relay permissions. */

      if (sender_address != old_sender_address)
        sender_domain = strchr(sender_address, '@') + 1 - sender_address;
      }

    /* The sender address is acceptable - for now. If verification is running
    in warning mode, errmess is set non-NULL if there is a warning to be
    given. This is also the case when a bad address has been received 3 times
    and refuse_all_rcpts is set. If the address has been verified, a new sender
    address is always produced on success. However, we reflect the original one
    to the outside world. */

    if (errmess != NULL)
      {
      DEBUG(3) debug_printf("%d %s <%s>\n", errcode, errmess, raw_sender);
      fprintf(smtp_out, "%d %s <%s>\r\n", errcode, errmess, raw_sender);
      log_write(1, LOG_MAIN|LOG_REJECT, "%s <%s>%s%s", errmess, raw_sender,
        (sender_fullhost != NULL)? " H=" : "",
        (sender_fullhost != NULL)? host_and_ident() : "");
      }
    else
      {
      DEBUG(3) debug_printf("250 <%s> is syntactically correct\n",
        raw_sender);
      fprintf(smtp_out, "250 <%s> is syntactically correct\r\n", raw_sender);
      }

    /* Note that we haven't yet checked this sender for permission to relay
    messages through this host. */

    sender_allow_relay_anywhere_set = FALSE;

    /* RFC 821 says MAIL FROM resets state at start of message */

    accept_free_recipients();
    rcount = 0;
    toomany = FALSE;
    break;


    /* The RCPT TO command requires an address as an operand. All we do
    here is to parse it for syntactic correctness. There may be any number
    of RCPT TO commands, specifying multiple senders. We build them all into
    a data structure that is in argc/argv format. The start/end values
    given by parse_extract_address are not used, as we keep only the
    extracted address. */

    case RCPT_CMD:
    if (sender_address == NULL)
      {
      DEBUG(3) debug_printf("503 No sender yet given\n");
      fprintf(smtp_out, "503 No sender yet given\r\n");
      break;
      }

    if (smtp_data[0] == 0)
      {
      DEBUG(3) debug_printf("501 RCPT TO must have an address operand\n");
      fprintf(smtp_out, "501 RCPT TO must have an address operand\r\n");
      break;
      }

    /* Don't allow "<>" as a recipient address */

    orig_receiver =
      parse_extract_address(smtp_data, &errmess, &start, &end,
        &receiver_domain, FALSE);
    if (orig_receiver == NULL)
      {
      DEBUG(3) debug_printf("501 %s: %s\n", smtp_data, errmess);
      fprintf(smtp_out, "501 %s: %s\r\n", smtp_data, errmess);
      break;
      }
    receiver = string_copy(orig_receiver);

    /* If the receiver address is unqualified, reject it, unless this is a
    locally generated message. However, unqualified addresses are permitted
    from a configured list of hosts and nets - typically when behaving as
    MUAs rather than MTAs. Sad that SMTP is used for both types of traffic,
    really. The flag is set at the start of the SMTP connection.

    RFC 1123 talks about supporting "the reserved mailbox postmaster"; I always
    assumed this meant "reserved local part", but the revision of RFC 821 and
    friends now makes it absolutely clear that it means *mailbox*. Consequently
    we must always qualify this address, regardless. */

    if (receiver_domain == 0 && receiver[0] != '@')
      {
      if (allow_unqualified_recipient || strcmpic(receiver, "postmaster") == 0)
        {
        DEBUG(9) debug_printf("unqualified address %s accepted\n",
          receiver);
        receiver_domain = (int)strlen(receiver) + 1;
        receiver = rewrite_address_qualify(receiver, TRUE);
        }
      else
        {
        DEBUG(3)
          debug_printf("501 %s: recipient address must contain a domain\n",
            smtp_data);
        fprintf(smtp_out, "501 %s: recipient address must contain a domain\r\n",
          smtp_data);
        log_write(1, LOG_MAIN|LOG_REJECT, "unqualified recipient rejected: "
          "<%s>%s%s%s",
          receiver,
          (sender_fullhost != NULL)? " H=" : "",
          (sender_fullhost != NULL)? host_and_ident() : "",
          host_findbyaddr_msg);
        break;
        }
      }

    /* Handle various cases when all recipients are to be refused. This is
    the only known way of getting some remote mailers to give up on attempting
    to send a message.

    (1) The sender verify function sets refuse_all_rcpts when a bad sender has
        been received at least twice from the same host.
    (2) This function sets host_refuse_all_rcpts if the sending host is on
        a reject by recipients list.
    (3) This function sets sender_refuse_all_rcpts if the sender is on a
        rejection by recipients list.

    If any of these flags is set, bounce all recipients, using 550, which is
    the only version of "no" that some mailers understand, apparently.
    However, there is an exception list for the policy rejections. */

    if (refuse_all_rcpts)
      {
      DEBUG(3) debug_printf("550 cannot route to sender address <%s>\n",
        sender_address);
      fprintf(smtp_out, "550 cannot route to sender address <%s>\r\n",
        sender_address);
      break;
      }

    if (host_refuse_all_rcpts || sender_refuse_all_rcpts)
      {
      if (recipients_reject_except == NULL ||
          !match_address_list(receiver, receiver_domain,
          recipients_reject_except, &re_recipients_reject_except, -1, ':'))
        {
        if (rbl_msg_buffer[0] != 0)
          fprintf(smtp_out, "550-%s\r\n", rbl_msg_buffer);
        else
          smtp_send_prohibition_message(smtp_out, 550, host_refuse_all_rcpts?
            "host_reject_recipients" : "sender_reject_recipients");
        fprintf(smtp_out, "550 rejected: administrative prohibition%s\r\n",
          host_refuse_all_rcpts? host_findbyaddr_msg : "");
        DEBUG(3) debug_printf("550 rejected: administrative prohibition%s\n",
          host_refuse_all_rcpts? host_findbyaddr_msg : "");
        if (log_refused_recipients)
          log_write(1, LOG_MAIN|LOG_REJECT, "recipient <%s> refused from %s%s",
            receiver, host_and_ident(),
            host_refuse_all_rcpts? host_findbyaddr_msg : "");
        break;
        }
      else
        log_write(1, LOG_MAIN|LOG_REJECT,
          "exception list recipient %s accepted from <%s>%s%s",
          receiver,
          raw_sender,
          (sender_fullhost != NULL)? " H=" : "",
          (sender_fullhost != NULL)? host_and_ident() : "");
      }

    /* Check maximum allowed */

    if (recipients_max > 0 && ++rcount > recipients_max)
      {
      char *code = recipients_max_reject? "550" : "421";
      DEBUG(3) debug_printf("%s too many recipients\n", code);
      fprintf(smtp_out, "%s too many recipients\r\n", code);
      toomany = TRUE;
      break;
      }

    /* If configured to check for mail relaying, ensure that the domain
    of the recipient is acceptable, either because it is one that is allowed
    for all sending hosts and sending addresses, or because the host and
    the sender are permitted to relay to all domains. */

    if (check_relay)
      {
      char *lcdomain = string_copylc(receiver + receiver_domain);

      /* The host test is needed only for non-local domains that are not in
      relay_domains. Host_allow_relay_anywhere gets set the first time we need
      its value. (The test for for MXed domains is later.) */

      if (!match_isinlist(lcdomain, local_domains, &re_local_domains, TRUE) &&
          (relay_domains == NULL ||
          !match_isinlist(lcdomain, relay_domains, &re_relay_domains, TRUE)))
        {
        BOOL permitted = FALSE;
        char *msg1 = "550 relaying to <%s> prohibited by administrator\r\n";
        char *msg2 = "";

        /* If we haven't yet checked this host, do so. The rule for allowing
        relaying to any address is "the host is in an accept list or there are
        no accept lists, and it is not in a reject list". */

        if (!host_allow_relay_anywhere_set)
          {
          host_allow_relay_anywhere = TRUE;

          if ((sender_host_accept_relay != NULL ||
               sender_net_accept_relay != NULL)
             &&
             !verify_check_net(sender_net_accept_relay,
               &sender_net_accept_relay_nets)
             &&
             !verify_check_host(sender_host_accept_relay,
               &sender_host_accept_relay_hosts, FALSE)
             )
            {
            host_allow_relay_anywhere = FALSE;
            nonrelay_host_failed_accept = TRUE;
            }

          else if ((sender_host_reject_relay != NULL ||
                    sender_net_reject_relay != NULL)
             &&
             (
             verify_check_host(sender_host_reject_relay,
               &sender_host_accept_relay_hosts, TRUE)
             ||
             verify_check_net(sender_net_reject_relay,
               &sender_net_reject_relay_nets)
             ))
            {
            host_allow_relay_anywhere = FALSE;
            nonrelay_host_failed_accept = FALSE;
            }

          host_allow_relay_anywhere_set = TRUE;
          DEBUG(9) debug_printf("host_allow_relay_anywhere set %s\n",
            host_allow_relay_anywhere? "TRUE" : "FALSE");
          }

        /* If the host is acceptable, check up on the sender address when
        configured to require both to be valid. If the host is not
        acceptable, check up on the sender address when configured to
        require only one of them. The "set" flag gets cleared for each new
        "MAIL FROM". This saves doing the check for multiple recipients,
        as it could involve a file lookup and therefore be expensive. */

        if (host_allow_relay_anywhere != relay_need_either &&
            !sender_allow_relay_anywhere_set)
          {
          sender_allow_relay_anywhere = (sender_address_relay == NULL)? TRUE :
            match_address_list(sender_address, sender_domain,
              sender_address_relay, &re_sender_address_relay, -1, ':');
          sender_allow_relay_anywhere_set = TRUE;
          DEBUG(9) debug_printf("sender_allow_relay_anywhere set %s\n",
            sender_allow_relay_anywhere? "TRUE" : "FALSE");
          }

        /* An option determines whether only one or both tests are needed */

        permitted = relay_need_either?
          (host_allow_relay_anywhere || sender_allow_relay_anywhere) :
          (host_allow_relay_anywhere && sender_allow_relay_anywhere);

        /* There is one last possibility, which is that the domain is one
        that is MXed to this host, and Exim is configured to permit relaying
        to all such. We delay the test till this point because it requires
        a DNS lookup and so it seems better to do the host checking first.
        It also means we can give a temporary error for DNS timeouts etc. */

        if (!permitted && relay_domains_include_local_mx)
          {
          host_item h;
          BOOL removed;
          int rc;

          h.next = NULL;
          h.name = lcdomain;
          h.address = NULL;

          rc = host_find_bydns(&h,
            TRUE,       /* DNS only */
            FALSE,      /* not A only */
            FALSE,      /* no widening */
            FALSE,      /* no widening */
            NULL,       /* no feedback FQDN */
            &removed);  /* feedback if local removed */

          if (rc == HOST_FOUND_LOCAL || (rc == HOST_FOUND && removed))
            permitted = TRUE;
          else if (rc == HOST_FIND_AGAIN)
            {
            msg1 = "450 temporarily unable to check <%s> for relaying "
              "permission\r\n";
            msg2 = "temporarily ";
            }
          }

        /* Forbidden relaying. */

        if (!permitted)
          {
          smtp_send_prohibition_message(smtp_out, (msg1[0] == '5')? 550 : 450,
            relay_need_either?
              (nonrelay_host_failed_accept?
                 "sender+host_accept_relay" : "sender+host_reject_relay") :
              (host_allow_relay_anywhere? "sender_relay" :
                 nonrelay_host_failed_accept?
                   "host_accept_relay" : "host_reject_relay"));
          fprintf(smtp_out, msg1, orig_receiver);
          DEBUG(3) debug_printf(msg1, orig_receiver);
          log_write(1, LOG_MAIN|LOG_REJECT, "%srefused relay (%s) to <%s> from "
            "<%s>%s%s",
            msg2,
            relay_need_either?
              (nonrelay_host_failed_accept?
                 "sender & host accept" : "sender & host reject") :
              (host_allow_relay_anywhere? "sender" :
                 nonrelay_host_failed_accept? "host accept" : "host reject"),
            orig_receiver, sender_address,
            (sender_fullhost == NULL)? "" : " H=",
            (sender_fullhost == NULL)? "" : host_and_ident());
          break;     /* End of handling the RCPT TO command */
          }
        }
      }

    /* If configured to check the receiver address now, do so, but not if
    the host matches one of the exception lists. Verification can also be
    conditional on the sender or recipient matching a pattern. */

    if ((receiver_verify || receiver_try_verify) &&
        (receiver_verify_addresses == NULL ||
         match_address_list(receiver, receiver_domain,
           receiver_verify_addresses, &re_receiver_verify_addresses,
             -1, ':')
        ) &&
        (receiver_verify_senders == NULL ||
         match_address_list(sender_address, sender_domain,
           receiver_verify_senders, &re_receiver_verify_senders,
             -1, ':')
        ) &&
        (receiver_verify_senders_except == NULL ||
         !match_address_list(sender_address, sender_domain,
           receiver_verify_senders_except, &re_receiver_verify_senders_except,
             -1, ':')
        ) &&
        !verify_check_host(receiver_verify_except_hosts,
          &receiver_verify_except_hostlist, FALSE) &&
        !verify_check_net(receiver_verify_except_nets,
          &receiver_verify_except_netlist))
      {
      BOOL receiver_local;
      int rc = verify_address(receiver, NULL, &receiver_local, NULL,
        vopt_is_recipient);

      /* Failure causes a hard error */

      if (rc == FAIL)
        {
        if (receiver_local)
          {
          DEBUG(3) debug_printf("550 Unknown local part in <%s>\n",
            orig_receiver);
          fprintf(smtp_out, "550 Unknown local part in <%s>\r\n", orig_receiver);
          }
        else
          {
          DEBUG(3) debug_printf("550 Cannot route to <%s>\n", orig_receiver);
          fprintf(smtp_out, "550 Cannot route to <%s>\r\n", orig_receiver);
          }
        log_write(3, LOG_MAIN,
          "verify failed for SMTP recipient %s from <%s>%s%s",
          orig_receiver,
          sender_address,
          (sender_fullhost == NULL)? "" : " H=",
          (sender_fullhost == NULL)? "" : host_and_ident());
        break;     /* End of handling the RCPT TO command */
        }

      /* If verification can't be done now, give a temporary error unless
      receiver_try_verify is set, in which case accept the address, but say
      it's unverified. */

      if (rc == DEFER)
        {
        if (!receiver_try_verify)
          {
          DEBUG(3) debug_printf("451 Cannot check <%s> at this time - "
            "please try later\n", orig_receiver);
          fprintf(smtp_out, "451 Cannot check <%s> at this time - "
            "please try later\r\n", orig_receiver);
          break;   /* End of handling the RCPT TO command */
          }

        DEBUG(3) debug_printf("250 Cannot check <%s> at this time - "
          "accepted unverified\n", orig_receiver);
        fprintf(smtp_out, "250 Cannot check <%s> at this time - "
          "accepted unverified\r\n", orig_receiver);
        }

      /* Verification succeeded */

      else
        {
        DEBUG(3) debug_printf("250 <%s> verified\n", orig_receiver);
        fprintf(smtp_out, "250 <%s> verified\r\n", orig_receiver);
        }
      }

    /* Otherwise the receiver address is only known to be syntactically
    acceptable. Any delivery errors will happen later. */

    else
      {
      DEBUG(3) debug_printf("250 <%s> is syntactically correct\n",
        orig_receiver);
      fprintf(smtp_out, "250 <%s> is syntactically correct\r\n", orig_receiver);
      }

    /* Add to the list of receivers, and set value to NULL to prevent
    freeing. */

    accept_add_recipient(receiver);
    receiver = NULL;
    break;


    /* The DATA command is legal only if it follows successful MAIL FROM
    and RCPT TO commands. This function is complete when a valid DATA
    command is encountered. */

    case DATA_CMD:
    if (sender_address == NULL)
      {
      DEBUG(3) debug_printf("503 MAIL FROM command must precede DATA\n");
      fprintf(smtp_out, "503 MAIL FROM command must precede DATA\r\n");
      break;
      }
    if (recipients_count <= 0)
      {
      DEBUG(3) debug_printf("503 Valid RCPT TO <recipient> must precede DATA\n");
      fprintf(smtp_out, "503 Valid RCPT TO <recipient> must precede DATA\r\n");
      break;
      }
    if (toomany && recipients_max_reject)
      {
      accept_free_recipients();
      sender_address = NULL;
      DEBUG(3) debug_printf("554 Too many recipients\n");
      fprintf(smtp_out, "554 Too many recipients\r\n");
      break;
      }

    DEBUG(3) debug_printf("354 Enter message, ending with \".\" on a line by itself\n");
    fprintf(smtp_out, "354 Enter message, ending with \".\" on a line by itself\r\n");
    done = 3;
    break;


    /* The VRFY command is enabled by a configuration option. Despite RFC1123
    it defaults disabled. */

    case VRFY_CMD:
    if (!smtp_verify)
      {
      DEBUG(3) debug_printf("550 VRFY command not available\n");
      fprintf(smtp_out, "550 VRFY command not available\r\n");
      }

    /* When VRFY is enabled, it verifies only addresses that contain no domain
    or one of the local domains. However, we have to let the verify function
    and the routers and directors decide what is local. */

    else
      {
      BOOL is_local_domain;
      int rc;
      char *address = parse_extract_address(smtp_data, &errmess, &start, &end,
        &receiver_domain, FALSE);

      if (address == NULL)
        {
        DEBUG(3) debug_printf("501 %s\n", errmess);
        fprintf(smtp_out, "501 %s\r\n", errmess);
        break;
        }

      rc = verify_address(address, NULL, &is_local_domain, NULL,
        vopt_is_recipient | vopt_local);

      if (!is_local_domain)
        {
        DEBUG(3) debug_printf("551 Not a local domain\n");
        fprintf(smtp_out, "551 Not a local domain\r\n");
        }

      else switch (rc)
        {
        case OK:
        DEBUG(3) debug_printf("250 Deliverable\n");
        fprintf(smtp_out, "250 Deliverable\r\n");
        break;

        case DEFER:
        DEBUG(3) debug_printf("450 Cannot resolve at this time\n");
        fprintf(smtp_out, "450 Cannot resolve at this time\r\n");
        break;

        case FAIL:
        DEBUG(3) debug_printf("550 Not deliverable\n");
        fprintf(smtp_out, "550 Not deliverable\r\n");
        break;
        }
      }
    break;

    /* The EXPN command is available only from specified hosts or nets. */

    case EXPN_CMD:
    if (!verify_check_host(smtp_expn_hosts, &smtp_expn_hostlist,
          FALSE) &&
        !verify_check_net(smtp_expn_nets, &smtp_expn_netlist))
      {
      DEBUG(3) debug_printf("550 EXPN command not available\n");
      fprintf(smtp_out, "550 EXPN command not available\r\n");
      }
    else
      {
      (void) verify_address(smtp_data, smtp_out, NULL, NULL,
        vopt_is_recipient | vopt_address_test | vopt_local | vopt_expn);
      }
    break;


    case QUIT_CMD:
    DEBUG(3) debug_printf("221 %s closing connection\n", primary_hostname);
    fprintf(smtp_out, "221 %s closing connection\r\n", primary_hostname);
    accept_free_recipients();
    done = 2;
    break;


    case RSET_CMD:
    accept_free_recipients();
    sender_address = NULL;
    DEBUG(3) debug_printf("250 Reset OK\n");
    fprintf(smtp_out, "250 Reset OK\r\n");
    break;


    case NOOP_CMD:
    DEBUG(3) debug_printf("250 OK\n");
    fprintf(smtp_out, "250 OK\r\n");
    break;


    case DEBUG_CMD:
    DEBUG(3) debug_printf("500 No way!\n");
    fprintf(smtp_out, "500 No way!\r\n");
    break;


    /* Show ETRN/EXPN if any hosts are permitted to use them; if actually
    used, a check will be done for permitted hosts. */

    case HELP_CMD:
    DEBUG(3)
      {
      debug_printf("214-Commands supported:\n");
      debug_printf("214-    HELO EHLO MAIL RCPT DATA%s%s\n",
        (smtp_etrn_hosts != NULL || smtp_etrn_nets != NULL)? " ETRN" :"",
        (smtp_expn_hosts != NULL || smtp_expn_nets != NULL)? " EXPN" :"");
      debug_printf("214     NOOP QUIT RSET HELP %s\n",
        smtp_verify? "VRFY" : "");
      }
    fprintf(smtp_out, "214-Commands supported:\r\n");
    fprintf(smtp_out, "214-    HELO EHLO MAIL RCPT DATA%s%s\r\n",
        (smtp_etrn_hosts != NULL || smtp_etrn_nets != NULL)? " ETRN" :"",
        (smtp_expn_hosts != NULL || smtp_expn_nets != NULL)? " EXPN" :"");
    fprintf(smtp_out, "214     NOOP QUIT RSET HELP %s\r\n",
      smtp_verify? "VRFY" : "");
    break;


    case EOF_CMD:
    DEBUG(3) debug_printf("421 %s lost input connection\n", primary_hostname);
    fprintf(smtp_out, "421 %s lost input connection\r\n", primary_hostname);

    /* Don't log unless in the middle of a message, as some mailers just
    drop the call rather than sending QUIT, and it clutters up the logs. */

    if (sender_address != NULL || recipients_count > 0)
      {
      if (sender_fullhost != NULL)
        log_write(4, LOG_MAIN, "%s unexpected disconnection", sender_fullhost);
      else
        log_write(0, LOG_MAIN, "unexpected EOF from local SMTP connection");
      }

    done = 1;
    break;


    case ETRN_CMD:
    if (smtp_etrn_hosts == NULL && smtp_etrn_nets == NULL)
      {
      DEBUG(3) debug_printf("500 Command unrecognized\n");
      fprintf(smtp_out, "500 Command unrecognized\r\n");
      break;
      }

    if (sender_address != NULL)
      {
      DEBUG(3) debug_printf("503 ETRN not permitted inside transaction\n");
      fprintf(smtp_out, "503 ETRN not permitted inside transaction\r\n");
      break;
      }

    if (smtp_data[0] != '#')
      {
      DEBUG(3) debug_printf("501 Syntax error\n");
      fprintf(smtp_out, "501 Syntax error\r\n");
      break;
      }

    /* Check that the current host is permitted to do this */

    if (!verify_check_host(smtp_etrn_hosts, &smtp_etrn_hostlist, FALSE) &&
        !verify_check_net(smtp_etrn_nets, &smtp_etrn_netlist))
      {
      DEBUG(3) debug_printf("458 Administrative prohibition\n");
      fprintf(smtp_out, "458 Administrative prohibition\r\n");
      break;
      }

    /* Fork a child process and call Exim with the -R option. We don't
    want to have to wait for the process at any point, so set SIGCHLD
    to SIG_IGN before forking. It should be set that way anyway for external
    incoming SMTP, but we save and restore to be tidy. */

    oldsignal = signal(SIGCHLD, SIG_IGN);

    if ((pid = fork()) == 0)
      {
      int i = 0;
      char *argv[6];

      argv[i++] = exim_path;
      if (config_changed)
        {
        argv[i++] = "-C";
        argv[i++] = config_filename;
        }
      argv[i++] = "-R";
      argv[i++] = smtp_data+1;

      argv[i++] = (char *)0;
      execv(argv[0], argv);
      log_write(0, LOG_PANIC_DIE, "exec of exim -R failed");
      _exit(EXIT_FAILURE);         /* paranoia */
      }

    signal(SIGCHLD, oldsignal);    /* restore */

    if (pid < 0)
      {
      log_write(0, LOG_PANIC, "fork of queue-runner process failed");
      fprintf(smtp_out, "458 Unable to start queue run\r\n");
      }
    else
      {
      DEBUG(3) debug_printf("250 OK, queue -R \'%s\' started\n", smtp_data+1);
      fprintf(smtp_out, "250 OK, queue -R \'%s\' started\r\n", smtp_data+1);
      }
    break;


    default:
    DEBUG(3) debug_printf("500 Command unrecognized\n");
    fprintf(smtp_out, "500 Command unrecognized\r\n");
    break;
    }

  /* Ensure output gets sent, and free temporary store. */

  fflush(smtp_out);

  if (orig_receiver != NULL) store_free(orig_receiver);
  if (receiver != NULL) store_free(receiver);
  }

/* Reset the signal handlers used in this function, and if no
message is in progress, ensure the store is cleaned up. */

signal(SIGALRM, SIG_DFL);
signal(SIGTERM, SIG_DFL);

if (done < 3)
  {
  if (raw_sender != NULL)
    {
    store_free(raw_sender);
    raw_sender = NULL;
    }
  accept_free_recipients();
  }

return done - 2;  /* Convert yield values */
}

/* End of smtp_in.c */
