/*************************************************
*     Exim - an Internet mail transport agent    *
*************************************************/

/* Copyright (c) University of Cambridge 1995 - 1997 */
/* See the file NOTICE for conditions of use and distribution. */

/* A set of functions to search databases in various formats. An open database
is represented by a void * value. This set of functions has no private state. */

#include "exim.h"


/* Returns from the file credential check function. */

enum { check_ok, check_stat };

/* Tree in which to cache open files until tidyup called. */

tree_node *search_tree = NULL;

/* Place to cache lookups for efficiency. */

static char cached_filename[256] = "";
static char cached_keystring[256] = "";
static int  cached_type = -1;
static char *cached_data = NULL;



/*************************************************
*         Check a file's credentials             *
*************************************************/

/* fstat can normally be expected to work on an open file, but there are some
NFS states where it may not.

Arguments:
  fd         an open file descriptor or -1
  filname    a file name if fd is -1

Returns:     check_stat if stat() or fstat() fails
             check_ok if all well

Side effect: sets errno to what fstat set it to.
*/

static int
check_file(int fd, char *filename)
{
int i;
struct stat statbuf;

if (fd >= 0)
  { if (fstat(fd, &statbuf) != 0) return check_stat; }
else if (stat(filename, &statbuf) != 0) return check_stat;

return check_ok;
}



/*************************************************
*               Release cached resources         *
*************************************************/

/* When search_open is called, it caches the file that it opens in search_tree.
The name of the tree node is a concatenation of the search type with the file
name. These files are closed only when this tidyup routine is called, typically
at the end of sections of code where a number of lookups might occur.

First, there is an internal, recursive subroutine.

Argument:   a pointer to a search_openfile tree node
Returns:    nothing
*/

static void
tidyup_subtree(tree_node *t)
{
if (t->left != NULL) tidyup_subtree(t->left);
if (t->right != NULL) tidyup_subtree(t->right);
if (t->name[0] == stype_lsearch) fclose((FILE *)(t->data.ptr));
  else if (t->name[0] == stype_dbm) EXIM_DBCLOSE((EXIM_DB *)(t->data.ptr));
store_free(t);
}

/* The external entry point

Argument: none
Returns:  nothing
*/

void
search_tidyup(void)
{
DEBUG(9) debug_printf("search_tidyup called\n");
if (search_tree != NULL)
  {
  tidyup_subtree(search_tree);
  search_tree = NULL;
  }
}




/*************************************************
*             Open search database               *
*************************************************/

/* There are two styles of query: (1) in the "single-key+file" style,
a single key string and a file name are given, for example, for linear
searches or DBM files.

Before opening, scan the tree of cached files to see if this "file" is already
open for the correct search type. If so, return the saved handle. If not, put
the handle in the tree for possible subsequent use. See search_tidyup above for
closing the cached files.

Arguments:
  filename       the name of the file for single-key+file style lookups
                 NULL for query-style lookups
  search_type    the type of search required
  message        points to a char * which can be set to point to an error
                 message when something goes wrong

Returns: an identifying handle for the open database */

void *
search_open(char *filename, int search_type, char **message)
{
int rc;
FILE *f;
EXIM_DB *d;
void *handle;
tree_node *t;
char keybuffer[256];

/* See if we already have this file open for this type of search, and if so,
pass back the previous handle. The key for the tree node is the search type
concatenated with the file name. */

if (filename != NULL)
  {
  sprintf(keybuffer, "%c%.254s", search_type, filename);
  if ((t = tree_search(search_tree, keybuffer)) != NULL)
    {
    DEBUG(9) debug_printf("search_open found (%c) %s cached\n",
      search_type, filename);
    return t->data.ptr;
    }
  }

/* Otherwise, open the file - each search type has its own code. */

DEBUG(9) debug_printf("search_open (%c) %s\n", search_type,
  (filename == NULL)? "NULL" : filename);

errno = 0;

switch (search_type)
  {
  /* Linear search */

  case stype_lsearch:
  f = os_fopen(filename, "r");
  if (f == NULL)
    {
    int save_errno = errno;
    *message = string_sprintf("failed to open %s for linear search: %s",
      filename, strerror(errno));
    errno = save_errno;
    return NULL;
    }

  if ((rc = check_file(fileno(f), NULL)) != check_ok)
    {
    int save_errno = errno;
    *message = string_sprintf("%s: failed to fstat open file", filename);
    errno = save_errno;
    return NULL;
    }
  handle = f;
  break;


  /* DBM search */

  case stype_dbm:
  EXIM_DBOPEN(filename, O_RDONLY, &d);
  if (d == NULL)
    {
    int save_errno = errno;
    *message = string_sprintf("failed to open %s as a %s file: %s", filename,
      EXIM_DBTYPE, strerror(errno));
    errno = save_errno;
    return NULL;
    }

  /* This needs to know more about the underlying files than is good for it!
  We need to know what the real file names are in order to check the owners
  and modes. If USE_DB is set, we know it is Berkeley DB, which an unmodified
  file name. Otherwise, for safety, we have to check for x.db or x.dir and
  x.pag. */

  rc = check_file(-1, filename);

  /* Something is wrong about the file. */

  if (rc != check_ok)
    {
    int save_errno = errno;
    *message = string_sprintf("%s: failed to stat file", filename);
    errno = save_errno;
    return NULL;
    }


  handle = d;
  break;

  /* Oops */

  default:
  *message = string_sprintf("unknown search type (%d) in search_open%s%s",
    search_type,
    (filename == NULL)? "" : " for file ",
    (filename == NULL)? "" : filename);
  return NULL;
  }

/* Get here only if the file has been successfully opened. If there is
a filename enter the file (with type concatenated) into the tree. */

if (filename != NULL)
  {
  t = store_malloc(sizeof(tree_node) + (int)strlen(keybuffer));
  strcpy(t->name, keybuffer);
  t->data.ptr = handle;
  tree_insertnode(&search_tree, t);
  }

return handle;
}





/*************************************************
*  Internal function: Find one item in database  *
*************************************************/

/*The answer is always put into dynamic store. If the key contains a colon,
then it is treated as a double key: the first part is the key for the record in
the file, and the remainder is a subkey that is used to extract a subfield from
the main data. Subfields are specified as subkey=value in the records.

The last lookup is cached by file name and key - using the handle is no good as
it isn't unique enough.

Arguments:
  handle       the handle from search_open
  filename     the filename that was handed to search_open, or
               NULL for query-style searches
  keystring    the keystring for single-key+file lookups, or
               the querystring for query-style lookups
  type         the type of lookup
  errmsg       somewhere to point to an error message

Returns:       NULL if the query failed, or
               a pointer to a string containing the answer
*/

static char *
internal_search_find(void *handle, char *filename, char *keystring, int type,
  char **errmsg)
{
int length;
FILE *f;
EXIM_DB *d;
EXIM_DATUM key, data;
char *subkey = NULL;
char *colon = NULL;
char *nis_data;
char buffer[4096];

*errmsg = "";    /* for safety */

/* If there's a colon in a key for a single-key+file lookup, temporarily
terminate the main key, and set up the subkey. */

if (filename != NULL)
  {
  colon = strchr(keystring, ':');
  if (colon != NULL)
    {
    subkey = colon + 1;
    *colon = 0;
    }
  }

/* Turn a NULL filename into an empty string so that it gets copied and
compared in the cache without special code. */

else filename = "";

DEBUG(9) debug_printf("internal_search_find: file=\"%s\" type=%c key=\"%s\"\n",
  filename, type, keystring);

/* Insurance. If the keystring is empty, just fail. */

if (keystring[0] == 0) return NULL;

/* If there is no cached record of the right type, or if this lookup is for a
new key, or in a different file, we must search the file and set up new cached
data. */

if (type != cached_type ||
    strcmp(keystring, cached_keystring) != 0 ||
    strcmp(filename, cached_filename) != 0)
  {
  DEBUG(9)
    {
    if (filename[0] != 0)
      debug_printf("file lookup required for %s%s%s in %s\n",
        keystring,
        (subkey == NULL)? "" : ":",
        (subkey == NULL)? "" : subkey,
        filename);
    else
      debug_printf("database lookup required for %s\n", keystring);
    }

  if (cached_data != NULL)
    {
    store_free(cached_data);    /* free previous */
    cached_data = NULL;
    }

  /* Length of key to match */

  length = (int)strlen(keystring);

  /* Code for the different kinds of search. The answer should be put in
  dynamic store as a zero-terminated string, and pointed to by cached_data.
  Leave cached_data = NULL on failure. */

  switch(type)
    {
    /* Linear search */

    case stype_lsearch:
    f = (FILE *)handle;
    rewind(f);

    while (fgets(buffer, sizeof(buffer), f) != NULL)
      {
      int ptr, size;
      int p = (int)strlen(buffer);
      char *s = buffer;

      while (p > 0 && isspace(buffer[p-1])) p--;
      buffer[p] = 0;
      if (buffer[0] == 0 || buffer[0] == '#' || isspace(buffer[0])) continue;
      while (*s != 0 && *s != ':' && !isspace(*s)) s++;
      if (s-buffer != length || strncmpic(buffer, keystring, length) != 0)
        continue;

      if (*s == ':') s++;
      while (isspace(*s)) s++;

      size = 100;
      ptr = 0;
      cached_data = store_malloc(size);
      if (*s != 0)
        cached_data = string_cat(cached_data, &size, &ptr, s, (int)strlen(s));

      while (fgets(buffer, sizeof(buffer), f) != NULL)
        {
        p = (int)strlen(buffer);
        while (p > 0 && isspace(buffer[p-1])) p--;
        buffer[p] = 0;
        if (buffer[0] == 0 || buffer[0] == '#') continue;
        if (!isspace(buffer[0])) break;
        cached_data = string_cat(cached_data, &size, &ptr, buffer, (int)strlen(buffer));
        }

      cached_data[ptr] = 0;
      break;
      }
    break;


    /* DBM search. */

    case stype_dbm:
    d = (EXIM_DB *)handle;
    EXIM_DATUM_DATA(key) = keystring;
    EXIM_DATUM_SIZE(key) = length + 1;
    if (EXIM_DBGET(d, key, data))
      cached_data = string_copy(EXIM_DATUM_DATA(data));
    break;
    }

  /* A record that has been found is now in cached_data, which is either NULL
  or points to a bit of dynamic store. Remember the file name, main key, and
  lookup type, but only if the file name and main key are < 256 characters
  long (the size of the cache slots). Longer keys are presumably exceedingly
  rare... */

  if ((int)strlen(filename) < 256 && length < 256)
    {
    strcpy(cached_filename, filename);
    strcpy(cached_keystring, keystring);
    cached_type = type;
    }
  else cached_type = -1;    /* Force lookup next time */
  }

else DEBUG(9) debug_printf("cached data used for lookup of %s%s%s%s%s\n",
  keystring,
  (subkey == NULL)? "" : ":",
  (subkey == NULL)? "" : subkey,
  (filename[0] == 0)? "" : " in ",
  (filename[0] == 0)? "" : filename);

/* Put back the colon if it was overwritten */

if (colon != NULL) *colon = ':';

/* If we have found data, pick out the subfield if required. Otherwise
make a fresh copy of the whole cached string. */

if (subkey != NULL && cached_data != NULL)
  return expand_getkeyed(subkey, cached_data);
    else if (cached_data != NULL) return string_copy(cached_data);
      else return NULL;
}




/*************************************************
* Find one item in database, possibly wildcarded *
*************************************************/

/* This function calls the internal function above; once only if there
is no partial matching, but repeatedly when partial matching is requested.

Arguments:
  handle       the handle from search_open
  filename     the filename that was handed to search_open, or
               NULL for query-style searches
  keystring    the keystring for single-key+file lookups, or
               the querystring for query-style lookups
  type         the type of lookup
  partial      if the value is greater than 1000, subtract 1024 to
               get the real value, and set the "try '*' if all else
               fails" flag; after this,
               -1 means no partial matching;
               otherwise it's the minimum number of components;
  expand_setup pointer to offset for setting up expansion strings;
               don't do any if < 0
  errmsg       somewhere to point to an error message

Returns:       NULL if the query failed, or
               a pointer to a string containing the answer
*/

char *
search_find(void *handle, char *filename, char *keystring, int type,
  int partial, int *expand_setup, char **errmsg)
{
BOOL set_null_wild = FALSE;
BOOL do_star = FALSE;
char *yield;

DEBUG(9) debug_printf("search_find: file=\"%s\" type=%c key=\"%s\" "
  "partial=%d\n",
  (filename == NULL)? "NULL" : filename, type, keystring, partial);

/* Determine whether independent "*" matching is wanted and adjust the
partial count. If there is no filename, partial matching can never happen. */

if (filename != NULL)
  {
  if (partial > 1000)
    {
    do_star = TRUE;
    partial -= 1024;
    }
  }
else partial = -1;

/* First of all, try to match the key string verbatim. If matched a complete
entry but could have been partial, flag to set up variables. */

yield = internal_search_find(handle, filename, keystring, type,
  errmsg);
if (yield != NULL) { if (partial >= 0) set_null_wild = TRUE; }

/* Not matched a complete entry; handle partial lookups */

else if (partial >= 0)
  {
  char *keystring2 = string_sprintf("*.%s", keystring);

  DEBUG(9) debug_printf("trying partial match %s\n", keystring2);
  yield = internal_search_find(handle, filename, keystring2, type, errmsg);

  /* The key in its entirety did not match a wild entry; try chopping off
  leading components. */

  if (yield == NULL)
    {
    int dotcount = 0;
    char *keystring3 = keystring2 + 2;
    char *s = keystring3;
    while (*s != 0) if (*s++ == '.') dotcount++;

    while (dotcount-- >= partial)
      {
      while (keystring3[1] != '.' && keystring3[1] != 0) keystring3++;
      *keystring3 = '*';
      DEBUG(9) debug_printf("trying partial match %s\n", keystring3);
      yield = internal_search_find(handle, filename, keystring3, type, errmsg);
      if (yield != NULL)
        {
        /* First variable is the wild part; second is the fixed part. Take care
        to get it right when keystring3 is just "*". */

        if (expand_setup != NULL && *expand_setup >= 0)
          {
          int fixedlength = (int)strlen(keystring3) - 2;
          int wildlength = (int)strlen(keystring) - fixedlength - 1;
          *expand_setup += 1;
          expand_nstring[*expand_setup] = keystring;
          expand_nlength[*expand_setup] = wildlength;
          *expand_setup += 1;
          expand_nstring[*expand_setup] = keystring + wildlength + 1;
          expand_nlength[*expand_setup] = (fixedlength < 0)? 0 : fixedlength;
          }
        break;
        }
      keystring3 += 2;
      }
    }

  else set_null_wild = TRUE; /* Matched a wild entry without any wild part */

  store_free(keystring2);
  }

/* If we still haven't matched anything, and the option to look for "*" is
set, try that, but not if partial == 0, because in that case it will already
have been searched for. If we do match, the first variable (the wild part)
is the whole domain, and the second is empty. */

if (yield == NULL && do_star && partial != 0)
  {
  DEBUG(9) debug_printf("trying to match *\n");
  yield = internal_search_find(handle, filename, "*", type, errmsg);
  if (yield != NULL && expand_setup != NULL && *expand_setup >= 0)
    {
    *expand_setup += 1;
    expand_nstring[*expand_setup] = keystring;
    expand_nlength[*expand_setup] = (int)strlen(keystring);
    *expand_setup += 1;
    expand_nstring[*expand_setup] = keystring;
    expand_nlength[*expand_setup] = 0;
    }
  }

/* If this was a potentially partial lookup, and we matched either a
complete non-wild domain entry, or we matched a wild-carded entry without
chopping off any of the domain components, set up the expansion variables
(if required) so that the first one is empty, and the second one is the
fixed part of the domain. */

if (set_null_wild && expand_setup != NULL && *expand_setup >= 0)
  {
  *expand_setup += 1;
  expand_nstring[*expand_setup] = keystring;
  expand_nlength[*expand_setup] = 0;
  *expand_setup += 1;
  expand_nstring[*expand_setup] = keystring;
  expand_nlength[*expand_setup] = (int)strlen(keystring);
  }

return yield;
}

/* End of search.c */
