/*
** id_open.c                 Establish/initiate a connection to an IDENT server
**
** Author: Peter Eriksson <pen@lysator.liu.se>
** Fixes: Pr Emanuelsson <pell@lysator.liu.se>
** IPV6 mods: Philip Hazel <ph10@cus.cam.ac.uk>
*/

#include <stdio.h>
#include <errno.h>
#include <fcntl.h>

#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/wait.h>
#include <sys/time.h>
#include <sys/file.h>

#define IN_LIBIDENT_SRC
#include "ident.h"

#include <arpa/inet.h>

ident_t *id_open __P3(struct sockaddr *, lsock,
                      struct sockaddr *, fsock,
                      struct timeval *, timeout)
{
    ident_t *id;
    int res, tmperrno;
    int af_type = lsock->sa_family;

    struct sockaddr_in sin_laddr, sin_faddr;

    fd_set rs, ws, es;

    if ((id = (ident_t *) malloc(sizeof(*id))) == 0)
        return 0;

    if ((id->fd = socket(af_type, SOCK_STREAM, 0)) < 0)
    {
        free(id);
        return 0;
    }

    if (timeout)
    {
        if ((res = fcntl(id->fd, F_GETFL, 0)) < 0)
            goto ERROR;

    }

    id->buf[0] = '\0';

    bzero((char *)&sin_laddr, sizeof(sin_laddr));

    sin_laddr.sin_family = AF_INET;
    sin_laddr.sin_port = 0;
    sin_laddr.sin_addr = ((struct sockaddr_in *)lsock)->sin_addr;

    if (bind(id->fd, (struct sockaddr *) &sin_laddr, sizeof(sin_laddr)) < 0)
    {
#ifdef DEBUG
        perror("libident: bind");
#endif
        goto ERROR;
    }

    bzero((char *)&sin_faddr, sizeof(sin_faddr));

    sin_faddr.sin_family = AF_INET;
    sin_faddr.sin_port = htons(IDPORT);
    sin_faddr.sin_addr = ((struct sockaddr_in *)fsock)->sin_addr;

    errno = 0;
    res = connect(id->fd, (struct sockaddr *) &sin_faddr, sizeof(sin_faddr));
    if (res < 0 && errno != EINPROGRESS)
    {
#ifdef DEBUG
        perror("libident: connect");
#endif
        goto ERROR;
    }

    if (timeout)
    {
        FD_ZERO(&rs);
        FD_ZERO(&ws);
        FD_ZERO(&es);

        FD_SET(id->fd, &rs);
        FD_SET(id->fd, &ws);
        FD_SET(id->fd, &es);

        if ((res = select(FD_SETSIZE, &rs, &ws, &es, timeout)) < 0)
        {
#ifdef DEBUG
            perror("libident: select");
#endif
            goto ERROR;
        }

        if (res == 0)
        {
            errno = ETIMEDOUT;
            goto ERROR;
        }

        if (FD_ISSET(id->fd, &es))
            goto ERROR;

        if (!FD_ISSET(id->fd, &rs) && !FD_ISSET(id->fd, &ws))
            goto ERROR;
    }

    return id;

  ERROR:
    tmperrno = errno;           /* Save, so close() won't erase it */
    close(id->fd);
    free(id);
    errno = tmperrno;
    return 0;
}
