/*************************************************
*     Exim - an Internet mail transport agent    *
*************************************************/

/* Copyright (c) University of Cambridge 1995 - 1997 */
/* See the file NOTICE for conditions of use and distribution. */


#include "../exim.h"
#include "smartuser.h"


#define NSUBEXP  10


/* Options specific to the smartuser director. */

optionlist smartuser_director_options[] = {

  { "expire_force_fail",   opt_bool,
      (void *)(offsetof(smartuser_director_options_block, expire_force_fail)) },
  { "expire_pattern", opt_stringptr,
      (void *)(offsetof(smartuser_director_options_block, expire_pattern)) },
  { "expire_subject", opt_stringptr,
      (void *)(offsetof(smartuser_director_options_block, expire_subject)) },
  { "expire_time", opt_stringptr,
      (void *)(offsetof(smartuser_director_options_block, expire_time)) },

  { "new_address", opt_stringptr,
      (void *)(offsetof(smartuser_director_options_block, new_address)) },
  { "panic_expansion_fail", opt_bool,
      (void *)(offsetof(smartuser_director_options_block, panic_expansion_fail)) }
};

/* Size of the options list. An extern variable has to be used so that its
address can appear in the tables drtables.c. */

int smartuser_director_options_count =
  sizeof(smartuser_director_options)/sizeof(optionlist);

/* Default private options block for the smartuser director. */

smartuser_director_options_block smartuser_director_option_defaults = {
  NULL,         /* new_address */

  NULL,         /* expire_pattern */
  NULL,         /* expire_subject */
  NULL,         /* expire_time */
  NULL,         /* expire_pattern_re */
  FALSE,        /* expire_force_fail */

  TRUE          /* panic_expansion_fail */
};



/*************************************************
*          Initialization entry point            *
*************************************************/

/* Called for each instance, after its options have been read, to
enable consistency checks to be done, or anything else that needs
to be set up.

Argument: points to the director instance block
Returns:  nothing
*/

void
smartuser_director_init(director_instance *dblock)
{
smartuser_director_options_block *ob =
  (smartuser_director_options_block *)(dblock->options_block);

/* If no transport is specified, a new user *must* be given. */

if (dblock->transport == NULL && dblock->expand_transport == NULL)
  {
  if (ob->new_address == NULL)
    log_write(0, LOG_PANIC_DIE|LOG_CONFIG2, "%s director:\n  "
      "neither transport nor new_address specified", dblock->name);
  }

/* A new address must be fully qualified, but can check here only if
it isn't going to expand. */

if (ob->new_address != NULL && strchr(ob->new_address, '$') == NULL &&
  strchr(ob->new_address, '@') == NULL)
    log_write(0, LOG_PANIC_DIE|LOG_CONFIG2, "%s director:\n  "
      "new address has no domain", dblock->name);

/* If expire_pattern is set, compile it now, and check that expire_time
is set. */

if (ob->expire_pattern != NULL)
  {
  ob->expire_pattern_re = regex_must_compile(ob->expire_pattern);
  if (ob->expire_time == NULL)
    log_write(0, LOG_PANIC_DIE|LOG_CONFIG2, "%s director:\n  "
      "expire_pattern set without expire_time", dblock->name);
  }

}




/*************************************************
*           Check for timed-out address          *
*************************************************/

/* This function is called if ob->expire_pattern is set. Its function is
to check for a timed out address. The string that is checked is created by
expanding expire_subject, or just the local part if that is null, and it is
checked against the expire_pattern, with expire_time giving the extraction
string. Initialization checks that expire_time is set if expire_pattern is set.
The result must be a single digit sequence, or two sequences of digits,
separated by "+", which is interpreted as an expiry time for the address. The
first is the year, and the second is the date and time within the year. If a
time cannot be extracted, or if the address is too old, this function fails.

Arguments:
  name       name of the director, for messages
  ob         local options block
  addr       the address in question

Returns:     OK if the address has an expiry time that is in the future;
             FORCEDFAIL if the expiry time was in the past and the
               expire_force_fail option is set;
             FAIL otherwise.
*/

#ifdef never

int
check_timed_address(char *name, smartuser_director_options_block *ob,
  address_item *addr)
{
int expire_time;
char *subject, *value, *sm;

/* Get the subject string which is to be matched against the pattern; expansion
allows it to involve the local part, suffix/prefix, and/or domain. The default
is just to use the local part. */

if (ob->expire_subject == NULL)
  subject = string_copy(addr->local_part);
else
  {
  subject = expand_string(ob->expire_subject);
  if (subject == NULL)
    log_write(0, LOG_MAIN|LOG_PANIC_DIE, "Expansion of %s in %s director "
      "failed: %s", ob->expire_subject, name, expand_string_message);
  }

/* Match the subject to the pattern; failure => no timestamp exists */

if (pcre_exec(ob->expire_pattern_re, NULL, subject, (int)strlen(subject),
  pcre_eopt, NULL, 0) < 0) return FAIL;

DEBUG(9) debug_printf("expire_pattern %s matched %s\n",
  ob->expire_pattern, subject);

/* Set up the numerical expansion variables */

for (expand_nmax = 1; expand_nmax < NSUBEXP; expand_nmax++)
  {
  expand_nstring[expand_nmax] = ob->expire_pattern_re->startp[expand_nmax];
  expand_nlength[expand_nmax] = ob->expire_pattern_re->endp[expand_nmax] -
    expand_nstring[expand_nmax];
  }
expand_nmax--;

/* Expand expire_time to pick out the expiry time. */

value = expand_string(ob->expire_time);

if (value == NULL)
  log_write(0, LOG_MAIN|LOG_PANIC_DIE, "Expansion of %s in %s director "
    "failed: %s", ob->expire_time, name, expand_string_message);

DEBUG(9) debug_printf("expire time is %s\n", value);

/* If the value consists entirely of digits, it is taken as a Unix time.
Otherwise it must contain a "-" to separate the year from the rest. */

if ((sm = strchr(value, '-')) == NULL)
  {
  char *endptr;
  expire_time = strtol(value, &endptr, 10);
  if (*endptr != 0)
    {
    DEBUG(9) debug_printf("failed to extract unix time\n");
    return FAIL;
    }
  }

/* Handle a human-readable time format. The year is terminated by a -
character, and may be in 2-digit or 4-digit form. Then follows month, and then
optional day, hour, minute, second. */

else
  {
  char *s, *endptr;
  int len = 0;
  int ylen = sm - value;
  int year = strtol(value, &endptr, 10);
  struct tm broken_time;

  /* Check that the portion before the - consists of digits only, and get
  their value. Point value to the remainder of the string. */

  if (endptr != sm)
    {
    DEBUG(9) debug_printf("failed to extract year");
    return FAIL;   /* Non-digits before the "-" */
    }
  value = sm + 1;

  /* Initialize the broken-down time structure. */

  memset(&broken_time, 0, sizeof(broken_time));
  broken_time.tm_isdst = -1;

  /* Sort out the year; the value required is the number of years since 1900.
  Any value less than 90 implies the 21st century. This introduces planned
  obsolescence that will need fixing some time before the year 2090. I won't
  be around, and I don't suppose Exim will be either. */

  if (year < 90) year += 100;
    else if (year > 1900) year -= 1900;
  broken_time.tm_year = year;

  /* Check that the remaining string consists entirely of digits, and compute
  their number. There must be at least two digits and no more than ten, and the
  number must be even. While we are at it, subtract '0' from each of them to
  save having to do it every time later. */

  for (s = value; *s != 0; s++)
    {
    if (!isdigit(*s))
      {
      DEBUG(9) debug_printf("found non-digit in date");
      return FAIL;
      }
    len++;
    *s = *s - '0';
    }
  if (len < 2 || len > 10 || (len & 1) != 0)
    {
    DEBUG(9) debug_printf("wrong length date");
    return FAIL;
    }

  /* We know there are at least two digits to specify the month. Any further
  ones are optional. Note that tm_mon starts at zero in struct tm but in Real
  Life it starts at one. The day of the month does *not* start at zero in the
  structure. There's consistency for you. */

  switch(len)
    {
    case 10: broken_time.tm_sec  = value[8] * 10 + value[9];
    case  8: broken_time.tm_min  = value[6] * 10 + value[7];
    case  6: broken_time.tm_hour = value[4] * 10 + value[5];
    case  4: broken_time.tm_mday = value[2] * 10 + value[3];
    default: broken_time.tm_mon  = value[0] * 10 + value[1] - 1;
    }

  expire_time = (int)mktime(&broken_time);
  }

/* Return OK if not expired, otherwise FAIL or FORCEDFAIL as configured. */

return (time(NULL) < expire_time)? OK :
  ob->expire_force_fail? FORCEFAIL : FAIL;
}

#endif


/*************************************************
*              Main entry point                  *
*************************************************/

/* See local README for interface description. */

int
smartuser_director_entry(
  director_instance *dblock,      /* data for this instantiation */
  address_item *addr,             /* address we are working on */
  address_item **addr_local,      /* add it to this if it's local */
  address_item **addr_remote,     /* add it to this if it's remote */
  address_item **addr_new,        /* put new addresses on here */
  address_item **addr_succeed,    /* put finished with addresses here */
  BOOL verify)                    /* TRUE when verifying */
{
smartuser_director_options_block *ob =
  (smartuser_director_options_block *)(dblock->options_block);
char *new_address = NULL;
char *errmess = NULL;
int start, end, domain, rc;

/* Perform file existence and sender verification checks. */

rc = direct_check_fsc(dblock, addr);
if (rc != OK) return rc;

DEBUG(2) debug_printf("%s director called for %s\n", dblock->name, addr->orig);

/* If an expiry pattern is defined, check to see if the address has timed out
or isn't in the correct form for a time-tagged address. In either case, this
director fails, but there is an option to return FORCEFAIL after a timeout. */

/*
if (ob->expire_pattern != NULL &&
    (rc = check_timed_address(dblock->name, ob, addr)) != OK)
  {
  DEBUG(2) debug_printf("no expiry time found or address expired: "
    "returning %d\n", rc);
  expand_nmax = -1;
  return rc;
  }
*/

/* If we get here, this director always matches the local address. */

addr->director = dblock;

/* If a new address was specified, expand it and check that its syntax is
valid. */

if (ob->new_address != NULL)
  {
  char *raw_new_address = expand_string(ob->new_address);

  /* Numeric variables from expiry stuff are now finished with. */

  expand_nmax = -1;

  /* Expansion failure is either a panic or a director fail, according
  to an option. A forced fail in the expansion is always soft. */

  if (raw_new_address == NULL)
    {
    if (ob->panic_expansion_fail && !expand_string_forcedfail)
      {
      log_write(0, LOG_MAIN|LOG_PANIC_DIE, "Expansion of %s in %s director "
        "failed: %s", ob->new_address, dblock->name, expand_string_message);
      }
    else
      {
      DEBUG(4) debug_printf("expansion of %s in %s director failed: %s\n",
        ob->new_address, dblock->name, expand_string_message);
      DEBUG(2) debug_printf("%s director failed for %s\n", dblock->name,
        addr->local_part);
      return FAIL;
      }
    }

  /* Malformed new address or domainless address causes freezing */

  new_address = parse_extract_address(raw_new_address, &errmess, &start,
    &end, &domain, FALSE);

  if (new_address == NULL || domain == 0)
    {
    addr->basic_errno = ERRNO_BADADDRESS2;
    if (errmess == NULL) errmess = "missing domain";
    addr->message =
      string_sprintf("<%s> - bad address generated by %s director: %s\n",
      raw_new_address, dblock->name, errmess);
    addr->errors_address = errors_address;
    addr->special_action = SPECIAL_FREEZE;
    return ERROR;
    }

  new_address = rewrite_address(new_address, TRUE, FALSE);
  domain = strchr(new_address, '@') - new_address + 1;
  }


/* If a transport is configured, queue the address for that transport,
changing the address if required. */

if (dblock->transport != NULL)
  {
  addr->transport = dblock->transport;

  /* Set up the local part and domain from the new address */

  if (new_address != NULL)
    {
    addr->local_part = string_copyn(new_address, domain-1);
    addr->domain = new_address + domain;
    }

  /* Handle local transport */

  if (addr->transport->info->local)
    {
    addr->next = *addr_local;
    *addr_local = addr;
    }

  /* Handle remote transport */

  else
    {
    addr->next = *addr_remote;
    *addr_remote = addr;
    }

  DEBUG(2) debug_printf("  queued for %s transport with local_part: %s\n",
    dblock->transport->name, addr->local_part);
  }

/* Otherwise the new_address field must be set (this is checked in the init
function). Create a new address, copying the parent's errors address and or-ing
the ignore_error flag, and set the parent's child count. Put the parent onto
the succeed chain so that any retry item attached to it gets processed. */

else
  {
  address_item *new_addr = deliver_make_addr(new_address);
  new_addr->next = *addr_new;
  new_addr->errors_address = addr->errors_address;
  new_addr->ignore_error |= addr->ignore_error;
  new_addr->start_director = dblock->new;
  *addr_new = new_addr;
  new_addr->parent = addr;
  addr->child_count++;
  addr->next = *addr_succeed;
  *addr_succeed = addr;

  DEBUG(2) debug_printf("  generated new address: %s%s%s%s\n",
    new_addr->orig,
    (new_addr->errors_address != NULL)? " (errors to " : "",
    (new_addr->errors_address != NULL)? new_addr->errors_address : "",
    (new_addr->errors_address != NULL)? ")" : "");
  }

return OK;
}

/* End of directors/smartuser.c */
