/*************************************************
*     Exim - an Internet mail transport agent    *
*************************************************/

/* Copyright (c) University of Cambridge 1995 - 1997 */
/* See the file NOTICE for conditions of use and distribution. */


#include "../exim.h"
#include "forwardfile.h"



/* Options specific to the forwardfile director. */

optionlist forwardfile_director_options[] = {
  { "check_ancestor",     opt_bool,
      (void *)(offsetof(forwardfile_director_options_block, check_ancestor)) },
  { "check_local_user",   opt_bool,
      (void *)(offsetof(forwardfile_director_options_block, check_local_user)) },
  { "current_directory",  opt_stringptr,
      (void *)(offsetof(forwardfile_director_options_block, current_dir)) },
  { "directory",          opt_stringptr | opt_hidden,
      (void *)(offsetof(forwardfile_director_options_block, file_dir)) },
  { "directory2_transport",opt_transportptr,
      (void *)(offsetof(forwardfile_director_options_block, directory2_transport)) },
  { "directory_transport",opt_transportptr,
      (void *)(offsetof(forwardfile_director_options_block, directory_transport)) },
  { "errors_to",          opt_stringptr,
      (void *)(offsetof(forwardfile_director_options_block, errors_to)) },
  { "file",               opt_stringptr,
      (void *)(offsetof(forwardfile_director_options_block, file)) },
  { "file_directory",     opt_stringptr,
      (void *)(offsetof(forwardfile_director_options_block, file_dir)) },
  { "file_transport",     opt_transportptr,
      (void *)(offsetof(forwardfile_director_options_block, file_transport)) },
  { "filter",             opt_bool,
      (void *)(offsetof(forwardfile_director_options_block, filter)) },
  { "forbid_file",        opt_bool,
      (void *)(offsetof(forwardfile_director_options_block, forbid_file)) },
  { "forbid_filter_log",  opt_bool,
      (void *)(offsetof(forwardfile_director_options_block, forbid_filter_log)) },
  { "forbid_include",     opt_bool,
      (void *)(offsetof(forwardfile_director_options_block, forbid_include)) },
  { "forbid_pipe",        opt_bool,
      (void *)(offsetof(forwardfile_director_options_block, forbid_pipe)) },
  { "forbid_reply",        opt_bool,
      (void *)(offsetof(forwardfile_director_options_block, forbid_reply)) },
  { "freeze_missing_include", opt_bool,
      (void *)(offsetof(forwardfile_director_options_block, freeze_missing_include)) },
  { "home_directory",     opt_stringptr,
      (void *)(offsetof(forwardfile_director_options_block, home_dir)) },
  { "ignore_eacces",     opt_bool,
      (void *)(offsetof(forwardfile_director_options_block, ignore_eacces)) },
  { "ignore_enotdir",     opt_bool,
      (void *)(offsetof(forwardfile_director_options_block, ignore_enotdir)) },
  { "pipe_transport",     opt_transportptr,
      (void *)(offsetof(forwardfile_director_options_block, pipe_transport)) },
  { "reply_transport",    opt_transportptr,
      (void *)(offsetof(forwardfile_director_options_block, reply_transport)) },
  { "rewrite",            opt_bool,
      (void *)(offsetof(forwardfile_director_options_block, rewrite)) },
  { "skip_syntax_errors", opt_bool,
      (void *)(offsetof(forwardfile_director_options_block, skip_syntax_errors)) },
  { "syntax_errors_to",   opt_stringptr,
      (void *)(offsetof(forwardfile_director_options_block, syntax_errors_to)) },
};

/* Size of the options list. An extern variable has to be used so that its
address can appear in the tables drtables.c. */

int forwardfile_director_options_count =
  sizeof(forwardfile_director_options)/sizeof(optionlist);

/* Default private options block for the forwardfile director. */

forwardfile_director_options_block forwardfile_director_option_defaults = {
  NULL,     /* file_dir */
  NULL,     /* home_dir */
  NULL,     /* current_dir */
  NULL,     /* file */
  NULL,     /* errors_to */
  NULL,     /* syntax_errors_to */
  FALSE,    /* check_ancestor */
  TRUE,     /* check_local_user */
  FALSE,    /* filter */
  FALSE,    /* forbid_filter_log */
  FALSE,    /* forbid_file */
  FALSE,    /* forbid_include */
  FALSE,    /* forbid_pipe */
  FALSE,    /* forbid_reply */
  FALSE,    /* ignore_eacces */
  FALSE,    /* ignore_enotdir */
  TRUE,     /* freeze_missing_include */
  TRUE,     /* rewrite */
  FALSE,    /* skip_syntax_errors */
  NULL,     /* directory_transport */
  NULL,     /* directory2_transport */
  NULL,     /* file_transport */
  NULL,     /* pipe_transport */
  NULL,     /* reply_transport */
};



/*************************************************
*          Initialization entry point            *
*************************************************/

/* Called for each instance, after its options have been read, to
enable consistency checks to be done, or anything else that needs
to be set up. */

void forwardfile_director_init(director_instance *dblock)
{
forwardfile_director_options_block *ob =
  (forwardfile_director_options_block *)(dblock->options_block);

/* A file name is mandatory */

if (ob->file == NULL)
  log_write(0, LOG_PANIC_DIE|LOG_CONFIG2, "%s director:\n  "
    "no file name specified", dblock->name);

/* A directory setting is optional, but if it exists it must be absolute,
though we can't check for certain until it has been expanded. */

/*
if (ob->file_dir != NULL && ob->file_dir[0] != '/' &&
    ob->file_dir[0] != '$')
  log_write(0, LOG_PANIC_DIE|LOG_CONFIG2, "%s director:\n  "
    "absolute path name required for the 'directory' option", dblock->name);
*/
    
/* Permit relative paths only if local user checking is set, or if the
directory option (which must be absolute) is set. */

/*
if (!ob->check_local_user && ob->file[0] != '/' && ob->file[0] != '$' &&
     ob->file_dir == NULL)
  log_write(0, LOG_PANIC_DIE|LOG_CONFIG2, "%s director:\n  "
    "absolute file path required when check_local_user and directory are not set",
    dblock->name);
*/
    
/* A transport must *not* be specified */

if (dblock->transport != NULL || dblock->expand_transport != NULL)
  log_write(0, LOG_PANIC_DIE|LOG_CONFIG2, "%s director:\n  "
    "a transport is not allowed for this director", dblock->name);
}



/*************************************************
*              Main entry point                  *
*************************************************/

/* See local README for interface description. */

int forwardfile_director_entry(
  director_instance *dblock,      /* data for this instantiation */
  address_item *addr,             /* address we are working on */
  address_item **addr_local,      /* add it to this if it's local */
  address_item **addr_remote,     /* add it to this if it's remote */
  address_item **addr_new,        /* put new addresses on here */
  address_item **addr_succeed,    /* put old address here on success */
  BOOL verify)                    /* TRUE when verifying */
{
forwardfile_director_options_block *ob =
  (forwardfile_director_options_block *)(dblock->options_block);
address_item *generated = NULL;
char *directory = NULL;
char *errors_to = addr->errors_address;
char *filename;
char *filebuf;
char *error;
char *s, *tag;
struct stat statbuf;
struct passwd *pw;
error_block *eblock = NULL;
FILE *fwd;
int   yield = OK;
int   extracted = 0;
BOOL  stat_directory = TRUE;
BOOL  delivered = FALSE;
BOOL  is_filter = FALSE;
BOOL  filter_header = FALSE;

/* If the check_local_user option is set, check that the local_part is
the login of a local user, and fail if not. Note: the third argument to
direct_finduser() must be NULL here, to prevent a numeric string being
taken as a numeric uid. If the user is found, set directory to the home
directory, and the home expansion variable as well, so that it can be
used while expanding ob->file_dir! */

if (ob->check_local_user)
  {
  if (!direct_finduser(addr->local_part, &pw))
    {
    DEBUG(2) debug_printf("%s director failed for %s (not a user)\n",
      dblock->name, addr->local_part);
    return FAIL;
    }
  else
    {
    directory = pw->pw_dir;
    deliver_home = directory;
    }
  }

/* If the file_directory option is set expand the string, and set it as the
"home" directory. The expansion can contain $home if check_local_user
is set. */

if (ob->file_dir != NULL)
  {
  directory = expand_string(ob->file_dir);
  if (directory == NULL)
    {
    log_write(0, LOG_MAIN, "%s director failed to expand %s: %s", dblock->name,
      ob->file_dir, expand_string_message);
    addr->special_action = SPECIAL_FREEZE;
    return ERROR;
    }
  deliver_home = directory;
  }

/* Perform file existence and sender verification checks now that we
have $home available. */

yield = direct_check_fsc(dblock, addr);
if (yield != OK) return yield;

/* Get the required file name and expand it. If the expansion fails, log the
incident and indicate an internal error. */

filename = expand_string(ob->file);

if (filename == NULL)
  {
  log_write(0, LOG_MAIN, "%s director failed to expand %s: %s", dblock->name,
    ob->file, expand_string_message);
  addr->special_action = SPECIAL_FREEZE;
  return ERROR;
  }

DEBUG(2) debug_printf("%s director: file = %s\n", dblock->name,
  filename);

/* If a directory is set and the file name is not absolute, construct the
complete file name. Otherwise set a flag to prevent an attempt at statting the
directory below. */

if (directory != NULL && filename[1] != ':')
  filename = string_sprintf("%s/%s", directory, filename);
else stat_directory = FALSE;

/* Check that the file name is absolute. Simple checks are done in the
init function, but expansions mean that we have to do a final check here. */

/*
if (filename[0] != '/')
  {
  log_write(0, LOG_MAIN, "%s is not an absolute path for the %s director",
    filename, dblock->name);
  addr->special_action = SPECIAL_FREEZE;
  return ERROR;
  }
/*

/* You might think we could just test for the existence of the required file by
attempting to open it, but life isn't that simple. In many large systems,
.forward files in users' home directories are used, with the home directories
possibly NFS-mounted from some remote place. It doesn't seem possible to detect
the state of "NFS mount inaccessible" just by trying to open a file.

The common case is specified with a relative path name (relative to the home
directory or to a specified directory), and in that case we try to do a bit
better by statting the directory first. If it cannot be statted, assume there
is some mounting problem, and defer the delivery. */

if (directory != NULL && stat_directory)
  {
  if (stat(directory, &statbuf) != 0)
    {
    DEBUG(2) debug_printf("%s director failed to stat %s: deferred\n",
      dblock->name, directory);
    addr->message = string_sprintf("%s director failed to stat %s",
      dblock->name, directory);
    yield = DEFER;
    goto RESTORE_UID;             /* skip forward */
    }
  DEBUG(2) debug_printf("successful stat of %s\n", directory);
  }

/* Now try to open the file for reading. If this fails with a non-existence
error, we have no option but to believe that the file does not exist, so the
director gives up on this address. Some other cases are more dubious, and may
indicate configuration errors. For this reason, their handling is controlled by
options.

ENOTDIR means that something along the line is not a directory: there are
installations that set home directories to be /dev/null for non-login accounts
but in normal circumstances this indicates some kind of configuration error.

EACCES means there's a permissions failure. Some users turn off read
permission on a .forward file to suspend forwarding, but this is probably an
error in any kind of mailing list processing. */

fwd = os_fopen(filename, "rb");
if (fwd == NULL)
  {
  switch(errno)
    {
    case ENOENT:          /* File does not exist */
    DEBUG(2) debug_printf("%s director: no file found\n", dblock->name);
    yield = FAIL;
    break;

    case ENOTDIR:         /* Something on the path isn't a directory */
    if (ob->ignore_enotdir)
      {
      DEBUG(2) debug_printf("%s director: non-directory on path: file assumed "
        "not to exist\n", dblock->name);
      yield = FAIL;
      break;
      }
    goto DEFAULT_ERROR;

    case EACCES:           /* Permission denied */
    if (ob->ignore_eacces)
      {
      DEBUG(2) debug_printf("%s director: permission denied: file assumed not "
        "to exist\n", dblock->name);
      yield = FAIL;
      break;
      }
    /* Else fall through */

    DEFAULT_ERROR:
    default:
    DEBUG(2) debug_printf("%s director failed to open %s: %s: deferred\n",
      dblock->name, filename, strerror(errno));
    addr->message = string_sprintf("%s director failed to open %s: %s",
      dblock->name, filename, strerror(errno));
    /* addr->special_action = SPECIAL_FREEZE; */
    yield = ERROR;
    break;
    }

  goto RESTORE_UID;               /* skip forward */
  }


/* Now check up on the mode of the file. It is tempting to do this stat before
opening the file, and use it as an existence check. However, doing that opens a
small security loophole in that the status could be changed before the file is
opened. Can't quite see what problems this might lead to, but you can't be too
careful where security is concerned. Fstat() on an open file can normally be
expected to succeed, but there are some NFS states where it does not. */

if (fstat(fileno(fwd), &statbuf) != 0)
  {
  yield = DEFER;
  goto CLOSE_RESTORE_UID;         /* skip forward */
  }

/* Read the .forward file and generate new addresses for each entry therein.
We read the file in one go in order to minimize the time we have it open. */

filebuf = store_malloc(statbuf.st_size + 1);
if ((fread(filebuf, 1, statbuf.st_size -1, fwd) != statbuf.st_size -1) && (statbuf.st_size != 0))
  {
  addr->basic_errno = errno;
  addr->message =
    string_sprintf("<%s> - error while reading forward file (%s director)\n",
    addr->orig, dblock->name);
  yield = DEFER;
  goto CLOSE_RESTORE_UID;         /* skip forward */
  }
filebuf[statbuf.st_size -1] = 0;

/* Don't pass statbuf.st_size directly to debug_printf. On some systems it is a
long, which may not be the same as an int. */

DEBUG(2)
  {
  int size = statbuf.st_size;
  debug_printf("%d bytes read from %s\n", size, filename);
  }

/* If the filter option is set, the file is to be interpreted as a filter
file instead of a straight list of addresses, if it starts with
"# Exim filter ..." (any capitilization, spaces optional). Check for this
in all cases, so as to give a warning message if it is found when filtering
is not enabled. */

s = filebuf;
tag = "# exim filter";
while (isspace(*s)) s++;           /* Skips initial blank lines */
for (; *tag != 0; s++, tag++)
  {
  if (*tag == ' ')
    {
    while (*s == ' ' || *s == '\t') s++;
    s--;
    }
  else if (tolower(*s) != tolower(*tag)) break;
  }
if (*tag == 0) filter_header = TRUE;

/* Filter interpretation is done by a general function that is also called from
the filter testing option (-bf). The second-last argument specifies whether the
log command should be locked out or skipped; this is done when verifying, or if
we have not set the uid to a local user, or when explicitly configured. Set up
the value of extracted to be the same as it is from parse_extract_addresses().
*/

if (filter_header && ob->filter)
  {
  DEBUG(2) debug_printf("file is a filter file\n");
  is_filter = TRUE;
  extracted = filter_interpret(filebuf, &generated, &delivered, NULL,
    &error,
      verify? 1 :                    /* skip logging if verifying */
      ob->forbid_filter_log? 2 : 0,  /* lock out if forbidden */
    ob->rewrite)? 0 : -1;
  }

/* Otherwise it's a vanilla .forward file; call parse_extract_addresses()
to get the values. The yield is 0=>OK, -1=>error, +1=> failed to open an
:include: file. */

else
  {
  DEBUG(2) debug_printf("file is not a filter file\n");

  /* There is a common function for use by forwarding and aliasing
  directors that extracts a list of addresses from a text string.
  Setting the fourth argument TRUE says that generating no addresses
  (from valid syntax) is no error.

  The forward file may include :include: items, and we have to be
  careful about permissions for reading them. The extracting function
  will check that include files begin with a specified string, unless
  NULL is supplied. Supplying "*" locks out :include: files, since they
  must be absolute paths. We lock them out if (a) requested to do so or
  (b) we haven't used seteuid and there's no directory to check. If
  seteuid has been used, just try to read anything; otherwise restrict
  to the directory or lock out if none. */

  extracted = parse_extract_addresses(filebuf, &generated, &error,
    TRUE,                                      /* no addresses => no error */
    FALSE,                                     /* don't recognize :blackhole: */
    ob->rewrite,                               /* rewrite if configured */
    ob->forbid_include? "*" :                  /* includes forbidden */
    (directory == NULL)? "*" :                 /* if no directory, lock out */
    directory,                                 /* else restrain to directory */
    ob->skip_syntax_errors? &eblock : NULL);
  }

/* The store for holding the forward file is now finished with. */

store_free(filebuf);

/* At this point we are finished with the .forward file. Close it, and, if
seteuid was used above, restore the previous effective uid and gid. The dreaded
goto is used above to skip to this code when errors are detected. */

CLOSE_RESTORE_UID:

fclose(fwd);

RESTORE_UID:

/* If there has been an error, return the error value now. Subsequently we
can just return directly on error, since there is no further need to mess with
the uid or close the file. */

if (yield != OK) return yield;

/* Extraction failed */

if (extracted != 0)
  {
  /* If extraction from a filter file failed, it is a "probably user error", to
  use a good old IBM term. Just defer delivery and let the user clean things
  up. */

  if (is_filter)
    {
    addr->basic_errno = ERRNO_BADFORWARD;
    addr->message =
      string_sprintf("<%s> - error in filter file: %s", addr->orig,
        error);
    return DEFER;
    }

  /* If extraction from a .forward file failed, freeze and yield ERROR if
  it was a missing :include: file and freeze_missing_include is TRUE. Other-
  wise just DEFER and hope things get fixed eventually. */

  else
    {
    addr->basic_errno = ERRNO_BADFORWARD;
    addr->message =
      string_sprintf("<%s> - error in forward file%s: %s", addr->orig,
        filter_header? " (filtering not enabled)" : "",
        error);
    if (extracted > 0 && ob->freeze_missing_include)
      {
      addr->special_action = SPECIAL_FREEZE;
      return ERROR;
      }
    else return DEFER;
    }
  }

/* If skip_syntax_errors was set and there were syntax errors in the list,
error messages will be present in eblock. Log them and send a message if
so configured, using a function that is common to aliasfile and forwardfile. */

if (eblock != NULL && !moan_skipped_syntax_errors(dblock->name, filename,
    eblock, verify? NULL : ob->syntax_errors_to))
  {
  addr->special_action = SPECIAL_FREEZE;
  return ERROR;
  }

/* If this director has a local errors_to setting for where to send error
messages for its children, expand it, and then check that it is a valid
address before using it, except when verifying. Otherwise there could be
directing loops if a silly config is set. */

if (ob->errors_to != NULL)
  {
  char *s = expand_string(ob->errors_to);
  if (s == NULL)
    {
    log_write(0, LOG_MAIN, "%s director failed to expand %s: %s", dblock->name,
      ob->errors_to, expand_string_message);
    addr->special_action = SPECIAL_FREEZE;
    return ERROR;
    }

  /* While verifying, set the sender address to null, because that's what
  it will be when sending an error message, and there are now configuration
  options that control the running of directors and routers by checking
  the sender address. When testing an address, there may not be a sender
  address. */

  if (verify) errors_to = s; else
    {
    char *snew;
    int save1 = 0;
    if (sender_address != NULL)
      {
      save1 = sender_address[0];
      sender_address[0] = 0;
      }
    if (verify_address(s, NULL, NULL, &snew, vopt_is_recipient | vopt_local)
      == OK) errors_to = snew;
    if (sender_address != NULL) sender_address[0] = save1;
    }
  }

/* Add the new addresses to the list of new addresses, copying in the
uid, gid and permission flags for use by pipes and files and autoreplies,
setting the parent, and or-ing its ignore_error flag.

If the generated address is the same as one of its ancestors, and the
check_ancestor flag is set, do not use this generated address, but replace it
with a copy of the input address. This is to cope with cases where A is aliased
to B and B has a .forward file pointing to A. We can't just pass on the old
address by returning FAIL, because it must act as a general parent for
generated addresses, and only get marked "done" when all its children are
delivered. */

while (generated != NULL)
  {
  address_item *next = generated;
  generated = next->next;
  next->parent = addr;
  next->ignore_error |= addr->ignore_error;
  next->start_director = dblock->new;
  addr->child_count++;
  next->next = *addr_new;
  *addr_new = next;

  if (ob->check_ancestor)
    {
    address_item *parent;
    for (parent = addr; parent != NULL; parent = parent->parent)
      {
      if (strcmp(next->orig, parent->orig) == 0)
        {
        DEBUG(2) debug_printf("generated parent replaced by child\n");
        next->orig = string_copy(addr->orig);
        break;
        }
      }
    }

  if (errors_to != NULL) next->errors_address = errors_to;

  if (next->pfr)
    {
    next->director = dblock;
    if (ob->home_dir != NULL) next->home_dir = ob->home_dir;
      else if (directory != NULL) next->home_dir = string_copy(directory);
    next->current_dir = ob->current_dir;
    next->allow_pipe = !ob->forbid_pipe;
    next->allow_file = !ob->forbid_file;
    next->allow_reply = !ob->forbid_reply;

    /* Forwardfile can produce opipes or files or autoreplies; if the transport
    setting is null, the global setting will get used later. */

    if (next->orig[0] == '|')
      next->transport = ob->pipe_transport;
    else
      if (next->orig[0] == '>') next->transport = ob->reply_transport;
    else
      {
      int len = (int)strlen(next->orig);
      if (next->orig[len-1] == '/')
        {
        next->transport = ob->directory_transport;
        if (len > 1 && next->orig[len-2] == '/' &&
            ob->directory2_transport != NULL)
          next->transport = ob->directory2_transport;
        }
      else next->transport = ob->file_transport;
      }
    }

  DEBUG(2) debug_printf("%s director generated %s\n%s%s%s%s%s%s%s",
    dblock->name,
    next->orig,
    next->pfr? "  pipe, file, or autoreply\n" : "",
    (errors_to != NULL)? "  errors to " : "",
    (errors_to != NULL)? errors_to : "",
    (errors_to != NULL)? "\n" : "",
    (next->transport == NULL)? "" : "  transport=",
    (next->transport == NULL)? "" : (next->transport)->name,
    (next->transport == NULL)? "" : "\n");
  }

/* If the filter interpreter returned "delivered" then we have succeeded
in completely handling this address. Otherwise, we have to arrange for this
address to be passed on to subsequent directors. Returning FAIL would appear to
be the answer, but it isn't, because successful delivery of the base address
gets it marked "done", so deferred generated addresses never get tried again.
We have to generate a new version of the base address, as if there were a
"deliver" command in the filter file, with the original address as parent.
However, we don't need to do this if there were no generated addresses. */

if (is_filter)
  {
  address_item *next;

  if (!delivered)
    {
    if (addr->child_count <= 0) return FAIL;

    next = deliver_make_addr(addr->orig);
    next->parent = addr;
    next->ignore_error |= addr->ignore_error;
    next->pfr = addr->pfr;
    addr->child_count++;
    next->next = *addr_new;
    *addr_new = next;

    if (errors_to != NULL) next->errors_address = errors_to;

    DEBUG(2) debug_printf("%s director generated %s\n%s%s%s%s",
      dblock->name,
      next->orig,
      next->pfr? "  pipe, file, or autoreply\n" : "",
      (errors_to != NULL)? "  errors to " : "",
      (errors_to != NULL)? errors_to : "",
      (errors_to != NULL)? "\n" : "");
    }

  yield = OK;
  }

/* If the forward file generated no addresses, it is not an error. The
director just fails. Compare aliasfile, which is different. */

else yield = (addr->child_count <= 0)? FAIL : OK;

/* If the yield is OK, put the original address onto the succeed queue so
that any retry items that get attached to it get processed. */

if (yield == OK)
  {
  addr->next = *addr_succeed;
  *addr_succeed = addr;
  }

return yield;
}

/* End of directors/forwardfile.c */
