/*************************************************
*     Exim - an Internet mail transport agent    *
*************************************************/

/* Copyright (c) University of Cambridge 1995 - 1997 */
/* See the file NOTICE for conditions of use and distribution. */


#include "../exim.h"
#include "aliasfile.h"



/* Options specific to the aliasfile director. */

optionlist aliasfile_director_options[] = {
  { "*partial_match", opt_int | opt_hidden,
      (void *)(offsetof(aliasfile_director_options_block, partial_match)) },
  { "current_directory",     opt_stringptr,
      (void *)(offsetof(aliasfile_director_options_block, current_dir)) },
  { "directory",          opt_stringptr | opt_hidden,
      (void *)(offsetof(aliasfile_director_options_block, home_dir)) },
  { "directory2_transport",opt_transportptr,
      (void *)(offsetof(aliasfile_director_options_block, directory2_transport)) },
  { "directory_transport",opt_transportptr,
      (void *)(offsetof(aliasfile_director_options_block, directory_transport)) },
  { "errors_to",          opt_stringptr,
      (void *)(offsetof(aliasfile_director_options_block, errors_to)) },
  { "expand",             opt_bool,
      (void *)(offsetof(aliasfile_director_options_block, expand)) },
  { "file",               opt_stringptr,
      (void *)(offsetof(aliasfile_director_options_block, file)) },
  { "file_transport",     opt_transportptr,
      (void *)(offsetof(aliasfile_director_options_block, file_transport)) },
  { "forbid_file",        opt_bool,
      (void *)(offsetof(aliasfile_director_options_block, forbid_file)) },
  { "forbid_pipe",        opt_bool,
      (void *)(offsetof(aliasfile_director_options_block, forbid_pipe)) },
  { "freeze_missing_include", opt_bool,
      (void *)(offsetof(aliasfile_director_options_block, freeze_missing_include)) },
  { "home_directory",     opt_stringptr,
      (void *)(offsetof(aliasfile_director_options_block, home_dir)) },
  { "include_domain",     opt_bool,
      (void *)(offsetof(aliasfile_director_options_block, include_domain)) },
  { "optional",           opt_bool,
      (void *)(offsetof(aliasfile_director_options_block, optional)) },
  { "pipe_transport",     opt_transportptr,
      (void *)(offsetof(aliasfile_director_options_block, pipe_transport)) },
  { "rewrite",            opt_bool,
      (void *)(offsetof(aliasfile_director_options_block, rewrite)) },
  { "search_type",        opt_searchtype,
      (void *)(offsetof(aliasfile_director_options_block, search_type)) },
  { "skip_syntax_errors", opt_bool,
      (void *)(offsetof(aliasfile_director_options_block, skip_syntax_errors)) },
  { "syntax_errors_to",   opt_stringptr,
      (void *)(offsetof(aliasfile_director_options_block, syntax_errors_to)) },
};

/* Size of the options list. An extern variable has to be used so that its
address can appear in the tables drtables.c. */

int aliasfile_director_options_count =
  sizeof(aliasfile_director_options)/sizeof(optionlist);

/* Default private options block for the aliasfile director. */

aliasfile_director_options_block aliasfile_director_option_defaults = {
  NULL,     /* file */
  NULL,     /* home_dir */
  NULL,     /* current_dir */
  NULL,     /* errors_to */
  NULL,     /* syntax_errors_to */
  -1,       /* partial_match */
  -1,       /* search_type */
  FALSE,    /* include_domain */
  FALSE,    /* expand */
  FALSE,    /* optional */
  FALSE,    /* forbid_file */
  FALSE,    /* forbid_pipe */
  TRUE,     /* freeze_missing_include */
  TRUE,     /* rewrite */
  FALSE,    /* skip_syntax_errors */
  NULL,     /* directory_transport */
  NULL,     /* directory2_transport */
  NULL,     /* file_transport */
  NULL      /* pipe_transport */
};



/*************************************************
*          Initialization entry point            *
*************************************************/

/* Called for each instance, after its options have been read, to
enable consistency checks to be done, or anything else that needs
to be set up.

Argument:
  dblock       a pointer to the director instance block

Returns:       nothing
*/

void
aliasfile_director_init(director_instance *dblock)
{
aliasfile_director_options_block *ob =
  (aliasfile_director_options_block *)(dblock->options_block);

/* A search type is mandatory */

if (ob->search_type < 0)
  log_write(0, LOG_PANIC_DIE|LOG_CONFIG2, "%s director:\n  "
    "a search type option is required", dblock->name);

/* For single-key+file search types, a file name is mandatory; for other types
a query is mandatory. An absolute file name is mandatory for lsearch and dbm;
but can't check for absoluteness if the name is being looked up. */

if (ob->search_type < stype_querystyle)
  {
  if (ob->file == NULL)
    log_write(0, LOG_PANIC_DIE|LOG_CONFIG2, "%s director:\n  "
      "no file name specified", dblock->name);

/*  if ((ob->search_type == stype_lsearch || ob->search_type == stype_dbm) &&
       ob->file[0] != '/' && ob->file[0] != '$')
    log_write(0, LOG_PANIC_DIE|LOG_CONFIG2, "%s director:\n  "
      "an absolute file path name is required for lsearch or dbm", dblock->name); */
  }
else
    log_write(0, LOG_PANIC_DIE|LOG_CONFIG2, "%s director:\n  "
      "invalid search type specified", dblock->name);
}

/*************************************************
*              Main entry point                  *
*************************************************/

/* See local README for interface description */

int
aliasfile_director_entry(
  director_instance *dblock,      /* data for this instantiation */
  address_item *addr,             /* address we are working on */
  address_item **addr_local,      /* add it to this if it's local */
  address_item **addr_remote,     /* add it to this if it's remote */
  address_item **addr_new,        /* put new addresses on here */
  address_item **addr_succeed,    /* put the old one here on success */
  BOOL verify)                    /* true if verifying */
{
aliasfile_director_options_block *ob =
  (aliasfile_director_options_block *)(dblock->options_block);
address_item *generated = NULL;
char *errors_to = addr->errors_address;
char *filename;
char *query;
char *aliastext;
char *error;
void *handle;
error_block *eblock = NULL;
int  extracted, rc;

/* Perform file existence and sender verification checks. */

rc = direct_check_fsc(dblock, addr);
if (rc != OK) return rc;

/* For single-key+file search types, set the required file name and expand it.
If the expansion fails, log the incident and indicate an internal error. The
file name has already been checked for absoluteness, at initialization time,
but only if it did not start with an expansion, so we double check here. */

if (ob->search_type < stype_querystyle)
  {
  query = (ob->include_domain)?
    string_sprintf("%s@%s", addr->local_part, addr->domain) : addr->local_part;
  filename = expand_string(ob->file);
  if (filename == NULL)
    {
    log_write(0, LOG_MAIN|LOG_PANIC, "%s director failed to expand %s: %s",
      dblock->name, ob->file, expand_string_message);
    addr->message = string_sprintf("%s director failed to expand %s: %s",
      dblock->name, ob->file, expand_string_message);
    addr->special_action = SPECIAL_FREEZE;
    return ERROR;
    }
/*  else if ((ob->search_type == stype_lsearch || ob->search_type == stype_dbm) &&
           filename[0] != '/')
    {
    log_write(0, LOG_MAIN|LOG_PANIC, "%s director requires absolute file name "
      "for lsearch or dbm: "
      "%s generated from expanding %s", dblock->name, filename, ob->file);
    addr->special_action = SPECIAL_FREEZE;
    return ERROR;
    } */

  DEBUG(2) debug_printf("%s director: file = %s search type = %d\n",
    dblock->name, filename, ob->search_type);
  }

/* For query-style lookups, expand the query and set the filename NULL */

/* Open the file (or whatever) for searching, according to the search type that
is set. The search functions return a handle identifying the search. For files
this is a FILE * or a DBM *; for other things is is < 0. If the optional flag
is set, failure to open is not an error; we just fail to direct. */

DEBUG(9) debug_printf("file=%s query=%s\n",
  (filename == NULL)? "NULL" : filename, query);

handle = search_open(filename, ob->search_type, &error);

if (handle == NULL)
  {
  if (ob->optional && errno == ENOENT)
    {
    DEBUG(2) debug_printf("%s director skipped: file failed to open and "
      "optional flag set\n", dblock->name);
    return FAIL;
    }
  addr->basic_errno = ERRNO_BADALIAS;
  addr->message = error;
  log_write(0, LOG_MAIN|LOG_PANIC, "%s director: %s", dblock->name, error);
  return ERROR;
  }

/* Now search the file (or whatever) for the entry we are interested in.
The text is returned in dynamic store. */

aliastext = search_find(handle, filename, query, ob->search_type,
  ob->partial_match, NULL, &error);

if (aliastext == NULL)
  {
  DEBUG(2) debug_printf("%s director failed for %s: %s\n", dblock->name,
    addr->local_part, error);
  return FAIL;
  }

/* If the expand option is set, pass the text through the string expander. */

if (ob->expand)
  {
  char *newtext = expand_string(aliastext);
  if (newtext == NULL)
    {
    log_write(0, LOG_MAIN, "%s director failed to expand %s (generated from "
      "local part %s)", dblock->name, aliastext, addr->local_part);
    addr->special_action = SPECIAL_FREEZE;
    return ERROR;
    }
  store_free(aliastext);
  aliastext = newtext;
  }

/* If this director has a local errors_to setting for where to send error
messages for its children, expand it, and then check that it is a valid
address before using it, except when just verifying an address. Otherwise
there could be directing loops if someone sets up a silly configuration. */

if (ob->errors_to != NULL)
  {
  char *s = expand_string(ob->errors_to);
  if (s == NULL)
    {
    log_write(0, LOG_MAIN, "%s director failed to expand %s", dblock->name,
      ob->errors_to);
    addr->special_action = SPECIAL_FREEZE;
    return ERROR;
    }

  /* While verifying, set the sender address to null, because that's what
  it will be when sending an error message, and there are now configuration
  options that control the running of directors and routers by checking
  the sender address. When testing an address, there may not be a sender
  address. */

  if (verify) errors_to = s; else
    {
    char *snew;
    int save1 = 0;
    if (sender_address != NULL)
      {
      save1 = sender_address[0];
      sender_address[0] = 0;
      }
    if (verify_address(s, NULL, NULL, &snew, vopt_is_recipient | vopt_local)
      == OK) errors_to = snew;
    if (sender_address != NULL) sender_address[0] = save1;
    }
  }

/* If there is a transport specified for the director, then set up this
address to use that transport. Ignore the alias text. */

if (dblock->transport != NULL)
  {
  addr->transport = dblock->transport;

  if (errors_to != NULL) addr->errors_address = errors_to;
  addr->director = dblock;
  addr->home_dir = ob->home_dir;
  addr->current_dir = ob->current_dir;

  if (addr->transport->info->local)
    {
    addr->next = *addr_local;
    *addr_local = addr;
    }

  else
    {
    addr->next = *addr_remote;
    *addr_remote = addr;
    }

  DEBUG(2)
    {
    debug_printf("  queued for %s transport\n", dblock->transport->name);
    }

  return OK;
  }

/* There is a common function for use by aliasing and aliasing directors that
extracts a list of addresses from a text string. Setting the fourth-last
argument FALSE makes generating no addresses an error. However, setting the
third-last TRUE allows an address that is ":blackhole:" in the alias file
to generate nothing without causing an error. Setting the last argument NULL
causes no checks to be made on :include: files. */

extracted = parse_extract_addresses(aliastext, &generated, &error, FALSE,
  TRUE, ob->rewrite, NULL, ob->skip_syntax_errors? &eblock : NULL);
store_free(aliastext);

/* If extraction failed, return error and freeze, unless it was a missing
include file and no_freeze_missing_include is set. */

if (extracted != 0)
  {
  addr->basic_errno = ERRNO_BADALIAS;
  addr->message = (filename != NULL)?
    string_sprintf("<%s> - error in alias file %s: %s", addr->orig, filename,
      error) :
    string_sprintf("<%s> - error in alias lookup: %s", addr->orig, error);
  if (extracted > 0 && !ob->freeze_missing_include) return DEFER;
  addr->special_action = SPECIAL_FREEZE;
  return ERROR;
  }

/* If skip_syntax_errors was set and there were syntax errors in the list,
error messages will be present in eblock. Log them, and sent them off in
a mail message if so configured. A common function is used by aliasfile
and forwardfile. */

if (eblock != NULL && !moan_skipped_syntax_errors(dblock->name, filename,
    eblock, verify? NULL : ob->syntax_errors_to))
  {
  addr->special_action = SPECIAL_FREEZE;
  return ERROR;
  }

/* Add the new addresses to the list of new addresses, setting the
parent, and or-ing its ignore_error flag. Also record the setting for
any starting director. */

while (generated != NULL)
  {
  address_item *next = generated;
  generated = next->next;
  next->parent = addr;
  next->ignore_error |= addr->ignore_error;
  next->start_director = dblock->new;
  addr->child_count++;

  next->next = *addr_new;
  *addr_new = next;

  if (errors_to != NULL) next->errors_address = errors_to;

  if (next->pfr)
    {
    next->director = dblock;
    next->home_dir = ob->home_dir;
    next->home_dir = ob->current_dir;
    next->allow_pipe = !ob->forbid_pipe;
    next->allow_file = !ob->forbid_file;

    /* Aliasfile can produce only pipes or files; if the transport
    setting is null, the global setting will get used later. */

    if (next->orig[0] == '|') next->transport = ob->pipe_transport; else
      {
      int len = (int)strlen(next->orig);
      if (next->orig[len-1] == '/')
        {
        next->transport = ob->directory_transport;
        if (len > 1 && next->orig[len-2] == '/' &&
            ob->directory2_transport != NULL)
          next->transport = ob->directory2_transport;
        }
      else next->transport = ob->file_transport;
      }
    }

  DEBUG(2)
    {
    debug_printf("%s director generated %s\n%s%s%s%s%s%s%s",
      dblock->name,
      next->orig,
      next->pfr? "  pipe, file, or autoreply\n" : "",
      (errors_to != NULL)? "  errors to " : "",
      (errors_to != NULL)? errors_to : "",
      (errors_to != NULL)? "\n" : "",
      (next->transport == NULL)? "" : "  transport=",
      (next->transport == NULL)? "" : (next->transport)->name,
      (next->transport == NULL)? "" : "\n");

    debug_printf("home=%s\n", (next->home_dir == NULL)? "null" :
      next->home_dir);
    }
  }

/* If no children were generated for this address, but no error was given,
it means that there was just a :blackhole: entry in the alias file. Log
something. */

if (addr->child_count <= 0 && !verify && !address_test_mode)
  log_write(0, LOG_MAIN, "=> :blackhole: <%s> D=%s", addr->orig, dblock->name);

/* Put the original address onto the succeed queue. This ensures that any
retry item that it acquires gets processed. */

addr->next = *addr_succeed;
*addr_succeed = addr;

return OK;
}

/* End of director/aliasfile.c */
