/*////////////////////////////////////////////////////////////////////////
Copyright (c) 1994-1998 Electrotechnical Laboratry (ETL), AIST, MITI
Copyright (c) 1994-1998 Yutaka Sato

Permission to use, copy, and distribute this material for any purpose
and without fee is hereby granted, provided that the above copyright
notice and this permission notice appear in all copies.
ETL MAKES NO REPRESENTATIONS ABOUT THE ACCURACY OR SUITABILITY OF THIS
MATERIAL FOR ANY PURPOSE.  IT IS PROVIDED "AS IS", WITHOUT ANY EXPRESS
OR IMPLIED WARRANTIES.
/////////////////////////////////////////////////////////////////////////
Content-Type:	program/C; charset=US-ASCII
Program:	inets.c (INET Socket manipulation)
Author:		Yutaka Sato <ysato@etl.go.jp>
Description:
History:
	March94	created
//////////////////////////////////////////////////////////////////////#*/

#include <stdio.h>
#include <string.h>
#include "vsignal.h"
#include "vsocket.h"
extern double Time();

char inets_errmsg[256];

#ifndef Verbose
extern int LOG_VERBOSE;
#define Verbose LOG_VERBOSE==0 ? 0 : sv1vlog
#endif

int (*STORE_DB)();
extern char *Sprintf();
extern char *strdup();
extern char **dupv();
char *inet_ntoaV4();

typedef struct sockaddr *SAP;
typedef struct sockaddr_in *SAIP;
static struct sockaddr_in ina0;

int DNS_TIMEOUT = 60;
int ACC_TIMEOUT = 30;
int CON_TIMEOUT = 30;
int LIN_TIMEOUT = 30;

int SOCKS_client = 0;

int CONNERR_TIMEOUT;
int CONNERR_CANTRESOLV;
int CONNERR_UNREACH;
int CONNERR_REFUSED;

static int CACHE_ONLY = 0;
static int HOSTS_PREDEF;
       int HOSTS_CACHE_LIFE_MIN = 10;
       int HOSTS_CACHE_LIFE_MAX = 0;
extern int RES_CACHE_DISABLE;

static jmp_buf jmpEnv;
static void sigALRM(sig){ longjmp(jmpEnv,SIGALRM); }
static void sigPIPE(sig){ longjmp(jmpEnv,SIGPIPE); }

Cbind(s,name,namelen,dsthost_ipa)
	struct sockaddr *name;
{
	if( SOCKS_client ){
		Verbose("Rbind ...\n");
		return Rbind(s,name,namelen,dsthost_ipa);
	}else	return bind(s,name,namelen);
}
Caccept(s,addr,addrlen)
	struct sockaddr *addr;
	int *addrlen;
{
	if( SOCKS_client ){
		Verbose("Raccept...\n");
		return Raccept(s,addr,addrlen);
	}else	return accept(s,addr,addrlen);
}
Clisten(s,backlog)
{
	if( SOCKS_client ){
		Verbose("Rlisten...\n");
		return Rlisten(s,backlog);
	}else	return listen(s,backlog);
}
Cgetsockname(s,name,namelen)
	struct sockaddr *name;
	int *namelen;
{
	if( SOCKS_client ){
		Verbose("Rgetsockname...\n");
		if( Rgetsockname(s,name,namelen)==0 && ((SAIP)name)->sin_port )
			return 0;
		Verbose("-- retry normal getsockname...\n");
	}
	return getsockname(s,name,namelen);
}
Bconnect(s,name,namelen,direct)
	struct sockaddr *name;
{	int rcode;
	int timer;

	if( CON_TIMEOUT ){
		timer = pushTimer("Bconnect",sigALRM,CON_TIMEOUT);
		if( setjmp(jmpEnv) != 0 ){
			sv1log("*** CON_TIMEOUT: %d secs.\n",CON_TIMEOUT);
			popTimer(timer);
			CONNERR_TIMEOUT = 1;
			return -1;
		}
	}

	if( SOCKS_client && !direct ){
		Verbose("Rconnect...\n");
		rcode = Rconnect(s,name,namelen);
	}else	rcode = connect(s,name,namelen);

	if( CON_TIMEOUT )
		popTimer(timer);

	return rcode;
}

extern struct hostent *Rgethostbyname();
struct hostent *Dgethostbyname(name)
	char *name;
{	struct hostent *ht;
	int fd;

	if( name[0] == 0 )
		return NULL;

	/* SPECIAL HOST NAME IN DeleGate */
	if( strcmp(name,"-")==0 || strcmp(name,"-.-")==0 )
		return NULL;

	fd = nextFD();
	ht = NULL;

	if( ht == NULL )
	if( SOCKS_client ){
		Verbose("Rgethostbyname(%s).\n",name);
		ht = Rgethostbyname(name);
	}
	if( ht == NULL ){
		Verbose("gethostbyname(%s).\n",name);
		ht = gethostbyname(name);
	}
	usedFD(fd);
	return ht;
}
struct hostent *Dgethostbyaddr(addr,len,type)
	char *addr;
{	struct hostent *ht;
	int fd;

	if( *(int*)addr == 0 || *(int*)addr == -1 )
		return NULL;

	fd = nextFD();
	ht = gethostbyaddr(addr,len,type);

	usedFD(fd);
	return ht;
}
static struct hostent *Dgethostbynameaddr(name,addr,len,type)
	char *name,*addr;
{
	if( name != NULL )
		return Dgethostbyname(name);
	else	return Dgethostbyaddr(addr,len,type);
}

extern char *getenv();
static setenv(name,value)
	char *name,*value;
{	char *oval,env[256];

	if( value && value[0] ){
		if( oval = getenv(name) )
			if( strcmp(oval,value) == 0 )
				return;
		sprintf(env,"%s=%s",name,value);
		putenv(strdup(env));
	}
}

static order1(typespec,order,servers)
	char *typespec,*order,*servers[];
{	char type[128],arg[256];

	type[0] = arg[0] = 0;
	sscanf(typespec,"%[^:]:%s",type,arg);

	if( strcasecmp(type,"cache") == 0 )
		strcat(order,"C");
	else
	if( strcasecmp(type,"file") == 0 )
		strcat(order,"F");
	else
	if( strcasecmp(type,"nis") == 0 )
		strcat(order,"N");
	else
	if( strcasecmp(type,"dns") == 0 )
		strcat(order,"D");
	else
	if( strcasecmp(type,"sys") == 0 )
		strcat(order,"S");

	if( arg[0] )
		sprintf(order+strlen(order),":%s,",arg);
	return 0;
}

#define RESOLVERS_SIZ	512

static FILE *res_logf;
extern int (*RES_log)();
static res_log(which,byname,name,rv,cname)
	char *name,*rv[],*cname;
{	static double Start;
	double Now;

	Now = Time();
	if( which == 0 )
		Start = Now;
	else{
		fprintf(res_logf,"%.3f %.4f %d %c %s\n",
			Now,Now-Start,Getpid(),which,name);
		fflush(res_logf);
	}
}

extern char *strrpbrk();
init_inets(name,resolv,conf,ns,verify,rr,debug,log,socks,addenv)
	char *name,*resolv,*conf,*ns,*verify,*rr,*debug,*log,*socks;
	int (*addenv)();
{	char *dp;

	if( debug != NULL )
		RES_debug(debug);
	if( rr != NULL )
		RES_roundrobin(rr);

	if( log != NULL ){
		if( res_logf = fopen(log,"a") ){
			setCloseOnExec(fileno(res_logf));
			RES_log = res_log;
		}
	}

	if( resolv != NULL ){
		char order[RESOLVERS_SIZ],porder[RESOLVERS_SIZ];
		order[0] = 0;
		scan_commaList(resolv,0,order1,order);
		RES_order(order,porder);
		if( *resolv == 0 )
			CACHE_ONLY = 1;
	}
	if( conf != NULL )
		RES_conf(conf);
	if( ns != NULL )
		RES_ns(ns);
	if( verify != NULL )
		RES_verify(verify);

	if( dp = strrpbrk(name,"/\\") )
		name = dp + 1;

	if( resolv == NULL ){
		if( strcmp(name,"delegatedR") == 0 )
			(*addenv)("RESOLV","dns");
	}

	if( socks == NULL ){
		if( strcmp(name,"delegatedS") == 0 ){
			(*addenv)("SOCKS","");
			(*addenv)("RESOLV","dns");
			SOCKS_client = 1;
		}
		socks_init(name,NULL,NULL,NULL);
	}else{
		char socksb[128],*sv[4];
		int ni;

		SOCKS_client = 1;
		strcpy(socksb,socks);
		sv[0] = sv[1] = sv[2] = 0;
		ni = stoV(socksb,3,sv,'/');
		if( 0 < ni && sv[0] ) setenv("SOCKS_SERVER",sv[0]);
		if( 1 < ni && sv[1] ) setenv("SOCKS_NS",    sv[1]);
		if( 2 < ni && sv[2] ) setenv("SOCKS_DNAME", sv[2]);
		socks_init(name,sv[0],sv[1],sv[2]);
	}
	if( SOCKS_client )
		SOCKSinit(name);
}

int myownSOCKS = 0;
withoutSOCKSLIB()
{
	SOCKS_client = 0;
	myownSOCKS = 1;
}


#define NHOSTS 128
static int NHosts = 0;
typedef struct {
	int	hc_index;
	int	hc_freq;
	int	hc_predef;
	int	hc_mtime;
 struct hostent	hc_hostent;
} Hostent;
static Hostent *HostsCache[NHOSTS];

static char *myFQDN = NULL;
static char  myFQDN_unknown[32];

static int ADDRLIST_RR;
static shuffle_addrlist(hp)
	struct hostent *hp;
{	char *haddr,*haddr0;
	int hi;

	if( hp->h_addr_list[0] == NULL || hp->h_addr_list[1] == NULL )
		return;

	if( ADDRLIST_RR ){
		haddr0 = hp->h_addr_list[0];
		for( hi = 0; haddr = hp->h_addr_list[hi+1]; hi++ )
			hp->h_addr_list[hi] = hp->h_addr_list[hi+1];
		hp->h_addr_list[hi] = haddr0;
	}
}

static findName1(hp,name)
	struct hostent *hp;
	char *name;
{	char *name1;
	int ai;

	if( hp->h_name )
		if( strcmp(name,hp->h_name) == 0 )
			return 1;

	if( hp->h_aliases )
	for( ai = 0; name1 = hp->h_aliases[ai]; ai++ )
		if( strcmp(name,name1) == 0 )
			return 1;
	return 0;
}
static findAddr1(hp,addr,len,type)
	struct hostent *hp;
	char *addr;
{	int ai;
	char *addr1;

	for( ai = 0; addr1 = hp->h_addr_list[ai]; ai++ )
		if( bcmp(addr,addr1,len) == 0 )
			return 1;
	return 0;
}
static char *dump_HOST1();
static Hostent *findHostCache(name,addr,len,type)
	char *name,*addr;
{	int hi,ai,freq;
	Hostent *Hp;
	struct hostent *hp;

	for( hi = 0; hi < NHosts; hi++ ){
		Hp = HostsCache[hi];
		hp = &Hp->hc_hostent;
		if( name ){
			if( findName1(hp,name) )
				goto found;
		}else{
			if( findAddr1(hp,addr,len,type) )
				goto found;
		}
	}
	return 0;
found:
	freq = Hp->hc_freq += 1;
/*
	if( name )
		Verbose("*** HIT[%d] gethostbyname(%s)\n",freq,name);
	else	Verbose("*** HIT[%d] gethostbyaddr(): %s\n",freq,hp->h_name);
*/
	return Hp;
}
static char **addAliases(hp,aliasesb,name)
	struct hostent *hp;
	char *aliasesb[],*name;
{	int ai;
	char *name1;

	if( name == 0 )
		return hp->h_aliases;
	if( findName1(hp,name) )
		return hp->h_aliases;

	for( ai = 0; name1 = hp->h_aliases[ai]; ai++ )
		aliasesb[ai] = name1;
	aliasesb[ai++] = name;
	aliasesb[ai] = 0;
	return aliasesb;
}

/*
 * TODO: chache should be normalized so that only N to 1 correspondence
 *       (N names to an address) are included, to make update easy ???
 */
static struct hostent *addHostCache(name,aliases,len,addrlist)
	char *name,**aliases,**addrlist;
{	struct hostent *chp;
	int na,ai;
	Hostent *He;
	char hosts[2048];

	chp = NULL;

	/*
	 *  chp = first occurence of entry which include
	 *        name, aliases, or addrlist in it, if exists.
	 */

	if( chp == NULL && NHosts < NHOSTS ){
		He = (Hostent*)calloc(sizeof(Hostent),1);
		He->hc_index = NHosts;
		HostsCache[NHosts++] = He;
		He->hc_predef = HOSTS_PREDEF;
		He->hc_mtime = time(NULL);
		chp = &He->hc_hostent;
	}
	if( chp != NULL ){
		chp->h_name = strdup(name);
		chp->h_aliases = dupv(aliases,0);
		chp->h_length = len;
		chp->h_addr_list = dupv(addrlist,len);
		dump_HOST1(hosts,chp);
		Verbose("HOSTS[%d]=%s %s\n",He->hc_index,hosts,He->hc_predef?"(PREDEF)":"");
		return chp;
	}
	return 0;
}
static replaceHostCache(name,addr,len,type, now,Hp,hp)
	char *name,*addr;
	Hostent *Hp;
	struct hostent *hp;
{	char *hname,*addrlistb[2],**addrlist;
	int addrleng;
	struct hostent *chp = &Hp->hc_hostent;
	int replaced;

	if( hp ){
		if( name && hp->h_name )
		if( strcmp(name,hp->h_name) != 0 ){
			sv1log("HOSTS[%d] cache can't be overwritten: %s->%s\n",
				Hp->hc_index,name,hp->h_name);
			Hp->hc_mtime = now;
			return 0;
		}
		hname = hp->h_name;
		addrlist = hp->h_addr_list;
		addrleng = hp->h_length;
	}else{
		addrlist = addrlistb;
		if( name ){
			hname = name;
			addrlist[0] = 0;
			addrleng = 4;
		}else{
			hname = "";
			addrlist[0] = addr;
			addrlist[1] = 0;
			addrleng = len;
		}
	}
	replaced = 0;
	if( name ){
		if( cmpv(chp->h_addr_list,addrlist,addrleng) ){
			freev(chp->h_addr_list);
			chp->h_addr_list = dupv(addrlist,addrleng);
			chp->h_length = addrleng;
			replaced = 1;
		}
	}else{
		if( strcmp(chp->h_name,hname) != 0 ){
			free(chp->h_name);
			chp->h_name = strdup(hname);
			replaced = 1;
		}
	}
	if( replaced ){
		sv1log("HOSTS[%d] cache by-%s of '%s' replaced (age=%d)\n",
			Hp->hc_index,
			name?"name":"addr",name?name:hname,now-Hp->hc_mtime);
	}
	Hp->hc_mtime = now;
	return 1;
}

static struct hostent *addHostCacheOk(hp,name,func,Start)
	struct hostent *hp;
	char *name;
	char *func;
	double Start;
{	char **aliases,*aliasesb[1024];

	daemonlog("D","*** %s: %s / %4.2f secs. has_alias:%d\n",
		func,hp->h_name,Time()-Start,
		(hp->h_aliases[0] ? 1:0));

	aliases = addAliases(hp,aliasesb,name);
	return addHostCache(hp->h_name,aliases,hp->h_length,hp->h_addr_list);
}
static char *dumpAddr(aaddr,ba)
	char *aaddr;
	unsigned char *ba;
{
	return Sprintf(aaddr,"%d.%d.%d.%d",ba[0],ba[1],ba[2],ba[3]);
}
static char *scanAddr(aaddr,ba)
	char *aaddr;
	unsigned char *ba;
{	int a0,a1,a2,a3;

	if( sscanf(aaddr,"%d.%d.%d.%d",&a0,&a1,&a2,&a3) != 4 )
		return 0;

	ba[0] = a0;
	ba[1] = a1;
	ba[2] = a2;
	ba[3] = a3;
	return (char*)ba;
}

static dumpAddrs(hp,addrlist)
	struct hostent *hp;
	char *addrlist;
{	char *lp,*addr;
	int ai;

	addrlist[0] = 0;
	if( hp->h_addr_list[0] ){
		lp = addrlist;
		if( hp->h_addr_list[1] )
			lp = Sprintf(lp,"{");
		for( ai = 0; addr = hp->h_addr_list[ai]; ai++ ){ 
			if( 0 < ai )
				lp = Sprintf(lp,",");
			lp = dumpAddr(lp,hp->h_addr_list[ai]);
		}
		if( hp->h_addr_list[1] )
			lp = Sprintf(lp,"}");
	}
}
static char *dump_HOST1(hosts,hp)
	char *hosts;
	struct hostent *hp;
{	int ai;
	char *sp;
	char *name;

	sp = hosts;
	if( hp->h_aliases[0] ){
		sp = Sprintf(sp,"{");
		sp = Sprintf(sp,"%s",hp->h_name);
		for( ai = 0; name = hp->h_aliases[ai]; ai++ )
		sp = Sprintf(sp,",%s",name);
		sp = Sprintf(sp,"}");
	}else	sp = Sprintf(sp,"%s",hp->h_name);
	sp = Sprintf(sp,"/");
	dumpAddrs(hp,sp);
	return sp + strlen(sp);
}
extern struct hostent *gethostbyNameAddr();
make_HOSTS(hosts,hostname,cacheonly)
	char *hosts,*hostname;
{	struct hostent *hp;

	if( hp = gethostbyNameAddr(cacheonly,hostname) ){
		dump_HOST1(hosts,hp);
		return 1;
	}
	return 0;
}
dump_HOSTS(hosts)
	char *hosts;
{	char *sp;
	int hi;

	hosts[0] = 0;
	sp = hosts;
	for( hi = 0; hi < NHosts; hi++ ){
		if( 0 < hi )
			sp = Sprintf(sp,",");
		sp = dump_HOST1(sp,&HostsCache[hi]->hc_hostent);
	}
	if( ADDRLIST_RR ){
		if( 0 < NHosts )
			sp = Sprintf(sp,",");
		strcpy(sp,"*/*/RR");
	}
	return NHosts;
}
static list2v(slist,vlist,isaddr)
	char *slist,*vlist[];
{	int ni = 0;
	char *np,*dp,*tp;

	if( slist[0] == '{' || strchr(slist,',') ){
		if( slist[0] == '{' ){
			np = slist + 1;
			if( tp = strchr(np,'}') )
				*tp = 0;
		}else	np = slist;

		for(; np; np = dp){
			if( dp = strchr(np,',') )
				*dp++ = 0;
			Verbose("[%d] %s\n",ni,np);
			if( isaddr )
				np = scanAddr(np,np);
			vlist[ni++] = np;

		}
		vlist[ni] = 0;
	}else{
		Verbose("[-] %s\n",slist);
		if( isaddr )
			slist = scanAddr(slist,slist);
		vlist[0] = slist;
		vlist[1] = 0;
		ni = 1;
	}
	return ni;
}

static gethostbyname1(name,addrb)
	char *name,*addrb;
{	struct hostent *hp;

	if( hp = Dgethostbyname(name) ){
		if( *addrb != 0 ){
			addrb += strlen(addrb);
			*addrb++ = ',';
		}
		dumpAddrs(hp,addrb);
	}
	return 0;
}
static gethostbynames(names,addrb)
	char *names,*addrb;
{	char nameb[0x4000];

	addrb[0] = 0;
	strcpy(nameb,names);
	scan_commaListL(nameb,0,gethostbyname1,addrb);
}
static addhost1(nameaddr)
	char *nameaddr;
{	char namesb[512],*names=namesb;
	char *addrs,addrb[512],*opts;
	char *addrlist[64];
	char *namelist[64],**aliases;

	if( strcmp(nameaddr,"CACHE_ONLY") == 0){
		CACHE_ONLY = 1;
		goto EXIT;
	}

	if( strlen(nameaddr) < sizeof(namesb) )
		strcpy(names,nameaddr);
	else	names = strdup(nameaddr);

	if( addrs = strchr(names,'/') ){
		*addrs++ = 0;
		if( opts = strchr(addrs+1,'/') )
			*opts++ = 0;
	}else	opts = 0;
	if( strcmp(nameaddr,"*/*/RR") == 0 ){
		ADDRLIST_RR = 1;
		goto EXIT;
	}

	if( addrs == 0 ){
		gethostbynames(names,addrb);
		if( addrb[0] == 0 )
			goto EXIT;
		addrs = addrb;
	}

	if( list2v(names,namelist,0) <= 0 )
		goto EXIT;
	if( list2v(addrs,addrlist,1) <= 0 )
		goto EXIT;

	if( namelist[0] )
		aliases = &namelist[1];
	else	aliases = &namelist[0];
	addHostCache(namelist[0],aliases,4,addrlist,opts);

EXIT:
	if( names != namesb )
		free(names);
	return 0;
}
scan_HOSTS(_,hosts)
	char *_,*hosts;
{
	HOSTS_PREDEF = 1;
	scan_commaList(hosts,0,addhost1);
	HOSTS_PREDEF = 0;
}
static storeHOSTS(chp)
	char *chp;
{	char data[0x8000];

	dump_HOST1(data,chp);
	(*STORE_DB)("HOSTS",data);
}

static gotsig;
static struct hostent *gethostbyNameAddrNocache(name,addr,len,type)
	char *name,*addr;
{	struct hostent *hp;

	if( 0 < RES_timeout(DNS_TIMEOUT) ){
		hp = Dgethostbynameaddr(name,addr,len,type);
	}else{
		void (*sPIPE)();
		int timer;

		sPIPE = Vsignal(SIGPIPE,sigPIPE);
		timer = pushTimer("gethostbyNameAddr",sigALRM,DNS_TIMEOUT);
		if( (gotsig = setjmp(jmpEnv)) == 0 )
			hp = Dgethostbynameaddr(name,addr,len,type);
		else	hp = 0;
		signal(SIGPIPE,sPIPE);
		popTimer(timer);
	}
	return hp;

}
RES_CACHEONLY(flag)
{	int oflag;

	oflag = CACHE_ONLY;
	CACHE_ONLY = flag;
	return oflag;
}
extern char resolv_errmsg[512];
#define is_inetaddr(name)	(name != NULL && inet_addrV4(name) != -1)

IsInetaddr(addr)
	char *addr;
{
	return is_inetaddr(addr);
}

static struct hostent *gethostbyNameAddr(cache_only,name,addr,len,type)
	char *name,*addr;
{	struct hostent *hp;
	struct hostent *chp;
	Hostent *Hp;
	double Start;
	char func[256];
	int nocache;
	int now;

	nocache = RES_CACHE_DISABLE;
	if( cache_only || CACHE_ONLY )
		nocache = 0;

	now = time(NULL);
	if( Hp = findHostCache(name,addr,len,type) ){
		int expired = 0;
		int age = 0;

		if( !Hp->hc_predef ){
			age = now - Hp->hc_mtime;
			if( nocache ){
				if( HOSTS_CACHE_LIFE_MIN < age )
					expired = 1;
			}else{
				if( HOSTS_CACHE_LIFE_MAX )
				if( HOSTS_CACHE_LIFE_MAX < age )
					expired = 1;
			}
		}
		if( !expired ){
			hp = &Hp->hc_hostent;
			if( hp->h_name[0] == 0 || hp->h_addr_list[0] == 0 )
				return 0;
			else	return hp;
		}
	}
	if( cache_only || CACHE_ONLY )
		return 0;

	Start = Time();
	resolv_errmsg[0] = 0;

	hp = gethostbyNameAddrNocache(name,addr,len,type);

	if( name != NULL )
		sprintf(func,"gethostbyname(%s)",name);
	else	sprintf(func,"gethostbyaddr(%s)",inet_ntoaV4(*(struct in_addr*)addr));

	if( hp == NULL && name != NULL && !is_inetaddr(name) )
		daemonlog("E","%s unknown[%4.2fs] %s\n",func,
			Time()-Start,resolv_errmsg);

	if( gotsig ){
		if( gotsig == SIGALRM )
			sv1log("*** TIMEOUT %s: %d secs.\n",func,DNS_TIMEOUT);
		else	sv1log("*** SIGPIPE on %s to DNS.\n",func);
/*
endhostent();
hostent_init = 0;
*/
	}else{
		if( Hp && replaceHostCache(name,addr,len,type, now,Hp,hp) )
			return hp;

		if( hp == 0 ){
			char *aliases[2],*addrlist[2];
			int addrleng;

			if( name ){
				addrlist[0] = 0;
				addrleng = 4;
			}else{
				addrlist[0] = addr;
				addrleng = len;
				name = "";
			}
			aliases[0] = 0;
			addrlist[1] = 0;
			chp = addHostCache(name,aliases,addrleng,addrlist);
		}else{
			chp = addHostCacheOk(hp,name,func,Start);
		}
		if( chp && STORE_DB )
			storeHOSTS(chp);
	}
	return hp;
}

static char *satoa(sa,host)
	struct sockaddr_in *sa;
	char *host;
{	char *addr;

	if( addr = inet_ntoaV4(sa->sin_addr) )
		return strcpy(host,addr);
	else	return strcpy(host,"(AF_UNIX)");
}
static char *saton(sa,host)
	struct sockaddr_in *sa;
	char *host;
{	static struct hostent *hp;

	hp = gethostbyNameAddr(0,NULL,
		(char*)&sa->sin_addr.s_addr,
		sizeof(sa->sin_addr.s_addr), AF_INET);

	if( hp != 0 )
		return strcpy(host,hp->h_name);
	else	return satoa(sa,host);
}
char *gethostbyAddr(addr,host)
	char *addr,*host;
{	struct sockaddr_in sa;

	sa.sin_addr.s_addr = inet_addrV4(addr);
	return saton(&sa,host);
}

printSock(sa,sockname,form)
	struct sockaddr_in *sa;
	char *sockname,*form;
{	char *addr;
	char *sp,*fp,fc;
	char host[512];

	sp = sockname;
	for( fp = form; fc = *fp; fp++ ){
	    if( fc != '%' ){
		*sp++ = fc;
		continue;
	    }
	    fp++;
	    switch( *fp ){
		default: sp = Sprintf(sp,"%%%c",*fp); break;
		case '%': *sp++ = fc; break;
		case 'A': sp = Sprintf(sp,"%s",satoa(sa,host)); break;
		case 'H': sp = Sprintf(sp,"%s",saton(sa,host)); break;
		case 'P': sp = Sprintf(sp,"%d",ntohs(sa->sin_port)); break;
	    }
	}
	*sp = 0;
}

static setFIFOaddr(sap)
	struct sockaddr_in *sap;
{
	sap->sin_addr.s_addr = gethostint_nbo("localhost");
	sap->sin_port = htons(1);
}
static gethostpeerName(sock,sockname,form,func)
	char *sockname;
	char *form;
	int (*func)();
{	struct sockaddr_in sa;
	int rcode;
	int len;

	len = sizeof(sa);
	sa.sin_addr.s_addr = 0;
	sa.sin_port = 0;

	if( (*func)(sock,&sa,&len) == 0 ){
		if( len == 0 && file_isfifo(sock) )
			setFIFOaddr(&sa);

		printSock(&sa,sockname,form);
		return 1;
	}else{
		strcpy(sockname,"?");
		return 0;
	}
}
gethostpeerNAME(sock,name,addr,port,func)
	char *name;
	int *addr;
	int *port;
	int (*func)();
{	struct sockaddr_in sa;
	int len;

	len = sizeof(sa);
	sa.sin_addr.s_addr = 0;

	if( (*func)(sock,(SAP)&sa,&len) == 0 ){
		if( len == 0 && file_isfifo(sock) )
			setFIFOaddr(&sa);

		if(name) printSock(&sa,name,"%H");
		if(addr) *addr = ntohl(sa.sin_addr.s_addr);
		if(port) *port = ntohs(sa.sin_port);
		return 1;
	}else{
		if(name) strcpy(name,"?");
		if(addr) *addr = -1;
		if(port) *port = 0;
		return 0;
	}
}

gethostName(sock,sockname,form)
	char *sockname,*form;
{
	return gethostpeerName(sock,sockname,form,getsockname);
}
gethostAddr(sock,sockaddr)
	char *sockaddr;
{
	return gethostName(sock,sockaddr,"%A");
}

getPrimName(host,prim)
	char *host,*prim;
{	struct hostent *hp;

	if( hp = gethostbyname(host) )
		strcpy(prim,hp->h_name);
	else	strcpy(prim,host);
}
getFQDN(name,fqdn)
	char *name,*fqdn;
{	char porder[RESOLVERS_SIZ];
	struct hostent *hp;

	RES_order("D",porder);
	hp = gethostbyname(name);
	RES_order(porder,NULL);
	if( hp != NULL ){
		strcpy(fqdn,hp->h_name);
		return 1;
	}
	if( name != fqdn )
		strcpy(fqdn,name);
	return 0;
}
gethostFQDN(fqdn,size)
	char *fqdn;
{	char host[1024];

	gethostname(host,sizeof(host));
	return getFQDN(host,fqdn);
}

getpeerName(sock,sockname,form)
	char *sockname,*form;
{	int rcode;
	int sresolv;
	char porder[RESOLVERS_SIZ],torder[RESOLVERS_SIZ];

	RES_order("",porder);
	if( strchr(porder,'D') == 0 ){
		sprintf(torder,"%sD",porder); /* use DNS anyway */
		RES_order(torder,NULL);
	}else	RES_order(porder,NULL);

	rcode = gethostpeerName(sock,sockname,form,getpeername);

	RES_order(porder,NULL);
	return rcode;
}
getpeerAddr(sock,sockaddr)
	char *sockaddr;
{
	return getpeerName(sock,sockaddr,"%A");
}

gethostNAME(sock,name,addr,port)
	char *name;
	int *addr;
	int *port;
{
	return gethostpeerNAME(sock,name,addr,port,getsockname);
}
getpeerNAME(sock,name,addr,port)
	char *name;
	int *addr;
	int *port;
{	int rcode;
	int sresolv;
	char porder[RESOLVERS_SIZ],torder[RESOLVERS_SIZ];

	RES_order("",porder);
	if( strchr(porder,'D') == 0 ){
		sprintf(torder,"%sD",porder); /* use DNS anyway */
		RES_order(torder,NULL);
	}else	RES_order(porder,NULL);

	rcode = gethostpeerNAME(sock,name,addr,port,getpeername);

	RES_order(porder,NULL);
	return rcode;
}
inetNtoa(addr,saddr)
	char *saddr;
{	struct sockaddr_in sa;

	sa.sin_addr.s_addr = addr;
	strcpy(saddr,inet_ntoaV4(sa.sin_addr));
}
getpeersNAME(sock,name,saddr,port)
	char *name,*saddr;
	int *port;
{	int addr;

	if( getpeerNAME(sock,name,&addr,port) ){
		inetNtoa(addr,saddr);
		return 1;
	}else	return 0;
}
inetNtoah(addr,saddr)
	char *saddr;
{
	inetNtoa(htonl(addr),saddr);
}

static Accept(sock,isServer,lockfd,sockname)
	char *sockname;
{	int clsock;
	struct sockaddr addr;
	int addrlen;

	addrlen = sizeof(addr);
	if( isServer )
		clsock =  accept(sock,&addr,&addrlen);
	else	clsock = Caccept(sock,&addr,&addrlen);

	if( 0 <= clsock && addrlen == 0 ){
		sv1log("#### accept addrlen==0 [%d]\n",clsock);
		close(clsock);
		clsock = -1;
	}else
	if( sockname ){
		satoa(&addr,sockname);
		sprintf(sockname+strlen(sockname),":%d",
			ntohs(((SAIP)&addr)->sin_port));
	}
	return clsock;
}
/*
 *	a valid (non-negative) lockfd means that parallel accept() should
 *	be mutually excluded, and accept() should be protected from
 *	(SIGALRM) interrupt.
 */
ACCEPT1(sock,isServer,lockfd,timeout,sockname)
	char *sockname;
{	int clsock;
	int NB;

	if( lockfd < 0 && timeout == 0 )
		return Accept(sock,isServer,lockfd,sockname);

	clsock = -1;

	if( 0 <= lockfd && lock_exclusiveTO(lockfd,timeout*1000,NULL) != 0 ){
		sv1log("## accept failed locking [%d]\n",errno);
		goto EXIT;
	}

	if( PollIn(sock,timeout*1000) <= 0 )
		sv1log("## accept failed polling [%d]\n",errno);
	else{
		clsock = Accept(sock,isServer,lockfd,sockname);
	}

	if( 0 <= lockfd )
		lock_unlock(lockfd);

EXIT:
	return clsock;
}
ACCEPT(sock,isServer,lockfd,timeout)
{
	return ACCEPT1(sock,isServer,lockfd,timeout,NULL);
}

static char *SRCPORT;
static sockopen1(asock,in,what,portname,hostname,direct)
	struct sockaddr_in *in;
	char *what,*portname,*hostname;
{	int sock;
	int rcode;
	char aaddr[256];
	int iport;
	char remote[256],local[256];
	double Time(),Start;

	if( 0 <= asock )
		sock = asock;
	else	sock = socket(AF_INET,SOCK_STREAM,0);

	strcpy(aaddr,inet_ntoaV4(in->sin_addr));
	iport = ntohs(in->sin_port);

	Verbose("%s connect %s://%s:%d\n",what,portname,hostname,iport);
	Start = Time();

	if( SRCPORT == NULL ){
		SRCPORT = getenv("SRCIF");
		if( SRCPORT == NULL )
			SRCPORT = "";
	}
	if( SRCPORT[0] ){
int on = 1;
char sockname[256];
char ifhost[256];
int ifport;

ifhost[0] = 0;
ifport = 0;
sscanf(SRCPORT,"%[^:]:%d",ifhost,&ifport);
setsockREUSE(sock,1);
bind_insock(sock,ifhost,ifport);
gethostName(sock,sockname,"%A:%P");
Verbose("#### source port = %s\n",sockname);
fprintf(stderr,"####[%d]#### source port = %s (%s:%d)\n",
getpid(),sockname,ifhost,ifport);
/*
rcode = Setsockopt(sock, SOL_SOCKET, SO_DONTROUTE, &on, sizeof(on));
Verbose("#### [%s] dontroute = %d\n",sockname,rcode);
*/
	}

	rcode = Bconnect(sock,(SAP)in,sizeof(struct sockaddr_in),direct);
	inets_errmsg[0] = 0;
	if( rcode != 0 ){
		if( errno == ECONNREFUSED) CONNERR_REFUSED = 1;
		if( errno == ENETUNREACH ) CONNERR_UNREACH = 1;
		if( errno == EHOSTUNREACH) CONNERR_UNREACH = 1;
		if( errno == ETIMEDOUT   ) CONNERR_TIMEOUT = 1;
		
		if( errno == EISCONN || errno == EINPROGRESS )
			return sock;
		sprintf(inets_errmsg,"[%d] %s connect failed %s:%d [%4.2fs] %d",
			sock,what,aaddr,iport, Time()-Start,errno);
		sv1log("%s\n",inets_errmsg);
		if( sock != asock )
			close(sock);
		return -1;
	}
	getpeerName(sock,remote,"%A:%P");
	gethostName(sock,local,"%A:%P");
	daemonlog("I","%s connected [%d] {%s <- %s} [%4.3fs]\n",
		what,sock,remote,local, Time()-Start);

	return sock;
}
static sockopens(sock,hp,in,what,portname,hostname,direct)
	struct hostent *hp;
	struct sockaddr_in *in;
	char *what,*portname,*hostname;
{	int hi;
	char *haddr;

	shuffle_addrlist(hp);
	for( hi = 0; haddr = hp->h_addr_list[hi]; hi++ ){
		bcopy(haddr,(char*)&in->sin_addr,hp->h_length);
		sock = sockopen1(sock,in,what,portname,hostname,direct);
		if( 0 <= sock )
			return sock;
	}
	return -1;
}

static btoaddr(name,addr,leng)
	char *name,*addr;
{	struct in_addr sin_addr;

	if( addr == 0 || leng != 4 ){
		sv1tlog("#### NAME[%s] ADDR:%x LENG:%d\n",
			name?name:"?",addr,leng);
		return -1;
	}
	bcopy(addr,(char*)&sin_addr,leng);
	return sin_addr.s_addr;
}
gethostintMin(host)
	char *host;
{	struct hostent *hp;
	char *name;
	int leng,ai;
	unsigned int addr,addr1;
	struct in_addr in;

	in.s_addr = inet_addrV4(host);
	if( in.s_addr == -1 )
		hp = gethostbyNameAddr(0,host);
	else	hp = gethostbyNameAddr(0,NULL,
			(char*)&in.s_addr,sizeof(in.s_addr), AF_INET);

	if( hp != NULL ){
		name = hp->h_name;
		leng = hp->h_length;
		addr = btoaddr(name,hp->h_addr_list[0],leng);

		for( ai = 1; hp->h_addr_list[ai]; ai++ ){
			addr1 = btoaddr(name,hp->h_addr_list[ai],leng);
			if( addr1 < addr )
				addr = addr1;
		}
		return addr;
	}
	return -1;
}
static __gethostint_nbo(cacheonly,host,primname)
	char *host,*primname;
{	struct hostent *hp;
	int iaddr;

	iaddr = inet_addrV4(host);
	if( iaddr != -1 && primname == NULL )
		return iaddr;

	if( hp = gethostbyNameAddr(cacheonly,host) ){
		if( primname != NULL )
			strcpy(primname,hp->h_name);
		iaddr = btoaddr(hp->h_name,hp->h_addr,hp->h_length);
	}
	return iaddr;
}
gethostint_nbo(host)
	char *host;
{
	return __gethostint_nbo(0,host,NULL);
}
gethost_addr(cacheonly,host,primname)
	char *host,*primname;
{
	return ntohl(__gethostint_nbo(cacheonly,host,primname));
}
IsResolvable(host)
	char *host;
{
	return gethostint_nbo(host) != -1;
}
getinetAddr(addr)
	char *addr;
{
	return ntohl(inet_addrV4(addr));
}

static ipv4cmp(a1,a2)
	char *a1,*a2;
{	int ai;

	for( ai = 0; ai < 4; ai++ )
		if( a1[ai] != a2[ai] )
			return 1;
	return 0;
}

/*
 * return non-zero if hp2 includes some element which is not
 * included in hp1.
 */
hostentcmp(hp1,hp2)
	struct hostent *hp1,*hp2;
{	char *a1,*a2;
	int n1,n2;
	int diff;

	for( n1 = 0; hp1->h_addr_list[n1]; n1++ );
	for( n2 = 0; hp2->h_addr_list[n2]; n2++ );
	if( n1 < n2 )
		return 1;

	for( n2 = 0; a2 = hp2->h_addr_list[n2]; n2++ ){
		for( n1 = 0; a1 = hp1->h_addr_list[n1]; n1++ )
			if( ipv4cmp(a2,a1) == 0 )
				break;
		if( a1 == NULL )
			return 1;
	}
	return 0;
}

__connectServer(sock,what,portname,hostname,iport,direct)
	char *what,*portname,*hostname;
{	int addr;
	struct hostent *hp;
	struct sockaddr_in *in;
	struct sockaddr_in in_sockaddr;
	char hostnameb[256];

	if( hostname != NULL && hostname[0] == '/' )
		return client_open_un(what,hostname,0);

/*
	if( sock < 0 && ViaVSAPassociator(-1) && strncmp(portname,"VSAP",4) != 0 ){
		char sockname[256],peername[256];
		sockname[0] = 0;
		sprintf(peername,"%s:%d",hostname,iport);
		sock = VSAPconnect(sockname,peername);
		if( 0 <= sock ){
			sv1log("#### connected via TELEPORT\n");
			return sock;
		}
		sv1log("#### connection via TELEPORT failed.\n");
	}
*/

	in_sockaddr = ina0;
	in_sockaddr.sin_family = AF_INET;
	in_sockaddr.sin_port = (short)htons(iport);
	in = &in_sockaddr;
	inets_errmsg[0] = 0;

	if( hostname == NULL || hostname[0] == 0 ){
		hostname = hostnameb;
		strcpy(hostnameb,"localhost");
		if( gethostint_nbo(hostnameb) == -1 )
			gethostname(hostnameb,sizeof(hostnameb));
	}

	addr = inet_addrV4(hostname);
	if( addr != -1 ){
		in_sockaddr.sin_addr.s_addr = addr;
		return sockopen1(sock,in,what,portname,hostname,direct);
	}
	if( hp = gethostbyNameAddr(0,hostname) ){
		int nsock;
		struct hostent *nhp;

		nsock = sockopens(sock,hp,in,what,portname,hostname,direct);
		if( 0 <= nsock )
			return nsock;
/*
if( !nonblocking ){
	nhp = gethostbyNameAddrNocache(hostname);
	if( 0 < hostentcmp(hp,nhp) ){
		nsock = sockopens(sock,nhp,in,what,portname,hostname,direct);
		sv1log("## connect(%s) retried without hosts cache = %d\n",
			hostname,nsock);
		return nsock;
	}
}
*/
		return -1;
	}else
	if( (addr = gethostint_nbo(hostname)) != -1 ){
		in_sockaddr.sin_addr.s_addr = addr;
		return sockopen1(sock,in,what,portname,hostname,direct);
	}else{
		CONNERR_CANTRESOLV = 1;
		sprintf(inets_errmsg,"%s unknown host '%s'",what,hostname);
		sv1log( "%s\n",inets_errmsg);
		return -1;
	}
}
connectServer(what,portname,hostname,iport)
	char *what,*portname,*hostname;
{	int sock;

	sock = __connectServer(-1,what,portname,hostname,iport,1);
	if( 0 <= sock )
		return sock;
	sock = __connectServer(-1,what,portname,hostname,iport,0);
	return sock;
}
client_open(what,portname,hostname,iport)
	char *what,*portname,*hostname;
{
	return __connectServer(-1,what,portname,hostname,iport,0);
}

UDP_client_open1(what,portname,hostname,iport,lhost,lport)
	char *what,*portname,*hostname,*lhost;
{	int asock,rsock;

	asock = socket(AF_INET,SOCK_DGRAM,0);
	sv1log("UDP_client_open[%d] %s://%s:%d ...\n",asock,portname,hostname,iport);
	if( 0 <= lport )
		bind_insock(asock,lhost,lport);
	rsock = __connectServer(asock,what,portname,hostname,iport,0);
	if( rsock < 0 )
		close(asock);
	return rsock;
}
UDP_client_open(what,portname,hostname,iport)
	char *what,*portname,*hostname;
{
	return UDP_client_open1(what,portname,hostname,iport,NULL,-1);
}

hostIFfor(rhost,hostIF)
	char *rhost,*hostIF;
{
	if( hostIFfor1(hostIF,1,"time",rhost,37) )
		return 1;
	if( hostIFfor1(hostIF,0,"time",rhost,37) )
		return 1;
	if( hostIFfor1(hostIF,0,"echo",rhost,7) )
		return 1;
	if( hostIFfor1(hostIF,0,"domain",rhost,53) )
		return 1;
	if( hostIFfor1(hostIF,0,"http",rhost,80) )
		return 1;
	return 0;
}
hostIFfor1(hostIF,udp,proto,rhost,rport)
	char *hostIF,*proto,*rhost;
{	int sock;

	hostIF[0] = 0;
	if( udp )
		sock = UDP_client_open("checkIF-UDP",proto,rhost,rport);
	else	sock = client_open("checkIF-TCP",proto,rhost,rport);

	if( 0 <= sock ){
		gethostNAME(sock,hostIF,NULL,NULL);
		close(sock);
		if( hostIF[0] && strcmp(hostIF,"0.0.0.0") != 0 )
			return 1;
	}
	return 0;
}

hostIFfor0(hostIF,udp,proto,rhost,rport,dontroute)
	char *hostIF,*proto,*rhost;
{	int sock,on;
	struct sockaddr_in sin;
	int got;

	got = 0;
	sock = socket(AF_INET,udp?SOCK_DGRAM:SOCK_STREAM,0);
	sin.sin_family = AF_INET;
	sin.sin_addr.s_addr = gethostint_nbo(rhost);
	sin.sin_port = htons(rport);
	if( dontroute ){
		on = 1;
		Setsockopt(sock, SOL_SOCKET, SO_DONTROUTE, &on, sizeof(on));
	}
	if( connect(sock,(SAP)&sin,sizeof(sin)) == 0 )
		got = gethostAddr(sock,hostIF);
	close(sock);
	return got;
}

IsConnected(sock,reason)
	char **reason;
{	int host,port;
	int nready,rcc;
	char buf[1];

	port = -1;
	host = peerHostport(sock,&port);
	if( host == -1 || port == -1 ){
		if(reason) *reason = "cant_getpeername";
		return 0;
	}

if(reason) *reason = "connected";
return 1;

/*
	nready = rPollIn(sock,1);
	if( nready < 0 ){
		if(reason) *reason = "peer_reset";
		return 0;
	}

	if( nready == 0 ){
		if(reason) *reason = "poll_none_ready";
		return 1;
	}

	rcc = recv(sock,buf,1,MSG_PEEK);
	if( rcc == 1 ){
		if(reason) *reason = "recv_ok";
		return 1;
	}

	if(reason) *reason = "cant_recv";
	return 0;
*/
}
Peek1(sock){
	char buf[1];
	return recv(sock,buf,1,MSG_PEEK);
}

#if defined(sun) && !defined(NC_TPI_CLTS)
#define SunOS4bin 1 /* this binary is compiled on SunOS4.X */
#else
#define SunOS4bin 0
#endif

static Getsockopt(s,level,optname,optval,optlen)
	char *optval;
	int *optlen;
{
	int rcode;

	if( IsBOW1_5() ){
		Verbose("BOW1.5: ignore getsockopt(%d,%x)...\n",s,optname);
		return -1; /* avoid a bug of BOW1.5 ... */
	}
	rcode = getsockopt(s,level,optname,optval,optlen);
	if( rcode == 0 && optname == SO_TYPE && SunOS4bin && IsSolaris() ){
		switch( *(int*)optval ){
			case SOCK_DGRAM:  *(int*)optval = SOCK_STREAM; break;
			case SOCK_STREAM: *(int*)optval = SOCK_DGRAM;  break;
		}
	}
	return rcode;
}
static Setsockopt(s,level,optname,optval,optlen)
	char *optval;
{
	return setsockopt(s,level,optname,optval,optlen);
}

set_nodelay(sock,onoff)
{	int ooo,oon,len;

	len = sizeof(ooo);
	Getsockopt(sock,IPPROTO_TCP,TCP_NODELAY,&ooo,&len);
	Setsockopt(sock,IPPROTO_TCP,TCP_NODELAY,&onoff,sizeof(onoff));
	len = sizeof(oon);
	Getsockopt(sock,IPPROTO_TCP,TCP_NODELAY,&oon,&len);
	Verbose("TCP_NODELAY[%d] %d -> %d\n",sock,ooo,oon);
}
setsockSHARE(sock,onoff)
{
#ifdef SO_REUSEPORT
	return Setsockopt(sock,SOL_SOCKET,SO_REUSEPORT,&onoff,sizeof(onoff));
#else
	return -1;
#endif
}
setsockREUSE(sock,onoff)
{
	Setsockopt(sock,SOL_SOCKET,SO_REUSEADDR,&onoff,sizeof(onoff));
}
set_keepalive(sock,on)
{	int On = 1, No = 0;	
	int Ov = -1, len = sizeof(Ov);

	if( on)
		Setsockopt(sock, SOL_SOCKET, SO_KEEPALIVE, &On, sizeof(On));
	else	Setsockopt(sock, SOL_SOCKET, SO_KEEPALIVE, &No, sizeof(No));
	Getsockopt(sock, SOL_SOCKET, SO_KEEPALIVE, &Ov, &len);
	Verbose("KeepAlive[%d] = %d\n",sock,Ov);
	return Ov;
}

int MAX_BUFF_SOCKSEND = 0x4000;
int MAX_BUFF_SOCKRECV = 0x4000;
std_setsockopt(sock)
{	int bsize;
	int On = 1, No = 0;

	Setsockopt(sock, SOL_SOCKET, SO_REUSEADDR, &No, sizeof(No));
	bsize = MAX_BUFF_SOCKRECV;
	Setsockopt(sock, SOL_SOCKET, SO_RCVBUF, &bsize,sizeof(bsize));
	bsize = MAX_BUFF_SOCKSEND;
	Setsockopt(sock, SOL_SOCKET, SO_SNDBUF, &bsize,sizeof(bsize));
}
getsocktype(sock)
{	int type,len;

	len = sizeof(type);
	if( Getsockopt(sock,SOL_SOCKET,SO_TYPE,&type,&len) == 0 )
		return type;
	return -1;
}
isUDPsock(sock)
{	int type;

	type = getsocktype(sock);
	return type == SOCK_DGRAM;
}
int file_ISSOCK(fd)
{
	return 0 <= getsocktype(fd);
}
getsockbuf(sock,in,out)
	int *in,*out;
{	int len;
	int rcode;

	*in = 0;
	*out = 0;

	len = sizeof(int);
	rcode = Getsockopt(sock, SOL_SOCKET, SO_RCVBUF, in,&len);
	len = sizeof(int);
	rcode = Getsockopt(sock, SOL_SOCKET, SO_SNDBUF, out,&len);
	return rcode;
}
setsockbuf(sock,in,out)
{	int ois,is,oos,os;
	int len;

	if( 0 < in ){
		len = sizeof(ois);
		Getsockopt(sock, SOL_SOCKET, SO_RCVBUF, &ois,&len);
		is = in;
		Setsockopt(sock, SOL_SOCKET, SO_RCVBUF, &is,sizeof(is));
	}else	ois = is = 0;

	if( 0 < out ){
		len = sizeof(oos);
		Getsockopt(sock, SOL_SOCKET, SO_SNDBUF, &oos,&len);
		os = out;
		Setsockopt(sock, SOL_SOCKET, SO_SNDBUF, &os,sizeof(os));
	}else	oos = os = 0;
	Verbose("setsockbuf[%d] in:%d->%d out:%d->%d\n",sock,ois,is,oos,os);
}
expsockbuf(sock,in,out)
{	int isize,osize;

	if( getsockbuf(sock,&isize,&osize) == 0 ){
		if( in  <= isize ) in = 0;
		if( out <= osize ) out = 0;
		if( in || out )
			return setsockbuf(sock,in,out);
		else	return 0;
	}
	return -1;
}
set_oobinline(sock,on)
{	int oobil;

#ifdef SO_OOBINLINE
	oobil = on;
	Setsockopt(sock, SOL_SOCKET, SO_OOBINLINE, &oobil, sizeof(oobil));
	sv1log("OOBINLINE: %d\n",on);
#endif
}

set_linger(sock,secs)
{	struct linger sl,gl;
	int len,rcode;

	if( secs ){
		sl.l_onoff = 1;	/* on */
		sl.l_linger = secs;	/* seconds */
	}else{
#ifdef hpux
		sl.l_onoff = 1;
		sl.l_linger = 0;
#else
		sl.l_onoff = 0;
		sl.l_linger = 0;
#endif
	}
	rcode = Setsockopt(sock, SOL_SOCKET, SO_LINGER, &sl, sizeof(sl));
	if( rcode != 0 ){
		Verbose("setsockopt(%d,LINGER) faield, errno=%d\n",sock,errno);
		return;
	}

	len = sizeof(gl);
	rcode = Getsockopt(sock, SOL_SOCKET, SO_LINGER, &gl, &len);
	if( rcode != 0 )
		Verbose("getsockopt(%d,LINGER) failed, errno=%d\n",sock,errno);
	else	Verbose("LINGER: [%d] %d %d{%d,%d}\n",sock,secs,
			len,gl.l_onoff,gl.l_linger);
}

static mkclnt_sock(src,sin,direct)
	struct sockaddr_in *src,*sin;
{	int sock;
	char sockname[256],peername[256];

	sin->sin_family = AF_INET;
	sock = socket(AF_INET,SOCK_STREAM,0);
	if( sock == -1 ){
		sv1log("mkclnt_sock: cannot create socket: %d.\n",errno);
		return -1;
	}

	if( src && (src->sin_addr.s_addr || src->sin_port) ){
		setsockREUSE(sock,1);
		setsockSHARE(sock,1);
		if( bind(sock,(SAP)src,sizeof(*src)) != 0 ){
			src->sin_port = 0;
			bind(sock,(SAP)src,sizeof(*src));
		}
	}

	printSock(sin,peername,"%H/%A:%P");
	if( Bconnect(sock,(SAP)sin,sizeof(struct sockaddr_in),direct) != 0 ){
		gethostName(sock,sockname,"%A:%P");
		close(sock);
		sv1log("mkclnt_sock: connection refused %s->%s\n",sockname,peername);
		return -1;
	}
	gethostName(sock,sockname,"%A:%P");
	sv1log("mkclnt_sock: connected %s->%s [%d]\n",sockname,peername,sock);
	return sock;
}

/*
 *	REUSE ON to restart server immediately.
 */
static int REUSE = 1;

static bind_inet(sock,hostaddr,port,nlisten,useSocks,dsthost)
	struct in_addr *hostaddr;
{	struct sockaddr_in ina;
	int rcode;

	ina = ina0;
	ina.sin_family = AF_INET;
	if( hostaddr != NULL )
		bcopy(hostaddr,&ina.sin_addr,sizeof(struct in_addr));
/*
		ina.sin_addr = *hostaddr;
*/
	else	ina.sin_addr.s_addr = INADDR_ANY;
	ina.sin_port = htons(port);

/*
if( useSocks ){
Verbose("==== do NOT REUSE the port\n");
REUSE = 0;
}
*/

	Setsockopt(sock,SOL_SOCKET,SO_REUSEADDR,&REUSE,sizeof(REUSE));

	if( useSocks )
		rcode = Cbind(sock,(SAP)&ina,sizeof(ina),dsthost);
	else	rcode =  bind(sock,(SAP)&ina,sizeof(ina));
	if( rcode != 0 ){
		svlog("bind_inet failed: ERRNO=%d\n",errno);
		return -1;
	}
	if( nlisten < 0 ) /* UDP bind */
		return rcode;

	return set_listen(sock,useSocks,nlisten);
}
set_listen(sock,useSocks,nlisten)
{	int rcode;

	if( nlisten <= 0 )
		return -1;

	if( useSocks )
		rcode = Clisten(sock,nlisten);
	else	rcode =  listen(sock,nlisten);

	if( rcode == 0 )
		Verbose("listen(%d,%d) OK.\n",sock,nlisten);
	else	svlog("listen(%d,%d) failed: ERRNO=%d\n",sock,nlisten,errno);
	return rcode;
}

bind_insock(sock,host,port)
	char *host;
{	struct sockaddr_in ina;
	int rcode;
	int addr;

	if( host != NULL && host[0] != 0 )
		addr = gethostint_nbo(host);
	else	addr = -1;
	if( addr == -1 )
		addr = INADDR_ANY;

	ina = ina0;
	ina.sin_family = AF_INET;
	ina.sin_addr.s_addr = addr;
	ina.sin_port = htons(port);
	rcode = bind(sock,(SAP)&ina,sizeof(ina));
	sv1log("bind_insock(%d,%s,%d) = %d\n",sock,host?host:"",port,rcode);
	return rcode;
}

connectTimeout(sock,host,port,timeout)
	char *host;
{	struct hostent *hp;
	struct sockaddr_in ina;

	if( hp = gethostbyNameAddr(0,host) ){
		ina.sin_family = AF_INET;
		bcopy(hp->h_addr,&ina.sin_addr,hp->h_length);
		ina.sin_port = htons(port);
		return connectTO(sock,&ina,sizeof(ina),timeout);
	}
	close(sock);
	return -1;
}

server_open(portname,hostname,portnum,nlisten)
	char *hostname,*portname;
{	int (*logf)(),sv1log(),sv1vlog();
	int rcode;
	int sock;
	struct hostent *hp;
	struct in_addr *ha;

	if( hostname != NULL && hostname[0] == '/' )
		return server_open_un(portname,hostname,nlisten);

	/* dolog = streq(portname,"RESPDIST"); */
	if( hostname && strcmp(hostname,"localhost") == 0 && portnum == 0 )
		logf = sv1vlog;
	else	logf = sv1log;

	(*logf)("server_open(%s,%s:%d,listen=%d)\n",
		portname,hostname?hostname:"*",portnum,nlisten);

	switch( nlisten ){
		case -1: sock = socket(AF_INET,SOCK_DGRAM,0); break;
		default: sock = socket(AF_INET,SOCK_STREAM,0);
	}
	ha = NULL;
	if( hostname != NULL && hostname[0] != 0 ){
		if( hp = gethostbyNameAddr(0,hostname) ){
			ha = (struct in_addr *)hp->h_addr;
			(*logf)("server_open: %s:%d\n",hostname,portnum);
		}
	}
	if( bind_inet(sock,ha,portnum,nlisten,0,0) != 0 ){
		close(sock);
		/*
		sock = bindViaSocks(hostname,portnum);
		if( 0 <= sock )
			return sock;
		*/
		sv1log("server_open() failed\n");
		return -1;
	}
	(*logf)("server_open(%s,%s:%d) BOUND\n",portname,hostname?hostname:"*",portnum);
	return sock;
}

static
char *_gethostaddr(host,cache_only)
	char *host;
{	struct hostent *hp;
	struct in_addr in;

	if( inet_addrV4(host) != -1 )
		return host;

	if( hp = gethostbyNameAddr(cache_only,host) ){
		bcopy(hp->h_addr,(char*)&in,hp->h_length);
		return inet_ntoaV4(in);
	}
	return 0;
}
char *gethostaddr(host) char *host; {
	return _gethostaddr(host,0);
}
char *gethostaddr_fromcache(host) char *host; {
	return _gethostaddr(host,1);
}

adduniqlist(names,name1)
	char *names[],*name1;
{	int nx;
	char *cname;

	for( nx = 0; cname = names[nx] ; nx++ )
		if( strcasecmp(cname,name1) == 0 )
			return;

	names[nx++] = strdup(name1);
	names[nx] = NULL;
}
static adduniqlist_host(names,hp)
	char *names[];
	struct hostent *hp;
{	int nx;

	if( hp->h_name )
		adduniqlist(names,hp->h_name);

	if( hp->h_aliases )
	for( nx = 0; hp->h_aliases[nx]; nx++ )
		adduniqlist(names,hp->h_aliases[nx]);
}

sethostcache(host,mark_predef)
	char *host;
{	int iaddr;
	struct in_addr in;
	struct hostent *chp,*hp;
	char sorder[RESOLVERS_SIZ],porder[RESOLVERS_SIZ];
	char order1[RESOLVERS_SIZ];
	int ox;
	char *aliases[128];
	int found,nx;
	int predef_sav;

	found = -1;
	predef_sav = HOSTS_PREDEF;
	HOSTS_PREDEF = mark_predef;

	in.s_addr = gethostint_nbo(host);
	if( in.s_addr == -1 )
		goto EXIT;

	chp = gethostbyNameAddr(1,NULL,
		(char*)&in.s_addr,sizeof(in.s_addr), AF_INET);
	if( chp == NULL )
		goto EXIT;

	aliases[0] = aliases[1] = 0;
	adduniqlist_host(aliases,chp);

	found = 0;
	RES_order("",sorder);

	/*
	 * Try gather host name aliases in all of available resolvers.
	 * Maybe this is necessary to make revese-MOUNT work well...
	 */
	for( ox = 0; ox = RES_next_res(sorder,ox,order1,NULL); ){
		if( order1[0] == 'C' /* cache */ ){
			/* Maybe cache is the first alternative in resolvers
			 * so the result found above is that of the cache ...
			 */
			continue;
		}
		if( order1[0] == 'S' /* sys */ ){
			/* Some system (Windows, Solaris, ...) would block
			 * long time in system's standard gethostbyXXXX() ...
			 */
			continue;
		}

		RES_order(order1,porder);

		hp = gethostbyNameAddrNocache(NULL,
			(char*)&in.s_addr,sizeof(in.s_addr), AF_INET);

		if( hp != NULL ){
			adduniqlist_host(aliases,hp);
			found++;
		}
	}

	chp->h_aliases = dupv(&aliases[1],0);

	for( nx = 0; aliases[nx]; nx++ )
		free(aliases[nx]);

	RES_order(sorder,porder);

/*{ char hosts[4096]; dump_HOSTS(hosts); fprintf(stderr,"%s\n",hosts); }*/

EXIT:
	HOSTS_PREDEF = predef_sav;
	return found;
}
sethostcache_predef(name,addr,len,type)
	char *name,*addr;
{	Hostent *Hp;
	char hosts[2048];

	if( Hp = findHostCache(name,addr,len,type) )
	if( !Hp->hc_predef ){
		dump_HOST1(hosts,&Hp->hc_hostent);
		Verbose("HOSTS[%d]=%s marked PREDEF\n",Hp->hc_index,hosts);
		Hp->hc_predef = 1;
	}
}

sockHostport(sock,portp)
	int *portp;
{	int len;
	struct sockaddr_in sin;

	len = sizeof(sin);
	if( getsockname(sock,(SAP)&sin,&len) == 0 ){
		if( len == 0 && file_isfifo(sock) )
			setFIFOaddr(&sin);
		if( portp != 0 )
			*portp = ntohs(sin.sin_port);
		return ntohl(sin.sin_addr.s_addr);
	}
	if( portp != 0 )
		*portp = 0;
	return -1;
}
peerHostport(sock,portp)
	int *portp;
{	int len;
	struct sockaddr_in sin;

	len = sizeof(sin);
	if( getpeername(sock,(SAP)&sin,&len) == 0 ){
		if( len == 0 && file_isfifo(sock) )
			setFIFOaddr(&sin);
		if( portp != 0 )
			*portp = ntohs(sin.sin_port);
		return ntohl(sin.sin_addr.s_addr);
	}
	return -1;
}
sockFromMyself(sock)
{
	return sockHostport(sock,NULL) == peerHostport(sock,NULL);
}
flush_socket(fd)
{
	send(fd,"",0,MSG_OOB);  /* push packets before the timeout */
}

/*
 * Data connection should be via Socks if the control connection is via Socks.
 * Source IP address of data connection should be same with that of control
 * connection, and source port number should be be L-1 where the port number
 * of control connection is L.
 */
static struct sockaddr_in ldata_addr;
connect_ftp_data(port,cntrlsock)
	char *port;
{	int ia[4],ip[2];
	unsigned char *sa,*sp;
	int lsock;
	char hostnam[128];
	int portnum;
	int r_ina,c_ina,trydirect;
	int len;
	struct sockaddr_in srcport_buff,*srcport;

	sscanf(port,"%d,%d,%d,%d,%d,%d",&ia[0],&ia[1],&ia[2],&ia[3],
		&ip[0],&ip[1]);
	sa = (unsigned char*)&ldata_addr.sin_addr;
	sp = (unsigned char*)&ldata_addr.sin_port;
	sa[0]=ia[0];sa[1]=ia[1];sa[2]=ia[2];sa[3]=ia[3];sp[0]=ip[0];sp[1]=ip[1];

	sprintf(hostnam,"%d.%d.%d.%d",ia[0],ia[1],ia[2],ia[3]);
	portnum = (ip[0] << 8) | ip[1];

	srcport = NULL;
	if( cntrlsock < 0 ){
		trydirect = 1;
	}else{
		r_ina = inet_addrV4(hostnam);
		c_ina = peerHostport(cntrlsock,NULL);
		if( r_ina == c_ina )
			trydirect = 1;
		else	trydirect = 0; /* maybe via Socks */

		len = sizeof(srcport_buff);
		if( getsockname(cntrlsock,(SAP)&srcport_buff,&len) == 0 )
		if( len != 0 ){
			srcport = &srcport_buff;
			srcport->sin_port = htons(ntohs(srcport->sin_port)-1);
		}
	}

	lsock = -1;
	if( lsock == -1 && ViaVSAPassociator(cntrlsock) ){
		char sockname[256],peername[256];
		sockname[0] = 0;

		if( VSAPgetsockname(cntrlsock,sockname) == 0 ){
			char *dp;
			int port;
			if( dp = strchr(sockname,':') ){
				dp++;
				/*if( port = atoi(dp) )
					sprintf(dp,"%d",port-1);
				else*/	sprintf(dp,"0");
			}
			sv1log("## FTP/VSAP CONNECT SOCK=%s\n",sockname);
		}

		sprintf(peername,"%s:%d",hostnam,portnum);
		lsock = VSAPconnect(sockname,peername);
	}

	if( lsock == -1 && trydirect )
		lsock = mkclnt_sock(srcport,&ldata_addr,1);

	if( lsock == -1 ){
		if( myownSOCKS && getViaSocks(hostnam,portnum) )
			lsock = connectViaSocks(hostnam,portnum);
		else	lsock = mkclnt_sock(srcport,&ldata_addr,0);
	}
	return lsock;
}

mkserv_sock_ftp(mport,server,port,cntrlsock,direct)
	char *mport;
	char *server;
{	int dsock;
	unsigned char *sa,*sp;
	char *host;
	int ftpsv_ina,r_ina,l_ina,c_ina;
	struct in_addr rina,lina;
	struct sockaddr_in bina;
	int len;
	int use_SOCKS;

	if( ViaVSAPassociator(cntrlsock) ){
		char sockname[256];
		int ax[4],bport;

		sockname[0] = 0;
		if( VSAPgetsockname(cntrlsock,sockname) == 0 ){
			char *dp;
			if( dp = strchr(sockname,':') )
				strcpy(dp+1,"0");
		}
		sv1log("## FTP/VSAP BIND SOCK=%s\n",sockname);
		dsock = VSAPbind(sockname,1);
		if( 0 <= dsock ){
			sscanf(sockname,"%d.%d.%d.%d:%d",
				&ax[0],&ax[1],&ax[2],&ax[3],&bport);
			sprintf(mport,"%d,%d,%d,%d,%d,%d",
				ax[0],ax[1],ax[2],ax[3],
				(bport>>8)&0xFF,bport&0xFF);
			return dsock;
		}
	}

	use_SOCKS = SOCKS_client && !direct;

	/* ftpsv_ina wll be necessary only when SOCKS is used... */
	if( (r_ina = gethostint_nbo(server)) != -1 ){
		ftpsv_ina = r_ina;
		rina.s_addr = r_ina;
		sv1log("FTP-data-remote: %s\n",inet_ntoaV4(rina));
	}

	l_ina = sockHostport(cntrlsock,0);
	if( l_ina == 0 || l_ina == -1 ){
		sv1log("cannot make FTP data port: no control connection.\n");
		return -1;
	}

	c_ina = peerHostport(cntrlsock,NULL);
	if( r_ina != c_ina ) /* not connected directly */
	if( myownSOCKS && getViaSocks(server,0) )
	{	unsigned char bhost[32];
		int bport;

		dsock = bindViaSocks(server,port,bhost,&bport);
		if( 0 <= dsock ){
			sa = bhost;
			sprintf(mport,"%d,%d,%d,%d,%d,%d",
				sa[0],sa[1],sa[2],sa[3],
				(bport>>8)&0xFF,bport&0xFF);
			return dsock;
		}
	}

	dsock = socket(AF_INET,SOCK_STREAM,0);
	if( bind_inet(dsock,NULL,0,1,use_SOCKS,ftpsv_ina) != 0 ){
		sv1log("ERROR: Cannot bind.\n");
		close(dsock);
		return -1;
	}

	len = sizeof(bina);
	if( use_SOCKS )
		Cgetsockname(dsock, (SAP)&bina, &len);
	else	getsockname(dsock, (SAP)&bina, &len);
	sp = (unsigned char*)&bina.sin_port;

	if( use_SOCKS && bina.sin_addr.s_addr == 0 ){
		sv1log("FTP-data-local: don't use SOCKS for local host?\n");
		use_SOCKS = 0;
	}

	if( use_SOCKS ){
		host = inet_ntoaV4(bina.sin_addr);
		sv1log("FTP-data-local:  %s [SOCKS]\n",host?host:"LocalHost");
		sa = (unsigned char*)&bina.sin_addr.s_addr;
	}else{
		lina.s_addr = htonl(l_ina);
		host = inet_ntoaV4(lina);
		sv1log("FTP-data-local:  %s : %d\n",host?host:"LocalHost",
			ntohs(bina.sin_port));
		sa = (unsigned char*)&lina.s_addr;
	}

	sprintf(mport,"%d,%d,%d,%d,%d,%d",sa[0],sa[1],sa[2],sa[3],sp[0],sp[1]);
	return dsock;
}

/* OBSOLETED */
/*
dsock = _mkserv_sock(host,0,&mdata_addr,ftpsv_ina,2);
sa = (unsigned char*)&mdata_addr.sin_addr;
sp = (unsigned char*)&mdata_addr.sin_port;
*/
/*
static _ineta2socka(addr,ina)
	char *addr;
	struct sockaddr_in *ina;
{	int iaddr;

	if( (iaddr = inet_addrV4(addr)) != -1){
		bcopy(&iaddr,&ina->sin_addr,sizeof(iaddr));
		return iaddr;
	}
	return -1;
}
static _mkserv_sock(host,iport,rina,dsthost,nlisten)
	char *host;
	struct sockaddr_in *rina;
{	struct sockaddr_in ina;
	int sock;
	int len;
	static struct hostent *hp;
	char hostb[256];

	sock = socket(AF_INET,SOCK_STREAM,0);
	if( bind_inet(sock,NULL,iport,nlisten,1,dsthost) != 0 )
		return -1;

	len = sizeof(ina);
	Cgetsockname(sock, (SAP)&ina, &len);

	if( !SOCKS_client )
	{

Is the following code necessary ???
941224 yes, because getsockname doesn't set sin_addr member (why ?)

		if( host == 0 ){
			GetHostname(hostb,sizeof(hostb));
			host = hostb;
		}
		if( _ineta2socka(host,&ina) == -1 )
		if( hp = gethostbyNameAddr(0,host) )
			bcopy(hp->h_addr,(char*)&ina.sin_addr,hp->h_length);
		else	sv1log("Error: failed gethostbyname(%s)\n",host);
	}

	sv1log("_mkserv_sock: [%d] %x:%d\n",
		sock, ina.sin_addr.s_addr,ina.sin_port);

	*rina = ina;
	return sock;
}
static struct sockaddr_in mdata_addr;
*/

UDPaccept(svsock,lockfd,timeout)
{	struct sockaddr_in from,svme,xme,peer;
	int clsock;
	int xsock;
	int len,rcc,wcc;
	char buf[0x8000];
	char sfrom[128],speer[128],sme[128];
	int rcode,try,rbind,nsvsock;

	if( 0 <= lockfd && lock_exclusiveTO(lockfd,timeout*1000,NULL) != 0 ){
		sv1log("## UDP accept failed locking [%d]\n",errno);
		return -1;
	}
	if( PollIn(svsock,1) <= 0 ){
		sv1log("## UDP accept failed polling [%d]\n",errno);
		clsock = -1;
		goto EXIT;
	}

	len = sizeof(from);
	rcc = recvfrom(svsock,buf,sizeof(buf),0,(SAP)&from,&len);
	if( rcc <= 0 ){
		sv1log("## UDP accept failed receiving [%d]\n",errno);
		clsock = -1;
		goto EXIT;
	}

	printSock(&from,sfrom,"%A:%P");
	sv1log("UDP-SV[%d] ready=%d from=%s\n",svsock,rcc,sfrom);

	len = sizeof(svme);
	rcode = getsockname(svsock,(SAP)&svme,&len);
	setsockREUSE(svsock,1);
	close(svsock);

	clsock = socket(AF_INET,SOCK_DGRAM,0);
	if( clsock == svsock ){
		clsock = dup(clsock);
		close(svsock);
	}
	setsockREUSE(clsock,1);

	xme = svme;

/*
xme.sin_addr.s_addr = 0;
xme.sin_port = 0;
*/

	for( try = 0; try < 10; try++ ){
		rcode = bind(clsock,(SAP)&xme,sizeof(xme));
		if( rcode == 0 )
			break;
		sleep(1);
	}
	if( rcode != 0 ){
		sv1log("UDP accept failed in bind %d\n",errno);
		close(clsock);
		clsock = -1;
		goto EXIT;
	}

	len = sizeof(xme);
	rcode = getsockname(clsock,(SAP)&xme,&len);

	xsock = socket(AF_INET,SOCK_DGRAM,0);
	wcc = sendto(xsock,buf,rcc,0,(SAP)&xme,sizeof(xme));
	close(xsock);

	rcode = connect(clsock,(SAP)&from,sizeof(from));
	len = sizeof(peer);
	rcode = getpeername(clsock,(SAP)&peer,&len);
	printSock(&peer,speer,"%A:%P");

	rcc = recvfrom(clsock,buf,sizeof(buf),MSG_PEEK,(SAP)&from,&len);
	sv1log("UDP-CL[%d] ready=%d peer=%s\n",clsock,rcc,speer);

	nsvsock = socket(AF_INET,SOCK_DGRAM,0);
	setsockREUSE(nsvsock,1);
	for( try = 0; try < 10; try++ ){
		errno = 0;
		rbind = bind(nsvsock,(SAP)&svme,sizeof(svme));
		if( rbind == 0 )
			break;
		sleep(1);
	}
	if( nsvsock != svsock ){
		dup2(nsvsock,svsock);
		close(nsvsock);
	}

	len = sizeof(svme);
	getsockname(svsock,(SAP)&svme,&len);
	printSock(&svme,sme,"%A:%P");
	setsockREUSE(svsock,1);
	sv1log("UDP-SV[%d]: NEW bind=%d %s\n",svsock,rbind,sme);

EXIT:
	if( 0 < lockfd )
		lock_unlock(lockfd);
	return clsock;
}

Recv(s,buf,len)
	char *buf;
{	struct sockaddr from;
	int rcc,fromlen;

	fromlen = sizeof(from);
errno = 0;
	rcc = recvfrom(s,buf,len,0,&from,&fromlen);
{
char sfrom[128];
printSock(&from,sfrom,"%H[%A]:%P");
fprintf(stderr,"recvfrom(%d,%x,%d)=%d // %s [%d]\n",s,buf,len,rcc, sfrom,fromlen);
if( errno ) perror("RECV");
}
}
Send(s,buf,len)
	char *buf;
{	int wcc;

/*
char sto[128],sfrom[128];
getpeerName(s,sto,"%H[%A]:%P");
gethostName(s,sfrom,"%H[%A]:%P");
fprintf(stderr,"send(%d,%x,%d) // %s >> %s\n",s,buf,len,sfrom,sto);
*/

	wcc = send(s,buf,len,0);
	return wcc;
}

RecvFrom(sock,buf,len,froma,fromp)
	char *buf;
	char *froma;
	int *fromp;
{	struct sockaddr_in from;
	int rcc,fromlen;

	fromlen = sizeof(from);
	rcc = SOCKS_recvfrom(sock,buf,len,0,(SAP)&from,&fromlen);
	strcpy(froma,inet_ntoaV4(from.sin_addr));
	*fromp = ntohs(from.sin_port);
	return rcc;
}
SendTo(sock,buf,len,host,port)
	char *buf,*host;
{	struct sockaddr_in me,to;
	int tolen;

	to = ina0;
	to.sin_family = AF_INET;
	to.sin_addr.s_addr = inet_addrV4(host);
	to.sin_port = htons(port);
	tolen = sizeof(to);
	return SOCKS_sendto(sock,buf,len,0,(SAP)&to,tolen);
}
SOCKS_udpassocsock(sock,lhost,lport,rhost,rport)
	char *rhost;
	int *rport;
{	struct sockaddr_in me,rme;

	me = ina0;
	me.sin_family = AF_INET;
	me.sin_addr.s_addr = gethostint_nbo(lhost);
	me.sin_port = htons(lport);

	if( SOCKS_udpassoc(sock,&me,&rme) != 0 )
		return -1;
	
	strcpy(rhost,inet_ntoaV4(rme.sin_addr));
	*rport = ntohs(rme.sin_port);
	return 0;
}

sendTo(sock,sto,msg,len)
	char *sto,*msg;
{	struct sockaddr_in to;
	char addr[64];
	int port;
	int wcc;

	port = 0;
	sscanf(sto,"%[^:]:%d",addr,&port);
	if( port == 0 ){
		sv1log("sendTo(%s:%d) ? \n",addr,port);
		return -1;
	}

	to = ina0;
	to.sin_family = AF_INET;
	to.sin_addr.s_addr = inet_addrV4(addr);
	to.sin_port = htons(port);
	wcc = sendto(sock,msg,len,0,(SAP)&to,sizeof(to));
	return wcc;
}

peekfrom(sock,sfrom)
	char *sfrom;
{	char buf[2048];
	struct sockaddr_in from;
	int fromlen;
	int rcc;

	strcpy(sfrom,"0.0.0.0:0");
	fromlen = sizeof(from);
	rcc = recvfrom(sock,buf,sizeof(buf),MSG_PEEK,(SAP)&from,&fromlen);
	if( rcc <= 0 )
		return rcc;

	printSock(&from,sfrom,"%A:%P");
	return rcc;
}

readfrom(sock,buff,size,sfrom)
	char *buff,*sfrom;
{	struct sockaddr_in from;
	int fromlen;
	int rcc;

	strcpy(sfrom,"0.0.0.0:0");
	fromlen = sizeof(from);
	rcc = recvfrom(sock,buff,size,0,(SAP)&from,&fromlen);
	if( rcc <= 0 )
		return rcc;

	printSock(&from,sfrom,"%A:%P");
	return rcc;
}

GetHostname(name,size)
	char *name;
{	struct hostent *hp;
	char host[256];

	if( myFQDN == NULL ){
		gethostname(host,sizeof(host));
		if( hp = gethostbyNameAddr(0,host) )
			myFQDN = strdup(hp->h_name);
		else{
			strcpy(myFQDN_unknown,host);
			myFQDN = myFQDN_unknown;
		}
	}
	strncpy(name,myFQDN,size);
}
IsMyself(host)
	char *host;
{	char myhost[256];
	int hosti;

	hosti = gethostintMin(host);
	if( hosti == -1 )
		return 0;

	if( hosti == gethostintMin("localhost") )
		return 1;

	gethostname(myhost,sizeof(myhost));
	if( hosti == gethostintMin(myhost) )
		return 1;

	return 0;
}

localsocket(sock)
{	int phosti,pporti;
	int mhosti,mporti;

	phosti = peerHostport(sock,&pporti);
	if( phosti == -1 )
		return 1;

	mhosti = sockHostport(sock,&mporti);
	if( mhosti == -1 )
		return 1;

	if( phosti == mhosti )
		return 1;

	return 0;
}

hostismyself(host,sockfp)
	char *host;
	FILE *sockfp;
{	struct sockaddr_in ina;
	int sock,rcode;

	if( gethostint_nbo("localhost") == gethostint_nbo(host) )
		return 1;

	if( sockfp != NULL )
		if( localsocket(fileno(sockfp)) )
			return 1;

	sock = socket(AF_INET,SOCK_STREAM,0);
	ina = ina0;
	ina.sin_family = AF_INET;
	ina.sin_addr.s_addr = gethostint_nbo(host);
	ina.sin_port = 0;
	rcode = bind(sock,(SAP)&ina,sizeof(ina));
	close(sock);

	if( rcode == 0 )
		return 1;

	return 0;
}

hostcmp(host1,host2)
	char *host1,*host2;
{	int hi1,hi2;

	if( strcasecmp(host1,host2) == 0 )
		return 0;

	if( (hi1 = gethostintMin(host1)) != -1 )
	if( (hi2 = gethostintMin(host2)) != -1 )
		if( hi1 == hi2 )
			return 0;
	return 1;
}

strNetaddr(host,net)
	char *host,*net;
{	unsigned int hosti;

	hosti = ntohl(gethostint_nbo(host));
	switch( (hosti >> 24) & 0xC0 ){
		case 0x00: hosti &= 0xFF000000; break;
		case 0x80: hosti &= 0xFFFF0000; break;
		case 0xC0: hosti &= 0xFFFFFF00; break;
	}
	if( net != NULL )
		sprintf(net,"%d.%d.%d.%d",
			0xFF&(hosti>>24),
			0xFF&(hosti>>16),
			0xFF&(hosti>>8),
			0xFF&(hosti));

	return htonl(hosti);
}

sendOOB(sock,buff,size)
	char *buff;
{
	return send(sock,buff,size,MSG_OOB);
}
recvOOB(sock,buff,size)
	char *buff;
{
	if( 0 < recv(sock,buff,size,MSG_OOB|MSG_PEEK) )
		return recv(sock,buff,size,MSG_OOB);
	else	return -1;
}

SetNonblockingIO(what,sock,on)
	char *what;
{	int onbio,nnbio;

	if( sock < 0 )
		return;

	onbio = getNonblockingIO(sock);
	setNonblockingIO(sock,on);
	nnbio = getNonblockingIO(sock);
	if( onbio != nnbio )
		Verbose("NBIO[%s][%d] %d -> %d\n",what,sock,onbio,nnbio);
}

getpairName(clsock,sockname,peername)
	char *sockname,*peername;
{
	gethostName(clsock,sockname,"%A:%P");
	getpeerName(clsock,peername,"%A:%P");
}

Socket1(what, sock,domain,type,proto, lhost,lport, rhost,rport, nlisten,opts,NB)
	char *what,*domain,*type,*proto,*lhost,*rhost,*opts;
{	int Domain,Type,Proto;

	if( sock < 0 ){
		if( domain && strcasecmp(domain,"unix") == 0
		 || lhost != NULL && lhost[0] == '/'  )
			Domain = AF_UNIX;
		else	Domain = AF_INET;
		if( type && (strcaseeq(type,"udp")||strcaseeq(type,"dgram")||strcaseeq(proto,"udp")) )
			Type = SOCK_DGRAM;
		else	Type = SOCK_STREAM;
		Proto = 0;
		sock = socket(Domain,Type,Proto);
		if( sock < 0 )
			return -1;
	}
	if( lhost != NULL && lhost[0] == '/' ){
	}else
	if( lhost != NULL || 0 < lport ){
		if( strcmp(lhost,"*") == 0 )
			lhost[0] = 0;
		setsockREUSE(sock,1);
		setsockSHARE(sock,1);
		if( bind_insock(sock,lhost,lport) != 0 ){
			close(sock);
			return -1;
		}
	}
	if( 0 < nlisten ){
		if( listen(sock,nlisten) != 0 ){
			close(sock);
			return -1;
		}
	}
	if( rhost != NULL || 0 < rport
	 || rhost != NULL && rhost[0] == '/' ){
		if( NB ) setNonblockingIO(sock,1);
		if( __connectServer(sock,"Socket",what,rhost,rport,1) < 0 ){
			if( !NB ){
				close(sock);
				return -1;
			}
		}
	}
	return sock;
}
Listen(sock,backlog)
{
	return listen(sock,backlog);
}


#define NIFTO	32
static struct {
	int	dest;
	int	mask;
	int	ifto;
} ifto[NIFTO];

hostIFto(dest,mask)
	int dest;
{	char desthost[128],hostif[128];
	int dest1,hostaddr;
	struct sockaddr_in sin;
	int host;
	int ii;

	for( ii = 0; ii < NIFTO && (dest1 = ifto[ii].dest); ii++ )
		if( (dest & mask) == dest1 )
			return ifto[ii].ifto;

	sin.sin_addr.s_addr = htonl(dest);
	strcpy(desthost,inet_ntoaV4(sin.sin_addr));
	if( hostIFfor0(hostif,1,"time",desthost,37,0) == 0 ){
		strcpy(hostif,"?");
		host = -1;
	}else	host = ntohl(inet_addr(hostif));

	if( ii < NIFTO ){
		ifto[ii].dest = dest & mask;
		ifto[ii].ifto = host;
	}
	sv1log("## hostIFto %s < %s (%x)\n",desthost,hostif,mask);
	return host;
}
