/*////////////////////////////////////////////////////////////////////////
Copyright (c) 1994 Electrotechnical Laboratry (ETL), AIST, MITI

Permission to use, copy, modify, and distribute this material for any
purpose and without fee is hereby granted, provided that the above
copyright notice and this permission notice appear in all copies, and
that the name of ETL not be used in advertising or publicity pertaining
to this material without the specific, prior written permission of an
authorized representative of ETL.
ETL MAKES NO REPRESENTATIONS ABOUT THE ACCURACY OR SUITABILITY OF THIS
MATERIAL FOR ANY PURPOSE.  IT IS PROVIDED "AS IS", WITHOUT ANY EXPRESS
OR IMPLIED WARRANTIES.
/////////////////////////////////////////////////////////////////////////
Content-Type:	program/C; charset=US-ASCII
Program:	hostlist.c
Author:		Yutaka Sato <ysato@etl.go.jp>
Description:

    Each element in HostList is one of follows:

	IP-address matching with mask:
		host/255.255.255.0
		host/@C

	IP-address range:
		192.31.200.[65-94]
		192.31.[197-206]
		150.29.[0-255]

	Domain name pattern macthing by name:
		*.ac.jp
		*.jp

    A special host-name "?" matches with hosts which name cannot be
    retrieved with DNS, thus unknown.

    Each element can be prefixed with UserList.
    A special user-name "?" matches when the user cannot be
    identified with IDENTd.

History:
	940623	added simple masking (suggested by <yoko@huie.hokudai.ac.jp>)
	940807	separated from access.c
//////////////////////////////////////////////////////////////////////#*/
#include <string.h>
#include <ctype.h>
#include "hostlist.h"
extern char *strdup();
extern char *strstr();
extern char *frex_create(),*frex_match();

static char opsyms[] = "+!-?";
#define ON_EQ	0	/* ON  if matched */
#define ON_NEQ	1	/* ON  if not matched */
#define OFF_EQ	2	/* OFF if matched */
#define OFF_NEQ	3	/* OFF if not matched */

#define	DO_NOP	0
#define	DO_ON	1
#define DO_OFF	2

static matchProto(protolist,proto)
	char *protolist[],*proto;
{	int isin,pi;
	char *proto1;
	int op,match;

	if( *protolist[0] == '!' )
		isin = 1;
	else	isin = 0;

	for( pi = 0; proto1 = protolist[pi]; pi++ ){
		if( *proto1 == '!' ){
			proto1++;
			op = OFF_EQ;
		}else	op = ON_EQ;

		if( streq(proto1,"*") )
			match = 1;
		else	match = streq(proto1,proto);

		if( match && op == ON_EQ )	isin = 1; else
		if( match && op == OFF_EQ )	isin = 0;
		Verbose("--> %d : %d : %s : %s : %s\n",
			isin, match, op==OFF_EQ?"NEQ":"EQ",
			proto, proto1);
	}
	return isin;
}
static char **makeUserRexp(uc,userlist)
	char *userlist[];
{	char **rexpv;
	char *user;
	int ui;

	rexpv = (char**)malloc(sizeof(char*)*uc);
	for( ui = 0; ui < uc; ui++ ){
		user = userlist[ui];
		if( *user == '!' )
			user++;
		if( !streq(user,"*") && strpbrk(user,"*[") ){
			sv1log("##REGEXP COMPILED: %s\n",user);
			rexpv[ui] = frex_create(user);
		}else	rexpv[ui] = 0;
	}
	return rexpv;
}
static matchUser(userlist,rexplist,username)
	char *userlist[];
	char *rexplist[];
	char *username;
{	int isin,ui;
	char *user;
	char *rexp,*tail;
	int op,match;

	if( *userlist[0] == '!' )
		isin = 1;
	else	isin = 0;

	for( ui = 0; user = userlist[ui]; ui++ ){
		if( *user == '!' ){
			user++;
			op = OFF_EQ;
		}else	op = ON_EQ;

		if( streq(user,"*") )
			match = 1;
		else
		if( rexp = rexplist[ui] ){
			tail = frex_match(rexp,username);
			match = tail && *tail == 0;
			sv1log("##REGEXP EXECUTED ==> %d : %s : %s\n",
				match,username,user);
		}
		else	match = strcmp(user,username) == 0;

		if( match && op == ON_EQ )	isin = 1; else
		if( match && op == OFF_EQ )	isin = 0;
		Verbose("--> %d : %d : %s : %s : %s\n",
			isin, match, op==OFF_EQ?"NEQ":"EQ",
			username,user);
	}
	return isin;
}
static matchPort(portlist,portnum)
	char *portlist[];
{	int isin,pi;
	char *ports;
	int op,match;
	int port0,port1;

	if( *portlist[0] == '!' )
		isin = 1;
	else	isin = 0;

	for( pi = 0; ports = portlist[pi]; pi++ ){
		if( *ports == '!' ){
			ports++;
			op = OFF_EQ;
		}else	op = ON_EQ;

		if( streq(ports,"*") )
			match = 1;
		else{
			switch( sscanf(ports,"%d-%d",&port0,&port1) ){
			case 0: match = 0; break;
			case 1: match = port0 == portnum; break;
			case 2: match = port0 <= portnum && portnum <= port1;
				break;
			}
		}

		if( match && op == ON_EQ )	isin = 1; else
		if( match && op == OFF_EQ )	isin = 0;
		Verbose("--> %d : %d : %s : %d : %s\n",
			isin, match, op==OFF_EQ?"NEQ":"EQ",
			portnum,ports);
	}
	return isin;
}

static match_addrrange(Hp,addr)
	Host *Hp;
{	int ai;
	Uchar a1;

	for(ai = 0; ai < 4; ai++){
		a1 = (addr >> ((3-ai)*8)) & 0xFF;
		if( a1 < Hp->ranges[ai].h_low || Hp->ranges[ai].h_high < a1 )
			return 0;
	}
	return 1;
}
default_netmask(addr)
{	int a1,mask;

	a1 = (addr >> 24) & 0xFF;
	if( a1 < 128 ) mask = 0xFF000000; else
	if( a1 < 192 ) mask = 0xFFFF0000; else
	       mask = 0xFFFFFF00;
	return mask;
}

static isNotResolved(addr,name)
	char *name;
{
	if(addr == 0 || addr == -1)
		return 1;
	if(getinetAddr(name) != -1) /* name is in "dd.dd.dd.dd" IP-address */
		return 1;
	return 0;
}
static char *scanVdomain(hostname,hostnameb,vdomain)
	char *hostname,*hostnameb,*vdomain;
{	char *vdom,*vp,*sp,*dp,ch;

	*vdomain = 0;
	vdom = strstr(hostname,".-");
	if( vdom == 0 )
		return 0;

	strcpy(hostnameb,hostname);
	hostname = hostnameb;
	vp = sp = strstr(hostname,".-");
	dp = vdomain;
	for( vp += 2; ch = *vp; vp++ ){
		if( isalnum(ch) )
			*dp++ = ch;
	}
	if( dp == vdomain )
		return 0;

	*dp = 0;
	while( *sp++ = *vp++ );
	return vdom;
}

#define HostOutgoingIF(hp)	(hp[0]=='.' && hp[1]=='o' && hp[2]==0)
#define HostIncomingIF(hp)	(hp[0]=='.' && hp[1]=='i' && hp[2]==0\
			      || hp[0]=='-' && hp[1]==0)
#define HostOnClientside(hp)	(hp[0]=='-' && hp[1]=='C' && hp[2]==0)
#define HostUnknown(hp)		(hp[0]=='?' && hp[1]==0)
#define HostMatchExactly(hp,hn)	(strcasecmp(hp,hn) == 0)

int ClientsideHost = -1;
int ClientSockHost = -1;

hostIsinList(hostlist,proto,hostname,hosti,portnum,username)
	HostList *hostlist;
	char *proto,*hostname;
	int hosti;
	char *username;
{	int hi,nhosts;
	int mask,ahost,chost;
	Host *Hp;
	char *hostpat;
	int op;
	int isin,match,do_onoff;
	char hostnameb[256],*vdom,vdomb[128],*vdom1;

	if( scanVdomain(hostname,hostnameb,vdomb) ){
		hostname = hostnameb;
		vdom = vdomb;
	}else	vdom = 0;

	op = 0;
	isin = 0;
	nhosts = hostlist->hl_cnt;

	for( hi = 1; hi <= nhosts; hi++ ){
		Hp = hostlist->hl_List[hi];
		vdom1 = Hp->h_vdomain;
		if( vdom == 0 && vdom1 != 0
		 || vdom != 0 && vdom1 == 0
		 || vdom != 0 && vdom1 != 0 && !streq(vdom,vdom1) )
			continue;

		op = Hp->h_op;
		if( isin && (op == ON_EQ || op == ON_NEQ) ||
		   !isin && (op ==OFF_EQ || op==OFF_NEQ) ){
			/* nothing will happen */
			continue;
		}
		match = 0;

		mask = Hp->h_mask;
		hostpat = Hp->h_name;

		if( HostOnClientside(hostpat) ){
			Verbose("- >> ClientsideHost = %x\n",ClientsideHost);
			if( mask == 0 )
				mask = default_netmask(ClientsideHost);
			chost = mask & ClientsideHost;
		}else
		if( HostOutgoingIF(hostpat) ){
			Verbose("- >> HostOutgoingIF = %x\n",hosti);
			if( hosti == 0 || hosti == -1 ){
				chost = ~hosti; /* any different from hosti */
				mask = -1;
			}else{
				if( mask == 0 )
					mask = default_netmask(hosti);
				chost = hostIFto(hosti,mask) & mask;
			}
		}else
		if( HostIncomingIF(hostpat) ){
			Verbose("- >> ClientSocktHost = %x\n",ClientSockHost);
			if( mask == 0 )
				mask = default_netmask(ClientSockHost);
			chost = mask & ClientSockHost;
		}else	chost = mask & Hp->h_addr;
		ahost = hosti & mask;

		if( HostUnknown(hostpat) && isNotResolved(hosti,hostname) ){
			Verbose("UNKNOWN HOST\n");
			match = 1;
		}else
		if( HostMatchExactly(hostpat,hostname) ){
			Verbose("EXACT NAME MATCHING [%d/%d] %s %c= %s\n",
				hi,nhosts,hostname,opsyms[op],hostpat);
			match = 1;
		}else
		if( ahost != 0 && chost != 0 ){
			Verbose("ADDR MATCHING [%d/%d] %x %c= %x ?\n",hi,nhosts,
				ahost,opsyms[op],chost);
			match = ahost == chost;
		}else
		if( ahost != 0 && Hp->ranges[0].h_high ){
			Verbose("ADDR RANGE [%d/%d] %s %c= %s ?\n",hi,nhosts,
				hostname,opsyms[op],hostpat);
			match = match_addrrange(Hp,ahost);
		}else
		if( !match ){
			Verbose("REGEXP NAME MATCHING [%d/%d] %s %c= %s ?\n",
				hi,nhosts,hostname,opsyms[op],hostpat);
			match = rexpmatch(hostpat,hostname)
		     		|| streq(hostpat,hostname);
		}
		if( match && Hp->h_proto ){
			if( proto != ANYP && !matchProto(Hp->h_proto,proto) ){
				Verbose("--> proto name unmatch: %s\n",proto);
				match = 0;
			}
		}
		if( match && Hp->h_port ){
			if( portnum && !matchPort(Hp->h_port,portnum) ){
				Verbose("--> port num. unmatch: %d\n",portnum);
				match = 0;
			}
		}
		if( match && Hp->h_user ){
			if( username == 0 )
				username = "?";
			if( !matchUser(Hp->h_user,Hp->h_userRexp,username) ){
				Verbose("--> user name unmatch: %s\n",username);
				match = 0;
			}
		}

		if( match && op == ON_EQ  || !match && op == ON_NEQ )
			do_onoff = DO_ON;
		else
		if( match && op == OFF_EQ || !match && op == OFF_NEQ )
			do_onoff = DO_OFF;
		else	do_onoff = DO_NOP;

		switch( do_onoff ){
			case DO_ON:	isin = 1; break;
			case DO_OFF:	isin = 0; break;
		}
		Verbose("==> %d (%s %s)\n",isin,hostlist->hl_what,hostname);
	}
	return isin ? hi : 0;
}
hostIsinList_byname(hostlist,proto,hostname,portnum,username)
	HostList *hostlist;
	char *proto,*hostname;
	char *username;
{	int hosti;
	char *vdom,vdomb[256],hostnameb[512],*rhostname;
	char primaryname[512];

	if( vdom = scanVdomain(hostname,hostnameb,vdomb) )
		rhostname = hostnameb;
	else	rhostname = hostname;

	if( hostname[0] == '-' ){
		hostname += 1;
		hosti = -1;
	}else{
		primaryname[0] = 0;
		hosti = gethost_addr(0,rhostname,primaryname);
		if( primaryname[0] ){
			hostname = primaryname;
			if( vdom ) strcat(hostname,vdom);
		}
	}
	return hostIsinList(hostlist,proto,hostname,hosti,portnum,username);
}
pathIsinList(hostlist,path)
	HostList *hostlist;
	char *path[];
{
}

static addUSER1(user,uv,uc)
	char *user,*uv[];
	int *uc;
{
	uv[*uc] = user;
	*uc += 1;
	return 0;
}
static addStrVec1(str,sv,sc)
	char *str,*sv[];
	int *sc;
{
	sv[*sc] = str;
	*sc += 1;
	return 0;
}

static scan_addrrange(Hp,addr)
	Host *Hp;
	char *addr;
{	char abuf[128],*ap,*dp;
	int ai,iaddr,lo,hi;

	strcpy(abuf,addr);
	ap = abuf;
	for( ai = 0; ai < 4;  ){
		if( dp = strchr(ap,'.') )
			*dp++ = 0;
		if( *ap == '[' ){
			if( sscanf(ap+1,"%d-%d",&lo,&hi) != 2 )
				goto error;
		}else	lo = hi = atoi(ap);

		Hp->ranges[ai].h_low = lo;
		Hp->ranges[ai].h_high = hi;
		ai++;
		if( dp == 0 )
			break;
		else	ap = dp;
	}
	for(; ai < 4; ai++ ){
		Hp->ranges[ai].h_low = 0;
		Hp->ranges[ai].h_high  = 255;
	}
	return 0;
error:
	sv1log("ERROR address range syntax: %s\n",addr);
	return -1;
}

static addHL1(hostmask,hostlist,op,users)
	char *hostmask;
	HostList *hostlist;
	char *users;
{	char op_hostmask[2048];

	if( users )
		sprintf(op_hostmask,"%c%s@%s",op,users,hostmask);
	else	sprintf(op_hostmask,"%c%s",op,hostmask);
	addHostList1(op_hostmask,hostlist);
	return 0;
}
static char *hostpart(hostmask,split)
	char *hostmask;
{	char *dp;

	if( dp = strchr(hostmask,'@') )
	if( dp == hostmask || dp[-1] != '/' ){ /* NOT host/@Cn */
		if( split )
			*dp = 0;
		return dp + 1;
	}
	return 0;
}

extern char *DELEGATE_LOCALNET;
static addHL(hostmask,hostlist)
	char *hostmask;
	HostList *hostlist;
{	char list[2048],*dp;
	int op;
	char *hosts,*users;

	if( strchr(opsyms,*hostmask) )
		op = *hostmask++;
	else	op = '+';
	strcpy(list,hostmask);

	if( dp = hostpart(list,1) ){
		hosts = dp;
		users = list;
	}else{
		hosts = list;
		users = 0;
	}

	if( streq(hosts,".localnet") ){
		strcpy(list,DELEGATE_LOCALNET);
		hosts = list;
	}else
	if( *hosts != '{' )
		return 0;

	Verbose("addHL: %c [ %s ] @ [ %s ]\n",op,users?users:"*",hosts);
	scan_commaListL(hosts,0,addHL1,hostlist,op,users);
	return 1;
}

extern char **dupv();

addHostList1(hostmask,hostlist)
	char *hostmask;
	HostList *hostlist;
{	int idx,siz,nsiz;
	Host *Hp,**oHlist,**nHlist;

	if( hostmask[0] == '!' && hostmask[1] == '!' )
		hostmask += 2;
	if( hostmask[0] == '+' && hostmask[1] == '!' )
		hostmask += 1;
	if( addHL(hostmask,hostlist) )
		return 0;
	if( hostlist->hl_cnt == 0 && *hostmask == '!' )
		addHostList1("*",hostlist);

	Hp = (Host*)calloc(sizeof(Host),1);
	idx = ++hostlist->hl_cnt;
	siz = hostlist->hl_size;
	if( siz <= idx ){
		oHlist = hostlist->hl_List;
		if( hostlist->hl_inc )
			nsiz = siz + hostlist->hl_inc;
		else	nsiz = siz + 1;
		while( nsiz <= idx )
			nsiz++;
		hostlist->hl_size = nsiz;
		if( oHlist == (Host**)0 )
			nHlist = (Host**)calloc(nsiz,sizeof(Host*));
		else	nHlist = (Host**)realloc(oHlist,nsiz*sizeof(Host*));
		hostlist->hl_List = nHlist;
	}
	hostlist->hl_List[idx] = Hp;
	makeHost(hostlist,Hp,0,hostmask);
	return 0;
}

canonWildcardAddr(hostname)
	char *hostname;
{	char *hp,ch,*bp,buff[128],*ap,addr[128];
	int noct,octv,olen,isdg;

	if( strchr(hostname,'*') && strchr(hostname,'.') ){
		hp = hostname;
		ap = addr;
		noct = 0;

		while( *hp ){
			bp = buff;
			isdg = 1;
			while( ch = *hp ){
				hp++;
				*bp++ = ch;
				if( ch == '.' ) 
					break;
				if( !isdigit(ch) )
					isdg = 0;
			}
			*bp = 0;

			if( buff[0]=='*' && (buff[1]==0||buff[1]=='.') ){
				sprintf(ap,"[0-255]%c",buff[1]);
			}else
			if( isdigit(buff[0]) && isdg ){
				octv = atoi(buff);
				if( octv < 0 || 255 < octv )
					goto NOTADDR;
				strcpy(ap,buff);
			}else	goto NOTADDR;
			ap += strlen(ap);

			noct++;
			if( 4 <= noct && *hp != 0 )
				goto NOTADDR;
		}
		if( 2 <= noct && noct <= 4 ){
			*ap = 0;
			strcpy(hostname,addr);
		}
	} NOTADDR:;
}

static isnetmasklen(mask)
	char *mask;
{	int masklen;

	if( isdigit(mask[0]) && mask[1] == 0
	 || isdigit(mask[0]) && isdigit(mask[1]) && mask[2] == 0 ){
		masklen = atoi(mask);
		if( 1 <= masklen && masklen <= 31 )
			return masklen;
	}
	return 0;
}

makeHost(hostlist,Hp,lev,hostmask)
	HostList *hostlist;
	Host *Hp;
	char *hostmask;
{	char *mp,hostnameb[1024],primaryname[512];
	int addr,mask,submask;
	char proto[256],hostmaskb[1024];
	char hostmaskb2[1024],vdomain[256];
	char *route;

	proto[0] = 0;
	if( strstr(hostmask,"://") ){
		sscanf(hostmask,"%[^:]://%s",proto,hostmaskb);
		hostmask = hostmaskb;
	}
	if( route = strstr(hostmask,"//") ){
		*route = 0;
		Hp->h_route = (Host*)calloc(sizeof(Host),1);
		makeHost(hostlist,Hp->h_route,lev+1,route+2);
	}

	if( hostmask[0] == '!' && hostmask[1] == '!' )
		hostmask += 2;

	if( *hostmask == '!' ){ Hp->h_op = OFF_EQ;  hostmask++; }else
/*
	if( *hostmask == '!' ){ Hp->h_op = ON_NEQ;  hostmask++; }else
	if( *hostmask == '?' ){ Hp->h_op = OFF_NEQ; hostmask++; }else
	if( *hostmask == '-' ){ Hp->h_op = OFF_EQ;  hostmask++; }else
*/
			      { Hp->h_op = ON_EQ;
				if(*hostmask=='+')  hostmask++; }

	vdomain[0] = 0;
	if( scanVdomain(hostmask,hostmaskb2,vdomain) )
		hostmask = hostmaskb2;

	mask = 0xFFFFFFFF;
	if( mp = strchr(hostmask,'/') ){
		*mp++ = 0;
		if( submask = isnetmasklen(mp) )
			mask = 0x80000000 >> (submask-1);
		else
		if( strchr(mp,'.') )
			mask = getinetAddr(mp);
		else
		if( mp[0] == '@' ){
			switch( mp[1] ){
			    case 'A': mask = 0xFF000000; break;
			    case 'B': mask = 0xFFFF0000; break;
			    case 'C': mask = 0xFFFFFF00; break;
			    default:  mask = 0; break;
			}
			if( submask = atoi(&mp[2]) )
				mask = mask >> submask;
		}else	sscanf(mp,"%x",&mask);
	}
	Hp->h_mask = mask;
	strcpy(hostnameb,hostmask);

	if( streq(hostmask,".") ){
		char me[128];
		GetHostname(me,sizeof(me));
		strcpy(hostnameb,me);
	}

	{	char *dp,*uv[128],users[2048];
		int ui;

		if( dp = strchr(hostnameb,'@') ){
			if( hostnameb[0] == '{' ){
				char *pp;
				strcpy(users,hostnameb+1);
				if( pp = strchr(users,'}') )
					*pp = 0;
				else	*strchr(users,'@') = 0;
			}else{
				strcpy(users,hostnameb);
				*strchr(users,'@') = 0;
			}
			ui = 0;
			scan_commaList(users,2,addUSER1,uv,&ui);
			uv[ui] = 0;
			Hp->h_user = dupv(uv,0);
			Hp->h_userRexp = makeUserRexp(ui,uv);

			strcpy(hostnameb,dp+1);

			if( !hostlist->hl_noIdent )
			if( hostnameb[0] != '!' )
				enableClientIdent(hostnameb);

			if( hostnameb[0] == '!' ){
				strcpy(hostnameb,hostnameb+1);
				if( Hp->h_op == ON_EQ )
					Hp->h_op = OFF_EQ;
				else	Hp->h_op = ON_EQ;
			}
			/* USER part of USER@!host should be ignored ?? */
		}
	}
	{	char *dp,*pv[128],ports[2048];
		int pi;

		if( dp = strchr(hostnameb,':') ){
			*dp++ = 0;
			strcpy(ports,dp);
			pi = 0;
			scan_commaList(ports,2,addStrVec1,pv,&pi);
			pv[pi] = 0;
			Hp->h_port = dupv(pv,0);
		}
	}
	if( proto[0] != 0 ){
		char *pv[128],protos[2048];
		int pi;

		pi = 0;
		strcpy(protos,proto);
		scan_commaList(protos,2,addStrVec1,pv,&pi);
		pv[pi] = 0;
		Hp->h_proto = dupv(pv,0);
	}

	primaryname[0] = 0;
	canonWildcardAddr(hostnameb);

	if( HostOnClientside(hostnameb) ){
		sv1log("%s[%d]: %c (ClientsideHost) = -C / %x\n",
			hostlist->hl_what,hostlist->hl_cnt,
			opsyms[Hp->h_op],Hp->h_mask);
	}else
	if( HostIncomingIF(hostnameb) ){
		sv1log("%s[%d]: %c (ClientSockHost) = %s / %x\n",
			hostlist->hl_what,hostlist->hl_cnt,
			opsyms[Hp->h_op],hostnameb,Hp->h_mask);
	}else
	if( HostOutgoingIF(hostnameb) ){
		sv1log("%s[%d]: %c (OutgoingIF) = %s / %x\n",
			hostlist->hl_what,hostlist->hl_cnt,
			opsyms[Hp->h_op],hostnameb,Hp->h_mask);
	}else
	if( hostnameb[0] == '-' ){
		Verbose("%s[%d]: %c %s\n",
			hostlist->hl_what,hostlist->hl_cnt,
			opsyms[Hp->h_op], hostnameb);
		if( (addr = _inet_addrV4(hostnameb+1)) != -1 )
			Hp->h_addr = addr;
		else{
			strcpy(hostnameb,hostnameb+1);
			/* to suppress resolution on reload which will
			 * always result in 'unknown' */
			if( gethost_addr(0,hostnameb,primaryname) == -1 ){
				sv1log("Virtual Host Name: '%s'\n",hostnameb);
				sethostcache_predef(hostnameb);
			}
		}
	}else
	if( strchr(hostnameb,'[') ){
		if( scan_addrrange(Hp,hostnameb) == 0 )
		Verbose("%s[%d]: %c %s\n",
			hostlist->hl_what,hostlist->hl_cnt,
			opsyms[Hp->h_op],hostnameb);
	}else
	if( strchr(hostnameb,'*') || streq(hostnameb,"?") ){
		Verbose("%s[%d]: %c %s\n",
			hostlist->hl_what,hostlist->hl_cnt,
			opsyms[Hp->h_op], hostnameb);
	}else
	if( (addr = gethost_addr(0,hostnameb,primaryname)) == -1 ){
		sv1log("ERROR %s[%d] %s ? unknown\n",
			hostlist->hl_what,hostlist->hl_cnt, hostnameb);
	}else{
		if( primaryname[0] )
			strcpy(hostnameb,primaryname);
		Hp->h_addr = addr;
		if( Hp->h_mask == 0 )
			Hp->h_mask = default_netmask(addr);

		Verbose("%s[%d]: %c %s = %x / %x\n",
			hostlist->hl_what,hostlist->hl_cnt,
			opsyms[Hp->h_op], hostnameb,addr,Hp->h_mask);
	}

	Hp->h_name = strdup(hostnameb);
	Hp->h_vdomain = vdomain[0] ? strdup(vdomain) : 0;
}


typedef struct {
	char	 *name;
	char	 *path;
	HostList *hlist;
} PathList; 
static PathList *pathlists;
static int pathlists_siz;
static int pathlists_cnt;

int makePathList(what,path)
	char *what,*path;
{	HostList *HL;
	PathList *PL;
	int nsize,hi;

	if( pathlists_siz <= pathlists_cnt ){
		if( pathlists_siz == 0 ){
			pathlists_siz = 32;
			nsize = sizeof(PathList) * pathlists_siz;
			pathlists = (PathList*)malloc(nsize);
		}else{
			pathlists_siz += 32;
			nsize = sizeof(PathList) * pathlists_siz;
			pathlists = (PathList*)realloc(pathlists,nsize);
		}
	}
	for( hi = 0; hi < pathlists_cnt; hi++ )
		if( strcmp(path,pathlists[hi].path) == 0 )
			return hi+1;

	HL = (HostList*)calloc(sizeof(HostList),1);
	HL->hl_what = strdup(what);

	scan_commaListL(path,0,addHostList1,HL);

	PL = &pathlists[pathlists_cnt++];
	PL->hlist = HL;
	PL->path = strdup(path);

	return pathlists_cnt;
}
matchPathList(hlid,path)
	char *path;
{
}
matchPath1(hlid,user,host,port)
	char *user,*host;
{	HostList *HL;
	int match;

	HL = pathlists[hlid-1].hlist;
	match = hostIsinList_byname(HL,"*",host,port,user);
Verbose("###### matchPath1(%d,%s,%s:%d) = %d\n",hlid,user,host,port,match);
	return match;
}
