/*////////////////////////////////////////////////////////////////////////
Copyright (c) 1997 Electrotechnical Laboratry (ETL), AIST, MITI

Permission to use, copy, modify, and distribute this material for any
purpose and without fee is hereby granted, provided that the above
copyright notice and this permission notice appear in all copies, and
that the name of ETL not be used in advertising or publicity pertaining
to this material without the specific, prior written permission of an
authorized representative of ETL.
ETL MAKES NO REPRESENTATIONS ABOUT THE ACCURACY OR SUITABILITY OF THIS
MATERIAL FOR ANY PURPOSE.  IT IS PROVIDED "AS IS", WITHOUT ANY EXPRESS
OR IMPLIED WARRANTIES.
/////////////////////////////////////////////////////////////////////////
Content-Type:	program/C; charset=US-ASCII
Program:	filter.c (external/internal filter for each connection)
Author:		Yutaka Sato <ysato@etl.go.jp>
Description:

    remote filter ...

        FFROMCL=cfi://host:port
	Filter: cfi://host:port
	CGI:    cgi://host:port

History:
	970814	extracted from service.c and delegated.c
//////////////////////////////////////////////////////////////////////#*/
#include "delegate.h"
#include "vsignal.h"
#include "param.h"
#include <ctype.h> /* isdigit() */

#define FTOCL_FIELD     "X-Request"

extern char **environ;
extern char *DELEGATE_getEnv();
extern char *strfConn();
extern char *malloc();
extern FILE *URLget();
extern char D_RPORT[];

#define F_CL		 1
#define F_TOCL		 2
#define F_FROMCL	 3
#define F_SV		 4
#define F_TOSV		 5
#define F_FROMSV	 6
#define F_MD		 7
#define F_TOMD		 8
#define F_FROMMD	 9
#define F_RPORT		10

#define F_CACHE		 0
#define F_TOCACHE	 0
#define F_FROMCACHE	 0
#define F_PROXY		 0

typedef struct {
	char	*f_name;
	char	*f_filter;
} Filter;
static Filter filters[16] = {
	{ ""		},
	{ P_FCL		},
	{ P_FTOCL	},
	{ P_FFROMCL	},
	{ P_FSV		},
	{ P_FTOSV	},
	{ P_FFROMSV	},
	{ P_FMD		},
	{ P_FTOMD	},
	{ P_FFROMMD	},
	{ P_RPORT	},
	{ ""		},
};

#define FS_RPORT	filters[F_RPORT].f_filter
#define FS_FTOCL	filters[F_TOCL].f_filter
#define FS_FTOSV	filters[F_TOSV].f_filter

#define CFIMAGIC	"#!cfi"
#define isCFI(filter)	(strncmp(filter,CFIMAGIC,strlen(CFIMAGIC)) == 0)

filter_isCFI(which)
{	char *filter;

	if( which == XF_FTOCL )
		return FS_FTOCL != NULL && isCFI(FS_FTOCL);
	if( which == XF_FTOSV )
		return FS_FTOSV != NULL && isCFI(FS_FTOSV);
	return 0;
}

static void sigTERM(sig)
{
	exit(0);
}
static close_all(ifd,ofd,lfd)
{	int fdx,efd;
	int rcode;

	efd = fileno(stderr);
	for( fdx = 0; fdx < 32; fdx++ ){
		if( fdx == SessionFd() )
			continue;
		if( fdx != ifd && fdx != ofd && fdx != lfd && fdx != efd ){
			rcode = close(fdx);
		}
	}
}

static char *open_cfi(file)
	char *file;
{	FILE *fp;
	char desc[128];
	char convspec[0x10000];
	char *sp;
	int size;
	int off;

	fp = fopen(file,"r");
	if( fp == NULL )
		if( isLoadableURL(file) )
			fp = URLget(file,1,NULL);

	sp = NULL;
	if( fp != NULL ){
		desc[0] = 0;
		off = ftell(fp);
		fgets(desc,sizeof(desc),fp);
		fseek(fp,off,0);

		if( strncmp(desc,CFIMAGIC,strlen(CFIMAGIC)) == 0 ){
			size = file_size(fileno(fp));
			sp = malloc(size+1);
			fread(sp,1,size,fp);
			sp[size] = 0;
			if( strncasecmp(file,"data:",5) == 0 )
				sv1log("#### cfi data: scheme ####\n%s\n",sp);
		}
		fclose(fp);
	}
	return sp;
}

static setF1(fi,filter)
	char *filter;
{	char *fname,*cfi;

	if( filter == NULL ){
		fname = filters[fi].f_name;
		filter = DELEGATE_getEnv(fname);
	}
	if( filter && filters[fi].f_filter == NULL ){
		if( cfi = open_cfi(filter) )
			filters[fi].f_filter = cfi;
		else	filters[fi].f_filter = strdup(filter);
	}
}
static setF0(fi)
{
	setF1(fi,NULL);
}

static char *getFilter(Conn,fi)
	Connection *Conn;
{	char *fname,*filter;
	static char filterb[2048];

	filter = NULL;

	fname = filters[fi].f_name;
	filterb[0] = 0;
	if( 0 <= find_CMAP(Conn,fname,filterb) ){
		if( filterb[0] ){
			filter = filterb;
			Verbose("## gotFilter[%s][%s]\n",fname,filter);
		}
	}
	if( filter == NULL )
		filter = filters[fi].f_filter;

	return filter;
}

char *getFTOCL(Conn) Connection *Conn; { return getFilter(Conn,F_TOCL); }
setFTOCL(ftocl) char *ftocl; { FS_FTOCL = ftocl; }
scan_FTOCL(filter) char *filter; { setF1(F_TOCL,filter); }

char *getFTOSV(Conn) Connection *Conn; { return getFilter(Conn,F_TOSV); }
setFTOSV(ftosv) char *ftosv; { FS_FTOSV = ftosv; }
scan_FTOSV(filter) char *filter; { setF1(F_TOSV,filter); }


scan_FILTERS(Conn)
	Connection *Conn;
{
	setF0(F_CL    );
	setF0(F_FROMCL);
	setF0(F_TOCL  );

	setF0(F_SV    );
	setF0(F_FROMSV);
	setF0(F_TOSV  );

	setF0(F_MD    );
	setF0(F_TOMD  );
	setF0(F_FROMMD);

	FS_RPORT    = DELEGATE_getEnv(P_RPORT);
      /*scan_RPORT(Conn,DELEGATE_getEnv(P_RPORT));*/
}

getTimestamp(stime)
	char *stime;
{	int now,usec;

	now = Gettimeofday(&usec);
	StrftimeLocal(stime,sizeof(stime),"%m/%d-%H:%M:%S%.3s",now,usec);
}
teeThru(in,outfp,teefp)
	FILE *in,*outfp,*teefp;
{	int ch;

	while( 0 < ready_cc(in) ){
		ch = getc(in);
		if( ch == EOF )
			return;
		if( outfp ) putc(ch,outfp);
		if( teefp ) putc(ch,teefp);
	}
	if( outfp ) fflush(outfp);
	if( teefp ) fflush(teefp);

	if( outfp == 0 ) simple_relay(fileno(in),fileno(teefp)); else
	if( teefp == 0 ) simple_relay(fileno(in),fileno(outfp)); else
			 relay_tee(fileno(in),fileno(outfp),fileno(teefp));
}

#define STAT	1
#define HEAD	2
#define	EOH	3
#define BODY	4
#define	EOR	5
#define BINARY	6

/*   TEEFILTER
 *	Output to "outfp" is through always to a client/servers
 *	Output to "teefp" may be prefixed with something by "-t -p -n"
 *	When in "tee" mode with "-h -b", only specified part of input will
 *	be relayed to "teefp"
 *	When in non-"tee" mode (maybe in "cat" mode), "teefp" is directed
 *	to original "outfp".  All of output is relayed to "teefp" in this
 *	case and "-h -b" controls to which parts "-t -p -n" is applied
 *	selectively.
 */
teeFilter(infp,outfp,filter,opts,tee)
	FILE *infp,*outfp;
	char *filter,*opts;
{	FILE *teefp;
	char line[1024];
	char *dp;
	int headonly = 0;
	int bodyonly = 0;
	int append = 0;
	int withlnum = 0;
	int processid = 0;
	int timestamp = 0;
	int delcr = 0;
	int ml = 0,mb = 0,NL = 0;
	int where;
	int pid;
	char stime[32];

	dp = opts;
	where = STAT;

	for(;;){
		if( strncmp(dp,"-s",2)==0 ){ where = HEAD;  dp += 2; }else
		if( strncmp(dp,"-h",2)==0 ){ headonly = 1;  dp += 2; }else
		if( strncmp(dp,"-b",2)==0 ){ bodyonly = 1;  dp += 2; }else
		if( strncmp(dp,"-t",2)==0 ){ timestamp = 1; dp += 2; }else
		if( strncmp(dp,"-p",2)==0 ){ processid = 1; dp += 2; pid = getpid(); }else
		if( strncmp(dp,"-n",2)==0 ){ withlnum = 1;  dp += 2; }else
		if( strncmp(dp,"-a",2)==0 ){ append = 1;    dp += 2; }else
		if( strncmp(dp,"-cr",3)==0 ){ delcr = 1;    dp += 3; }else
			break;
	}

	if( *dp == ' ' )
		dp++;
	
	if( dp[0] == 0 )
		teefp = stderr;
	else
	if( append )
		teefp = fopen(dp,"a");
	else	teefp = fopen(dp,"w");
	if( teefp == NULL ){
		sv1log("#### -tee: cannot open %s\n",dp);
		return 0;
	}
	if( !tee ){
		teefp = outfp;
		outfp = NULL;
	}

	for(;;){
		if( fPollIn(infp,100) == 0 ){
			if( outfp ) if( fflush(outfp) == EOF ) break;
			if( teefp ) if( fflush(teefp) == EOF ) break;
		}
		if( fgets(line,sizeof(line),infp) == NULL )
			break;

if( headonly || bodyonly )
if( where == HEAD && NL == 1 )
if( line[3]==' ' && isdigit(line[0])&&isdigit(line[1])&&isdigit(line[2])
/* maybe it's a response status line (previous message was status line only */
 || line[0]=='+' || line[0]=='-'
/* maybe it's a response status line of POP */
){
where = STAT;
NL = 0;
}

		mb += strlen(line);
		ml++;

		if( where <= HEAD && strncmp(line,FTOCL_FIELD,strlen(FTOCL_FIELD))==0 )
			continue;

		if( line[0] == '.' && (line[1] == '\n' || line[1] == '\r') )
			where = BINARY;

		if( delcr ){
			if( dp = strchr(line,'\r') )
				strcpy(dp,dp+1);
		}

		if( outfp )
			fputs(line,outfp);

		if( where == HEAD && strchr(line,':') == NULL )
		if( line[0] != '\r' && line[0] != '\n' )
			where = BODY;

		if( teefp ){
			if( !tee && where == STAT )
				fputs(line,teefp);
			else
			if( headonly && HEAD < where || bodyonly && where != BODY ){
				if( !tee )
				fputs(line,teefp);
			}else{
				NL++;
				if( timestamp ){
					getTimestamp(stime);
					fprintf(teefp,"%s ",stime);
				}
				if( processid )
					fprintf(teefp,"[%d] ",pid);
				if( withlnum )
					fprintf(teefp,"%6d\t",NL);
				fputs(line,teefp);
			}
		}

		if( where == STAT )
			where = HEAD;

		if( where <= HEAD  && (line[0] == '\n' || line[0] == '\r') )
			where = EOH;

		if( where == EOH ){
			where = BODY;
			if( headonly ){
				if( tee )
					simple_relayf(infp,outfp);
				else	simple_relayf(infp,teefp);
				break;
			}
		}

if( headonly || bodyonly )
if( line[0] == '.' && (line[1] == '\n' || line[1] == '\r') ){
where = STAT;
NL = 0;
}
	}
	if( tee && teefp != stderr )
		fclose(teefp);

	Verbose("#### %s [%d] bytes / [%d] lines\n",filter,mb,ml);
}

remote_cfi(filter,in,out)
	char *filter;
	FILE *in,*out;
{	char host[1024],options[1024];
	int port,sock;
	int sv[2][2],rv[2],xv[2];
	FILE *ts,*fs;
	int ch;

	host[0] = 0;
	port = 0;
	options[0] = 0;
	if( sscanf(filter,"cfi://%[^:]:%d/%s",host,&port,options) ){
		sock = client_open("cfi","data",host,port);
		if( sock < 0 )
			return -1;
	}else	return -1;

	ts = fdopen(sock,"w");
	if( options[0] ){
		fprintf(ts,"POST /%s HTTP/1.0\r\n",options);
		fprintf(ts,"\r\n");
		fflush(ts);
	}
	while( 0 < ready_cc(in) ){
		ch = getc(in);
		putc(ch,ts);
	}
	fflush(ts);

	sv[0][0] = fileno(in); sv[0][1] = sock;        xv[0] = 1;
	sv[1][0] = sock;       sv[1][1] = fileno(out); xv[1] = 0;
	relaysx(0,2,sv,xv,rv,NULL,NULL);

	fclose(ts);
	return 0;
}

extern char *wordscan();
builtin_filter(filter,in,out)
	char *filter;
	FILE *in,*out;
{	int rcc;

	if( strncmp(filter,"cfi:",4) == 0 ){
		remote_cfi(filter,in,out);
		return 1;
	}
	if( strcmp(filter,"-thru") == 0 ){
		rcc = simple_relayf(in,out);
		Verbose("#### %s [%d] bytes\n",filter,rcc);
		return 1;
	}
	if( strncmp(filter,"-cat",4) == 0 ){
		teeFilter(in,out,filter,filter+4,0);
		return 1;
	}else
	if( strncmp(filter,"-tee",4) == 0 ){
		teeFilter(in,out,filter,filter+4,1);
		return 1;
	}
	if( strcmp(filter,"-jis") == 0 ){
		rcc = CCV_relay_textX("JIS.JP",in,out);
		Verbose("#### %s [%d]\n",filter,rcc);
		return 1;
	}
	if( strcmp(filter,"-euc") == 0 ){
		rcc = CCV_relay_textX("EUC.JP",in,out);
		Verbose("#### %s [%d]\n",filter,rcc);
		return 1;
	}
	if( strcmp(filter,"-sjis") == 0 ){
		rcc = CCV_relay_textX("SJIS.JP",in,out);
		Verbose("#### %s [%d]\n",filter,rcc);
		return 1;
	}
	return -1;
}

static addConnEnviron(Conn)
	Connection *Conn;
{	char env[1024];

	strfConn(Conn,"REMOTE_IDENT=%u",env); putenv(strdup(env));
	strfConn(Conn,"REMOTE_HOST=%h", env); putenv(strdup(env));
	strfConn(Conn,"REMOTE_ADDR=%a", env); putenv(strdup(env));
}
static addSockEnviron(Conn,what,sock)
	Connection *Conn;
	char *what;
{	int sockhandle;
	char env[128];

	if( (sockhandle = getsockHandle(sock) ) != -1 ){
		sprintf("SOCKHANDLE_%s=%d",what,sockhandle);
		putenv(strdup(env));
	}
}

extern int IO_TIMEOUT;
static recvPeek1(ifd,buff,size)
	char *buff;
{	int rcc;

	if( 0 < PollIn(ifd,IO_TIMEOUT*1000) ){
		setNonblockingIO(ifd,1);
		rcc = RecvPeek(ifd,buff,size);
		setNonblockingIO(ifd,0);
		if( 0 < rcc ) buff[rcc] = 0;
		return rcc;
	}
	return -1;
}
static scanComArg(command,execpath,av,mac,argb)
	char *command,*execpath,*av[],*argb;
{	int ac;
	char *dp;

	if( command[0] != '[' )
		return -1;

	if( dp = strchr(command,']') ){
		execpath[0] = 0;
		sscanf(command+1,"%[^]]",execpath);
		command = dp + 1;
	}else{
		strcpy(execpath,command+1);
		command = "";
	}
	if( *command == ' ' )
		command++;

	ac = decomp_args(av,mac,command,argb);
	sv1log("#### [%s](%d) %s\n",execpath,ac,command);
	return ac;
}
packComArg(command,execpath,args)
	char *command,*execpath,*args;
{
	sprintf(command,"[%s]%s",execpath,args);
}
execsystem(what,command)
	char *what,*command;
{	char execpath[1024],*av[128],argb[1024];

	if( 0 <= scanComArg(command,execpath,av,128,argb) )
		Execvp(what,execpath,av);
	else	Finish(system(command));
}

static execFilter(Conn,in,out,ifd,ofd,what,isresp,filter)
	Connection *Conn;
	FILE *in,*out;
	char *what,*filter;
{	char *av[128];
	int ac,ai;
	char execpath[1024];
	char abuff[0x2000];
	char argb[0x2000];
	int tofil[2];
	int bi;

	if( in  == NULL ) in = fdopen(ifd,"r");
	if( out == NULL ) out = fdopen(ofd,"w");

	if( 0 <= builtin_filter(filter,in,out) ){
		fflush(out);
		Finish(0);
	}

	/* DeleGate's "MASTER" protocol header must not be passed to the
	 * FCL or FFROMCL filter of this delegated if the client delegated have
	 * FTOMD or FMD filter which is not passed such header also.
	 * That is, a pair of FCL/FFROMCL and FMD/FTOMD must communicate
	 * transparently.
	 */
	if( DFLT_PROTO[0] == 0 && ready_cc(in) <= 0 )
	if( streq(what,"FCL") || streq(what,"FFROMCL") ){
		int li,rcc;
		char line[1024];

		rcc = recvPeek1(ifd,line,32);
		if( isHelloRequest(line) ){
			sv1log("#%s: don't pass DeleGate-HELLO to the filter\n",
				what);
			for( li = 0; ; li++ ){
				rcc = RecvLine(ifd,line,sizeof(line));
				if( rcc <= 0 )
					break;
				Verbose("#%s: %s",what,line);
				write(ofd,line,rcc);
				if( line[0] == '\r' || line[0] == '\n' )
					break;
			}
		}
	}

	if( isCFI(filter) ){
		char conninfo[2048];

		sv1log("#### execFilter[%s] CFI\n",what);
		make_conninfo(Conn,conninfo,filter);

		addConnEnviron(Conn);
		close_all(ifd,ofd,curLogFd());
		cfi(isresp,in,out,conninfo,filter);
		Finish(0);
	}else{
		sv1log("#### execFilter[%s] %s\n",what,filter);
		ac = scanComArg(filter,execpath,av,128,argb);

		if( ac < 0 && INHERENT_fork() /* NOT Windows :-) */ ){
			ac = 0;
			av[ac++] = "sh";
			strcpy(execpath,av[0]);
			av[ac++] = "-c";
			av[ac++] = filter;
		}
		if( ac < 0 ){
			ac = decomp_args(av,128,filter,argb);
			strcpy(execpath,av[0]);
			sv1log("#### [%s](%d) %s\n",execpath,ac,filter);
		}
		av[ac] = 0;
		addConnEnviron(Conn);
		arg2env("CFI_");

		for( ai = 0; ai < ac; ai++ )
			Verbose("%s arg[%d] %s\n",what,ai,av[ai]);

		if( 0 < ready_cc(in) ){
			sv1log("#### relay buffered input\n");
			pipe(tofil);
			if( Fork("FILTER-INBUFF") == 0 ){
				FILE *tout;
				close(tofil[0]);
				tout = fdopen(tofil[1],"w");
				simple_relayf(in,tout);
				Finish(0);
			}else{
				close(tofil[1]);
				ifd = tofil[0];
			}
		}

		bi = streq(what,"FCL")||streq(what,"FSV")||streq(what,"FMD");

		if( !bi || INHERENT_fork() /* NOT Windows :-) */ ){
			dup2(ifd,0);
			dup2(ofd,1);
			if( Conn->xf_stderr2out ){
				sv1log("%s: direct stderr to stdout\n",what);
				dup2(ofd,2);
			}
			Execvp(what,execpath,av);
			Finish(-1);
		}else{
			close_all(ifd,ofd,curLogFd());
			setclientsock(ifd);
			setserversock(ofd);
			/* use spawn to inherit socket descriptors ... */
			Spawnvp(what,execpath,av);

			/* if( Windows95() )
			 * to make inherited sockets be disconnected ? */
				wait(0);

			Finish(0);
		}
	}
}

#define CFI_TYPE	"CFI_TYPE"

static callF2(Conn,clsock,svsock,ac,av,arg)
	Connection *Conn;
	char *av[];
	char *arg;
{	char what[32],*filter;
	char type[32];

	filter = wordscan(arg,what);
	sprintf(type,"%s=%s",CFI_TYPE,what);
	putenv(strdup(type));

	while( *filter == ' ' ) filter++;
	sv1log("[%s] callFilter2: %d=%d %d=%d %s\n",what,
		clsock,file_ISSOCK(clsock), svsock,file_ISSOCK(svsock), filter);
	execFilter(Conn,NULL,NULL,clsock,svsock,what,2,filter);
	close(clsock);
	close(svsock);
	return 0;
}
static forkspawnFilter(Conn,what,clsock,svsock,oclsock,osvsock,filter)
	Connection *Conn;
	char *what;
	char *filter;
{	char type[32];

	if( INHERENT_fork() ){
	    if( Fork(what) == 0 ){
		if( 0 <= oclsock ) close(oclsock);
		if( 0 <= osvsock ) close(osvsock);
		sprintf(type,"%s=%s",CFI_TYPE,what);
		putenv(strdup(type));
		execFilter(Conn,NULL,NULL,clsock,svsock,what,2,filter);
	    }
	}else{
		char arg[1024];
		sprintf(arg,"%s %s",what,filter);
		if( 0 <= oclsock) setCloseOnExecSocket(oclsock);
		if( 0 <= osvsock) setCloseOnExecSocket(osvsock);
		execFunc(Conn,clsock,svsock,callF2,arg);
		if( 0 <= oclsock) clearCloseOnExecSocket(oclsock);
		if( 0 <= osvsock) clearCloseOnExecSocket(osvsock);
	}
}

static callF1(Conn,in,out,filter)
	Connection *Conn;
	FILE *in,*out;
	char *filter;
{
	execFilter(Conn,in,out,fileno(in),fileno(out),
		Conn->fi_what,Conn->fi_isresp,filter);
}
static forkexecFilter1X(Conn,dst,src,what,iomode,isresp,filter,pidp)
	Connection *Conn;
	char *what,*filter;
	int *pidp;
{	int tofil[2],fd;
	int pid;

	pipe(tofil);
	if( INHERENT_fork() ){
		if( (pid = Fork(what)) == 0 ){
			ProcTitle(Conn,"(filter:%s)",what);
			Vsignal(SIGTERM,sigTERM);
			Vsignal(SIGINT, sigTERM);
			if( iomode & 1 ){
				if( 0 <= src && src != dst ) close(src);
				close(tofil[1]);
				execFilter(Conn,NULL,NULL,tofil[0],dst,
					what,isresp,filter);
			}else{
				if( 0 <= dst && dst != src ) close(dst);
				close(tofil[0]);
				execFilter(Conn,NULL,NULL,src,tofil[1],
					what,isresp,filter);
			}
		}
		if( iomode & 1 ){
			close(tofil[0]);
			if( pidp ) *pidp = pid;
			return tofil[1];
		}else{
			close(tofil[1]);
			if( pidp ) *pidp = pid;
			return tofil[0];
		}
	}else{
		strcpy(Conn->fi_what,what);
		Conn->fi_isresp = isresp;

		if( iomode & 1 ){
			if( src != dst ) setCloseOnExecSocket(src);
			pid = spawnFilter(Conn,iomode,tofil,dst,callF1,filter);
			if( src != dst ) clearCloseOnExecSocket(src);
			close(tofil[0]);
			if( pidp ) *pidp = pid;
			return tofil[1];
		}else{
			if( dst != src ) setCloseOnExecSocket(dst);
			pid = spawnFilter(Conn,iomode,tofil,src,callF1,filter);
			if( dst != src ) clearCloseOnExecSocket(dst);
			close(tofil[1]);
			if( pidp ) *pidp = pid;
			return tofil[0];
		}
	}
}

#define IS_BUILTIN(filter)	(filter[0] == '-')

static forkexecFilter1(Conn,dst,src,what,iomode,isresp,filter,pidp)
	Connection *Conn;
	char *what,*filter;
	int *pidp;
{	char xwhat[64];
	int sock,psock,fsock,pid;

	if( pidp != NULL )
		*pidp = 0;

	if( iomode & 1 )
		sock = dst;
	else	sock = src;

	if( pidp == NULL )
	if( IS_BUILTIN(filter)
	 || WithSocketFile() || !file_ISSOCK(sock)
	)
	return forkexecFilter1X(Conn,dst,src,what,iomode,isresp,filter,NULL);

	if( !isresp && DFLT_PROTO[0] == 0 && isCFI(filter) ){
		char buff[32];

		if( 0 < recvPeek1(src,buff,16) )
			if( HTTP_isMethod(buff) )
				strcpy(DFLT_PROTO,"http");
	}

	sprintf(xwhat,"%s-P",what);
	psock = forkexecFilter1X(Conn,dst,src,xwhat,iomode,isresp,"-thru",&pid);
	if( pidp != NULL )
		*pidp = pid;
	sv1log("#### pre-filter inserted: %d\n",pid);

	if( iomode & 1 )
		dst = psock;
	else	src = psock;

	fsock = forkexecFilter1X(Conn,dst,src,what, iomode,isresp,filter, NULL);
	close(psock);
	return fsock;
}

/*
 * RPORT : redirection port for response from the MASTER
 */
scan_RPORT(Conn,portin)
	Connection *Conn;
	char *portin;
{	char host[256],tcp_udp[128];
	int sock,port,listen;

	RPORTsock = -1;
	if( portin == NULL )
		return;

	tcp_udp[0] = host[0] = 0;
	port = 0;
	sscanf(portin,"%[^:]:%[^:]:%d",tcp_udp,host,&port);
	if( strcasecmp(tcp_udp,"tcp") == 0 ) listen =  1; else
	if( strcasecmp(tcp_udp,"udp") == 0 ) listen = -1; else{
		sv1tlog("%s ? %s\n",tcp_udp,portin);
		Finish(-1);
	}
	RPORTudp = (listen < 0);
	sock = server_open("RPORT",host,port,listen);
	sockHostport(sock,&port);
	if( host[0] == 0 )
		gethostname(host,sizeof(host));

	sprintf(D_RPORT,"%s:%s:%d",tcp_udp,host,port);
	RPORTsock = sock;
}
static accept_RPORT(Conn)
	Connection *Conn;
{	int sock;

	if( RPORTsock < 0 )
		return -1;

	sv1log("ACCEPT RPORT[%d]...\n",RPORTsock);
	if( RPORTudp ){
		sock = RPORTsock;
		sv1log("ACCEPT RPORT[%d][%d] UDP\n",RPORTsock,sock);
	}else{
		sock = ACCEPT(RPORTsock,0,-1,0);
		sv1log("ACCEPT RPORT[%d][%d]\n",RPORTsock,sock);
		close(RPORTsock);
	}
	RPORTsock = -1;
	return sock;
}
static connect_RPORTX(Conn,portin)
	Connection *Conn;
	char *portin;
{	char host[256],tcp_udp[128];
	int sock,port,listen;

	if( D_RPORTX[0] == 0 )
		return -1;

	sv1log("CONNECT RPORTX[%s]\n",portin);
	if( sscanf(portin,"%[^:]:%[^:]:%d",tcp_udp,host,&port) != 3 )
		return -1;

	if( strcasecmp(tcp_udp,"udp") == 0 )
		sock = UDP_client_open("RPORTX","raw",host,port);
	else	sock = client_open("RPORTX","raw",host,port);
	return sock;
}
insertFTOCL(Conn,client,server)
	Connection *Conn;
{	char *filter;
	int fsock;

	filter = getFilter(Conn,F_TOCL);
	if( filter == NULL )
		return client;

	fsock = forkexecFilter1(Conn,client,server,"FTOCL",  1,1,filter,NULL);

	EchoRequest = 1;
	Conn->xf_filters |= XF_FTOCL;
	return fsock;
}
static insertFFROMCL(Conn,client,fpidp)
	Connection *Conn;
	int *fpidp;
{	char *filter;
	int fromclp[2];
	int fsock;

	filter = getFilter(Conn,F_FROMCL);
	if( filter == NULL )
		return -1;

	/* V.3.0.12 insert a simple relay process
	 * to cause normal EOF at the input of FFROMCL filter ?
	 */
	fsock = forkexecFilter1(Conn,-1,client,"FFROMCL", 0,0,filter,fpidp);
	Conn->xf_filters |= XF_FFROMCL;
	return fsock;
}

static insertFCL(Conn,fromC)
	Connection *Conn;
{	char *filter;
	int fromcl[2];

	filter = getFilter(Conn,F_CL);
	if( filter == NULL )
		return -1;

	Socketpair(fromcl);
	forkspawnFilter(Conn,"FCL",fromC,fromcl[1],fromcl[0],-1,filter);
	close(fromcl[1]);

	Conn->xf_filters |= XF_FCL;
	return fromcl[0];
}

static HTTPtunnel(Conn,what,serv)
	Connection *Conn;
	char *what;
{	char connectmsg[1024];
	char resp[1024];
	int rcc;

	if( !streq(CLNT_PROTO,"http") || !streq(DST_PROTO,"https") )
		return 0;

	sprintf(connectmsg,"CONNECT %s:%d HTTP/1.0\r\n\r\n",DST_HOST,DST_PORT);
	write(serv,connectmsg,strlen(connectmsg));

	for(;;){
		rcc = RecvLine(serv,resp,sizeof(resp));
		if( rcc <= 0 )
			break;
		sv1log("[%s] %s",what,resp);
		if( *resp == '\r' || *resp == '\n' )
			break;
	}
	return 1;
}

static insertFMD(Conn,client,msock)
	Connection *Conn;
{	char *filter;
	int tosv[2];

	if( msock < 0 )
		return -1;

	filter = getFilter(Conn,F_MD);
	if( filter == NULL )
		return -1;

	HTTPtunnel(Conn,"FMD",msock);

	Socketpair(tosv);
	forkspawnFilter(Conn,"FMD",tosv[0],msock,client,tosv[1],filter);
	close(tosv[0]);

	Conn->xf_filters |= XF_FMD;
	return tosv[1];
}
static insertFSV(Conn,client,toS)
	Connection *Conn;
{	char *filter;
	int tosv[2];

	if( toS < 0 )
		return -1;

	filter = getFilter(Conn,F_SV);
	if( filter == NULL )
		return -1;

	if( toProxy )
	HTTPtunnel(Conn,"FSV",toS);

	Socketpair(tosv);
	forkspawnFilter(Conn,"FSV",tosv[0],toS,client,tosv[1],filter);
	close(tosv[0]);

	Conn->xf_filters |= XF_FSV;
	return tosv[1];
}
insertFTOSV(Conn,client,server,pidp)
	Connection *Conn;
	int *pidp;
{	char *filter;
	int fsock;

	filter = getFilter(Conn,F_TOSV);
	if( filter == NULL )
		return server;

	fsock = forkexecFilter1(Conn,server,client,"FTOSV",  1,0,filter,pidp);
	Conn->xf_filters |= XF_FTOSV;
	return fsock;
}
static insertFFROMSV(Conn,client,server)
	Connection *Conn;
{	char *filter;
	int fsock;

	filter = getFilter(Conn,F_FROMSV);
	if( filter == NULL )
		return server;

	fsock = forkexecFilter1(Conn,client,server,"FFROMSV",0,1,filter,NULL);
	Conn->xf_filters |= XF_FFROMSV;
	return fsock;
}

static insertFFROMMD(Conn,master)
	Connection *Conn;
{	char *filter;
	int fsock;

	filter = getFilter(Conn,F_FROMMD);
	if( filter == NULL )
		return -1;

	fsock = forkexecFilter1(Conn,-1,    master,"FFROMMD",0,1,filter,NULL);
	Conn->xf_filters |= XF_FFROMMD;
	return fsock;
}
static insertFTOMD(Conn,master)
	Connection *Conn;
{	int pipe[2];
	char *filter;

	filter = getFilter(Conn,F_TOMD);
	if( filter == NULL )
		return -1;

	return -1;
}


/*
 * Insert filters before the invocation of protocol interpreters
 * Filters which proces all of data from client (FCL and FFROMCL)
 * must be invoked at the start of interpretation.
 */
insert_FCLIENTS(Conn,fromCp,toCp)
	Connection *Conn;
	int *fromCp,*toCp;
{	char *filter;
	int fpid;
	int fromC,toC,toF;

	fromC = *fromCp;
	toC = *toCp;
	fpid = 0;

	if( 0 <= (toF = insertFCL(Conn,fromC)) )
		fromC = toC = toF;
	else
	if( 0 <= (toF = insertFFROMCL(Conn,fromC,&fpid)) )
		fromC = toF;

	scan_RPORT(Conn,FS_RPORT);

	*fromCp = fromC;
	*toCp = toC;
	return fpid;
}

/*
 * Insert filters right after the connection establishment to the server
 */
insert_FSERVER(Conn,fromC)
	Connection *Conn;
{	int toS;
	int toF;

	if( ImCC ) return;

	if( 0 <= (toF = insertFSV(Conn,fromC,ToS)) ){
		ToSX = dup(ToS);
		dup2(toF,ToS);
		/* close(toF); bad for Windows */
	}else
	if( 0 <= ToS ){
		toS = ToS;
		ToS = insertFTOSV(Conn,fromC,ToS,NULL);

		if( ToS != toS )
			ToSX = toS;
		else	ToSX = -1;
	}

	if( 0 <= FromS )
		FromS = insertFFROMSV(Conn,fromC,FromS);
}
close_FSERVER(Conn,realclose)
	Connection *Conn;
{
	if( 0 <= ToSX ){
		close(ToS);
		if( realclose ){
			close(ToSX);
			ToS = -1;
		}else{
			ToS = ToSX;
			/* maybe ImCC */
		}
		ToSX = -1;

		if( Conn->xf_filters & XF_FTOSV ){
			Conn->xf_filters &= ~XF_FTOSV;
			if( (Conn->xf_isremote & XF_FTOSV) == 0 )
				wait(0);
			Conn->xf_isremote &= ~XF_FTOSV;
		}
	}
}

/*
 * Insert filters right after the connection establishment to the MASTER
 */
insert_FMASTER(Conn,msock)
	Connection *Conn;
{	int fromS,toS;
	int toF;

	if( 0 <= (toF = insertFMD(Conn,Conn->cl_sock,msock)) ){
		if( ToServ ) fflush(ToServ);
		dup2(toF,msock);
		close(toF);
		return;
	}

	if( (fromS = accept_RPORT(Conn)) < 0 )
		fromS = msock;

	if( (FromS = insertFFROMMD(Conn,fromS)) < 0 )
		FromS = fromS;
	else	FromSX = FromS; /* FFROMMD is inserted */

	if( 0 <= (toS = insertFTOMD(Conn,msock)) ){
		if( ToServ ) fflush(ToServ);
		ToS = toS;
		/*ToSX = ToS;*/
	}
}

/*
 *  Insert filters after some protocol (HTTP) header interpretation
 */
insert_FPROTO(Conn,toCp,toS,fromSp)
	Connection *Conn;
	int *toCp,*fromSp;
{	int xtoC,fromS,toC;

	fromS = *fromSp;
	toC = *toCp;

	if( fromS < 0 )
		fromS = toS;

	if( 0 <= (xtoC = connect_RPORTX(Conn,D_RPORTX)) ){
		if( toC != Conn->cl_sock )
			close(toC);
		toC = xtoC;
	}

	if( (Conn->xf_filters & XF_FTOCL) == 0 ) /* to avoid
		duplicate insertion in FTP-proxy ... */
	if( 0 <= (xtoC = insertFTOCL(Conn,toC,toS)) && xtoC != toC ){
		if( toC != Conn->cl_sock )
			close(toC);
		toC = xtoC;
	}

	*fromSp = fromS;
	*toCp = toC;
}


filter_withCFI(Conn,which)
	Connection *Conn;
{
	if( ImCC == 0 )
	if( Conn->xf_filters & which )
	if( filter_isCFI(which) )
		return 1;
	return 0;
}

static callFsystem(Conn,in,out,command)
	Connection *Conn;
	FILE *in,*out;
	char *command;
{	int fd;

	/* direct redirection will overwrite another if fileno(out)==0 */
	dup2(fileno(out),3);
	dup2(fileno(in),0);
	dup2(3,1);
	dup2(3,2);
	for( fd = 3; fd < 32; fd++ )
		close(fd);
	system(command);
}

systemFilterNowait(command,in,out,tofd)
	char *command;
	FILE *in,*out;
	int tofd[];
{	int pid;
	Connection ConnBuf, *Conn = &ConnBuf;
	int fdi,fdo;

	sv1log("systemFilter[with buffered input = %d]: %s\n",
		ready_cc(in),command);

	pipe(tofd);
	if( INHERENT_fork() ){
		if( (pid = fork()) == 0 ){
			fdi = dup(tofd[0]);
			fdo = dup(fileno(out));
			close(tofd[1]);
			close(fileno(in));
			dup2(fdi,0); close(fdi);
			dup2(fdo,1); close(fdo);
			execsystem("systemFilterNowait",command);
			Finish(-1);
		}
	}else{
		pid = spawnFilter(Conn,1,tofd,fileno(out),callFsystem,command);
	}

	return pid;
}
systemFilter(command,in,out)
	char *command;
	FILE *in,*out;
{	int fd0,fd1;
	int code;
	int pid;
	int tofd[2];
	FILE *tofp;
	int fdi,fdo;
	void *sig;

	fflush(out);
	if( ready_cc(in) <= 0 ){
		sv1log("systemFilter: %s\n",command);
		fdi = dup(fileno(in));
		fdo = dup(fileno(out));
		fd0 = dup(0); dup2(fdi,0); close(fdi);
		fd1 = dup(1); dup2(fdo,1); close(fdo);
		code = system(command);
		dup2(fd0,0); close(fd0);
		dup2(fd1,1); close(fd1);
		return 0;
	}

	pid = systemFilterNowait(command,in,out,tofd);
	close(tofd[0]);

	tofp = fdopen(tofd[1],"w");
	sig = Vsignal(SIGPIPE,SIG_IGN);
	simple_relayf(in,tofp);
	Vsignal(SIGPIPE,sig);

	fclose(tofp);

	wait(0);
	return pid;
}

relay_tofilter(pid,fromC,toF,fromF,toC)
{	int nready,rcc,nrelayed;
	char buf[4096];
	int wpid,xpid;

	wpid = pid;
	for(;;){
		nrelayed = 0;
		if( 0 <= fromC ){
			nready = PollIn(fromC,10);
			if( nready < 0 )
				goto EXIT;
			if( 0 < nready ){
				rcc = read(fromC,buf,sizeof(buf));
				if( rcc <= 0 )
					goto EXIT;
				nrelayed++;
				write(toF,buf,rcc);
			}
		}
		for(;;){
			nready = PollIn(fromF,100);
			if( nready < 0 )
				goto EXIT;
			if( 0 < nready ){
				rcc = read(fromF,buf,sizeof(buf));
				if( rcc <= 0 )
					goto EXIT;
				nrelayed++;
				write(toC,buf,rcc);
			}
			if( nready == 0 )
				break;
		}
		if( nrelayed == 0 ){ /* Win95 cant sense EOF of PIPE on Poll */
			xpid = NoHangWait();
			if( xpid == pid ){
				wpid = 0;
				goto EXIT;
			}
		}
	}
EXIT:
	return wpid;
}

doXCOM(Conn,in,out,err)
	Connection *Conn;
{	char *command;

	command = DELEGATE_getEnv(P_XCOM);
	if( command == NULL )
		return 0;

	sv1log("exec-COMMAND: %s\n",command);
	if( Conn ){
		addConnEnviron(Conn);
		addSockEnviron(Conn,"CLIENT",ClientSock);
	}
	dup2(in,0);
	dup2(out,1);
	dup2(err,2);

	execsystem("XCOM",command);
	Finish(-1);
}
doXFIL(Conn,in,out,err)
	Connection *Conn;
{	char *filter;
	int toF,fromF,pidF,wpid;
	int tosv[2];

	filter = DELEGATE_getEnv(P_XFIL);
	sv1log("exec-FILTER: %s\n",filter?filter:"NONE (echo)");
	if( filter == NULL )
		return 0;

	Conn->xf_stderr2out = 1;
	fromF = forkexecFilter1(Conn,-1,in,"XFIL", 0,0,filter,&pidF);

/*
wpid = pidF; simple_relay(fromF,out);
*/
wpid = relay_tofilter(pidF,-1,-1,fromF,out);

	if( 0 < wpid ){
		close(fromF);
		close(Conn->cl_sock);
		Kill(wpid,SIGTERM);
		wait(0);
	}
	return 1;
}

service_exec(Conn)
	Connection *Conn;
{
	if( service_permitted(Conn,"exec") == 0 )
		return;

	if( doXCOM(Conn,FromC,ToC,ToC) == 0 )
	if( doXFIL(Conn,FromC,ToC,ToC) == 0 )
		simple_relay(FromC,ToC);
}

connect_main(ac,av,Conn)
	char *av[];
	Connection *Conn;
{	int ai;
	char *arg,host[128],type[128];
	int port;
	int sock;

	host[0] = 0;
	port = 0;
	type[0] = 0;

	for( ai = 1; ai < ac; ai++ ){
		arg = av[ai];
		if( strncmp(arg,"-F",2) == 0 )
			continue;
		if( host[0] == 0 )
			strcpy(host,arg);
		else	sscanf(arg,"%d/%s",&port,type);
	}
	if( host[0] == 0 || port <= 0 ){
		fprintf(stderr,
			"Usage: %s host port[/udp] [XCOM=command]\n",
			1<ac&&strncmp(av[1],"-F",2)==0?av[1]:"connect");
		exit(-1);
	}

	if( strcasecmp(type,"udp") == 0 )
		scan_CONNECT(Conn,"udp");

	set_realserver(Conn,"tcprelay",host,port);
	Conn->from_myself = 1;
	sock = connect_to_serv(Conn,0,1,0);

	dup2(0,4);
	dup2(1,5);
	dup2(2,6);
	if( doXCOM(NULL,sock,sock,fileno(stderr)) == 0 )
	if( doXFIL(Conn,sock,sock,fileno(stderr)) == 0 )
		relay_svcl(Conn,0,1,sock,sock);
	exit(0);
}
