/*////////////////////////////////////////////////////////////////////////
Copyright (c) 1996 Electrotechnical Laboratry (ETL), AIST, MITI

Permission to use, copy, modify, and distribute this material for any
purpose and without fee is hereby granted, provided that the above
copyright notice and this permission notice appear in all copies, and
that the name of ETL not be used in advertising or publicity pertaining
to this material without the specific, prior written permission of an
authorized representative of ETL.
ETL MAKES NO REPRESENTATIONS ABOUT THE ACCURACY OR SUITABILITY OF THIS
MATERIAL FOR ANY PURPOSE.  IT IS PROVIDED "AS IS", WITHOUT ANY EXPRESS
OR IMPLIED WARRANTIES.
/////////////////////////////////////////////////////////////////////////
Content-Type:	program/C; charset=US-ASCII
Program:	fstat.c
Author:		Yutaka Sato <ysato@etl.go.jp>
Description:
History:
	961010	extracted from misc.c
//////////////////////////////////////////////////////////////////////#*/
#include <stdio.h>
#include <string.h>
extern char *strstr();
extern char *malloc();
extern char *realloc();
extern char *strdup();

#include <sys/types.h>
#include <sys/stat.h>

#ifndef S_ISDIR
#define S_ISDIR(m)  (((m)&S_IFMT) == S_IFDIR)
#endif
#ifndef S_ISFIFO
#ifdef S_IFIFO
#define S_ISFIFO(m) (((m)&S_IFMT) == S_IFIFO)
#else
#define S_ISFIFO(m) 0
#endif
#endif

#if defined(S_ISUID) && defined(S_IRWXU)
#define GETEUID()	geteuid()
#define GETEGID()	getegid()
#define STUID(st)	st.st_uid
#define STGID(st)	st.st_gid
#else
#define GETEUID()	0
#define GETEGID()	0
#define STUID(st)	0
#define STGID(st)	0
#define S_IRWXU		(S_IREAD | S_IWRITE | S_IEXEC)
#define S_IRWXG		(S_IREAD | S_IWRITE | S_IEXEC)
#define S_IRWXO		(S_IREAD | S_IWRITE | S_IEXEC)
#endif

/*
 * This check should be done by access() function, but the function
 * can be platform dependent ??
 */
int access_RWX(path)
	char *path;
{	struct stat st;
	int mode,uid,gid;

	if( stat(path,&st) == 0 ){
		mode = st.st_mode;
		if( STUID(st) == GETEUID() && (mode & S_IRWXU) == S_IRWXU )
			return 0;
		if( STGID(st) == GETEGID() && (mode & S_IRWXG) == S_IRWXG )
			return 0;
		if( (mode & S_IRWXO) == S_IRWXO )
			return 0;
	}
	return -1;
}

int file_is(fd)
{	struct stat stat;

	if( fstat(fd,&stat) == 0 )
		return 1;
	else	return 0;
}

valid_fds(where)
	char *where;
{	int fd;

	fprintf(stderr,"##[%d][%s]##",getpid(),where);
	for(fd=0;fd<32;fd++)
		if(file_is(fd))
			fprintf(stderr,"[%2d]",fd);
	fprintf(stderr,"\n");
}

static touch_ctime(fd){
	struct stat st0;
	int omode,xmode;

	if( fstat(fd,&st0) != 0 )
		return -1;

	omode = st0.st_mode;
	xmode = (omode & ~S_IEXEC) | (~(omode & S_IEXEC) & S_IEXEC);
	if( fchmod(fd,xmode) != 0 )
		return -1;
	return fchmod(fd,omode);
}

int fileIsremote1(fd)
{	struct stat st1;
	int time0,time1;

	if( isatty(fd) )
		return 0;

	time0 = time(0);
	if( touch_ctime(fd) < 0 )
		return -1;
	time1 = time(0);
	if( fstat(fd,&st1) != 0 )
		return -1;

	if( st1.st_ctime < time0 || time1 < st1.st_ctime )
		return 1;
	else	return 0;
}

/* 
 * ctime is not updated for TMPFS files on SunOS
 * atime is not updated for NFS files
 */
int file_timeoff(fd,created_now)
{	struct stat st0,st1;
	int now;

	if( !file_isreg(fd) )
		return 0;

	if( fstat(fd,&st0) != 0 )
		return 0;

	if( touch_ctime(fd) != 0 )
		return 0;
	now = time(0);

	if( fstat(fd,&st1) != 0 )
		return 0;

	if( !created_now )
	if( st1.st_ctime == st0.st_ctime ){
		/* this will not work right after REAL CHANGE */
		return 0;
	}

	return now - st1.st_ctime;
}

int file_mtime(fd)
{	struct stat stat;

	if( fstat(fd,&stat) == 0 )
		return stat.st_mtime;
	else	return -1;
}
int file_atime(fd)
{	struct stat stat;

	if( fstat(fd,&stat) == 0 )
		return stat.st_atime;
	else	return -1;
}
int file_times(fd,ctime,mtime,atime)
	int *ctime,*mtime,*atime;
{	struct stat stat;

	if( fstat(fd,&stat) == 0 ){
		*ctime = stat.st_ctime;
		*mtime = stat.st_mtime;
		*atime = stat.st_atime;
		return 0;
	}else	return -1;
}
int file_size(fd)
{	struct stat stat;

	if( fstat(fd,&stat) == 0 )
		return stat.st_size;
	else	return -1;
}
int file_ino(fd)
{	struct stat stat;

	if( fstat(fd,&stat) == 0 )
		return stat.st_ino;
	else	return 0;
}
int file_nlink(fd)
{	struct stat stat;

	if( fstat(fd,&stat) == 0 )
		return stat.st_nlink;
	else	return -1;
}
int file_uid(fd)
{	struct stat stat;

	if( fstat(fd,&stat) == 0 )
		return stat.st_uid;
	else	return -1;
}
int file_stat(fd,sizep,mtimep,atimep)
	int *sizep,*mtimep,*atimep;
{	struct stat stat;

	if( fstat(fd,&stat) == 0 ){
		*sizep = stat.st_size;
		*mtimep = stat.st_mtime;
		*atimep = stat.st_atime;
		return 0;
	}else	return -1;
}

int file_isfifo(fd)
{	struct stat status;

	if( fstat(fd,&status) == 0 )
		return S_ISFIFO(status.st_mode);
	return 0;
}

file_device(fd)
{	struct stat status;

	if( fstat(fd,&status) == 0 )
		return ((status.st_dev) >> 8) & 0xFF;
	return -1;
}
File_device(path)
	char *path;
{	struct stat status;

	if( stat(path,&status) == 0 )
		return ((status.st_dev) >> 8) & 0xFF;
	return -1;
}

File_stat(path,size,time,isdir)
	char *path;
	int *size,*time,*isdir;
{	struct stat status;

	if( stat(path,&status) == 0 ){
		*size = status.st_size;
		*time = status.st_mtime;
		if( S_ISDIR(status.st_mode) )
			*isdir = 1;
		else	*isdir = 0;
		return 0;
	}
	*size = -1;
	*time = -1;
	*isdir = 0;
	return -1;
}

dump_file_stat(fd)
{	struct stat stat;

	if( fstat(fd,&stat) != 0 )
		return;
	fprintf(stderr,"size: %d\n",stat.st_size);
	fprintf(stderr,"mode: 0%o\n",stat.st_mode);
	fprintf(stderr,"uid: %d\n",stat.st_uid);
	fprintf(stderr,"gid: %d\n",stat.st_gid);
	fprintf(stderr,"atime: %d\n",stat.st_atime);
	fprintf(stderr,"mtime: %d\n",stat.st_mtime);
	fprintf(stderr,"ctime: %d\n",stat.st_ctime);
}

/*
 *	UNIX LS COMMAND
 */
char *FileModes(mode){
	static char smode[10];
	int maski;

	for( maski = 0; maski < 9; maski++ ){
		if( (1 << maski) & mode )
			smode[8-maski] = "xwrxwrxwr"[maski];
		else	smode[8-maski] = '-';
	}
	return smode;
}

#ifdef S_IFBLK
#define Fkbytes(st)	(st.st_blocks/2)
stat_blocks(stp) struct stat *stp; {
	return stp->st_blocks;
}
#else
#define Fkbytes(st)	((st.st_size+1023)/1024)
stat_blocks(stp) struct stat *stp; {
	return (stp->st_size / 0x8000 + 1) * 64;
}
#endif

extern char *getUsernameCached();
#define Ftypec(st)	(S_ISDIR(st.st_mode)?'d':'-')
#define Fmodes(st)	FileModes(st.st_mode)
#define Fnlink(st)	st.st_nlink
#define Fowner(st,b)	getUsernameCached(st.st_uid,b)
#define Fbytes(st)	st.st_size
#define Ftime(t,b)	rsctime(t,b)
#define Fmtime(st,b)	Ftime(st.st_mtime,b)
#define Fatime(st,b)	Ftime(st.st_atime,b)

extern void *frex_create();
extern char *frex_match();

typedef struct {
	int	 s_ikey;
	char	*s_skey;
	char	*s_line;
} Elem;

typedef struct {
	FILE	*l_out;
	char	*l_dir;
	void	*l_maskrexp;
struct stat	*l_stp;
	char	*l_fmt;
	int	 l_all;
	int	 l_reflink;

	int	 l_ikey;
	char	*l_buf;

	int	 l_nsize;
	int	 l_nfill;
	Elem	*l_lines;
} LsArg;

static ls1(file,lsa)
	char *file;
	LsArg *lsa;
{	char *dir = lsa->l_dir;
	char *fmt = lsa->l_fmt;
	char *oline = lsa->l_buf;
	char *sp,sc;
	struct stat st;
	int rcode;
	int st_ok;
	int width;
	char path[1024],buf[1024],*op;
	char *tail;

	oline[0] = 0;
	lsa->l_ikey = 0;
	if( file[0] == '.' && !lsa->l_all )
		return 0;

	if( lsa->l_maskrexp ){
		tail = frex_match(lsa->l_maskrexp,file);
		if( tail == NULL || *tail != 0 )
			return 0;
	}

	st_ok = 0;
	path[0] = 0;
	if( dir && dir[0] )
		strcpy(path,dir);
	if( strtailchr(path) != '/' )
		strcat(path,"/");
	strcat(path,file);

	if( lsa->l_stp ){
		st = *lsa->l_stp;
		st_ok = 1;
	}

	op = oline;
	for( sp = fmt; sc = *sp; sp++ ){
		if( op[0] != 0 )
			op += strlen(op);

		if( sc != '%' || sp[1] == 0 ){
			*op++ = sc; *op = 0;
			continue;
		}
		sp++;
		width = numscan(&sp);
		sc = *sp;

		if( sc == 'N' ){
			strcpy(op,file);
			continue;
		}
		if( sc == 'A' ){
			strcpy(op,path);
			continue;
		}
		if( st_ok == 0 ){
			if( lsa->l_reflink || !INHERENT_lstat() )
				rcode = stat(path,&st);
			else	rcode = lstat(path,&st);
			if( rcode != 0 ){
				oline[0] = 0;
				break;
			}
			st_ok = 1;
		}
		switch( sc ){
			case 'm': lsa->l_ikey = st.st_mtime; break;
			case 'a': lsa->l_ikey = st.st_atime; break;
			case 'T': *op++ = Ftypec(st); *op = 0; break;
			case 'M': strcpy(op,Fmodes(st)); break;
			case 'L': sprintf(op,"%*d",width,Fnlink(st)); break;
			case 'O': sprintf(op,"%*s",width,Fowner(st,buf)); break;
			case 'S': sprintf(op,"%*d",width,Fbytes(st)); break;
			case 'K': sprintf(op,"%*d",width,Fkbytes(st)); break;
			case 'D': sprintf(op,"%*s",width,Fmtime(st,buf)); break;
			case 'U': sprintf(op,"%*s",width,Fatime(st,buf)); break;
		}
	}
	return 0;
}
static ls2(file,lsa)
	char *file;
	LsArg *lsa;
{	int li,osize,nsize;

	ls1(file,lsa);
	if( lsa->l_buf[0] ){
		if( lsa->l_nfill == lsa->l_nsize ){
			osize = lsa->l_nsize;
			nsize = (lsa->l_nsize += 2048) * sizeof(Elem);
			if( osize == 0 )
				lsa->l_lines=(Elem*)malloc(nsize);
			else	lsa->l_lines=(Elem*)realloc(lsa->l_lines,nsize);
		}
		li = lsa->l_nfill++;
		lsa->l_lines[li].s_ikey = lsa->l_ikey;
		lsa->l_lines[li].s_skey = strdup(file);
		lsa->l_lines[li].s_line = strdup(lsa->l_buf);
	}
	return 0;
}
static cmpline(e1,e2)
	Elem *e1,*e2;
{	int diff;

	if( diff = e2->s_ikey - e1->s_ikey )
		return diff;
	else	return strcmp(e1->s_skey,e2->s_skey);
}
static sort_ls(dirpath,lsa)
	char *dirpath;
	LsArg *lsa;
{	int li;
	char *line;

	lsa->l_nfill = 0;
	lsa->l_nsize = 0;
	lsa->l_lines = NULL;

	Scandir(dirpath,ls2,lsa);
	qsort(lsa->l_lines,lsa->l_nfill,sizeof(Elem),cmpline);

	for( li = 0; li < lsa->l_nfill; li++ ){
		line = lsa->l_lines[li].s_line;
		fprintf(lsa->l_out,"%s\r\n",line);
		free(line);
		free(lsa->l_lines[li].s_skey);
	}
	free(lsa->l_lines);
}

#define O_PUTSELF	1
#define O_TIMESORT	2
#define O_BYATIME	4
#define O_FORM_L	0x10
#define O_FORM_S	0x20
#define O_REXP		0x40

extern int TIME_NOW;
static int DoFileRexp;

dir2ls(dirpath,stp,opt,fmt,fp)
	char *dirpath;
	struct stat *stp;
	char *opt,*fmt;
	FILE *fp;
{	int onow;
	int flags;
	char sortspec;
	LsArg lsa;
	char *op,*dp;
	char fmt_s[256];
	char fmt_a[256];
	char line[2048];
	int f_l,f_s;
	char fmt_b[256];
	char spath[1024];

	if( opt == NULL )
		opt = "";

	bzero(&lsa,sizeof(lsa));
	flags = 0;

	for( op = opt; *op; op++ ){
		switch( *op ){
			case 'a':
				lsa.l_all = 1;
				break;
			case 'l':
				flags |= O_FORM_L;
				break;
			case 's':
				flags |= O_FORM_S;
				break;
			case 'L':
				lsa.l_reflink = 1;
				break;
			case 'd':
				flags |= O_PUTSELF;
				break;
			case 't':
				flags |= O_TIMESORT;
				break;
			case 'u':
				flags |= O_BYATIME;
				break;
			case '*':
				flags |= O_REXP;
				break;
		}
	}
	if( (DoFileRexp || (flags & O_REXP)) && strpbrk(dirpath,"*?[") ){
		strcpy(spath,dirpath);
		dirpath = spath;
		if( dp = strrchr(dirpath,'/') ){
			lsa.l_maskrexp = frex_create(dp + 1);
			*dp = 0;
		}else{
			lsa.l_maskrexp = frex_create(dirpath);
			dirpath = ".";
		}
	}

	if( fmt == NULL ){
		fmt = fmt_b;
		fmt[0] = 0;

		if( flags & O_FORM_S )
			strcat(fmt,"%4K ");
		if( flags & O_FORM_L )
			strcat(fmt,"%T%M%3L %-8O %8S %D ");

		strcat(fmt,"%N");
	}

	if( flags & O_TIMESORT ){
		if( flags & O_BYATIME )
			sortspec = 'a';
		else	sortspec = 'm';
		sprintf(fmt_s,"%%%c%s",sortspec,fmt);
		fmt = fmt_s;
	}
	if( flags & O_BYATIME ){
		strcpy(fmt_a,fmt);
		fmt = fmt_a;
		if( dp = strstr(fmt,"%D") )
			dp[1] = 'U';
	}

	lsa.l_dir = dirpath;
	lsa.l_stp = stp;
	lsa.l_fmt = fmt;
	lsa.l_out = fp;
	lsa.l_buf = line;
	line[0] = 0;

	onow = TIME_NOW;
	TIME_NOW = time(0);
	setpwent();

	if( !(flags & O_PUTSELF) && fileIsdir(dirpath) )
		sort_ls(dirpath,&lsa);
	else{
		lsa.l_dir = "";
		lsa.l_all = 1;
		ls1(dirpath,&lsa);
		fprintf(fp,"%s\r\n",lsa.l_buf);
	}

	endpwent();
	TIME_NOW = onow;

	if( lsa.l_maskrexp )
		frex_free(lsa.l_maskrexp);
	return;
}

FILE *ls_unix(fp,opt,fmt,dir,stp)
	FILE *fp;
	char *opt,*fmt,*dir;
	struct stat *stp;
{	int io[2];

	if( dir == NULL )
		dir = ".";

	if( fp != NULL ){
		dir2ls(dir,stp,opt,fmt,fp);
		return fp;
	}else{
		pipe(io);
		if( fork() == 0 ){
			close(io[0]);
			fp = fdopen(io[1],"w");
			dir2ls(dir,stp,opt,fmt,fp);
			exit(0);
		}else{
			close(io[1]);
			fp = fdopen(io[0],"r");
			return fp;
		}
	}
}

ls_main(ac,av)
	char *av[];
{	char *fmt,*getenv();

	fmt = getenv("LSFMT");
	if( ac <= 1 )
		ls_unix(stdout,NULL,fmt,".",NULL);
	else
	if( av[1][0] == '-' )
		ls_unix(stdout,av[1],fmt,av[2],NULL);
	else	ls_unix(stdout,NULL,fmt,av[1],NULL);
}
#ifdef LSMAIN
main(ac,av) char *av[]; { DoFileRexp = 1; ls_main(ac,av); }
#endif
