/*////////////////////////////////////////////////////////////////////////
Copyright (c) 1994 Electrotechnical Laboratry (ETL), AIST, MITI

Permission to use, copy, modify, and distribute this material for any
purpose and without fee is hereby granted, provided that the above
copyright notice and this permission notice appear in all copies, and
that the name of ETL not be used in advertising or publicity pertaining
to this material without the specific, prior written permission of an
authorized representative of ETL.
ETL MAKES NO REPRESENTATIONS ABOUT THE ACCURACY OR SUITABILITY OF THIS
MATERIAL FOR ANY PURPOSE.  IT IS PROVIDED "AS IS", WITHOUT ANY EXPRESS
OR IMPLIED WARRANTIES.
/////////////////////////////////////////////////////////////////////////
Content-Type:   program/C; charset=US-ASCII
Program:        ftime.c (portable strftime())
Author:         Yutaka Sato <ysato@etl.go.jp>
Description:
History:
        940719	created
	940727	introduced gmtoff() for portability
//////////////////////////////////////////////////////////////////////#*/
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <time.h>

char *TIMEFORM_mdHMS  = "%m/%d %H:%M:%S";
char *TIMEFORM_HTTPD  = "%d/%b/%Y:%H:%M:%S %z";
char *TIMEFORM_HTTPDs = "%d/%b/%Y:%H:%M:%S%.3s %z";
char *TIMEFORM_GOPHER = "%Y%m%d%H%M%S";
char *TIMEFORM_USENET = "%d %b %Y %H:%M:%S %z";
char *TIMEFORM_RFC822 = "%a, %d %b %Y %H:%M:%S %z";
char *TIMEFORM_RFC850 = "%A, %d-%b-%y %H:%M:%S %z";
char *TIMEFORM_ANSI_C = "%a %b %d %H:%M:%S %Y";
char *TIMEFORM_YMDHMS = "%y %m %d %H %M %S";
char *TIMEFORM_ymdhms = "%y%m%d%H%M%S";
char *TIMEFORM_LS     = "%y/%m/%d %H:%M:%S";
char *TIMEFORM_SYSLOG = "%b %d %H:%M:%S";

extern char *ctime(),*asctime();
extern struct tm *localtime(),*gmtime();

/*
 *	the return value of localtime(), struct tm in global area,
 *	should be saved befre Gmtoff()
 */
static long gmt_off;
static int gmt_off_got = 0;
Gmtoff(){
	struct tm *tm;
	int clock;

	if( gmt_off_got )
		return gmt_off;

	clock = 24*3600;
	tm = localtime(&clock);
	gmt_off = Timegm(tm) - clock;

	gmt_off_got = 1;
	return gmt_off;
}

char *gmtoff(){
	long clock;
	int gmtoff;
	char sctime[32],*satime;
	static char sgmtoff[8];

	if( sgmtoff[0] != 0 )
		return sgmtoff;
	strcpy(sgmtoff,"+0000");

	clock = 24*3600;
	strcpy(sctime,ctime(&clock));
	for( gmtoff = -12; gmtoff <= 12; gmtoff++ ){
		clock = 24*3600 + gmtoff*3600;
		satime = asctime(gmtime(&clock));
		if( strcmp(sctime,satime) == 0 ){
			sprintf(sgmtoff,"%s%04d",
				0<=gmtoff?"+":"-",gmtoff*100);
			break;
		}
	}
	return sgmtoff;
}

static NthWeek(tm,wday1)
	struct tm *tm;
{	int nth,off,sun1st,rem,remain;
	int yday;
	int wbase;

	if( 94 <= tm->tm_year && tm->tm_year <= 96 ) /* historical BUG */
		wbase = 1;
	else    wbase = 0;

	yday = tm->tm_yday;
	rem = yday % 7;

	if( rem < tm->tm_wday )
		off = tm->tm_wday - rem;
	else	off = tm->tm_wday+7 - rem;

	sun1st = 7 - off;
	if( wday1 == 1 )
		sun1st++;
	sun1st = sun1st % 7;

	if( yday < sun1st )
		nth = wbase;
	else{
		remain = yday - sun1st + 1;
		nth = wbase + (remain/7) + (remain % 7 != 0 ? 1:0);
	}
	return nth;
}

static char *Month[] =
{"Jan","Feb","Mar","Apr","May","Jun","Jul","Aug","Sep","Oct","Nov","Dec",0};

static char *Wdays[] =
{"Sun","Mon","Tue","Wed","Thu","Fri","Sat",0};

static char *WDays[] =
{"Sunday","Monday","Tuesday","Wednesday","Thursday","Friday","Saturday",0};

static montoi(mon)
	char *mon;
{	int i;
	char *mo;

	for( i = 0; mo = Month[i]; i++ )
		if( strcmp(mo,mon) == 0 )
			return i;
	return -1;
}
wdaytoi(wday)
	char *wday;
{	int len,i;
	char *wd;

	len = strlen(wday);
	if( len == 2 )
	for( i = 0; wd = Wdays[i]; i++ )
		if( strncasecmp(wd,wday,2) == 0 )
			return i;

	for( i = 0; wd = Wdays[i]; i++ )
		if( strcmp(wd,wday) == 0 )
			return i;
	for( i = 0; wd = WDays[i]; i++ )
		if( strcmp(wd,wday) == 0 )
			return i;
	return -1;
}

int TIME_NOW;
static char *lsDate(clock,tm,lsdate)
	struct tm *tm;
	char *lsdate;
{	int now;

	if( clock < 0 )
		clock = Timegm(tm);

	if( 0 < TIME_NOW )
		now = TIME_NOW;
	else	now = time(0);

	if( clock+(3600*24*31*11) < now )
		sprintf(lsdate,"%s %2d %5d",
			Month[tm->tm_mon],
			tm->tm_mday,
			tm->tm_year + 1900
		);
	else
		sprintf(lsdate,"%s %2d %02d:%02d",
			Month[tm->tm_mon],
			tm->tm_mday,
			tm->tm_hour,
			tm->tm_min
		);
	return lsdate;
}
char *rsctime(clock,lsdate)
	long clock;
	char *lsdate;
{	struct tm *tm;

	tm = localtime(&clock);
	return lsDate(clock,tm,lsdate);
}

long timeBaseDayLocal(clock)
	long clock;
{	struct tm *tm;
	long base;

	tm = localtime(&clock);
	tm->tm_hour = 0;
	tm->tm_min = 0;
	tm->tm_sec = 0;
	base = Timelocal(tm);
	return base;
}

static char *sprint02d(d,n)
	char *d;
{
/*
Year 2000 problem ? :-)

	if( 100 <= n )
		*d++ = "0123456789"[(n/100)%10];
*/
	*d++ = "0123456789"[(n/10)%10];
	*d++ = "0123456789"[n%10];
	*d = 0;
	return d;
}
static char *sprintstr(d,s)
	char *d,*s;
{
	while( *d = *s++ ) d++;
	return d;
}

static Strftime0(atime,size,fmt,tm,usecond,zone)
	char *atime,*fmt;
	struct tm *tm;
	char *zone;
{	char *mon;
	char *fp,*ap;
	char precision[16],*pcp,fm1[16];
	char buf[128];

	ap = atime;
	for( fp = fmt; *fp; fp++ ){
	    if( *fp != '%' ){
		*ap++ = *fp;
		continue;
	    }
	    fp++;

	    pcp = precision;
	    while( *fp == '.' || isdigit(*fp) )
		*pcp++ = *fp++;
	    *pcp = 0;

	    switch( *fp ){
		default:  ap = sprintstr(ap,"");	break;
		case '%': ap = sprintstr(ap,"%");	break;

		case 'A': ap = sprintstr(ap,WDays[tm->tm_wday]);break;
		case 'a': ap = sprintstr(ap,Wdays[tm->tm_wday]);break;
		case 'b': ap = sprintstr(ap,Month[tm->tm_mon]);	break;
		case 'z': ap = sprintstr(ap,zone);		break;
		case 'L': ap = sprintstr(ap,lsDate(-1,tm,buf));	break;

		case 'w': ap = sprint02d(ap,tm->tm_wday);	break;
		case 'd': ap = sprint02d(ap,tm->tm_mday);	break;
		case 'm': ap = sprint02d(ap,tm->tm_mon+1);	break;
		case 'H': ap = sprint02d(ap,tm->tm_hour);	break;
		case 'M': ap = sprint02d(ap,tm->tm_min);	break;
		case 'S': ap = sprint02d(ap,tm->tm_sec);	break;
		case 'U': ap = sprint02d(ap,NthWeek(tm,0));	break;
		case 'W': ap = sprint02d(ap,NthWeek(tm,1));	break;
		case 'y': ap = sprint02d(ap,tm->tm_year);	break;

		case 'Y':
			sprintf(buf,"%d",tm->tm_year+1900);
			ap = sprintstr(ap,buf);
			break;

		case 's':
			sprintf(fm1,"%%%sf",precision);
			sprintf(buf,fm1,usecond/1000000.0);
			ap = sprintstr(ap,buf+1);
			break;
	    }
	}
	*ap = 0;
	return strlen(atime);
}

static struct tm tmerr;
StrftimeLocal(atime,size,fmt,clock,usecond)
	char *atime,*fmt;
	long clock;
{	struct tm *tm;
	char *zone;

	zone = gmtoff(); /* *tm will be overwritten by side effect */
	tm = localtime(&clock);
	if( tm == NULL )
		tm = &tmerr;
	return Strftime0(atime,size,fmt,tm,usecond,zone);
}
StrftimeGMT(atime,size,fmt,clock,usecond)
	char *atime,*fmt;
	long clock;
{	struct tm *tm;
	char *zone;

	tm = gmtime(&clock);
	zone = "GMT";
	return Strftime0(atime,size,fmt,tm,usecond,zone);
}

extern char *scanint();
extern char *awordscan();
extern char *wordscan();
scanftime(stime,fmt)
	char *stime,*fmt;
{	int clock = -1;
	char ch,*fp,*sp;
	int year,mon,mday,hour,min,sec;
	char sym[128];
	char zone[128];
	int gmtoff;
	struct tm tm;

	year = mon = mday = hour = min = sec = -1;
	zone[0] = 0;

	sp = stime;
	for( fp = fmt; ch = *fp; fp++ ){
	    /*fprintf(stderr,"[%s][%s]\n",fp,sp);*/

	    if( ch == ' ' || ch == '\t'  ){
		while( *sp == ' ' || *sp == '\t' )
			sp++;
		continue;
	    }
	    if( ch != '%' ){
		if( *sp != ch )
			goto EXIT;
		sp++;
		continue;
	    }
	    fp++;
	    switch( *fp ){
		case '%': if(*sp != '%') goto EXIT; else sp++; break;
		case 'A': sp = awordscan(sp,sym); break;
		case 'a': sp = awordscan(sp,sym); break;
		case 'b': sp = awordscan(sp,sym); mon = montoi(sym); break;
		case 'd': if((sp = scanint(sp,&mday)) == NULL) goto EXIT; break;
		case 'm': if((sp = scanint(sp,&mon)) == NULL ) goto EXIT;
				mon -= 1; break;
		case 'z': sp = wordscan(sp,zone); break;
		case 'H': if((sp = scanint(sp,&hour)) == NULL) goto EXIT; break;
		case 'M': if((sp = scanint(sp,&min) ) == NULL) goto EXIT; break;
		case 'S': if((sp = scanint(sp,&sec) ) == NULL) goto EXIT; break;
		case 'U': sp = awordscan(sp,sym); break;
		case 'W': sp = awordscan(sp,sym); break;
		case 'y':
		case 'Y': if((sp = scanint(sp,&year)) == NULL) goto EXIT;
			if( 1900 <= year )
				year -= 1900;
			break;
	    }
	}
EXIT:
	if( year<0 || mon<0 || mday<0 || hour<0 || min<0 || sec<0 )
		return -1;

	tm.tm_sec = sec;
	tm.tm_min = min;
	tm.tm_hour = hour;
	tm.tm_mday = mday;
	tm.tm_mon = mon;
	tm.tm_year = year;

	tm.tm_wday = 0;
	tm.tm_yday = 0;
	tm.tm_isdst = 0;
	/*tm.tm_zone = zone;*/
	/*tm.tm_gmtoff = 0;*/
	clock = Timegm(&tm);

	if( 0 <= clock && zone[0] ){
		int off;

		gmtoff = 0;
		if( zone[0] == '+' || zone[0] == '-' ){
			off = atoi(zone+1);
			gmtoff = 3600*(off/100) + 60*(off%100);
		}
		if( zone[0] == '+' ) clock -= gmtoff;
		if( zone[0] == '-' ) clock += gmtoff;
	}
	/*fprintf(stderr,"%d/%d/%d %d:%d:%d [%s] %d\n",
		year,mon+1,mday,hour,min,sec,zone,clock);*/
	return clock;
}
toclockLocal(y,m,d,H,M,S)
{	struct tm tm;
	int clock;

	tm.tm_year = y;
	tm.tm_mon  = m - 1;
	tm.tm_mday = d;

	tm.tm_hour = H;
	tm.tm_min  = M;
	tm.tm_sec  = S;

	tm.tm_wday = 0;
	tm.tm_yday = 0;
	tm.tm_isdst = 0;
	/*tm.tm_gmtoff = 0;*/

	clock = Timelocal(&tm);
	return clock;
}
fromclockLocal(clock,w,y,m,d,H,M,S)
	long clock;
	int *w,*y,*m,*d,*H,*M,*S;
{	struct tm *tm;

	if( (tm = localtime(&clock)) == NULL )
		return -1;

	*w = tm->tm_wday;

	*y = tm->tm_year;
	*m = tm->tm_mon + 1;
	*d = tm->tm_mday;

	*H = tm->tm_hour;
	*M = tm->tm_min;
	*S = tm->tm_sec;

	return 0;
}

scanHTTPtime(stime)
	char *stime;
{	int itime;

	if( 0 <= (itime = scanftime(stime,TIMEFORM_RFC822)) ) return itime;
	if( 0 <= (itime = scanftime(stime,TIMEFORM_RFC850)) ) return itime;
	if( 0 <= (itime = scanftime(stime,TIMEFORM_ANSI_C)) ) return itime;
	return -1;
}
scanNNTPtime(stime)
	char *stime;
{	int itime;

	if( 0 <= (itime = scanftime(stime,TIMEFORM_RFC822)) ) return itime;
	if( 0 <= (itime = scanftime(stime,TIMEFORM_RFC850)) ) return itime;
	if( 0 <= (itime = scanftime(stime,TIMEFORM_ANSI_C)) ) return itime;
	if( 0 <= (itime = scanftime(stime,TIMEFORM_USENET)) ) return itime;
	return -1;
}
scanUNIXFROMtime(stime)
	char *stime;
{	int itime;

	if( 0 <= (itime = scanftime(stime,TIMEFORM_ANSI_C)) )
		return itime - Gmtoff();
	else	return scanNNTPtime(stime);
}

scantime(stime,tm)
	char *stime;
	struct tm *tm;
{	int mday,year,hour,min,sec;
	char swday[32],smon[32];
	int ni;
	int clock;
	struct tm tmb;

	if( tm == NULL ){
		tm = &tmb;
		bzero(tm,sizeof(tmb));
	}

	ni = sscanf(stime,"%[^,], %d-%[^-]-%d %d:%d:%d",
		swday,&mday,smon,&year,&hour,&min,&sec);

	if( ni != 7 )
	ni = sscanf(stime,"%[^,], %d %s %d %d:%d:%d",
		swday,&mday,smon,&year,&hour,&min,&sec);

	clock = -1;
	if( ni == 7 ){
		tm->tm_wday = wdaytoi(swday);
		tm->tm_year = 1900 < year ? (year - 1900) : year;
		tm->tm_mon  = montoi(smon);
		tm->tm_mday = mday;
		tm->tm_hour = hour;
		tm->tm_min  = min;
		tm->tm_sec  = sec;
		clock = Timegm(tm);
		return clock;
	}
	return clock;
}

char *scanLsDate(str,date)
	char *str,*date;
{	char *sp,mon[128],day[128],plus[128];
	int monlen,mi,len;

	while( *str == ' ' )
		str++;
	sp = wordscan(str,mon);
	monlen = strlen(mon);

	for( mi = 0; mi < 12; mi++ ){
		if( strncmp(Month[mi],mon,monlen) == 0 ){
			sp = wordscan(sp,day);
			sp = wordscan(sp,plus);
			if( 0 < atoi(day) ){
				len = sp - str;
				strncpy(date,str,len);
				date[len] = 0;
				return sp;
			}
			return NULL;
		}
	}
	return NULL;
}
LsDateClock(date,now)
	char *date;
	long now;
{	char smon[128],smday[128],plus[128];
	struct tm tmnow,tm;
	int monnow,clock;

	tmnow = *gmtime(&now);
	tm = tmnow;
	monnow = tm.tm_mon;

	if( sscanf(date,"%s %s %s",smon,smday,plus) != 3 )
		return -1;
	if( (tm.tm_mon = montoi(smon)) < 0 )
		return -1;
	if( (tm.tm_mday = atoi(smday)) < 0 )
		return -1;

	tm.tm_sec = 59;
	if( sscanf(plus,"%d:%d",&tm.tm_hour,&tm.tm_min) == 2 ){
		if( monnow < tm.tm_mon )
			tm.tm_year -= 1;
	}else{
		tm.tm_year = atoi(plus);
		if( 1900 < tm.tm_year )
			tm.tm_year -= 1900;
		tm.tm_hour = 23;
		tm.tm_min = 59;
	}
	clock = Timegm(&tm);
/* {
	char buff[128];
	StrftimeGMT(buff,128,TIMEFORM_RFC822,clock);
	fprintf(stderr,"#### [%-20s][%s]\n",date,buff);
} */
	return clock;
}

tmcmp(tm1,tm2)
	struct tm *tm1,*tm2;
{
	if(tm1->tm_year != tm2->tm_year) return tm1->tm_year-tm2->tm_year;
	if(tm1->tm_mon  != tm2->tm_mon ) return tm1->tm_mon -tm2->tm_mon ;
	if(tm1->tm_mday != tm2->tm_mday) return tm1->tm_mday-tm2->tm_mday;
	if(tm1->tm_hour != tm2->tm_hour) return tm1->tm_hour-tm2->tm_hour;
	if(tm1->tm_min  != tm2->tm_min ) return tm1->tm_min -tm2->tm_min ;
	if(tm1->tm_sec  != tm2->tm_sec ) return tm1->tm_sec -tm2->tm_sec ;
	return 0;
}

static int yday_base[] = { 0,31,59,90,120,151,181,212,243,273,304,334,1000 };
Timegm(tm)
	struct tm *tm;
{	struct tm *tm0;
	int yoff,leapy,clock,yday;

	yoff = tm->tm_year - 70;
	yday = yday_base[tm->tm_mon] + (tm->tm_mday - 1);
	if( 1 < tm->tm_mon && (yoff-2) % 4 == 0 )
		yday += 1;

	clock = (
		 (
		  (
		   (
		    (yoff*365 + (yoff+1)/4 + yday) * 24
		   ) + tm->tm_hour
		  ) * 60
		 ) + tm->tm_min
		) * 60
		 + tm->tm_sec;
	return clock;
}
/*
 *	convert the "tm" time as if it is a local time, then detuct
 *	the Gmtoff()
 */
Timelocal(tm)
	struct tm *tm;
{	struct tm tms;

	tms = *tm;
	return Timegm(&tms) - Gmtoff();
}

/*
main(){
char buf[64],buf2[64];
int clock = time(0);
static struct tm tm1,tm2,*tm;
char *st;
int i,nw,nnw;

nw = 0;
clock -= 24*60*60*280;
for(i = 0; i < 2000; i++)
{	char buf[1024];

	tm = localtime(&clock);
	nnw = NthWeek(tm,1);
	if( nw != nnw ){
		strftime(buf,sizeof(buf),"%y/%m/%d(%w)",tm);
		printf("%2d: %-12s",nnw,buf);
	}
	nw = nnw;
	clock += 24*60*60;
}
exit();

StrftimeLocal(buf,32,TIMEFORM_RFC822,clock); printf("%s\n",buf);
StrftimeGMT(buf,32,TIMEFORM_RFC822,clock); printf("%s\n",buf);
sleep(1);
StrftimeGMT(buf2,32,TIMEFORM_RFC822,time(0)); printf("%s\n",buf2);

scantime(buf,&tm1); st = asctime(&tm1); printf("== %s\n",st);
scantime(buf2,&tm2); st = asctime(&tm2); printf("== %s\n",st);
printf("%d\n",tmcmp(&tm1,&tm2));
printf("%d\n",tmcmp(&tm2,&tm1));

}
*/
/*
main(){
	char date[1024];
	int now,t;

	now = time(0);

	StrftimeGMT(date,sizeof(date),TIMEFORM_RFC850,now);
	t = scanHTTPtime(date);
	printf("RFC850 %d %d %s\n",now,t,date);

	StrftimeGMT(date,sizeof(date),TIMEFORM_RFC822,now);
	t = scanHTTPtime(date);
	printf("RFC822 %d %d %s\n",now,t,date);

	StrftimeGMT(date,sizeof(date),TIMEFORM_ANSI_C,now);
	t = scanHTTPtime(date);
	printf("ANSI_C %d %d %s\n",now,t,date);
}
*/

/*
main()
{	unsigned long clock,clock1,clock2;
	struct tm *tm;
	char fmt[64];
	char date1[256],date2[256];

	strcpy(fmt,"%y/%m/%d W=%W");
	for( clock = 0; clock < 0xFFFFFFFF; clock += 60*60*24 ){
		tm = gmtime(&clock);
		clock1 = timegm(tm);
		clock2 = Timegm(tm);
		if( clock1 != clock2 ){
			printf("%8d %8d %8d %s",clock,clock1,clock2,ctime(&clock));
		}
		strftime(date1,sizeof(date1),fmt,tm);
		StrftimeGMT(date2,sizeof(date2),fmt,clock);
		if( strcmp(date1,date2) == 0 ){
			fprintf(stderr,"%10d [%s][%s]\r",clock,date1,date2);
			fflush(stderr);
		}else{
			fprintf(stdout,"%10d [%s][%s]\n",clock,date1,date2);
		}
	}
}
*/

/*
main()
{
	LsDateClock("Dec 22 01:46",time(0));
	LsDateClock("Dec 22 1996", time(0));
	LsDateClock("Apr 14 01:46",time(0));
	LsDateClock("Apr 14 1995", time(0));
	LsDateClock("Jan 22 19:55",time(0));
}
*/
