/* 
 *  AutoPPP is a REXX script for periodic or continuous connection 
 *  to a PPP line after which an optional Internet app (eg. mail 
 *  reader or news reader to download messages) is launched.
 *
 *  Author:  David Forrai <dforrai@erinet.com>
 *  Revision 1:  9 Nov 1996
 *    Corrected logic to avoid creation multiple queues per connect attempt.
 *    Added logic to prevent orphaned Wait4PPP jobs after PPP exits on error.
 * 
 *  First, load the REXX Util functions to access SysSleep() 
 */ 
CALL RxFuncAdd 'SysLoadFuncs','RexxUtil','SysLoadFuncs' 
CALL SysLoadFuncs 
/* 
 *  Default values for start time, interval, and queue name
 */ 
hmsTime = '' 
hmsInterval = '' 
start = 0        /* Start immediately  */ 
interval = 86400 /* repeat in 24 hours */ 
newQueue = ''
/* 
 * Parse the command line info 
 */ 
PARSE ARG args 
allDone = 0 
DO UNTIL allDone = 1 
/* 
 *  Parse the first word out of the args string, then convert it to 
 *  uppercase. 
 */ 
   PARSE VAR args program args 
   PARSE UPPER VAR program program 
   allDone = 1 
/* 
 *  If the first word is preceeded by /H, it specifies the start time 
 *  in hr:min:sec. Loop again to parse out the program names and 
 *  arguments. 
 */ 
   IF POS('/H', program) = 1 THEN DO 
/* 
 *  If the first word is preceeded by /I, it specifies the interval 
 *  time in hr:min:sec.  Loop again to parse out the program names 
 *  and arguments. 
 */ 
      hmsTime = SUBSTR(program, 3) 
      allDone = 0 
      SAY 'Start time =' hmsTime 
   END  /* IF */ 
   IF POS('/I', program) = 1 THEN DO 
      hmsInterval = SUBSTR(program, 3) 
      allDone = 0 
      SAY 'Interval =' hmsInterval 
   END  /* IF */ 
END /* DO */ 
/* 
 *  If a start time was given on the command line, calculate the 
 *  interval between the current time and the time to launch the 
 *  passed program. 
 */ 
IF hmsTime \= '' THEN DO 
   PARSE VALUE hmsTime WITH hour ':' minute ':' second 
   PARSE VALUE TIME() WITH curHr ':' curMin ':' curSec 
   current = 3600 * curHr + 60 * curMin 
   IF DATATYPE(hour, 'N') THEN DO 
      IF (hour > 23) | (hour < 0) THEN DO 
         SAY 'Invalid start hour ('||hour||'), aborting ...' 
         'EXIT' 
      END  /* If */ 
      ELSE hour = 3600 * hour 
   END /* IF */ 
   ELSE DO 
      SAY 'Invalid start hour ('||hour||'), aborting ...' 
     'EXIT' 
   END  /* Else */ 
   IF DATATYPE(minute, 'N') THEN DO 
      IF (minute > 59) | (minute < 0) THEN DO 
         SAY 'Invalid start minute('||minute||'), aborting ...' 
         'EXIT' 
      END  /* If */ 
      ELSE minute = 60 * minute 
   END /* IF */ 
   ELSE DO 
      SAY 'Invalid start minute('||minute||'), aborting ...' 
     'EXIT' 
   END  /* Else */ 
   start = hour + minute - current 
   IF start < 0 THEN start = 86400 + start 
END  /* IF */ 
/* 
 *  If an interval was given as an argument, calculate the interval 
 *  between logins. 
 */ 
IF hmsInterval \= '' THEN DO 
   PARSE VALUE hmsInterval WITH hour ':' minute ':' second 
   IF DATATYPE(hour, 'N') THEN DO 
      IF (hour > 23) | (hour < 0) THEN DO 
         SAY 'Invalid interval hour ('||hour||'), aborting ...' 
         'EXIT' 
      END /* If */ 
      ELSE interval = 3600 * hour 
   END /* If */ 
   ELSE DO 
      SAY 'Invalid interval hour ('||hour||'), aborting ...' 
      'EXIT' 
   END /* If */ 
   IF DATATYPE(minute, 'N') THEN DO 
      IF (minute > 59) | (minute < 0) THEN DO 
         SAY 'Invalid interval minute ('||minute||'), aborting ...' 
         'EXIT' 
      END /* If */ 
      ELSE interval = interval + 60 * minute 
   END /* If */ 
   ELSE DO 
      SAY 'Invalid interval minute ('||minute||'), aborting ...' 
      'EXIT' 
   END /* If */ 
END  /* IF */ 
/* 
 *  Set the program running flag to 0 (= not running). 
 */ 
programUp = 0 
connectionFailed = 0 
hour = start / 3600 
PARSE VALUE hour WITH hour '.' minute 
minute = ((start / 3600) - hour) * 60 + 0.5 
PARSE VALUE minute WITH minute '.' second 
SAY 'PPP will start in' hour 'hours and' minute 'minutes.' 
/* 
 *  Now sleep until the start time 
 */ 
IF start > 0 THEN CALL SysSleep(start) 
/* 
 *  Start the main timing loop 
 */
DO FOREVER
/* 
 *  Reset the timer if the program is not currently running 
 */ 
   IF (programUp = 0) & (connectionFailed = 0) THEN CALL TIME 'R' 
/* 
 *  Turn off the IP gate. 
 */ 
   '@IPGATE off' 
/* 
 *  If a program was passed to automatically launch, start the 
 *  "wait for PPP connect" script in the background, passing it 
 *  the name of the queue which will receive output from PPP.EXE. 
 *  By monitoring the queue's contents, we can tell when the PPP 
 *  connection is made. 
 */ 
   IF (program \= '') & programUp = 0 THEN DO 
/* 
 *  Create a queue and set it as the default queue if no queue was created.
 */
      IF newQueue = '' THEN DO
         newQueue = RXQUEUE('CREATE') 
         oldQueue = RXQUEUE('SET', newQueue)
      END /* If */
      'START /B /MIN Wait4PPP.CMD' newQueue program args 
/* 
 *  Start the PPP connection and pipe the output into the write queue 
 */ 
      CALL LINEOUT 'QUEUE:', 'AutoPPP: PPP started' 
      'ppp.exe | rxqueue' newQueue 
      CALL LINEOUT 'QUEUE:', 'AutoPPP: PPP ended' 
/* 
 *  Sleep for 3 seconds to let Wait4PPP process the PPP ended message 
 *  if it is still running. 
 */ 
      CALL SysSleep(3) 
   END /* If */ 
/*       CALL LINEOUT 'QUEUE:'
 *  If there is no program to automatically start or the program 
 *  is already running, just launch PPP. 
 */ 
   ELSE DO
      'ppp.exe'
      IF newQueue \= '' THEN DO
         CALL LINEOUT 'QUEUE:', 'AutoPPP: PPP ended' 
         CALL SysSleep(3) 
      END /* If */
   END /* Else */
/* 
 *  PPP has ended. 
 */ 
   IF program \= '' THEN DO 
      line = '' 
      pppEnded = 0 
      DO UNTIL (programUp = 0) | (pppEnded = 1) | (connectionFailed = 1) 
         IF QUEUED() \= 0 THEN DO 
            DO WHILE QUEUED() \= 0 
               line = LINEIN('QUEUE:') 
               SAY line /* Echo line for diagnostic purposes */ 
               IF POS('Program started', line) > 0 THEN DO 
                  programUp = 1 
                  connectionFailed = 0 
               END /* If */ 
               IF POS('Program ended', line) > 0 THEN programUp = 0 
               IF POS('Timed out', line) > 0 THEN connectionFailed = 1 
               IF POS('PPP ended', line) > 0 THEN pppEnded = 1 
            END /* Do While */ 
         END  /* If */ 
         ELSE 
            CALL SysSleep(1) 
      END /* Do While */ 
      IF programUp = 0 & connectionFailed = 0 THEN DO 
/* 
 * If the program isn't running, destroy the queue, 
 */ 
         CALL RXQUEUE 'SET', oldQueue 
         CALL RXQUEUE 'DELETE', newQueue
         newQueue = ''
/* 
 *  turn on the IP gate, 
 */ 
         '@IPGATE on' 
         wait = interval - TIME('E') 
/* 
 *  and sleep until it's time to run again. 
 */ 
         SAY 'PPP ended at '||TIME() 
         hour = wait / 3600 
         PARSE VALUE hour WITH hour '.' minute 
         minute = ((wait / 3600) - hour) * 60 - 0.5 
         PARSE VALUE minute WITH minute '.' second 
         SAY 'PPP will start in' hour 'hours and' minute 'minutes.' 
         wait = interval - TIME('R') 
         PARSE VALUE wait WITH wait '.' dummy 
         CALL SysSleep(wait) 
      END /* If */ 
/* 
 *  For all other cases, loop back and try to connect again. 
 */ 
   END /* If */ 
END /* do */ 
