/******************************* REXX *********************************/
/*  REXX External function to calculate a relative day of the month   */
/*  (for example, the second Tuesday of March).  Input is a four      */
/*  character code of the format "rdmm" where "r" = the relative day  */
/*  of the month (1 - 5 for first through fifth), or "L" for the      */
/*  last day of the month (if you specify "5" and there is no fifth   */
/*  day of the month, you will get an error).  "d" is a number from   */
/*  1 through 7 indicating which day of the week (1=Sunday,           */
/*  7=Saturday).  "mm" is the two-digit number of the month           */
/*  (01=January, 12=December).  The result of the function call is    */
/*  extended calendar date of the form "mm/dd/yyyy" where "yyyy" =    */
/*  the current year, "mm" = the month, and "dd" is the day of the    */
/*  month.  Use a function call to "JCCalJul()" to convert this to a  */
/*  standard Julian date.                                             */
/*                                                                    */
/*  Original program written by Jaime A. Cruz, Jr. and released to    */
/*  the public domain.  If you wish to contact the author, you may    */
/*  do so at 72267.1372@compuserve.com, or jcruz@ibm.net.             */
/**********************************************************************/
/*                   Table of days in each month.                     */
/**********************************************************************/
month.0 = 12
month.1 = 31
month.2 = 28
month.3 = 31
month.4 = 30
month.5 = 31
month.6 = 30
month.7 = 31
month.8 = 31
month.9 = 30
month.10 = 31
month.11 = 30
month.12 = 31
 
Parse Upper Arg code, .
/**********************************************************************/
/*  Extract the relative date information, parsing it into            */
/*  variables.  Also extract the current year from the system.        */
/**********************************************************************/
Parse Value code With 1 which 2 ,
                      2 d 3 ,
                      3 mm 5
year = Left(Date('S'), 4)
If JCLepYer(year) Then
   month.2 = 29
 
/**********************************************************************/
/*  Calculate the first day of the month into Julian format, then     */
/*  determine the last day of that month.                             */
/**********************************************************************/
jul_date = JCCalJul(Right(mm, 2, '0') || '/01/' || ,
                    year)
mm = Strip(mm, 'L', '0')
jul_max = jul_date + month.mm - 1
 
/**********************************************************************/
/*  Calculate upon which day of the week the first day of the month   */
/*  falls.                                                            */
/**********************************************************************/
first_day = JCDoW(jul_date)
 
/**********************************************************************/
/*        Calculate the first occurence of our specified day.         */
/**********************************************************************/
delta = d - first_day
If delta < 0 Then
   delta = delta + 7
jul_date = jul_date + delta
 
/**********************************************************************/
/*  Calculate the date of our requested day.  If the user specified   */
/*  the fifth occurence, and it doesn't exist, return an error.       */
/**********************************************************************/
If which \= 'L' Then
   Do
      x = (which - 1) * 7
      jul_date = jul_date + x
      If jul_date > jul_max Then
         jul_date = 'ERROR'
   End
Else
 
/**********************************************************************/
/*  Calculate the last occurence of our requested day (if "L"         */
/*  specified).                                                       */
/**********************************************************************/
   Do
      x = 28
      Do Forever
         If jul_date + x > jul_max Then
            x = x - 7
         Else
            Do
               jul_date = jul_date + x
               Leave
            End
      End
   End
date = JCJulCal(jul_date)
 
/**********************************************************************/
/*       Return to the invoker with the requested information.        */
/**********************************************************************/
Return date
