/***************************************************************************************************************************

  FILE NAME :
    tcpsock.cpp


  DESCRIPTION :
    This module contains the methods used to implement public interface to the class TCPSocket.  This is a
    base class that provides some VERY simple TCPIP socket capabilities.


  METHODS DEFINED IN THIS MODULE :
    TCPSocket :: TCPSocket ()
    TCPSocket :: TCPSocket ( const TCPSocket  & rctcpsocket     )
    TCPSocket :: TCPSocket ( const SockAddr   & rcsokaddrLocal , const SockAddr & rcsokaddrRemot = SockAddr () ) )

    SINT32        TCPSocket :: accept         ()                 
    const char  * TCPSocket :: className      () const           
    int           TCPSocket :: isStream       () const           
    void          TCPSocket :: listen         ()                 
    int           TCPSocket :: maxPending     () const           
    int           TCPSocket :: protocol       () const           
    int           TCPSocket :: setMaxPending  ( int nNumToPend ) 


  EXCEPTION CLASSES THROWN FROM THIS MODULE :
    TCPIPError


  Copyright 1997, Peter Garner
    This code may be used for any NON-MILITARY purpose.
 
***************************************************************************************************************************/
//  Standard C Headers
# include <time.h>


//  TCPIP C Headers
extern "C"
{
  # include <netdb.h>
  # include <sys/types.h>
  # include <sys/ioctl.h>
  # include <netinet/in.h>
}


//  Local C++ Headers
# include <ipsock.hpp>
# include <tcperror.hpp>


//  Locally Defined C Headers
//  We include <genmacro.h> last, since it has conditional defines of MIN and MAX.  Since other headers may define MIN and
//  MAX unconditionally, we need to include this one last.
# include <genmacro.h>






/***************************************************************************************************************************
  TCPSocket :: TCPSocket ()
****************************************************************************************************************************
  This is the TCPSockets' default constructor.  It obtains a socket handle and uses that socket to
  set the internal socket handle.  The rest of the objects member data is set to 0.

***************************************************************************************************************************/
TCPSocket :: TCPSocket () : IPSocket () , nMaxPendingConnections ( enMAX_PEND )
{

} /* TCPSocket :: TCPSocket () */



/***************************************************************************************************************************
  TCPSocket :: TCPSocket ( const TCPSocket & rctcpsocket )
****************************************************************************************************************************

  ARGUMENTS     :
    const IPSocket  & rctcpsocket
      A reference to the constant TCPSocket source object.


  DESCRIPTION   :
    This is the TCPSocket DEEP copy constructor.

***************************************************************************************************************************/
TCPSocket :: TCPSocket ( const TCPSocket & rctcpsocket )
  : IPSocket ( rctcpsocket ) , nMaxPendingConnections ( rctcpsocket.maxPending () )
{

} /* TCPSocket :: TCPSocket ( const TCPSocket & rctcpsocket ) */



/***************************************************************************************************************************
  TCPSocket :: TCPSocket ( const SockAddr & rcsokaddrLocal , const SockAddr & rcsokaddrRemot = SockAddr () )
****************************************************************************************************************************

  ARGUMENTS     :
    const SockAddr  & rcsokaddrLocal
      A reference to a constant SockAddr object representing the (local) IP Address and port with which to associate the
      object being constructed
      
    const SockAddr  & rcsokaddrRemot = SockAddr ()
      A reference to a constant SockAddr object representing the (remote) IP Address and port with which to associate the
      object being constructed.



  DESCRIPTION   :
    This creates an TCPSocket given both a local and remote (i.e., read and write) socket address AND OPENS the socket,
    setting the receivers' s32Socket member to the handle returned from the attempt to open that socket.

****************************************************************************************************************************/
TCPSocket :: TCPSocket ( const SockAddr & rcsokaddrLocal , const SockAddr & rcsokaddrRemot )
  : IPSocket ( rcsokaddrLocal , rcsokaddrRemot ) , nMaxPendingConnections ( enMAX_PEND )
{
  int nErr    = 0 ;
  int nSocket = :: socket ( AF_INET , SOCK_STREAM , 0 ) ;

  ESCAPE_BAD_TCPIP ( nSocket , "TCPIP Error on create stream socket" , nSocket ) ;

  setHandle ( nSocket ) ;

  //  We just open the socket handle with this constructor, so tell the object it is responsible for closing the handle
  setDestroy ( true ) ;

  //  Make the address reusable
  int nOn = 1 ;
  
  nErr = setSockOptions ( SO_REUSEADDR , (void *) & nOn , sizeof ( nOn ) ) ;

} /* TCPSocket :: TCPSocket ( const SockAddr & rcsokaddrLocal , const SockAddr & rcsokaddrRemot ) */



/***************************************************************************************************************************
  SINT32 TCPSocket :: accept ()
****************************************************************************************************************************

  RETURNS       :
    The socket descriptor obtained by the TCPIP API call "::accept"


  DESCRIPTION   :
    This calls the TCPIP API accept using the receivers' Socket Handle to accept a new connection request.

***************************************************************************************************************************/
SINT32 TCPSocket :: accept ()
{
  int         nRetVal   = 0                         ;
  size_t      stAddrLen = SockAddr :: addrSize  ()  ;
  sockaddr  * psockaddr = remotAddr ().sockAddr ()  ;


  memset ( psockaddr , 0 , stAddrLen ) ;



// For some reason, AIX expects this to be a size_t!
# ifdef __AIX__
  nRetVal = :: accept ( handle () , psockaddr , (size_t *) & stAddrLen ) ;
# else
  nRetVal = :: accept ( handle () , psockaddr , (int *) & stAddrLen ) ;
# endif /* # ifdef __AIX__*/

  ESCAPE_BAD_TCPIP ( nRetVal, "TCPIP Error on accept" , handle () ) ;

  //  Return the new socket handle
  return  nRetVal ;

} /* SINT32 TCPSocket :: accept () */



/***************************************************************************************************************************
  const char * TCPSocket :: className () const
****************************************************************************************************************************

  RETURNS       :
    A constant char pointer to the receivers' class name as an ASCIIZ string,
    i.e. "TCPSocket".


  DESCRIPTION   :
    This method return the class name of the receiver object as a const
    char *.  This information may be used as a primitive RTTI, or anything
    else you like.  Note that this IS a virtual function, and will early bind
    in the constructors and destructor.  Thus you can also use this method to
    determine in which constructorlevel you are, although I don't know why
    you would want to know that!

***************************************************************************************************************************/
const char * TCPSocket :: className () const
{
  return  "TCPSocket" ;

} /* const char * TCPSocket :: className () const */



/***************************************************************************************************************************
  int TCPSocket :: isStream () const
****************************************************************************************************************************
    
  RETURNS     :
    (int) TRUE (1)


  DESCRIPTION :
    Returns true if the receiver is a STREAM socket and false otherwise.

***************************************************************************************************************************/
int TCPSocket :: isStream () const
{
  return  TRUE ;

} /* int TCPSocket :: isStream () const */



/***************************************************************************************************************************
  void TCPSocket :: listen ()
****************************************************************************************************************************

  DESCRIPTION   :
    This calls the TCPIP API listen using the receivers' Socket Handle.

***************************************************************************************************************************/
void TCPSocket :: listen ()
{
  int   nRetVal = :: listen ( handle () , maxPending () ) ;

  ESCAPE_BAD_TCPIP ( nRetVal, "TCPIP Error on listen" , handle () ) ;

} /* void TCPSocket :: listen () */



/***************************************************************************************************************************
  int TCPSocket :: maxPending () const
****************************************************************************************************************************

  RETURNS       :
    The current number of pending TCP Connection requests that the receiver will accept during a 'listen'
    call.


  DESCRIPTION   :
    This method returns the number of TCP connection requests that the receiver will allow to pend while
    'listen' is called.  This number is changed via the method 'TCPSocket :: setMaxPending ( int )'.

***************************************************************************************************************************/
int TCPSocket :: maxPending () const
{
  return  nMaxPendingConnections ;

} /* int TCPSocket :: maxPending () const */



/***************************************************************************************************************************
  int TCPSocket :: protocol () const
****************************************************************************************************************************
    
  RETURNS     :
    (int) SOCK_STREAM


  DESCRIPTION :
    Returns the receivers' protocol type, in the case of a TCPIP object this
    is SOCK_STREAM.

***************************************************************************************************************************/
int TCPSocket :: protocol () const
{
  return  SOCK_STREAM ;

} /* int TCPSocket :: protocol () const */



/***************************************************************************************************************************
  int TCPSocket :: setMaxPending ( int nNumToPend )
****************************************************************************************************************************

  RETURNS       :
    The current number of pending TCP Connection requests that the receiver will accept during a 'listen'
    call.


  DESCRIPTION   :
    This method returns the number of TCP connection requests that the receiver will allow to pend while
    'listen' is called.  This number is changed via the method 'TCPSocket :: setMaxPending ( int )'.

***************************************************************************************************************************/
int TCPSocket :: setMaxPending ( int nNumToPend )
{
  return  nMaxPendingConnections = MIN ( SOMAXCONN , nNumToPend ) ;

} /* int TCPSocket :: setMaxPending ( int nNumToPend ) */



/************************************************   end of file tcpsock.cpp   *********************************************/


