/***************************************************************************************************************************
 
    FILE NAME :
      tcpobj.cpp
 
 
    DESCRIPTION :
      This module contains the methods used to implement public interface to the class TCPObject.  This is an
      abstract base class that encapsulates the TCPSocket class.
 
 
    METHODS DEFINED IN THIS MODULE :
 
      TCPObject :: TCPObject  ()                                       
      TCPObject :: TCPObject  ( const SockAddr  & rcsokaddrLocal , const SockAddr & rcsokaddrRemot = SockAddr () )
      TCPObject :: TCPObject  ( const TCPObject & rctcpobject     )                    
      TCPObject :: ~TCPObject ()                                                       

      const char      * TCPObject :: className      () const                           
      void              TCPObject :: close          ()                                 
      TCPObject       & TCPObject :: init           ( const TCPObject & rctcpobject )  
      SockAddr        & TCPObject :: tcpSocket      ()                                 
      const SockAddr  & TCPObject :: tcpSocket      () const                           
      TCPSocket       & TCPObject :: setTCPSocket   ( const TCPSocket & rctcpsocket )  
      short             TCPObject :: setWaitTime    ( short sWaitTimeSeconds =  -1  )
      short             TCPObject :: waitTime       ()  const


      TCPServer :: TCPServer  ( const SockAddr & rcsokaddrLocal )                      

      SINT32            TCPServer :: accept         ( TCPStream & rtcpstream )          
      const char      * TCPServer :: className      () const                            
      void              TCPServer :: listen         ()                                  


      TCPStream :: TCPStream ()                                                        
      TCPStream :: TCPStream ( const TCPObject & rctcpobject )                          


      const char      * TCPStream :: className      ()                                          const                            
      SINT32            TCPStream :: read           ( void          * pvBuff  , SINT32 s32Len ) const
      void              TCPStream :: readTimeOut    ( const SINT16    cs16NextRead            ) const
      SINT32            TCPStream :: write          ( const void    * pcvBuff , SINT32 s32Len ) const


    OPERATORS DEFINED IN THIS MODULE :

      TCPObject   & TCPObject :: operator =         ( const TCPObject & rctcpobject )    


    EXCEPTION CLASSES THROWN FROM THIS MODULE :
      TCPIPError


  Copyright 1997, Peter Garner
    This code may be used for any NON-MILITARY purpose.
 
***************************************************************************************************************************/
//  Standard C Headers
# ifdef __UNX__
  # include <unistd.h>
# else
  # include <os2port.h>
# endif


//  TCPIP C Headers
extern "C"
{
  # include <netdb.h>
  # include <sys/types.h>
  # include <sys/ioctl.h>
  # include <netinet/in.h>
}


//  Locally Defined C Headers
# include <strfcns.h>


//  Locally Defined C++ Headers
# include <tcperror.hpp>
# include <tcpobj.hpp>





/***************************************************************************************************************************
  TCPObject :: TCPObject ()
****************************************************************************************************************************

  DESCRIPTION :
    This is the default TCPObject constructor.  It creates a TCPObject whose address and handle are set to 0.

***************************************************************************************************************************/
TCPObject :: TCPObject ()
{
  //  Set the wait time to whatever the default is
  setWaitTime () ;

} /* TCPObject :: TCPObject () */



/***************************************************************************************************************************
  TCPObject :: TCPObject ( const SockAddr & rcsokaddrLocal , const SockAddr & rcsokaddrRemot = SockAddr () )
****************************************************************************************************************************

  ARGUMENTS   :
    const SockAddr  & rcsokaddrLocal
      A reference to a constant SockAddr object representing the (local) IP Address and port with which to associate the
      object being constructed
      
    const SockAddr  & rcsokaddrRemot = SockAddr ()
      A reference to a constant SockAddr object representing the (remote) IP Address and port with which to associate the
      object being constructed.


  DESCRIPTION :
    This creates an TCPObject given both a local and remote (i.e., read and write) socket address AND OPENS the socket,
    setting the receivers' s32Socket member to the handle returned from the attempt to open that socket.


***************************************************************************************************************************/
TCPObject :: TCPObject ( const SockAddr & rcsokaddrLocal , const SockAddr & rcsokaddrRemot )
  : tcpsocket ( rcsokaddrLocal , rcsokaddrRemot )
{
  //  Set the wait time to whatever the default is
  setWaitTime () ;

} /* TCPObject :: TCPObject ( const SockAddr & rcsokaddrLocal , const SockAddr & rcsokaddrRemot ) */



/***************************************************************************************************************************
  TCPObject :: TCPObject ( const TCPObject & rctcpobject )
****************************************************************************************************************************

  ARGUMENTS   :
    const TCPObject & rctcpobject
      A reference to the constant TCPObject that is to be the source for this DEEP copy constructor


  DESCRIPTION :
    This method is the TCPObject DEEP copy constructor.  This constructs a TCPObject object from a constant TCPObject.

***************************************************************************************************************************/
TCPObject :: TCPObject ( const TCPObject & rctcpobject )
  : tcpsocket ( rctcpobject.tcpSocket () ) , sWaitSeconds ( rctcpobject.waitTime () )
{

} /* TCPObject :: TCPObject ( const TCPObject & rctcpobject ) */



/***************************************************************************************************************************
  const char * TCPObject :: className () const
****************************************************************************************************************************

  DESCRIPTION :
    This method returns the class name "TCPObject" as a const char *.
  
***************************************************************************************************************************/
const char * TCPObject :: className () const
{
  return  "TCPObject" ;

} /* const char * TCPObject :: className () const */



/***************************************************************************************************************************
  TCPObject :: close ()
****************************************************************************************************************************

  RETURNS     :
    The integer result code from the close attempt.

  DESCRIPTION :
    This method CLOSES the TCPIP socket handle.

***************************************************************************************************************************/
int TCPObject :: close ()
{
  //  Attempt to close the socket handle.  Note that we are saving the result code and returning it in 2 steps
  //  so that we can more easily view the result code with a symbolic debugger.  (When the program has been
  //  debugged and then recompiled with optimizations turned on, the compiler will optimize these 2 steps into
  //  "return tcpSocket ().close () ;", giving us the best of both worlds!)
  int nRetVal = tcpSocket ().close () ;

  return  nRetVal ;

} /* int TCPObject :: close () */



/***************************************************************************************************************************
  TCPObject & TCPObject :: init ( const TCPObject & rctcpobject )
****************************************************************************************************************************

  ARGUMENTS   :
      A reference to the constant TCPObject that is to be the source for this DEEP copy initializer.


  DESCRIPTION :
    This is the TCPObject copy operator.  This intializes a TCPObject object as a DEEP copy from a constant TCPObject.

***************************************************************************************************************************/
TCPObject & TCPObject :: init ( const TCPObject & rctcpobject )
{
  tcpSocket ().init ( rctcpobject.tcpSocket () ) ;

  //  Set the wait time to that of the argument
  setWaitTime ( rctcpobject.waitTime () ) ;

  return  * this ;

} /* TCPObject & TCPObject :: init ( const TCPObject & rctcpobject ) */



/***************************************************************************************************************************
  TCPSocket & TCPObject :: tcpSocket ()
****************************************************************************************************************************

  RETURNS     :
    The TCPSocket reference that the receiver encapsulates internally

  DESCRIPTION :
    This method provides an implementation independent means for the other classes to access the internal TCPSocket that
    TCPObject encapsulates.
    (NON-CONSTANT OVERLOAD)

***************************************************************************************************************************/
TCPSocket & TCPObject :: tcpSocket ()
{
  //  Return a reference to the receivers' internal TCPSocket object
  return  tcpsocket ;

} /* TCPSocket & TCPObject :: tcpSocket () */



/***************************************************************************************************************************
  TCPSocket & TCPObject :: tcpSocket () const
****************************************************************************************************************************

  RETURNS     :
    The TCPSocket reference that the receiver encapsulates internally

  DESCRIPTION :
    This method provides an implementation independent means for the other classes to access the internal TCPSocket that
    TCPObject encapsulates.
    (CONSTANT OVERLOAD)

***************************************************************************************************************************/
const TCPSocket & TCPObject :: tcpSocket () const
{
  //  Return a reference to the receivers' internal TCPSocket object
  return  tcpsocket ;

} /* const TCPSocket & TCPObject :: tcpSocket () const */



/***************************************************************************************************************************
  TCPSocket & TCPObject :: setTCPSocket ( const TCPSocket & rctcpsocket )
****************************************************************************************************************************

  ARGUMENTS   :
    const TCPSocket & rctcpsocket
      A reference to the CONSTANT TCPSocket object that we want to use to initialize the receiver.

  RETURNS     :
    The TCPSocket reference that the receiver encapsulates internally

  DESCRIPTION :
    This method provides an implementation independent means for the other classes to initialize the internal TCPSocket that
    TCPObject encapsulates.  In short, this method lets the receivers' internal TCPSocket member become equivalent to that
    in the input argument 'rtcpsocket'.

***************************************************************************************************************************/
TCPSocket & TCPObject :: setTCPSocket ( const TCPSocket & rctcpsocket )
{
  tcpSocket ().init ( rctcpsocket ) ;

  return  tcpSocket () ;

} /* TCPSocket & TCPObject :: setTCPSocket ( const TCPSocket & rctcpsocket ) */



/***************************************************************************************************************************
  short TCPObject :: setWaitTime ( short sWaitTimeSeconds ) 
****************************************************************************************************************************

  ARGUMENTS   :
    short sWaitTimeSeconds
      The number of seconds


  RETURNS     :
      The value of input argument 'sWaitTimeSeconds'


  DESCRIPTION :
    This method sets the time that the receiver will wait (in seconds) for I/O.  If this value is negative, the receiver
    will wait forever!
  
***************************************************************************************************************************/
short TCPObject :: setWaitTime ( short sWaitTimeSeconds ) 
{
  return  sWaitSeconds  = sWaitTimeSeconds  ;

} /* short TCPObject :: setWaitTime ( short sWaitTimeSeconds )  */



/***************************************************************************************************************************
  short TCPObject :: waitTime () const
****************************************************************************************************************************

  DESCRIPTION :
    This method returns the time that the receiver will wait (in seconds) for I/O.  If this value is negative, the receiver
    will wait forever!
  
***************************************************************************************************************************/
short TCPObject :: waitTime () const
{
  return  sWaitSeconds ;

} /* short TCPObject :: waitTime () const */




/*********************************************    TCPObject OPERATORS      ************************************************/



/***************************************************************************************************************************
  TCPObject & TCPObject :: operator = ( const TCPObject & rctcpobject )
****************************************************************************************************************************

  ARGUMENTS   :
      A reference to the constant TCPObject that is to be the source for this DEEP copy initializer.


  DESCRIPTION :
    This is the TCPObject copy initializer.  This intializes a TCPObject object as a DEEP copy from a TCPObject.

***************************************************************************************************************************/
TCPObject & TCPObject :: operator = ( const TCPObject & rctcpobject )
{
  return  init ( rctcpobject ) ;

} /* TCPObject & TCPObject :: operator = ( const TCPObject & rctcpobject ) */





/*********************************************    TCPServer METHODS        ************************************************/



/***************************************************************************************************************************
  TCPServer :: TCPServer ( const SockAddr & rcsokaddrLocal )
****************************************************************************************************************************

  ARGUMENTS   :
    const SockAddr  & rcsokaddrLocal
      The local address to which to bind the TCPServer object being constructed.


  DESCRIPTION :
    This method constructs a TCPServer object from a SockAddr reference.  This method opens the socket and binds
    the socket to the address specified in 'rcsokaddrLocal'.

***************************************************************************************************************************/
TCPServer :: TCPServer ( const SockAddr & rcsokaddrLocal ) : TCPObject ( rcsokaddrLocal )
{
  tcpSocket ().bind () ;

} /* TCPServer :: TCPServer ( const SockAddr & rcsokaddrLocal ) */



/***************************************************************************************************************************
  SINT32 TCPServer :: accept ( TCPStream & tcpstream )
****************************************************************************************************************************

  ARGUMENTS   :
    TCPStream   & tcpstream
      A reference to a TCPStream object.  Upon successful completion of this method the contents of this argument will be
      overwritten.  This arguments Local Address will be set to the local address of this methods' receiver and the remote
      address will be set to the address of the client with whom the connection has been made.  This arguments' socket
      handle will be the handle of the CONNECTED socket, NOT the socket handle of the receiver (server)!


  RETURNS     :
    This method returns the newly created handle of the CONNECTED socket as a signed 32 bit integer.


  DESCRIPTION :
    This method tells the receiver to accept a connection as requested.  THIS METHOD ASSUMES AS A PRECONDITION THAT THE
     SOCKET IS BLOCKED AND IN LISTEN MODE!  A TCPIPError will be thrown if these condition are not true!  A side effect of
     this method is that the Remote Address of BOTH the receiver and the TCPStream arguments' internal TCPSocket members
     will be set to the address of the client with whom the connection has been made.
  
***************************************************************************************************************************/
SINT32 TCPServer :: accept ( TCPStream & tcpstream )
{
  //  Attempt to accept a connection to a client.  Again this socket should be blocked, and any errors will result in a
  //  TCPIPError being thrown!
  SINT32  s32ChildHandle = tcpSocket ().accept  () ;

  //  If we get this far, we have successfully created a new connected socket (the value being stored in 's32ChildHandle'),
  //  and the receivers' TCPSocket member will contain both the local and remote addresses of the connection.  So we
  //  initialize the argument TCPStream reference with these address values and the parent (server) socket handle.
  tcpstream.init ( * this ) ;

  //  Finally set the TCPStream argument handle to that of the newly created CONNECTED socket handle, this completes
  //  the initialization of the TCPStream argument, it is now ready for use in a stream conversation.
  tcpstream.tcpSocket ().setHandle ( s32ChildHandle ) ;

  //  Finally return the new connected child handle!
  return  s32ChildHandle ;

} /* SINT32 TCPServer :: accept ( TCPStream & tcpstream ) */



/***************************************************************************************************************************
  const char * TCPServer :: className () const
****************************************************************************************************************************

  DESCRIPTION :
    This method returns the class name "TCPServer" as a const char *.
  
***************************************************************************************************************************/
const char * TCPServer :: className () const
{
  return  "TCPServer" ;

} /* const char * TCPServer :: className () const */



/***************************************************************************************************************************
  void TCPServer :: listen ()
****************************************************************************************************************************

  DESCRIPTION :
    This method tells the receiver to make itself available to listen for connections.  This method is analogous to the
     TCPIP "C" API 'listen', however this method returns no error code but rather throws a TCPIPError if it fails!
  
***************************************************************************************************************************/
void TCPServer :: listen ()
{
  tcpSocket ().listen  () ;

} /* void TCPServer :: listen () */



/***************************************************************************************************************************
  TCPStream :: TCPStream ()
****************************************************************************************************************************

  DESCRIPTION :
    This is the default TCPStream constructor.  It creates a TCPStream whose address and handle are set to 0.

***************************************************************************************************************************/
TCPStream :: TCPStream () 
{

} /* TCPStream :: TCPStream () */



/***************************************************************************************************************************
  TCPStream :: TCPStream ( const SockAddr & rcsokaddrLocal , const SockAddr & rcsokaddrRemot = SockAddr () )
****************************************************************************************************************************

  ARGUMENTS   :
    const SockAddr  & rcsokaddrLocal
      A reference to a constant SockAddr object representing the (local) IP Address and port with which to associate the
      object being constructed
      
    const SockAddr  & rcsokaddrRemot = SockAddr ()
      A reference to a constant SockAddr object representing the (remote) IP Address and port with which to associate the
      object being constructed.


  DESCRIPTION :
    This creates an TCPStream given both a local and remote (i.e., read and write) socket address AND OPENS the socket,
    setting the receivers' s32Socket member to the handle returned from the attempt to open that socket.


***************************************************************************************************************************/
TCPStream :: TCPStream ( const SockAddr & rcsokaddrLocal , const SockAddr & rcsokaddrRemot )
  : TCPObject ( rcsokaddrLocal , rcsokaddrRemot )
{

} /* TCPStream :: TCPStream ( const SockAddr & rcsokaddrLocal , const SockAddr & rcsokaddrRemot ) */



/***************************************************************************************************************************
  TCPStream :: TCPStream ( const TCPObject & rctcpobject )
****************************************************************************************************************************

  ARGUMENTS   :
    const TCPObject & rctcpobject
    A reference to the constant TCPObject that is to be the source for this DEEP copy constructor


  DESCRIPTION :
    This is the TCPStream copy constructor.  It creates a TCPStream object from a constant reference to a TCPObject.
    This does a deep copy on the argument 'rctcpobject', thus copying the TCPSocket member data.  Note that the sWaitTime
    member is set to the default.

***************************************************************************************************************************/
TCPStream :: TCPStream ( const TCPObject & rctcpobject ) : TCPObject ( rctcpobject )
{

} /* TCPStream :: TCPStream ( const TCPObject & rctcpobject ) */



/***************************************************************************************************************************
  const char * TCPStream :: className () const
****************************************************************************************************************************

  DESCRIPTION :
    This method returns the class name "TCPStream" as a const char *.
  
***************************************************************************************************************************/
const char * TCPStream :: className () const
{
  return  "TCPStream" ;

} /* const char * TCPStream :: className () const */



/***************************************************************************************************************************
  SINT32 TCPStream :: read ( void * pvBuff , SINT32 s32Len ) const
****************************************************************************************************************************
  ARGUMENTS     :
    char    * pcBuff
      A pointer to the character buffer into which to read the message.
      
    SINT32    s32Len
      The amount of data to read


    RETURNS       :
      The number of bytes of data actually read.


  DESCRIPTION :
    This method reads a block of data from a socket.  The purpose of this method is to read chunks of data larger than the
    TCPIP 32K limit.  This method ASSUMES that socket is in blocking mode and has been initialized, connected, etc.  If
    these preconditions are NOT true, a TCPIPError will be thrown!
  
***************************************************************************************************************************/
SINT32 TCPStream :: read ( void * pvBuff , SINT32 s32Len ) const
{
  //  Initialized local data
  char    * pcBuff        = (char *) pvBuff ;   //  Cast the void parameter to a char * so we can do pointer arithmetic
  SINT16    s16LastRead   = 0               ;   //  Number of Bytes read in last low level read operation
  SINT32    s32Left       = s32Len          ;   //  Number of Bytes left to read to fulfill read request
  SINT16    s16NextRead   = 0               ;   //  Number of Bytes to read in the next low level read operation
  SINT32    s32TotalRead  = 0               ;   //  Total number of Bytes read so far
  UINT16    u16BytesAvail = 0               ;   //  Number of Bytes currently available to read on the socket

  while ( 0 < s32Left )
  {
    //  Read the LESSER of the two quantities, Maximum Packet Size, and Amount Left to Read from the socket handle.
    //  Remember cs16GblMAX_PACKET is global constant defined in "tcpobj.h" as our packet size.
    s16NextRead   = MIN ( cs16GblMAX_PACKET , s32Left ) ;

    //  Wait for usMaxWait seconds for s16NextRead bytes to become available for reading.  If at least usMaxWait seconds
    //  elapses before 16NextRead bytes become available for reading, readTimeOut will throw a TCPIPError with an Error Id
    //  of EAGAIN.
    readTimeOut   ( s16NextRead ) ;
    
    //  Having made it out of the last while loop without throwing an exception, we can assume that we have at least
    //  's16NextRead' bytes to read.  Remember that the error checking is done by 'TCPSocket :: read ( char * , SINT16 )'.
    s16LastRead   = tcpSocket ().read ( pcBuff , s16NextRead ) ;

    //  Add the number of bytes just read to our running total of ALL bytes read
    s32TotalRead  += s16LastRead ;

    //  Subtract the number of bytes just read from our number of bytes left to read.
    s32Left       -= s16LastRead ;

    //  And finally increment the buffer!
    pcBuff        += s16LastRead ;
  
  } /* endwhile  */

  return  s32TotalRead ;

} /* SINT32 TCPStream :: read ( void * pvBuff , SINT32 s32Len ) const */



/***************************************************************************************************************************
  void TCPStream :: readTimeOut ( const SINT16 cs16NextRead ) const
****************************************************************************************************************************

  ARGUMENTS     :
    const SINT16  cs16NextRead
      The number of bytes we are waiting to read.  (The number of bytes we are waiting to become available)


  DESCRIPTION :
    This method waits for at least 'waitTime ()' for 's16NextRead' bytes to become available on the socket for reading.
    Note that we could have used the BSD select function for this method instead of sleeping inside of a while loop!  The 
    BSD select would allow us to fine tune the wait resolution to microseconds. However, we do not need this fine a level of
    timing resolution, and not all operating systems support the select method.  Thus for this to remain portable, we used a
    while loop.  In addition the select routine will tell us WHETHER data is available, it WILL NOT tell us how much data is
    available.  Thus we would STILL need some kind of a while loop that called 'dataAvailable' to test how much data was on
    the socket.  Thus using select would not simplify our code at all.


  EXCEPTIONS  :
    If at least usSeconds have passed, and the specified number of bytes has NOT become available, a TCPIPError is thrown
    with an Error Id of EAGAIN.

***************************************************************************************************************************/
void TCPStream :: readTimeOut ( const SINT16 cs16NextRead ) const
{
  //  Find out how long we should wait for the data to become available
  short   sWaitSeconds  = waitTime () ;

  //  If the value is negative, we are supposed to wait forever, so just return as if nothing happened!
  if ( 0 > sWaitSeconds )
  {
    return  ;

  } /* endif */

  //  Find out how many bytes are available to read from the socket.
  UINT16  u16BytesAvail = tcpSocket ().dataAvailable () ;

  //  Check and see if we have enough data available to read as much as we expect.
  while ( u16BytesAvail < cs16NextRead )
  {
    //  If we DO NOT have as much data as we expect, surrender the processor, (sleep for 1 second) and wait for more data.
    sleep ( 1 ) ;

    //  Find out how many bytes are available to read from the socket after sleeping.
    u16BytesAvail = tcpSocket ().dataAvailable () ;

    //  Check to see if usSeconds has reached 0 in this (inner) loop yet.  If it has we want to throw a TCPIPError with a
    //  an Error Id of ETIME (the system error code that says TIMER EXPIRED).
    if ( 0 == sWaitSeconds )
    {
      TCPError  tcperror ( ETIME, "MAXIMUM WAIT TIME REACHED FOR DATA, READ TIMED OUT" , tcpSocket ().handle () ) ;

      THROW ( tcperror ) ;

    } /* endif */

    //  Otherwise decrement our local variable sWaitSeconds by one to show that one second (roughly) has elapsed.
    sWaitSeconds-- ;

  } /* endwhile  */
  
} /* void TCPStream :: readTimeOut ( const SINT16 cs16NextRead ) const */



/***************************************************************************************************************************
  SINT32 TCPStream :: write ( const void * pcvBuff , SINT32 s32Len ) const
****************************************************************************************************************************
  ARGUMENTS     :
    cosnt char  * pcBuff
      A pointer to the character buffer from which to write the message.
      
    SINT32        s32Len
      The amount of data to write


    RETURNS       :
      The number of bytes of data actually written.


  DESCRIPTION :
    This method writes a block of data from a socket.  The purpose of this method is to write chunks of data larger than
    the TCPIP 32K limit.  This method ASSUMES that socket is in blocking mode and has been initialized, connected, etc.  If
    these preconditions are NOT true, a TCPIPError will be thrown!
  
***************************************************************************************************************************/
SINT32 TCPStream :: write ( const void * pcvBuff , SINT32 s32Len ) const
{
  //  Initialized local data
  const char  * pccBuff       = (const char *) pcvBuff  ;   //  Cast to const char * so we can do pointer arithmetic
  SINT16        s16LastWrite  = 0                       ;
  SINT32        s32TotalWrite = 0                       ;
  SINT32        s32Left       = s32Len                  ;

  while ( 0 < s32Left )
  {
    //  Write the LESSER of the two quantities, Maximum Packet Size, and Amount Left to Write from the socket handle.
    //  Remember cs16GblMAX_PACKET is global constant defined in "tcpobj.h" as our packet size.  Also remember that the
    //  error checking is done by 'TCPSocket :: write ( char * , SINT16 )'.
    s16LastWrite  = tcpSocket ().write ( pccBuff , MIN ( cs16GblMAX_PACKET , s32Left ) ) ;

    //  Add the number of bytes just written to our running total of ALL bytes written
    s32TotalWrite += s16LastWrite;

    //  Subtract the number of bytes just written from our number of bytes left to write.
    s32Left       -= s16LastWrite;

    //  And finally increment the buffer!
    pccBuff       += s16LastWrite;
  
  } /* endwhile  */

  return  s32TotalWrite ;

} /* SINT32 TCPStream :: write ( const void * pcvBuff , SINT32 s32Len ) const */



/*********************************************    end of file tcpobj.cpp   ************************************************/


