/***************************************************************************************************************************

    FILE NAME   :
      sockaddr.cpp


    APPLICATION :
      IMPLEMENTATION OF THE PUBLIC INTERFACE FOR THE SOCKADDR CLASS.


    DESCRIPTION :
      This module contains the methods used to implement the interface to the class SockAddr.  This is a
      class that is abstract the TCPIP sockaddr and sockaddr_in structures.


    METHODS DEFINED IN THIS MODULE :

      SockAddr :: SockAddr ( ushort usPort = 0 , const char * cszHostName = 0  )                        
      SockAddr :: SockAddr ( const SockAddr & rcsokaddr                 )                        

      const char        * SockAddr :: className     () const                                     
      SINT32              SockAddr :: host          ( const char * cszHost )
      const sockaddr_in * SockAddr :: iNetAddr      () const                                     
      sockaddr_in       * SockAddr :: iNetAddr      ()
      const char        * SockAddr :: localHostName ( char * szHostName   , size_t stMaxLen               )
      ushort              SockAddr :: port          () const
      ushort              SockAddr :: servicePort   ( const char * cszService , const char * cszProtocol  )
      sockaddr          * SockAddr :: sockAddr      ()
      void                SockAddr :: init          ( ushort usPort = 0 , const char * cszHostName = 0    )
      void                SockAddr :: init          ( const SockAddr & rcsokaddr                          )
      SINT32              SockAddr :: setAddr       ( SINT32 s32INetAddr                                  )
      SockAddr          & SockAddr :: operator =    ( const SockAddr & rcsokaddr                          )


    EXCEPTION CLASSES THROWN FROM THIS MODULE :
      TCPIPError


  Copyright 1997, Peter Garner
    This code may be used for any NON-MILITARY purpose.
 
***************************************************************************************************************************/
//  System Defined C Headers
# ifdef __UNX__
  # include <arpa/inet.h>
# endif


//  TCPIP C Headers
extern "C"
{
  # include <netdb.h>
  # include <sys/types.h>
  # include <sys/ioctl.h>

  # ifdef __OS2__
    # include <utils.h>
  # endif  /* # ifdef __OS2__ */

}


//  Locally Defined C Headers
# include <strfcns.h>


//  Locally Defined C++ Headers
# include <sockaddr.hpp>
# include <tcperror.hpp>


//  Locally Defined C Headers
//  We include <genmacro.h> last, since it has conditional defines of MIN and MAX.  Since other headers may define MIN and
//  MAX unconditionally, we need to include this one last.
# include <genmacro.h>



const int   cnGblMaxHostName  = 255 ;



/***************************************************************************************************************************
  SockAddr :: SockAddr ( ushort usPort , const char * cszHostName )
****************************************************************************************************************************

  ARGUMENTS     :
    ushort    usPort
      The port number which the new object will store.

    char    * cszHostName
      The IP Address with which to associate the receiver.  This may be in the form "nnn.nnn.nnn.nnn", or as
      a host name.  In either case, it will be converted into a network ordered binary int, and the internal
      IP Address of the receiver will be set to that BINARY IP Address.


  DESCRIPTION   :
    This creates an SockAddr Object given both a port and an IP Address.

***************************************************************************************************************************/
SockAddr :: SockAddr ( ushort usPort , const char * cszHostName )
{
  init ( usPort , cszHostName ) ;  

} /* SockAddr :: SockAddr ( ushort usPort , const char * cszHostName ) */



/***************************************************************************************************************************
  SockAddr :: SockAddr ( const SockAddr & rcsokaddr )
****************************************************************************************************************************

  ARGUMENTS     :
    const SockAddr & rcsokaddr
      The SockAddr structure used as the source for this copy constructor.

  DESCRIPTION   :
    This is the SockAddr copy constructor.  It does a DEEP copy on the source argument 'rcsokaddr'.

***************************************************************************************************************************/
SockAddr :: SockAddr ( const SockAddr & rcsokaddr )
{
  init ( rcsokaddr ) ;

} /* SockAddr :: SockAddr ( const SockAddr & rcsokaddr ) */



/***************************************************************************************************************************
  const char * SockAddr :: className () const
****************************************************************************************************************************

  RETURNS       :
    (const char *) "SockAddr"


  DESCRIPTION   :
    This method returns the class name "SockAddr" as a const char *.

***************************************************************************************************************************/
const char * SockAddr :: className () const
{
  return  "SockAddr" ;

} /* const char * SockAddr :: className () const */



/***************************************************************************************************************************
  SINT32 SockAddr :: host ( const char * cszHost )
****************************************************************************************************************************

  ARGUMENTS     :
    const char  * cszHost
      The IP Address we want to convert.  This may be in the form "nnn.nnn.nnn.nnn", or as a host name.  In either case, it
      will be converted into a network ordered binary int, and the internal IP Address of the receiver will be set to that
      BINARY IP Address.

      If the string is 0 or of length 0, 'const char * SockAddr :: localHostName ( char * szHostName , size_t stMaxLen )' is
      called to get the current (local machine) host name.


  RETURNS       :
    The IP Address in network ordered binary form.


  DESCRIPTION   :
    If 'cszHost' is a string containing a host name, this function operates similiarly to the BSD gethostbyname, and
    converts that host name to a network ordered, binary IP Address via a call to to "::gethostbyname".  If the string is in
    the form "nnn.nnn.nnn.nnn", it is converted to a network ordered, binary IP Address via a call to "::inet_addr".  In
    either case, it sets the receivers internal IP Address to that corresponding to 'cszHost' in network ordered binary form
    and returns that IP address.

***************************************************************************************************************************/
SINT32 SockAddr :: host ( const char * cszHost )
{
  char        szLocalHost [cnGblMaxHostName + 1]  ; //  Storage for a local host name, in case the argument 'cszHost' is 0.
  int         nRetVal                             ;
  SINT32      s32AddrTemp                         ;


  //  If we were given a null pointer or an empty string for the host name, get the local (machine) host name
  if ( ! strIsValid ( cszHost ) )
  {
    cszHost = localHostName ( szLocalHost , cnGblMaxHostName ) ;
  
  } /* endif */

  //  If cszHost is an INTERNET address of the form "nnn.nnn.nnn.nnn"
  if ( isINAddress ( cszHost ) )
  {
    //  Then convert it to a network ordered, binary, INTERNET address
    s32AddrTemp = :: inet_addr ( cszHost ) ;    

  } /* endif */
  else
  {
    //  Get the host name information from the hosts file
    hostent   * phostent = :: gethostbyname ( cszHost ) ;

    //  If the host name could not be found, throw a TCPIP exception
    if ( 0 == phostent )
    {
      THROW_TCPIP_ERROR ( "TCPIP Error on gethostbyname" , -1 ) ;

    } /* endif */

    //  If it COULD be found, then set the internet address found in the structure!
    struct  in_addr   * pin_addr = (struct in_addr *) * phostent -> h_addr_list ;

    # ifdef __DOS__
      s32AddrTemp = pin_addr -> S_un.S_addr ;

    # else
      s32AddrTemp = pin_addr -> s_addr      ;

    # endif  /* # ifdef __DOS__ */

  } /* endelse */

  //  Set the receivers' internal INTERNET address to the value we have obtained
  setAddr ( s32AddrTemp ) ;

  //  And return the INTERNET IP address in network ordered, binary form.
  return  s32AddrTemp ;

} /* SINT32 SockAddr :: host ( const char * cszHost ) */



/***************************************************************************************************************************
  const sockaddr_in * SockAddr :: iNetAddr () const
****************************************************************************************************************************

  DESCRIPTION   :
    This method returns a pointer to the receivers internal C socket address structure as a pointer to a
    sockaddr_in structure.  (constant overload)

***************************************************************************************************************************/
const sockaddr_in * SockAddr :: iNetAddr () const
{
  return  & saddrUnion.saint  ;

} /* const sockaddr_in * SockAddr :: iNetAddr () const */



/***************************************************************************************************************************
  sockaddr_in * SockAddr :: iNetAddr ()
****************************************************************************************************************************

  DESCRIPTION   :
    This method returns a pointer to the receivers internal C socket address structure as a pointer to a
    sockaddr_in structure.  (non-constant overload)

****************************************************************************************************************************/
sockaddr_in * SockAddr :: iNetAddr ()
{
  return  & saddrUnion.saint  ;

} /* sockaddr_in * SockAddr :: iNetAddr () */



/***************************************************************************************************************************
  sockaddr * SockAddr :: sockAddr ()
****************************************************************************************************************************

  RETURNS       :
    A pointer to the receivers internal C socket address.


  DESCRIPTION   :
    This method returns a pointer to the receivers internal socket address structure as a pointer to a
    sockaddr structure.

***************************************************************************************************************************/
sockaddr * SockAddr :: sockAddr ()
{
  return  & saddrUnion.skadr ;

} /* sockaddr * SockAddr :: sockAddr () */



/***************************************************************************************************************************
  ushort SockAddr :: port () const
****************************************************************************************************************************

  DESCRIPTION   :
    This method returns the receivers' associated TCP Port Number.

***************************************************************************************************************************/
ushort SockAddr :: port () const
{
  return  iNetAddr () -> sin_port ;

} /* ushort SockAddr :: port () const */



/***************************************************************************************************************************
  void SockAddr :: init ( ushort usPort , const char * cszHostName )
****************************************************************************************************************************

  ARGUMENTS     :
    ushort    usPort
      The port number with which to initialize (associate) the receiver.

    char    * cszHostName
      The IP Address with which to associate the receiver.  This may be in the form "nnn.nnn.nnn.nnn", or as
      a host name.  In either case, it will be converted into a network ordered binary int, and the internal
      IP Address of the receiver will be set to that BINARY IP Address.

  DESCRIPTION   :
    This initializes the receiver' so that it is associated with the port number and host name passed in
    the arguments 'usPort' and 'cszHostName'.

***************************************************************************************************************************/
void SockAddr :: init ( ushort usPort , const char * cszHostName )
{
  //  Initialize the internal sockaddr_in structure to be all 0.
  memset ( (void *) sockAddr () , 0 , sizeof ( sockaddr_in ) ) ;

  //  Set the Family to AF_INET
  iNetAddr   () -> sin_family  = AF_INET ;
  //  Set the port number to the input port, using htons to portably do any needed byte swapping
  iNetAddr   () -> sin_port    = htons ( usPort ) ;

  //  If the string 'cszHostName' is not null, see if we can find it in the hosts file.
  if ( strIsValid ( cszHostName ) )
  {
    //  If the string IS valid, call host to associate the receiver with the IP Address found in
    //  TCPIP\ETC\HOSTS.
    //  Note this will throw a TCPIP exception if cszHostName is not found in the hosts file!
    host ( cszHostName ) ;    

  } /* endif */

} /* void SockAddr :: init ( ushort usPort , const char * cszHostName ) */



/***************************************************************************************************************************
  void SockAddr :: init ( const SockAddr & rcsokaddr )
****************************************************************************************************************************

  ARGUMENTS     :
    const SockAddr & rcsokaddr
      The SockAddr structure used as the source for this initializer.

  DESCRIPTION   :
    This is the SockAddr copy initializer.  It does a DEEP copy on the source argument 'rcsokaddr'.

***************************************************************************************************************************/
void SockAddr :: init ( const SockAddr & rcsokaddr )
{
  memcpy ( (void *) sockAddr () , rcsokaddr.iNetAddr () , addrSize () ) ;

} /* void SockAddr :: init ( const SockAddr & rcsokaddr ) */



/***************************************************************************************************************************
  SINT32 SockAddr :: setAddr ( SINT32 s32NewAddr )
****************************************************************************************************************************

  ARGUMENTS     :
    SINT32  s32NewAddr
      The IP address with which to associate the receiver as a 32 bit network ordered integer.


  RETURNS       :
    The input argument 's32NewAddr'.


  DESCRIPTION   :
    This method associates the receiver with the IP address in 's32NewAddr'.  This method uses some defines
    to accomplish this portably, as different TCPIP implementations define the in_addr structure differently!

***************************************************************************************************************************/
SINT32 SockAddr :: setAddr ( SINT32 s32NewAddr )
{
  in_addr     inaddrTemp  ;

 # ifdef __DOS__
  inaddrTemp.S_un.S_addr = s32NewAddr ;

 # else
  inaddrTemp.s_addr = (unsigned long) s32NewAddr ;

 # endif  /* # ifdef __DOS__ */

  iNetAddr () -> sin_addr = inaddrTemp ;

  return  s32NewAddr ;

} /* SINT32 SockAddr :: setAddr ( SINT32 s32NewAddr ) */



/****************************************   SOCKADDR STATIC METHODS             *******************************************/



/***************************************************************************************************************************
  size_t SockAddr :: addrSize ()
****************************************************************************************************************************

  RETURNS       :
    A reference to the receiver.


  DESCRIPTION   :
    This method returns the size in bytes of the receivers' internal sockaddr structure.

***************************************************************************************************************************/
size_t SockAddr :: addrSize ()
{
  return  sizeof ( struct sockaddr_in ) ;

} /* size_t SockAddr :: addrSize () */



/***************************************************************************************************************************
  int SockAddr :: isINAddress ( const char * pccNewAddr )
****************************************************************************************************************************

  ARGUMENTS     :
    const char  * pccNewAddr
      The string we wish to check to see if is an INTERNET address of the form "nnn.nnn.nnn.nnn".


  RETURNS       :
    TRUE    (1) if the argument 'pccNewAddr' is an INTERNET address of the form "nnn.nnn.nnn.nnn".
    FALSE   (0) otherwise.

  DESCRIPTION   :
    This static method checks if the input argument 'pccNewAddr' is an INTERNET address of the form 
    "nnn.nnn.nnn.nnn".

***************************************************************************************************************************/
int SockAddr :: isINAddress ( const char * pccNewAddr )
{
  size_t  stLen   = strlen ( pccNewAddr ) ;

  //  The MAXIMUM length of an internet address ("nnn.nnn.nnn.nnn") is 15, and the MINIMUM length is 4 ("nnnn")
  //  so if we are not in the range 4, 15, return false, as this cannot be an internet address.
  if ( !INRANGE ( stLen , 4 , 15 ) )
  {
    return  0 ;

  } /* endif */

  //  Check to see if there are any characters in the string NOT in the set "1234567890."
  if ( strspn ( pccNewAddr , "1234567890." ) < stLen )
  {
    return  0 ;

  } /* endif */

  //  If there are MORE than 3 '.' characters in the string it CANNOT be an INTERNET address either!
  if ( 3 < memCount ( pccNewAddr , '.' , stLen ) )
  {
    return  0 ;

  } /* endif */

  return  1 ;

} /* int SockAddr :: isINAddress ( const char * pccNewAddr ) */



/***************************************************************************************************************************
  const char * SockAddr :: localHostName ( char * szHostName , size_t stMaxLen )
****************************************************************************************************************************

  ARGUMENTS     :
    char    * szHostName
      The address of the character array into which the host name of the current processor will be copied
      
    size_t    stMaxLen
      The length of the array provided in the argument 'szHostName'.  At most 'stMaxLen' characters will be copied into the
      array 'szHostName'.
    

  RETURNS       :
    The argument address 'char * szHostName' as a constant char *.
    
    
  DESCRIPTION   :
    This static method looks the name for the current host and copies the first 'stMaxLen' bytes of that name into the array
    pointed to by the argument 'szHostName'.  If this method is successfull, the argument 'szHostName' will be null
    terminated, even if the length of the host name is greater than 'stMaxLen'.  This is eally just a wrapper for the "C"
    API gethostname ( char * , size_t ).  This method extends that API by throwing a TCPIPError if anything goes wrong.
    
***************************************************************************************************************************/
const char * SockAddr :: localHostName ( char * szHostName , size_t stMaxLen )
{
  int   nRetVal = gethostname ( szHostName , stMaxLen ) ;

  ESCAPE_BAD_TCPIP ( nRetVal, "TCPIP Error on gethostname" , -1 ) ;

  return  szHostName ;

} /* const char * SockAddr :: localHostName ( char * szHostName , size_t stMaxLen ) */



/***************************************************************************************************************************
  ushort SockAddr :: servicePort ( const char * cszService , const char * cszProtocol = 0 )
****************************************************************************************************************************

  ARGUMENTS     :
    const char  * cszService
      The name of the TCPIP service whose port number we wish to find
      
    const char  * cszProtocol = 0
      The protocol (usually TCP or UDP) of the service.  If this is 0, the first matching port number will be returned
      irregardless of the associated protocol.
    

  RETURNS       :
    The port number associated with argument service and protocol as found in the file "/etc/services" in HOST byte
    order
    
    
  DESCRIPTION   :
    This static method looks up the port number of a service given its' service name and protocol.  This static method is
    really just a wrapper for the "C" API getservbyname ( char * , char * ).  This method constricts the API by just
    returning the port number and extends that API by throwing a TCPIPError if anything goes wrong.
    
***************************************************************************************************************************/
ushort SockAddr :: servicePort ( const char * cszService , const char * cszProtocol )
{
  //  Declare a constant for the size of the argument strings, 128 should be enough!
  const size_t  cstMaxLen = 127 ;

  //  Declare some local objects so that we do not have to cast these arguments to non-constants
  char          szServiceName   [cstMaxLen + 1] ;
  char        * szProtocol                      ;
  char          szProtocolType  [cstMaxLen + 1] ;
  servent     * pservent                        ;
  ushort        usPort                          ;

  //  Copy the constant arguments into the non constant local variables
  strTermCpy    ( szServiceName   , cszService  , cstMaxLen ) ;
  strTermCpy    ( szProtocolType  , cszProtocol , cstMaxLen ) ;
  
  //  Check to see if service name was 0 or empty
  if ( ! strIsValid ( cszService ) )
  {
    //  Set the GLOBAL VARIABLE errno to Invalid Argument
    errno = EINVAL ;

    //  Throw a TCPIP error
    THROW_TCPIP_ERROR ( "NULL Service Name passed to SockAddr :: servicePort ( const char * , const char * )" , -1 ) ;

  } /* endif */

  //  Check to see if protocol was 0 or empty
  szProtocol  = strIsValid ( cszProtocol ) ? szProtocolType : 0 ;

  //  THIS IS NOT THREAD SAFE AT THIS POINT!
  // Call the C "API" to lookup in the service file
  pservent = getservbyname ( szServiceName , szProtocol ) ;

  //  If we failed, throw a TCPIPError
  if ( 0 == pservent )
  {
    //  Create a string for the secondary error message
    String  strgError  = String ( "getservbyname could not resolve the service : " ) + String ( cszService ) ;

    if ( szProtocol )
    {
      strgError += String ( " / " ) + String ( szProtocol ) ;

    } /* endif */

    //  Throw the error
    THROW_TCPIP_ERROR ( strgError.c_str () , -1 ) ;

  } /* endif */

  usPort  = pservent -> s_port ;

  //  AT THIS POINT WE ARE THREAD SAFE AGAIN!
  //  Convert to HOST byte order
  ntohs ( usPort ) ;

  return  usPort   ;

} /* ushort SockAddr :: servicePort ( const char * cszService , const char * cszProtocol ) */



/****************************************  SOCKADDR OPERATORS         ***************************************/



/***************************************************************************************************************************
  SockAddr & SockAddr :: operator = ( const SockAddr & rcsokaddr )
****************************************************************************************************************************

  ARGUMENTS     :
    const SockAddr & rcsokaddr
      The SockAddr structure used as the source for this copy operator.


  RETURNS       :
    A reference to the receiver.


  DESCRIPTION   :
    This is the SockAddr copy operator.  It does a DEEP copy on the source argument 'rcsokaddr'.

***************************************************************************************************************************/
SockAddr & SockAddr :: operator = ( const SockAddr & rcsokaddr )
{
  init ( rcsokaddr ) ;

  return  * this ;

} /* SockAddr & SockAddr :: operator = ( const SockAddr & rcsokaddr ) */



/*************************************************  end of file sockaddr.cpp   ********************************************/

