/***************************************************************************************************************************

  FILE NAME :
    ipsock.cpp


  DESCRIPTION :
    This module contains the methods used to implement public interface to the class IPSocket.  This is a
    base class that provides some VERY simple TCPIP socket capabilities.


  METHODS DEFINED IN THIS MODULE :

    IPSocket :: IPSocket  ()
    IPSocket :: IPSocket  ( const IPSocket & rcipsocket )
    IPSocket :: IPSocket  ( const SockAddr & rcsokaddrLocal , const SockAddr & rcsokaddrRemot )
    IPSocket :: ~IPSocket ()

    void              IPSocket :: bind            ()
    int               IPSocket :: close           ()
    void              IPSocket :: connect         ()
    UINT16            IPSocket :: dataAvailable   () const
    int               IPSocket :: getSockOptions  ( int nOption , void * pvOptVal , int nOptLen , int nLevel = SOL_SOCKET ) const
    SINT32            IPSocket :: handle          ()  const
    IPSocket        & IPSocket :: init            ( const IPSocket  & rcipsocket )
    void              IPSocket :: ioctl           ( int    nOption  , char * pcBuff , int nBuffLen ) const
    SockAddr        & IPSocket :: localAddr       ()
    const SockAddr  & IPSocket :: localAddr       ()
    int               IPSocket :: read            ( char * pcBuff   , SINT16 s16Length ) const
    SockAddr        & IPSocket :: remotAddr       ()
    int               IPSocket :: setSockOptions  ( int nOption , const void * pcvOptVal , int nOptLen , int nLevel = SOL_SOCKET )
    SINT32            IPSocket :: setHandle       ( SINT32 s32NewSocket = 0 )
    SockAddr        & IPSocket :: setLocalAddr    ( const SockAddr  & rcsokaddrNew )
    SockAddr        & IPSocket :: setRemotAddr    ( const SockAddr  & rcsokaddrNew )
    int               IPSocket :: write           ( const char * pccBuff   , SINT16 s16Length ) const


  OPERATORS DEFINED IN THIS MODULE :
    IPSocket    & IPSocket :: operator =      ( const IPSocket  & rcipsocket )


  EXCEPTION CLASSES THROWN FROM THIS MODULE :
    TCPIPError


  Copyright 1997, Peter Garner
    This code may be used for any NON-MILITARY purpose.

***************************************************************************************************************************/
//  System Defined C Headers
# ifdef __UNX__
  # include <unistd.h>
# endif /* # ifdef __UNX__ */


//  TCPIP C Headers
extern "C"
{
  # include <netdb.h>
  # include <sys/types.h>
  # include <sys/ioctl.h>
  # include <netinet/in.h>
}


//  Locally Defined C Headers
# include <strfcns.h>
# include <tcpport.h>


//  Locally Defined C++ Headers
# include <ipsock.hpp>
# include <tcperror.hpp>




/***************************************************************************************************************************
  IPSocket :: IPSocket ()
****************************************************************************************************************************

  DESCRIPTION  :
    This is the IPSockets' default constructor.  It sets the member data of the object being constructed to 0.

***************************************************************************************************************************/
IPSocket :: IPSocket () : s32Socket ( 0 )
{
  //  Don't automatically close this unless we open the handle ourselves
  setDestroy ( false ) ;

} /* IPSocket :: IPSocket () */



/***************************************************************************************************************************
  IPSocket :: IPSocket ( const IPSocket & rcipsocket )
****************************************************************************************************************************

  ARGUMENTS     :
    const IPSocket  & rcipsocket
      A reference to the constant IPSocket source object.


  DESCRIPTION   :
    This is the IPSocket DEEP copy constructor.

***************************************************************************************************************************/
IPSocket :: IPSocket ( const IPSocket & rcipsocket )
  : s32Socket     ( rcipsocket.handle    ()  ) ,
    sokaddrLocal  ( rcipsocket.localAddr ()  ) ,
    sokaddrRemot  ( rcipsocket.remotAddr ()  )
{
  //  Don't automatically close this unless we open the handle ourselves
  setDestroy ( false ) ;

} /* IPSocket :: IPSocket ( const IPSocket & rcipsocket ) */



/***************************************************************************************************************************
  IPSocket :: IPSocket ( const SockAddr & rcsokaddrLocal , const SockAddr & rcsokaddrRemot )
****************************************************************************************************************************

  ARGUMENTS     :
    const SockAddr & rcsokaddrLocal
      The LOCAL  SockAddr structure

    const SockAddr & rcsokaddrRemot
      The REMOTE SockAddr structure


  DESCRIPTION   :
    This constructor creates an IPSocket given both a local and remote socket address

***************************************************************************************************************************/
IPSocket :: IPSocket ( const SockAddr & rcsokaddrLocal , const SockAddr & rcsokaddrRemot )
  : s32Socket ( 0 ) , sokaddrLocal ( rcsokaddrLocal ) , sokaddrRemot ( rcsokaddrRemot )
{
  //  Don't automatically close this unless we open the handle ourselves
  setDestroy ( false ) ;

} /* IPSocket :: IPSocket ( const SockAddr & rcsokaddrLocal , const SockAddr & rcsokaddrRemot ) */



/***************************************************************************************************************************
  virtual IPSocket :: ~IPSocket ()
****************************************************************************************************************************

  DESCRIPTION   :
    This is the IPSocket objects destructor.  If the 'AutoObject' destroy flag is on it will call
      'AutoObject freeResource ()' to shutdown and close the connection.

***************************************************************************************************************************/
IPSocket :: ~IPSocket ()
{
  debugMsg  ( "\nDestroying an object of class : "  ) ;
  debugLnF  ( className ()                          ) ;
  debugMsg  ( "The objects' socket handle is   : "  ) ;
  debugLnF  ( handle    ()                          ) ;

  //  Call the 'AutoObject :: destroy ()' method, this will call 'IPSocket :: freeResource ()' to shutdown and close the
  //  connection if the destroy flag is set.
  destroy   () ;

  debugLnF  ( "Leaving the destructor\n"            ) ;

} /* IPSocket :: ~IPSocket () */



/***************************************************************************************************************************
  void IPSocket :: bind ()
****************************************************************************************************************************

  DESCRIPTION :
    Binds the socket to the receivers' current LOCAL address.

***************************************************************************************************************************/
void IPSocket :: bind ()
{
  int         nRetVal   = 0                                 ;
  size_t      stAddrLen = SockAddr :: addrSize     ()    ;
  //  Get a pointer to the receivers' LOCAL socket address
  sockaddr  * psockaddr = localAddr ().sockAddr ()    ;

  //  Bind the receivers' handle to the receivers' LOCAL socket address using the TCPIP "C" API ':: bind'.
  nRetVal = ::bind ( handle () , psockaddr , stAddrLen ) ;

  //  Check for error conditions and throw a TCPIPError if one has occured
  ESCAPE_BAD_TCPIP ( nRetVal , "TCPIP Error on BIND" , handle () ) ;

} /* void IPSocket :: bind () */



/***************************************************************************************************************************
  virtual int IPSocket :: close ()
****************************************************************************************************************************

  DESCRIPTION :
    This method terminates (closes) the receivers' communications connection.

***************************************************************************************************************************/
int IPSocket :: close ()
{
  //  Attempt to close the socket handle.  Note that we are saving the result code and returning it in 2 steps
  //  so that we can more easily view the result code with a symbolic debugger.  (When the program has been
  //  debugged and then recompiled with optimizations turned on, the compiler will optimize these 2 steps into
  //  "return :: soclose ( handle () ) ;", giving us the best of both worlds!)
  int nRetVal = :: soclose ( handle () ) ;

  return  nRetVal ;

} /* int IPSocket :: close () */



/***************************************************************************************************************************
  virtual void IPSocket :: connect ()
****************************************************************************************************************************

  DESCRIPTION :
    This method attempts to open the receivers' communications connection.  If the connection fails, a TPIPError will be
    thrown.

***************************************************************************************************************************/
void IPSocket :: connect ()
{
  //  Use the TCPIP "C" API to connect the socket to it's remote (peer) address
  int nRetVal = :: connect ( handle () , remotAddr ().sockAddr () , SockAddr :: addrSize () ) ;

  //  Check for error conditions and throw a TCPIPError if one has occured
  ESCAPE_BAD_TCPIP ( nRetVal , "TCPIP Error on CONNECT" , handle () ) ;

} /* void IPSocket :: connect () */



/***************************************************************************************************************************
  UINT16 IPSocket :: dataAvailable () const
****************************************************************************************************************************

  RETURNS     :
    The number of bytes currently readable on the socket.


  DESCRIPTION :
    Returns the number of bytes currently retrievable on the socket.  If the call fails, a TPCPIPError is thrown.

***************************************************************************************************************************/
UINT16 IPSocket :: dataAvailable () const
{
  unsigned  uCount  = 0 ;

  //  Check if data is available using the IOCTL option FIONREAD, note that we are using this Classes overload of 'ioctl',
  //  (void IPSocket :: ioctl) and that method does the error checking and exception throwing.
  ioctl  ( FIONREAD , (char *) & uCount , sizeof ( uCount ) ) ;

  //  Return the number of bytes available
  return  (UINT16) uCount ;

} /* UINT16 IPSocket :: dataAvailable () const */



/***************************************************************************************************************************
  void IPSocket :: freeResource ()
****************************************************************************************************************************

  DESCRIPTION :
    This method shuts down and closes the receivers' socket handle and sets that handle to 0, (Preventing a Run Time Error
    if it is called twice.)  Note that socket is closed irregardless of the status of the shouldDestroy flag.  In order to
    call this method based on that flag, call "virtual void AutoObject :: destroy ()".

***************************************************************************************************************************/
void IPSocket :: freeResource ()
{
  debugMsg  ( "\nEntering 'void IPSocket :: freeResource ()', about to shutdown and close socket handle : " ) ;
  debugLnF  ( handle () ) ;

  shutdown  ( handle () , 2 ) ;
  close     () ;

  debugLnF  ( "Leaving  'void IPSocket :: freeResource ()'\n" ) ;

} /* void IPSocket :: freeResource () */



/***************************************************************************************************************************
  int IPSocket :: getSockOptions ( int nOption , void * pvOptVal , int * pnOptLen , int nLevel = SOL_SOCKET ) const
****************************************************************************************************************************

  ARGUMENTS     :
    int     nOption
      The option value as an integer, i.e. SO_REUSEADDR, SO_LINGER, SO_KEEPALIVE

    void  * pvOptVal
      This is the address of the data structure that will hold the values returned by the system.  For MOST options this
      will be a simple integer that will be set to true (1) or false (0) upon return.

    int   * pnOptLen
      A pointer to an integer that will contain the size in bytes of the data copied into 'pvOptVal' by the system.  This
      value should be initialized to the size of the buffer 'pvOptVal' before this method is called.

    int     nLevel  = SOL_SOCKET
      The protocol level of the socket option requested.  This defaults to SOL_SOCKET which is usually the intended level.
      Otherwise this should be one of the protocol numbers (levels) listed in <netinet/in.h> (for example, IPPROTO_TCP).


  DESCRIPTION   :
    This calls the TCPIP "C" API 'getsockopt' using the receivers' Socket Handle.

***************************************************************************************************************************/
int IPSocket :: getSockOptions ( int nOption , void * pvOptVal , int * pnOptLen , int nLevel ) const
{
  //  Call the "C" API ':: getsockopt'
  int   nRetVal = :: getsockopt ( handle () , nLevel, nOption , pvOptVal , pnOptLen ) ;

  //  Check for error conditions and throw a TCPIPError if one has occured
  ESCAPE_BAD_TCPIP ( nRetVal , "TCPIP Error on getsockopt" , handle () ) ;

  return  nRetVal ;

} /* int IPSocket :: getSockOptions ( int nOption , void * pvOptVal , int * pnOptLen , int nLevel ) const */



/***************************************************************************************************************************
  SINT32 IPSocket :: handle () const
****************************************************************************************************************************

  RETURNS     :
    The receivers' current TCPIP Socket handle.


  DESCRIPTION :
    Returns the receivers' socket handle as a 32 bit signed integer.

***************************************************************************************************************************/
SINT32 IPSocket :: handle () const
{
  return  s32Socket ;

} /* SINT32 IPSocket :: handle () const */



/***************************************************************************************************************************
  IPSocket & IPSocket :: init ( const IPSocket & rcipsocket )
****************************************************************************************************************************

  ARGUMENTS     :
    const IPSocket  & rcipsocket
      A reference to the constant IPSocket source object for this initializer.


  RETURNS       :
    A reference to the receiver


  DESCRIPTION   :
    This is a IPSocket DEEP copy initializer, returns a reference to the receiver

***************************************************************************************************************************/
IPSocket & IPSocket :: init ( const IPSocket & rcipsocket )
{
  //  Copy the Addresses
  localAddr  ().init ( rcipsocket.localAddr () ) ;
  remotAddr  ().init ( rcipsocket.remotAddr () ) ;

  //  Copy the socket handle
  setHandle     ( rcipsocket.handle  ()  ) ;

  return  * this ;

} /* IPSocket & IPSocket :: init ( const IPSocket & rcipsocket ) */



/***************************************************************************************************************************
  void IPSocket :: ioctl ( int nOption , char * pcBuff , int nBuffLen ) const
****************************************************************************************************************************

  ARGUMENTS     :
    int     nOption
      The option number of the IOCTL request being made.

    char  * pcBuff
      The address of the buffer or data structure containing the information to be passed to or received from the ioctl API.

    int     nBuffLen
      The length in byte sof the buffer 'pcBuff'

  DESCRIPTION   :
    This method calls the TCPIP API 'ioctl' using the receivers' Socket Handle.

***************************************************************************************************************************/
void IPSocket :: ioctl ( int nOption , char * pcBuff , int nBuffLen ) const
{
  //  Call the "C" API ':: ioctl'
  int   nRetVal = :: ioctl ( handle () , nOption , pcBuff , nBuffLen ) ;

  //  Check for error conditions and throw a TCPIPError if one has occured
  ESCAPE_BAD_TCPIP ( nRetVal , "TCPIP Error on ioctl" , handle () ) ;

} /* void IPSocket :: ioctl ( int nOption , char * pcBuff , int nBuffLen ) const */



/***************************************************************************************************************************
  SockAddr & IPSocket :: localAddr ()
****************************************************************************************************************************

  RETURNS     :
    (SockAddr &) sockaddrLocal - A reference to the object that represents the receivers' LOCAL endpoint ( port and
    IP address)


  DESCRIPTION :
    Returns a reference to the receivers' Local SockAddr member.
    (NON CONSTANT OVERLOAD)

***************************************************************************************************************************/
SockAddr & IPSocket :: localAddr ()
{
  return  sokaddrLocal ;

} /* SockAddr & IPSocket :: localAddr () */



/***************************************************************************************************************************
  const SockAddr & IPSocket :: localAddr () const
****************************************************************************************************************************

  RETURNS     :
    (const SockAddr &) sockaddrLocal


  DESCRIPTION :
    Returns a constant reference to the receivers' Local SockAddr member.
    (CONSTANT OVERLOAD)

***************************************************************************************************************************/
const SockAddr & IPSocket :: localAddr () const
{
  return  sokaddrLocal ;

} /* const SockAddr & IPSocket :: localAddr () const */



/***************************************************************************************************************************
  int IPSocket :: read ( char * pcBuff , SINT16 s16Length ) const
****************************************************************************************************************************
  ARGUMENTS     :
    char    * pcBuff
      A pointer to the character buffer into which to read the message.

    SINT16    s16Length
      The amount of data to read


  RETURNS       :
    The number of bytes of data actually read.


  DESCRIPTION   :
    This method reads a TCPIP message into the buffer pointed to by 'pcBuff' on a connected socket.

***************************************************************************************************************************/
int IPSocket :: read ( char * pcBuff , SINT16 s16Length ) const
{
# ifndef __TEST__
  //  Attempt to read the requested number of bytes using the "C" API ':: recv'
  int nRetVal = :: recv ( handle () , pcBuff , s16Length , 0 ) ;

# else
  //  Attempt to read the requested number of bytes using the "C" API ':: read'
  int nRetVal = :: read ( handle () , pcBuff , s16Length ) ;

# endif  /* # ifndef __TEST__ */

  //  Check for error conditions and throw a TCPIPError if one has occured
  ESCAPE_BAD_TCPIP ( nRetVal , "TCPIP Error on READ" , handle () ) ;

  return  nRetVal ;

} /* int IPSocket :: read ( char * pcBuff , SINT16 s16Length ) const */



/***************************************************************************************************************************
  SockAddr & IPSocket :: remotAddr ()
****************************************************************************************************************************

  RETURNS     :
    (SockAddr &) sockaddrRemot


  DESCRIPTION :
    Returns a reference to the receivers' Remote SockAddr member.
    (NON CONSTANT OVERLOAD)

***************************************************************************************************************************/
SockAddr & IPSocket :: remotAddr ()
{
  return  sokaddrRemot ;

} /* SockAddr & IPSocket :: remotAddr () */



/***************************************************************************************************************************
  const SockAddr & IPSocket :: remotAddr () const
****************************************************************************************************************************

  RETURNS     :
    (const SockAddr &) sockaddrRemot - A reference to the object that represents the receivers' REMOTE endpoint ( port and
    IP address)


  DESCRIPTION :
    Returns a constant reference to the receivers' Remote SockAddr member.
    (CONSTANT OVERLOAD)

***************************************************************************************************************************/
const SockAddr & IPSocket :: remotAddr () const
{
  return  sokaddrRemot ;

} /* const SockAddr & IPSocket :: remotAddr () const */



/***************************************************************************************************************************
  SINT32 IPSocket :: setHandle ( SINT32 s32NewSocket )
****************************************************************************************************************************

  ARGUMENTS   :
    SINT32  s32NewSocket
      The value of the new socket handle with which we want to associate the receiver.

  RETURNS     :
    Returns the receivers' socket handle as a 32 bit signed integer.


  DESCRIPTION :
    Sets the receivers' socket handle to 's32NewSocket' and returns
    that socket handle as a 32 bit signed integer.

***************************************************************************************************************************/
SINT32 IPSocket :: setHandle ( SINT32 s32NewSocket )
{
  return  s32Socket = s32NewSocket ;

} /* SINT32 IPSocket :: setHandle ( SINT32 s32NewSocket ) */



/***************************************************************************************************************************
  SockAddr & IPSocket :: setLocalAddr ( const SockAddr & rcsaddrLocal )
****************************************************************************************************************************

  ARGUMENTS   :
    const SockAddr  & rcsokaddrLocal
      A reference to a const SockAddr object.  This SockAddr object will be used as the source of a DEEP
      copy into the receivers' local bind (read/receive) address.


  RETURNS     :
    A reference to the receivers' local read SockAddr structure.


  DESCRIPTION :
    This method does a DEEP copy from the input argument 'rcsokaddrLocal' into the receivers' local bind
    (read/receive) address.

***************************************************************************************************************************/
SockAddr & IPSocket :: setLocalAddr ( const SockAddr & rcsokaddrLocal )
{
  localAddr ().init ( rcsokaddrLocal ) ;

  return  localAddr () ;

} /* SockAddr & IPSocket :: setRemotAddr ( const SockAddr & rcsokaddrLocal ) */



/***************************************************************************************************************************
  SockAddr & IPSocket :: setRemotAddr ( const SockAddr & rcsaddrNewRemot )
****************************************************************************************************************************

  ARGUMENTS   :
    const SockAddr  & rcsokaddrNewRemot
      A reference to a const SockAddr object.  This SockAddr object will be used as the source of a DEEP
      copy into the receivers' remote connect (write/send) address.


  RETURNS     :
    A reference to the receivers' local read SockAddr structure.


  DESCRIPTION :
    This method does a DEEP copy from the input argument 'rcsokaddrRemot' into the receivers' remote
    connect (write/send) address.

***************************************************************************************************************************/
SockAddr & IPSocket :: setRemotAddr ( const SockAddr & rcsokaddrRemot )
{
  remotAddr ().init ( rcsokaddrRemot ) ;

  return  remotAddr () ;

} /* SockAddr & IPSocket :: setRemotAddr ( const SockAddr & rcsokaddrNewRemot ) */



/***************************************************************************************************************************
  int IPSocket :: setSockOptions ( int nOption , const void * pcvOptVal , int * pnOptLen , int nLevel = SOL_SOCKET )
****************************************************************************************************************************

  ARGUMENTS     :
    int     nOption
      The option value as an integer, i.e. SO_REUSEADDR, SO_LINGER, SO_KEEPALIVE

    const void  * pcvOptVal
      This is the address of the data structure that holds the option values to be passed to the system.  For MOST options
      this will be a simple integer that will be set to true (1) or false (0) upon return.

    int           nOptLen
      A n integer that contains the size in bytes of the data buffer copied into 'pcvOptVal'.

    int     nLevel  = SOL_SOCKET
      The protocol level of the socket option requested.  This defaults to SOL_SOCKET which is usually the intended level.
      Otherwise this should be one of the protocol numbers (levels) listed in <netinet/in.h> (for example, IPPROTO_TCP).


  DESCRIPTION   :
    This calls the TCPIP "C" API 'setsockopt' using the receivers' Socket Handle.

***************************************************************************************************************************/
int IPSocket :: setSockOptions ( int nOption , const void * pcvOptVal , int nOptLen , int nLevel )
{
  //  Call the "C" API ':: setsockopt'
  int   nRetVal = :: setsockopt ( handle () , nLevel, nOption , pcvOptVal , nOptLen ) ;

  //  Check for error conditions and throw a TCPIPError if one has occured
  ESCAPE_BAD_TCPIP ( nRetVal , "TCPIP Error on setsockopt" , handle () ) ;

  return  nRetVal ;

} /* int IPSocket :: setSockOptions ( int nOption , const void * pcvOptVal , int nOptLen , int nLevel ) */



/***************************************************************************************************************************
  int IPSocket :: write ( const char *  pccBuff , SINT16 s16Length ) const
****************************************************************************************************************************
  ARGUMENTS     :
    const char  * pccBuff
      A pointer to the character buffer containing the data to write to the socket.

    SINT16        s16Length
      The amount of data to write


  RETURNS       :
    The number of bytes of data actually written.


  DESCRIPTION   :
    This method writes a TCPIP message from the buffer pointed to by 'pcBuff' onto a connected socket.

***************************************************************************************************************************/
int IPSocket :: write ( const char *  pccBuff , SINT16 s16Length ) const
{
# ifndef __TEST__
  //  Attempt to read the requested number of bytes using the "C" API ':: send'
  int nRetVal = :: send ( handle () , pccBuff , s16Length , 0 ) ;

# else
  //  Attempt to write the requested number of bytes using the "C" API ':: write'
  int   nRetVal = :: write ( handle () , pccBuff , s16Length ) ;

# endif  /* # ifndef __TEST__ */

  //  Check for error conditions and throw a TCPIPError if one has occured
  ESCAPE_BAD_TCPIP ( nRetVal , "TCPIP Error on WRITE" , handle () ) ;

  return  nRetVal ;

} /* int IPSocket :: write ( const char *  pccBuff , SINT16 s16Length ) const */




/*********************************************    IPSocket OPERATORS    ***************************************************/



/***************************************************************************************************************************
  IPSocket & IPSocket :: operator = ( const IPSocket & rcipsocket )
****************************************************************************************************************************

  ARGUMENTS     :
    const IPSocket  & rcipsocket
      A reference to the constant IPSocket source object for this initializer.


  RETURNS       :
    A reference to the receiver.


  DESCRIPTION   :
    This is the copy '=' operator, does a DEEP copy from rcipsocket and returns a reference to the receiver

***************************************************************************************************************************/
IPSocket & IPSocket :: operator = ( const IPSocket & rcipsocket )
{
  //  Simply call the already defined init copy method!
  return  init  ( rcipsocket ) ;

} /* IPSocket & IPSocket :: operator = ( const IPSocket & rcipsocket ) */




/************************************************    end of file ipsock.cpp   *********************************************/

