/***************************************************************************************************************************

  Copyright - Peter Garner, 1997.
     This code is released into the public domain, and you are free to use and modify this without credit to the author

***************************************************************************************************************************/
/***************************************************************************************************************************
 
    FILE NAME :
      CRESLOCK.CPP
 
 
    APPLICATION :
      GENERAL REPLACEMENT FOR IBM IPrivateResource AND IResourceLock CLASSES UNDER AIX
 
 
    DESCRIPTION :
      This module implements the most commonly used methods in the interface of the IBM IPrivateResource AND IResourceLock
      classes.  These classes as defined by IBM do not seem to work under our current version (V 3.1.3.0) of CSet++/6000
     
 
    METHODS DEFINED IN THIS MODULE :
      PrivateResource :: PrivateResource                  ()
      PrivateResource :: ~PrivateResource                 ()
      pthread_mutex_t   * PrivateResource :: getMutexAddr ()
      int                 PrivateResource :: lock         ()
      int                 PrivateResource :: unlock       ()

 
***************************************************************************************************************************/
//  System Defined C Headers
# include <pthread.h>


//  System Defined C++ Headers
# include <iostream.h>


//  IBM OCL Defined C++ Headers
# include <iexcept.hpp>


//  Locally Defined C++ Headers
# include <namedobj.hpp>
# include <creslock.hpp>



/***************************************************************************************************************************
  PrivateResource :: PrivateResource ()
****************************************************************************************************************************

  DESCRIPTION :
    This is the PrivateResource constructor.  This constructor initializes the member mutex semaphore using the system
    defaults.


  EXCEPTIONS  :
    If any error occurs, an IAccessError is thrown, so that the behaviour is similiar to that defined for the IBM
    IPrivateResource class.

***************************************************************************************************************************/
PrivateResource :: PrivateResource ()
{
  int nErrCode = pthread_mutex_init ( getMutexAddr () , pthread_mutexattr_default ) ;

  //  Check to see if we could initialize the mutex semphore
  if ( nErrCode )
  {
    //  Get any error text information the operating system can provide
    ICLibErrorInfo  ierrinfo      ( "Could not initialize mutex semaphore " ) ;

    //  Create an error object using the OS text and the the returned error code
    IAccessError    iaccesserror  ( ierrinfo.text () , nErrCode ) ;

    ITHROW ( iaccesserror ) ;

  } /* endif */

} /* PrivateResource :: PrivateResource () */



/***************************************************************************************************************************
  PrivateResource :: ~PrivateResource ()
****************************************************************************************************************************

  DESCRIPTION :
    This is the PrivateResource destructor.  This destructor destroys the member mutex semaphore used by the receiver.

***************************************************************************************************************************/
PrivateResource :: ~PrivateResource ()
{
  //  Destroy the member mutex
  pthread_mutex_destroy ( getMutexAddr () ) ;

} /* PrivateResource :: ~PrivateResource () */



/***************************************************************************************************************************
  pthread_mutex_t * PrivateResource :: getMutexAddr ()
****************************************************************************************************************************

  RETURNS     :
    A pointer to the receivers' member mutex semaphore.


  DESCRIPTION :
    This is the accessor for the address of receivers' member mutex.

***************************************************************************************************************************/
pthread_mutex_t * PrivateResource :: getMutexAddr ()
{
  return  & mutex ;

} /* pthread_mutex_t * PrivateResource :: getMutexAddr () */



/***************************************************************************************************************************
  int PrivateResource :: lock ()
****************************************************************************************************************************

  RETURNS     :
    The error code (should be 0) from the operating systems attempt to LOCK the mutex.


  DESCRIPTION :
    This methods attempts to LOCK the receivers' member mutex.

***************************************************************************************************************************/
int PrivateResource :: lock ()
{
  return  pthread_mutex_lock ( getMutexAddr () ) ;

} /* int PrivateResource :: lock () */



/***************************************************************************************************************************
  int PrivateResource :: unlock ()
****************************************************************************************************************************

  RETURNS     :
    The error code (should be 0) from the operating systems attempt to UNLOCK the mutex.


  DESCRIPTION :
    This methods attempts to UNLOCK the receivers' member mutex.

***************************************************************************************************************************/
int PrivateResource :: unlock ()
{
  return  pthread_mutex_unlock ( getMutexAddr () ) ;

} /* int PrivateResource :: unlock () */

/************************************************  END OF FILE CRESLOCK.CPP  **********************************************/