# ifndef __templptr_hpp
  # define __templptr_hpp

/***************************************************************************************************************************

  Copyright - Peter Garner, 1997.
     This code is released into the public domain, and you are free to use and modify this without credit to the author

***************************************************************************************************************************/

//  Locally Defined C++ Headers
# include <autoobj.hpp>



//  This template class represents a single generic pointer.  It's sole purpose is to assure that dynamic objects are
//  deleted when they go out of scope.

template < class T > class TemplatePointer : public AutoObject
{
public    :
//  Constructor
  TemplatePointer ( T * ptInit = 0 ) ;

//  Destructor
  virtual ~TemplatePointer () ;

//  Virtual Constant Methods
  virtual const char  * className  () const ;

//  Virtual Methods
  virtual void          deleteMem () ;

//  Constant Methods
  T                   * getPointer    ()  const ;

//  Methods
  T                   * getPointer    () ;
  const T             * init          ( T * ptInit = 0 ) ;
  
//  Constant Operators
  operator const        T *           () const ;
  operator const        T &           () const ;

//  Operators
  operator              T *           () ;
  operator              T &           () ;

private   :
  T         *   ptObject  ;

} ; /* template < class T > class TemplatePointer : public NamedObject */



/***************************************************************************************************************************
  template < class T > inline TemplatePointer < T > :: TemplatePointer ( T * ptInit = 0 )
****************************************************************************************************************************

  ARGUMENTS   :
    T   * ptInit
      A pointer to an object of type T.  This object being constructed will encapsulate this pointer.


  DESCRIPTION :
    This constructor creates a TemplatePointer < T > object that points to the argument 'ptInit'.

***************************************************************************************************************************/
template < class T > inline TemplatePointer < T > :: TemplatePointer ( T * ptInit )
  : ptObject ( 0 )
{
  init  ( ptInit ) ;

} /* template < class T > TemplatePointer < T > :: TemplatePointer ( T * ptInit ) */



/***************************************************************************************************************************
  virtual template < class T > inline TemplatePointer < T > :: ~TemplatePointer ()
****************************************************************************************************************************

  DESCRIPTION :
    This is the TemplatePointer < T > & destructor.  It calls 'AutoObject :: destroy ()' to delete the receivers' memory.


***************************************************************************************************************************/
template < class T > inline TemplatePointer < T > :: ~TemplatePointer ()
{
  destroy () ;

} /* template < class T > TemplatePointer < T > :: ~TemplatePointer () */



/***************************************************************************************************************************
  virtual template < class T > inline const char * TemplatePointer < T > :: className () const
****************************************************************************************************************************

  RETURNS     :
    A constant pointer to the ASCIIZ string "template < class T > class TemplatePointer".


  DESCRIPTION :
    This VIRTUAL constant method always returns the class name of the receiver.  This may be used to implement
    a primitive Run Time Type Identification, (RTTI), or anything else you like.  Note that this IS a virtual
    function, and will early bind in the constructors and destructor.  Thus you can also use this method to
    determine in which constructor level you are, although I don't know why you would want to know that!

***************************************************************************************************************************/
template < class T > inline const char * TemplatePointer < T > :: className  () const
{
  return  "TemplatePointer" ;

} /* template < class T > const char * TemplatePointer < T > :: className () const */



/***************************************************************************************************************************
  virtual template < class T > inline void TemplatePointer < T > :: deleteMem ()
****************************************************************************************************************************

  DESCRIPTION :
    This method deletes the receivers' internal memory pointer and sets that pointer to point to 0, (Preventing a Run Time
    Error if it is called twice.)

***************************************************************************************************************************/
template < class T > inline void TemplatePointer < T > :: deleteMem ()
{
  delete  ptObject ;

  ptObject  = 0 ;

} /* template < class T > void TemplatePointer < T > :: deleteMem () */



/***************************************************************************************************************************
  template < class T > inline T * TemplatePointer < T > :: getPointer ()
****************************************************************************************************************************

  RETURNS     :
    The address of the memory encapsulated by the receiver as a pointer to < T >.


  DESCRIPTION :
    This method is the NON CONSTANT accessor for the address of the memory encapsulated by the receiver.

***************************************************************************************************************************/
template < class T > inline T * TemplatePointer < T > :: getPointer ()
{
  return  ptObject ;

} /* template < class T > T * TemplatePointer < T > :: getPointer () */



/***************************************************************************************************************************
  template < class T > inline T * TemplatePointer < T > :: getPointer () const
****************************************************************************************************************************

  RETURNS     :
    The address of the memory encapsulated by the receiver as a pointer to a constant < T >.


  DESCRIPTION :
    This method is the CONSTANT accessor for the address of the memory encapsulated by the receiver.

***************************************************************************************************************************/
template < class T > inline T * TemplatePointer < T > :: getPointer () const
{
  return  ptObject ;

} /* template < class T > T * TemplatePointer < T > :: getPointer () const */



/***************************************************************************************************************************
  template < class T > inline const T * TemplatePointer < T > :: init ( T * ptInit )
****************************************************************************************************************************

  ARGUMENTS   :
    T   * ptInit
      A pointer to an object of type T.  This object being initalized will encapsulate this pointer.


  RETURNS     :
    The value of the input argument 'ptInit' as a pointer to a constant T.


  DESCRIPTION :
    This method initializes a TemplatePointer < T > object to point to the argument 'ptInit'.

***************************************************************************************************************************/
template < class T > inline const T * TemplatePointer < T > :: init ( T * ptInit )
{
  delete  ptObject ;

  ptObject = ptInit ;

  return  getPointer  ()  ;

} /* template < class T > const T * TemplatePointer < T > :: init ( T * ptInit ) */



/***************************************************************************************************************************
  template < class T > inline TemplatePointer < T > :: operator T * ()
****************************************************************************************************************************

  RETURNS     :
    This method returns the address of the memory encapsulated by the receiver as a pointer to T.


  DESCRIPTION :
    This operator is the NON CONSTANT operator for the address of the memory encapsulated by the receiver.  This operator
    converts the object to a pointer to T.

***************************************************************************************************************************/
template < class T > inline TemplatePointer < T > :: operator T * ()
{
  return  getPointer  ()  ;

} /* template < class T > inline TemplatePointer < T > :: operator T * () */



/***************************************************************************************************************************
  template < class T > inline TemplatePointer < T > :: operator const T * () const
****************************************************************************************************************************

  RETURNS     :
    This method returns the address of the memory encapsulated by the receiver as a pointer to a constant < T >.


  DESCRIPTION :
    This operator is the CONSTANT operator for the address of the memory encapsulated by the receiver.  This operator
    converts the object to a pointer to a CONSTANT < T >.

***************************************************************************************************************************/
template < class T > inline TemplatePointer < T > :: operator const T * () const
{
  return  getPointer  ()  ;

} /* template < class T > inline TemplatePointer < T > :: operator const T * () const */



/***************************************************************************************************************************
  template < class T > inline TemplatePointer < T > :: operator T * ()
****************************************************************************************************************************

  RETURNS     :
    This method returns the object encapsulated by the receiver as a reference to T.


  DESCRIPTION :
    This operator is the NON CONSTANT operator for the the memory object encapsulated by the receiver.  This operator
    converts the receiver to a reference to T.

***************************************************************************************************************************/
template < class T > inline TemplatePointer < T > :: operator T & ()
{
  return  * getPointer  ()  ;

} /* template < class T > inline TemplatePointer < T > :: operator T & () */



/***************************************************************************************************************************
  template < class T > inline TemplatePointer < T > :: operator const T & () const
****************************************************************************************************************************

  RETURNS     :
    This method returns the object encapsulated by the receiver as a reference to a CONSTANT T.


  DESCRIPTION :
    This operator is the CONSTANT operator for the the memory object encapsulated by the receiver.  This operator
    converts the receiver to a reference to a CONSTANT T.

***************************************************************************************************************************/
template < class T > inline TemplatePointer < T > :: operator const T & () const
{
  return  * getPointer  ()  ;

} /* template < class T > inline TemplatePointer < T > :: operator const T & () const */




//  This is a commonly used synonym for for a character pointer.
typedef  TemplatePointer < char >  CharPointer  ;



# endif  /*  # ifndef __templptr_hpp  */
