# ifndef __templary_hpp
  # define __templary_hpp

/***************************************************************************************************************************

  Copyright - Peter Garner, 1997.
     This code is released into the public domain, and you are free to use and modify this without credit to the author

***************************************************************************************************************************/
//  System Defined C Headers
# include <memory.h>
# include <stddef.h>


//  Locally Defined C++ Headers
# include <autoobj.hpp>
# include <checkmem.hpp>




template < class T > class TemplateArray : public AutoObject
{
public    :
//  Constructors
  TemplateArray                         ( size_t stNumElements = 0 ) ;
  TemplateArray                         ( size_t stNumElements , const T * pctSrcArray ) ;
  TemplateArray                         ( const TemplateArray < T > & rctemplarray ) ;

//  Destructor
  virtual ~TemplateArray                ()  ;

//  Virtual Constant Methods
  virtual const char    * className     ()  const                 ;

//  Virtual Methods
  virtual void            deleteMem     ()                        ;  

//  Constant Methods
  const T               * getBase       () const                  ; //  Return the buffer as an array of < class T >
  const void            * getMem        () const                  ; //  Return a pointer to the base of array as void *.
  size_t                  numElems      () const                  ; //  Gets number of elements array can hold
  size_t                  size          () const                  ; //  Gets size of array in bytes
                        
//  Methods             
  T                     * getBase       ()                        ; //  Return the buffer as an array of < class T >
  void                  * getMem        ()                        ; //  Return a pointer to the base of array as void *.
  void                    init          ( size_t stNumELem , const T * pctSrcArray  = 0 ) ;
  void                    init          ( const TemplateArray < T > & rctemplarray      ) ;
  void                    resize        ( size_t stNumElements )  ; //  Reallocate so that it holds stNumElements Elements
                        
//  Operators           
  operator  const T     *               ()              const     ; //  Reference Constant Object as a Constant Pointer to T
  operator        T     *               ()                        ; //  Reference Object as a Pointer to T
  const T               & operator []   ( size_t stN )  const     ; //  Return a Constant Reference to the Nth Element starting at 0
  T                     & operator []   ( size_t stN )            ; //  Return a Reference to the Nth Element starting at 0
                        
protected :             
  virtual void            clear         ()                        ; //  Set entire internal buffer to 0
  T                     * setBase       ( T * ptSrcArray       )  ;
  size_t                  setNumElems   ( size_t stNumElements )  ; //  Sets number of elements array can hold

private   :
  size_t        stNumElem                                         ; //  Number of Elements allocated to hold
  T           * pt                                                ; //  Base Pointer

} ; /* template < class T > class TemplateArray : public NamedObject */



/***************************************************************************************************************************
  template < class T > TemplateArray < T > :: TemplateArray ( size_t stNumElements = 0 ) 
****************************************************************************************************************************

  ARGUMENTS   :
    size_t  stNumElements
      The number of elements the array should hold.


  DESCRIPTION :
    This constructor creates a TemplateArray < T > object large enough to hold 'stNumElements'.


  EXCEPTIONS  :
    If the attempt to allocate memory fails, an IOutOfMemory exception will be thrown.

***************************************************************************************************************************/
template < class T > inline TemplateArray < T > :: TemplateArray ( size_t stNumElements ) 
  : pt ( 0 ) , stNumElem ( 0 )
{
  //  If we have 0 elements to allocate, just return, but be sure the ALL THE DATA is initialized to 0 in
  //  the initializer list!
  if ( 0 == stNumElements )
  {
    return  ; 

  } /* endif */

  //  Set the number of elements data member to the input argument
  setNumElems ( stNumElements ) ;

  //  And allocate an array of that many elements
  pt = new  T [stNumElements] ;

  //  Finally check to make sure the allocation was successfull.
  CHECK_MEM ( pt ) ;

} /* template < class T > inline TemplateArray < T > :: TemplateArray ( size_t stNumElements ) */



/***************************************************************************************************************************
  template < class T > TemplateArray < T > :: TemplateArray ( size_t stNumElements , const T * pctSrcArray )
****************************************************************************************************************************

  ARGUMENTS   :
    size_t    stNumElements
      The number of elements the array should hold.

    const T * pctSrcArray
      The address of an array of T objects.  The first 'stNumElements' from this array will be DEEP copied into the
      object being constructed.


  DESCRIPTION :
    This constructor creates a TemplateArray < T > object large enough to hold 'stNumElements', of type < T >and copies the
    first 'stNumElements' into that array.  In otherwords this constructor creates an array of 'stNumElements' as a DEEP copy
    of the argument 'pctSrcArray'.


  NOTES       :
    The element type must have an '=' operator defined for it in order to use this method.


  EXCEPTIONS  :
    If the attempt to allocate memory fails, an IOutOfMemory exception will be thrown.

***************************************************************************************************************************/
template < class T > inline TemplateArray < T > :: TemplateArray ( size_t stNumElements , const T * pctSrcArray )
  : pt ( 0 ) , stNumElem ( 0 )
{
  init ( stNumElements , pctSrcArray ) ;

} /* template < class T > inline TemplateArray < T > :: TemplateArray ( size_t stNumElements , const T * pctSrcArray ) */



/***************************************************************************************************************************
  template < class T > inline TemplateArray < T > :: TemplateArray ( const TemplateArray < T > & rctemplarray )
****************************************************************************************************************************

  ARGUMENTS   :
    const TemplateArray < T > & rctemplarray
      A reference to a constant TemplateArray object that will be the source for this DEEP copy constructor


  DESCRIPTION :
    This is the const TemplateArray < T > & rctemplarray copy constructor.  It creates a
    const TemplateArray < T > & rctemplarray object that is a DEEP copy of the argument 'rctemplarray'.


***************************************************************************************************************************/
template < class T > inline TemplateArray < T > :: TemplateArray ( const TemplateArray < T > & rctemplarray )
  : pt ( 0 ) , stNumElem ( 0 )
{
  init ( rctemplarray ) ;

} /* template < class T > inline TemplateArray < T > :: TemplateArray ( const TemplateArray < T > & rctemplarray ) */



/***************************************************************************************************************************
  template < class T > inline TemplateArray < T > :: ~TemplateArray ()
****************************************************************************************************************************

  DESCRIPTION :
    This is the TemplateArray < T > destructor.  It calls 'AutoObject :: destroy ()' to delete the
    receivers' memory array.


***************************************************************************************************************************/
template < class T > inline TemplateArray < T > :: ~TemplateArray ()
{
  destroy () ;

} /* template < class T > inline TemplateArray < T > :: ~TemplateArray () */



/***************************************************************************************************************************
   template < class T > inline const char * TemplateArray < T > :: className () const
****************************************************************************************************************************

  RETURNS     :
    A constant pointer to the ASCIIZ string "template < class T > class TemplateArray".


  DESCRIPTION :
    This VIRTUAL constant method always returns the class name of the receiver.  This may be used to implement
    a primitive Run Time Type Identification, (RTTI), or anything else you like.  Note that this IS a virtual
    function, and will early bind in the constructors and destructor.  Thus you can also use this method to
    determine in which constructor level you are, although I don't know why you would want to know that!

***************************************************************************************************************************/
template < class T > inline const char * TemplateArray < T > :: className () const
{
  return  "template < class T > class TemplateArray" ;    

} /* template < class T > inline const char * TemplateArray < T > :: className () const */



/***************************************************************************************************************************
  virtual template < class T > inline void TemplateArray < T > :: clear ()
****************************************************************************************************************************

  DESCRIPTION :
    This PROTECTED method sets the receivers' ENTIRE memory buffer to 0.  NO DESTRUCTORS are called.  This is a very
    dangerous method and can really screw things up!  This method is really intended to be used when the template is
    instantiated for a "C" type, (char, int, struct, etc.).  Since the destructor is not called for type T, only use
    this method for types that have no destructor.  This method is defined as virtual so that you may override it to
    do nothing when this template is instantiated for a type that DOES have a destructor defined.

***************************************************************************************************************************/
template < class T > inline void TemplateArray < T > :: clear ()
{
  memset ( getMem () , 0 , sizeof ( T ) * numElems () ) ;

} /* template < class T > inline void TemplateArray < T > :: clear () */



/***************************************************************************************************************************
  virtual template < class T > inline void TemplateArray < T > :: deleteMem ()
****************************************************************************************************************************

  DESCRIPTION :
    This method deletes the receivers' ENTIRE memory array and sets the internal buffer to point to 0, (Preventing a Run
    Time Error if it is called twice.)  Note that the memory is deleted irregardless of the status of the shouldDestroy
    flag.  To delete the contents of the receiver based on that flag, call "virtual void AutoObject :: destroy ()".

***************************************************************************************************************************/
template < class T > inline void TemplateArray < T > :: deleteMem ()
{
  delete  []  pt ;

  pt  = 0 ;

} /* template < class T > inline void TemplateArray < T > :: deleteMem () */



/***************************************************************************************************************************
   template < class T > inline T * TemplateArray < T > :: getBase () 
****************************************************************************************************************************

  RETURNS     :
    The address of the array encapsulated by the receiver as a pointer to T.


  DESCRIPTION :
    This method is the NON CONSTANT accessor for the address of the array encapsulated by the receiver.

***************************************************************************************************************************/
template < class T > inline T * TemplateArray < T > :: getBase () 
{
  return  pt ;

} /* template < class T > inline T * TemplateArray < T > :: getBase ()  */



/***************************************************************************************************************************
   template < class T > inline const T * TemplateArray < T > :: getBase () const
****************************************************************************************************************************

  RETURNS     :
    The address of the array encapsulated by the receiver as a constant pointer to < T >.


  DESCRIPTION :
    This method is the CONSTANT accessor for the address of the array encapsulated by the receiver.

***************************************************************************************************************************/
template < class T > inline const T * TemplateArray < T > :: getBase () const
{
  return  pt ;

} /* template < class T > inline const T * TemplateArray < T > :: getBase () const */



/***************************************************************************************************************************
   template < class T > inline  void * TemplateArray < T > :: getMem () 
****************************************************************************************************************************

  RETURNS     :
    This method returns the address of the array encapsulated by the receiver as a void pointer.


  DESCRIPTION :
    This method is the NON CONSTANT accessor for the address of the array encapsulated by the receiver as generic pointer to
    void.

***************************************************************************************************************************/
template < class T > inline  void * TemplateArray < T > :: getMem () 
{
  return  (void *) getBase () ;

} /* template < class T > inline  void * TemplateArray < T > :: getMem ()  */



/***************************************************************************************************************************
   template < class T > inline const void * TemplateArray < T > :: getMem () const
****************************************************************************************************************************

  RETURNS     :
    This method returns the address of the array encapsulated by the receiver as a constant void pointer.


  DESCRIPTION :
    This method is the CONSTANT accessor for the address of the array encapsulated by the receiver.

***************************************************************************************************************************/
template < class T > inline const void * TemplateArray < T > :: getMem () const
{
  return  (void *) getBase () ;

} /* template < class T > inline const void * TemplateArray < T > :: getMem () const */



/***************************************************************************************************************************
  template < class T > inline void TemplateArray < T > :: init ( size_t stNumElements , const T * pctSrcArray )
****************************************************************************************************************************

  ARGUMENTS   :
    size_t    stNumElements
      The number of elements the array should hold.

    const T * pctSrcArray
      The address of an array of T objects.  The first 'stNumElements' from this array will be DEEP copied into the
      receiver.


  DESCRIPTION :
    This initializer initializes a TemplateArray < T > object large enough to hold 'stNumElements', and copies the first
    'stNumElements' into that array.  In otherwords this initializer creates an array of 'stNumElements' as a DEEP copy
    of the argument 'pctSrcArray'.


  NOTES       :
    The element type must have an '=' operator defined for it in order to use this method.


  EXCEPTIONS  :
    If the attempt to allocate memory fails, an IOutOfMemory exception will be thrown.

***************************************************************************************************************************/
template < class T > inline void TemplateArray < T > :: init ( size_t stNumElements , const T * pctSrcArray )
{
  size_t  st ;

  delete [] pt ;

  setBase ( 0 ) ;

  T * ptNew = new  T [setNumElems ( stNumElements )] ;

  CHECK_MEM ( ptNew ) ;

  setBase ( ptNew ) ;

  // If the source array is 0, just return!
  if ( 0 == pctSrcArray )
  {
    return  ;

  } /* endif */

  for ( st = 0 ; st < stNumElements ; st++ )
  {
    ptNew [st] = pctSrcArray [st] ;

  } /* endfor */

} /* template < class T > inline void TemplateArray < T > :: init ( size_t stNumElements , const T * pctSrcArray ) */



/***************************************************************************************************************************
  template < class T > inline TemplateArray < T > :: init ( const TemplateArray < T > & rctemplarray )
****************************************************************************************************************************

  ARGUMENTS   :
    const TemplateArray < T > & rctemplarray
      A reference to a constant TemplateArray object that will be the source for this DEEP copy initializer


  DESCRIPTION :
    This is the const TemplateArray < T > copy initializer.  It initializes the receiver to be a DEEP copy of the argument
    'rctemplarray'.


***************************************************************************************************************************/
template < class T > inline void TemplateArray < T > :: init ( const TemplateArray < T > & rctemplarray )
{
  init ( rctemplarray.numElems () , rctemplarray.getBase () ) ;  

} /* template < class T > inline void TemplateArray < T > :: init ( const TemplateArray < T > & rctemplarray ) */



/***************************************************************************************************************************
   template < class T > inline size_t TemplateArray < T > :: numElems () const
****************************************************************************************************************************

  RETURNS     :
    This method returns the number of elements in the array encapsulated by the receiver as a size_t.


  DESCRIPTION :
    This method is the CONSTANT accessor for the number of elements in the array encapsulated by the receiver.

***************************************************************************************************************************/
template < class T > inline size_t TemplateArray < T > :: numElems () const
{
  return  stNumElem ;

} /* template < class T > inline size_t TemplateArray < T > :: numElems () const */



/***************************************************************************************************************************
   template < class T > inline size_t TemplateArray < T > :: size () const
****************************************************************************************************************************

  RETURNS     :
    This method returns the number of bytes of memory contained by the receiver as a size_t.


  DESCRIPTION :
    This method is a CONSTANT accessor the number of bytes of memory in the array encapsulated by the receiver.  Note that
    the number returned may NOT be accurate if typoe < T > has a variable size.  This method simply returns the number of
    elements times the sizeof ( < T > ).  If T has a variable size, this number will be incorrect.

***************************************************************************************************************************/
template < class T > inline size_t TemplateArray < T > :: size () const
{
  return  stNumElem * sizeof ( T ) ;

} /* template < class T > inline size_t TemplateArray < T > :: size () const */



/***************************************************************************************************************************
  template < class T > inline void TemplateArray < T > :: resize ( size_t stNumElements )
****************************************************************************************************************************

  ARGUMENTS   :
    size_t    stNumElements
      The number of elements the array should hold.


  DESCRIPTION :
    This method reszies a TemplateArray < T > object so that it holds 'stNumElements', and copies the first 'stNumElements'
    into that array.  If the original array is smaller than the argument 'stNumElements', then only the smaller number of
    elements will be copied.


  NOTES       :
    The element type must have an '=' operator defined for it in order to use this method.


  EXCEPTIONS  :
    If the attempt to allocate memory fails, an IOutOfMemory exception will be thrown.

***************************************************************************************************************************/
template < class T > inline void TemplateArray < T > :: resize ( size_t stNumElements )
{
  //  Uninitialized Data
  size_t  st                            ;

  //  Initialized Data
  size_t    stOldSize     = numElems () ;
  size_t    stSmallerSize = stOldSize < stNumElements ? stOldSize : stNumElements ;
  T       * ptOld         = getBase    () ;
  T       * ptNew         = new T      [stNumElements] ;

  CHECK_MEM ( ptNew ) ;

  setBase ( ptNew ) ;

  setNumElems ( stNumElements ) ;

  for ( st = 0 ; st < stSmallerSize ; st++ )
  {
    ptNew [st] = ptOld [st] ;

  } /* endfor */

  delete  []  ptOld ;

} /* template < class T > inline void TemplateArray < T > :: resize ( size_t stNumElements ) */



/***************************************************************************************************************************
  template < class T > inline T * TemplateArray < T > :: setBase ( T * ptSrcArray )
****************************************************************************************************************************

  ARGUMENTS   :
    T   * pctSrcArray
      The address of an array of T objects.  The receivers' internal pointer will be set to this address.


  RETURNS     :
    This method returns the value of the argument 'pctSrcArray'.


  DESCRIPTION :
    This accessor initializes a TemplateArray < T > object to point to 'ptSrcArray'.  This is a shallow copy, since the
    receivers' internal pointer is simply set to 'ptSrcArray'.  This method should be used with care and is really only 
    intended for use by the constructors or initializers.

***************************************************************************************************************************/
template < class T > inline T * TemplateArray < T > :: setBase ( T * ptSrcArray )
{
  return  pt  = ptSrcArray ;

} /* template < class T > inline T * TemplateArray < T > :: setBase ( T * ptSrcArray ) */



/***************************************************************************************************************************
  template < class T > inline size_t TemplateArray < T > :: setNumElems ( size_t stNumElements )
****************************************************************************************************************************

  ARGUMENTS   :
    size_t  stNumElements
      The number of elements encapsulated by the receiver.


  RETURNS     :
    This method returns the value of the argument 'stNumElements'.


  DESCRIPTION :
    This accessor initializes the number of elements returned by the receivers' :
      template < class T > inline size_t TemplateArray < T > :: numElems () const
    method.  This method should be used with care and is really only intended for use by the constructors or initializers.

***************************************************************************************************************************/
template < class T > inline size_t TemplateArray < T > :: setNumElems ( size_t stNumElements )
{
  return  stNumElem = stNumElements ;

} /* template < class T > inline size_t TemplateArray < T > :: setNumElems ( size_t stNumElements ) */



/***************************************************************************************************************************
   template < class T > inline TemplateArray < T > :: operator T * ()
****************************************************************************************************************************

  RETURNS     :
    This method returns the address of the array encapsulated by the receiver as a pointer to < T >.


  DESCRIPTION :
    This operator is the NON CONSTANT operator for the address of the array encapsulated by the receiver.  This operator
    converts the receiver to a pointer to < T >.

***************************************************************************************************************************/
template < class T > inline TemplateArray < T > :: operator T * ()
{
  return  getBase () ;

} /* template < class T > inline TemplateArray < T > :: operator T * () */



/***************************************************************************************************************************
   template < class T > inline TemplateArray < T > :: operator const T * () const
****************************************************************************************************************************

  RETURNS     :
    This operator returns the address of the array encapsulated by the receiver as a pointer to a constant < T >.


  DESCRIPTION :
    This operator is the CONSTANT operator for the address of the array encapsulated by the receiver.  This operator
    converts the receiver to a pointer to a CONSTANT < T >.

***************************************************************************************************************************/
template < class T > inline TemplateArray < T > :: operator const T * () const
{
  return  getBase () ;

} /* template < class T > inline TemplateArray < T > :: operator const T * () const */



/***************************************************************************************************************************
   template < class T > inline T & TemplateArray < T > :: operator [] ( size_t stN )
****************************************************************************************************************************

  ARGUMENTS   :
    size_t  stN
      The index of the element we wish to retrieve.

  RETURNS     :
    This operator returns a reference to the 'stN'th element in the array encapsulated by the receiver.


  DESCRIPTION :
    This operator is the NON CONSTANT accessor operator for a TemplateArray < T > as an indexed array.  I.e. if one
    declares an object as TemplateArray < char > chararray ( 10 ), then charrarray [0] returns the first element in that
    array.

***************************************************************************************************************************/
template < class T > inline T & TemplateArray < T > :: operator [] ( size_t stN )
{
  return  * ( getBase () + stN ) ;

} /* template < class T > inline TemplateArray < T > :: operator T & [] () */



/***************************************************************************************************************************
   template < class T > inline const T & TemplateArray < T > :: operator [] ( size_t stN ) const
****************************************************************************************************************************

  RETURNS     :
    This method returns a reference to the 'stN'th element in the array encapsulated by the receiver as a constant.


  DESCRIPTION :
    This operator is the CONSTANT accessor operator for a TemplateArray < T > as an indexed array.  I.e. if one
    declares an object as const TemplateArray < char > cchararray ( 10 ), then ccharrarray [0] returns the first element in
    that array.

***************************************************************************************************************************/
template < class T > inline const T & TemplateArray < T > :: operator [] ( size_t stN ) const
{
  return  * ( getBase () + stN ) ;

} /* template < class T > inline TemplateArray < T > :: operator const T & [] () const */


//  This is a commonly used synonym for for an array of characters.
typedef TemplateArray < char > CharArray ;


# endif  /*  # ifndef __templary_hpp  */


