# ifndef __keyeditem_hpp
  # define __keyeditem_hpp


/***************************************************************************************************************************
 
    FILE NAME :
      KEYEDITEM.HPP
 
 
    DESCRIPTION :
      This module contains the declarations for the classes KeyedItem and AttrKeyedItem.  These classes return a key value
      and is used to instantiate Keyed Collection Classes.
 
 
    Copyright 1997, Peter Garner
    This code may be used for any NON-MILITARY purpose.
 
***************************************************************************************************************************/

# include <templptr.hpp>


//  This class is a simple template class that returns a key value.  This class is intended for use with the Keyed
//  Collection Classes available under OCL.  It may also work with STL.
template < class T > class KeyedItem
{
public    :
  virtual T const         & key ()  const = 0 ;

} ; /* template < class T > class KeyedItem */



//  This class extends the template class KeyedItem by STORING the key value as an attribute and has a public accessor
//  and mutator for that attribute.
template < class T > class AttrKeyedItem : public KeyedItem < T >
{
public    :
  AttrKeyedItem   ( const T & tKeyNew ) ;
  AttrKeyedItem   ( const AttrKeyedItem < T > & rattrkeyeditem ) ;

  virtual T const         & key     ()  const ;
  virtual T const         & setKey  ( const T & tKeyNew ) ;

private   :
  T   tKey  ;

} ; /* template < class T > class AttrKeyedItem : public KeyedItem < T > */



//  This class is a Descendant of TemplatePointer that is intended for use with Keyed Collection Classes.  This is an auto
//  destroy pointer template and is intended to be used as a member of Keyed Pointer Collections.  A pointer of this type
//  can be added to a Collection, and is GUARENTEED to be deleted when the Collections' destructor is called.  The REAL
//  reason this class is declared is so that we can define the 'key' utility method (required by the OCL Keyed Collections)
//  as a template parameterized with both the item type and key type.  C++ will not allow a template utility method to be
//  defined unless BOTH parameters are used in the argument list.  Thus this class is really just a place holder that allows
//  both parameters to be used in the declaration of :
//
//      template < class I , class T > inline T const & key ( TemplateKeyedItemPtr < I , T > const & crtemplptr )
//
//  The latter template utility method lets us define the key method only once.  Note that for this to work, the type I
//  must have a method defined as  "const T & I :: key () const".  template < class T > KeyedItem < T > has such a method
//  defined, thus this class is really intended to be instantiated with a KeyedItem or one of its' descendents.
template < class I , class T > class TemplateKeyedItemPtr : public TemplatePointer < I >
{
public    :
//  Constructors
  //  Simply call the Parent Constructor
  TemplateKeyedItemPtr ( I * piInit ) : TemplatePointer < I > ( piInit ) {}

} ; /* template < class I , class T > class TemplateKeyedItemPtr : public TemplatePointer < I > */



/***************************************************************************************************************************
  template < class T > inline AttrKeyedItem < T > :: AttrKeyedItem ( const T & rcKeyNew )
****************************************************************************************************************************

  ARGUMENTS   :
    const T   & rcKeyNew
      A reference to a constant object of type < T > that will be used to initialize the key attribute.


  DESCRIPTION :
    This constructor creates an AttrKeyedItem with a key value equivalent to the input argument 'rcKeyNew'.


  NOTES       :
    In order to use this constructor, the type < T > must have a copy constructor defined that takes a constant source.

***************************************************************************************************************************/
template < class T > inline AttrKeyedItem < T > :: AttrKeyedItem ( const T & rcKeyNew )
  : tKey ( rcKeyNew )
{

} /* template < class T > inline AttrKeyedItem < T > :: AttrKeyedItem ( const T & rcKeyNew ) */



/***************************************************************************************************************************
  template < class T > inline AttrKeyedItem < T > :: AttrKeyedItem ( const AttrKeyedItem < T > & rattrkeyeditem )
****************************************************************************************************************************

  ARGUMENTS   :
    const AttrKeyedItem < T > & rattrkeyeditem
      A reference to the constant AttrKeyedItem < T > that will be used as the source for this copy constructor.


  DESCRIPTION :
    This copy constructor creates an AttrKeyedItem with a key value that is a COPY of the arguments key value.


  NOTES       :
    In order to use this constructor, the type < T > must have a copy constructor defined that takes a constant source.

***************************************************************************************************************************/
template < class T > inline AttrKeyedItem < T > :: AttrKeyedItem ( const AttrKeyedItem < T > & rattrkeyeditem )
  : tKey ( rattrkeyeditem.key () )
{

} /* template < class T > inline AttrKeyedItem < T > :: AttrKeyedItem ( const AttrKeyedItem < T > & rattrkeyeditem ) */



/***************************************************************************************************************************
  virtual template < class T > inline T const & AttrKeyedItem < T > :: key () const
****************************************************************************************************************************

  RETURNS     :
    This accessor returns a constant reference to the key value attribute of the receiver.

***************************************************************************************************************************/
template < class T > inline T const & AttrKeyedItem < T > :: key () const
{
  return  tKey ;

} /* template < class T > inline T const & AttrKeyedItem < T > :: key () const */


/***************************************************************************************************************************
  virtual template < class T > inline T const & AttrKeyedItem < T > :: setKey ( const T & tKeyNew )
****************************************************************************************************************************

  ARGUMENTS   :
    const T   & tKeyNew    
      A reference to a constant object of type < T > that will be used to initialize the key attribute.


  RETURNS     :
    This mutator returns a constant reference to the NEW key value attribute of the receiver.


  NOTES       :
    In order to use this method, the type < T > must have a copy operator '=' defined that takes a constant source.

***************************************************************************************************************************/
template < class T > inline T const & AttrKeyedItem < T > :: setKey ( const T & crtKeyNew )
{
  return  tKey = crtKeyNew ;

} /* template < class T > inline T const & AttrKeyedItem < T > :: setKey ( const T & tKeyNew ) */



//
//  Utility Methods defined for OCL Compatibility
//



/***************************************************************************************************************************
  template < class T > inline T const & key ( KeyedItem < T > const & crkeyeditem )
****************************************************************************************************************************

  ARGUMENTS   :
    KeyedItem < T > const & crkeyeditem
      A constant reference to the KeyedItem object whose key will be returned


  RETURNS     :
    The key value of the argument 'crkeyeditem' as a T const &.


  DESCRIPTION :
    This is a utility method that returns a reference to the key value of the argument KeyedItem reference.  This method is
    needed by the OCL Keyed Collection Class Templates.


***************************************************************************************************************************/
template < class T > inline T const & key ( KeyedItem < T > const & crkeyeditem )
{
  return  crkeyeditem.key () ;

} /* template < class T > inline T const & key ( KeyedItem < T > ) */



/***************************************************************************************************************************
  template < class I , class T > inline T const & key ( TemplateKeyedItemPtr < I , T > const & crtemplptr )
****************************************************************************************************************************

  ARGUMENTS   :
    TemplateKeyedItemPtr < I , T > const & crtemplptr
      A const reference to the TemplateKeyedItemPtr whose key value we wish to retrieve


  RETURNS     :
    The key value of the argument 'crtemplptr' as a T const &.


  DESCRIPTION :
    This is a utility method that returns a reference to the key value of the argument reference.  This method is needed by
    the OCL Keyed Collection Class Templates.  Note that for this method to work the parameter type I must have a member
    function defined as :

      const T & I :: key () const


***************************************************************************************************************************/
template < class I , class T > inline T const & key ( TemplateKeyedItemPtr < I , T > const & crtemplptr )
{
  return  crtemplptr.getPointer () -> key () ;

} /* inline int const & key ( KeyItemPtr const & crkeyeditemptr ) */



//  Typedef some common uses of these templates
typedef   AttrKeyedItem < int   >   IntKeyedItem    ;
typedef   AttrKeyedItem < long  >   LongKeyedItem   ;
typedef   AttrKeyedItem < short >   ShortKeyedItem  ;

# endif  /*  # ifndef __keyeditem_hpp  */
