/***************************************************************************************************************************

  FILE NAME :
    workunit.cpp


  DESCRIPTION :
    This header contains the declaration for the class WorkUnit.  This class uses the "resource acquisition is
    initialization" technique to manage a database transaction (Unit of Work).  The concept behind this technique of 
    "resource acquisition is initialization" is described by Bjarne Stroustrup in "The C++ Programming Language", 1997, pp.
    366 - 7.


  METHODS DEFINED IN THIS MODULE :
    WorkUnit :: WorkUnit   ()
    WorkUnit :: ~WorkUnit  ()

    virtual const char      * WorkUnit :: className         ()  const
    long                      WorkUnit :: commit            ()
    virtual void              WorkUnit :: freeResource      ()
    long                      WorkUnit :: rollback          ()

***************************************************************************************************************************/
//  Locally Defined C++ Headers
# include <sqlfcns.hpp>
# include <workunit.hpp>




/***************************************************************************************************************************
  WorkUnit :: WorkUnit ()
****************************************************************************************************************************

  DESCRIPTION :
    This constructor creates a WorkUnit object and in doing so begins a database transaction. (Starts a unit of work.)

***************************************************************************************************************************/
WorkUnit :: WorkUnit ()
{
  sqlBeginTrans () ;

} /* WorkUnit :: WorkUnit () */



/***************************************************************************************************************************
  WorkUnit :: ~WorkUnit ()
****************************************************************************************************************************

  DESCRIPTION :
    This destructor checks the value returned by the parent method 'bool AutoObject :: willDestroy ()' and rolls back the
    current transaction (unit of work) if that parent method returns true.

***************************************************************************************************************************/
WorkUnit :: ~WorkUnit ()
{
  try
  {
    destroy () ;

  } /* endtry */
  catch ( ... )
  {
    //  We may want to log an error to openview here, but destructors must NEVER throw exceptions!

  } /* endcatch ( ... ) */

} /* WorkUnit :: ~WorkUnit () */



/***************************************************************************************************************************
  const char * WorkUnit :: className () const
****************************************************************************************************************************

  RETURNS     :
    A constant pointer to the ASCIIZ string "WorkUnit".


  DESCRIPTION :
    This VIRTUAL constant method always returns the class name of the receiver.  This may be used to implement
    a primitive Run Time Type Identification, (RTTI), or anything else you like.  Note that this IS a virtual
    function, and will early bind in the constructors and destructor.  Thus you can also use this method to
    determine in which constructor level you are, although I don't know why you would want to know that!

***************************************************************************************************************************/
const char * WorkUnit :: className () const
{
  return  "WorkUnit" ;

} /* const char * WorkUnit :: className () const */



/***************************************************************************************************************************
  long WorkUnit :: commit ()
****************************************************************************************************************************

  RETURNS     :
    The sqlca.sqlcode resulting from the database call to commit


  DESCRIPTION :
    This method commits the current database transaction (unit of work).

***************************************************************************************************************************/
long WorkUnit :: commit () 
{
  long  lRet = sqlCommit ()  ;

  //  Set the auto destroy flag to false so that the destructor does not issue a rollback!
  setDestroy ( false ) ;

  return  lRet ;

} /* long WorkUnit :: commit () */



/***************************************************************************************************************************
  void WorkUnit :: freeResource ()
****************************************************************************************************************************

  DESCRIPTION :
    This method rolls back the current database transaction (unit of work).

***************************************************************************************************************************/
void WorkUnit :: freeResource () 
{
  rollback ()  ;

} /* void WorkUnit :: freeResource () */



/***************************************************************************************************************************
  long WorkUnit :: rollback ()
****************************************************************************************************************************

  RETURNS     :
    The sqlca.sqlcode resulting from the database call to rollback


  DESCRIPTION :
    This method rolls back the current database transaction (unit of work).

***************************************************************************************************************************/
long WorkUnit :: rollback () 
{
  long  lRet = sqlRollback ()  ;

  //  Set the auto destroy flag to false so that the destructor does not issue a rollback!
  setDestroy ( false ) ;

  return  lRet ;

} /* long WorkUnit :: rollback () */


/**************************************     end of file workunit.cpp           *********************************************/


