/***************************************************************************************************************************

   FILE NAME :
     STRFCNS.C


   APPLICATION :
     GENERIC C MODULE FOR HANDLING STRING AND MEMORY BUFFERS



   DESCRIPTION :
     This module contains some generic functions that deal with string and memory handling.


   FUNCTIONS DEFINED IN THIS MODULE :
      unsigned      memCount          ( const char * pccSrch , char c , size_t stLen )
      short         memIsBlank        ( const char * pcTest , size_t stLen )
      const void  * memrchr           ( const void * pcvSrch , char c , size_t stLen )
      char        * strCopyFill       ( char * szDest , const char * cszSrc , char c , size_t stLen )
      unsigned      strCount          ( const char * pccSrch , char c )
      short         strIsBlank        ( const char * cszTest)
      short         strIsValid        ( const char * szTest )
      size_t        strLen            ( const char * csz )
      char        * strLTrim          ( char * szLine )
      char        * strReset          ( char ** ppcDest , const char * cszSrc )
      char        * strRTrim          ( char * szLine )
      char        * strStripNewLine   ( char * szLine )
      char        * strTermCpy        ( char * szDest , const char * cszSrc , size_t stLen )
      char        * strTermFill       ( char * szDest , const char * cszSrc , char c , size_t stLen )
      char        * strToLower        ( char * sz )
      char        * strToUpper        ( char * sz )


***************************************************************************************************************************/



# include <ctype.h>
# include <stdlib.h>
# include <string.h>


# include "genmacro.h"
# include "strfcns.h"







/***************************************************************************************************************************
  unsigned memCount ( const char * pccSrch , char c , size_t stLen )
****************************************************************************************************************************

  ARGUMENTS   :
    const char  * pccSrch    
      The address of the area of memory to search for the number occurences of 'c'.

    int           c         
      The character for which to search.

    size_t        stLen
      The size of the area or string to search.  NOTE this is the ACTUAL size starting at 1, not 0!  E.g. if this parameter
      is 2, then first 2 bytes of 'pccSrch' will be checked for equivalency to 'c'.


  RETURNS     :
    An unsigned integer that is the number of occurences of 'c' in the first 'stLen' bytes of 'pccSrch'.
    If c cannot be found, 0 is returned.


  DESCRIPTION :
    This searches an area of memory (pccSrch) for the ALL occurences of the character c.  This function returns the number
    of occurences of 'c' in 'pccSrch'.


***************************************************************************************************************************/
unsigned memCount ( const char * pccSrch , char c , size_t stLen )
{
  /*  Initialized Data  */
  /*  I know this is dangerous to cast a const to a non const, but memchr takes a non constant parameter!  */
  char      * pcFound = (char *) pccSrch    ;
  size_t      stFound = 0                   ;
  size_t      stLeft  = stLen               ;
  unsigned    uRetVal = 0                   ;

  while ( 0 != ( pcFound = (char *) memchr ( (char *) pcFound , c , stLeft ) ) )
  {
    /*  An occurance of c was found so increment the return value (count)   */
    uRetVal++ ;
    /*  Increment pcFound to point to the NEXT character after that found!  */
    pcFound++ ;
    /*  Calculate the offset of that found character                        */
    stFound = pcFound - pccSrch ;
    /*  And calculate the number of bytes REMAINING in the search area      */
    /*  using that offset                                                   */
    stLeft = stLen - stFound ;
   
  } /* endwhile */

  return  uRetVal ;

} /* unsigned memCount ( const char * pccSrch , char c , size_t stLen ) */



/***************************************************************************************************************************
  short memIsBlank ( const char * pcTest , size_t stLen ) ;
****************************************************************************************************************************

  ARGUMENTS   :
    char    * pcTest
      This argument points to the area of memory that we wish to test.

    size_t    stLen
      This argument is the length of the area that we wish to test.


  RETURNS     :
    TRUE  (1) if all of the bytes in the memory area are whitespace, or if the stLen argument is 0.
    FALSE (0) otherwise.


  DESCRIPTION :
    This function checks an area of memory beginning at 'pcTest' and extending for 'stLen' bytes to see if that area is all
    white space.


  NOTES       :
    If the stLen argument is 0, TRUE is returned.

***************************************************************************************************************************/
short memIsBlank ( const char * pcTest , size_t stLen )
{
  size_t st ;

  for ( st = 0 ; st < stLen; st++ )
  {
    if ( 0 == isspace ( pcTest [st] ) )
    {
      return 0 ;

    } /* endif */

  } /* endfor */

  return 1 ;

} /* short memIsBlank ( char * pcTest , size_t stLen ) */



/***************************************************************************************************************************
  void * memrchr ( const void * pcvSrch , char c , size_t stLen )
****************************************************************************************************************************

  ARGUMENTS   :
    void  * pcvSrch    
      The address of the area of memory to search for the last occurence of 
      'c'.

    int     c         
      The character for which to search.

    size_t  stLen
      The size of the area or string to search.  NOTE this is the ACTUAL size starting at 1, not 0!  E.g. if this parameter
      is 2, pvSrch[1], then pvSrch[0] will be checked for equivalency to 'c'.


  RETURNS     :
    A pointer to the LAST occurence of c in pvSrch, if c cannot be found, NULL (0) is returned.


  DESCRIPTION :
    This searches an area of memory (pvSrch) for the LAST occurence of the character c.  This function is analogous to
    memchr in the same manner in which strrchr is analogous to strchr.

***************************************************************************************************************************/
const void * memrchr ( const void * pcvSrch , char c , size_t stLen )
{
  const char  * pccSearchMem  = (char * ) pcvSrch ;
  const void  * pcvLastOccur  = 0 ;

  while ( stLen )
  {
    if ( pccSearchMem[--stLen] == c )
    {
      pcvLastOccur = pccSearchMem + stLen ;
      break ;

    } /* endif */

  } /* endwhile */

  return pcvLastOccur ;

}    /* const void * memrchr ( const void * pvSrch , char c , size_t stLen ) */



/***************************************************************************************************************************
  char * strCopyFill ( char * szDest , const char * cszSrc , char c , size_t stLen )
****************************************************************************************************************************

  ARGUMENTS   :
    char        * szDest
      The address of the DESTINATION memory

    const char  * cszSrc
      The address of the SOURCE string

    char      c
      The fill (pad) character.  If strlen ( 'cszSrce' ) is smaller than 'stLen', the memory up to stLen bytes will be padded
      with this character

    size_t    stLen
      The maximum number of bytes to copy or pad.


  RETURNS     :
    'szDest', the address of the DESTINATION memory in all cases.


  DESCRIPTION :
    This function copies the string 'cszSrc' into the location 'szDest', up to a total of 'stLen' bytes, and padding any
    extra bytes up to 'stLen' bytes, much like the standard C library strncpy.  However, this function lets one specify
    the fill character by the argument 'c'.


  NOTES       :
    Unlike strncpy, this function will NOT NULL terminate the destination string, unless the fill character is specified to
    be '\0'.

***************************************************************************************************************************/
char * strCopyFill ( char * szDest , const char * cszSrc , char c , size_t stLen )
{
  size_t stStrLen = strlen ( cszSrc ) ;

  /*  If the total fill length is 0, there is nothing to do, so simply  */
  /*  return the destination address.                                   */
  if ( 0 == stLen )
  {
     return szDest ;

  } /* endif */

  stStrLen  = MIN ( stStrLen , stLen ) ;

  /*  Set all characters in the destination to the argument c  */
  memset ( szDest , c , stLen ) ;

  /*  Copy the specified number of bytes to the target  */
  return  (char *) memmove ( szDest , cszSrc , stStrLen ) ;

}  /*  char * strCopyFill ( char * , const char * , char , size_t )  */



/***************************************************************************************************************************
  unsigned strCount ( const char * pccSrch , char c )
****************************************************************************************************************************

  ARGUMENTS   :
    const char  * pccSrch    
      The starting address of the constant ASCIIZ string in which to count the number occurences of 'c'.

    int           c         
      The character for which to search.


  RETURNS     :
    An unsigned integer that is the number of occurences of 'c' in 'pccSrch'.


  DESCRIPTION :
    This searches an ASCIIZ string (pccSrch) for the ALL occurences of the character c.  This function returns the number
    of occurences of 'c' in 'pccSrch'.

***************************************************************************************************************************/
unsigned strCount ( const char * pccSrch , char c )
{
  size_t  stLen = strlen ( pccSrch ) ;

  return  memCount ( pccSrch , c , stLen ) ;

} /* unsigned strCount ( const char * pccSrch , char c ) */



/***************************************************************************************************************************
  short strIsBlank ( const char * cszTest ) ;
****************************************************************************************************************************

 ARGUMENTS    :
   const char  * cszTest
     The starting address of the constant ASCIIZ string that we wish to test


 RETURNS      :
   TRUE  (1) if all of the bytes in the string are whitespace.
   FALSE (0) otherwise.


 DESCRIPTION  :
   This function checks the constant ASCIIZ string pointed to by 'cszTest' to see if that string is all white space.


 NOTES        :
   If the strings' length is 0, TRUE is returned.

***************************************************************************************************************************/
short strIsBlank ( const char * cszTest)
{
  /*  Initialized Data    */
  size_t stLen = strlen ( cszTest ) ;

  return memIsBlank ( cszTest , stLen ) ;

} /*  short strIsBlank ( const char * cszTest)  */



/***************************************************************************************************************************
  short strIsValid ( const char * cszTest )
****************************************************************************************************************************

  ARGUMENTS   :
    const char  * cszTest
     The starting address of the constant ASCIIZ string that we wish to test


  RETURNS     :
    TRUE (1) if the string doesn't point to NULL and AT LEAST the first character is not '\0'.  I.e. the string length is at
    least 1
    FALSE (0) Otherwise


  DESCRIPTION :
    This function checks a constant ASCIIZ string to make sure that it is non-NULL AND that it has a length > 0.

***************************************************************************************************************************/
short strIsValid ( const char * szTest )
{
  short sRetVal = ( szTest && szTest [0] != '\0' ) ;

  return sRetVal ;

} /* short strIsValid ( const char * szTest ) */



/***************************************************************************************************************************
  size_t strLen ( const char * csz )
****************************************************************************************************************************

  ARGUMENTS   :
    char  * csz
     The starting address of the constant ASCIIZ string whose length we wish to know.
  
  
  RETURNS     :
    The length of the string 'csz' if this string is not null
    not NULL and of length > 0

    0 Otherwise


  DESCRIPTION :
    This function returns the length of the ASCIIZ string supplied in the argument 'csz'.  The purpose of this function is
    to return the length of a string portably and robustly handling NULL pointers and const strings.  Most compiler
    implementations of the 'C' library function 'strlen' have problems when they are passed NULL pointer, this method
    does not!

***************************************************************************************************************************/
size_t strLen ( const char * csz )
{
  return  strIsValid ( csz ) ? strlen ( csz ) : 0  ;

} /* size_t strLen ( const char * csz ) */



/***************************************************************************************************************************
  char * strLTrim ( char * szLine )
****************************************************************************************************************************

  ARGUMENTS   :
   char   * szLine
     A pointer to the ASCIIZ string we wish to trim. 
 
 
  RETURNS     :
    The address the argument string IF this string is not NULL and of length > 0
    0 Otherwise


  DESCRIPTION :
    This function finds the first non whitespace character in a string and then copies from that point to the beginning of
    the string.  In other words this function trims all whitespace from the beginning (left side) of a string.
 
***************************************************************************************************************************/
char * strLTrim ( char * szLine )
{
  char   * szNonWhite = szLine ;

  if ( !strIsValid ( szLine ) )
  {
    return 0 ;

  } /* endif */

  while ( * szNonWhite && isspace ( * szNonWhite ) )
  {
    szNonWhite++ ;

  } /* endwhile */

  if ( szNonWhite != szLine )
  {
    strcpy ( szLine, szNonWhite ) ;

  } /* endif */

  return szLine ;

} /* char * strLTrim ( char * szLine ) */



/***************************************************************************************************************************
  char * strReset ( char ** ppcDest , const char * cszSrc )
****************************************************************************************************************************

  ARGUMENTS   :
   char         **  ppcDest
     A pointer to the ADDRESS of the memory area we wish to realloc.  This MUST point to a buffer that can be manipulated
     with 'free'.

   const char   *   cszSrc 
     The starting address of the constant ASCIIZ string that we wish to copy into 'ppcDest' after the latter has been
     reallocated.
 
 
  RETURNS     :
    The NEW address the argument string '* ppcDest' in ALL cases.  This may be 0 if 'cszrc' is 0 or if the system cannot
    allocate enough memory.


  DESCRIPTION :
    This function frees the memory pointed to by 'ppcDest', allocates enough memory to hold a string of length 'cszSrc',
    sets 'ppcDest' to point to that memory, then strcpys 'szSrc' to 'ppcDest'.  I.e. this function operates just like
    the 'C' library function 'realloc', but additionally does a string copy.


  NOTES       :
    This function calls "free ( ppcDest ) ;", It will go totally haywire  if 'ppcDest' is not 0 or pointing to a malloced
    area of memory!
 
***************************************************************************************************************************/
char * strReset ( char ** ppcDest , const char * cszSrc )
{
  size_t  stLen ;

  free ( * ppcDest ) ;

  if ( 0 == cszSrc )
  {
    return  * ppcDest = 0 ;
    
  } /* endif */

  stLen = strlen ( cszSrc ) ;

  * ppcDest = (char *) malloc ( stLen + 1 ) ;

  if ( * ppcDest )
  {
    strcpy ( * ppcDest , cszSrc ) ;

  } /* endif */

  return  * ppcDest ;

}  /*  char * strReset ( char ** , char * )  */



/***************************************************************************************************************************
  char * strRTrim ( char * szLine )
****************************************************************************************************************************

  ARGUMENTS   :
   char   * szLine
     A pointer to the ASCIIZ string we wish to trim. 
 
 
  RETURNS     :
    The address the argument string IF this string is not NULL and of length > 0
    0 Otherwise


  DESCRIPTION :
    This function finds the LAST non whitespace character in a string and then truncates the string at point. In other words
    this function trims all whitespace from the end (right side) of a string.
 
***************************************************************************************************************************/
char * strRTrim ( char * szLine )
{
  size_t stLen = strlen ( szLine ) ;

  while ( stLen-- )
  {
    if ( isspace ( szLine [stLen] ) )
    {
      szLine [stLen] = '\0' ;
 
    }
    else
    {
      /*  Right end of string is not white space so break out of loop  */
      break ;

    } /* endif */

  } /* endwhile */

  return szLine ;

} /*  char * strRTrim ( char * szLine )  */



/***************************************************************************************************************************
  char * strStripNewLine ( char * szLine )
****************************************************************************************************************************

  ARGUMENTS   :
    char  * szLine    
     A pointer to the ASCIIZ string we wish to trim. 


  RETURNS     :
    A pointer to the input argument 'szLine'.


  DESCRIPTION :
    This function strips any trailing '\n' or '\r' characters from the argument string 'szLine'.

***************************************************************************************************************************/
char * strStripNewLine ( char * szLine )
{
  size_t stLen = strlen ( szLine ) ;

  while ( stLen-- )
  {
    if ( szLine [stLen] == '\n' || szLine [stLen] == '\r' )
    {
      szLine [stLen] = '\0' ;
 
    }
    else
    {
      /*  Right end of string is not a CR or an LF so break out of loop  */
      break ;

    } /* endif */

  } /* endwhile */

  return  szLine ;

} /*  char * strStripNewLine ( char * szLine )  */



/***************************************************************************************************************************
  char * strTermCpy ( char * szDest , const char * cszSrc , size_t stLen )
****************************************************************************************************************************

  ARGUMENTS   :
    char        * szDest
      The address of the ASCIIZ DESTINATION string

    const char  * cszSrc
      The address of the ASCIIZ SOURCE string

    size_t        stLen
      The maximum number of bytes to copy or pad.


  RETURNS     :
    'szDest', the address of the DESTINATION string in all cases.


  DESCRIPTION :
    This function copies the string 'cszSrc' into the location 'szDest', up to a total of 'stLen' bytes, and padding any
    extra bytes up to 'stLen' bytes, JUST like the standard C library strncpy.  However, this function will ALWAYS NULL
    terminate 'szDest' IN ALL CASES at either the length of the SOURCE string or at 'stLen', whichever comes first.  I.e.
    this method guarentees that szDest will be a NULL terminated (ASCIIZ) string upon completion, provided 'szDest' is not
    NULL.

***************************************************************************************************************************/
char * strTermCpy ( char * szDest , const char * cszSrc , size_t stLen )
{
  /*  If the copy length is 0, there is nothing to do!  */
  if ( 0 != stLen && 0 != cszSrc )
  {
    strncpy ( szDest , cszSrc , stLen ) ;

  } /* endif */

  szDest [stLen] = 0 ;

  return  szDest ;

} /*  char * strTermCpy ( char * szDest , const char * cszSrc , size_t stLen )  */



/***************************************************************************************************************************
  char * strTermFill ( char * szDest , const char * cszSrc , char c , size_t stLen )
****************************************************************************************************************************

  ARGUMENTS   :
    char        * szDest
      The address of the ASCIIZ DESTINATION string

    const char  * cszSrc
      The address of the ASCIIZ SOURCE string


    char      c
      The fill (pad) character.  If strlen ( 'cszSrc' ) is smaller than 'stLen', the destination string up to 'stLen' bytes
      will be padded with the character in this argument.


    size_t        stLen
      The maximum number of bytes to copy or pad.


  RETURNS     :
    'szDest', the address of the DESTINATION string in all cases.


  DESCRIPTION :
    This function copies the string 'cszSrc' into the location 'szDest', up to a total of 'stLen' bytes, and padding any
    extra bytes up to 'stLen' bytes, JUST like the standard C library strncpy.  However, this function will ALWAYS NULL
    terminate 'szDest' IN ALL CASES at either the length of the SOURCE string or at 'stLen', whichever comes first.  I.e.
    this method guarentees that szDest will be a NULL terminated (ASCIIZ) string upon completion, provided 'szDest' is not
    NULL.  In addition this function allows you to specify the fill character.

***************************************************************************************************************************/
char * strTermFill ( char * szDest , const char * cszSrc , char c , size_t stLen )
{
  strCopyFill ( szDest, cszSrc, c , stLen ) ;

  /*  Do not write to szDest if it is 0! */
  if ( szDest )
  {
    /*  Only decrement stLen if it is non 0!  */
    stLen = stLen ? stLen - 1 : 0 ;

    szDest [stLen] = '\0' ;    

  } /* endif */

  return szDest ;

}  /*  char * strTermFill ( char * , const char * , char , size_t )  */



/***************************************************************************************************************************
  char * strToLower ( char * sz )
****************************************************************************************************************************

  ARGUMENTS   :
    char        * sz
      The address of the ASCIIZ string we wish to convert to lowercase.



  RETURNS     :
    'sz', the address of the argument string in all cases.


  DESCRIPTION :
    This function converts all of the characters in 'sz' to lowercase, and will not cause an error if 'sz' is NULL.

***************************************************************************************************************************/
char * strToLower ( char * sz )
{
  size_t  st , stLen ;

  if ( 0 == sz )
  {
    return  0 ;

  } /* endif */

  stLen = strlen ( sz ) ;

  for ( st = 0 ; st < stLen ; st++ )
  {
    sz [st] = tolower ( sz [st] ) ;

  } /* endfor */

  return  sz ;

}  /*  char * strToLower ( char * sz )  */



/***************************************************************************************************************************
  char * strToUpper ( char * sz )
****************************************************************************************************************************

  ARGUMENTS   :
    char        * sz
      The address of the ASCIIZ string we wish to convert to uppercase.



  RETURNS     :
    'sz', the address of the argument string in all cases.


  DESCRIPTION :
    This function converts all of the characters in 'sz' to uppercase, and will not cause an error if 'sz' is NULL.

***************************************************************************************************************************/
char * strToUpper ( char * sz )
{
  size_t  st , stLen ;

  if ( 0 == sz )
  {
    return  0 ;

  } /* endif */

  stLen = strlen ( sz ) ;

  for ( st = 0 ; st < stLen ; st++ )
  {
    sz [st] = toupper ( sz [st] ) ;

  } /* endfor */

  return  sz ;

}  /*  char * strToUpper ( char * sz )  */


/**************************************     END OF FILE STRFCNS.C             *********************************************/
