/***************************************************************************************************************************
 
    FILE NAME :
      FSTRMCHK.CPP
 
 
    DESCRIPTION :
      This module contains the methods used to implement the class FStreamCheck.   This class is a descendant of fstream.
      The difference between FStreamCheck and fstream is that FStreamCheck checks for errors after its' IO and throws an
      FStreamError (a descendant of IAccessError) if any IO errors occur.  In addition this module also contains the
      definitions for the class FStreamError.
 
 
    METHODS DEFINED IN THIS MODULE :
      FStreamCheck :: FStreamCheck ()
      FStreamCheck :: FStreamCheck ( const char * cszName , int nIO_Mode , int nProt )
      FStreamCheck :: FStreamCheck ( int nFileDesc )
      FStreamCheck :: FStreamCheck ( int nFileDesc , char * pcBuffer , int nLength )

      void          FStreamCheck :: checkState  ()
      const char  * FStreamCheck :: className   () const
      streampos     FStreamCheck :: fSize       ()
      const char  * FStreamCheck :: fileName    () const
      void          FStreamCheck :: open        ( const char * cszFileName , int nIO_Mode , int nProt )
      istream     & FStreamCheck :: read        ( char * pcData , int nLength )
      istream     & FStreamCheck :: read        ( signed char * pscData , int nLength )
      istream     & FStreamCheck :: read        ( unsigned char * pucData , int nLength )
      streampos     FStreamCheck :: seekoff     ( streamoff so , seek_dir sd , int nIO_Mode )
      streampos     FStreamCheck :: seekpos     ( streampos sp , int nIO_Mode )
      const char  * FStreamCheck :: setFileName ( const char * cszFileName )
      ostream     & FStreamCheck :: write       ( const char * pccData , int nLength )
      ostream     & FStreamCheck :: write       ( const signed char * pcscData , int nLength )   
      ostream     & FStreamCheck :: write       ( const unsigned char * pcucData , int nLength ) 


      FStreamError :: FStreamError ( const char * cszErrMsg , long lErrorID )
      FStreamError :: FStreamError ( const FStreamError & rcfstreamerror )

      const char  * FStreamError :: className () const



    Copyright 1997, Peter Garner
    This code may be used for any NON-MILITARY purpose.
 
***************************************************************************************************************************/
//  Locally Defined C++ Headers
# include <fstrmchk.hpp>



/***************************************************************************************************************************
  FStreamCheck :: FStreamCheck () : fstream ()
****************************************************************************************************************************

  DESCRIPTION :
    This attempts to construct an FStreamCheck object from no arguments


  EXCEPTIONS  :
    If any IO error occurs, an FStreamError is thrown

***************************************************************************************************************************/
FStreamCheck :: FStreamCheck () : fstream ()
{
  CHECK_STATE () ;

} /* FStreamCheck :: FStreamCheck () : fstream () */



/***************************************************************************************************************************
  FStreamCheck :: FStreamCheck ( const char * cszName , int nIO_Mode , int nProt )
****************************************************************************************************************************

  ARGUMENTS   :
    const char  * cszName
      The name of the file to open and associate with the object being constructed.

    int           nIO_Mode
      The ios :: flags to use in opening the file, e.g. ios :: in  or  ios :: ate, etc.

    int           nProt
      The filebuf protections flag.  This defaults to filebuf :: openprot


  DESCRIPTION :
    This attempts to construct an FStreamCheck object by opening the named file with the given flags and protection mode.


  EXCEPTIONS  :
    If there is not enough memory left on the system to allocate a buffer large enough to hold the contents of the entire
    file, an IOutOfMemory exception is thrown.

***************************************************************************************************************************/
FStreamCheck :: FStreamCheck ( const char * cszName , int nIO_Mode , int nProt )
  : fstream ( cszName , nIO_Mode , nProt ) , ourstrgFileName ( cszName )
{
  CHECK_STATE () ;

} /* FStreamCheck :: FStreamCheck ( const char * cszName , int nIO_Mode , int nProt ) */



/***************************************************************************************************************************
  FStreamCheck :: FStreamCheck ( int nFileDesc ) : fstream ( nFileDesc )
****************************************************************************************************************************

  ARGUMENTS   :
  int           nFileDesc
    The file descriptor of an open file.  If this handle is not open, a FStreamError exception will be thrown.


  DESCRIPTION :
    This attempts to construct an FStreamCheck object by associating it with the given file descriptor


  EXCEPTIONS  :
    If any IO error occurs, an FStreamError is thrown

***************************************************************************************************************************/
FStreamCheck :: FStreamCheck ( int nFileDesc ) : fstream ( nFileDesc )
{
  CHECK_STATE () ;

} /* FStreamCheck :: FStreamCheck ( int nFileDesc ) : fstream ( nFileDesc ) */



/***************************************************************************************************************************
  FStreamCheck :: FStreamCheck ( int nFileDesc , char * pcBuffer , int nLength )
****************************************************************************************************************************

  ARGUMENTS   :
    int           nFileDesc
      The file descriptor of an open file.  If this handle is not open, a FStreamError exception will be thrown.

    char        * pcBuffer
      A pointer to a buffer area to be used in buffering the file
    
    int           nLength
      The length of the buffer area to be used in buffering the file


  DESCRIPTION :
    This version constructs an FStreamCheck object that is attached to the file descriptor nFileDesc.  If nFileDesc is not
    open, an FStreamError will be thrown. This constructor also sets up an associated filebuf object with a stream buffer
    that has nLength bytes and begins at the position pointed to by pcBuffer. If pcBuffer or nLength is 0, the associated
    filebuf object is unbuffered.


  EXCEPTIONS  :
    If any IO error occurs, an FStreamError is thrown

***************************************************************************************************************************/
FStreamCheck :: FStreamCheck ( int nFileDesc , char * pcBuffer , int nLength ) : fstream ( nFileDesc , pcBuffer , nLength )
{
  CHECK_STATE () ;

} /* FStreamCheck :: FStreamCheck ( int nFileDesc , char * pcBuffer , int nLength ) : fstream ( nFileDesc , pcBuffer , nLength ) */



/***************************************************************************************************************************
  void FStreamCheck :: checkState ()
****************************************************************************************************************************

  DESCRIPTION :
    This method checks the state of the receiver via a call to ios :: good ().  If the stream is not in a good state an
    FStreamError is thrown.  This is the method used to check the IO operations for this class.


  EXCEPTIONS  :
    If the call to ios :: good returns false, an FStreamError is thrown.

***************************************************************************************************************************/
void FStreamCheck :: checkState ()
{
  CHECK_STATE () ;

} /* void FStreamCheck :: checkState () */



/***************************************************************************************************************************
    const char * FStreamCheck :: className () const
****************************************************************************************************************************

  RETURNS     :
    A constant pointer to the ASCIIZ string "FStreamCheck".


  DESCRIPTION :
    This VIRTUAL constant method always returns the class name of the receiver.  This may be used to implement
    a primitive Run Time Type Identification, (RTTI), or anything else you like.  Note that this IS a virtual
    function, and will early bind in the constructors and destructor.  Thus you can also use this method to
    determine in which constructor level you are, although I don't know why you would want to know that!

***************************************************************************************************************************/
const char * FStreamCheck :: className () const
{
  return  "FStreamCheck" ;

} /* const char * FStreamCheck :: className () const */



/***************************************************************************************************************************
  const char * FStreamCheck :: fileName () const
****************************************************************************************************************************

  DESCRIPTION :
    This method returns the file name of the file associated with the receiver if the receiver was constructed with a file
    name argument.

***************************************************************************************************************************/
const char * FStreamCheck :: fileName () const
{
  return  ourstrgFileName ;

} /*  const char * FStreamCheck :: fileName () const  */



/***************************************************************************************************************************
  streampos FStreamCheck :: fSize ()
****************************************************************************************************************************

  RETURNS     :
    The size of the file in bytes


  DESCRIPTION :
    This method returns the size of the file associated with the receiver in bytes.  This may not work properly on a binary
    file opened in text mode, since the two byte carriage return - line feed sequence is translated into a one byte line
    feed in text mode.


  EXCEPTIONS  :
    If any IO error occurs, an FStreamError is thrown

***************************************************************************************************************************/
streampos FStreamCheck :: fSize ()
{
  //  Save current fpointer where size WILL go!
  streampos   streamposStart  = seekoff ( (streamoff) 0 , ios :: cur )  ;
  //  Seek to the end and get the offset at the end, this will be the size
  streampos   streamposSize   = seekoff ( (streamoff) 0 , ios :: end )  ;

  //  Return to the position from which we started
  seekoff ( streamposStart , ios :: beg ) ;
  
  //  Return the size
  return  streamposSize ;

} /* streampos FStreamCheck :: fSize () */



/***************************************************************************************************************************
  void FStreamCheck :: open ( const char * cszFileName , int nIO_Mode , int nProt )
****************************************************************************************************************************

  ARGUMENTS   :
    const char  * cszFileName
      The name of the file to open

    int           nIO_Mode
      the ios :: flags to use in opening the file, e.g. ios :: in  or  ios :: ate, etc.

    int           nProt
      The filebuf protections flag.  This defaults to filebuf :: openprot


  DESCRIPTION :
    This method attempts to open the file 'cszFileName' using the specified mode and protection flags.  This method operates
    identically to fstream :: open, but this class will throw an exception if there is an IO error.


  EXCEPTIONS  :
    If any IO error occurs, an FStreamError is thrown

***************************************************************************************************************************/
void FStreamCheck :: open ( const char * cszFileName , int nIO_Mode , int nProt )
{
  clear           () ;
  setFileName     ( cszFileName ) ;
  fstream :: open ( cszFileName , nIO_Mode , nProt ) ;
  CHECK_STATE     () ;

} /* void FStreamCheck :: open ( const char * cszFileName , int nIO_Mode , int nProt ) */



/***************************************************************************************************************************
  istream & FStreamCheck :: read ( char * pcData , int nLength )
****************************************************************************************************************************

  ARGUMENTS   :
    char        * pcData
      A pointer to the buffer area that will receive the data read from the file.
        
    int           nLength
      The number of bytes to read


  DESCRIPTION :
    This method attempts to read 'nLength' bytes from the file associated with the receiver into the buffer 'pcData'.  This
    method operates identically to fstream :: read, but this class will throw an exception if there is an IO error.


  EXCEPTIONS  :
    If any IO error occurs, an FStreamError is thrown

***************************************************************************************************************************/
istream & FStreamCheck :: read ( char * pcData , int nLength )
{
  int   nBytesRead  = 0 ;

  clear       () ;
  fstream :: read ( pcData , nLength ) ;

  nBytesRead  = gcount () ;

  //  If EOF file was encountered, failbit and eofbit will be set.  So if we got end of file, but read at least one
  //  character, DO NOT throw an exception
  if ( 0 != nBytesRead && ( ios :: eofbit | ios :: failbit ==  rdstate () ) )
  {
    return  (istream &) * this ;

  } /* endif */

  CHECK_STATE () ;

  return  (istream &) * this ;

} /* istream & FStreamCheck :: read ( char * pcData , int nLength ) */



/***************************************************************************************************************************
  istream & FStreamCheck :: read ( signed char * pscData , int nLength )
****************************************************************************************************************************

  ARGUMENTS   :
    char        * pscData
      A pointer to the buffer area that will receive the data read from the file.
            
    int           nLength
      The number of bytes to read


  DESCRIPTION :
    This method attempts to read 'nLength' bytes from the file associated with the receiver into the buffer 'pscData'.  This
    method operates identically to fstream :: read, but this class will throw an exception if there is an IO error.


  EXCEPTIONS  :
    If any IO error occurs, an FStreamError is thrown

***************************************************************************************************************************/
istream & FStreamCheck :: read ( signed char * pscData , int nLength )
{
  return  read ( (char *) pscData , nLength ) ;    

} /* istream & FStreamCheck :: read ( signed char * pscData , int nLength ) */



/***************************************************************************************************************************
  istream & FStreamCheck :: read ( unsigned char * pucData , int nLength )
****************************************************************************************************************************

  ARGUMENTS   :
    char        * pucData
    A pointer to the buffer area that will receive the data read from the file.
                
    int           nLength
    The number of bytes to read


  DESCRIPTION :
    This method attempts to read 'nLength' bytes from the file associated with the receiver into the buffer 'pucData'.  This
    method operates identically to fstream :: read, but this class will throw an exception if there is an IO error.


  EXCEPTIONS  :
    If any IO error occurs, an FStreamError is thrown

***************************************************************************************************************************/
istream & FStreamCheck :: read ( unsigned char * pucData , int nLength )
{
  return  read ( (char *) pucData , nLength ) ;  

} /* istream & FStreamCheck :: read ( unsigned char * pucData , int nLength ) */



/***************************************************************************************************************************
  streampos FStreamCheck :: seekoff ( streamoff so , seek_dir sd , int nIO_Mode )
****************************************************************************************************************************

  ARGUMENTS
    streamoff   so
      The offset to which to seek

    seek_dir    sd
      The direction in which to seek ( ios :: beg , ios :: curr , ios :: end  )

    int         nIO_Mode
      The IO mode to use for the seek operation


  RETURNS     :
    The new file position.


  DESCRIPTION :
    This method attempts to seek the receivers' rdbuffer to the position specified by the argument 'streamoff so'.  This
    method operates identically to 'streambuf :: seekoff' but this class will throw an exception if there is an IO error.


  EXCEPTIONS  :
    If any IO error occurs, an FStreamError is thrown

***************************************************************************************************************************/
streampos FStreamCheck :: seekoff ( streamoff so , seek_dir sd , int nIO_Mode )
{
  streampos   strmpos = rdbuf () -> seekoff ( so , sd , nIO_Mode ) ;

  if ( EOF == strmpos )
  {
    THROW_FSTREAM_ERROR ( "Attempt to seek on FStreamCheck to invalid position" , EBADPOS ) ;

  } /* endif */

  return  strmpos ;

} /* streampos FStreamCheck :: seekoff ( streamoff so , seek_dir sd , int nIO_Mode ) */



/***************************************************************************************************************************
  streampos FStreamCheck :: seekpos ( streampos sp , int nIO_Mode )
****************************************************************************************************************************

  ARGUMENTS
    streampos   ps
      The position to which to seek

    int         nIO_Mode
      The IO mode to use for the seek operation


  DESCRIPTION :
    This method attempts to seek the receivers' rdbuffer to the position specified by the argument 'streampos sp'.  This
    method operates identically to 'streambuf :: seekpos' but this class will throw an exception if there is an IO error.


  EXCEPTIONS  :
    If any IO error occurs, an FStreamError is thrown

***************************************************************************************************************************/
streampos FStreamCheck :: seekpos ( streampos sp , int nIO_Mode )
{
  return  seekoff ( (streamoff) sp , ios :: beg , nIO_Mode )  ;

} /* streampos FStreamCheck :: seekpos ( streampos sp , int nIO_Mode ) */



/***************************************************************************************************************************
  const char * FStreamCheck :: setFileName ( const char * cszFileName )
****************************************************************************************************************************

  ARGUMENTS   :
    const char  * cszFileName
      The name of the file associated with the receiver


  RETURNS     :
    This method returns new file name as stored by the receiver.


  DESCRIPTION :
    This PROTECTED accessor provides the means of setting the file name associated with the receiver.  This method is
    protected since we do not want to allow the outside world to accidentally mismatch this name and the opened file!

***************************************************************************************************************************/
const char * FStreamCheck :: setFileName ( const char * cszFileName )
{
  return  ourstrgFileName = cszFileName ;

} /*  const char * FStreamCheck :: setFileName ( const char * cszFileName  )  */



/***************************************************************************************************************************
  ostream & FStreamCheck :: write ( const char * pccData , int nLength )
****************************************************************************************************************************

  ARGUMENTS   :
    char        * pccData
      A pointer to the buffer area that is the source of the data to be written.
            
    int           nLength
    The number of bytes to write


  DESCRIPTION :
    This method attempts to write 'nLength' bytes from the buffer 'pccData' to the file associated with the receiver.  This
    method operates identically to fstream :: write, but this class will throw an exception if there is an IO error.


  EXCEPTIONS  :
    If any IO error occurs, an FStreamError is thrown

***************************************************************************************************************************/
ostream & FStreamCheck :: write ( const char * pccData , int nLength )
{
  clear       () ;
  fstream :: write ( pccData , nLength ) ;
  CHECK_STATE () ;

  return  (ostream &) * this ;

} /* ostream & FStreamCheck :: write ( const char * pccData , int nLength ) */



/***************************************************************************************************************************
  ostream & FStreamCheck :: write ( const signed char * pcscData , int nLength )
****************************************************************************************************************************

  ARGUMENTS   :
    char        * pcscData
      A pointer to the buffer area that is the source of the data to be written.
                
    int           nLength
      The number of bytes to write


  DESCRIPTION :
    This method attempts to write 'nLength' bytes from the buffer 'pcscData' to the file associated with the receiver.  This
    method operates identically to fstream :: write, but this class will throw an exception if there is an IO error.


  EXCEPTIONS  :
    If any IO error occurs, an FStreamError is thrown

***************************************************************************************************************************/
ostream & FStreamCheck :: write ( const signed char * pcscData , int nLength )
{
  return  write ( (char *) pcscData , nLength ) ;  

} /* ostream & FStreamCheck :: write ( const signed char * pcscData , int nLength ) */



/***************************************************************************************************************************
  ostream & FStreamCheck :: write ( const unsigned char * pcucData , int nLength )
****************************************************************************************************************************

  ARGUMENTS   :
    char        * pcucData
      A pointer to the buffer area that is the source of the data to be written.
                    
    int           nLength
      The number of bytes to write


  DESCRIPTION :
    This method attempts to write 'nLength' bytes from the buffer 'pcucData' to the file associated with the receiver.  This
    method operates identically to fstream :: write, but this class will throw an exception if there is an IO error.


  EXCEPTIONS  :
    If any IO error occurs, an FStreamError is thrown

***************************************************************************************************************************/
ostream & FStreamCheck :: write ( const unsigned char * pcucData , int nLength )
{
  return  write ( (char *) pcucData , nLength ) ;

} /* ostream & FStreamCheck :: write ( const unsigned char * pcucData , int nLength ) */



/***************************************************************************************************************************
  FStreamError :: FStreamError ( const char * cszErrMsg , long lErrorId )
****************************************************************************************************************************

  ARGUMENTS   :
    const char  * cszErrMsg
      A pointer to the error message as a constant ASCIIZ string

    long          lErrorId
      The numeric error id to be used in constructing the object.
    

  DESCRIPTION :
    This constructs an FStreamError object with the given error message and error id

***************************************************************************************************************************/
FStreamError :: FStreamError ( const char * cszErrMsg , long lErrorId ) : IAccessError ( cszErrMsg , lErrorId )
{

} /* FStreamError :: FStreamError ( const char * cszErrMsg , long lErrorId ) */



/***************************************************************************************************************************
  FStreamError :: FStreamError ( const FStreamError & rcfstreamerror )
****************************************************************************************************************************

  ARGUMENTS   :
    const FStreamError  & rcfstreamerror
      A reference to the constant FStreamError object that will be used as the source for this DEEP copy constructor.
    

  DESCRIPTION :
    This is the copy constructor for an FStreamError.  This is the constructor the compiler uses when throwing an object of
    this type.

***************************************************************************************************************************/
FStreamError :: FStreamError ( const FStreamError & rcfstreamerror ) : IAccessError ( rcfstreamerror )
{
  //  This copy constructor is needed so that we can throw an object of this type.  The parent class
  //  (IAccessError) constructor should work fine for this class as this class defines no new data.

} /* FStreamError :: FStreamError ( const FStreamError & rcfstreamerror ) : IAccessError ( rcfstreamerror ) */



/***************************************************************************************************************************
  const char * FStreamError :: className () const
****************************************************************************************************************************

  RETURNS     :
    A constant pointer to the ASCIIZ string "FStreamError".


  DESCRIPTION :
    This VIRTUAL constant method always returns the class name of the receiver.  This may be used to implement
    a primitive Run Time Type Identification, (RTTI), or anything else you like.  Note that this IS a virtual
    function, and will early bind in the constructors and destructor.  Thus you can also use this method to
    determine in which constructor level you are, although I don't know why you would want to know that!

***************************************************************************************************************************/
const char * FStreamError :: className () const
{
  return  "FStreamError" ;

} /* const char * FStreamError :: className () const */

           

/***************************************************************************************************************************
  const char * FStreamError :: name () const
****************************************************************************************************************************

  RETURNS     :
    A constant pointer to the ASCIIZ string "FStreamError".


  DESCRIPTION :
  This method overides the IException name method defined by OCL so that it returns our NamedObject :: className method.
  I.e. this method overloads IException :: name so that it returns a value identical to NamedObject :: className.

***************************************************************************************************************************/
const char * FStreamError :: name () const
{
  return  className () ;

} /* const char * FStreamError :: name () const */




/**************************************     END OF FILE FSTRMCHK.CPP          *********************************************/

