/***************************************************************************************************************************
 
    FILE NAME :
      FILENAME.C


    DESCRIPTION :
      This module contains specific C functions for handling file names.  Nearly every operating system or C compiler
      provides a similiar set of functions.  The purpose of this module is to provide a PORTABLE set of such functions.
 
 
    FUNCTIONS DEFINED IN THIS MODULE :
      char        * makeFullFileName ( char * szFullName , const char * cszPath , const char * cszFileName )
      const char  * replaceExtension ( char * szFilePath , const char * cszNewExt ) 
 
 
    Copyright 1997, Peter Garner
    This code may be used for any NON-MILITARY purpose.

/***************************************************************************************************************************/

/*   System  Defined C Headers  */
# include <string.h>
# include <strfcns.h>

/*   Locally Defined C Headers  */
# include <ospath.h>
# include <filename.h>




/***************************************************************************************************************************
  char * makeFullFileName ( char * szFullName , const char * cszPath , const char * cszFileName )
****************************************************************************************************************************

  ARGUMENTS   :
    char        * szFullName
      The address of the buffer that will contain the assembled file name as an ASCIIZ string.

    const char  * cszPath
      The address of the buffer containing the directory information as a CONSTANT ASCIIZ string

    const char  * cszFileName
      The address of the buffer containing the file name as a CONSTANT ASCIIZ string


  RETURNS     :
    The address of the string argument 'szFullName'.
    If the argument 'szFullName' is 0, or the argument cszFileName is blank or 0, 0 will be returned.


  DESCRIPTION :
    This function concatenates its' arguments 'cszPath' and 'cszFileName' and copies the result to the address pointed to by
    the argument 'szFullPath'.  If there is no path separator (defined as DIR_SEPARATOR in "ospath.h") either at the end of
    'cszPath' OR at the beginning of 'cszFileName' one will be inserted between them during the concatenation.

***************************************************************************************************************************/
char * makeFullFileName ( char * szFullName , const char * cszPath , const char * cszFileName )
{
  size_t  stPathLength = 0 ;
  size_t  stNameLength = 0 ;

  if ( 0 == szFullName || 0 == strIsValid ( cszFileName ) )
  {
    return 0 ;

  } /* endif */

  stNameLength = strlen ( cszFileName ) ;

  if ( cszPath )
  {
    stPathLength  = strlen ( cszPath ) ;
    memmove ( szFullName , cszPath , stPathLength )  ;

  } /* endif */

  if ( 0 == stPathLength || DIR_SEPARATOR != szFullName [stPathLength-1] )
  {
    szFullName [stPathLength++] = DIR_SEPARATOR ;

  } /* endif */

  memmove ( szFullName + stPathLength , cszFileName , stNameLength + 1 ) ;

  return szFullName ;

} /* char * makeFullFileName ( char * szFullName , const char * cszPath , const char * cszFileName ) */



/***************************************************************************************************************************
  const char * replaceExtension ( char * szFilePath , const char * cszNewExt )
****************************************************************************************************************************

  ARGUMENTS   :
    char        * szFilePath 
      The address of the buffer containing the file name as an ASCIIZ string

    const char  * cszNewExt
      The address of the buffer containing the new file extension as a CONSTANT ASCIIZ string


  RETURNS     :
    The address of the string argument 'szFilePath'.
    If the argument 'szFilePath' is blank or 0, or the argument 'cszNewExt' is blank or 0, 0 will be returned.



  DESCRIPTION :
    This either appends the argument 'cszNewExt' to the argument 'szFileName' or replaces the file extension on 'szFileName'
    with 'cszNewExt'.  The string 'szFileName' is reverse searched for a '.' character.  If none is found, then cszFileName
    is simply appended to 'szFileName'.  If one IS found, it is assumed to be the extension separator and 'cszNewExt' is
    copied into 'szFileName' at that point.  In any case, one and only one '.' character will be placed between 'szFileName'
    and 'cszNewExtension'.  This means that one can supply a string for the argument 'cszNewExt' with OR without a '.'
    character, and the result will be identical, e.g. any combination of the following arguments :

      szFilePath  = 'filename'
      szFilePath  = 'filename.'
      cszNewExt   = '.ext'
      cszNewExt   = 'ext'

    will still result in the string 'filename.ext'

***************************************************************************************************************************/
const char * replaceExtension ( char * szFilePath , const char * cszNewExt )
{
  char    * szExtension = 0 ;
  size_t    stPathLen   = 0 ;
    
  //  Make sure that both pointers are valid
  if ( ! strIsValid ( szFilePath ) || ! strIsValid ( cszNewExt ) )
  {
    return  0 ;

  } /* endif */

  //  See if there was a '.' character in the file name
  szExtension   = strrchr ( szFilePath , '.'  ) ;
  stPathLen     = strlen  ( szFilePath        ) ;

  //  If there was, point the extension pointer to that '.', otherwise point it to the end of the string cszFilePath
  szExtension = szExtension ? szExtension : szFilePath + stPathLen ;

  //  See if the supplied extension begins with a period
  if ( '.' != cszNewExt [0] )
  {
    szFilePath [stPathLen++] = '.'  ;

  } /* endif */

  strcpy ( szExtension , cszNewExt ) ;

  return  szFilePath ;

} /* const char * replaceExtension ( char * szFilePath , const char * cszNewExt ) */



/**************************************     END OF FILE FILENAME.C             *********************************************/
