/***************************************************************************************************************************
 
    FILE NAME :
      BUFFILE.CPP
 
 
    DESCRIPTION :
      This module contains the methods used to implement the class BuffedFile.  This class reads an entire file into memory
      and then closes the file.  Its' purpose is to provide the ability to treat a file as a memory buffer.  Thus this class
      is a little bit like a UNIX memory mapped file.  However, unlike a UNIX memory mapped file, once the physical disk
      file has been read into memory, it is closed, unlocked and is available to other processes.
 
 
    METHODS DEFINED IN THIS MODULE :
      BuffedFile :: BuffedFile ( const char * cszFileName = 0 )
      BuffedFile :: BuffedFile ( const char * ccaBuffer , size_t stBuffLength )

      char              * BuffedFile :: buffer        ()
      const             * BuffedFile :: buffer        () const
      CharArray         & BuffedFile :: charArray     ()
      const char        * BuffedFile :: className     () const 
      void                BuffedFile :: flush         ( const char * cszFileName , int nIO_Mode )
      size_t              BuffedFile :: length        () const
      virtual char      * BuffedFile :: read          ( const char * cszFileName , int nIO_Mode )
      size_t              BuffedFile :: setLength     ( size_t stNew )
 
 
    Copyright 1997, Peter Garner
    This code may be used for any NON-MILITARY purpose.


***************************************************************************************************************************/
//  Locally Defined C++ Headers
# include <buffile.hpp>




/***************************************************************************************************************************
  BuffedFile :: BuffedFile ( const char * cszFileName = 0 , int nIO_Mode = cnDEF_IN_MODE )
****************************************************************************************************************************

  ARGUMENTS   :
    const char  * cszFileName   = 0
      The name of the file we wish to read into memory

    int           nIO_Mode      = cnDEF_IN_MODE
      The set of ios flags to be used in opening the file for reading.

  
  DESCRIPTION :
    This attempts to construct a BuffedFile object by reading the file named in the argument 'cszFileName' into memory.


  EXCEPTIONS  :
    If any IO error occurs, an FStreamError exception is thrown
    If there is not enough memory left on the system to allocate the buffer, an IOutOfMemory exception is thrown

***************************************************************************************************************************/
BuffedFile :: BuffedFile ( const char * cszFileName , int nIO_Mode )
{
  if ( 0 == cszFileName )
  {
    return  ;

  } /* endif */

  read ( cszFileName , nIO_Mode ) ;

} /* BuffedFile :: BuffedFile ( const char * cszFileName = 0 , int nIO_Mode = cnDEF_IN_MODE ) */



/***************************************************************************************************************************
  BuffedFile :: BuffedFile ( size_t stBuffLength , const char * ccaBuffer )
****************************************************************************************************************************

  ARGUMENTS   :
    const char  * ccaBuffer
      A pointer to a constant area of memory.  The first 'stBuffLength' bytes of this buffer will be copied into the the
      internal buffer of the object being constructed.

    size_t        stBuffLength
      The initial buffer size that the object being constructed should allocate.  I.e. the object being constructed will
      allocate an internal buffer of this size.

  
  DESCRIPTION :
    This method attempts to construct a BuffedFile object as a DEEP copy of a pre-existing buffer.  Alternatively, by
    providing just the length, and a buffer argument of 0, a BuffedFile Object will be constructed with a buffer of length
    'stBuffLength', whose contents are all 0.  This constructor is useful when one wishes to manipulate an area of memory
    and then write the contents of that area to a disk file.


  EXCEPTIONS  :
    If any IO error occurs, an FStreamError exception is thrown
    If there is not enough memory left on the system to allocate a buffer large enough to hold the contents of the entire
    file, an IOutOfMemory exception is thrown.

***************************************************************************************************************************/
BuffedFile :: BuffedFile ( const char * ccaBuffer , size_t stBuffLength ) : chararray ( stBuffLength , ccaBuffer )
{

} /* BuffedFile :: BuffedFile ( const char * ccaBuffer , size_t stBuffLength ) */



/***************************************************************************************************************************
  char * BuffedFile :: buffer ()
****************************************************************************************************************************

  RETURNS     :
    The address of the receivers' internal memory buffer as pointer to constant.

  
  DESCRIPTION :
    This is the accessor for internal memory area encapsulated by this class (the buffer where the contents of the file are
    stored).  This is the preferred means of gaining access that that buffer.

***************************************************************************************************************************/
char * BuffedFile :: buffer ()
{
  return  (char *) charArray () ;

} /*  char * BuffedFile :: buffer ()  */



/***************************************************************************************************************************
  const char * BuffedFile :: buffer () const
****************************************************************************************************************************

  RETURNS     :
    The address of the receivers' internal memory buffer.

  
  DESCRIPTION :
  This is the accessor for internal memory area encapsulated by this class (the buffer where the contents of the file are
  stored).  This is the preferred means of gaining access that that buffer.

***************************************************************************************************************************/
const char * BuffedFile :: buffer () const
{
  return  (const char *) charArray () ;

} /*  const char * BuffedFile :: buffer () const  */



/***************************************************************************************************************************
  CharArray & BuffedFile :: charArray ()
****************************************************************************************************************************

  RETURNS     :
    A reference to the TemplateArray < char > (typedefined as CharArray) encapsulated by the receiver

  
  DESCRIPTION :
    This is the PROTECTED accessor for the CharArray object this class uses to manage its' internal memory buffer.
    This is the NON-CONSTANT overload of this accessor.

***************************************************************************************************************************/
CharArray & BuffedFile :: charArray ()
{
  return  chararray ;

} /*  CharArray & BuffedFile :: charArray ()  */



/***************************************************************************************************************************
  const CharArray & BuffedFile :: charArray () const
****************************************************************************************************************************

  RETURNS     :
    A reference to the TemplateArray < char > (typedefined as CharArray) encapsulated by the receiver as a constant

  
  DESCRIPTION :
    This is the PROTECTED accessor for the CharArray object this class uses to manage its' internal memory buffer.
    This is the CONSTANT overload of this accessor.

***************************************************************************************************************************/
const CharArray & BuffedFile :: charArray () const
{
  return  chararray ;

} /*  const CharArray & BuffedFile :: charArray () const  */



/***************************************************************************************************************************
    const char * BuffedFile :: className () const
****************************************************************************************************************************

  RETURNS     :
    A constant pointer to the ASCIIZ string "BuffedFile".


  DESCRIPTION :
    This VIRTUAL constant method always returns the class name of the receiver.  This may be used to implement
    a primitive Run Time Type Identification, (RTTI), or anything else you like.  Note that this IS a virtual
    function, and will early bind in the constructors and destructor.  Thus you can also use this method to
    determine in which constructor level you are, although I don't know why you would want to know that!

***************************************************************************************************************************/
const char * BuffedFile :: className () const
{
  return  "BuffedFile" ;

} /* const char * BuffedFile :: className () const */



/***************************************************************************************************************************
    void BuffedFile :: flush ( const char * cszFileName , int nIO_Mode )
****************************************************************************************************************************

  ARGUMENTS   :
    const char  * cszFileName
      The name of the file to which to write the contents of the receivers' internal memory buffer.  I.e. the contents of
      the receivers' buffer will be saved (flushed) to this file name.

    int           nIO_Mode
      The set of ios flags to be used in opening the file for writing.  This argument defaults to cnDEF_OUT_MODE.

  DESCRIPTION :
    This method writes the contents of the receivers' internal memory buffer to the physical disk file named in the input
    argument 'cszFileName'.


  EXCEPTIONS  :
    If any IO error occurs, an FStreamError exception is thrown

***************************************************************************************************************************/
void BuffedFile :: flush ( const char * cszFileName , int nIO_Mode )
{
  open  ( cszFileName , nIO_Mode  ) ;
  write ( buffer ()   , length () ) ;
  close ()                          ;

} /* void BuffedFile :: flush ( const char * cszFileName , int nIO_Mode ) */



/***************************************************************************************************************************
  size_t BuffedFile :: length () const
****************************************************************************************************************************

  RETURNS     :
    The length in bytes of the receivers' internal memory buffer.

  
  DESCRIPTION :
    This is the accessor for size in bytes of the internal memory area encapsulated by this class.

***************************************************************************************************************************/
size_t BuffedFile :: length () const
{
  return  charArray ().numElems () ;

} /*  size_t BuffedFile :: length () const  */



/***************************************************************************************************************************
    char * BuffedFile :: read ( const char * cszFileName , int nIO_Mode = cnDEF_IN_MODE )
****************************************************************************************************************************

  ARGUMENTS   :
    const char  * cszFileName
      The file name we wish to read into the receivers' internal memory buffer.

    int           nIO_Mode = cnDEF_IN_MODE
      The set of ios flags to be used in opening the file for reading.


  RETURNS     :
    The address of the receivers' internal memory buffer.


  DESCRIPTION :
    This method attempts to read the contents of the file named in the input argument 'cszFileName' into the receivers'
    internal memory buffer.  This method will resize the receivers' internal buffer to the length of that file and read
    the entire contents of that file into the buffer.


  EXCEPTIONS  :
    If any IO error occurs, an FStreamError exception is thrown
    If there is not enough memory left on the system to allocate the buffer, an IOutOfMemory exception is thrown

***************************************************************************************************************************/
char * BuffedFile :: read ( const char * cszFileName , int nIO_Mode )
{
  open        ( cszFileName , nIO_Mode    ) ;
  //  Resize the internal buffer to hold the entire contents of the file
  setLength   ( FStreamCheck :: fSize ()  ) ;
  //  Read the entire file into memory
  FStreamCheck :: read     ( buffer ()   , length  ()  ) ;
  close () ;

  return  buffer () ;

} /* char * BuffedFile :: read ( const char * cszFileName , int nIO_Mode ) */



/***************************************************************************************************************************
  size_t BuffedFile :: setLength ( size_t stBuffLength )
****************************************************************************************************************************

  ARGUMENTS   :
    size_t      stBuffLength
      The new buffer size that the receiver should allocate.


  RETURNS     :
    The new buffer size, i.e. the value of the input argument 'szBuffLength'

  
  DESCRIPTION :
    This method reallocates the receivers' internal buffer to the size specified in the input argument 'stBuffLength'.
    This is done via a call to CharArray.resize ( size_t stBuffLength ).


  EXCEPTIONS  :
    If there is not enough memory left on the system to allocate the buffer, an IOutOfMemory exception is thrown

***************************************************************************************************************************/
size_t BuffedFile :: setLength ( size_t stBuffLength )
{
  charArray ().resize ( stBuffLength ) ;

  return  stBuffLength ;

} /*  size_t BuffedFile :: setLength ( size_t stNew )  */



/**************************************     END OF FILE BUFFILE.CPP           *********************************************/